/*! DataTables 1.10.23
 * ©2008-2020 SpryMedia Ltd - datatables.net/license
 */

/**
 * @summary     DataTables
 * @description Paginate, search and order HTML tables
 * @version     1.10.23
 * @file        jquery.dataTables.js
 * @author      SpryMedia Ltd
 * @contact     www.datatables.net
 * @copyright   Copyright 2008-2020 SpryMedia Ltd.
 *
 * This source file is free software, available under the following license:
 *   MIT license - http://datatables.net/license
 *
 * This source file is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
 *
 * For details please refer to: http://www.datatables.net
 */

/*jslint evil: true, undef: true, browser: true */
/*globals $,require,jQuery,define,_selector_run,_selector_opts,_selector_first,_selector_row_indexes,_ext,_Api,_api_register,_api_registerPlural,_re_new_lines,_re_html,_re_formatted_numeric,_re_escape_regex,_empty,_intVal,_numToDecimal,_isNumber,_isHtml,_htmlNumeric,_pluck,_pluck_order,_range,_stripHtml,_unique,_fnBuildAjax,_fnAjaxUpdate,_fnAjaxParameters,_fnAjaxUpdateDraw,_fnAjaxDataSrc,_fnAddColumn,_fnColumnOptions,_fnAdjustColumnSizing,_fnVisibleToColumnIndex,_fnColumnIndexToVisible,_fnVisbleColumns,_fnGetColumns,_fnColumnTypes,_fnApplyColumnDefs,_fnHungarianMap,_fnCamelToHungarian,_fnLanguageCompat,_fnBrowserDetect,_fnAddData,_fnAddTr,_fnNodeToDataIndex,_fnNodeToColumnIndex,_fnGetCellData,_fnSetCellData,_fnSplitObjNotation,_fnGetObjectDataFn,_fnSetObjectDataFn,_fnGetDataMaster,_fnClearTable,_fnDeleteIndex,_fnInvalidate,_fnGetRowElements,_fnCreateTr,_fnBuildHead,_fnDrawHead,_fnDraw,_fnReDraw,_fnAddOptionsHtml,_fnDetectHeader,_fnGetUniqueThs,_fnFeatureHtmlFilter,_fnFilterComplete,_fnFilterCustom,_fnFilterColumn,_fnFilter,_fnFilterCreateSearch,_fnEscapeRegex,_fnFilterData,_fnFeatureHtmlInfo,_fnUpdateInfo,_fnInfoMacros,_fnInitialise,_fnInitComplete,_fnLengthChange,_fnFeatureHtmlLength,_fnFeatureHtmlPaginate,_fnPageChange,_fnFeatureHtmlProcessing,_fnProcessingDisplay,_fnFeatureHtmlTable,_fnScrollDraw,_fnApplyToChildren,_fnCalculateColumnWidths,_fnThrottle,_fnConvertToWidth,_fnGetWidestNode,_fnGetMaxLenString,_fnStringToCss,_fnSortFlatten,_fnSort,_fnSortAria,_fnSortListener,_fnSortAttachListener,_fnSortingClasses,_fnSortData,_fnSaveState,_fnLoadState,_fnSettingsFromNode,_fnLog,_fnMap,_fnBindAction,_fnCallbackReg,_fnCallbackFire,_fnLengthOverflow,_fnRenderer,_fnDataSource,_fnRowAttributes*/

(function( factory ) {
	"use strict";

	if ( typeof define === 'function' && define.amd ) {
		// AMD
		define( ['jquery'], function ( $ ) {
			return factory( $, window, document );
		} );
	}
	else if ( typeof exports === 'object' ) {
		// CommonJS
		module.exports = function (root, $) {
			if ( ! root ) {
				// CommonJS environments without a window global must pass a
				// root. This will give an error otherwise
				root = window;
			}

			if ( ! $ ) {
				$ = typeof window !== 'undefined' ? // jQuery's factory checks for a global window
					require('jquery') :
					require('jquery')( root );
			}

			return factory( $, root, root.document );
		};
	}
	else {
		// Browser
		factory( jQuery, window, document );
	}
}
(function( $, window, document, undefined ) {
	"use strict";

	/**
	 * DataTables is a plug-in for the jQuery Javascript library. It is a highly
	 * flexible tool, based upon the foundations of progressive enhancement,
	 * which will add advanced interaction controls to any HTML table. For a
	 * full list of features please refer to
	 * [DataTables.net](href="http://datatables.net).
	 *
	 * Note that the `DataTable` object is not a global variable but is aliased
	 * to `jQuery.fn.DataTable` and `jQuery.fn.dataTable` through which it may
	 * be  accessed.
	 *
	 *  @class
	 *  @param {object} [init={}] Configuration object for DataTables. Options
	 *    are defined by {@link DataTable.defaults}
	 *  @requires jQuery 1.7+
	 *
	 *  @example
	 *    // Basic initialisation
	 *    $(document).ready( function {
	 *      $('#example').dataTable();
	 *    } );
	 *
	 *  @example
	 *    // Initialisation with configuration options - in this case, disable
	 *    // pagination and sorting.
	 *    $(document).ready( function {
	 *      $('#example').dataTable( {
	 *        "paginate": false,
	 *        "sort": false
	 *      } );
	 *    } );
	 */
	var DataTable = function ( options )
	{
		/**
		 * Perform a jQuery selector action on the table's TR elements (from the tbody) and
		 * return the resulting jQuery object.
		 *  @param {string|node|jQuery} sSelector jQuery selector or node collection to act on
		 *  @param {object} [oOpts] Optional parameters for modifying the rows to be included
		 *  @param {string} [oOpts.filter=none] Select TR elements that meet the current filter
		 *    criterion ("applied") or all TR elements (i.e. no filter).
		 *  @param {string} [oOpts.order=current] Order of the TR elements in the processed array.
		 *    Can be either 'current', whereby the current sorting of the table is used, or
		 *    'original' whereby the original order the data was read into the table is used.
		 *  @param {string} [oOpts.page=all] Limit the selection to the currently displayed page
		 *    ("current") or not ("all"). If 'current' is given, then order is assumed to be
		 *    'current' and filter is 'applied', regardless of what they might be given as.
		 *  @returns {object} jQuery object, filtered by the given selector.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Highlight every second row
		 *      oTable.$('tr:odd').css('backgroundColor', 'blue');
		 *    } );
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Filter to rows with 'Webkit' in them, add a background colour and then
		 *      // remove the filter, thus highlighting the 'Webkit' rows only.
		 *      oTable.fnFilter('Webkit');
		 *      oTable.$('tr', {"search": "applied"}).css('backgroundColor', 'blue');
		 *      oTable.fnFilter('');
		 *    } );
		 */
		this.$ = function ( sSelector, oOpts )
		{
			return this.api(true).$( sSelector, oOpts );
		};
		
		
		/**
		 * Almost identical to $ in operation, but in this case returns the data for the matched
		 * rows - as such, the jQuery selector used should match TR row nodes or TD/TH cell nodes
		 * rather than any descendants, so the data can be obtained for the row/cell. If matching
		 * rows are found, the data returned is the original data array/object that was used to
		 * create the row (or a generated array if from a DOM source).
		 *
		 * This method is often useful in-combination with $ where both functions are given the
		 * same parameters and the array indexes will match identically.
		 *  @param {string|node|jQuery} sSelector jQuery selector or node collection to act on
		 *  @param {object} [oOpts] Optional parameters for modifying the rows to be included
		 *  @param {string} [oOpts.filter=none] Select elements that meet the current filter
		 *    criterion ("applied") or all elements (i.e. no filter).
		 *  @param {string} [oOpts.order=current] Order of the data in the processed array.
		 *    Can be either 'current', whereby the current sorting of the table is used, or
		 *    'original' whereby the original order the data was read into the table is used.
		 *  @param {string} [oOpts.page=all] Limit the selection to the currently displayed page
		 *    ("current") or not ("all"). If 'current' is given, then order is assumed to be
		 *    'current' and filter is 'applied', regardless of what they might be given as.
		 *  @returns {array} Data for the matched elements. If any elements, as a result of the
		 *    selector, were not TR, TD or TH elements in the DataTable, they will have a null
		 *    entry in the array.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Get the data from the first row in the table
		 *      var data = oTable._('tr:first');
		 *
		 *      // Do something useful with the data
		 *      alert( "First cell is: "+data[0] );
		 *    } );
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Filter to 'Webkit' and get all data for
		 *      oTable.fnFilter('Webkit');
		 *      var data = oTable._('tr', {"search": "applied"});
		 *
		 *      // Do something with the data
		 *      alert( data.length+" rows matched the search" );
		 *    } );
		 */
		this._ = function ( sSelector, oOpts )
		{
			return this.api(true).rows( sSelector, oOpts ).data();
		};
		
		
		/**
		 * Create a DataTables Api instance, with the currently selected tables for
		 * the Api's context.
		 * @param {boolean} [traditional=false] Set the API instance's context to be
		 *   only the table referred to by the `DataTable.ext.iApiIndex` option, as was
		 *   used in the API presented by DataTables 1.9- (i.e. the traditional mode),
		 *   or if all tables captured in the jQuery object should be used.
		 * @return {DataTables.Api}
		 */
		this.api = function ( traditional )
		{
			return traditional ?
				new _Api(
					_fnSettingsFromNode( this[ _ext.iApiIndex ] )
				) :
				new _Api( this );
		};
		
		
		/**
		 * Add a single new row or multiple rows of data to the table. Please note
		 * that this is suitable for client-side processing only - if you are using
		 * server-side processing (i.e. "bServerSide": true), then to add data, you
		 * must add it to the data source, i.e. the server-side, through an Ajax call.
		 *  @param {array|object} data The data to be added to the table. This can be:
		 *    <ul>
		 *      <li>1D array of data - add a single row with the data provided</li>
		 *      <li>2D array of arrays - add multiple rows in a single call</li>
		 *      <li>object - data object when using <i>mData</i></li>
		 *      <li>array of objects - multiple data objects when using <i>mData</i></li>
		 *    </ul>
		 *  @param {bool} [redraw=true] redraw the table or not
		 *  @returns {array} An array of integers, representing the list of indexes in
		 *    <i>aoData</i> ({@link DataTable.models.oSettings}) that have been added to
		 *    the table.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    // Global var for counter
		 *    var giCount = 2;
		 *
		 *    $(document).ready(function() {
		 *      $('#example').dataTable();
		 *    } );
		 *
		 *    function fnClickAddRow() {
		 *      $('#example').dataTable().fnAddData( [
		 *        giCount+".1",
		 *        giCount+".2",
		 *        giCount+".3",
		 *        giCount+".4" ]
		 *      );
		 *
		 *      giCount++;
		 *    }
		 */
		this.fnAddData = function( data, redraw )
		{
			var api = this.api( true );
		
			/* Check if we want to add multiple rows or not */
			var rows = Array.isArray(data) && ( Array.isArray(data[0]) || $.isPlainObject(data[0]) ) ?
				api.rows.add( data ) :
				api.row.add( data );
		
			if ( redraw === undefined || redraw ) {
				api.draw();
			}
		
			return rows.flatten().toArray();
		};
		
		
		/**
		 * This function will make DataTables recalculate the column sizes, based on the data
		 * contained in the table and the sizes applied to the columns (in the DOM, CSS or
		 * through the sWidth parameter). This can be useful when the width of the table's
		 * parent element changes (for example a window resize).
		 *  @param {boolean} [bRedraw=true] Redraw the table or not, you will typically want to
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable( {
		 *        "sScrollY": "200px",
		 *        "bPaginate": false
		 *      } );
		 *
		 *      $(window).on('resize', function () {
		 *        oTable.fnAdjustColumnSizing();
		 *      } );
		 *    } );
		 */
		this.fnAdjustColumnSizing = function ( bRedraw )
		{
			var api = this.api( true ).columns.adjust();
			var settings = api.settings()[0];
			var scroll = settings.oScroll;
		
			if ( bRedraw === undefined || bRedraw ) {
				api.draw( false );
			}
			else if ( scroll.sX !== "" || scroll.sY !== "" ) {
				/* If not redrawing, but scrolling, we want to apply the new column sizes anyway */
				_fnScrollDraw( settings );
			}
		};
		
		
		/**
		 * Quickly and simply clear a table
		 *  @param {bool} [bRedraw=true] redraw the table or not
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Immediately 'nuke' the current rows (perhaps waiting for an Ajax callback...)
		 *      oTable.fnClearTable();
		 *    } );
		 */
		this.fnClearTable = function( bRedraw )
		{
			var api = this.api( true ).clear();
		
			if ( bRedraw === undefined || bRedraw ) {
				api.draw();
			}
		};
		
		
		/**
		 * The exact opposite of 'opening' a row, this function will close any rows which
		 * are currently 'open'.
		 *  @param {node} nTr the table row to 'close'
		 *  @returns {int} 0 on success, or 1 if failed (can't find the row)
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable;
		 *
		 *      // 'open' an information row when a row is clicked on
		 *      $('#example tbody tr').click( function () {
		 *        if ( oTable.fnIsOpen(this) ) {
		 *          oTable.fnClose( this );
		 *        } else {
		 *          oTable.fnOpen( this, "Temporary row opened", "info_row" );
		 *        }
		 *      } );
		 *
		 *      oTable = $('#example').dataTable();
		 *    } );
		 */
		this.fnClose = function( nTr )
		{
			this.api( true ).row( nTr ).child.hide();
		};
		
		
		/**
		 * Remove a row for the table
		 *  @param {mixed} target The index of the row from aoData to be deleted, or
		 *    the TR element you want to delete
		 *  @param {function|null} [callBack] Callback function
		 *  @param {bool} [redraw=true] Redraw the table or not
		 *  @returns {array} The row that was deleted
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Immediately remove the first row
		 *      oTable.fnDeleteRow( 0 );
		 *    } );
		 */
		this.fnDeleteRow = function( target, callback, redraw )
		{
			var api = this.api( true );
			var rows = api.rows( target );
			var settings = rows.settings()[0];
			var data = settings.aoData[ rows[0][0] ];
		
			rows.remove();
		
			if ( callback ) {
				callback.call( this, settings, data );
			}
		
			if ( redraw === undefined || redraw ) {
				api.draw();
			}
		
			return data;
		};
		
		
		/**
		 * Restore the table to it's original state in the DOM by removing all of DataTables
		 * enhancements, alterations to the DOM structure of the table and event listeners.
		 *  @param {boolean} [remove=false] Completely remove the table from the DOM
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      // This example is fairly pointless in reality, but shows how fnDestroy can be used
		 *      var oTable = $('#example').dataTable();
		 *      oTable.fnDestroy();
		 *    } );
		 */
		this.fnDestroy = function ( remove )
		{
			this.api( true ).destroy( remove );
		};
		
		
		/**
		 * Redraw the table
		 *  @param {bool} [complete=true] Re-filter and resort (if enabled) the table before the draw.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Re-draw the table - you wouldn't want to do it here, but it's an example :-)
		 *      oTable.fnDraw();
		 *    } );
		 */
		this.fnDraw = function( complete )
		{
			// Note that this isn't an exact match to the old call to _fnDraw - it takes
			// into account the new data, but can hold position.
			this.api( true ).draw( complete );
		};
		
		
		/**
		 * Filter the input based on data
		 *  @param {string} sInput String to filter the table on
		 *  @param {int|null} [iColumn] Column to limit filtering to
		 *  @param {bool} [bRegex=false] Treat as regular expression or not
		 *  @param {bool} [bSmart=true] Perform smart filtering or not
		 *  @param {bool} [bShowGlobal=true] Show the input global filter in it's input box(es)
		 *  @param {bool} [bCaseInsensitive=true] Do case-insensitive matching (true) or not (false)
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Sometime later - filter...
		 *      oTable.fnFilter( 'test string' );
		 *    } );
		 */
		this.fnFilter = function( sInput, iColumn, bRegex, bSmart, bShowGlobal, bCaseInsensitive )
		{
			var api = this.api( true );
		
			if ( iColumn === null || iColumn === undefined ) {
				api.search( sInput, bRegex, bSmart, bCaseInsensitive );
			}
			else {
				api.column( iColumn ).search( sInput, bRegex, bSmart, bCaseInsensitive );
			}
		
			api.draw();
		};
		
		
		/**
		 * Get the data for the whole table, an individual row or an individual cell based on the
		 * provided parameters.
		 *  @param {int|node} [src] A TR row node, TD/TH cell node or an integer. If given as
		 *    a TR node then the data source for the whole row will be returned. If given as a
		 *    TD/TH cell node then iCol will be automatically calculated and the data for the
		 *    cell returned. If given as an integer, then this is treated as the aoData internal
		 *    data index for the row (see fnGetPosition) and the data for that row used.
		 *  @param {int} [col] Optional column index that you want the data of.
		 *  @returns {array|object|string} If mRow is undefined, then the data for all rows is
		 *    returned. If mRow is defined, just data for that row, and is iCol is
		 *    defined, only data for the designated cell is returned.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    // Row data
		 *    $(document).ready(function() {
		 *      oTable = $('#example').dataTable();
		 *
		 *      oTable.$('tr').click( function () {
		 *        var data = oTable.fnGetData( this );
		 *        // ... do something with the array / object of data for the row
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Individual cell data
		 *    $(document).ready(function() {
		 *      oTable = $('#example').dataTable();
		 *
		 *      oTable.$('td').click( function () {
		 *        var sData = oTable.fnGetData( this );
		 *        alert( 'The cell clicked on had the value of '+sData );
		 *      } );
		 *    } );
		 */
		this.fnGetData = function( src, col )
		{
			var api = this.api( true );
		
			if ( src !== undefined ) {
				var type = src.nodeName ? src.nodeName.toLowerCase() : '';
		
				return col !== undefined || type == 'td' || type == 'th' ?
					api.cell( src, col ).data() :
					api.row( src ).data() || null;
			}
		
			return api.data().toArray();
		};
		
		
		/**
		 * Get an array of the TR nodes that are used in the table's body. Note that you will
		 * typically want to use the '$' API method in preference to this as it is more
		 * flexible.
		 *  @param {int} [iRow] Optional row index for the TR element you want
		 *  @returns {array|node} If iRow is undefined, returns an array of all TR elements
		 *    in the table's body, or iRow is defined, just the TR element requested.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Get the nodes from the table
		 *      var nNodes = oTable.fnGetNodes( );
		 *    } );
		 */
		this.fnGetNodes = function( iRow )
		{
			var api = this.api( true );
		
			return iRow !== undefined ?
				api.row( iRow ).node() :
				api.rows().nodes().flatten().toArray();
		};
		
		
		/**
		 * Get the array indexes of a particular cell from it's DOM element
		 * and column index including hidden columns
		 *  @param {node} node this can either be a TR, TD or TH in the table's body
		 *  @returns {int} If nNode is given as a TR, then a single index is returned, or
		 *    if given as a cell, an array of [row index, column index (visible),
		 *    column index (all)] is given.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      $('#example tbody td').click( function () {
		 *        // Get the position of the current data from the node
		 *        var aPos = oTable.fnGetPosition( this );
		 *
		 *        // Get the data array for this row
		 *        var aData = oTable.fnGetData( aPos[0] );
		 *
		 *        // Update the data array and return the value
		 *        aData[ aPos[1] ] = 'clicked';
		 *        this.innerHTML = 'clicked';
		 *      } );
		 *
		 *      // Init DataTables
		 *      oTable = $('#example').dataTable();
		 *    } );
		 */
		this.fnGetPosition = function( node )
		{
			var api = this.api( true );
			var nodeName = node.nodeName.toUpperCase();
		
			if ( nodeName == 'TR' ) {
				return api.row( node ).index();
			}
			else if ( nodeName == 'TD' || nodeName == 'TH' ) {
				var cell = api.cell( node ).index();
		
				return [
					cell.row,
					cell.columnVisible,
					cell.column
				];
			}
			return null;
		};
		
		
		/**
		 * Check to see if a row is 'open' or not.
		 *  @param {node} nTr the table row to check
		 *  @returns {boolean} true if the row is currently open, false otherwise
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable;
		 *
		 *      // 'open' an information row when a row is clicked on
		 *      $('#example tbody tr').click( function () {
		 *        if ( oTable.fnIsOpen(this) ) {
		 *          oTable.fnClose( this );
		 *        } else {
		 *          oTable.fnOpen( this, "Temporary row opened", "info_row" );
		 *        }
		 *      } );
		 *
		 *      oTable = $('#example').dataTable();
		 *    } );
		 */
		this.fnIsOpen = function( nTr )
		{
			return this.api( true ).row( nTr ).child.isShown();
		};
		
		
		/**
		 * This function will place a new row directly after a row which is currently
		 * on display on the page, with the HTML contents that is passed into the
		 * function. This can be used, for example, to ask for confirmation that a
		 * particular record should be deleted.
		 *  @param {node} nTr The table row to 'open'
		 *  @param {string|node|jQuery} mHtml The HTML to put into the row
		 *  @param {string} sClass Class to give the new TD cell
		 *  @returns {node} The row opened. Note that if the table row passed in as the
		 *    first parameter, is not found in the table, this method will silently
		 *    return.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable;
		 *
		 *      // 'open' an information row when a row is clicked on
		 *      $('#example tbody tr').click( function () {
		 *        if ( oTable.fnIsOpen(this) ) {
		 *          oTable.fnClose( this );
		 *        } else {
		 *          oTable.fnOpen( this, "Temporary row opened", "info_row" );
		 *        }
		 *      } );
		 *
		 *      oTable = $('#example').dataTable();
		 *    } );
		 */
		this.fnOpen = function( nTr, mHtml, sClass )
		{
			return this.api( true )
				.row( nTr )
				.child( mHtml, sClass )
				.show()
				.child()[0];
		};
		
		
		/**
		 * Change the pagination - provides the internal logic for pagination in a simple API
		 * function. With this function you can have a DataTables table go to the next,
		 * previous, first or last pages.
		 *  @param {string|int} mAction Paging action to take: "first", "previous", "next" or "last"
		 *    or page number to jump to (integer), note that page 0 is the first page.
		 *  @param {bool} [bRedraw=true] Redraw the table or not
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *      oTable.fnPageChange( 'next' );
		 *    } );
		 */
		this.fnPageChange = function ( mAction, bRedraw )
		{
			var api = this.api( true ).page( mAction );
		
			if ( bRedraw === undefined || bRedraw ) {
				api.draw(false);
			}
		};
		
		
		/**
		 * Show a particular column
		 *  @param {int} iCol The column whose display should be changed
		 *  @param {bool} bShow Show (true) or hide (false) the column
		 *  @param {bool} [bRedraw=true] Redraw the table or not
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Hide the second column after initialisation
		 *      oTable.fnSetColumnVis( 1, false );
		 *    } );
		 */
		this.fnSetColumnVis = function ( iCol, bShow, bRedraw )
		{
			var api = this.api( true ).column( iCol ).visible( bShow );
		
			if ( bRedraw === undefined || bRedraw ) {
				api.columns.adjust().draw();
			}
		};
		
		
		/**
		 * Get the settings for a particular table for external manipulation
		 *  @returns {object} DataTables settings object. See
		 *    {@link DataTable.models.oSettings}
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *      var oSettings = oTable.fnSettings();
		 *
		 *      // Show an example parameter from the settings
		 *      alert( oSettings._iDisplayStart );
		 *    } );
		 */
		this.fnSettings = function()
		{
			return _fnSettingsFromNode( this[_ext.iApiIndex] );
		};
		
		
		/**
		 * Sort the table by a particular column
		 *  @param {int} iCol the data index to sort on. Note that this will not match the
		 *    'display index' if you have hidden data entries
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Sort immediately with columns 0 and 1
		 *      oTable.fnSort( [ [0,'asc'], [1,'asc'] ] );
		 *    } );
		 */
		this.fnSort = function( aaSort )
		{
			this.api( true ).order( aaSort ).draw();
		};
		
		
		/**
		 * Attach a sort listener to an element for a given column
		 *  @param {node} nNode the element to attach the sort listener to
		 *  @param {int} iColumn the column that a click on this node will sort on
		 *  @param {function} [fnCallback] callback function when sort is run
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Sort on column 1, when 'sorter' is clicked on
		 *      oTable.fnSortListener( document.getElementById('sorter'), 1 );
		 *    } );
		 */
		this.fnSortListener = function( nNode, iColumn, fnCallback )
		{
			this.api( true ).order.listener( nNode, iColumn, fnCallback );
		};
		
		
		/**
		 * Update a table cell or row - this method will accept either a single value to
		 * update the cell with, an array of values with one element for each column or
		 * an object in the same format as the original data source. The function is
		 * self-referencing in order to make the multi column updates easier.
		 *  @param {object|array|string} mData Data to update the cell/row with
		 *  @param {node|int} mRow TR element you want to update or the aoData index
		 *  @param {int} [iColumn] The column to update, give as null or undefined to
		 *    update a whole row.
		 *  @param {bool} [bRedraw=true] Redraw the table or not
		 *  @param {bool} [bAction=true] Perform pre-draw actions or not
		 *  @returns {int} 0 on success, 1 on error
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *      oTable.fnUpdate( 'Example update', 0, 0 ); // Single cell
		 *      oTable.fnUpdate( ['a', 'b', 'c', 'd', 'e'], $('tbody tr')[0] ); // Row
		 *    } );
		 */
		this.fnUpdate = function( mData, mRow, iColumn, bRedraw, bAction )
		{
			var api = this.api( true );
		
			if ( iColumn === undefined || iColumn === null ) {
				api.row( mRow ).data( mData );
			}
			else {
				api.cell( mRow, iColumn ).data( mData );
			}
		
			if ( bAction === undefined || bAction ) {
				api.columns.adjust();
			}
		
			if ( bRedraw === undefined || bRedraw ) {
				api.draw();
			}
			return 0;
		};
		
		
		/**
		 * Provide a common method for plug-ins to check the version of DataTables being used, in order
		 * to ensure compatibility.
		 *  @param {string} sVersion Version string to check for, in the format "X.Y.Z". Note that the
		 *    formats "X" and "X.Y" are also acceptable.
		 *  @returns {boolean} true if this version of DataTables is greater or equal to the required
		 *    version, or false if this version of DataTales is not suitable
		 *  @method
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *      alert( oTable.fnVersionCheck( '1.9.0' ) );
		 *    } );
		 */
		this.fnVersionCheck = _ext.fnVersionCheck;
		

		var _that = this;
		var emptyInit = options === undefined;
		var len = this.length;

		if ( emptyInit ) {
			options = {};
		}

		this.oApi = this.internal = _ext.internal;

		// Extend with old style plug-in API methods
		for ( var fn in DataTable.ext.internal ) {
			if ( fn ) {
				this[fn] = _fnExternApiFunc(fn);
			}
		}

		this.each(function() {
			// For each initialisation we want to give it a clean initialisation
			// object that can be bashed around
			var o = {};
			var oInit = len > 1 ? // optimisation for single table case
				_fnExtend( o, options, true ) :
				options;

			/*global oInit,_that,emptyInit*/
			var i=0, iLen, j, jLen, k, kLen;
			var sId = this.getAttribute( 'id' );
			var bInitHandedOff = false;
			var defaults = DataTable.defaults;
			var $this = $(this);
			
			
			/* Sanity check */
			if ( this.nodeName.toLowerCase() != 'table' )
			{
				_fnLog( null, 0, 'Non-table node initialisation ('+this.nodeName+')', 2 );
				return;
			}
			
			/* Backwards compatibility for the defaults */
			_fnCompatOpts( defaults );
			_fnCompatCols( defaults.column );
			
			/* Convert the camel-case defaults to Hungarian */
			_fnCamelToHungarian( defaults, defaults, true );
			_fnCamelToHungarian( defaults.column, defaults.column, true );
			
			/* Setting up the initialisation object */
			_fnCamelToHungarian( defaults, $.extend( oInit, $this.data() ), true );
			
			
			
			/* Check to see if we are re-initialising a table */
			var allSettings = DataTable.settings;
			for ( i=0, iLen=allSettings.length ; i<iLen ; i++ )
			{
				var s = allSettings[i];
			
				/* Base check on table node */
				if (
					s.nTable == this ||
					(s.nTHead && s.nTHead.parentNode == this) ||
					(s.nTFoot && s.nTFoot.parentNode == this)
				) {
					var bRetrieve = oInit.bRetrieve !== undefined ? oInit.bRetrieve : defaults.bRetrieve;
					var bDestroy = oInit.bDestroy !== undefined ? oInit.bDestroy : defaults.bDestroy;
			
					if ( emptyInit || bRetrieve )
					{
						return s.oInstance;
					}
					else if ( bDestroy )
					{
						s.oInstance.fnDestroy();
						break;
					}
					else
					{
						_fnLog( s, 0, 'Cannot reinitialise DataTable', 3 );
						return;
					}
				}
			
				/* If the element we are initialising has the same ID as a table which was previously
				 * initialised, but the table nodes don't match (from before) then we destroy the old
				 * instance by simply deleting it. This is under the assumption that the table has been
				 * destroyed by other methods. Anyone using non-id selectors will need to do this manually
				 */
				if ( s.sTableId == this.id )
				{
					allSettings.splice( i, 1 );
					break;
				}
			}
			
			/* Ensure the table has an ID - required for accessibility */
			if ( sId === null || sId === "" )
			{
				sId = "DataTables_Table_"+(DataTable.ext._unique++);
				this.id = sId;
			}
			
			/* Create the settings object for this table and set some of the default parameters */
			var oSettings = $.extend( true, {}, DataTable.models.oSettings, {
				"sDestroyWidth": $this[0].style.width,
				"sInstance":     sId,
				"sTableId":      sId
			} );
			oSettings.nTable = this;
			oSettings.oApi   = _that.internal;
			oSettings.oInit  = oInit;
			
			allSettings.push( oSettings );
			
			// Need to add the instance after the instance after the settings object has been added
			// to the settings array, so we can self reference the table instance if more than one
			oSettings.oInstance = (_that.length===1) ? _that : $this.dataTable();
			
			// Backwards compatibility, before we apply all the defaults
			_fnCompatOpts( oInit );
			_fnLanguageCompat( oInit.oLanguage );
			
			// If the length menu is given, but the init display length is not, use the length menu
			if ( oInit.aLengthMenu && ! oInit.iDisplayLength )
			{
				oInit.iDisplayLength = Array.isArray( oInit.aLengthMenu[0] ) ?
					oInit.aLengthMenu[0][0] : oInit.aLengthMenu[0];
			}
			
			// Apply the defaults and init options to make a single init object will all
			// options defined from defaults and instance options.
			oInit = _fnExtend( $.extend( true, {}, defaults ), oInit );
			
			
			// Map the initialisation options onto the settings object
			_fnMap( oSettings.oFeatures, oInit, [
				"bPaginate",
				"bLengthChange",
				"bFilter",
				"bSort",
				"bSortMulti",
				"bInfo",
				"bProcessing",
				"bAutoWidth",
				"bSortClasses",
				"bServerSide",
				"bDeferRender"
			] );
			_fnMap( oSettings, oInit, [
				"asStripeClasses",
				"ajax",
				"fnServerData",
				"fnFormatNumber",
				"sServerMethod",
				"aaSorting",
				"aaSortingFixed",
				"aLengthMenu",
				"sPaginationType",
				"sAjaxSource",
				"sAjaxDataProp",
				"iStateDuration",
				"sDom",
				"bSortCellsTop",
				"iTabIndex",
				"fnStateLoadCallback",
				"fnStateSaveCallback",
				"renderer",
				"searchDelay",
				"rowId",
				[ "iCookieDuration", "iStateDuration" ], // backwards compat
				[ "oSearch", "oPreviousSearch" ],
				[ "aoSearchCols", "aoPreSearchCols" ],
				[ "iDisplayLength", "_iDisplayLength" ]
			] );
			_fnMap( oSettings.oScroll, oInit, [
				[ "sScrollX", "sX" ],
				[ "sScrollXInner", "sXInner" ],
				[ "sScrollY", "sY" ],
				[ "bScrollCollapse", "bCollapse" ]
			] );
			_fnMap( oSettings.oLanguage, oInit, "fnInfoCallback" );
			
			/* Callback functions which are array driven */
			_fnCallbackReg( oSettings, 'aoDrawCallback',       oInit.fnDrawCallback,      'user' );
			_fnCallbackReg( oSettings, 'aoServerParams',       oInit.fnServerParams,      'user' );
			_fnCallbackReg( oSettings, 'aoStateSaveParams',    oInit.fnStateSaveParams,   'user' );
			_fnCallbackReg( oSettings, 'aoStateLoadParams',    oInit.fnStateLoadParams,   'user' );
			_fnCallbackReg( oSettings, 'aoStateLoaded',        oInit.fnStateLoaded,       'user' );
			_fnCallbackReg( oSettings, 'aoRowCallback',        oInit.fnRowCallback,       'user' );
			_fnCallbackReg( oSettings, 'aoRowCreatedCallback', oInit.fnCreatedRow,        'user' );
			_fnCallbackReg( oSettings, 'aoHeaderCallback',     oInit.fnHeaderCallback,    'user' );
			_fnCallbackReg( oSettings, 'aoFooterCallback',     oInit.fnFooterCallback,    'user' );
			_fnCallbackReg( oSettings, 'aoInitComplete',       oInit.fnInitComplete,      'user' );
			_fnCallbackReg( oSettings, 'aoPreDrawCallback',    oInit.fnPreDrawCallback,   'user' );
			
			oSettings.rowIdFn = _fnGetObjectDataFn( oInit.rowId );
			
			/* Browser support detection */
			_fnBrowserDetect( oSettings );
			
			var oClasses = oSettings.oClasses;
			
			$.extend( oClasses, DataTable.ext.classes, oInit.oClasses );
			$this.addClass( oClasses.sTable );
			
			
			if ( oSettings.iInitDisplayStart === undefined )
			{
				/* Display start point, taking into account the save saving */
				oSettings.iInitDisplayStart = oInit.iDisplayStart;
				oSettings._iDisplayStart = oInit.iDisplayStart;
			}
			
			if ( oInit.iDeferLoading !== null )
			{
				oSettings.bDeferLoading = true;
				var tmp = Array.isArray( oInit.iDeferLoading );
				oSettings._iRecordsDisplay = tmp ? oInit.iDeferLoading[0] : oInit.iDeferLoading;
				oSettings._iRecordsTotal = tmp ? oInit.iDeferLoading[1] : oInit.iDeferLoading;
			}
			
			/* Language definitions */
			var oLanguage = oSettings.oLanguage;
			$.extend( true, oLanguage, oInit.oLanguage );
			
			if ( oLanguage.sUrl )
			{
				/* Get the language definitions from a file - because this Ajax call makes the language
				 * get async to the remainder of this function we use bInitHandedOff to indicate that
				 * _fnInitialise will be fired by the returned Ajax handler, rather than the constructor
				 */
				$.ajax( {
					dataType: 'json',
					url: oLanguage.sUrl,
					success: function ( json ) {
						_fnLanguageCompat( json );
						_fnCamelToHungarian( defaults.oLanguage, json );
						$.extend( true, oLanguage, json );
						_fnInitialise( oSettings );
					},
					error: function () {
						// Error occurred loading language file, continue on as best we can
						_fnInitialise( oSettings );
					}
				} );
				bInitHandedOff = true;
			}
			
			/*
			 * Stripes
			 */
			if ( oInit.asStripeClasses === null )
			{
				oSettings.asStripeClasses =[
					oClasses.sStripeOdd,
					oClasses.sStripeEven
				];
			}
			
			/* Remove row stripe classes if they are already on the table row */
			var stripeClasses = oSettings.asStripeClasses;
			var rowOne = $this.children('tbody').find('tr').eq(0);
			if ( $.inArray( true, $.map( stripeClasses, function(el, i) {
				return rowOne.hasClass(el);
			} ) ) !== -1 ) {
				$('tbody tr', this).removeClass( stripeClasses.join(' ') );
				oSettings.asDestroyStripes = stripeClasses.slice();
			}
			
			/*
			 * Columns
			 * See if we should load columns automatically or use defined ones
			 */
			var anThs = [];
			var aoColumnsInit;
			var nThead = this.getElementsByTagName('thead');
			if ( nThead.length !== 0 )
			{
				_fnDetectHeader( oSettings.aoHeader, nThead[0] );
				anThs = _fnGetUniqueThs( oSettings );
			}
			
			/* If not given a column array, generate one with nulls */
			if ( oInit.aoColumns === null )
			{
				aoColumnsInit = [];
				for ( i=0, iLen=anThs.length ; i<iLen ; i++ )
				{
					aoColumnsInit.push( null );
				}
			}
			else
			{
				aoColumnsInit = oInit.aoColumns;
			}
			
			/* Add the columns */
			for ( i=0, iLen=aoColumnsInit.length ; i<iLen ; i++ )
			{
				_fnAddColumn( oSettings, anThs ? anThs[i] : null );
			}
			
			/* Apply the column definitions */
			_fnApplyColumnDefs( oSettings, oInit.aoColumnDefs, aoColumnsInit, function (iCol, oDef) {
				_fnColumnOptions( oSettings, iCol, oDef );
			} );
			
			/* HTML5 attribute detection - build an mData object automatically if the
			 * attributes are found
			 */
			if ( rowOne.length ) {
				var a = function ( cell, name ) {
					return cell.getAttribute( 'data-'+name ) !== null ? name : null;
				};
			
				$( rowOne[0] ).children('th, td').each( function (i, cell) {
					var col = oSettings.aoColumns[i];
			
					if ( col.mData === i ) {
						var sort = a( cell, 'sort' ) || a( cell, 'order' );
						var filter = a( cell, 'filter' ) || a( cell, 'search' );
			
						if ( sort !== null || filter !== null ) {
							col.mData = {
								_:      i+'.display',
								sort:   sort !== null   ? i+'.@data-'+sort   : undefined,
								type:   sort !== null   ? i+'.@data-'+sort   : undefined,
								filter: filter !== null ? i+'.@data-'+filter : undefined
							};
			
							_fnColumnOptions( oSettings, i );
						}
					}
				} );
			}
			
			var features = oSettings.oFeatures;
			var loadedInit = function () {
				/*
				 * Sorting
				 * @todo For modularisation (1.11) this needs to do into a sort start up handler
				 */
			
				// If aaSorting is not defined, then we use the first indicator in asSorting
				// in case that has been altered, so the default sort reflects that option
				if ( oInit.aaSorting === undefined ) {
					var sorting = oSettings.aaSorting;
					for ( i=0, iLen=sorting.length ; i<iLen ; i++ ) {
						sorting[i][1] = oSettings.aoColumns[ i ].asSorting[0];
					}
				}
			
				/* Do a first pass on the sorting classes (allows any size changes to be taken into
				 * account, and also will apply sorting disabled classes if disabled
				 */
				_fnSortingClasses( oSettings );
			
				if ( features.bSort ) {
					_fnCallbackReg( oSettings, 'aoDrawCallback', function () {
						if ( oSettings.bSorted ) {
							var aSort = _fnSortFlatten( oSettings );
							var sortedColumns = {};
			
							$.each( aSort, function (i, val) {
								sortedColumns[ val.src ] = val.dir;
							} );
			
							_fnCallbackFire( oSettings, null, 'order', [oSettings, aSort, sortedColumns] );
							_fnSortAria( oSettings );
						}
					} );
				}
			
				_fnCallbackReg( oSettings, 'aoDrawCallback', function () {
					if ( oSettings.bSorted || _fnDataSource( oSettings ) === 'ssp' || features.bDeferRender ) {
						_fnSortingClasses( oSettings );
					}
				}, 'sc' );
			
			
				/*
				 * Final init
				 * Cache the header, body and footer as required, creating them if needed
				 */
			
				// Work around for Webkit bug 83867 - store the caption-side before removing from doc
				var captions = $this.children('caption').each( function () {
					this._captionSide = $(this).css('caption-side');
				} );
			
				var thead = $this.children('thead');
				if ( thead.length === 0 ) {
					thead = $('<thead/>').appendTo($this);
				}
				oSettings.nTHead = thead[0];
			
				var tbody = $this.children('tbody');
				if ( tbody.length === 0 ) {
					tbody = $('<tbody/>').appendTo($this);
				}
				oSettings.nTBody = tbody[0];
			
				var tfoot = $this.children('tfoot');
				if ( tfoot.length === 0 && captions.length > 0 && (oSettings.oScroll.sX !== "" || oSettings.oScroll.sY !== "") ) {
					// If we are a scrolling table, and no footer has been given, then we need to create
					// a tfoot element for the caption element to be appended to
					tfoot = $('<tfoot/>').appendTo($this);
				}
			
				if ( tfoot.length === 0 || tfoot.children().length === 0 ) {
					$this.addClass( oClasses.sNoFooter );
				}
				else if ( tfoot.length > 0 ) {
					oSettings.nTFoot = tfoot[0];
					_fnDetectHeader( oSettings.aoFooter, oSettings.nTFoot );
				}
			
				/* Check if there is data passing into the constructor */
				if ( oInit.aaData ) {
					for ( i=0 ; i<oInit.aaData.length ; i++ ) {
						_fnAddData( oSettings, oInit.aaData[ i ] );
					}
				}
				else if ( oSettings.bDeferLoading || _fnDataSource( oSettings ) == 'dom' ) {
					/* Grab the data from the page - only do this when deferred loading or no Ajax
					 * source since there is no point in reading the DOM data if we are then going
					 * to replace it with Ajax data
					 */
					_fnAddTr( oSettings, $(oSettings.nTBody).children('tr') );
				}
			
				/* Copy the data index array */
				oSettings.aiDisplay = oSettings.aiDisplayMaster.slice();
			
				/* Initialisation complete - table can be drawn */
				oSettings.bInitialised = true;
			
				/* Check if we need to initialise the table (it might not have been handed off to the
				 * language processor)
				 */
				if ( bInitHandedOff === false ) {
					_fnInitialise( oSettings );
				}
			};
			
			/* Must be done after everything which can be overridden by the state saving! */
			if ( oInit.bStateSave )
			{
				features.bStateSave = true;
				_fnCallbackReg( oSettings, 'aoDrawCallback', _fnSaveState, 'state_save' );
				_fnLoadState( oSettings, oInit, loadedInit );
			}
			else {
				loadedInit();
			}
			
		} );
		_that = null;
		return this;
	};

	
	/*
	 * It is useful to have variables which are scoped locally so only the
	 * DataTables functions can access them and they don't leak into global space.
	 * At the same time these functions are often useful over multiple files in the
	 * core and API, so we list, or at least document, all variables which are used
	 * by DataTables as private variables here. This also ensures that there is no
	 * clashing of variable names and that they can easily referenced for reuse.
	 */
	
	
	// Defined else where
	//  _selector_run
	//  _selector_opts
	//  _selector_first
	//  _selector_row_indexes
	
	var _ext; // DataTable.ext
	var _Api; // DataTable.Api
	var _api_register; // DataTable.Api.register
	var _api_registerPlural; // DataTable.Api.registerPlural
	
	var _re_dic = {};
	var _re_new_lines = /[\r\n\u2028]/g;
	var _re_html = /<.*?>/g;
	
	// This is not strict ISO8601 - Date.parse() is quite lax, although
	// implementations differ between browsers.
	var _re_date = /^\d{2,4}[\.\/\-]\d{1,2}[\.\/\-]\d{1,2}([T ]{1}\d{1,2}[:\.]\d{2}([\.:]\d{2})?)?$/;
	
	// Escape regular expression special characters
	var _re_escape_regex = new RegExp( '(\\' + [ '/', '.', '*', '+', '?', '|', '(', ')', '[', ']', '{', '}', '\\', '$', '^', '-' ].join('|\\') + ')', 'g' );
	
	// http://en.wikipedia.org/wiki/Foreign_exchange_market
	// - \u20BD - Russian ruble.
	// - \u20a9 - South Korean Won
	// - \u20BA - Turkish Lira
	// - \u20B9 - Indian Rupee
	// - R - Brazil (R$) and South Africa
	// - fr - Swiss Franc
	// - kr - Swedish krona, Norwegian krone and Danish krone
	// - \u2009 is thin space and \u202F is narrow no-break space, both used in many
	// - Ƀ - Bitcoin
	// - Ξ - Ethereum
	//   standards as thousands separators.
	var _re_formatted_numeric = /['\u00A0,$£€¥%\u2009\u202F\u20BD\u20a9\u20BArfkɃΞ]/gi;
	
	
	var _empty = function ( d ) {
		return !d || d === true || d === '-' ? true : false;
	};
	
	
	var _intVal = function ( s ) {
		var integer = parseInt( s, 10 );
		return !isNaN(integer) && isFinite(s) ? integer : null;
	};
	
	// Convert from a formatted number with characters other than `.` as the
	// decimal place, to a Javascript number
	var _numToDecimal = function ( num, decimalPoint ) {
		// Cache created regular expressions for speed as this function is called often
		if ( ! _re_dic[ decimalPoint ] ) {
			_re_dic[ decimalPoint ] = new RegExp( _fnEscapeRegex( decimalPoint ), 'g' );
		}
		return typeof num === 'string' && decimalPoint !== '.' ?
			num.replace( /\./g, '' ).replace( _re_dic[ decimalPoint ], '.' ) :
			num;
	};
	
	
	var _isNumber = function ( d, decimalPoint, formatted ) {
		var strType = typeof d === 'string';
	
		// If empty return immediately so there must be a number if it is a
		// formatted string (this stops the string "k", or "kr", etc being detected
		// as a formatted number for currency
		if ( _empty( d ) ) {
			return true;
		}
	
		if ( decimalPoint && strType ) {
			d = _numToDecimal( d, decimalPoint );
		}
	
		if ( formatted && strType ) {
			d = d.replace( _re_formatted_numeric, '' );
		}
	
		return !isNaN( parseFloat(d) ) && isFinite( d );
	};
	
	
	// A string without HTML in it can be considered to be HTML still
	var _isHtml = function ( d ) {
		return _empty( d ) || typeof d === 'string';
	};
	
	
	var _htmlNumeric = function ( d, decimalPoint, formatted ) {
		if ( _empty( d ) ) {
			return true;
		}
	
		var html = _isHtml( d );
		return ! html ?
			null :
			_isNumber( _stripHtml( d ), decimalPoint, formatted ) ?
				true :
				null;
	};
	
	
	var _pluck = function ( a, prop, prop2 ) {
		var out = [];
		var i=0, ien=a.length;
	
		// Could have the test in the loop for slightly smaller code, but speed
		// is essential here
		if ( prop2 !== undefined ) {
			for ( ; i<ien ; i++ ) {
				if ( a[i] && a[i][ prop ] ) {
					out.push( a[i][ prop ][ prop2 ] );
				}
			}
		}
		else {
			for ( ; i<ien ; i++ ) {
				if ( a[i] ) {
					out.push( a[i][ prop ] );
				}
			}
		}
	
		return out;
	};
	
	
	// Basically the same as _pluck, but rather than looping over `a` we use `order`
	// as the indexes to pick from `a`
	var _pluck_order = function ( a, order, prop, prop2 )
	{
		var out = [];
		var i=0, ien=order.length;
	
		// Could have the test in the loop for slightly smaller code, but speed
		// is essential here
		if ( prop2 !== undefined ) {
			for ( ; i<ien ; i++ ) {
				if ( a[ order[i] ][ prop ] ) {
					out.push( a[ order[i] ][ prop ][ prop2 ] );
				}
			}
		}
		else {
			for ( ; i<ien ; i++ ) {
				out.push( a[ order[i] ][ prop ] );
			}
		}
	
		return out;
	};
	
	
	var _range = function ( len, start )
	{
		var out = [];
		var end;
	
		if ( start === undefined ) {
			start = 0;
			end = len;
		}
		else {
			end = start;
			start = len;
		}
	
		for ( var i=start ; i<end ; i++ ) {
			out.push( i );
		}
	
		return out;
	};
	
	
	var _removeEmpty = function ( a )
	{
		var out = [];
	
		for ( var i=0, ien=a.length ; i<ien ; i++ ) {
			if ( a[i] ) { // careful - will remove all falsy values!
				out.push( a[i] );
			}
		}
	
		return out;
	};
	
	
	var _stripHtml = function ( d ) {
		return d.replace( _re_html, '' );
	};
	
	
	/**
	 * Determine if all values in the array are unique. This means we can short
	 * cut the _unique method at the cost of a single loop. A sorted array is used
	 * to easily check the values.
	 *
	 * @param  {array} src Source array
	 * @return {boolean} true if all unique, false otherwise
	 * @ignore
	 */
	var _areAllUnique = function ( src ) {
		if ( src.length < 2 ) {
			return true;
		}
	
		var sorted = src.slice().sort();
		var last = sorted[0];
	
		for ( var i=1, ien=sorted.length ; i<ien ; i++ ) {
			if ( sorted[i] === last ) {
				return false;
			}
	
			last = sorted[i];
		}
	
		return true;
	};
	
	
	/**
	 * Find the unique elements in a source array.
	 *
	 * @param  {array} src Source array
	 * @return {array} Array of unique items
	 * @ignore
	 */
	var _unique = function ( src )
	{
		if ( _areAllUnique( src ) ) {
			return src.slice();
		}
	
		// A faster unique method is to use object keys to identify used values,
		// but this doesn't work with arrays or objects, which we must also
		// consider. See jsperf.com/compare-array-unique-versions/4 for more
		// information.
		var
			out = [],
			val,
			i, ien=src.length,
			j, k=0;
	
		again: for ( i=0 ; i<ien ; i++ ) {
			val = src[i];
	
			for ( j=0 ; j<k ; j++ ) {
				if ( out[j] === val ) {
					continue again;
				}
			}
	
			out.push( val );
			k++;
		}
	
		return out;
	};
	
	// Surprisingly this is faster than [].concat.apply
	// https://jsperf.com/flatten-an-array-loop-vs-reduce/2
	var _flatten = function (out, val) {
		if (Array.isArray(val)) {
			for (var i=0 ; i<val.length ; i++) {
				_flatten(out, val[i]);
			}
		}
		else {
			out.push(val);
		}
	  
		return out;
	}
	
	// Array.isArray polyfill.
	// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/isArray
	if (! Array.isArray) {
	    Array.isArray = function(arg) {
	        return Object.prototype.toString.call(arg) === '[object Array]';
	    };
	}
	
	// .trim() polyfill
	// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/trim
	if (!String.prototype.trim) {
	  String.prototype.trim = function () {
	    return this.replace(/^[\s\uFEFF\xA0]+|[\s\uFEFF\xA0]+$/g, '');
	  };
	}
	
	/**
	 * DataTables utility methods
	 * 
	 * This namespace provides helper methods that DataTables uses internally to
	 * create a DataTable, but which are not exclusively used only for DataTables.
	 * These methods can be used by extension authors to save the duplication of
	 * code.
	 *
	 *  @namespace
	 */
	DataTable.util = {
		/**
		 * Throttle the calls to a function. Arguments and context are maintained
		 * for the throttled function.
		 *
		 * @param {function} fn Function to be called
		 * @param {integer} freq Call frequency in mS
		 * @return {function} Wrapped function
		 */
		throttle: function ( fn, freq ) {
			var
				frequency = freq !== undefined ? freq : 200,
				last,
				timer;
	
			return function () {
				var
					that = this,
					now  = +new Date(),
					args = arguments;
	
				if ( last && now < last + frequency ) {
					clearTimeout( timer );
	
					timer = setTimeout( function () {
						last = undefined;
						fn.apply( that, args );
					}, frequency );
				}
				else {
					last = now;
					fn.apply( that, args );
				}
			};
		},
	
	
		/**
		 * Escape a string such that it can be used in a regular expression
		 *
		 *  @param {string} val string to escape
		 *  @returns {string} escaped string
		 */
		escapeRegex: function ( val ) {
			return val.replace( _re_escape_regex, '\\$1' );
		}
	};
	
	
	
	/**
	 * Create a mapping object that allows camel case parameters to be looked up
	 * for their Hungarian counterparts. The mapping is stored in a private
	 * parameter called `_hungarianMap` which can be accessed on the source object.
	 *  @param {object} o
	 *  @memberof DataTable#oApi
	 */
	function _fnHungarianMap ( o )
	{
		var
			hungarian = 'a aa ai ao as b fn i m o s ',
			match,
			newKey,
			map = {};
	
		$.each( o, function (key, val) {
			match = key.match(/^([^A-Z]+?)([A-Z])/);
	
			if ( match && hungarian.indexOf(match[1]+' ') !== -1 )
			{
				newKey = key.replace( match[0], match[2].toLowerCase() );
				map[ newKey ] = key;
	
				if ( match[1] === 'o' )
				{
					_fnHungarianMap( o[key] );
				}
			}
		} );
	
		o._hungarianMap = map;
	}
	
	
	/**
	 * Convert from camel case parameters to Hungarian, based on a Hungarian map
	 * created by _fnHungarianMap.
	 *  @param {object} src The model object which holds all parameters that can be
	 *    mapped.
	 *  @param {object} user The object to convert from camel case to Hungarian.
	 *  @param {boolean} force When set to `true`, properties which already have a
	 *    Hungarian value in the `user` object will be overwritten. Otherwise they
	 *    won't be.
	 *  @memberof DataTable#oApi
	 */
	function _fnCamelToHungarian ( src, user, force )
	{
		if ( ! src._hungarianMap ) {
			_fnHungarianMap( src );
		}
	
		var hungarianKey;
	
		$.each( user, function (key, val) {
			hungarianKey = src._hungarianMap[ key ];
	
			if ( hungarianKey !== undefined && (force || user[hungarianKey] === undefined) )
			{
				// For objects, we need to buzz down into the object to copy parameters
				if ( hungarianKey.charAt(0) === 'o' )
				{
					// Copy the camelCase options over to the hungarian
					if ( ! user[ hungarianKey ] ) {
						user[ hungarianKey ] = {};
					}
					$.extend( true, user[hungarianKey], user[key] );
	
					_fnCamelToHungarian( src[hungarianKey], user[hungarianKey], force );
				}
				else {
					user[hungarianKey] = user[ key ];
				}
			}
		} );
	}
	
	
	/**
	 * Language compatibility - when certain options are given, and others aren't, we
	 * need to duplicate the values over, in order to provide backwards compatibility
	 * with older language files.
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnLanguageCompat( lang )
	{
		// Note the use of the Hungarian notation for the parameters in this method as
		// this is called after the mapping of camelCase to Hungarian
		var defaults = DataTable.defaults.oLanguage;
	
		// Default mapping
		var defaultDecimal = defaults.sDecimal;
		if ( defaultDecimal ) {
			_addNumericSort( defaultDecimal );
		}
	
		if ( lang ) {
			var zeroRecords = lang.sZeroRecords;
	
			// Backwards compatibility - if there is no sEmptyTable given, then use the same as
			// sZeroRecords - assuming that is given.
			if ( ! lang.sEmptyTable && zeroRecords &&
				defaults.sEmptyTable === "No data available in table" )
			{
				_fnMap( lang, lang, 'sZeroRecords', 'sEmptyTable' );
			}
	
			// Likewise with loading records
			if ( ! lang.sLoadingRecords && zeroRecords &&
				defaults.sLoadingRecords === "Loading..." )
			{
				_fnMap( lang, lang, 'sZeroRecords', 'sLoadingRecords' );
			}
	
			// Old parameter name of the thousands separator mapped onto the new
			if ( lang.sInfoThousands ) {
				lang.sThousands = lang.sInfoThousands;
			}
	
			var decimal = lang.sDecimal;
			if ( decimal && defaultDecimal !== decimal ) {
				_addNumericSort( decimal );
			}
		}
	}
	
	
	/**
	 * Map one parameter onto another
	 *  @param {object} o Object to map
	 *  @param {*} knew The new parameter name
	 *  @param {*} old The old parameter name
	 */
	var _fnCompatMap = function ( o, knew, old ) {
		if ( o[ knew ] !== undefined ) {
			o[ old ] = o[ knew ];
		}
	};
	
	
	/**
	 * Provide backwards compatibility for the main DT options. Note that the new
	 * options are mapped onto the old parameters, so this is an external interface
	 * change only.
	 *  @param {object} init Object to map
	 */
	function _fnCompatOpts ( init )
	{
		_fnCompatMap( init, 'ordering',      'bSort' );
		_fnCompatMap( init, 'orderMulti',    'bSortMulti' );
		_fnCompatMap( init, 'orderClasses',  'bSortClasses' );
		_fnCompatMap( init, 'orderCellsTop', 'bSortCellsTop' );
		_fnCompatMap( init, 'order',         'aaSorting' );
		_fnCompatMap( init, 'orderFixed',    'aaSortingFixed' );
		_fnCompatMap( init, 'paging',        'bPaginate' );
		_fnCompatMap( init, 'pagingType',    'sPaginationType' );
		_fnCompatMap( init, 'pageLength',    'iDisplayLength' );
		_fnCompatMap( init, 'searching',     'bFilter' );
	
		// Boolean initialisation of x-scrolling
		if ( typeof init.sScrollX === 'boolean' ) {
			init.sScrollX = init.sScrollX ? '100%' : '';
		}
		if ( typeof init.scrollX === 'boolean' ) {
			init.scrollX = init.scrollX ? '100%' : '';
		}
	
		// Column search objects are in an array, so it needs to be converted
		// element by element
		var searchCols = init.aoSearchCols;
	
		if ( searchCols ) {
			for ( var i=0, ien=searchCols.length ; i<ien ; i++ ) {
				if ( searchCols[i] ) {
					_fnCamelToHungarian( DataTable.models.oSearch, searchCols[i] );
				}
			}
		}
	}
	
	
	/**
	 * Provide backwards compatibility for column options. Note that the new options
	 * are mapped onto the old parameters, so this is an external interface change
	 * only.
	 *  @param {object} init Object to map
	 */
	function _fnCompatCols ( init )
	{
		_fnCompatMap( init, 'orderable',     'bSortable' );
		_fnCompatMap( init, 'orderData',     'aDataSort' );
		_fnCompatMap( init, 'orderSequence', 'asSorting' );
		_fnCompatMap( init, 'orderDataType', 'sortDataType' );
	
		// orderData can be given as an integer
		var dataSort = init.aDataSort;
		if ( typeof dataSort === 'number' && ! Array.isArray( dataSort ) ) {
			init.aDataSort = [ dataSort ];
		}
	}
	
	
	/**
	 * Browser feature detection for capabilities, quirks
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnBrowserDetect( settings )
	{
		// We don't need to do this every time DataTables is constructed, the values
		// calculated are specific to the browser and OS configuration which we
		// don't expect to change between initialisations
		if ( ! DataTable.__browser ) {
			var browser = {};
			DataTable.__browser = browser;
	
			// Scrolling feature / quirks detection
			var n = $('<div/>')
				.css( {
					position: 'fixed',
					top: 0,
					left: $(window).scrollLeft()*-1, // allow for scrolling
					height: 1,
					width: 1,
					overflow: 'hidden'
				} )
				.append(
					$('<div/>')
						.css( {
							position: 'absolute',
							top: 1,
							left: 1,
							width: 100,
							overflow: 'scroll'
						} )
						.append(
							$('<div/>')
								.css( {
									width: '100%',
									height: 10
								} )
						)
				)
				.appendTo( 'body' );
	
			var outer = n.children();
			var inner = outer.children();
	
			// Numbers below, in order, are:
			// inner.offsetWidth, inner.clientWidth, outer.offsetWidth, outer.clientWidth
			//
			// IE6 XP:                           100 100 100  83
			// IE7 Vista:                        100 100 100  83
			// IE 8+ Windows:                     83  83 100  83
			// Evergreen Windows:                 83  83 100  83
			// Evergreen Mac with scrollbars:     85  85 100  85
			// Evergreen Mac without scrollbars: 100 100 100 100
	
			// Get scrollbar width
			browser.barWidth = outer[0].offsetWidth - outer[0].clientWidth;
	
			// IE6/7 will oversize a width 100% element inside a scrolling element, to
			// include the width of the scrollbar, while other browsers ensure the inner
			// element is contained without forcing scrolling
			browser.bScrollOversize = inner[0].offsetWidth === 100 && outer[0].clientWidth !== 100;
	
			// In rtl text layout, some browsers (most, but not all) will place the
			// scrollbar on the left, rather than the right.
			browser.bScrollbarLeft = Math.round( inner.offset().left ) !== 1;
	
			// IE8- don't provide height and width for getBoundingClientRect
			browser.bBounding = n[0].getBoundingClientRect().width ? true : false;
	
			n.remove();
		}
	
		$.extend( settings.oBrowser, DataTable.__browser );
		settings.oScroll.iBarWidth = DataTable.__browser.barWidth;
	}
	
	
	/**
	 * Array.prototype reduce[Right] method, used for browsers which don't support
	 * JS 1.6. Done this way to reduce code size, since we iterate either way
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnReduce ( that, fn, init, start, end, inc )
	{
		var
			i = start,
			value,
			isSet = false;
	
		if ( init !== undefined ) {
			value = init;
			isSet = true;
		}
	
		while ( i !== end ) {
			if ( ! that.hasOwnProperty(i) ) {
				continue;
			}
	
			value = isSet ?
				fn( value, that[i], i, that ) :
				that[i];
	
			isSet = true;
			i += inc;
		}
	
		return value;
	}
	
	/**
	 * Add a column to the list used for the table with default values
	 *  @param {object} oSettings dataTables settings object
	 *  @param {node} nTh The th element for this column
	 *  @memberof DataTable#oApi
	 */
	function _fnAddColumn( oSettings, nTh )
	{
		// Add column to aoColumns array
		var oDefaults = DataTable.defaults.column;
		var iCol = oSettings.aoColumns.length;
		var oCol = $.extend( {}, DataTable.models.oColumn, oDefaults, {
			"nTh": nTh ? nTh : document.createElement('th'),
			"sTitle":    oDefaults.sTitle    ? oDefaults.sTitle    : nTh ? nTh.innerHTML : '',
			"aDataSort": oDefaults.aDataSort ? oDefaults.aDataSort : [iCol],
			"mData": oDefaults.mData ? oDefaults.mData : iCol,
			idx: iCol
		} );
		oSettings.aoColumns.push( oCol );
	
		// Add search object for column specific search. Note that the `searchCols[ iCol ]`
		// passed into extend can be undefined. This allows the user to give a default
		// with only some of the parameters defined, and also not give a default
		var searchCols = oSettings.aoPreSearchCols;
		searchCols[ iCol ] = $.extend( {}, DataTable.models.oSearch, searchCols[ iCol ] );
	
		// Use the default column options function to initialise classes etc
		_fnColumnOptions( oSettings, iCol, $(nTh).data() );
	}
	
	
	/**
	 * Apply options for a column
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iCol column index to consider
	 *  @param {object} oOptions object with sType, bVisible and bSearchable etc
	 *  @memberof DataTable#oApi
	 */
	function _fnColumnOptions( oSettings, iCol, oOptions )
	{
		var oCol = oSettings.aoColumns[ iCol ];
		var oClasses = oSettings.oClasses;
		var th = $(oCol.nTh);
	
		// Try to get width information from the DOM. We can't get it from CSS
		// as we'd need to parse the CSS stylesheet. `width` option can override
		if ( ! oCol.sWidthOrig ) {
			// Width attribute
			oCol.sWidthOrig = th.attr('width') || null;
	
			// Style attribute
			var t = (th.attr('style') || '').match(/width:\s*(\d+[pxem%]+)/);
			if ( t ) {
				oCol.sWidthOrig = t[1];
			}
		}
	
		/* User specified column options */
		if ( oOptions !== undefined && oOptions !== null )
		{
			// Backwards compatibility
			_fnCompatCols( oOptions );
	
			// Map camel case parameters to their Hungarian counterparts
			_fnCamelToHungarian( DataTable.defaults.column, oOptions, true );
	
			/* Backwards compatibility for mDataProp */
			if ( oOptions.mDataProp !== undefined && !oOptions.mData )
			{
				oOptions.mData = oOptions.mDataProp;
			}
	
			if ( oOptions.sType )
			{
				oCol._sManualType = oOptions.sType;
			}
	
			// `class` is a reserved word in Javascript, so we need to provide
			// the ability to use a valid name for the camel case input
			if ( oOptions.className && ! oOptions.sClass )
			{
				oOptions.sClass = oOptions.className;
			}
			if ( oOptions.sClass ) {
				th.addClass( oOptions.sClass );
			}
	
			$.extend( oCol, oOptions );
			_fnMap( oCol, oOptions, "sWidth", "sWidthOrig" );
	
			/* iDataSort to be applied (backwards compatibility), but aDataSort will take
			 * priority if defined
			 */
			if ( oOptions.iDataSort !== undefined )
			{
				oCol.aDataSort = [ oOptions.iDataSort ];
			}
			_fnMap( oCol, oOptions, "aDataSort" );
		}
	
		/* Cache the data get and set functions for speed */
		var mDataSrc = oCol.mData;
		var mData = _fnGetObjectDataFn( mDataSrc );
		var mRender = oCol.mRender ? _fnGetObjectDataFn( oCol.mRender ) : null;
	
		var attrTest = function( src ) {
			return typeof src === 'string' && src.indexOf('@') !== -1;
		};
		oCol._bAttrSrc = $.isPlainObject( mDataSrc ) && (
			attrTest(mDataSrc.sort) || attrTest(mDataSrc.type) || attrTest(mDataSrc.filter)
		);
		oCol._setter = null;
	
		oCol.fnGetData = function (rowData, type, meta) {
			var innerData = mData( rowData, type, undefined, meta );
	
			return mRender && type ?
				mRender( innerData, type, rowData, meta ) :
				innerData;
		};
		oCol.fnSetData = function ( rowData, val, meta ) {
			return _fnSetObjectDataFn( mDataSrc )( rowData, val, meta );
		};
	
		// Indicate if DataTables should read DOM data as an object or array
		// Used in _fnGetRowElements
		if ( typeof mDataSrc !== 'number' ) {
			oSettings._rowReadObject = true;
		}
	
		/* Feature sorting overrides column specific when off */
		if ( !oSettings.oFeatures.bSort )
		{
			oCol.bSortable = false;
			th.addClass( oClasses.sSortableNone ); // Have to add class here as order event isn't called
		}
	
		/* Check that the class assignment is correct for sorting */
		var bAsc = $.inArray('asc', oCol.asSorting) !== -1;
		var bDesc = $.inArray('desc', oCol.asSorting) !== -1;
		if ( !oCol.bSortable || (!bAsc && !bDesc) )
		{
			oCol.sSortingClass = oClasses.sSortableNone;
			oCol.sSortingClassJUI = "";
		}
		else if ( bAsc && !bDesc )
		{
			oCol.sSortingClass = oClasses.sSortableAsc;
			oCol.sSortingClassJUI = oClasses.sSortJUIAscAllowed;
		}
		else if ( !bAsc && bDesc )
		{
			oCol.sSortingClass = oClasses.sSortableDesc;
			oCol.sSortingClassJUI = oClasses.sSortJUIDescAllowed;
		}
		else
		{
			oCol.sSortingClass = oClasses.sSortable;
			oCol.sSortingClassJUI = oClasses.sSortJUI;
		}
	}
	
	
	/**
	 * Adjust the table column widths for new data. Note: you would probably want to
	 * do a redraw after calling this function!
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnAdjustColumnSizing ( settings )
	{
		/* Not interested in doing column width calculation if auto-width is disabled */
		if ( settings.oFeatures.bAutoWidth !== false )
		{
			var columns = settings.aoColumns;
	
			_fnCalculateColumnWidths( settings );
			for ( var i=0 , iLen=columns.length ; i<iLen ; i++ )
			{
				columns[i].nTh.style.width = columns[i].sWidth;
			}
		}
	
		var scroll = settings.oScroll;
		if ( scroll.sY !== '' || scroll.sX !== '')
		{
			_fnScrollDraw( settings );
		}
	
		_fnCallbackFire( settings, null, 'column-sizing', [settings] );
	}
	
	
	/**
	 * Covert the index of a visible column to the index in the data array (take account
	 * of hidden columns)
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iMatch Visible column index to lookup
	 *  @returns {int} i the data index
	 *  @memberof DataTable#oApi
	 */
	function _fnVisibleToColumnIndex( oSettings, iMatch )
	{
		var aiVis = _fnGetColumns( oSettings, 'bVisible' );
	
		return typeof aiVis[iMatch] === 'number' ?
			aiVis[iMatch] :
			null;
	}
	
	
	/**
	 * Covert the index of an index in the data array and convert it to the visible
	 *   column index (take account of hidden columns)
	 *  @param {int} iMatch Column index to lookup
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {int} i the data index
	 *  @memberof DataTable#oApi
	 */
	function _fnColumnIndexToVisible( oSettings, iMatch )
	{
		var aiVis = _fnGetColumns( oSettings, 'bVisible' );
		var iPos = $.inArray( iMatch, aiVis );
	
		return iPos !== -1 ? iPos : null;
	}
	
	
	/**
	 * Get the number of visible columns
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {int} i the number of visible columns
	 *  @memberof DataTable#oApi
	 */
	function _fnVisbleColumns( oSettings )
	{
		var vis = 0;
	
		// No reduce in IE8, use a loop for now
		$.each( oSettings.aoColumns, function ( i, col ) {
			if ( col.bVisible && $(col.nTh).css('display') !== 'none' ) {
				vis++;
			}
		} );
	
		return vis;
	}
	
	
	/**
	 * Get an array of column indexes that match a given property
	 *  @param {object} oSettings dataTables settings object
	 *  @param {string} sParam Parameter in aoColumns to look for - typically
	 *    bVisible or bSearchable
	 *  @returns {array} Array of indexes with matched properties
	 *  @memberof DataTable#oApi
	 */
	function _fnGetColumns( oSettings, sParam )
	{
		var a = [];
	
		$.map( oSettings.aoColumns, function(val, i) {
			if ( val[sParam] ) {
				a.push( i );
			}
		} );
	
		return a;
	}
	
	
	/**
	 * Calculate the 'type' of a column
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnColumnTypes ( settings )
	{
		var columns = settings.aoColumns;
		var data = settings.aoData;
		var types = DataTable.ext.type.detect;
		var i, ien, j, jen, k, ken;
		var col, cell, detectedType, cache;
	
		// For each column, spin over the 
		for ( i=0, ien=columns.length ; i<ien ; i++ ) {
			col = columns[i];
			cache = [];
	
			if ( ! col.sType && col._sManualType ) {
				col.sType = col._sManualType;
			}
			else if ( ! col.sType ) {
				for ( j=0, jen=types.length ; j<jen ; j++ ) {
					for ( k=0, ken=data.length ; k<ken ; k++ ) {
						// Use a cache array so we only need to get the type data
						// from the formatter once (when using multiple detectors)
						if ( cache[k] === undefined ) {
							cache[k] = _fnGetCellData( settings, k, i, 'type' );
						}
	
						detectedType = types[j]( cache[k], settings );
	
						// If null, then this type can't apply to this column, so
						// rather than testing all cells, break out. There is an
						// exception for the last type which is `html`. We need to
						// scan all rows since it is possible to mix string and HTML
						// types
						if ( ! detectedType && j !== types.length-1 ) {
							break;
						}
	
						// Only a single match is needed for html type since it is
						// bottom of the pile and very similar to string
						if ( detectedType === 'html' ) {
							break;
						}
					}
	
					// Type is valid for all data points in the column - use this
					// type
					if ( detectedType ) {
						col.sType = detectedType;
						break;
					}
				}
	
				// Fall back - if no type was detected, always use string
				if ( ! col.sType ) {
					col.sType = 'string';
				}
			}
		}
	}
	
	
	/**
	 * Take the column definitions and static columns arrays and calculate how
	 * they relate to column indexes. The callback function will then apply the
	 * definition found for a column to a suitable configuration object.
	 *  @param {object} oSettings dataTables settings object
	 *  @param {array} aoColDefs The aoColumnDefs array that is to be applied
	 *  @param {array} aoCols The aoColumns array that defines columns individually
	 *  @param {function} fn Callback function - takes two parameters, the calculated
	 *    column index and the definition for that column.
	 *  @memberof DataTable#oApi
	 */
	function _fnApplyColumnDefs( oSettings, aoColDefs, aoCols, fn )
	{
		var i, iLen, j, jLen, k, kLen, def;
		var columns = oSettings.aoColumns;
	
		// Column definitions with aTargets
		if ( aoColDefs )
		{
			/* Loop over the definitions array - loop in reverse so first instance has priority */
			for ( i=aoColDefs.length-1 ; i>=0 ; i-- )
			{
				def = aoColDefs[i];
	
				/* Each definition can target multiple columns, as it is an array */
				var aTargets = def.targets !== undefined ?
					def.targets :
					def.aTargets;
	
				if ( ! Array.isArray( aTargets ) )
				{
					aTargets = [ aTargets ];
				}
	
				for ( j=0, jLen=aTargets.length ; j<jLen ; j++ )
				{
					if ( typeof aTargets[j] === 'number' && aTargets[j] >= 0 )
					{
						/* Add columns that we don't yet know about */
						while( columns.length <= aTargets[j] )
						{
							_fnAddColumn( oSettings );
						}
	
						/* Integer, basic index */
						fn( aTargets[j], def );
					}
					else if ( typeof aTargets[j] === 'number' && aTargets[j] < 0 )
					{
						/* Negative integer, right to left column counting */
						fn( columns.length+aTargets[j], def );
					}
					else if ( typeof aTargets[j] === 'string' )
					{
						/* Class name matching on TH element */
						for ( k=0, kLen=columns.length ; k<kLen ; k++ )
						{
							if ( aTargets[j] == "_all" ||
							     $(columns[k].nTh).hasClass( aTargets[j] ) )
							{
								fn( k, def );
							}
						}
					}
				}
			}
		}
	
		// Statically defined columns array
		if ( aoCols )
		{
			for ( i=0, iLen=aoCols.length ; i<iLen ; i++ )
			{
				fn( i, aoCols[i] );
			}
		}
	}
	
	/**
	 * Add a data array to the table, creating DOM node etc. This is the parallel to
	 * _fnGatherData, but for adding rows from a Javascript source, rather than a
	 * DOM source.
	 *  @param {object} oSettings dataTables settings object
	 *  @param {array} aData data array to be added
	 *  @param {node} [nTr] TR element to add to the table - optional. If not given,
	 *    DataTables will create a row automatically
	 *  @param {array} [anTds] Array of TD|TH elements for the row - must be given
	 *    if nTr is.
	 *  @returns {int} >=0 if successful (index of new aoData entry), -1 if failed
	 *  @memberof DataTable#oApi
	 */
	function _fnAddData ( oSettings, aDataIn, nTr, anTds )
	{
		/* Create the object for storing information about this new row */
		var iRow = oSettings.aoData.length;
		var oData = $.extend( true, {}, DataTable.models.oRow, {
			src: nTr ? 'dom' : 'data',
			idx: iRow
		} );
	
		oData._aData = aDataIn;
		oSettings.aoData.push( oData );
	
		/* Create the cells */
		var nTd, sThisType;
		var columns = oSettings.aoColumns;
	
		// Invalidate the column types as the new data needs to be revalidated
		for ( var i=0, iLen=columns.length ; i<iLen ; i++ )
		{
			columns[i].sType = null;
		}
	
		/* Add to the display array */
		oSettings.aiDisplayMaster.push( iRow );
	
		var id = oSettings.rowIdFn( aDataIn );
		if ( id !== undefined ) {
			oSettings.aIds[ id ] = oData;
		}
	
		/* Create the DOM information, or register it if already present */
		if ( nTr || ! oSettings.oFeatures.bDeferRender )
		{
			_fnCreateTr( oSettings, iRow, nTr, anTds );
		}
	
		return iRow;
	}
	
	
	/**
	 * Add one or more TR elements to the table. Generally we'd expect to
	 * use this for reading data from a DOM sourced table, but it could be
	 * used for an TR element. Note that if a TR is given, it is used (i.e.
	 * it is not cloned).
	 *  @param {object} settings dataTables settings object
	 *  @param {array|node|jQuery} trs The TR element(s) to add to the table
	 *  @returns {array} Array of indexes for the added rows
	 *  @memberof DataTable#oApi
	 */
	function _fnAddTr( settings, trs )
	{
		var row;
	
		// Allow an individual node to be passed in
		if ( ! (trs instanceof $) ) {
			trs = $(trs);
		}
	
		return trs.map( function (i, el) {
			row = _fnGetRowElements( settings, el );
			return _fnAddData( settings, row.data, el, row.cells );
		} );
	}
	
	
	/**
	 * Take a TR element and convert it to an index in aoData
	 *  @param {object} oSettings dataTables settings object
	 *  @param {node} n the TR element to find
	 *  @returns {int} index if the node is found, null if not
	 *  @memberof DataTable#oApi
	 */
	function _fnNodeToDataIndex( oSettings, n )
	{
		return (n._DT_RowIndex!==undefined) ? n._DT_RowIndex : null;
	}
	
	
	/**
	 * Take a TD element and convert it into a column data index (not the visible index)
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iRow The row number the TD/TH can be found in
	 *  @param {node} n The TD/TH element to find
	 *  @returns {int} index if the node is found, -1 if not
	 *  @memberof DataTable#oApi
	 */
	function _fnNodeToColumnIndex( oSettings, iRow, n )
	{
		return $.inArray( n, oSettings.aoData[ iRow ].anCells );
	}
	
	
	/**
	 * Get the data for a given cell from the internal cache, taking into account data mapping
	 *  @param {object} settings dataTables settings object
	 *  @param {int} rowIdx aoData row id
	 *  @param {int} colIdx Column index
	 *  @param {string} type data get type ('display', 'type' 'filter' 'sort')
	 *  @returns {*} Cell data
	 *  @memberof DataTable#oApi
	 */
	function _fnGetCellData( settings, rowIdx, colIdx, type )
	{
		var draw           = settings.iDraw;
		var col            = settings.aoColumns[colIdx];
		var rowData        = settings.aoData[rowIdx]._aData;
		var defaultContent = col.sDefaultContent;
		var cellData       = col.fnGetData( rowData, type, {
			settings: settings,
			row:      rowIdx,
			col:      colIdx
		} );
	
		if ( cellData === undefined ) {
			if ( settings.iDrawError != draw && defaultContent === null ) {
				_fnLog( settings, 0, "Requested unknown parameter "+
					(typeof col.mData=='function' ? '{function}' : "'"+col.mData+"'")+
					" for row "+rowIdx+", column "+colIdx, 4 );
				settings.iDrawError = draw;
			}
			return defaultContent;
		}
	
		// When the data source is null and a specific data type is requested (i.e.
		// not the original data), we can use default column data
		if ( (cellData === rowData || cellData === null) && defaultContent !== null && type !== undefined ) {
			cellData = defaultContent;
		}
		else if ( typeof cellData === 'function' ) {
			// If the data source is a function, then we run it and use the return,
			// executing in the scope of the data object (for instances)
			return cellData.call( rowData );
		}
	
		if ( cellData === null && type == 'display' ) {
			return '';
		}
		return cellData;
	}
	
	
	/**
	 * Set the value for a specific cell, into the internal data cache
	 *  @param {object} settings dataTables settings object
	 *  @param {int} rowIdx aoData row id
	 *  @param {int} colIdx Column index
	 *  @param {*} val Value to set
	 *  @memberof DataTable#oApi
	 */
	function _fnSetCellData( settings, rowIdx, colIdx, val )
	{
		var col     = settings.aoColumns[colIdx];
		var rowData = settings.aoData[rowIdx]._aData;
	
		col.fnSetData( rowData, val, {
			settings: settings,
			row:      rowIdx,
			col:      colIdx
		}  );
	}
	
	
	// Private variable that is used to match action syntax in the data property object
	var __reArray = /\[.*?\]$/;
	var __reFn = /\(\)$/;
	
	/**
	 * Split string on periods, taking into account escaped periods
	 * @param  {string} str String to split
	 * @return {array} Split string
	 */
	function _fnSplitObjNotation( str )
	{
		return $.map( str.match(/(\\.|[^\.])+/g) || [''], function ( s ) {
			return s.replace(/\\\./g, '.');
		} );
	}
	
	
	/**
	 * Return a function that can be used to get data from a source object, taking
	 * into account the ability to use nested objects as a source
	 *  @param {string|int|function} mSource The data source for the object
	 *  @returns {function} Data get function
	 *  @memberof DataTable#oApi
	 */
	function _fnGetObjectDataFn( mSource )
	{
		if ( $.isPlainObject( mSource ) )
		{
			/* Build an object of get functions, and wrap them in a single call */
			var o = {};
			$.each( mSource, function (key, val) {
				if ( val ) {
					o[key] = _fnGetObjectDataFn( val );
				}
			} );
	
			return function (data, type, row, meta) {
				var t = o[type] || o._;
				return t !== undefined ?
					t(data, type, row, meta) :
					data;
			};
		}
		else if ( mSource === null )
		{
			/* Give an empty string for rendering / sorting etc */
			return function (data) { // type, row and meta also passed, but not used
				return data;
			};
		}
		else if ( typeof mSource === 'function' )
		{
			return function (data, type, row, meta) {
				return mSource( data, type, row, meta );
			};
		}
		else if ( typeof mSource === 'string' && (mSource.indexOf('.') !== -1 ||
			      mSource.indexOf('[') !== -1 || mSource.indexOf('(') !== -1) )
		{
			/* If there is a . in the source string then the data source is in a
			 * nested object so we loop over the data for each level to get the next
			 * level down. On each loop we test for undefined, and if found immediately
			 * return. This allows entire objects to be missing and sDefaultContent to
			 * be used if defined, rather than throwing an error
			 */
			var fetchData = function (data, type, src) {
				var arrayNotation, funcNotation, out, innerSrc;
	
				if ( src !== "" )
				{
					var a = _fnSplitObjNotation( src );
	
					for ( var i=0, iLen=a.length ; i<iLen ; i++ )
					{
						// Check if we are dealing with special notation
						arrayNotation = a[i].match(__reArray);
						funcNotation = a[i].match(__reFn);
	
						if ( arrayNotation )
						{
							// Array notation
							a[i] = a[i].replace(__reArray, '');
	
							// Condition allows simply [] to be passed in
							if ( a[i] !== "" ) {
								data = data[ a[i] ];
							}
							out = [];
	
							// Get the remainder of the nested object to get
							a.splice( 0, i+1 );
							innerSrc = a.join('.');
	
							// Traverse each entry in the array getting the properties requested
							if ( Array.isArray( data ) ) {
								for ( var j=0, jLen=data.length ; j<jLen ; j++ ) {
									out.push( fetchData( data[j], type, innerSrc ) );
								}
							}
	
							// If a string is given in between the array notation indicators, that
							// is used to join the strings together, otherwise an array is returned
							var join = arrayNotation[0].substring(1, arrayNotation[0].length-1);
							data = (join==="") ? out : out.join(join);
	
							// The inner call to fetchData has already traversed through the remainder
							// of the source requested, so we exit from the loop
							break;
						}
						else if ( funcNotation )
						{
							// Function call
							a[i] = a[i].replace(__reFn, '');
							data = data[ a[i] ]();
							continue;
						}
	
						if ( data === null || data[ a[i] ] === undefined )
						{
							return undefined;
						}
						data = data[ a[i] ];
					}
				}
	
				return data;
			};
	
			return function (data, type) { // row and meta also passed, but not used
				return fetchData( data, type, mSource );
			};
		}
		else
		{
			/* Array or flat object mapping */
			return function (data, type) { // row and meta also passed, but not used
				return data[mSource];
			};
		}
	}
	
	
	/**
	 * Return a function that can be used to set data from a source object, taking
	 * into account the ability to use nested objects as a source
	 *  @param {string|int|function} mSource The data source for the object
	 *  @returns {function} Data set function
	 *  @memberof DataTable#oApi
	 */
	function _fnSetObjectDataFn( mSource )
	{
		if ( $.isPlainObject( mSource ) )
		{
			/* Unlike get, only the underscore (global) option is used for for
			 * setting data since we don't know the type here. This is why an object
			 * option is not documented for `mData` (which is read/write), but it is
			 * for `mRender` which is read only.
			 */
			return _fnSetObjectDataFn( mSource._ );
		}
		else if ( mSource === null )
		{
			/* Nothing to do when the data source is null */
			return function () {};
		}
		else if ( typeof mSource === 'function' )
		{
			return function (data, val, meta) {
				mSource( data, 'set', val, meta );
			};
		}
		else if ( typeof mSource === 'string' && (mSource.indexOf('.') !== -1 ||
			      mSource.indexOf('[') !== -1 || mSource.indexOf('(') !== -1) )
		{
			/* Like the get, we need to get data from a nested object */
			var setData = function (data, val, src) {
				var a = _fnSplitObjNotation( src ), b;
				var aLast = a[a.length-1];
				var arrayNotation, funcNotation, o, innerSrc;
	
				for ( var i=0, iLen=a.length-1 ; i<iLen ; i++ )
				{
					// Protect against prototype pollution
					if (a[i] === '__proto__' || a[i] === 'constructor') {
						throw new Error('Cannot set prototype values');
					}
	
					// Check if we are dealing with an array notation request
					arrayNotation = a[i].match(__reArray);
					funcNotation = a[i].match(__reFn);
	
					if ( arrayNotation )
					{
						a[i] = a[i].replace(__reArray, '');
						data[ a[i] ] = [];
	
						// Get the remainder of the nested object to set so we can recurse
						b = a.slice();
						b.splice( 0, i+1 );
						innerSrc = b.join('.');
	
						// Traverse each entry in the array setting the properties requested
						if ( Array.isArray( val ) )
						{
							for ( var j=0, jLen=val.length ; j<jLen ; j++ )
							{
								o = {};
								setData( o, val[j], innerSrc );
								data[ a[i] ].push( o );
							}
						}
						else
						{
							// We've been asked to save data to an array, but it
							// isn't array data to be saved. Best that can be done
							// is to just save the value.
							data[ a[i] ] = val;
						}
	
						// The inner call to setData has already traversed through the remainder
						// of the source and has set the data, thus we can exit here
						return;
					}
					else if ( funcNotation )
					{
						// Function call
						a[i] = a[i].replace(__reFn, '');
						data = data[ a[i] ]( val );
					}
	
					// If the nested object doesn't currently exist - since we are
					// trying to set the value - create it
					if ( data[ a[i] ] === null || data[ a[i] ] === undefined )
					{
						data[ a[i] ] = {};
					}
					data = data[ a[i] ];
				}
	
				// Last item in the input - i.e, the actual set
				if ( aLast.match(__reFn ) )
				{
					// Function call
					data = data[ aLast.replace(__reFn, '') ]( val );
				}
				else
				{
					// If array notation is used, we just want to strip it and use the property name
					// and assign the value. If it isn't used, then we get the result we want anyway
					data[ aLast.replace(__reArray, '') ] = val;
				}
			};
	
			return function (data, val) { // meta is also passed in, but not used
				return setData( data, val, mSource );
			};
		}
		else
		{
			/* Array or flat object mapping */
			return function (data, val) { // meta is also passed in, but not used
				data[mSource] = val;
			};
		}
	}
	
	
	/**
	 * Return an array with the full table data
	 *  @param {object} oSettings dataTables settings object
	 *  @returns array {array} aData Master data array
	 *  @memberof DataTable#oApi
	 */
	function _fnGetDataMaster ( settings )
	{
		return _pluck( settings.aoData, '_aData' );
	}
	
	
	/**
	 * Nuke the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnClearTable( settings )
	{
		settings.aoData.length = 0;
		settings.aiDisplayMaster.length = 0;
		settings.aiDisplay.length = 0;
		settings.aIds = {};
	}
	
	
	 /**
	 * Take an array of integers (index array) and remove a target integer (value - not
	 * the key!)
	 *  @param {array} a Index array to target
	 *  @param {int} iTarget value to find
	 *  @memberof DataTable#oApi
	 */
	function _fnDeleteIndex( a, iTarget, splice )
	{
		var iTargetIndex = -1;
	
		for ( var i=0, iLen=a.length ; i<iLen ; i++ )
		{
			if ( a[i] == iTarget )
			{
				iTargetIndex = i;
			}
			else if ( a[i] > iTarget )
			{
				a[i]--;
			}
		}
	
		if ( iTargetIndex != -1 && splice === undefined )
		{
			a.splice( iTargetIndex, 1 );
		}
	}
	
	
	/**
	 * Mark cached data as invalid such that a re-read of the data will occur when
	 * the cached data is next requested. Also update from the data source object.
	 *
	 * @param {object} settings DataTables settings object
	 * @param {int}    rowIdx   Row index to invalidate
	 * @param {string} [src]    Source to invalidate from: undefined, 'auto', 'dom'
	 *     or 'data'
	 * @param {int}    [colIdx] Column index to invalidate. If undefined the whole
	 *     row will be invalidated
	 * @memberof DataTable#oApi
	 *
	 * @todo For the modularisation of v1.11 this will need to become a callback, so
	 *   the sort and filter methods can subscribe to it. That will required
	 *   initialisation options for sorting, which is why it is not already baked in
	 */
	function _fnInvalidate( settings, rowIdx, src, colIdx )
	{
		var row = settings.aoData[ rowIdx ];
		var i, ien;
		var cellWrite = function ( cell, col ) {
			// This is very frustrating, but in IE if you just write directly
			// to innerHTML, and elements that are overwritten are GC'ed,
			// even if there is a reference to them elsewhere
			while ( cell.childNodes.length ) {
				cell.removeChild( cell.firstChild );
			}
	
			cell.innerHTML = _fnGetCellData( settings, rowIdx, col, 'display' );
		};
	
		// Are we reading last data from DOM or the data object?
		if ( src === 'dom' || ((! src || src === 'auto') && row.src === 'dom') ) {
			// Read the data from the DOM
			row._aData = _fnGetRowElements(
					settings, row, colIdx, colIdx === undefined ? undefined : row._aData
				)
				.data;
		}
		else {
			// Reading from data object, update the DOM
			var cells = row.anCells;
	
			if ( cells ) {
				if ( colIdx !== undefined ) {
					cellWrite( cells[colIdx], colIdx );
				}
				else {
					for ( i=0, ien=cells.length ; i<ien ; i++ ) {
						cellWrite( cells[i], i );
					}
				}
			}
		}
	
		// For both row and cell invalidation, the cached data for sorting and
		// filtering is nulled out
		row._aSortData = null;
		row._aFilterData = null;
	
		// Invalidate the type for a specific column (if given) or all columns since
		// the data might have changed
		var cols = settings.aoColumns;
		if ( colIdx !== undefined ) {
			cols[ colIdx ].sType = null;
		}
		else {
			for ( i=0, ien=cols.length ; i<ien ; i++ ) {
				cols[i].sType = null;
			}
	
			// Update DataTables special `DT_*` attributes for the row
			_fnRowAttributes( settings, row );
		}
	}
	
	
	/**
	 * Build a data source object from an HTML row, reading the contents of the
	 * cells that are in the row.
	 *
	 * @param {object} settings DataTables settings object
	 * @param {node|object} TR element from which to read data or existing row
	 *   object from which to re-read the data from the cells
	 * @param {int} [colIdx] Optional column index
	 * @param {array|object} [d] Data source object. If `colIdx` is given then this
	 *   parameter should also be given and will be used to write the data into.
	 *   Only the column in question will be written
	 * @returns {object} Object with two parameters: `data` the data read, in
	 *   document order, and `cells` and array of nodes (they can be useful to the
	 *   caller, so rather than needing a second traversal to get them, just return
	 *   them from here).
	 * @memberof DataTable#oApi
	 */
	function _fnGetRowElements( settings, row, colIdx, d )
	{
		var
			tds = [],
			td = row.firstChild,
			name, col, o, i=0, contents,
			columns = settings.aoColumns,
			objectRead = settings._rowReadObject;
	
		// Allow the data object to be passed in, or construct
		d = d !== undefined ?
			d :
			objectRead ?
				{} :
				[];
	
		var attr = function ( str, td  ) {
			if ( typeof str === 'string' ) {
				var idx = str.indexOf('@');
	
				if ( idx !== -1 ) {
					var attr = str.substring( idx+1 );
					var setter = _fnSetObjectDataFn( str );
					setter( d, td.getAttribute( attr ) );
				}
			}
		};
	
		// Read data from a cell and store into the data object
		var cellProcess = function ( cell ) {
			if ( colIdx === undefined || colIdx === i ) {
				col = columns[i];
				contents = (cell.innerHTML).trim();
	
				if ( col && col._bAttrSrc ) {
					var setter = _fnSetObjectDataFn( col.mData._ );
					setter( d, contents );
	
					attr( col.mData.sort, cell );
					attr( col.mData.type, cell );
					attr( col.mData.filter, cell );
				}
				else {
					// Depending on the `data` option for the columns the data can
					// be read to either an object or an array.
					if ( objectRead ) {
						if ( ! col._setter ) {
							// Cache the setter function
							col._setter = _fnSetObjectDataFn( col.mData );
						}
						col._setter( d, contents );
					}
					else {
						d[i] = contents;
					}
				}
			}
	
			i++;
		};
	
		if ( td ) {
			// `tr` element was passed in
			while ( td ) {
				name = td.nodeName.toUpperCase();
	
				if ( name == "TD" || name == "TH" ) {
					cellProcess( td );
					tds.push( td );
				}
	
				td = td.nextSibling;
			}
		}
		else {
			// Existing row object passed in
			tds = row.anCells;
	
			for ( var j=0, jen=tds.length ; j<jen ; j++ ) {
				cellProcess( tds[j] );
			}
		}
	
		// Read the ID from the DOM if present
		var rowNode = row.firstChild ? row : row.nTr;
	
		if ( rowNode ) {
			var id = rowNode.getAttribute( 'id' );
	
			if ( id ) {
				_fnSetObjectDataFn( settings.rowId )( d, id );
			}
		}
	
		return {
			data: d,
			cells: tds
		};
	}
	/**
	 * Create a new TR element (and it's TD children) for a row
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iRow Row to consider
	 *  @param {node} [nTrIn] TR element to add to the table - optional. If not given,
	 *    DataTables will create a row automatically
	 *  @param {array} [anTds] Array of TD|TH elements for the row - must be given
	 *    if nTr is.
	 *  @memberof DataTable#oApi
	 */
	function _fnCreateTr ( oSettings, iRow, nTrIn, anTds )
	{
		var
			row = oSettings.aoData[iRow],
			rowData = row._aData,
			cells = [],
			nTr, nTd, oCol,
			i, iLen, create;
	
		if ( row.nTr === null )
		{
			nTr = nTrIn || document.createElement('tr');
	
			row.nTr = nTr;
			row.anCells = cells;
	
			/* Use a private property on the node to allow reserve mapping from the node
			 * to the aoData array for fast look up
			 */
			nTr._DT_RowIndex = iRow;
	
			/* Special parameters can be given by the data source to be used on the row */
			_fnRowAttributes( oSettings, row );
	
			/* Process each column */
			for ( i=0, iLen=oSettings.aoColumns.length ; i<iLen ; i++ )
			{
				oCol = oSettings.aoColumns[i];
				create = nTrIn ? false : true;
	
				nTd = create ? document.createElement( oCol.sCellType ) : anTds[i];
				nTd._DT_CellIndex = {
					row: iRow,
					column: i
				};
				
				cells.push( nTd );
	
				// Need to create the HTML if new, or if a rendering function is defined
				if ( create || ((oCol.mRender || oCol.mData !== i) &&
					 (!$.isPlainObject(oCol.mData) || oCol.mData._ !== i+'.display')
				)) {
					nTd.innerHTML = _fnGetCellData( oSettings, iRow, i, 'display' );
				}
	
				/* Add user defined class */
				if ( oCol.sClass )
				{
					nTd.className += ' '+oCol.sClass;
				}
	
				// Visibility - add or remove as required
				if ( oCol.bVisible && ! nTrIn )
				{
					nTr.appendChild( nTd );
				}
				else if ( ! oCol.bVisible && nTrIn )
				{
					nTd.parentNode.removeChild( nTd );
				}
	
				if ( oCol.fnCreatedCell )
				{
					oCol.fnCreatedCell.call( oSettings.oInstance,
						nTd, _fnGetCellData( oSettings, iRow, i ), rowData, iRow, i
					);
				}
			}
	
			_fnCallbackFire( oSettings, 'aoRowCreatedCallback', null, [nTr, rowData, iRow, cells] );
		}
	}
	
	
	/**
	 * Add attributes to a row based on the special `DT_*` parameters in a data
	 * source object.
	 *  @param {object} settings DataTables settings object
	 *  @param {object} DataTables row object for the row to be modified
	 *  @memberof DataTable#oApi
	 */
	function _fnRowAttributes( settings, row )
	{
		var tr = row.nTr;
		var data = row._aData;
	
		if ( tr ) {
			var id = settings.rowIdFn( data );
	
			if ( id ) {
				tr.id = id;
			}
	
			if ( data.DT_RowClass ) {
				// Remove any classes added by DT_RowClass before
				var a = data.DT_RowClass.split(' ');
				row.__rowc = row.__rowc ?
					_unique( row.__rowc.concat( a ) ) :
					a;
	
				$(tr)
					.removeClass( row.__rowc.join(' ') )
					.addClass( data.DT_RowClass );
			}
	
			if ( data.DT_RowAttr ) {
				$(tr).attr( data.DT_RowAttr );
			}
	
			if ( data.DT_RowData ) {
				$(tr).data( data.DT_RowData );
			}
		}
	}
	
	
	/**
	 * Create the HTML header for the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnBuildHead( oSettings )
	{
		var i, ien, cell, row, column;
		var thead = oSettings.nTHead;
		var tfoot = oSettings.nTFoot;
		var createHeader = $('th, td', thead).length === 0;
		var classes = oSettings.oClasses;
		var columns = oSettings.aoColumns;
	
		if ( createHeader ) {
			row = $('<tr/>').appendTo( thead );
		}
	
		for ( i=0, ien=columns.length ; i<ien ; i++ ) {
			column = columns[i];
			cell = $( column.nTh ).addClass( column.sClass );
	
			if ( createHeader ) {
				cell.appendTo( row );
			}
	
			// 1.11 move into sorting
			if ( oSettings.oFeatures.bSort ) {
				cell.addClass( column.sSortingClass );
	
				if ( column.bSortable !== false ) {
					cell
						.attr( 'tabindex', oSettings.iTabIndex )
						.attr( 'aria-controls', oSettings.sTableId );
	
					_fnSortAttachListener( oSettings, column.nTh, i );
				}
			}
	
			if ( column.sTitle != cell[0].innerHTML ) {
				cell.html( column.sTitle );
			}
	
			_fnRenderer( oSettings, 'header' )(
				oSettings, cell, column, classes
			);
		}
	
		if ( createHeader ) {
			_fnDetectHeader( oSettings.aoHeader, thead );
		}
		
		/* ARIA role for the rows */
		$(thead).children('tr').attr('role', 'row');
	
		/* Deal with the footer - add classes if required */
		$(thead).children('tr').children('th, td').addClass( classes.sHeaderTH );
		$(tfoot).children('tr').children('th, td').addClass( classes.sFooterTH );
	
		// Cache the footer cells. Note that we only take the cells from the first
		// row in the footer. If there is more than one row the user wants to
		// interact with, they need to use the table().foot() method. Note also this
		// allows cells to be used for multiple columns using colspan
		if ( tfoot !== null ) {
			var cells = oSettings.aoFooter[0];
	
			for ( i=0, ien=cells.length ; i<ien ; i++ ) {
				column = columns[i];
				column.nTf = cells[i].cell;
	
				if ( column.sClass ) {
					$(column.nTf).addClass( column.sClass );
				}
			}
		}
	}
	
	
	/**
	 * Draw the header (or footer) element based on the column visibility states. The
	 * methodology here is to use the layout array from _fnDetectHeader, modified for
	 * the instantaneous column visibility, to construct the new layout. The grid is
	 * traversed over cell at a time in a rows x columns grid fashion, although each
	 * cell insert can cover multiple elements in the grid - which is tracks using the
	 * aApplied array. Cell inserts in the grid will only occur where there isn't
	 * already a cell in that position.
	 *  @param {object} oSettings dataTables settings object
	 *  @param array {objects} aoSource Layout array from _fnDetectHeader
	 *  @param {boolean} [bIncludeHidden=false] If true then include the hidden columns in the calc,
	 *  @memberof DataTable#oApi
	 */
	function _fnDrawHead( oSettings, aoSource, bIncludeHidden )
	{
		var i, iLen, j, jLen, k, kLen, n, nLocalTr;
		var aoLocal = [];
		var aApplied = [];
		var iColumns = oSettings.aoColumns.length;
		var iRowspan, iColspan;
	
		if ( ! aoSource )
		{
			return;
		}
	
		if (  bIncludeHidden === undefined )
		{
			bIncludeHidden = false;
		}
	
		/* Make a copy of the master layout array, but without the visible columns in it */
		for ( i=0, iLen=aoSource.length ; i<iLen ; i++ )
		{
			aoLocal[i] = aoSource[i].slice();
			aoLocal[i].nTr = aoSource[i].nTr;
	
			/* Remove any columns which are currently hidden */
			for ( j=iColumns-1 ; j>=0 ; j-- )
			{
				if ( !oSettings.aoColumns[j].bVisible && !bIncludeHidden )
				{
					aoLocal[i].splice( j, 1 );
				}
			}
	
			/* Prep the applied array - it needs an element for each row */
			aApplied.push( [] );
		}
	
		for ( i=0, iLen=aoLocal.length ; i<iLen ; i++ )
		{
			nLocalTr = aoLocal[i].nTr;
	
			/* All cells are going to be replaced, so empty out the row */
			if ( nLocalTr )
			{
				while( (n = nLocalTr.firstChild) )
				{
					nLocalTr.removeChild( n );
				}
			}
	
			for ( j=0, jLen=aoLocal[i].length ; j<jLen ; j++ )
			{
				iRowspan = 1;
				iColspan = 1;
	
				/* Check to see if there is already a cell (row/colspan) covering our target
				 * insert point. If there is, then there is nothing to do.
				 */
				if ( aApplied[i][j] === undefined )
				{
					nLocalTr.appendChild( aoLocal[i][j].cell );
					aApplied[i][j] = 1;
	
					/* Expand the cell to cover as many rows as needed */
					while ( aoLocal[i+iRowspan] !== undefined &&
					        aoLocal[i][j].cell == aoLocal[i+iRowspan][j].cell )
					{
						aApplied[i+iRowspan][j] = 1;
						iRowspan++;
					}
	
					/* Expand the cell to cover as many columns as needed */
					while ( aoLocal[i][j+iColspan] !== undefined &&
					        aoLocal[i][j].cell == aoLocal[i][j+iColspan].cell )
					{
						/* Must update the applied array over the rows for the columns */
						for ( k=0 ; k<iRowspan ; k++ )
						{
							aApplied[i+k][j+iColspan] = 1;
						}
						iColspan++;
					}
	
					/* Do the actual expansion in the DOM */
					$(aoLocal[i][j].cell)
						.attr('rowspan', iRowspan)
						.attr('colspan', iColspan);
				}
			}
		}
	}
	
	
	/**
	 * Insert the required TR nodes into the table for display
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnDraw( oSettings )
	{
		/* Provide a pre-callback function which can be used to cancel the draw is false is returned */
		var aPreDraw = _fnCallbackFire( oSettings, 'aoPreDrawCallback', 'preDraw', [oSettings] );
		if ( $.inArray( false, aPreDraw ) !== -1 )
		{
			_fnProcessingDisplay( oSettings, false );
			return;
		}
	
		var i, iLen, n;
		var anRows = [];
		var iRowCount = 0;
		var asStripeClasses = oSettings.asStripeClasses;
		var iStripes = asStripeClasses.length;
		var iOpenRows = oSettings.aoOpenRows.length;
		var oLang = oSettings.oLanguage;
		var iInitDisplayStart = oSettings.iInitDisplayStart;
		var bServerSide = _fnDataSource( oSettings ) == 'ssp';
		var aiDisplay = oSettings.aiDisplay;
	
		oSettings.bDrawing = true;
	
		/* Check and see if we have an initial draw position from state saving */
		if ( iInitDisplayStart !== undefined && iInitDisplayStart !== -1 )
		{
			oSettings._iDisplayStart = bServerSide ?
				iInitDisplayStart :
				iInitDisplayStart >= oSettings.fnRecordsDisplay() ?
					0 :
					iInitDisplayStart;
	
			oSettings.iInitDisplayStart = -1;
		}
	
		var iDisplayStart = oSettings._iDisplayStart;
		var iDisplayEnd = oSettings.fnDisplayEnd();
	
		/* Server-side processing draw intercept */
		if ( oSettings.bDeferLoading )
		{
			oSettings.bDeferLoading = false;
			oSettings.iDraw++;
			_fnProcessingDisplay( oSettings, false );
		}
		else if ( !bServerSide )
		{
			oSettings.iDraw++;
		}
		else if ( !oSettings.bDestroying && !_fnAjaxUpdate( oSettings ) )
		{
			return;
		}
	
		if ( aiDisplay.length !== 0 )
		{
			var iStart = bServerSide ? 0 : iDisplayStart;
			var iEnd = bServerSide ? oSettings.aoData.length : iDisplayEnd;
	
			for ( var j=iStart ; j<iEnd ; j++ )
			{
				var iDataIndex = aiDisplay[j];
				var aoData = oSettings.aoData[ iDataIndex ];
				if ( aoData.nTr === null )
				{
					_fnCreateTr( oSettings, iDataIndex );
				}
	
				var nRow = aoData.nTr;
	
				/* Remove the old striping classes and then add the new one */
				if ( iStripes !== 0 )
				{
					var sStripe = asStripeClasses[ iRowCount % iStripes ];
					if ( aoData._sRowStripe != sStripe )
					{
						$(nRow).removeClass( aoData._sRowStripe ).addClass( sStripe );
						aoData._sRowStripe = sStripe;
					}
				}
	
				// Row callback functions - might want to manipulate the row
				// iRowCount and j are not currently documented. Are they at all
				// useful?
				_fnCallbackFire( oSettings, 'aoRowCallback', null,
					[nRow, aoData._aData, iRowCount, j, iDataIndex] );
	
				anRows.push( nRow );
				iRowCount++;
			}
		}
		else
		{
			/* Table is empty - create a row with an empty message in it */
			var sZero = oLang.sZeroRecords;
			if ( oSettings.iDraw == 1 &&  _fnDataSource( oSettings ) == 'ajax' )
			{
				sZero = oLang.sLoadingRecords;
			}
			else if ( oLang.sEmptyTable && oSettings.fnRecordsTotal() === 0 )
			{
				sZero = oLang.sEmptyTable;
			}
	
			anRows[ 0 ] = $( '<tr/>', { 'class': iStripes ? asStripeClasses[0] : '' } )
				.append( $('<td />', {
					'valign':  'top',
					'colSpan': _fnVisbleColumns( oSettings ),
					'class':   oSettings.oClasses.sRowEmpty
				} ).html( sZero ) )[0];
		}
	
		/* Header and footer callbacks */
		_fnCallbackFire( oSettings, 'aoHeaderCallback', 'header', [ $(oSettings.nTHead).children('tr')[0],
			_fnGetDataMaster( oSettings ), iDisplayStart, iDisplayEnd, aiDisplay ] );
	
		_fnCallbackFire( oSettings, 'aoFooterCallback', 'footer', [ $(oSettings.nTFoot).children('tr')[0],
			_fnGetDataMaster( oSettings ), iDisplayStart, iDisplayEnd, aiDisplay ] );
	
		var body = $(oSettings.nTBody);
	
		body.children().detach();
		body.append( $(anRows) );
	
		/* Call all required callback functions for the end of a draw */
		_fnCallbackFire( oSettings, 'aoDrawCallback', 'draw', [oSettings] );
	
		/* Draw is complete, sorting and filtering must be as well */
		oSettings.bSorted = false;
		oSettings.bFiltered = false;
		oSettings.bDrawing = false;
	}
	
	
	/**
	 * Redraw the table - taking account of the various features which are enabled
	 *  @param {object} oSettings dataTables settings object
	 *  @param {boolean} [holdPosition] Keep the current paging position. By default
	 *    the paging is reset to the first page
	 *  @memberof DataTable#oApi
	 */
	function _fnReDraw( settings, holdPosition )
	{
		var
			features = settings.oFeatures,
			sort     = features.bSort,
			filter   = features.bFilter;
	
		if ( sort ) {
			_fnSort( settings );
		}
	
		if ( filter ) {
			_fnFilterComplete( settings, settings.oPreviousSearch );
		}
		else {
			// No filtering, so we want to just use the display master
			settings.aiDisplay = settings.aiDisplayMaster.slice();
		}
	
		if ( holdPosition !== true ) {
			settings._iDisplayStart = 0;
		}
	
		// Let any modules know about the draw hold position state (used by
		// scrolling internally)
		settings._drawHold = holdPosition;
	
		_fnDraw( settings );
	
		settings._drawHold = false;
	}
	
	
	/**
	 * Add the options to the page HTML for the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnAddOptionsHtml ( oSettings )
	{
		var classes = oSettings.oClasses;
		var table = $(oSettings.nTable);
		var holding = $('<div/>').insertBefore( table ); // Holding element for speed
		var features = oSettings.oFeatures;
	
		// All DataTables are wrapped in a div
		var insert = $('<div/>', {
			id:      oSettings.sTableId+'_wrapper',
			'class': classes.sWrapper + (oSettings.nTFoot ? '' : ' '+classes.sNoFooter)
		} );
	
		oSettings.nHolding = holding[0];
		oSettings.nTableWrapper = insert[0];
		oSettings.nTableReinsertBefore = oSettings.nTable.nextSibling;
	
		/* Loop over the user set positioning and place the elements as needed */
		var aDom = oSettings.sDom.split('');
		var featureNode, cOption, nNewNode, cNext, sAttr, j;
		for ( var i=0 ; i<aDom.length ; i++ )
		{
			featureNode = null;
			cOption = aDom[i];
	
			if ( cOption == '<' )
			{
				/* New container div */
				nNewNode = $('<div/>')[0];
	
				/* Check to see if we should append an id and/or a class name to the container */
				cNext = aDom[i+1];
				if ( cNext == "'" || cNext == '"' )
				{
					sAttr = "";
					j = 2;
					while ( aDom[i+j] != cNext )
					{
						sAttr += aDom[i+j];
						j++;
					}
	
					/* Replace jQuery UI constants @todo depreciated */
					if ( sAttr == "H" )
					{
						sAttr = classes.sJUIHeader;
					}
					else if ( sAttr == "F" )
					{
						sAttr = classes.sJUIFooter;
					}
	
					/* The attribute can be in the format of "#id.class", "#id" or "class" This logic
					 * breaks the string into parts and applies them as needed
					 */
					if ( sAttr.indexOf('.') != -1 )
					{
						var aSplit = sAttr.split('.');
						nNewNode.id = aSplit[0].substr(1, aSplit[0].length-1);
						nNewNode.className = aSplit[1];
					}
					else if ( sAttr.charAt(0) == "#" )
					{
						nNewNode.id = sAttr.substr(1, sAttr.length-1);
					}
					else
					{
						nNewNode.className = sAttr;
					}
	
					i += j; /* Move along the position array */
				}
	
				insert.append( nNewNode );
				insert = $(nNewNode);
			}
			else if ( cOption == '>' )
			{
				/* End container div */
				insert = insert.parent();
			}
			// @todo Move options into their own plugins?
			else if ( cOption == 'l' && features.bPaginate && features.bLengthChange )
			{
				/* Length */
				featureNode = _fnFeatureHtmlLength( oSettings );
			}
			else if ( cOption == 'f' && features.bFilter )
			{
				/* Filter */
				featureNode = _fnFeatureHtmlFilter( oSettings );
			}
			else if ( cOption == 'r' && features.bProcessing )
			{
				/* pRocessing */
				featureNode = _fnFeatureHtmlProcessing( oSettings );
			}
			else if ( cOption == 't' )
			{
				/* Table */
				featureNode = _fnFeatureHtmlTable( oSettings );
			}
			else if ( cOption ==  'i' && features.bInfo )
			{
				/* Info */
				featureNode = _fnFeatureHtmlInfo( oSettings );
			}
			else if ( cOption == 'p' && features.bPaginate )
			{
				/* Pagination */
				featureNode = _fnFeatureHtmlPaginate( oSettings );
			}
			else if ( DataTable.ext.feature.length !== 0 )
			{
				/* Plug-in features */
				var aoFeatures = DataTable.ext.feature;
				for ( var k=0, kLen=aoFeatures.length ; k<kLen ; k++ )
				{
					if ( cOption == aoFeatures[k].cFeature )
					{
						featureNode = aoFeatures[k].fnInit( oSettings );
						break;
					}
				}
			}
	
			/* Add to the 2D features array */
			if ( featureNode )
			{
				var aanFeatures = oSettings.aanFeatures;
	
				if ( ! aanFeatures[cOption] )
				{
					aanFeatures[cOption] = [];
				}
	
				aanFeatures[cOption].push( featureNode );
				insert.append( featureNode );
			}
		}
	
		/* Built our DOM structure - replace the holding div with what we want */
		holding.replaceWith( insert );
		oSettings.nHolding = null;
	}
	
	
	/**
	 * Use the DOM source to create up an array of header cells. The idea here is to
	 * create a layout grid (array) of rows x columns, which contains a reference
	 * to the cell that that point in the grid (regardless of col/rowspan), such that
	 * any column / row could be removed and the new grid constructed
	 *  @param array {object} aLayout Array to store the calculated layout in
	 *  @param {node} nThead The header/footer element for the table
	 *  @memberof DataTable#oApi
	 */
	function _fnDetectHeader ( aLayout, nThead )
	{
		var nTrs = $(nThead).children('tr');
		var nTr, nCell;
		var i, k, l, iLen, jLen, iColShifted, iColumn, iColspan, iRowspan;
		var bUnique;
		var fnShiftCol = function ( a, i, j ) {
			var k = a[i];
	                while ( k[j] ) {
				j++;
			}
			return j;
		};
	
		aLayout.splice( 0, aLayout.length );
	
		/* We know how many rows there are in the layout - so prep it */
		for ( i=0, iLen=nTrs.length ; i<iLen ; i++ )
		{
			aLayout.push( [] );
		}
	
		/* Calculate a layout array */
		for ( i=0, iLen=nTrs.length ; i<iLen ; i++ )
		{
			nTr = nTrs[i];
			iColumn = 0;
	
			/* For every cell in the row... */
			nCell = nTr.firstChild;
			while ( nCell ) {
				if ( nCell.nodeName.toUpperCase() == "TD" ||
				     nCell.nodeName.toUpperCase() == "TH" )
				{
					/* Get the col and rowspan attributes from the DOM and sanitise them */
					iColspan = nCell.getAttribute('colspan') * 1;
					iRowspan = nCell.getAttribute('rowspan') * 1;
					iColspan = (!iColspan || iColspan===0 || iColspan===1) ? 1 : iColspan;
					iRowspan = (!iRowspan || iRowspan===0 || iRowspan===1) ? 1 : iRowspan;
	
					/* There might be colspan cells already in this row, so shift our target
					 * accordingly
					 */
					iColShifted = fnShiftCol( aLayout, i, iColumn );
	
					/* Cache calculation for unique columns */
					bUnique = iColspan === 1 ? true : false;
	
					/* If there is col / rowspan, copy the information into the layout grid */
					for ( l=0 ; l<iColspan ; l++ )
					{
						for ( k=0 ; k<iRowspan ; k++ )
						{
							aLayout[i+k][iColShifted+l] = {
								"cell": nCell,
								"unique": bUnique
							};
							aLayout[i+k].nTr = nTr;
						}
					}
				}
				nCell = nCell.nextSibling;
			}
		}
	}
	
	
	/**
	 * Get an array of unique th elements, one for each column
	 *  @param {object} oSettings dataTables settings object
	 *  @param {node} nHeader automatically detect the layout from this node - optional
	 *  @param {array} aLayout thead/tfoot layout from _fnDetectHeader - optional
	 *  @returns array {node} aReturn list of unique th's
	 *  @memberof DataTable#oApi
	 */
	function _fnGetUniqueThs ( oSettings, nHeader, aLayout )
	{
		var aReturn = [];
		if ( !aLayout )
		{
			aLayout = oSettings.aoHeader;
			if ( nHeader )
			{
				aLayout = [];
				_fnDetectHeader( aLayout, nHeader );
			}
		}
	
		for ( var i=0, iLen=aLayout.length ; i<iLen ; i++ )
		{
			for ( var j=0, jLen=aLayout[i].length ; j<jLen ; j++ )
			{
				if ( aLayout[i][j].unique &&
					 (!aReturn[j] || !oSettings.bSortCellsTop) )
				{
					aReturn[j] = aLayout[i][j].cell;
				}
			}
		}
	
		return aReturn;
	}
	
	/**
	 * Create an Ajax call based on the table's settings, taking into account that
	 * parameters can have multiple forms, and backwards compatibility.
	 *
	 * @param {object} oSettings dataTables settings object
	 * @param {array} data Data to send to the server, required by
	 *     DataTables - may be augmented by developer callbacks
	 * @param {function} fn Callback function to run when data is obtained
	 */
	function _fnBuildAjax( oSettings, data, fn )
	{
		// Compatibility with 1.9-, allow fnServerData and event to manipulate
		_fnCallbackFire( oSettings, 'aoServerParams', 'serverParams', [data] );
	
		// Convert to object based for 1.10+ if using the old array scheme which can
		// come from server-side processing or serverParams
		if ( data && Array.isArray(data) ) {
			var tmp = {};
			var rbracket = /(.*?)\[\]$/;
	
			$.each( data, function (key, val) {
				var match = val.name.match(rbracket);
	
				if ( match ) {
					// Support for arrays
					var name = match[0];
	
					if ( ! tmp[ name ] ) {
						tmp[ name ] = [];
					}
					tmp[ name ].push( val.value );
				}
				else {
					tmp[val.name] = val.value;
				}
			} );
			data = tmp;
		}
	
		var ajaxData;
		var ajax = oSettings.ajax;
		var instance = oSettings.oInstance;
		var callback = function ( json ) {
			_fnCallbackFire( oSettings, null, 'xhr', [oSettings, json, oSettings.jqXHR] );
			fn( json );
		};
	
		if ( $.isPlainObject( ajax ) && ajax.data )
		{
			ajaxData = ajax.data;
	
			var newData = typeof ajaxData === 'function' ?
				ajaxData( data, oSettings ) :  // fn can manipulate data or return
				ajaxData;                      // an object object or array to merge
	
			// If the function returned something, use that alone
			data = typeof ajaxData === 'function' && newData ?
				newData :
				$.extend( true, data, newData );
	
			// Remove the data property as we've resolved it already and don't want
			// jQuery to do it again (it is restored at the end of the function)
			delete ajax.data;
		}
	
		var baseAjax = {
			"data": data,
			"success": function (json) {
				var error = json.error || json.sError;
				if ( error ) {
					_fnLog( oSettings, 0, error );
				}
	
				oSettings.json = json;
				callback( json );
			},
			"dataType": "json",
			"cache": false,
			"type": oSettings.sServerMethod,
			"error": function (xhr, error, thrown) {
				var ret = _fnCallbackFire( oSettings, null, 'xhr', [oSettings, null, oSettings.jqXHR] );
	
				if ( $.inArray( true, ret ) === -1 ) {
					if ( error == "parsererror" ) {
						_fnLog( oSettings, 0, 'Invalid JSON response', 1 );
					}
					else if ( xhr.readyState === 4 ) {
						_fnLog( oSettings, 0, 'Ajax error', 7 );
					}
				}
	
				_fnProcessingDisplay( oSettings, false );
			}
		};
	
		// Store the data submitted for the API
		oSettings.oAjaxData = data;
	
		// Allow plug-ins and external processes to modify the data
		_fnCallbackFire( oSettings, null, 'preXhr', [oSettings, data] );
	
		if ( oSettings.fnServerData )
		{
			// DataTables 1.9- compatibility
			oSettings.fnServerData.call( instance,
				oSettings.sAjaxSource,
				$.map( data, function (val, key) { // Need to convert back to 1.9 trad format
					return { name: key, value: val };
				} ),
				callback,
				oSettings
			);
		}
		else if ( oSettings.sAjaxSource || typeof ajax === 'string' )
		{
			// DataTables 1.9- compatibility
			oSettings.jqXHR = $.ajax( $.extend( baseAjax, {
				url: ajax || oSettings.sAjaxSource
			} ) );
		}
		else if ( typeof ajax === 'function' )
		{
			// Is a function - let the caller define what needs to be done
			oSettings.jqXHR = ajax.call( instance, data, callback, oSettings );
		}
		else
		{
			// Object to extend the base settings
			oSettings.jqXHR = $.ajax( $.extend( baseAjax, ajax ) );
	
			// Restore for next time around
			ajax.data = ajaxData;
		}
	}
	
	
	/**
	 * Update the table using an Ajax call
	 *  @param {object} settings dataTables settings object
	 *  @returns {boolean} Block the table drawing or not
	 *  @memberof DataTable#oApi
	 */
	function _fnAjaxUpdate( settings )
	{
		if ( settings.bAjaxDataGet ) {
			settings.iDraw++;
			_fnProcessingDisplay( settings, true );
	
			_fnBuildAjax(
				settings,
				_fnAjaxParameters( settings ),
				function(json) {
					_fnAjaxUpdateDraw( settings, json );
				}
			);
	
			return false;
		}
		return true;
	}
	
	
	/**
	 * Build up the parameters in an object needed for a server-side processing
	 * request. Note that this is basically done twice, is different ways - a modern
	 * method which is used by default in DataTables 1.10 which uses objects and
	 * arrays, or the 1.9- method with is name / value pairs. 1.9 method is used if
	 * the sAjaxSource option is used in the initialisation, or the legacyAjax
	 * option is set.
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {bool} block the table drawing or not
	 *  @memberof DataTable#oApi
	 */
	function _fnAjaxParameters( settings )
	{
		var
			columns = settings.aoColumns,
			columnCount = columns.length,
			features = settings.oFeatures,
			preSearch = settings.oPreviousSearch,
			preColSearch = settings.aoPreSearchCols,
			i, data = [], dataProp, column, columnSearch,
			sort = _fnSortFlatten( settings ),
			displayStart = settings._iDisplayStart,
			displayLength = features.bPaginate !== false ?
				settings._iDisplayLength :
				-1;
	
		var param = function ( name, value ) {
			data.push( { 'name': name, 'value': value } );
		};
	
		// DataTables 1.9- compatible method
		param( 'sEcho',          settings.iDraw );
		param( 'iColumns',       columnCount );
		param( 'sColumns',       _pluck( columns, 'sName' ).join(',') );
		param( 'iDisplayStart',  displayStart );
		param( 'iDisplayLength', displayLength );
	
		// DataTables 1.10+ method
		var d = {
			draw:    settings.iDraw,
			columns: [],
			order:   [],
			start:   displayStart,
			length:  displayLength,
			search:  {
				value: preSearch.sSearch,
				regex: preSearch.bRegex
			}
		};
	
		for ( i=0 ; i<columnCount ; i++ ) {
			column = columns[i];
			columnSearch = preColSearch[i];
			dataProp = typeof column.mData=="function" ? 'function' : column.mData ;
	
			d.columns.push( {
				data:       dataProp,
				name:       column.sName,
				searchable: column.bSearchable,
				orderable:  column.bSortable,
				search:     {
					value: columnSearch.sSearch,
					regex: columnSearch.bRegex
				}
			} );
	
			param( "mDataProp_"+i, dataProp );
	
			if ( features.bFilter ) {
				param( 'sSearch_'+i,     columnSearch.sSearch );
				param( 'bRegex_'+i,      columnSearch.bRegex );
				param( 'bSearchable_'+i, column.bSearchable );
			}
	
			if ( features.bSort ) {
				param( 'bSortable_'+i, column.bSortable );
			}
		}
	
		if ( features.bFilter ) {
			param( 'sSearch', preSearch.sSearch );
			param( 'bRegex', preSearch.bRegex );
		}
	
		if ( features.bSort ) {
			$.each( sort, function ( i, val ) {
				d.order.push( { column: val.col, dir: val.dir } );
	
				param( 'iSortCol_'+i, val.col );
				param( 'sSortDir_'+i, val.dir );
			} );
	
			param( 'iSortingCols', sort.length );
		}
	
		// If the legacy.ajax parameter is null, then we automatically decide which
		// form to use, based on sAjaxSource
		var legacy = DataTable.ext.legacy.ajax;
		if ( legacy === null ) {
			return settings.sAjaxSource ? data : d;
		}
	
		// Otherwise, if legacy has been specified then we use that to decide on the
		// form
		return legacy ? data : d;
	}
	
	
	/**
	 * Data the data from the server (nuking the old) and redraw the table
	 *  @param {object} oSettings dataTables settings object
	 *  @param {object} json json data return from the server.
	 *  @param {string} json.sEcho Tracking flag for DataTables to match requests
	 *  @param {int} json.iTotalRecords Number of records in the data set, not accounting for filtering
	 *  @param {int} json.iTotalDisplayRecords Number of records in the data set, accounting for filtering
	 *  @param {array} json.aaData The data to display on this page
	 *  @param {string} [json.sColumns] Column ordering (sName, comma separated)
	 *  @memberof DataTable#oApi
	 */
	function _fnAjaxUpdateDraw ( settings, json )
	{
		// v1.10 uses camelCase variables, while 1.9 uses Hungarian notation.
		// Support both
		var compat = function ( old, modern ) {
			return json[old] !== undefined ? json[old] : json[modern];
		};
	
		var data = _fnAjaxDataSrc( settings, json );
		var draw            = compat( 'sEcho',                'draw' );
		var recordsTotal    = compat( 'iTotalRecords',        'recordsTotal' );
		var recordsFiltered = compat( 'iTotalDisplayRecords', 'recordsFiltered' );
	
		if ( draw !== undefined ) {
			// Protect against out of sequence returns
			if ( draw*1 < settings.iDraw ) {
				return;
			}
			settings.iDraw = draw * 1;
		}
	
		_fnClearTable( settings );
		settings._iRecordsTotal   = parseInt(recordsTotal, 10);
		settings._iRecordsDisplay = parseInt(recordsFiltered, 10);
	
		for ( var i=0, ien=data.length ; i<ien ; i++ ) {
			_fnAddData( settings, data[i] );
		}
		settings.aiDisplay = settings.aiDisplayMaster.slice();
	
		settings.bAjaxDataGet = false;
		_fnDraw( settings );
	
		if ( ! settings._bInitComplete ) {
			_fnInitComplete( settings, json );
		}
	
		settings.bAjaxDataGet = true;
		_fnProcessingDisplay( settings, false );
	}
	
	
	/**
	 * Get the data from the JSON data source to use for drawing a table. Using
	 * `_fnGetObjectDataFn` allows the data to be sourced from a property of the
	 * source object, or from a processing function.
	 *  @param {object} oSettings dataTables settings object
	 *  @param  {object} json Data source object / array from the server
	 *  @return {array} Array of data to use
	 */
	function _fnAjaxDataSrc ( oSettings, json )
	{
		var dataSrc = $.isPlainObject( oSettings.ajax ) && oSettings.ajax.dataSrc !== undefined ?
			oSettings.ajax.dataSrc :
			oSettings.sAjaxDataProp; // Compatibility with 1.9-.
	
		// Compatibility with 1.9-. In order to read from aaData, check if the
		// default has been changed, if not, check for aaData
		if ( dataSrc === 'data' ) {
			return json.aaData || json[dataSrc];
		}
	
		return dataSrc !== "" ?
			_fnGetObjectDataFn( dataSrc )( json ) :
			json;
	}
	
	/**
	 * Generate the node required for filtering text
	 *  @returns {node} Filter control element
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlFilter ( settings )
	{
		var classes = settings.oClasses;
		var tableId = settings.sTableId;
		var language = settings.oLanguage;
		var previousSearch = settings.oPreviousSearch;
		var features = settings.aanFeatures;
		var input = '<input type="search" class="'+classes.sFilterInput+'"/>';
	
		var str = language.sSearch;
		str = str.match(/_INPUT_/) ?
			str.replace('_INPUT_', input) :
			str+input;
	
		var filter = $('<div/>', {
				'id': ! features.f ? tableId+'_filter' : null,
				'class': classes.sFilter
			} )
			.append( $('<label/>' ).append( str ) );
	
		var searchFn = function() {
			/* Update all other filter input elements for the new display */
			var n = features.f;
			var val = !this.value ? "" : this.value; // mental IE8 fix :-(
	
			/* Now do the filter */
			if ( val != previousSearch.sSearch ) {
				_fnFilterComplete( settings, {
					"sSearch": val,
					"bRegex": previousSearch.bRegex,
					"bSmart": previousSearch.bSmart ,
					"bCaseInsensitive": previousSearch.bCaseInsensitive
				} );
	
				// Need to redraw, without resorting
				settings._iDisplayStart = 0;
				_fnDraw( settings );
			}
		};
	
		var searchDelay = settings.searchDelay !== null ?
			settings.searchDelay :
			_fnDataSource( settings ) === 'ssp' ?
				400 :
				0;
	
		var jqFilter = $('input', filter)
			.val( previousSearch.sSearch )
			.attr( 'placeholder', language.sSearchPlaceholder )
			.on(
				'keyup.DT search.DT input.DT paste.DT cut.DT',
				searchDelay ?
					_fnThrottle( searchFn, searchDelay ) :
					searchFn
			)
			.on( 'mouseup', function(e) {
				// Edge fix! Edge 17 does not trigger anything other than mouse events when clicking
				// on the clear icon (Edge bug 17584515). This is safe in other browsers as `searchFn`
				// checks the value to see if it has changed. In other browsers it won't have.
				setTimeout( function () {
					searchFn.call(jqFilter[0]);
				}, 10);
			} )
			.on( 'keypress.DT', function(e) {
				/* Prevent form submission */
				if ( e.keyCode == 13 ) {
					return false;
				}
			} )
			.attr('aria-controls', tableId);
	
		// Update the input elements whenever the table is filtered
		$(settings.nTable).on( 'search.dt.DT', function ( ev, s ) {
			if ( settings === s ) {
				// IE9 throws an 'unknown error' if document.activeElement is used
				// inside an iframe or frame...
				try {
					if ( jqFilter[0] !== document.activeElement ) {
						jqFilter.val( previousSearch.sSearch );
					}
				}
				catch ( e ) {}
			}
		} );
	
		return filter[0];
	}
	
	
	/**
	 * Filter the table using both the global filter and column based filtering
	 *  @param {object} oSettings dataTables settings object
	 *  @param {object} oSearch search information
	 *  @param {int} [iForce] force a research of the master array (1) or not (undefined or 0)
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterComplete ( oSettings, oInput, iForce )
	{
		var oPrevSearch = oSettings.oPreviousSearch;
		var aoPrevSearch = oSettings.aoPreSearchCols;
		var fnSaveFilter = function ( oFilter ) {
			/* Save the filtering values */
			oPrevSearch.sSearch = oFilter.sSearch;
			oPrevSearch.bRegex = oFilter.bRegex;
			oPrevSearch.bSmart = oFilter.bSmart;
			oPrevSearch.bCaseInsensitive = oFilter.bCaseInsensitive;
		};
		var fnRegex = function ( o ) {
			// Backwards compatibility with the bEscapeRegex option
			return o.bEscapeRegex !== undefined ? !o.bEscapeRegex : o.bRegex;
		};
	
		// Resolve any column types that are unknown due to addition or invalidation
		// @todo As per sort - can this be moved into an event handler?
		_fnColumnTypes( oSettings );
	
		/* In server-side processing all filtering is done by the server, so no point hanging around here */
		if ( _fnDataSource( oSettings ) != 'ssp' )
		{
			/* Global filter */
			_fnFilter( oSettings, oInput.sSearch, iForce, fnRegex(oInput), oInput.bSmart, oInput.bCaseInsensitive );
			fnSaveFilter( oInput );
	
			/* Now do the individual column filter */
			for ( var i=0 ; i<aoPrevSearch.length ; i++ )
			{
				_fnFilterColumn( oSettings, aoPrevSearch[i].sSearch, i, fnRegex(aoPrevSearch[i]),
					aoPrevSearch[i].bSmart, aoPrevSearch[i].bCaseInsensitive );
			}
	
			/* Custom filtering */
			_fnFilterCustom( oSettings );
		}
		else
		{
			fnSaveFilter( oInput );
		}
	
		/* Tell the draw function we have been filtering */
		oSettings.bFiltered = true;
		_fnCallbackFire( oSettings, null, 'search', [oSettings] );
	}
	
	
	/**
	 * Apply custom filtering functions
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterCustom( settings )
	{
		var filters = DataTable.ext.search;
		var displayRows = settings.aiDisplay;
		var row, rowIdx;
	
		for ( var i=0, ien=filters.length ; i<ien ; i++ ) {
			var rows = [];
	
			// Loop over each row and see if it should be included
			for ( var j=0, jen=displayRows.length ; j<jen ; j++ ) {
				rowIdx = displayRows[ j ];
				row = settings.aoData[ rowIdx ];
	
				if ( filters[i]( settings, row._aFilterData, rowIdx, row._aData, j ) ) {
					rows.push( rowIdx );
				}
			}
	
			// So the array reference doesn't break set the results into the
			// existing array
			displayRows.length = 0;
			$.merge( displayRows, rows );
		}
	}
	
	
	/**
	 * Filter the table on a per-column basis
	 *  @param {object} oSettings dataTables settings object
	 *  @param {string} sInput string to filter on
	 *  @param {int} iColumn column to filter
	 *  @param {bool} bRegex treat search string as a regular expression or not
	 *  @param {bool} bSmart use smart filtering or not
	 *  @param {bool} bCaseInsensitive Do case insenstive matching or not
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterColumn ( settings, searchStr, colIdx, regex, smart, caseInsensitive )
	{
		if ( searchStr === '' ) {
			return;
		}
	
		var data;
		var out = [];
		var display = settings.aiDisplay;
		var rpSearch = _fnFilterCreateSearch( searchStr, regex, smart, caseInsensitive );
	
		for ( var i=0 ; i<display.length ; i++ ) {
			data = settings.aoData[ display[i] ]._aFilterData[ colIdx ];
	
			if ( rpSearch.test( data ) ) {
				out.push( display[i] );
			}
		}
	
		settings.aiDisplay = out;
	}
	
	
	/**
	 * Filter the data table based on user input and draw the table
	 *  @param {object} settings dataTables settings object
	 *  @param {string} input string to filter on
	 *  @param {int} force optional - force a research of the master array (1) or not (undefined or 0)
	 *  @param {bool} regex treat as a regular expression or not
	 *  @param {bool} smart perform smart filtering or not
	 *  @param {bool} caseInsensitive Do case insenstive matching or not
	 *  @memberof DataTable#oApi
	 */
	function _fnFilter( settings, input, force, regex, smart, caseInsensitive )
	{
		var rpSearch = _fnFilterCreateSearch( input, regex, smart, caseInsensitive );
		var prevSearch = settings.oPreviousSearch.sSearch;
		var displayMaster = settings.aiDisplayMaster;
		var display, invalidated, i;
		var filtered = [];
	
		// Need to take account of custom filtering functions - always filter
		if ( DataTable.ext.search.length !== 0 ) {
			force = true;
		}
	
		// Check if any of the rows were invalidated
		invalidated = _fnFilterData( settings );
	
		// If the input is blank - we just want the full data set
		if ( input.length <= 0 ) {
			settings.aiDisplay = displayMaster.slice();
		}
		else {
			// New search - start from the master array
			if ( invalidated ||
				 force ||
				 regex ||
				 prevSearch.length > input.length ||
				 input.indexOf(prevSearch) !== 0 ||
				 settings.bSorted // On resort, the display master needs to be
				                  // re-filtered since indexes will have changed
			) {
				settings.aiDisplay = displayMaster.slice();
			}
	
			// Search the display array
			display = settings.aiDisplay;
	
			for ( i=0 ; i<display.length ; i++ ) {
				if ( rpSearch.test( settings.aoData[ display[i] ]._sFilterRow ) ) {
					filtered.push( display[i] );
				}
			}
	
			settings.aiDisplay = filtered;
		}
	}
	
	
	/**
	 * Build a regular expression object suitable for searching a table
	 *  @param {string} sSearch string to search for
	 *  @param {bool} bRegex treat as a regular expression or not
	 *  @param {bool} bSmart perform smart filtering or not
	 *  @param {bool} bCaseInsensitive Do case insensitive matching or not
	 *  @returns {RegExp} constructed object
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterCreateSearch( search, regex, smart, caseInsensitive )
	{
		search = regex ?
			search :
			_fnEscapeRegex( search );
		
		if ( smart ) {
			/* For smart filtering we want to allow the search to work regardless of
			 * word order. We also want double quoted text to be preserved, so word
			 * order is important - a la google. So this is what we want to
			 * generate:
			 * 
			 * ^(?=.*?\bone\b)(?=.*?\btwo three\b)(?=.*?\bfour\b).*$
			 */
			var a = $.map( search.match( /"[^"]+"|[^ ]+/g ) || [''], function ( word ) {
				if ( word.charAt(0) === '"' ) {
					var m = word.match( /^"(.*)"$/ );
					word = m ? m[1] : word;
				}
	
				return word.replace('"', '');
			} );
	
			search = '^(?=.*?'+a.join( ')(?=.*?' )+').*$';
		}
	
		return new RegExp( search, caseInsensitive ? 'i' : '' );
	}
	
	
	/**
	 * Escape a string such that it can be used in a regular expression
	 *  @param {string} sVal string to escape
	 *  @returns {string} escaped string
	 *  @memberof DataTable#oApi
	 */
	var _fnEscapeRegex = DataTable.util.escapeRegex;
	
	var __filter_div = $('<div>')[0];
	var __filter_div_textContent = __filter_div.textContent !== undefined;
	
	// Update the filtering data for each row if needed (by invalidation or first run)
	function _fnFilterData ( settings )
	{
		var columns = settings.aoColumns;
		var column;
		var i, j, ien, jen, filterData, cellData, row;
		var fomatters = DataTable.ext.type.search;
		var wasInvalidated = false;
	
		for ( i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
			row = settings.aoData[i];
	
			if ( ! row._aFilterData ) {
				filterData = [];
	
				for ( j=0, jen=columns.length ; j<jen ; j++ ) {
					column = columns[j];
	
					if ( column.bSearchable ) {
						cellData = _fnGetCellData( settings, i, j, 'filter' );
	
						if ( fomatters[ column.sType ] ) {
							cellData = fomatters[ column.sType ]( cellData );
						}
	
						// Search in DataTables 1.10 is string based. In 1.11 this
						// should be altered to also allow strict type checking.
						if ( cellData === null ) {
							cellData = '';
						}
	
						if ( typeof cellData !== 'string' && cellData.toString ) {
							cellData = cellData.toString();
						}
					}
					else {
						cellData = '';
					}
	
					// If it looks like there is an HTML entity in the string,
					// attempt to decode it so sorting works as expected. Note that
					// we could use a single line of jQuery to do this, but the DOM
					// method used here is much faster http://jsperf.com/html-decode
					if ( cellData.indexOf && cellData.indexOf('&') !== -1 ) {
						__filter_div.innerHTML = cellData;
						cellData = __filter_div_textContent ?
							__filter_div.textContent :
							__filter_div.innerText;
					}
	
					if ( cellData.replace ) {
						cellData = cellData.replace(/[\r\n\u2028]/g, '');
					}
	
					filterData.push( cellData );
				}
	
				row._aFilterData = filterData;
				row._sFilterRow = filterData.join('  ');
				wasInvalidated = true;
			}
		}
	
		return wasInvalidated;
	}
	
	
	/**
	 * Convert from the internal Hungarian notation to camelCase for external
	 * interaction
	 *  @param {object} obj Object to convert
	 *  @returns {object} Inverted object
	 *  @memberof DataTable#oApi
	 */
	function _fnSearchToCamel ( obj )
	{
		return {
			search:          obj.sSearch,
			smart:           obj.bSmart,
			regex:           obj.bRegex,
			caseInsensitive: obj.bCaseInsensitive
		};
	}
	
	
	
	/**
	 * Convert from camelCase notation to the internal Hungarian. We could use the
	 * Hungarian convert function here, but this is cleaner
	 *  @param {object} obj Object to convert
	 *  @returns {object} Inverted object
	 *  @memberof DataTable#oApi
	 */
	function _fnSearchToHung ( obj )
	{
		return {
			sSearch:          obj.search,
			bSmart:           obj.smart,
			bRegex:           obj.regex,
			bCaseInsensitive: obj.caseInsensitive
		};
	}
	
	/**
	 * Generate the node required for the info display
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {node} Information element
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlInfo ( settings )
	{
		var
			tid = settings.sTableId,
			nodes = settings.aanFeatures.i,
			n = $('<div/>', {
				'class': settings.oClasses.sInfo,
				'id': ! nodes ? tid+'_info' : null
			} );
	
		if ( ! nodes ) {
			// Update display on each draw
			settings.aoDrawCallback.push( {
				"fn": _fnUpdateInfo,
				"sName": "information"
			} );
	
			n
				.attr( 'role', 'status' )
				.attr( 'aria-live', 'polite' );
	
			// Table is described by our info div
			$(settings.nTable).attr( 'aria-describedby', tid+'_info' );
		}
	
		return n[0];
	}
	
	
	/**
	 * Update the information elements in the display
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnUpdateInfo ( settings )
	{
		/* Show information about the table */
		var nodes = settings.aanFeatures.i;
		if ( nodes.length === 0 ) {
			return;
		}
	
		var
			lang  = settings.oLanguage,
			start = settings._iDisplayStart+1,
			end   = settings.fnDisplayEnd(),
			max   = settings.fnRecordsTotal(),
			total = settings.fnRecordsDisplay(),
			out   = total ?
				lang.sInfo :
				lang.sInfoEmpty;
	
		if ( total !== max ) {
			/* Record set after filtering */
			out += ' ' + lang.sInfoFiltered;
		}
	
		// Convert the macros
		out += lang.sInfoPostFix;
		out = _fnInfoMacros( settings, out );
	
		var callback = lang.fnInfoCallback;
		if ( callback !== null ) {
			out = callback.call( settings.oInstance,
				settings, start, end, max, total, out
			);
		}
	
		$(nodes).html( out );
	}
	
	
	function _fnInfoMacros ( settings, str )
	{
		// When infinite scrolling, we are always starting at 1. _iDisplayStart is used only
		// internally
		var
			formatter  = settings.fnFormatNumber,
			start      = settings._iDisplayStart+1,
			len        = settings._iDisplayLength,
			vis        = settings.fnRecordsDisplay(),
			all        = len === -1;
	
		return str.
			replace(/_START_/g, formatter.call( settings, start ) ).
			replace(/_END_/g,   formatter.call( settings, settings.fnDisplayEnd() ) ).
			replace(/_MAX_/g,   formatter.call( settings, settings.fnRecordsTotal() ) ).
			replace(/_TOTAL_/g, formatter.call( settings, vis ) ).
			replace(/_PAGE_/g,  formatter.call( settings, all ? 1 : Math.ceil( start / len ) ) ).
			replace(/_PAGES_/g, formatter.call( settings, all ? 1 : Math.ceil( vis / len ) ) );
	}
	
	
	
	/**
	 * Draw the table for the first time, adding all required features
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnInitialise ( settings )
	{
		var i, iLen, iAjaxStart=settings.iInitDisplayStart;
		var columns = settings.aoColumns, column;
		var features = settings.oFeatures;
		var deferLoading = settings.bDeferLoading; // value modified by the draw
	
		/* Ensure that the table data is fully initialised */
		if ( ! settings.bInitialised ) {
			setTimeout( function(){ _fnInitialise( settings ); }, 200 );
			return;
		}
	
		/* Show the display HTML options */
		_fnAddOptionsHtml( settings );
	
		/* Build and draw the header / footer for the table */
		_fnBuildHead( settings );
		_fnDrawHead( settings, settings.aoHeader );
		_fnDrawHead( settings, settings.aoFooter );
	
		/* Okay to show that something is going on now */
		_fnProcessingDisplay( settings, true );
	
		/* Calculate sizes for columns */
		if ( features.bAutoWidth ) {
			_fnCalculateColumnWidths( settings );
		}
	
		for ( i=0, iLen=columns.length ; i<iLen ; i++ ) {
			column = columns[i];
	
			if ( column.sWidth ) {
				column.nTh.style.width = _fnStringToCss( column.sWidth );
			}
		}
	
		_fnCallbackFire( settings, null, 'preInit', [settings] );
	
		// If there is default sorting required - let's do it. The sort function
		// will do the drawing for us. Otherwise we draw the table regardless of the
		// Ajax source - this allows the table to look initialised for Ajax sourcing
		// data (show 'loading' message possibly)
		_fnReDraw( settings );
	
		// Server-side processing init complete is done by _fnAjaxUpdateDraw
		var dataSrc = _fnDataSource( settings );
		if ( dataSrc != 'ssp' || deferLoading ) {
			// if there is an ajax source load the data
			if ( dataSrc == 'ajax' ) {
				_fnBuildAjax( settings, [], function(json) {
					var aData = _fnAjaxDataSrc( settings, json );
	
					// Got the data - add it to the table
					for ( i=0 ; i<aData.length ; i++ ) {
						_fnAddData( settings, aData[i] );
					}
	
					// Reset the init display for cookie saving. We've already done
					// a filter, and therefore cleared it before. So we need to make
					// it appear 'fresh'
					settings.iInitDisplayStart = iAjaxStart;
	
					_fnReDraw( settings );
	
					_fnProcessingDisplay( settings, false );
					_fnInitComplete( settings, json );
				}, settings );
			}
			else {
				_fnProcessingDisplay( settings, false );
				_fnInitComplete( settings );
			}
		}
	}
	
	
	/**
	 * Draw the table for the first time, adding all required features
	 *  @param {object} oSettings dataTables settings object
	 *  @param {object} [json] JSON from the server that completed the table, if using Ajax source
	 *    with client-side processing (optional)
	 *  @memberof DataTable#oApi
	 */
	function _fnInitComplete ( settings, json )
	{
		settings._bInitComplete = true;
	
		// When data was added after the initialisation (data or Ajax) we need to
		// calculate the column sizing
		if ( json || settings.oInit.aaData ) {
			_fnAdjustColumnSizing( settings );
		}
	
		_fnCallbackFire( settings, null, 'plugin-init', [settings, json] );
		_fnCallbackFire( settings, 'aoInitComplete', 'init', [settings, json] );
	}
	
	
	function _fnLengthChange ( settings, val )
	{
		var len = parseInt( val, 10 );
		settings._iDisplayLength = len;
	
		_fnLengthOverflow( settings );
	
		// Fire length change event
		_fnCallbackFire( settings, null, 'length', [settings, len] );
	}
	
	
	/**
	 * Generate the node required for user display length changing
	 *  @param {object} settings dataTables settings object
	 *  @returns {node} Display length feature node
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlLength ( settings )
	{
		var
			classes  = settings.oClasses,
			tableId  = settings.sTableId,
			menu     = settings.aLengthMenu,
			d2       = Array.isArray( menu[0] ),
			lengths  = d2 ? menu[0] : menu,
			language = d2 ? menu[1] : menu;
	
		var select = $('<select/>', {
			'name':          tableId+'_length',
			'aria-controls': tableId,
			'class':         classes.sLengthSelect
		} );
	
		for ( var i=0, ien=lengths.length ; i<ien ; i++ ) {
			select[0][ i ] = new Option(
				typeof language[i] === 'number' ?
					settings.fnFormatNumber( language[i] ) :
					language[i],
				lengths[i]
			);
		}
	
		var div = $('<div><label/></div>').addClass( classes.sLength );
		if ( ! settings.aanFeatures.l ) {
			div[0].id = tableId+'_length';
		}
	
		div.children().append(
			settings.oLanguage.sLengthMenu.replace( '_MENU_', select[0].outerHTML )
		);
	
		// Can't use `select` variable as user might provide their own and the
		// reference is broken by the use of outerHTML
		$('select', div)
			.val( settings._iDisplayLength )
			.on( 'change.DT', function(e) {
				_fnLengthChange( settings, $(this).val() );
				_fnDraw( settings );
			} );
	
		// Update node value whenever anything changes the table's length
		$(settings.nTable).on( 'length.dt.DT', function (e, s, len) {
			if ( settings === s ) {
				$('select', div).val( len );
			}
		} );
	
		return div[0];
	}
	
	
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 * Note that most of the paging logic is done in
	 * DataTable.ext.pager
	 */
	
	/**
	 * Generate the node required for default pagination
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {node} Pagination feature node
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlPaginate ( settings )
	{
		var
			type   = settings.sPaginationType,
			plugin = DataTable.ext.pager[ type ],
			modern = typeof plugin === 'function',
			redraw = function( settings ) {
				_fnDraw( settings );
			},
			node = $('<div/>').addClass( settings.oClasses.sPaging + type )[0],
			features = settings.aanFeatures;
	
		if ( ! modern ) {
			plugin.fnInit( settings, node, redraw );
		}
	
		/* Add a draw callback for the pagination on first instance, to update the paging display */
		if ( ! features.p )
		{
			node.id = settings.sTableId+'_paginate';
	
			settings.aoDrawCallback.push( {
				"fn": function( settings ) {
					if ( modern ) {
						var
							start      = settings._iDisplayStart,
							len        = settings._iDisplayLength,
							visRecords = settings.fnRecordsDisplay(),
							all        = len === -1,
							page = all ? 0 : Math.ceil( start / len ),
							pages = all ? 1 : Math.ceil( visRecords / len ),
							buttons = plugin(page, pages),
							i, ien;
	
						for ( i=0, ien=features.p.length ; i<ien ; i++ ) {
							_fnRenderer( settings, 'pageButton' )(
								settings, features.p[i], i, buttons, page, pages
							);
						}
					}
					else {
						plugin.fnUpdate( settings, redraw );
					}
				},
				"sName": "pagination"
			} );
		}
	
		return node;
	}
	
	
	/**
	 * Alter the display settings to change the page
	 *  @param {object} settings DataTables settings object
	 *  @param {string|int} action Paging action to take: "first", "previous",
	 *    "next" or "last" or page number to jump to (integer)
	 *  @param [bool] redraw Automatically draw the update or not
	 *  @returns {bool} true page has changed, false - no change
	 *  @memberof DataTable#oApi
	 */
	function _fnPageChange ( settings, action, redraw )
	{
		var
			start     = settings._iDisplayStart,
			len       = settings._iDisplayLength,
			records   = settings.fnRecordsDisplay();
	
		if ( records === 0 || len === -1 )
		{
			start = 0;
		}
		else if ( typeof action === "number" )
		{
			start = action * len;
	
			if ( start > records )
			{
				start = 0;
			}
		}
		else if ( action == "first" )
		{
			start = 0;
		}
		else if ( action == "previous" )
		{
			start = len >= 0 ?
				start - len :
				0;
	
			if ( start < 0 )
			{
			  start = 0;
			}
		}
		else if ( action == "next" )
		{
			if ( start + len < records )
			{
				start += len;
			}
		}
		else if ( action == "last" )
		{
			start = Math.floor( (records-1) / len) * len;
		}
		else
		{
			_fnLog( settings, 0, "Unknown paging action: "+action, 5 );
		}
	
		var changed = settings._iDisplayStart !== start;
		settings._iDisplayStart = start;
	
		if ( changed ) {
			_fnCallbackFire( settings, null, 'page', [settings] );
	
			if ( redraw ) {
				_fnDraw( settings );
			}
		}
	
		return changed;
	}
	
	
	
	/**
	 * Generate the node required for the processing node
	 *  @param {object} settings dataTables settings object
	 *  @returns {node} Processing element
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlProcessing ( settings )
	{
		return $('<div/>', {
				'id': ! settings.aanFeatures.r ? settings.sTableId+'_processing' : null,
				'class': settings.oClasses.sProcessing
			} )
			.html( settings.oLanguage.sProcessing )
			.insertBefore( settings.nTable )[0];
	}
	
	
	/**
	 * Display or hide the processing indicator
	 *  @param {object} settings dataTables settings object
	 *  @param {bool} show Show the processing indicator (true) or not (false)
	 *  @memberof DataTable#oApi
	 */
	function _fnProcessingDisplay ( settings, show )
	{
		if ( settings.oFeatures.bProcessing ) {
			$(settings.aanFeatures.r).css( 'display', show ? 'block' : 'none' );
		}
	
		_fnCallbackFire( settings, null, 'processing', [settings, show] );
	}
	
	/**
	 * Add any control elements for the table - specifically scrolling
	 *  @param {object} settings dataTables settings object
	 *  @returns {node} Node to add to the DOM
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlTable ( settings )
	{
		var table = $(settings.nTable);
	
		// Add the ARIA grid role to the table
		table.attr( 'role', 'grid' );
	
		// Scrolling from here on in
		var scroll = settings.oScroll;
	
		if ( scroll.sX === '' && scroll.sY === '' ) {
			return settings.nTable;
		}
	
		var scrollX = scroll.sX;
		var scrollY = scroll.sY;
		var classes = settings.oClasses;
		var caption = table.children('caption');
		var captionSide = caption.length ? caption[0]._captionSide : null;
		var headerClone = $( table[0].cloneNode(false) );
		var footerClone = $( table[0].cloneNode(false) );
		var footer = table.children('tfoot');
		var _div = '<div/>';
		var size = function ( s ) {
			return !s ? null : _fnStringToCss( s );
		};
	
		if ( ! footer.length ) {
			footer = null;
		}
	
		/*
		 * The HTML structure that we want to generate in this function is:
		 *  div - scroller
		 *    div - scroll head
		 *      div - scroll head inner
		 *        table - scroll head table
		 *          thead - thead
		 *    div - scroll body
		 *      table - table (master table)
		 *        thead - thead clone for sizing
		 *        tbody - tbody
		 *    div - scroll foot
		 *      div - scroll foot inner
		 *        table - scroll foot table
		 *          tfoot - tfoot
		 */
		var scroller = $( _div, { 'class': classes.sScrollWrapper } )
			.append(
				$(_div, { 'class': classes.sScrollHead } )
					.css( {
						overflow: 'hidden',
						position: 'relative',
						border: 0,
						width: scrollX ? size(scrollX) : '100%'
					} )
					.append(
						$(_div, { 'class': classes.sScrollHeadInner } )
							.css( {
								'box-sizing': 'content-box',
								width: scroll.sXInner || '100%'
							} )
							.append(
								headerClone
									.removeAttr('id')
									.css( 'margin-left', 0 )
									.append( captionSide === 'top' ? caption : null )
									.append(
										table.children('thead')
									)
							)
					)
			)
			.append(
				$(_div, { 'class': classes.sScrollBody } )
					.css( {
						position: 'relative',
						overflow: 'auto',
						width: size( scrollX )
					} )
					.append( table )
			);
	
		if ( footer ) {
			scroller.append(
				$(_div, { 'class': classes.sScrollFoot } )
					.css( {
						overflow: 'hidden',
						border: 0,
						width: scrollX ? size(scrollX) : '100%'
					} )
					.append(
						$(_div, { 'class': classes.sScrollFootInner } )
							.append(
								footerClone
									.removeAttr('id')
									.css( 'margin-left', 0 )
									.append( captionSide === 'bottom' ? caption : null )
									.append(
										table.children('tfoot')
									)
							)
					)
			);
		}
	
		var children = scroller.children();
		var scrollHead = children[0];
		var scrollBody = children[1];
		var scrollFoot = footer ? children[2] : null;
	
		// When the body is scrolled, then we also want to scroll the headers
		if ( scrollX ) {
			$(scrollBody).on( 'scroll.DT', function (e) {
				var scrollLeft = this.scrollLeft;
	
				scrollHead.scrollLeft = scrollLeft;
	
				if ( footer ) {
					scrollFoot.scrollLeft = scrollLeft;
				}
			} );
		}
	
		$(scrollBody).css('max-height', scrollY);
		if (! scroll.bCollapse) {
			$(scrollBody).css('height', scrollY);
		}
	
		settings.nScrollHead = scrollHead;
		settings.nScrollBody = scrollBody;
		settings.nScrollFoot = scrollFoot;
	
		// On redraw - align columns
		settings.aoDrawCallback.push( {
			"fn": _fnScrollDraw,
			"sName": "scrolling"
		} );
	
		return scroller[0];
	}
	
	
	
	/**
	 * Update the header, footer and body tables for resizing - i.e. column
	 * alignment.
	 *
	 * Welcome to the most horrible function DataTables. The process that this
	 * function follows is basically:
	 *   1. Re-create the table inside the scrolling div
	 *   2. Take live measurements from the DOM
	 *   3. Apply the measurements to align the columns
	 *   4. Clean up
	 *
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnScrollDraw ( settings )
	{
		// Given that this is such a monster function, a lot of variables are use
		// to try and keep the minimised size as small as possible
		var
			scroll         = settings.oScroll,
			scrollX        = scroll.sX,
			scrollXInner   = scroll.sXInner,
			scrollY        = scroll.sY,
			barWidth       = scroll.iBarWidth,
			divHeader      = $(settings.nScrollHead),
			divHeaderStyle = divHeader[0].style,
			divHeaderInner = divHeader.children('div'),
			divHeaderInnerStyle = divHeaderInner[0].style,
			divHeaderTable = divHeaderInner.children('table'),
			divBodyEl      = settings.nScrollBody,
			divBody        = $(divBodyEl),
			divBodyStyle   = divBodyEl.style,
			divFooter      = $(settings.nScrollFoot),
			divFooterInner = divFooter.children('div'),
			divFooterTable = divFooterInner.children('table'),
			header         = $(settings.nTHead),
			table          = $(settings.nTable),
			tableEl        = table[0],
			tableStyle     = tableEl.style,
			footer         = settings.nTFoot ? $(settings.nTFoot) : null,
			browser        = settings.oBrowser,
			ie67           = browser.bScrollOversize,
			dtHeaderCells  = _pluck( settings.aoColumns, 'nTh' ),
			headerTrgEls, footerTrgEls,
			headerSrcEls, footerSrcEls,
			headerCopy, footerCopy,
			headerWidths=[], footerWidths=[],
			headerContent=[], footerContent=[],
			idx, correction, sanityWidth,
			zeroOut = function(nSizer) {
				var style = nSizer.style;
				style.paddingTop = "0";
				style.paddingBottom = "0";
				style.borderTopWidth = "0";
				style.borderBottomWidth = "0";
				style.height = 0;
			};
	
		// If the scrollbar visibility has changed from the last draw, we need to
		// adjust the column sizes as the table width will have changed to account
		// for the scrollbar
		var scrollBarVis = divBodyEl.scrollHeight > divBodyEl.clientHeight;
		
		if ( settings.scrollBarVis !== scrollBarVis && settings.scrollBarVis !== undefined ) {
			settings.scrollBarVis = scrollBarVis;
			_fnAdjustColumnSizing( settings );
			return; // adjust column sizing will call this function again
		}
		else {
			settings.scrollBarVis = scrollBarVis;
		}
	
		/*
		 * 1. Re-create the table inside the scrolling div
		 */
	
		// Remove the old minimised thead and tfoot elements in the inner table
		table.children('thead, tfoot').remove();
	
		if ( footer ) {
			footerCopy = footer.clone().prependTo( table );
			footerTrgEls = footer.find('tr'); // the original tfoot is in its own table and must be sized
			footerSrcEls = footerCopy.find('tr');
		}
	
		// Clone the current header and footer elements and then place it into the inner table
		headerCopy = header.clone().prependTo( table );
		headerTrgEls = header.find('tr'); // original header is in its own table
		headerSrcEls = headerCopy.find('tr');
		headerCopy.find('th, td').removeAttr('tabindex');
	
	
		/*
		 * 2. Take live measurements from the DOM - do not alter the DOM itself!
		 */
	
		// Remove old sizing and apply the calculated column widths
		// Get the unique column headers in the newly created (cloned) header. We want to apply the
		// calculated sizes to this header
		if ( ! scrollX )
		{
			divBodyStyle.width = '100%';
			divHeader[0].style.width = '100%';
		}
	
		$.each( _fnGetUniqueThs( settings, headerCopy ), function ( i, el ) {
			idx = _fnVisibleToColumnIndex( settings, i );
			el.style.width = settings.aoColumns[idx].sWidth;
		} );
	
		if ( footer ) {
			_fnApplyToChildren( function(n) {
				n.style.width = "";
			}, footerSrcEls );
		}
	
		// Size the table as a whole
		sanityWidth = table.outerWidth();
		if ( scrollX === "" ) {
			// No x scrolling
			tableStyle.width = "100%";
	
			// IE7 will make the width of the table when 100% include the scrollbar
			// - which is shouldn't. When there is a scrollbar we need to take this
			// into account.
			if ( ie67 && (table.find('tbody').height() > divBodyEl.offsetHeight ||
				divBody.css('overflow-y') == "scroll")
			) {
				tableStyle.width = _fnStringToCss( table.outerWidth() - barWidth);
			}
	
			// Recalculate the sanity width
			sanityWidth = table.outerWidth();
		}
		else if ( scrollXInner !== "" ) {
			// legacy x scroll inner has been given - use it
			tableStyle.width = _fnStringToCss(scrollXInner);
	
			// Recalculate the sanity width
			sanityWidth = table.outerWidth();
		}
	
		// Hidden header should have zero height, so remove padding and borders. Then
		// set the width based on the real headers
	
		// Apply all styles in one pass
		_fnApplyToChildren( zeroOut, headerSrcEls );
	
		// Read all widths in next pass
		_fnApplyToChildren( function(nSizer) {
			headerContent.push( nSizer.innerHTML );
			headerWidths.push( _fnStringToCss( $(nSizer).css('width') ) );
		}, headerSrcEls );
	
		// Apply all widths in final pass
		_fnApplyToChildren( function(nToSize, i) {
			// Only apply widths to the DataTables detected header cells - this
			// prevents complex headers from having contradictory sizes applied
			if ( $.inArray( nToSize, dtHeaderCells ) !== -1 ) {
				nToSize.style.width = headerWidths[i];
			}
		}, headerTrgEls );
	
		$(headerSrcEls).height(0);
	
		/* Same again with the footer if we have one */
		if ( footer )
		{
			_fnApplyToChildren( zeroOut, footerSrcEls );
	
			_fnApplyToChildren( function(nSizer) {
				footerContent.push( nSizer.innerHTML );
				footerWidths.push( _fnStringToCss( $(nSizer).css('width') ) );
			}, footerSrcEls );
	
			_fnApplyToChildren( function(nToSize, i) {
				nToSize.style.width = footerWidths[i];
			}, footerTrgEls );
	
			$(footerSrcEls).height(0);
		}
	
	
		/*
		 * 3. Apply the measurements
		 */
	
		// "Hide" the header and footer that we used for the sizing. We need to keep
		// the content of the cell so that the width applied to the header and body
		// both match, but we want to hide it completely. We want to also fix their
		// width to what they currently are
		_fnApplyToChildren( function(nSizer, i) {
			nSizer.innerHTML = '<div class="dataTables_sizing">'+headerContent[i]+'</div>';
			nSizer.childNodes[0].style.height = "0";
			nSizer.childNodes[0].style.overflow = "hidden";
			nSizer.style.width = headerWidths[i];
		}, headerSrcEls );
	
		if ( footer )
		{
			_fnApplyToChildren( function(nSizer, i) {
				nSizer.innerHTML = '<div class="dataTables_sizing">'+footerContent[i]+'</div>';
				nSizer.childNodes[0].style.height = "0";
				nSizer.childNodes[0].style.overflow = "hidden";
				nSizer.style.width = footerWidths[i];
			}, footerSrcEls );
		}
	
		// Sanity check that the table is of a sensible width. If not then we are going to get
		// misalignment - try to prevent this by not allowing the table to shrink below its min width
		if ( table.outerWidth() < sanityWidth )
		{
			// The min width depends upon if we have a vertical scrollbar visible or not */
			correction = ((divBodyEl.scrollHeight > divBodyEl.offsetHeight ||
				divBody.css('overflow-y') == "scroll")) ?
					sanityWidth+barWidth :
					sanityWidth;
	
			// IE6/7 are a law unto themselves...
			if ( ie67 && (divBodyEl.scrollHeight >
				divBodyEl.offsetHeight || divBody.css('overflow-y') == "scroll")
			) {
				tableStyle.width = _fnStringToCss( correction-barWidth );
			}
	
			// And give the user a warning that we've stopped the table getting too small
			if ( scrollX === "" || scrollXInner !== "" ) {
				_fnLog( settings, 1, 'Possible column misalignment', 6 );
			}
		}
		else
		{
			correction = '100%';
		}
	
		// Apply to the container elements
		divBodyStyle.width = _fnStringToCss( correction );
		divHeaderStyle.width = _fnStringToCss( correction );
	
		if ( footer ) {
			settings.nScrollFoot.style.width = _fnStringToCss( correction );
		}
	
	
		/*
		 * 4. Clean up
		 */
		if ( ! scrollY ) {
			/* IE7< puts a vertical scrollbar in place (when it shouldn't be) due to subtracting
			 * the scrollbar height from the visible display, rather than adding it on. We need to
			 * set the height in order to sort this. Don't want to do it in any other browsers.
			 */
			if ( ie67 ) {
				divBodyStyle.height = _fnStringToCss( tableEl.offsetHeight+barWidth );
			}
		}
	
		/* Finally set the width's of the header and footer tables */
		var iOuterWidth = table.outerWidth();
		divHeaderTable[0].style.width = _fnStringToCss( iOuterWidth );
		divHeaderInnerStyle.width = _fnStringToCss( iOuterWidth );
	
		// Figure out if there are scrollbar present - if so then we need a the header and footer to
		// provide a bit more space to allow "overflow" scrolling (i.e. past the scrollbar)
		var bScrolling = table.height() > divBodyEl.clientHeight || divBody.css('overflow-y') == "scroll";
		var padding = 'padding' + (browser.bScrollbarLeft ? 'Left' : 'Right' );
		divHeaderInnerStyle[ padding ] = bScrolling ? barWidth+"px" : "0px";
	
		if ( footer ) {
			divFooterTable[0].style.width = _fnStringToCss( iOuterWidth );
			divFooterInner[0].style.width = _fnStringToCss( iOuterWidth );
			divFooterInner[0].style[padding] = bScrolling ? barWidth+"px" : "0px";
		}
	
		// Correct DOM ordering for colgroup - comes before the thead
		table.children('colgroup').insertBefore( table.children('thead') );
	
		/* Adjust the position of the header in case we loose the y-scrollbar */
		divBody.trigger('scroll');
	
		// If sorting or filtering has occurred, jump the scrolling back to the top
		// only if we aren't holding the position
		if ( (settings.bSorted || settings.bFiltered) && ! settings._drawHold ) {
			divBodyEl.scrollTop = 0;
		}
	}
	
	
	
	/**
	 * Apply a given function to the display child nodes of an element array (typically
	 * TD children of TR rows
	 *  @param {function} fn Method to apply to the objects
	 *  @param array {nodes} an1 List of elements to look through for display children
	 *  @param array {nodes} an2 Another list (identical structure to the first) - optional
	 *  @memberof DataTable#oApi
	 */
	function _fnApplyToChildren( fn, an1, an2 )
	{
		var index=0, i=0, iLen=an1.length;
		var nNode1, nNode2;
	
		while ( i < iLen ) {
			nNode1 = an1[i].firstChild;
			nNode2 = an2 ? an2[i].firstChild : null;
	
			while ( nNode1 ) {
				if ( nNode1.nodeType === 1 ) {
					if ( an2 ) {
						fn( nNode1, nNode2, index );
					}
					else {
						fn( nNode1, index );
					}
	
					index++;
				}
	
				nNode1 = nNode1.nextSibling;
				nNode2 = an2 ? nNode2.nextSibling : null;
			}
	
			i++;
		}
	}
	
	
	
	var __re_html_remove = /<.*?>/g;
	
	
	/**
	 * Calculate the width of columns for the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnCalculateColumnWidths ( oSettings )
	{
		var
			table = oSettings.nTable,
			columns = oSettings.aoColumns,
			scroll = oSettings.oScroll,
			scrollY = scroll.sY,
			scrollX = scroll.sX,
			scrollXInner = scroll.sXInner,
			columnCount = columns.length,
			visibleColumns = _fnGetColumns( oSettings, 'bVisible' ),
			headerCells = $('th', oSettings.nTHead),
			tableWidthAttr = table.getAttribute('width'), // from DOM element
			tableContainer = table.parentNode,
			userInputs = false,
			i, column, columnIdx, width, outerWidth,
			browser = oSettings.oBrowser,
			ie67 = browser.bScrollOversize;
	
		var styleWidth = table.style.width;
		if ( styleWidth && styleWidth.indexOf('%') !== -1 ) {
			tableWidthAttr = styleWidth;
		}
	
		/* Convert any user input sizes into pixel sizes */
		for ( i=0 ; i<visibleColumns.length ; i++ ) {
			column = columns[ visibleColumns[i] ];
	
			if ( column.sWidth !== null ) {
				column.sWidth = _fnConvertToWidth( column.sWidthOrig, tableContainer );
	
				userInputs = true;
			}
		}
	
		/* If the number of columns in the DOM equals the number that we have to
		 * process in DataTables, then we can use the offsets that are created by
		 * the web- browser. No custom sizes can be set in order for this to happen,
		 * nor scrolling used
		 */
		if ( ie67 || ! userInputs && ! scrollX && ! scrollY &&
		     columnCount == _fnVisbleColumns( oSettings ) &&
		     columnCount == headerCells.length
		) {
			for ( i=0 ; i<columnCount ; i++ ) {
				var colIdx = _fnVisibleToColumnIndex( oSettings, i );
	
				if ( colIdx !== null ) {
					columns[ colIdx ].sWidth = _fnStringToCss( headerCells.eq(i).width() );
				}
			}
		}
		else
		{
			// Otherwise construct a single row, worst case, table with the widest
			// node in the data, assign any user defined widths, then insert it into
			// the DOM and allow the browser to do all the hard work of calculating
			// table widths
			var tmpTable = $(table).clone() // don't use cloneNode - IE8 will remove events on the main table
				.css( 'visibility', 'hidden' )
				.removeAttr( 'id' );
	
			// Clean up the table body
			tmpTable.find('tbody tr').remove();
			var tr = $('<tr/>').appendTo( tmpTable.find('tbody') );
	
			// Clone the table header and footer - we can't use the header / footer
			// from the cloned table, since if scrolling is active, the table's
			// real header and footer are contained in different table tags
			tmpTable.find('thead, tfoot').remove();
			tmpTable
				.append( $(oSettings.nTHead).clone() )
				.append( $(oSettings.nTFoot).clone() );
	
			// Remove any assigned widths from the footer (from scrolling)
			tmpTable.find('tfoot th, tfoot td').css('width', '');
	
			// Apply custom sizing to the cloned header
			headerCells = _fnGetUniqueThs( oSettings, tmpTable.find('thead')[0] );
	
			for ( i=0 ; i<visibleColumns.length ; i++ ) {
				column = columns[ visibleColumns[i] ];
	
				headerCells[i].style.width = column.sWidthOrig !== null && column.sWidthOrig !== '' ?
					_fnStringToCss( column.sWidthOrig ) :
					'';
	
				// For scrollX we need to force the column width otherwise the
				// browser will collapse it. If this width is smaller than the
				// width the column requires, then it will have no effect
				if ( column.sWidthOrig && scrollX ) {
					$( headerCells[i] ).append( $('<div/>').css( {
						width: column.sWidthOrig,
						margin: 0,
						padding: 0,
						border: 0,
						height: 1
					} ) );
				}
			}
	
			// Find the widest cell for each column and put it into the table
			if ( oSettings.aoData.length ) {
				for ( i=0 ; i<visibleColumns.length ; i++ ) {
					columnIdx = visibleColumns[i];
					column = columns[ columnIdx ];
	
					$( _fnGetWidestNode( oSettings, columnIdx ) )
						.clone( false )
						.append( column.sContentPadding )
						.appendTo( tr );
				}
			}
	
			// Tidy the temporary table - remove name attributes so there aren't
			// duplicated in the dom (radio elements for example)
			$('[name]', tmpTable).removeAttr('name');
	
			// Table has been built, attach to the document so we can work with it.
			// A holding element is used, positioned at the top of the container
			// with minimal height, so it has no effect on if the container scrolls
			// or not. Otherwise it might trigger scrolling when it actually isn't
			// needed
			var holder = $('<div/>').css( scrollX || scrollY ?
					{
						position: 'absolute',
						top: 0,
						left: 0,
						height: 1,
						right: 0,
						overflow: 'hidden'
					} :
					{}
				)
				.append( tmpTable )
				.appendTo( tableContainer );
	
			// When scrolling (X or Y) we want to set the width of the table as 
			// appropriate. However, when not scrolling leave the table width as it
			// is. This results in slightly different, but I think correct behaviour
			if ( scrollX && scrollXInner ) {
				tmpTable.width( scrollXInner );
			}
			else if ( scrollX ) {
				tmpTable.css( 'width', 'auto' );
				tmpTable.removeAttr('width');
	
				// If there is no width attribute or style, then allow the table to
				// collapse
				if ( tmpTable.width() < tableContainer.clientWidth && tableWidthAttr ) {
					tmpTable.width( tableContainer.clientWidth );
				}
			}
			else if ( scrollY ) {
				tmpTable.width( tableContainer.clientWidth );
			}
			else if ( tableWidthAttr ) {
				tmpTable.width( tableWidthAttr );
			}
	
			// Get the width of each column in the constructed table - we need to
			// know the inner width (so it can be assigned to the other table's
			// cells) and the outer width so we can calculate the full width of the
			// table. This is safe since DataTables requires a unique cell for each
			// column, but if ever a header can span multiple columns, this will
			// need to be modified.
			var total = 0;
			for ( i=0 ; i<visibleColumns.length ; i++ ) {
				var cell = $(headerCells[i]);
				var border = cell.outerWidth() - cell.width();
	
				// Use getBounding... where possible (not IE8-) because it can give
				// sub-pixel accuracy, which we then want to round up!
				var bounding = browser.bBounding ?
					Math.ceil( headerCells[i].getBoundingClientRect().width ) :
					cell.outerWidth();
	
				// Total is tracked to remove any sub-pixel errors as the outerWidth
				// of the table might not equal the total given here (IE!).
				total += bounding;
	
				// Width for each column to use
				columns[ visibleColumns[i] ].sWidth = _fnStringToCss( bounding - border );
			}
	
			table.style.width = _fnStringToCss( total );
	
			// Finished with the table - ditch it
			holder.remove();
		}
	
		// If there is a width attr, we want to attach an event listener which
		// allows the table sizing to automatically adjust when the window is
		// resized. Use the width attr rather than CSS, since we can't know if the
		// CSS is a relative value or absolute - DOM read is always px.
		if ( tableWidthAttr ) {
			table.style.width = _fnStringToCss( tableWidthAttr );
		}
	
		if ( (tableWidthAttr || scrollX) && ! oSettings._reszEvt ) {
			var bindResize = function () {
				$(window).on('resize.DT-'+oSettings.sInstance, _fnThrottle( function () {
					_fnAdjustColumnSizing( oSettings );
				} ) );
			};
	
			// IE6/7 will crash if we bind a resize event handler on page load.
			// To be removed in 1.11 which drops IE6/7 support
			if ( ie67 ) {
				setTimeout( bindResize, 1000 );
			}
			else {
				bindResize();
			}
	
			oSettings._reszEvt = true;
		}
	}
	
	
	/**
	 * Throttle the calls to a function. Arguments and context are maintained for
	 * the throttled function
	 *  @param {function} fn Function to be called
	 *  @param {int} [freq=200] call frequency in mS
	 *  @returns {function} wrapped function
	 *  @memberof DataTable#oApi
	 */
	var _fnThrottle = DataTable.util.throttle;
	
	
	/**
	 * Convert a CSS unit width to pixels (e.g. 2em)
	 *  @param {string} width width to be converted
	 *  @param {node} parent parent to get the with for (required for relative widths) - optional
	 *  @returns {int} width in pixels
	 *  @memberof DataTable#oApi
	 */
	function _fnConvertToWidth ( width, parent )
	{
		if ( ! width ) {
			return 0;
		}
	
		var n = $('<div/>')
			.css( 'width', _fnStringToCss( width ) )
			.appendTo( parent || document.body );
	
		var val = n[0].offsetWidth;
		n.remove();
	
		return val;
	}
	
	
	/**
	 * Get the widest node
	 *  @param {object} settings dataTables settings object
	 *  @param {int} colIdx column of interest
	 *  @returns {node} widest table node
	 *  @memberof DataTable#oApi
	 */
	function _fnGetWidestNode( settings, colIdx )
	{
		var idx = _fnGetMaxLenString( settings, colIdx );
		if ( idx < 0 ) {
			return null;
		}
	
		var data = settings.aoData[ idx ];
		return ! data.nTr ? // Might not have been created when deferred rendering
			$('<td/>').html( _fnGetCellData( settings, idx, colIdx, 'display' ) )[0] :
			data.anCells[ colIdx ];
	}
	
	
	/**
	 * Get the maximum strlen for each data column
	 *  @param {object} settings dataTables settings object
	 *  @param {int} colIdx column of interest
	 *  @returns {string} max string length for each column
	 *  @memberof DataTable#oApi
	 */
	function _fnGetMaxLenString( settings, colIdx )
	{
		var s, max=-1, maxIdx = -1;
	
		for ( var i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
			s = _fnGetCellData( settings, i, colIdx, 'display' )+'';
			s = s.replace( __re_html_remove, '' );
			s = s.replace( /&nbsp;/g, ' ' );
	
			if ( s.length > max ) {
				max = s.length;
				maxIdx = i;
			}
		}
	
		return maxIdx;
	}
	
	
	/**
	 * Append a CSS unit (only if required) to a string
	 *  @param {string} value to css-ify
	 *  @returns {string} value with css unit
	 *  @memberof DataTable#oApi
	 */
	function _fnStringToCss( s )
	{
		if ( s === null ) {
			return '0px';
		}
	
		if ( typeof s == 'number' ) {
			return s < 0 ?
				'0px' :
				s+'px';
		}
	
		// Check it has a unit character already
		return s.match(/\d$/) ?
			s+'px' :
			s;
	}
	
	
	
	function _fnSortFlatten ( settings )
	{
		var
			i, iLen, k, kLen,
			aSort = [],
			aiOrig = [],
			aoColumns = settings.aoColumns,
			aDataSort, iCol, sType, srcCol,
			fixed = settings.aaSortingFixed,
			fixedObj = $.isPlainObject( fixed ),
			nestedSort = [],
			add = function ( a ) {
				if ( a.length && ! Array.isArray( a[0] ) ) {
					// 1D array
					nestedSort.push( a );
				}
				else {
					// 2D array
					$.merge( nestedSort, a );
				}
			};
	
		// Build the sort array, with pre-fix and post-fix options if they have been
		// specified
		if ( Array.isArray( fixed ) ) {
			add( fixed );
		}
	
		if ( fixedObj && fixed.pre ) {
			add( fixed.pre );
		}
	
		add( settings.aaSorting );
	
		if (fixedObj && fixed.post ) {
			add( fixed.post );
		}
	
		for ( i=0 ; i<nestedSort.length ; i++ )
		{
			srcCol = nestedSort[i][0];
			aDataSort = aoColumns[ srcCol ].aDataSort;
	
			for ( k=0, kLen=aDataSort.length ; k<kLen ; k++ )
			{
				iCol = aDataSort[k];
				sType = aoColumns[ iCol ].sType || 'string';
	
				if ( nestedSort[i]._idx === undefined ) {
					nestedSort[i]._idx = $.inArray( nestedSort[i][1], aoColumns[iCol].asSorting );
				}
	
				aSort.push( {
					src:       srcCol,
					col:       iCol,
					dir:       nestedSort[i][1],
					index:     nestedSort[i]._idx,
					type:      sType,
					formatter: DataTable.ext.type.order[ sType+"-pre" ]
				} );
			}
		}
	
		return aSort;
	}
	
	/**
	 * Change the order of the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 *  @todo This really needs split up!
	 */
	function _fnSort ( oSettings )
	{
		var
			i, ien, iLen, j, jLen, k, kLen,
			sDataType, nTh,
			aiOrig = [],
			oExtSort = DataTable.ext.type.order,
			aoData = oSettings.aoData,
			aoColumns = oSettings.aoColumns,
			aDataSort, data, iCol, sType, oSort,
			formatters = 0,
			sortCol,
			displayMaster = oSettings.aiDisplayMaster,
			aSort;
	
		// Resolve any column types that are unknown due to addition or invalidation
		// @todo Can this be moved into a 'data-ready' handler which is called when
		//   data is going to be used in the table?
		_fnColumnTypes( oSettings );
	
		aSort = _fnSortFlatten( oSettings );
	
		for ( i=0, ien=aSort.length ; i<ien ; i++ ) {
			sortCol = aSort[i];
	
			// Track if we can use the fast sort algorithm
			if ( sortCol.formatter ) {
				formatters++;
			}
	
			// Load the data needed for the sort, for each cell
			_fnSortData( oSettings, sortCol.col );
		}
	
		/* No sorting required if server-side or no sorting array */
		if ( _fnDataSource( oSettings ) != 'ssp' && aSort.length !== 0 )
		{
			// Create a value - key array of the current row positions such that we can use their
			// current position during the sort, if values match, in order to perform stable sorting
			for ( i=0, iLen=displayMaster.length ; i<iLen ; i++ ) {
				aiOrig[ displayMaster[i] ] = i;
			}
	
			/* Do the sort - here we want multi-column sorting based on a given data source (column)
			 * and sorting function (from oSort) in a certain direction. It's reasonably complex to
			 * follow on it's own, but this is what we want (example two column sorting):
			 *  fnLocalSorting = function(a,b){
			 *    var iTest;
			 *    iTest = oSort['string-asc']('data11', 'data12');
			 *      if (iTest !== 0)
			 *        return iTest;
			 *    iTest = oSort['numeric-desc']('data21', 'data22');
			 *    if (iTest !== 0)
			 *      return iTest;
			 *    return oSort['numeric-asc']( aiOrig[a], aiOrig[b] );
			 *  }
			 * Basically we have a test for each sorting column, if the data in that column is equal,
			 * test the next column. If all columns match, then we use a numeric sort on the row
			 * positions in the original data array to provide a stable sort.
			 *
			 * Note - I know it seems excessive to have two sorting methods, but the first is around
			 * 15% faster, so the second is only maintained for backwards compatibility with sorting
			 * methods which do not have a pre-sort formatting function.
			 */
			if ( formatters === aSort.length ) {
				// All sort types have formatting functions
				displayMaster.sort( function ( a, b ) {
					var
						x, y, k, test, sort,
						len=aSort.length,
						dataA = aoData[a]._aSortData,
						dataB = aoData[b]._aSortData;
	
					for ( k=0 ; k<len ; k++ ) {
						sort = aSort[k];
	
						x = dataA[ sort.col ];
						y = dataB[ sort.col ];
	
						test = x<y ? -1 : x>y ? 1 : 0;
						if ( test !== 0 ) {
							return sort.dir === 'asc' ? test : -test;
						}
					}
	
					x = aiOrig[a];
					y = aiOrig[b];
					return x<y ? -1 : x>y ? 1 : 0;
				} );
			}
			else {
				// Depreciated - remove in 1.11 (providing a plug-in option)
				// Not all sort types have formatting methods, so we have to call their sorting
				// methods.
				displayMaster.sort( function ( a, b ) {
					var
						x, y, k, l, test, sort, fn,
						len=aSort.length,
						dataA = aoData[a]._aSortData,
						dataB = aoData[b]._aSortData;
	
					for ( k=0 ; k<len ; k++ ) {
						sort = aSort[k];
	
						x = dataA[ sort.col ];
						y = dataB[ sort.col ];
	
						fn = oExtSort[ sort.type+"-"+sort.dir ] || oExtSort[ "string-"+sort.dir ];
						test = fn( x, y );
						if ( test !== 0 ) {
							return test;
						}
					}
	
					x = aiOrig[a];
					y = aiOrig[b];
					return x<y ? -1 : x>y ? 1 : 0;
				} );
			}
		}
	
		/* Tell the draw function that we have sorted the data */
		oSettings.bSorted = true;
	}
	
	
	function _fnSortAria ( settings )
	{
		var label;
		var nextSort;
		var columns = settings.aoColumns;
		var aSort = _fnSortFlatten( settings );
		var oAria = settings.oLanguage.oAria;
	
		// ARIA attributes - need to loop all columns, to update all (removing old
		// attributes as needed)
		for ( var i=0, iLen=columns.length ; i<iLen ; i++ )
		{
			var col = columns[i];
			var asSorting = col.asSorting;
			var sTitle = col.sTitle.replace( /<.*?>/g, "" );
			var th = col.nTh;
	
			// IE7 is throwing an error when setting these properties with jQuery's
			// attr() and removeAttr() methods...
			th.removeAttribute('aria-sort');
	
			/* In ARIA only the first sorting column can be marked as sorting - no multi-sort option */
			if ( col.bSortable ) {
				if ( aSort.length > 0 && aSort[0].col == i ) {
					th.setAttribute('aria-sort', aSort[0].dir=="asc" ? "ascending" : "descending" );
					nextSort = asSorting[ aSort[0].index+1 ] || asSorting[0];
				}
				else {
					nextSort = asSorting[0];
				}
	
				label = sTitle + ( nextSort === "asc" ?
					oAria.sSortAscending :
					oAria.sSortDescending
				);
			}
			else {
				label = sTitle;
			}
	
			th.setAttribute('aria-label', label);
		}
	}
	
	
	/**
	 * Function to run on user sort request
	 *  @param {object} settings dataTables settings object
	 *  @param {node} attachTo node to attach the handler to
	 *  @param {int} colIdx column sorting index
	 *  @param {boolean} [append=false] Append the requested sort to the existing
	 *    sort if true (i.e. multi-column sort)
	 *  @param {function} [callback] callback function
	 *  @memberof DataTable#oApi
	 */
	function _fnSortListener ( settings, colIdx, append, callback )
	{
		var col = settings.aoColumns[ colIdx ];
		var sorting = settings.aaSorting;
		var asSorting = col.asSorting;
		var nextSortIdx;
		var next = function ( a, overflow ) {
			var idx = a._idx;
			if ( idx === undefined ) {
				idx = $.inArray( a[1], asSorting );
			}
	
			return idx+1 < asSorting.length ?
				idx+1 :
				overflow ?
					null :
					0;
		};
	
		// Convert to 2D array if needed
		if ( typeof sorting[0] === 'number' ) {
			sorting = settings.aaSorting = [ sorting ];
		}
	
		// If appending the sort then we are multi-column sorting
		if ( append && settings.oFeatures.bSortMulti ) {
			// Are we already doing some kind of sort on this column?
			var sortIdx = $.inArray( colIdx, _pluck(sorting, '0') );
	
			if ( sortIdx !== -1 ) {
				// Yes, modify the sort
				nextSortIdx = next( sorting[sortIdx], true );
	
				if ( nextSortIdx === null && sorting.length === 1 ) {
					nextSortIdx = 0; // can't remove sorting completely
				}
	
				if ( nextSortIdx === null ) {
					sorting.splice( sortIdx, 1 );
				}
				else {
					sorting[sortIdx][1] = asSorting[ nextSortIdx ];
					sorting[sortIdx]._idx = nextSortIdx;
				}
			}
			else {
				// No sort on this column yet
				sorting.push( [ colIdx, asSorting[0], 0 ] );
				sorting[sorting.length-1]._idx = 0;
			}
		}
		else if ( sorting.length && sorting[0][0] == colIdx ) {
			// Single column - already sorting on this column, modify the sort
			nextSortIdx = next( sorting[0] );
	
			sorting.length = 1;
			sorting[0][1] = asSorting[ nextSortIdx ];
			sorting[0]._idx = nextSortIdx;
		}
		else {
			// Single column - sort only on this column
			sorting.length = 0;
			sorting.push( [ colIdx, asSorting[0] ] );
			sorting[0]._idx = 0;
		}
	
		// Run the sort by calling a full redraw
		_fnReDraw( settings );
	
		// callback used for async user interaction
		if ( typeof callback == 'function' ) {
			callback( settings );
		}
	}
	
	
	/**
	 * Attach a sort handler (click) to a node
	 *  @param {object} settings dataTables settings object
	 *  @param {node} attachTo node to attach the handler to
	 *  @param {int} colIdx column sorting index
	 *  @param {function} [callback] callback function
	 *  @memberof DataTable#oApi
	 */
	function _fnSortAttachListener ( settings, attachTo, colIdx, callback )
	{
		var col = settings.aoColumns[ colIdx ];
	
		_fnBindAction( attachTo, {}, function (e) {
			/* If the column is not sortable - don't to anything */
			if ( col.bSortable === false ) {
				return;
			}
	
			// If processing is enabled use a timeout to allow the processing
			// display to be shown - otherwise to it synchronously
			if ( settings.oFeatures.bProcessing ) {
				_fnProcessingDisplay( settings, true );
	
				setTimeout( function() {
					_fnSortListener( settings, colIdx, e.shiftKey, callback );
	
					// In server-side processing, the draw callback will remove the
					// processing display
					if ( _fnDataSource( settings ) !== 'ssp' ) {
						_fnProcessingDisplay( settings, false );
					}
				}, 0 );
			}
			else {
				_fnSortListener( settings, colIdx, e.shiftKey, callback );
			}
		} );
	}
	
	
	/**
	 * Set the sorting classes on table's body, Note: it is safe to call this function
	 * when bSort and bSortClasses are false
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnSortingClasses( settings )
	{
		var oldSort = settings.aLastSort;
		var sortClass = settings.oClasses.sSortColumn;
		var sort = _fnSortFlatten( settings );
		var features = settings.oFeatures;
		var i, ien, colIdx;
	
		if ( features.bSort && features.bSortClasses ) {
			// Remove old sorting classes
			for ( i=0, ien=oldSort.length ; i<ien ; i++ ) {
				colIdx = oldSort[i].src;
	
				// Remove column sorting
				$( _pluck( settings.aoData, 'anCells', colIdx ) )
					.removeClass( sortClass + (i<2 ? i+1 : 3) );
			}
	
			// Add new column sorting
			for ( i=0, ien=sort.length ; i<ien ; i++ ) {
				colIdx = sort[i].src;
	
				$( _pluck( settings.aoData, 'anCells', colIdx ) )
					.addClass( sortClass + (i<2 ? i+1 : 3) );
			}
		}
	
		settings.aLastSort = sort;
	}
	
	
	// Get the data to sort a column, be it from cache, fresh (populating the
	// cache), or from a sort formatter
	function _fnSortData( settings, idx )
	{
		// Custom sorting function - provided by the sort data type
		var column = settings.aoColumns[ idx ];
		var customSort = DataTable.ext.order[ column.sSortDataType ];
		var customData;
	
		if ( customSort ) {
			customData = customSort.call( settings.oInstance, settings, idx,
				_fnColumnIndexToVisible( settings, idx )
			);
		}
	
		// Use / populate cache
		var row, cellData;
		var formatter = DataTable.ext.type.order[ column.sType+"-pre" ];
	
		for ( var i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
			row = settings.aoData[i];
	
			if ( ! row._aSortData ) {
				row._aSortData = [];
			}
	
			if ( ! row._aSortData[idx] || customSort ) {
				cellData = customSort ?
					customData[i] : // If there was a custom sort function, use data from there
					_fnGetCellData( settings, i, idx, 'sort' );
	
				row._aSortData[ idx ] = formatter ?
					formatter( cellData ) :
					cellData;
			}
		}
	}
	
	
	
	/**
	 * Save the state of a table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnSaveState ( settings )
	{
		if ( !settings.oFeatures.bStateSave || settings.bDestroying )
		{
			return;
		}
	
		/* Store the interesting variables */
		var state = {
			time:    +new Date(),
			start:   settings._iDisplayStart,
			length:  settings._iDisplayLength,
			order:   $.extend( true, [], settings.aaSorting ),
			search:  _fnSearchToCamel( settings.oPreviousSearch ),
			columns: $.map( settings.aoColumns, function ( col, i ) {
				return {
					visible: col.bVisible,
					search: _fnSearchToCamel( settings.aoPreSearchCols[i] )
				};
			} )
		};
	
		_fnCallbackFire( settings, "aoStateSaveParams", 'stateSaveParams', [settings, state] );
	
		settings.oSavedState = state;
		settings.fnStateSaveCallback.call( settings.oInstance, settings, state );
	}
	
	
	/**
	 * Attempt to load a saved table state
	 *  @param {object} oSettings dataTables settings object
	 *  @param {object} oInit DataTables init object so we can override settings
	 *  @param {function} callback Callback to execute when the state has been loaded
	 *  @memberof DataTable#oApi
	 */
	function _fnLoadState ( settings, oInit, callback )
	{
		var i, ien;
		var columns = settings.aoColumns;
		var loaded = function ( s ) {
			if ( ! s || ! s.time ) {
				callback();
				return;
			}
	
			// Allow custom and plug-in manipulation functions to alter the saved data set and
			// cancelling of loading by returning false
			var abStateLoad = _fnCallbackFire( settings, 'aoStateLoadParams', 'stateLoadParams', [settings, s] );
			if ( $.inArray( false, abStateLoad ) !== -1 ) {
				callback();
				return;
			}
	
			// Reject old data
			var duration = settings.iStateDuration;
			if ( duration > 0 && s.time < +new Date() - (duration*1000) ) {
				callback();
				return;
			}
	
			// Number of columns have changed - all bets are off, no restore of settings
			if ( s.columns && columns.length !== s.columns.length ) {
				callback();
				return;
			}
	
			// Store the saved state so it might be accessed at any time
			settings.oLoadedState = $.extend( true, {}, s );
	
			// Restore key features - todo - for 1.11 this needs to be done by
			// subscribed events
			if ( s.start !== undefined ) {
				settings._iDisplayStart    = s.start;
				settings.iInitDisplayStart = s.start;
			}
			if ( s.length !== undefined ) {
				settings._iDisplayLength   = s.length;
			}
	
			// Order
			if ( s.order !== undefined ) {
				settings.aaSorting = [];
				$.each( s.order, function ( i, col ) {
					settings.aaSorting.push( col[0] >= columns.length ?
						[ 0, col[1] ] :
						col
					);
				} );
			}
	
			// Search
			if ( s.search !== undefined ) {
				$.extend( settings.oPreviousSearch, _fnSearchToHung( s.search ) );
			}
	
			// Columns
			//
			if ( s.columns ) {
				for ( i=0, ien=s.columns.length ; i<ien ; i++ ) {
					var col = s.columns[i];
	
					// Visibility
					if ( col.visible !== undefined ) {
						columns[i].bVisible = col.visible;
					}
	
					// Search
					if ( col.search !== undefined ) {
						$.extend( settings.aoPreSearchCols[i], _fnSearchToHung( col.search ) );
					}
				}
			}
	
			_fnCallbackFire( settings, 'aoStateLoaded', 'stateLoaded', [settings, s] );
			callback();
		};
	
		if ( ! settings.oFeatures.bStateSave ) {
			callback();
			return;
		}
	
		var state = settings.fnStateLoadCallback.call( settings.oInstance, settings, loaded );
	
		if ( state !== undefined ) {
			loaded( state );
		}
		// otherwise, wait for the loaded callback to be executed
	}
	
	
	/**
	 * Return the settings object for a particular table
	 *  @param {node} table table we are using as a dataTable
	 *  @returns {object} Settings object - or null if not found
	 *  @memberof DataTable#oApi
	 */
	function _fnSettingsFromNode ( table )
	{
		var settings = DataTable.settings;
		var idx = $.inArray( table, _pluck( settings, 'nTable' ) );
	
		return idx !== -1 ?
			settings[ idx ] :
			null;
	}
	
	
	/**
	 * Log an error message
	 *  @param {object} settings dataTables settings object
	 *  @param {int} level log error messages, or display them to the user
	 *  @param {string} msg error message
	 *  @param {int} tn Technical note id to get more information about the error.
	 *  @memberof DataTable#oApi
	 */
	function _fnLog( settings, level, msg, tn )
	{
		msg = 'DataTables warning: '+
			(settings ? 'table id='+settings.sTableId+' - ' : '')+msg;
	
		if ( tn ) {
			msg += '. For more information about this error, please see '+
			'http://datatables.net/tn/'+tn;
		}
	
		if ( ! level  ) {
			// Backwards compatibility pre 1.10
			var ext = DataTable.ext;
			var type = ext.sErrMode || ext.errMode;
	
			if ( settings ) {
				_fnCallbackFire( settings, null, 'error', [ settings, tn, msg ] );
			}
	
			if ( type == 'alert' ) {
				alert( msg );
			}
			else if ( type == 'throw' ) {
				throw new Error(msg);
			}
			else if ( typeof type == 'function' ) {
				type( settings, tn, msg );
			}
		}
		else if ( window.console && console.log ) {
			console.log( msg );
		}
	}
	
	
	/**
	 * See if a property is defined on one object, if so assign it to the other object
	 *  @param {object} ret target object
	 *  @param {object} src source object
	 *  @param {string} name property
	 *  @param {string} [mappedName] name to map too - optional, name used if not given
	 *  @memberof DataTable#oApi
	 */
	function _fnMap( ret, src, name, mappedName )
	{
		if ( Array.isArray( name ) ) {
			$.each( name, function (i, val) {
				if ( Array.isArray( val ) ) {
					_fnMap( ret, src, val[0], val[1] );
				}
				else {
					_fnMap( ret, src, val );
				}
			} );
	
			return;
		}
	
		if ( mappedName === undefined ) {
			mappedName = name;
		}
	
		if ( src[name] !== undefined ) {
			ret[mappedName] = src[name];
		}
	}
	
	
	/**
	 * Extend objects - very similar to jQuery.extend, but deep copy objects, and
	 * shallow copy arrays. The reason we need to do this, is that we don't want to
	 * deep copy array init values (such as aaSorting) since the dev wouldn't be
	 * able to override them, but we do want to deep copy arrays.
	 *  @param {object} out Object to extend
	 *  @param {object} extender Object from which the properties will be applied to
	 *      out
	 *  @param {boolean} breakRefs If true, then arrays will be sliced to take an
	 *      independent copy with the exception of the `data` or `aaData` parameters
	 *      if they are present. This is so you can pass in a collection to
	 *      DataTables and have that used as your data source without breaking the
	 *      references
	 *  @returns {object} out Reference, just for convenience - out === the return.
	 *  @memberof DataTable#oApi
	 *  @todo This doesn't take account of arrays inside the deep copied objects.
	 */
	function _fnExtend( out, extender, breakRefs )
	{
		var val;
	
		for ( var prop in extender ) {
			if ( extender.hasOwnProperty(prop) ) {
				val = extender[prop];
	
				if ( $.isPlainObject( val ) ) {
					if ( ! $.isPlainObject( out[prop] ) ) {
						out[prop] = {};
					}
					$.extend( true, out[prop], val );
				}
				else if ( breakRefs && prop !== 'data' && prop !== 'aaData' && Array.isArray(val) ) {
					out[prop] = val.slice();
				}
				else {
					out[prop] = val;
				}
			}
		}
	
		return out;
	}
	
	
	/**
	 * Bind an event handers to allow a click or return key to activate the callback.
	 * This is good for accessibility since a return on the keyboard will have the
	 * same effect as a click, if the element has focus.
	 *  @param {element} n Element to bind the action to
	 *  @param {object} oData Data object to pass to the triggered function
	 *  @param {function} fn Callback function for when the event is triggered
	 *  @memberof DataTable#oApi
	 */
	function _fnBindAction( n, oData, fn )
	{
		$(n)
			.on( 'click.DT', oData, function (e) {
					$(n).trigger('blur'); // Remove focus outline for mouse users
					fn(e);
				} )
			.on( 'keypress.DT', oData, function (e){
					if ( e.which === 13 ) {
						e.preventDefault();
						fn(e);
					}
				} )
			.on( 'selectstart.DT', function () {
					/* Take the brutal approach to cancelling text selection */
					return false;
				} );
	}
	
	
	/**
	 * Register a callback function. Easily allows a callback function to be added to
	 * an array store of callback functions that can then all be called together.
	 *  @param {object} oSettings dataTables settings object
	 *  @param {string} sStore Name of the array storage for the callbacks in oSettings
	 *  @param {function} fn Function to be called back
	 *  @param {string} sName Identifying name for the callback (i.e. a label)
	 *  @memberof DataTable#oApi
	 */
	function _fnCallbackReg( oSettings, sStore, fn, sName )
	{
		if ( fn )
		{
			oSettings[sStore].push( {
				"fn": fn,
				"sName": sName
			} );
		}
	}
	
	
	/**
	 * Fire callback functions and trigger events. Note that the loop over the
	 * callback array store is done backwards! Further note that you do not want to
	 * fire off triggers in time sensitive applications (for example cell creation)
	 * as its slow.
	 *  @param {object} settings dataTables settings object
	 *  @param {string} callbackArr Name of the array storage for the callbacks in
	 *      oSettings
	 *  @param {string} eventName Name of the jQuery custom event to trigger. If
	 *      null no trigger is fired
	 *  @param {array} args Array of arguments to pass to the callback function /
	 *      trigger
	 *  @memberof DataTable#oApi
	 */
	function _fnCallbackFire( settings, callbackArr, eventName, args )
	{
		var ret = [];
	
		if ( callbackArr ) {
			ret = $.map( settings[callbackArr].slice().reverse(), function (val, i) {
				return val.fn.apply( settings.oInstance, args );
			} );
		}
	
		if ( eventName !== null ) {
			var e = $.Event( eventName+'.dt' );
	
			$(settings.nTable).trigger( e, args );
	
			ret.push( e.result );
		}
	
		return ret;
	}
	
	
	function _fnLengthOverflow ( settings )
	{
		var
			start = settings._iDisplayStart,
			end = settings.fnDisplayEnd(),
			len = settings._iDisplayLength;
	
		/* If we have space to show extra rows (backing up from the end point - then do so */
		if ( start >= end )
		{
			start = end - len;
		}
	
		// Keep the start record on the current page
		start -= (start % len);
	
		if ( len === -1 || start < 0 )
		{
			start = 0;
		}
	
		settings._iDisplayStart = start;
	}
	
	
	function _fnRenderer( settings, type )
	{
		var renderer = settings.renderer;
		var host = DataTable.ext.renderer[type];
	
		if ( $.isPlainObject( renderer ) && renderer[type] ) {
			// Specific renderer for this type. If available use it, otherwise use
			// the default.
			return host[renderer[type]] || host._;
		}
		else if ( typeof renderer === 'string' ) {
			// Common renderer - if there is one available for this type use it,
			// otherwise use the default
			return host[renderer] || host._;
		}
	
		// Use the default
		return host._;
	}
	
	
	/**
	 * Detect the data source being used for the table. Used to simplify the code
	 * a little (ajax) and to make it compress a little smaller.
	 *
	 *  @param {object} settings dataTables settings object
	 *  @returns {string} Data source
	 *  @memberof DataTable#oApi
	 */
	function _fnDataSource ( settings )
	{
		if ( settings.oFeatures.bServerSide ) {
			return 'ssp';
		}
		else if ( settings.ajax || settings.sAjaxSource ) {
			return 'ajax';
		}
		return 'dom';
	}
	

	
	
	/**
	 * Computed structure of the DataTables API, defined by the options passed to
	 * `DataTable.Api.register()` when building the API.
	 *
	 * The structure is built in order to speed creation and extension of the Api
	 * objects since the extensions are effectively pre-parsed.
	 *
	 * The array is an array of objects with the following structure, where this
	 * base array represents the Api prototype base:
	 *
	 *     [
	 *       {
	 *         name:      'data'                -- string   - Property name
	 *         val:       function () {},       -- function - Api method (or undefined if just an object
	 *         methodExt: [ ... ],              -- array    - Array of Api object definitions to extend the method result
	 *         propExt:   [ ... ]               -- array    - Array of Api object definitions to extend the property
	 *       },
	 *       {
	 *         name:     'row'
	 *         val:       {},
	 *         methodExt: [ ... ],
	 *         propExt:   [
	 *           {
	 *             name:      'data'
	 *             val:       function () {},
	 *             methodExt: [ ... ],
	 *             propExt:   [ ... ]
	 *           },
	 *           ...
	 *         ]
	 *       }
	 *     ]
	 *
	 * @type {Array}
	 * @ignore
	 */
	var __apiStruct = [];
	
	
	/**
	 * `Array.prototype` reference.
	 *
	 * @type object
	 * @ignore
	 */
	var __arrayProto = Array.prototype;
	
	
	/**
	 * Abstraction for `context` parameter of the `Api` constructor to allow it to
	 * take several different forms for ease of use.
	 *
	 * Each of the input parameter types will be converted to a DataTables settings
	 * object where possible.
	 *
	 * @param  {string|node|jQuery|object} mixed DataTable identifier. Can be one
	 *   of:
	 *
	 *   * `string` - jQuery selector. Any DataTables' matching the given selector
	 *     with be found and used.
	 *   * `node` - `TABLE` node which has already been formed into a DataTable.
	 *   * `jQuery` - A jQuery object of `TABLE` nodes.
	 *   * `object` - DataTables settings object
	 *   * `DataTables.Api` - API instance
	 * @return {array|null} Matching DataTables settings objects. `null` or
	 *   `undefined` is returned if no matching DataTable is found.
	 * @ignore
	 */
	var _toSettings = function ( mixed )
	{
		var idx, jq;
		var settings = DataTable.settings;
		var tables = $.map( settings, function (el, i) {
			return el.nTable;
		} );
	
		if ( ! mixed ) {
			return [];
		}
		else if ( mixed.nTable && mixed.oApi ) {
			// DataTables settings object
			return [ mixed ];
		}
		else if ( mixed.nodeName && mixed.nodeName.toLowerCase() === 'table' ) {
			// Table node
			idx = $.inArray( mixed, tables );
			return idx !== -1 ? [ settings[idx] ] : null;
		}
		else if ( mixed && typeof mixed.settings === 'function' ) {
			return mixed.settings().toArray();
		}
		else if ( typeof mixed === 'string' ) {
			// jQuery selector
			jq = $(mixed);
		}
		else if ( mixed instanceof $ ) {
			// jQuery object (also DataTables instance)
			jq = mixed;
		}
	
		if ( jq ) {
			return jq.map( function(i) {
				idx = $.inArray( this, tables );
				return idx !== -1 ? settings[idx] : null;
			} ).toArray();
		}
	};
	
	
	/**
	 * DataTables API class - used to control and interface with  one or more
	 * DataTables enhanced tables.
	 *
	 * The API class is heavily based on jQuery, presenting a chainable interface
	 * that you can use to interact with tables. Each instance of the API class has
	 * a "context" - i.e. the tables that it will operate on. This could be a single
	 * table, all tables on a page or a sub-set thereof.
	 *
	 * Additionally the API is designed to allow you to easily work with the data in
	 * the tables, retrieving and manipulating it as required. This is done by
	 * presenting the API class as an array like interface. The contents of the
	 * array depend upon the actions requested by each method (for example
	 * `rows().nodes()` will return an array of nodes, while `rows().data()` will
	 * return an array of objects or arrays depending upon your table's
	 * configuration). The API object has a number of array like methods (`push`,
	 * `pop`, `reverse` etc) as well as additional helper methods (`each`, `pluck`,
	 * `unique` etc) to assist your working with the data held in a table.
	 *
	 * Most methods (those which return an Api instance) are chainable, which means
	 * the return from a method call also has all of the methods available that the
	 * top level object had. For example, these two calls are equivalent:
	 *
	 *     // Not chained
	 *     api.row.add( {...} );
	 *     api.draw();
	 *
	 *     // Chained
	 *     api.row.add( {...} ).draw();
	 *
	 * @class DataTable.Api
	 * @param {array|object|string|jQuery} context DataTable identifier. This is
	 *   used to define which DataTables enhanced tables this API will operate on.
	 *   Can be one of:
	 *
	 *   * `string` - jQuery selector. Any DataTables' matching the given selector
	 *     with be found and used.
	 *   * `node` - `TABLE` node which has already been formed into a DataTable.
	 *   * `jQuery` - A jQuery object of `TABLE` nodes.
	 *   * `object` - DataTables settings object
	 * @param {array} [data] Data to initialise the Api instance with.
	 *
	 * @example
	 *   // Direct initialisation during DataTables construction
	 *   var api = $('#example').DataTable();
	 *
	 * @example
	 *   // Initialisation using a DataTables jQuery object
	 *   var api = $('#example').dataTable().api();
	 *
	 * @example
	 *   // Initialisation as a constructor
	 *   var api = new $.fn.DataTable.Api( 'table.dataTable' );
	 */
	_Api = function ( context, data )
	{
		if ( ! (this instanceof _Api) ) {
			return new _Api( context, data );
		}
	
		var settings = [];
		var ctxSettings = function ( o ) {
			var a = _toSettings( o );
			if ( a ) {
				settings.push.apply( settings, a );
			}
		};
	
		if ( Array.isArray( context ) ) {
			for ( var i=0, ien=context.length ; i<ien ; i++ ) {
				ctxSettings( context[i] );
			}
		}
		else {
			ctxSettings( context );
		}
	
		// Remove duplicates
		this.context = _unique( settings );
	
		// Initial data
		if ( data ) {
			$.merge( this, data );
		}
	
		// selector
		this.selector = {
			rows: null,
			cols: null,
			opts: null
		};
	
		_Api.extend( this, this, __apiStruct );
	};
	
	DataTable.Api = _Api;
	
	// Don't destroy the existing prototype, just extend it. Required for jQuery 2's
	// isPlainObject.
	$.extend( _Api.prototype, {
		any: function ()
		{
			return this.count() !== 0;
		},
	
	
		concat:  __arrayProto.concat,
	
	
		context: [], // array of table settings objects
	
	
		count: function ()
		{
			return this.flatten().length;
		},
	
	
		each: function ( fn )
		{
			for ( var i=0, ien=this.length ; i<ien; i++ ) {
				fn.call( this, this[i], i, this );
			}
	
			return this;
		},
	
	
		eq: function ( idx )
		{
			var ctx = this.context;
	
			return ctx.length > idx ?
				new _Api( ctx[idx], this[idx] ) :
				null;
		},
	
	
		filter: function ( fn )
		{
			var a = [];
	
			if ( __arrayProto.filter ) {
				a = __arrayProto.filter.call( this, fn, this );
			}
			else {
				// Compatibility for browsers without EMCA-252-5 (JS 1.6)
				for ( var i=0, ien=this.length ; i<ien ; i++ ) {
					if ( fn.call( this, this[i], i, this ) ) {
						a.push( this[i] );
					}
				}
			}
	
			return new _Api( this.context, a );
		},
	
	
		flatten: function ()
		{
			var a = [];
			return new _Api( this.context, a.concat.apply( a, this.toArray() ) );
		},
	
	
		join:    __arrayProto.join,
	
	
		indexOf: __arrayProto.indexOf || function (obj, start)
		{
			for ( var i=(start || 0), ien=this.length ; i<ien ; i++ ) {
				if ( this[i] === obj ) {
					return i;
				}
			}
			return -1;
		},
	
		iterator: function ( flatten, type, fn, alwaysNew ) {
			var
				a = [], ret,
				i, ien, j, jen,
				context = this.context,
				rows, items, item,
				selector = this.selector;
	
			// Argument shifting
			if ( typeof flatten === 'string' ) {
				alwaysNew = fn;
				fn = type;
				type = flatten;
				flatten = false;
			}
	
			for ( i=0, ien=context.length ; i<ien ; i++ ) {
				var apiInst = new _Api( context[i] );
	
				if ( type === 'table' ) {
					ret = fn.call( apiInst, context[i], i );
	
					if ( ret !== undefined ) {
						a.push( ret );
					}
				}
				else if ( type === 'columns' || type === 'rows' ) {
					// this has same length as context - one entry for each table
					ret = fn.call( apiInst, context[i], this[i], i );
	
					if ( ret !== undefined ) {
						a.push( ret );
					}
				}
				else if ( type === 'column' || type === 'column-rows' || type === 'row' || type === 'cell' ) {
					// columns and rows share the same structure.
					// 'this' is an array of column indexes for each context
					items = this[i];
	
					if ( type === 'column-rows' ) {
						rows = _selector_row_indexes( context[i], selector.opts );
					}
	
					for ( j=0, jen=items.length ; j<jen ; j++ ) {
						item = items[j];
	
						if ( type === 'cell' ) {
							ret = fn.call( apiInst, context[i], item.row, item.column, i, j );
						}
						else {
							ret = fn.call( apiInst, context[i], item, i, j, rows );
						}
	
						if ( ret !== undefined ) {
							a.push( ret );
						}
					}
				}
			}
	
			if ( a.length || alwaysNew ) {
				var api = new _Api( context, flatten ? a.concat.apply( [], a ) : a );
				var apiSelector = api.selector;
				apiSelector.rows = selector.rows;
				apiSelector.cols = selector.cols;
				apiSelector.opts = selector.opts;
				return api;
			}
			return this;
		},
	
	
		lastIndexOf: __arrayProto.lastIndexOf || function (obj, start)
		{
			// Bit cheeky...
			return this.indexOf.apply( this.toArray.reverse(), arguments );
		},
	
	
		length:  0,
	
	
		map: function ( fn )
		{
			var a = [];
	
			if ( __arrayProto.map ) {
				a = __arrayProto.map.call( this, fn, this );
			}
			else {
				// Compatibility for browsers without EMCA-252-5 (JS 1.6)
				for ( var i=0, ien=this.length ; i<ien ; i++ ) {
					a.push( fn.call( this, this[i], i ) );
				}
			}
	
			return new _Api( this.context, a );
		},
	
	
		pluck: function ( prop )
		{
			return this.map( function ( el ) {
				return el[ prop ];
			} );
		},
	
		pop:     __arrayProto.pop,
	
	
		push:    __arrayProto.push,
	
	
		// Does not return an API instance
		reduce: __arrayProto.reduce || function ( fn, init )
		{
			return _fnReduce( this, fn, init, 0, this.length, 1 );
		},
	
	
		reduceRight: __arrayProto.reduceRight || function ( fn, init )
		{
			return _fnReduce( this, fn, init, this.length-1, -1, -1 );
		},
	
	
		reverse: __arrayProto.reverse,
	
	
		// Object with rows, columns and opts
		selector: null,
	
	
		shift:   __arrayProto.shift,
	
	
		slice: function () {
			return new _Api( this.context, this );
		},
	
	
		sort:    __arrayProto.sort, // ? name - order?
	
	
		splice:  __arrayProto.splice,
	
	
		toArray: function ()
		{
			return __arrayProto.slice.call( this );
		},
	
	
		to$: function ()
		{
			return $( this );
		},
	
	
		toJQuery: function ()
		{
			return $( this );
		},
	
	
		unique: function ()
		{
			return new _Api( this.context, _unique(this) );
		},
	
	
		unshift: __arrayProto.unshift
	} );
	
	
	_Api.extend = function ( scope, obj, ext )
	{
		// Only extend API instances and static properties of the API
		if ( ! ext.length || ! obj || ( ! (obj instanceof _Api) && ! obj.__dt_wrapper ) ) {
			return;
		}
	
		var
			i, ien,
			struct,
			methodScoping = function ( scope, fn, struc ) {
				return function () {
					var ret = fn.apply( scope, arguments );
	
					// Method extension
					_Api.extend( ret, ret, struc.methodExt );
					return ret;
				};
			};
	
		for ( i=0, ien=ext.length ; i<ien ; i++ ) {
			struct = ext[i];
	
			// Value
			obj[ struct.name ] = struct.type === 'function' ?
				methodScoping( scope, struct.val, struct ) :
				struct.type === 'object' ?
					{} :
					struct.val;
	
			obj[ struct.name ].__dt_wrapper = true;
	
			// Property extension
			_Api.extend( scope, obj[ struct.name ], struct.propExt );
		}
	};
	
	
	// @todo - Is there need for an augment function?
	// _Api.augment = function ( inst, name )
	// {
	// 	// Find src object in the structure from the name
	// 	var parts = name.split('.');
	
	// 	_Api.extend( inst, obj );
	// };
	
	
	//     [
	//       {
	//         name:      'data'                -- string   - Property name
	//         val:       function () {},       -- function - Api method (or undefined if just an object
	//         methodExt: [ ... ],              -- array    - Array of Api object definitions to extend the method result
	//         propExt:   [ ... ]               -- array    - Array of Api object definitions to extend the property
	//       },
	//       {
	//         name:     'row'
	//         val:       {},
	//         methodExt: [ ... ],
	//         propExt:   [
	//           {
	//             name:      'data'
	//             val:       function () {},
	//             methodExt: [ ... ],
	//             propExt:   [ ... ]
	//           },
	//           ...
	//         ]
	//       }
	//     ]
	
	_Api.register = _api_register = function ( name, val )
	{
		if ( Array.isArray( name ) ) {
			for ( var j=0, jen=name.length ; j<jen ; j++ ) {
				_Api.register( name[j], val );
			}
			return;
		}
	
		var
			i, ien,
			heir = name.split('.'),
			struct = __apiStruct,
			key, method;
	
		var find = function ( src, name ) {
			for ( var i=0, ien=src.length ; i<ien ; i++ ) {
				if ( src[i].name === name ) {
					return src[i];
				}
			}
			return null;
		};
	
		for ( i=0, ien=heir.length ; i<ien ; i++ ) {
			method = heir[i].indexOf('()') !== -1;
			key = method ?
				heir[i].replace('()', '') :
				heir[i];
	
			var src = find( struct, key );
			if ( ! src ) {
				src = {
					name:      key,
					val:       {},
					methodExt: [],
					propExt:   [],
					type:      'object'
				};
				struct.push( src );
			}
	
			if ( i === ien-1 ) {
				src.val = val;
				src.type = typeof val === 'function' ?
					'function' :
					$.isPlainObject( val ) ?
						'object' :
						'other';
			}
			else {
				struct = method ?
					src.methodExt :
					src.propExt;
			}
		}
	};
	
	_Api.registerPlural = _api_registerPlural = function ( pluralName, singularName, val ) {
		_Api.register( pluralName, val );
	
		_Api.register( singularName, function () {
			var ret = val.apply( this, arguments );
	
			if ( ret === this ) {
				// Returned item is the API instance that was passed in, return it
				return this;
			}
			else if ( ret instanceof _Api ) {
				// New API instance returned, want the value from the first item
				// in the returned array for the singular result.
				return ret.length ?
					Array.isArray( ret[0] ) ?
						new _Api( ret.context, ret[0] ) : // Array results are 'enhanced'
						ret[0] :
					undefined;
			}
	
			// Non-API return - just fire it back
			return ret;
		} );
	};
	
	
	/**
	 * Selector for HTML tables. Apply the given selector to the give array of
	 * DataTables settings objects.
	 *
	 * @param {string|integer} [selector] jQuery selector string or integer
	 * @param  {array} Array of DataTables settings objects to be filtered
	 * @return {array}
	 * @ignore
	 */
	var __table_selector = function ( selector, a )
	{
		if ( Array.isArray(selector) ) {
			return $.map( selector, function (item) {
				return __table_selector(item, a);
			} );
		}
	
		// Integer is used to pick out a table by index
		if ( typeof selector === 'number' ) {
			return [ a[ selector ] ];
		}
	
		// Perform a jQuery selector on the table nodes
		var nodes = $.map( a, function (el, i) {
			return el.nTable;
		} );
	
		return $(nodes)
			.filter( selector )
			.map( function (i) {
				// Need to translate back from the table node to the settings
				var idx = $.inArray( this, nodes );
				return a[ idx ];
			} )
			.toArray();
	};
	
	
	
	/**
	 * Context selector for the API's context (i.e. the tables the API instance
	 * refers to.
	 *
	 * @name    DataTable.Api#tables
	 * @param {string|integer} [selector] Selector to pick which tables the iterator
	 *   should operate on. If not given, all tables in the current context are
	 *   used. This can be given as a jQuery selector (for example `':gt(0)'`) to
	 *   select multiple tables or as an integer to select a single table.
	 * @returns {DataTable.Api} Returns a new API instance if a selector is given.
	 */
	_api_register( 'tables()', function ( selector ) {
		// A new instance is created if there was a selector specified
		return selector !== undefined && selector !== null ?
			new _Api( __table_selector( selector, this.context ) ) :
			this;
	} );
	
	
	_api_register( 'table()', function ( selector ) {
		var tables = this.tables( selector );
		var ctx = tables.context;
	
		// Truncate to the first matched table
		return ctx.length ?
			new _Api( ctx[0] ) :
			tables;
	} );
	
	
	_api_registerPlural( 'tables().nodes()', 'table().node()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTable;
		}, 1 );
	} );
	
	
	_api_registerPlural( 'tables().body()', 'table().body()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTBody;
		}, 1 );
	} );
	
	
	_api_registerPlural( 'tables().header()', 'table().header()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTHead;
		}, 1 );
	} );
	
	
	_api_registerPlural( 'tables().footer()', 'table().footer()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTFoot;
		}, 1 );
	} );
	
	
	_api_registerPlural( 'tables().containers()', 'table().container()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTableWrapper;
		}, 1 );
	} );
	
	
	
	/**
	 * Redraw the tables in the current context.
	 */
	_api_register( 'draw()', function ( paging ) {
		return this.iterator( 'table', function ( settings ) {
			if ( paging === 'page' ) {
				_fnDraw( settings );
			}
			else {
				if ( typeof paging === 'string' ) {
					paging = paging === 'full-hold' ?
						false :
						true;
				}
	
				_fnReDraw( settings, paging===false );
			}
		} );
	} );
	
	
	
	/**
	 * Get the current page index.
	 *
	 * @return {integer} Current page index (zero based)
	 *//**
	 * Set the current page.
	 *
	 * Note that if you attempt to show a page which does not exist, DataTables will
	 * not throw an error, but rather reset the paging.
	 *
	 * @param {integer|string} action The paging action to take. This can be one of:
	 *  * `integer` - The page index to jump to
	 *  * `string` - An action to take:
	 *    * `first` - Jump to first page.
	 *    * `next` - Jump to the next page
	 *    * `previous` - Jump to previous page
	 *    * `last` - Jump to the last page.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'page()', function ( action ) {
		if ( action === undefined ) {
			return this.page.info().page; // not an expensive call
		}
	
		// else, have an action to take on all tables
		return this.iterator( 'table', function ( settings ) {
			_fnPageChange( settings, action );
		} );
	} );
	
	
	/**
	 * Paging information for the first table in the current context.
	 *
	 * If you require paging information for another table, use the `table()` method
	 * with a suitable selector.
	 *
	 * @return {object} Object with the following properties set:
	 *  * `page` - Current page index (zero based - i.e. the first page is `0`)
	 *  * `pages` - Total number of pages
	 *  * `start` - Display index for the first record shown on the current page
	 *  * `end` - Display index for the last record shown on the current page
	 *  * `length` - Display length (number of records). Note that generally `start
	 *    + length = end`, but this is not always true, for example if there are
	 *    only 2 records to show on the final page, with a length of 10.
	 *  * `recordsTotal` - Full data set length
	 *  * `recordsDisplay` - Data set length once the current filtering criterion
	 *    are applied.
	 */
	_api_register( 'page.info()', function ( action ) {
		if ( this.context.length === 0 ) {
			return undefined;
		}
	
		var
			settings   = this.context[0],
			start      = settings._iDisplayStart,
			len        = settings.oFeatures.bPaginate ? settings._iDisplayLength : -1,
			visRecords = settings.fnRecordsDisplay(),
			all        = len === -1;
	
		return {
			"page":           all ? 0 : Math.floor( start / len ),
			"pages":          all ? 1 : Math.ceil( visRecords / len ),
			"start":          start,
			"end":            settings.fnDisplayEnd(),
			"length":         len,
			"recordsTotal":   settings.fnRecordsTotal(),
			"recordsDisplay": visRecords,
			"serverSide":     _fnDataSource( settings ) === 'ssp'
		};
	} );
	
	
	/**
	 * Get the current page length.
	 *
	 * @return {integer} Current page length. Note `-1` indicates that all records
	 *   are to be shown.
	 *//**
	 * Set the current page length.
	 *
	 * @param {integer} Page length to set. Use `-1` to show all records.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'page.len()', function ( len ) {
		// Note that we can't call this function 'length()' because `length`
		// is a Javascript property of functions which defines how many arguments
		// the function expects.
		if ( len === undefined ) {
			return this.context.length !== 0 ?
				this.context[0]._iDisplayLength :
				undefined;
		}
	
		// else, set the page length
		return this.iterator( 'table', function ( settings ) {
			_fnLengthChange( settings, len );
		} );
	} );
	
	
	
	var __reload = function ( settings, holdPosition, callback ) {
		// Use the draw event to trigger a callback
		if ( callback ) {
			var api = new _Api( settings );
	
			api.one( 'draw', function () {
				callback( api.ajax.json() );
			} );
		}
	
		if ( _fnDataSource( settings ) == 'ssp' ) {
			_fnReDraw( settings, holdPosition );
		}
		else {
			_fnProcessingDisplay( settings, true );
	
			// Cancel an existing request
			var xhr = settings.jqXHR;
			if ( xhr && xhr.readyState !== 4 ) {
				xhr.abort();
			}
	
			// Trigger xhr
			_fnBuildAjax( settings, [], function( json ) {
				_fnClearTable( settings );
	
				var data = _fnAjaxDataSrc( settings, json );
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					_fnAddData( settings, data[i] );
				}
	
				_fnReDraw( settings, holdPosition );
				_fnProcessingDisplay( settings, false );
			} );
		}
	};
	
	
	/**
	 * Get the JSON response from the last Ajax request that DataTables made to the
	 * server. Note that this returns the JSON from the first table in the current
	 * context.
	 *
	 * @return {object} JSON received from the server.
	 */
	_api_register( 'ajax.json()', function () {
		var ctx = this.context;
	
		if ( ctx.length > 0 ) {
			return ctx[0].json;
		}
	
		// else return undefined;
	} );
	
	
	/**
	 * Get the data submitted in the last Ajax request
	 */
	_api_register( 'ajax.params()', function () {
		var ctx = this.context;
	
		if ( ctx.length > 0 ) {
			return ctx[0].oAjaxData;
		}
	
		// else return undefined;
	} );
	
	
	/**
	 * Reload tables from the Ajax data source. Note that this function will
	 * automatically re-draw the table when the remote data has been loaded.
	 *
	 * @param {boolean} [reset=true] Reset (default) or hold the current paging
	 *   position. A full re-sort and re-filter is performed when this method is
	 *   called, which is why the pagination reset is the default action.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'ajax.reload()', function ( callback, resetPaging ) {
		return this.iterator( 'table', function (settings) {
			__reload( settings, resetPaging===false, callback );
		} );
	} );
	
	
	/**
	 * Get the current Ajax URL. Note that this returns the URL from the first
	 * table in the current context.
	 *
	 * @return {string} Current Ajax source URL
	 *//**
	 * Set the Ajax URL. Note that this will set the URL for all tables in the
	 * current context.
	 *
	 * @param {string} url URL to set.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'ajax.url()', function ( url ) {
		var ctx = this.context;
	
		if ( url === undefined ) {
			// get
			if ( ctx.length === 0 ) {
				return undefined;
			}
			ctx = ctx[0];
	
			return ctx.ajax ?
				$.isPlainObject( ctx.ajax ) ?
					ctx.ajax.url :
					ctx.ajax :
				ctx.sAjaxSource;
		}
	
		// set
		return this.iterator( 'table', function ( settings ) {
			if ( $.isPlainObject( settings.ajax ) ) {
				settings.ajax.url = url;
			}
			else {
				settings.ajax = url;
			}
			// No need to consider sAjaxSource here since DataTables gives priority
			// to `ajax` over `sAjaxSource`. So setting `ajax` here, renders any
			// value of `sAjaxSource` redundant.
		} );
	} );
	
	
	/**
	 * Load data from the newly set Ajax URL. Note that this method is only
	 * available when `ajax.url()` is used to set a URL. Additionally, this method
	 * has the same effect as calling `ajax.reload()` but is provided for
	 * convenience when setting a new URL. Like `ajax.reload()` it will
	 * automatically redraw the table once the remote data has been loaded.
	 *
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'ajax.url().load()', function ( callback, resetPaging ) {
		// Same as a reload, but makes sense to present it for easy access after a
		// url change
		return this.iterator( 'table', function ( ctx ) {
			__reload( ctx, resetPaging===false, callback );
		} );
	} );
	
	
	
	
	var _selector_run = function ( type, selector, selectFn, settings, opts )
	{
		var
			out = [], res,
			a, i, ien, j, jen,
			selectorType = typeof selector;
	
		// Can't just check for isArray here, as an API or jQuery instance might be
		// given with their array like look
		if ( ! selector || selectorType === 'string' || selectorType === 'function' || selector.length === undefined ) {
			selector = [ selector ];
		}
	
		for ( i=0, ien=selector.length ; i<ien ; i++ ) {
			// Only split on simple strings - complex expressions will be jQuery selectors
			a = selector[i] && selector[i].split && ! selector[i].match(/[\[\(:]/) ?
				selector[i].split(',') :
				[ selector[i] ];
	
			for ( j=0, jen=a.length ; j<jen ; j++ ) {
				res = selectFn( typeof a[j] === 'string' ? (a[j]).trim() : a[j] );
	
				if ( res && res.length ) {
					out = out.concat( res );
				}
			}
		}
	
		// selector extensions
		var ext = _ext.selector[ type ];
		if ( ext.length ) {
			for ( i=0, ien=ext.length ; i<ien ; i++ ) {
				out = ext[i]( settings, opts, out );
			}
		}
	
		return _unique( out );
	};
	
	
	var _selector_opts = function ( opts )
	{
		if ( ! opts ) {
			opts = {};
		}
	
		// Backwards compatibility for 1.9- which used the terminology filter rather
		// than search
		if ( opts.filter && opts.search === undefined ) {
			opts.search = opts.filter;
		}
	
		return $.extend( {
			search: 'none',
			order: 'current',
			page: 'all'
		}, opts );
	};
	
	
	var _selector_first = function ( inst )
	{
		// Reduce the API instance to the first item found
		for ( var i=0, ien=inst.length ; i<ien ; i++ ) {
			if ( inst[i].length > 0 ) {
				// Assign the first element to the first item in the instance
				// and truncate the instance and context
				inst[0] = inst[i];
				inst[0].length = 1;
				inst.length = 1;
				inst.context = [ inst.context[i] ];
	
				return inst;
			}
		}
	
		// Not found - return an empty instance
		inst.length = 0;
		return inst;
	};
	
	
	var _selector_row_indexes = function ( settings, opts )
	{
		var
			i, ien, tmp, a=[],
			displayFiltered = settings.aiDisplay,
			displayMaster = settings.aiDisplayMaster;
	
		var
			search = opts.search,  // none, applied, removed
			order  = opts.order,   // applied, current, index (original - compatibility with 1.9)
			page   = opts.page;    // all, current
	
		if ( _fnDataSource( settings ) == 'ssp' ) {
			// In server-side processing mode, most options are irrelevant since
			// rows not shown don't exist and the index order is the applied order
			// Removed is a special case - for consistency just return an empty
			// array
			return search === 'removed' ?
				[] :
				_range( 0, displayMaster.length );
		}
		else if ( page == 'current' ) {
			// Current page implies that order=current and fitler=applied, since it is
			// fairly senseless otherwise, regardless of what order and search actually
			// are
			for ( i=settings._iDisplayStart, ien=settings.fnDisplayEnd() ; i<ien ; i++ ) {
				a.push( displayFiltered[i] );
			}
		}
		else if ( order == 'current' || order == 'applied' ) {
			if ( search == 'none') {
				a = displayMaster.slice();
			}
			else if ( search == 'applied' ) {
				a = displayFiltered.slice();
			}
			else if ( search == 'removed' ) {
				// O(n+m) solution by creating a hash map
				var displayFilteredMap = {};
	
				for ( var i=0, ien=displayFiltered.length ; i<ien ; i++ ) {
					displayFilteredMap[displayFiltered[i]] = null;
				}
	
				a = $.map( displayMaster, function (el) {
					return ! displayFilteredMap.hasOwnProperty(el) ?
						el :
						null;
				} );
			}
		}
		else if ( order == 'index' || order == 'original' ) {
			for ( i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
				if ( search == 'none' ) {
					a.push( i );
				}
				else { // applied | removed
					tmp = $.inArray( i, displayFiltered );
	
					if ((tmp === -1 && search == 'removed') ||
						(tmp >= 0   && search == 'applied') )
					{
						a.push( i );
					}
				}
			}
		}
	
		return a;
	};
	
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 * Rows
	 *
	 * {}          - no selector - use all available rows
	 * {integer}   - row aoData index
	 * {node}      - TR node
	 * {string}    - jQuery selector to apply to the TR elements
	 * {array}     - jQuery array of nodes, or simply an array of TR nodes
	 *
	 */
	var __row_selector = function ( settings, selector, opts )
	{
		var rows;
		var run = function ( sel ) {
			var selInt = _intVal( sel );
			var i, ien;
			var aoData = settings.aoData;
	
			// Short cut - selector is a number and no options provided (default is
			// all records, so no need to check if the index is in there, since it
			// must be - dev error if the index doesn't exist).
			if ( selInt !== null && ! opts ) {
				return [ selInt ];
			}
	
			if ( ! rows ) {
				rows = _selector_row_indexes( settings, opts );
			}
	
			if ( selInt !== null && $.inArray( selInt, rows ) !== -1 ) {
				// Selector - integer
				return [ selInt ];
			}
			else if ( sel === null || sel === undefined || sel === '' ) {
				// Selector - none
				return rows;
			}
	
			// Selector - function
			if ( typeof sel === 'function' ) {
				return $.map( rows, function (idx) {
					var row = aoData[ idx ];
					return sel( idx, row._aData, row.nTr ) ? idx : null;
				} );
			}
	
			// Selector - node
			if ( sel.nodeName ) {
				var rowIdx = sel._DT_RowIndex;  // Property added by DT for fast lookup
				var cellIdx = sel._DT_CellIndex;
	
				if ( rowIdx !== undefined ) {
					// Make sure that the row is actually still present in the table
					return aoData[ rowIdx ] && aoData[ rowIdx ].nTr === sel ?
						[ rowIdx ] :
						[];
				}
				else if ( cellIdx ) {
					return aoData[ cellIdx.row ] && aoData[ cellIdx.row ].nTr === sel.parentNode ?
						[ cellIdx.row ] :
						[];
				}
				else {
					var host = $(sel).closest('*[data-dt-row]');
					return host.length ?
						[ host.data('dt-row') ] :
						[];
				}
			}
	
			// ID selector. Want to always be able to select rows by id, regardless
			// of if the tr element has been created or not, so can't rely upon
			// jQuery here - hence a custom implementation. This does not match
			// Sizzle's fast selector or HTML4 - in HTML5 the ID can be anything,
			// but to select it using a CSS selector engine (like Sizzle or
			// querySelect) it would need to need to be escaped for some characters.
			// DataTables simplifies this for row selectors since you can select
			// only a row. A # indicates an id any anything that follows is the id -
			// unescaped.
			if ( typeof sel === 'string' && sel.charAt(0) === '#' ) {
				// get row index from id
				var rowObj = settings.aIds[ sel.replace( /^#/, '' ) ];
				if ( rowObj !== undefined ) {
					return [ rowObj.idx ];
				}
	
				// need to fall through to jQuery in case there is DOM id that
				// matches
			}
			
			// Get nodes in the order from the `rows` array with null values removed
			var nodes = _removeEmpty(
				_pluck_order( settings.aoData, rows, 'nTr' )
			);
	
			// Selector - jQuery selector string, array of nodes or jQuery object/
			// As jQuery's .filter() allows jQuery objects to be passed in filter,
			// it also allows arrays, so this will cope with all three options
			return $(nodes)
				.filter( sel )
				.map( function () {
					return this._DT_RowIndex;
				} )
				.toArray();
		};
	
		return _selector_run( 'row', selector, run, settings, opts );
	};
	
	
	_api_register( 'rows()', function ( selector, opts ) {
		// argument shifting
		if ( selector === undefined ) {
			selector = '';
		}
		else if ( $.isPlainObject( selector ) ) {
			opts = selector;
			selector = '';
		}
	
		opts = _selector_opts( opts );
	
		var inst = this.iterator( 'table', function ( settings ) {
			return __row_selector( settings, selector, opts );
		}, 1 );
	
		// Want argument shifting here and in __row_selector?
		inst.selector.rows = selector;
		inst.selector.opts = opts;
	
		return inst;
	} );
	
	_api_register( 'rows().nodes()', function () {
		return this.iterator( 'row', function ( settings, row ) {
			return settings.aoData[ row ].nTr || undefined;
		}, 1 );
	} );
	
	_api_register( 'rows().data()', function () {
		return this.iterator( true, 'rows', function ( settings, rows ) {
			return _pluck_order( settings.aoData, rows, '_aData' );
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().cache()', 'row().cache()', function ( type ) {
		return this.iterator( 'row', function ( settings, row ) {
			var r = settings.aoData[ row ];
			return type === 'search' ? r._aFilterData : r._aSortData;
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().invalidate()', 'row().invalidate()', function ( src ) {
		return this.iterator( 'row', function ( settings, row ) {
			_fnInvalidate( settings, row, src );
		} );
	} );
	
	_api_registerPlural( 'rows().indexes()', 'row().index()', function () {
		return this.iterator( 'row', function ( settings, row ) {
			return row;
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().ids()', 'row().id()', function ( hash ) {
		var a = [];
		var context = this.context;
	
		// `iterator` will drop undefined values, but in this case we want them
		for ( var i=0, ien=context.length ; i<ien ; i++ ) {
			for ( var j=0, jen=this[i].length ; j<jen ; j++ ) {
				var id = context[i].rowIdFn( context[i].aoData[ this[i][j] ]._aData );
				a.push( (hash === true ? '#' : '' )+ id );
			}
		}
	
		return new _Api( context, a );
	} );
	
	_api_registerPlural( 'rows().remove()', 'row().remove()', function () {
		var that = this;
	
		this.iterator( 'row', function ( settings, row, thatIdx ) {
			var data = settings.aoData;
			var rowData = data[ row ];
			var i, ien, j, jen;
			var loopRow, loopCells;
	
			data.splice( row, 1 );
	
			// Update the cached indexes
			for ( i=0, ien=data.length ; i<ien ; i++ ) {
				loopRow = data[i];
				loopCells = loopRow.anCells;
	
				// Rows
				if ( loopRow.nTr !== null ) {
					loopRow.nTr._DT_RowIndex = i;
				}
	
				// Cells
				if ( loopCells !== null ) {
					for ( j=0, jen=loopCells.length ; j<jen ; j++ ) {
						loopCells[j]._DT_CellIndex.row = i;
					}
				}
			}
	
			// Delete from the display arrays
			_fnDeleteIndex( settings.aiDisplayMaster, row );
			_fnDeleteIndex( settings.aiDisplay, row );
			_fnDeleteIndex( that[ thatIdx ], row, false ); // maintain local indexes
	
			// For server-side processing tables - subtract the deleted row from the count
			if ( settings._iRecordsDisplay > 0 ) {
				settings._iRecordsDisplay--;
			}
	
			// Check for an 'overflow' they case for displaying the table
			_fnLengthOverflow( settings );
	
			// Remove the row's ID reference if there is one
			var id = settings.rowIdFn( rowData._aData );
			if ( id !== undefined ) {
				delete settings.aIds[ id ];
			}
		} );
	
		this.iterator( 'table', function ( settings ) {
			for ( var i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
				settings.aoData[i].idx = i;
			}
		} );
	
		return this;
	} );
	
	
	_api_register( 'rows.add()', function ( rows ) {
		var newRows = this.iterator( 'table', function ( settings ) {
				var row, i, ien;
				var out = [];
	
				for ( i=0, ien=rows.length ; i<ien ; i++ ) {
					row = rows[i];
	
					if ( row.nodeName && row.nodeName.toUpperCase() === 'TR' ) {
						out.push( _fnAddTr( settings, row )[0] );
					}
					else {
						out.push( _fnAddData( settings, row ) );
					}
				}
	
				return out;
			}, 1 );
	
		// Return an Api.rows() extended instance, so rows().nodes() etc can be used
		var modRows = this.rows( -1 );
		modRows.pop();
		$.merge( modRows, newRows );
	
		return modRows;
	} );
	
	
	
	
	
	/**
	 *
	 */
	_api_register( 'row()', function ( selector, opts ) {
		return _selector_first( this.rows( selector, opts ) );
	} );
	
	
	_api_register( 'row().data()', function ( data ) {
		var ctx = this.context;
	
		if ( data === undefined ) {
			// Get
			return ctx.length && this.length ?
				ctx[0].aoData[ this[0] ]._aData :
				undefined;
		}
	
		// Set
		var row = ctx[0].aoData[ this[0] ];
		row._aData = data;
	
		// If the DOM has an id, and the data source is an array
		if ( Array.isArray( data ) && row.nTr && row.nTr.id ) {
			_fnSetObjectDataFn( ctx[0].rowId )( data, row.nTr.id );
		}
	
		// Automatically invalidate
		_fnInvalidate( ctx[0], this[0], 'data' );
	
		return this;
	} );
	
	
	_api_register( 'row().node()', function () {
		var ctx = this.context;
	
		return ctx.length && this.length ?
			ctx[0].aoData[ this[0] ].nTr || null :
			null;
	} );
	
	
	_api_register( 'row.add()', function ( row ) {
		// Allow a jQuery object to be passed in - only a single row is added from
		// it though - the first element in the set
		if ( row instanceof $ && row.length ) {
			row = row[0];
		}
	
		var rows = this.iterator( 'table', function ( settings ) {
			if ( row.nodeName && row.nodeName.toUpperCase() === 'TR' ) {
				return _fnAddTr( settings, row )[0];
			}
			return _fnAddData( settings, row );
		} );
	
		// Return an Api.rows() extended instance, with the newly added row selected
		return this.row( rows[0] );
	} );
	
	
	
	var __details_add = function ( ctx, row, data, klass )
	{
		// Convert to array of TR elements
		var rows = [];
		var addRow = function ( r, k ) {
			// Recursion to allow for arrays of jQuery objects
			if ( Array.isArray( r ) || r instanceof $ ) {
				for ( var i=0, ien=r.length ; i<ien ; i++ ) {
					addRow( r[i], k );
				}
				return;
			}
	
			// If we get a TR element, then just add it directly - up to the dev
			// to add the correct number of columns etc
			if ( r.nodeName && r.nodeName.toLowerCase() === 'tr' ) {
				rows.push( r );
			}
			else {
				// Otherwise create a row with a wrapper
				var created = $('<tr><td></td></tr>').addClass( k );
				$('td', created)
					.addClass( k )
					.html( r )
					[0].colSpan = _fnVisbleColumns( ctx );
	
				rows.push( created[0] );
			}
		};
	
		addRow( data, klass );
	
		if ( row._details ) {
			row._details.detach();
		}
	
		row._details = $(rows);
	
		// If the children were already shown, that state should be retained
		if ( row._detailsShow ) {
			row._details.insertAfter( row.nTr );
		}
	};
	
	
	var __details_remove = function ( api, idx )
	{
		var ctx = api.context;
	
		if ( ctx.length ) {
			var row = ctx[0].aoData[ idx !== undefined ? idx : api[0] ];
	
			if ( row && row._details ) {
				row._details.remove();
	
				row._detailsShow = undefined;
				row._details = undefined;
			}
		}
	};
	
	
	var __details_display = function ( api, show ) {
		var ctx = api.context;
	
		if ( ctx.length && api.length ) {
			var row = ctx[0].aoData[ api[0] ];
	
			if ( row._details ) {
				row._detailsShow = show;
	
				if ( show ) {
					row._details.insertAfter( row.nTr );
				}
				else {
					row._details.detach();
				}
	
				__details_events( ctx[0] );
			}
		}
	};
	
	
	var __details_events = function ( settings )
	{
		var api = new _Api( settings );
		var namespace = '.dt.DT_details';
		var drawEvent = 'draw'+namespace;
		var colvisEvent = 'column-visibility'+namespace;
		var destroyEvent = 'destroy'+namespace;
		var data = settings.aoData;
	
		api.off( drawEvent +' '+ colvisEvent +' '+ destroyEvent );
	
		if ( _pluck( data, '_details' ).length > 0 ) {
			// On each draw, insert the required elements into the document
			api.on( drawEvent, function ( e, ctx ) {
				if ( settings !== ctx ) {
					return;
				}
	
				api.rows( {page:'current'} ).eq(0).each( function (idx) {
					// Internal data grab
					var row = data[ idx ];
	
					if ( row._detailsShow ) {
						row._details.insertAfter( row.nTr );
					}
				} );
			} );
	
			// Column visibility change - update the colspan
			api.on( colvisEvent, function ( e, ctx, idx, vis ) {
				if ( settings !== ctx ) {
					return;
				}
	
				// Update the colspan for the details rows (note, only if it already has
				// a colspan)
				var row, visible = _fnVisbleColumns( ctx );
	
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					row = data[i];
	
					if ( row._details ) {
						row._details.children('td[colspan]').attr('colspan', visible );
					}
				}
			} );
	
			// Table destroyed - nuke any child rows
			api.on( destroyEvent, function ( e, ctx ) {
				if ( settings !== ctx ) {
					return;
				}
	
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					if ( data[i]._details ) {
						__details_remove( api, i );
					}
				}
			} );
		}
	};
	
	// Strings for the method names to help minification
	var _emp = '';
	var _child_obj = _emp+'row().child';
	var _child_mth = _child_obj+'()';
	
	// data can be:
	//  tr
	//  string
	//  jQuery or array of any of the above
	_api_register( _child_mth, function ( data, klass ) {
		var ctx = this.context;
	
		if ( data === undefined ) {
			// get
			return ctx.length && this.length ?
				ctx[0].aoData[ this[0] ]._details :
				undefined;
		}
		else if ( data === true ) {
			// show
			this.child.show();
		}
		else if ( data === false ) {
			// remove
			__details_remove( this );
		}
		else if ( ctx.length && this.length ) {
			// set
			__details_add( ctx[0], ctx[0].aoData[ this[0] ], data, klass );
		}
	
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.show()',
		_child_mth+'.show()' // only when `child()` was called with parameters (without
	], function ( show ) {   // it returns an object and this method is not executed)
		__details_display( this, true );
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.hide()',
		_child_mth+'.hide()' // only when `child()` was called with parameters (without
	], function () {         // it returns an object and this method is not executed)
		__details_display( this, false );
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.remove()',
		_child_mth+'.remove()' // only when `child()` was called with parameters (without
	], function () {           // it returns an object and this method is not executed)
		__details_remove( this );
		return this;
	} );
	
	
	_api_register( _child_obj+'.isShown()', function () {
		var ctx = this.context;
	
		if ( ctx.length && this.length ) {
			// _detailsShown as false or undefined will fall through to return false
			return ctx[0].aoData[ this[0] ]._detailsShow || false;
		}
		return false;
	} );
	
	
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 * Columns
	 *
	 * {integer}           - column index (>=0 count from left, <0 count from right)
	 * "{integer}:visIdx"  - visible column index (i.e. translate to column index)  (>=0 count from left, <0 count from right)
	 * "{integer}:visible" - alias for {integer}:visIdx  (>=0 count from left, <0 count from right)
	 * "{string}:name"     - column name
	 * "{string}"          - jQuery selector on column header nodes
	 *
	 */
	
	// can be an array of these items, comma separated list, or an array of comma
	// separated lists
	
	var __re_column_selector = /^([^:]+):(name|visIdx|visible)$/;
	
	
	// r1 and r2 are redundant - but it means that the parameters match for the
	// iterator callback in columns().data()
	var __columnData = function ( settings, column, r1, r2, rows ) {
		var a = [];
		for ( var row=0, ien=rows.length ; row<ien ; row++ ) {
			a.push( _fnGetCellData( settings, rows[row], column ) );
		}
		return a;
	};
	
	
	var __column_selector = function ( settings, selector, opts )
	{
		var
			columns = settings.aoColumns,
			names = _pluck( columns, 'sName' ),
			nodes = _pluck( columns, 'nTh' );
	
		var run = function ( s ) {
			var selInt = _intVal( s );
	
			// Selector - all
			if ( s === '' ) {
				return _range( columns.length );
			}
	
			// Selector - index
			if ( selInt !== null ) {
				return [ selInt >= 0 ?
					selInt : // Count from left
					columns.length + selInt // Count from right (+ because its a negative value)
				];
			}
	
			// Selector = function
			if ( typeof s === 'function' ) {
				var rows = _selector_row_indexes( settings, opts );
	
				return $.map( columns, function (col, idx) {
					return s(
							idx,
							__columnData( settings, idx, 0, 0, rows ),
							nodes[ idx ]
						) ? idx : null;
				} );
			}
	
			// jQuery or string selector
			var match = typeof s === 'string' ?
				s.match( __re_column_selector ) :
				'';
	
			if ( match ) {
				switch( match[2] ) {
					case 'visIdx':
					case 'visible':
						var idx = parseInt( match[1], 10 );
						// Visible index given, convert to column index
						if ( idx < 0 ) {
							// Counting from the right
							var visColumns = $.map( columns, function (col,i) {
								return col.bVisible ? i : null;
							} );
							return [ visColumns[ visColumns.length + idx ] ];
						}
						// Counting from the left
						return [ _fnVisibleToColumnIndex( settings, idx ) ];
	
					case 'name':
						// match by name. `names` is column index complete and in order
						return $.map( names, function (name, i) {
							return name === match[1] ? i : null;
						} );
	
					default:
						return [];
				}
			}
	
			// Cell in the table body
			if ( s.nodeName && s._DT_CellIndex ) {
				return [ s._DT_CellIndex.column ];
			}
	
			// jQuery selector on the TH elements for the columns
			var jqResult = $( nodes )
				.filter( s )
				.map( function () {
					return $.inArray( this, nodes ); // `nodes` is column index complete and in order
				} )
				.toArray();
	
			if ( jqResult.length || ! s.nodeName ) {
				return jqResult;
			}
	
			// Otherwise a node which might have a `dt-column` data attribute, or be
			// a child or such an element
			var host = $(s).closest('*[data-dt-column]');
			return host.length ?
				[ host.data('dt-column') ] :
				[];
		};
	
		return _selector_run( 'column', selector, run, settings, opts );
	};
	
	
	var __setColumnVis = function ( settings, column, vis ) {
		var
			cols = settings.aoColumns,
			col  = cols[ column ],
			data = settings.aoData,
			row, cells, i, ien, tr;
	
		// Get
		if ( vis === undefined ) {
			return col.bVisible;
		}
	
		// Set
		// No change
		if ( col.bVisible === vis ) {
			return;
		}
	
		if ( vis ) {
			// Insert column
			// Need to decide if we should use appendChild or insertBefore
			var insertBefore = $.inArray( true, _pluck(cols, 'bVisible'), column+1 );
	
			for ( i=0, ien=data.length ; i<ien ; i++ ) {
				tr = data[i].nTr;
				cells = data[i].anCells;
	
				if ( tr ) {
					// insertBefore can act like appendChild if 2nd arg is null
					tr.insertBefore( cells[ column ], cells[ insertBefore ] || null );
				}
			}
		}
		else {
			// Remove column
			$( _pluck( settings.aoData, 'anCells', column ) ).detach();
		}
	
		// Common actions
		col.bVisible = vis;
	};
	
	
	_api_register( 'columns()', function ( selector, opts ) {
		// argument shifting
		if ( selector === undefined ) {
			selector = '';
		}
		else if ( $.isPlainObject( selector ) ) {
			opts = selector;
			selector = '';
		}
	
		opts = _selector_opts( opts );
	
		var inst = this.iterator( 'table', function ( settings ) {
			return __column_selector( settings, selector, opts );
		}, 1 );
	
		// Want argument shifting here and in _row_selector?
		inst.selector.cols = selector;
		inst.selector.opts = opts;
	
		return inst;
	} );
	
	_api_registerPlural( 'columns().header()', 'column().header()', function ( selector, opts ) {
		return this.iterator( 'column', function ( settings, column ) {
			return settings.aoColumns[column].nTh;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().footer()', 'column().footer()', function ( selector, opts ) {
		return this.iterator( 'column', function ( settings, column ) {
			return settings.aoColumns[column].nTf;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().data()', 'column().data()', function () {
		return this.iterator( 'column-rows', __columnData, 1 );
	} );
	
	_api_registerPlural( 'columns().dataSrc()', 'column().dataSrc()', function () {
		return this.iterator( 'column', function ( settings, column ) {
			return settings.aoColumns[column].mData;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().cache()', 'column().cache()', function ( type ) {
		return this.iterator( 'column-rows', function ( settings, column, i, j, rows ) {
			return _pluck_order( settings.aoData, rows,
				type === 'search' ? '_aFilterData' : '_aSortData', column
			);
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().nodes()', 'column().nodes()', function () {
		return this.iterator( 'column-rows', function ( settings, column, i, j, rows ) {
			return _pluck_order( settings.aoData, rows, 'anCells', column ) ;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().visible()', 'column().visible()', function ( vis, calc ) {
		var that = this;
		var ret = this.iterator( 'column', function ( settings, column ) {
			if ( vis === undefined ) {
				return settings.aoColumns[ column ].bVisible;
			} // else
			__setColumnVis( settings, column, vis );
		} );
	
		// Group the column visibility changes
		if ( vis !== undefined ) {
			this.iterator( 'table', function ( settings ) {
				// Redraw the header after changes
				_fnDrawHead( settings, settings.aoHeader );
				_fnDrawHead( settings, settings.aoFooter );
		
				// Update colspan for no records display. Child rows and extensions will use their own
				// listeners to do this - only need to update the empty table item here
				if ( ! settings.aiDisplay.length ) {
					$(settings.nTBody).find('td[colspan]').attr('colspan', _fnVisbleColumns(settings));
				}
		
				_fnSaveState( settings );
	
				// Second loop once the first is done for events
				that.iterator( 'column', function ( settings, column ) {
					_fnCallbackFire( settings, null, 'column-visibility', [settings, column, vis, calc] );
				} );
	
				if ( calc === undefined || calc ) {
					that.columns.adjust();
				}
			});
		}
	
		return ret;
	} );
	
	_api_registerPlural( 'columns().indexes()', 'column().index()', function ( type ) {
		return this.iterator( 'column', function ( settings, column ) {
			return type === 'visible' ?
				_fnColumnIndexToVisible( settings, column ) :
				column;
		}, 1 );
	} );
	
	_api_register( 'columns.adjust()', function () {
		return this.iterator( 'table', function ( settings ) {
			_fnAdjustColumnSizing( settings );
		}, 1 );
	} );
	
	_api_register( 'column.index()', function ( type, idx ) {
		if ( this.context.length !== 0 ) {
			var ctx = this.context[0];
	
			if ( type === 'fromVisible' || type === 'toData' ) {
				return _fnVisibleToColumnIndex( ctx, idx );
			}
			else if ( type === 'fromData' || type === 'toVisible' ) {
				return _fnColumnIndexToVisible( ctx, idx );
			}
		}
	} );
	
	_api_register( 'column()', function ( selector, opts ) {
		return _selector_first( this.columns( selector, opts ) );
	} );
	
	var __cell_selector = function ( settings, selector, opts )
	{
		var data = settings.aoData;
		var rows = _selector_row_indexes( settings, opts );
		var cells = _removeEmpty( _pluck_order( data, rows, 'anCells' ) );
		var allCells = $(_flatten( [], cells ));
		var row;
		var columns = settings.aoColumns.length;
		var a, i, ien, j, o, host;
	
		var run = function ( s ) {
			var fnSelector = typeof s === 'function';
	
			if ( s === null || s === undefined || fnSelector ) {
				// All cells and function selectors
				a = [];
	
				for ( i=0, ien=rows.length ; i<ien ; i++ ) {
					row = rows[i];
	
					for ( j=0 ; j<columns ; j++ ) {
						o = {
							row: row,
							column: j
						};
	
						if ( fnSelector ) {
							// Selector - function
							host = data[ row ];
	
							if ( s( o, _fnGetCellData(settings, row, j), host.anCells ? host.anCells[j] : null ) ) {
								a.push( o );
							}
						}
						else {
							// Selector - all
							a.push( o );
						}
					}
				}
	
				return a;
			}
			
			// Selector - index
			if ( $.isPlainObject( s ) ) {
				// Valid cell index and its in the array of selectable rows
				return s.column !== undefined && s.row !== undefined && $.inArray( s.row, rows ) !== -1 ?
					[s] :
					[];
			}
	
			// Selector - jQuery filtered cells
			var jqResult = allCells
				.filter( s )
				.map( function (i, el) {
					return { // use a new object, in case someone changes the values
						row:    el._DT_CellIndex.row,
						column: el._DT_CellIndex.column
	 				};
				} )
				.toArray();
	
			if ( jqResult.length || ! s.nodeName ) {
				return jqResult;
			}
	
			// Otherwise the selector is a node, and there is one last option - the
			// element might be a child of an element which has dt-row and dt-column
			// data attributes
			host = $(s).closest('*[data-dt-row]');
			return host.length ?
				[ {
					row: host.data('dt-row'),
					column: host.data('dt-column')
				} ] :
				[];
		};
	
		return _selector_run( 'cell', selector, run, settings, opts );
	};
	
	
	
	
	_api_register( 'cells()', function ( rowSelector, columnSelector, opts ) {
		// Argument shifting
		if ( $.isPlainObject( rowSelector ) ) {
			// Indexes
			if ( rowSelector.row === undefined ) {
				// Selector options in first parameter
				opts = rowSelector;
				rowSelector = null;
			}
			else {
				// Cell index objects in first parameter
				opts = columnSelector;
				columnSelector = null;
			}
		}
		if ( $.isPlainObject( columnSelector ) ) {
			opts = columnSelector;
			columnSelector = null;
		}
	
		// Cell selector
		if ( columnSelector === null || columnSelector === undefined ) {
			return this.iterator( 'table', function ( settings ) {
				return __cell_selector( settings, rowSelector, _selector_opts( opts ) );
			} );
		}
	
		// The default built in options need to apply to row and columns
		var internalOpts = opts ? {
			page: opts.page,
			order: opts.order,
			search: opts.search
		} : {};
	
		// Row + column selector
		var columns = this.columns( columnSelector, internalOpts );
		var rows = this.rows( rowSelector, internalOpts );
		var i, ien, j, jen;
	
		var cellsNoOpts = this.iterator( 'table', function ( settings, idx ) {
			var a = [];
	
			for ( i=0, ien=rows[idx].length ; i<ien ; i++ ) {
				for ( j=0, jen=columns[idx].length ; j<jen ; j++ ) {
					a.push( {
						row:    rows[idx][i],
						column: columns[idx][j]
					} );
				}
			}
	
			return a;
		}, 1 );
	
		// There is currently only one extension which uses a cell selector extension
		// It is a _major_ performance drag to run this if it isn't needed, so this is
		// an extension specific check at the moment
		var cells = opts && opts.selected ?
			this.cells( cellsNoOpts, opts ) :
			cellsNoOpts;
	
		$.extend( cells.selector, {
			cols: columnSelector,
			rows: rowSelector,
			opts: opts
		} );
	
		return cells;
	} );
	
	
	_api_registerPlural( 'cells().nodes()', 'cell().node()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			var data = settings.aoData[ row ];
	
			return data && data.anCells ?
				data.anCells[ column ] :
				undefined;
		}, 1 );
	} );
	
	
	_api_register( 'cells().data()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return _fnGetCellData( settings, row, column );
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().cache()', 'cell().cache()', function ( type ) {
		type = type === 'search' ? '_aFilterData' : '_aSortData';
	
		return this.iterator( 'cell', function ( settings, row, column ) {
			return settings.aoData[ row ][ type ][ column ];
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().render()', 'cell().render()', function ( type ) {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return _fnGetCellData( settings, row, column, type );
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().indexes()', 'cell().index()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return {
				row: row,
				column: column,
				columnVisible: _fnColumnIndexToVisible( settings, column )
			};
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().invalidate()', 'cell().invalidate()', function ( src ) {
		return this.iterator( 'cell', function ( settings, row, column ) {
			_fnInvalidate( settings, row, src, column );
		} );
	} );
	
	
	
	_api_register( 'cell()', function ( rowSelector, columnSelector, opts ) {
		return _selector_first( this.cells( rowSelector, columnSelector, opts ) );
	} );
	
	
	_api_register( 'cell().data()', function ( data ) {
		var ctx = this.context;
		var cell = this[0];
	
		if ( data === undefined ) {
			// Get
			return ctx.length && cell.length ?
				_fnGetCellData( ctx[0], cell[0].row, cell[0].column ) :
				undefined;
		}
	
		// Set
		_fnSetCellData( ctx[0], cell[0].row, cell[0].column, data );
		_fnInvalidate( ctx[0], cell[0].row, 'data', cell[0].column );
	
		return this;
	} );
	
	
	
	/**
	 * Get current ordering (sorting) that has been applied to the table.
	 *
	 * @returns {array} 2D array containing the sorting information for the first
	 *   table in the current context. Each element in the parent array represents
	 *   a column being sorted upon (i.e. multi-sorting with two columns would have
	 *   2 inner arrays). The inner arrays may have 2 or 3 elements. The first is
	 *   the column index that the sorting condition applies to, the second is the
	 *   direction of the sort (`desc` or `asc`) and, optionally, the third is the
	 *   index of the sorting order from the `column.sorting` initialisation array.
	 *//**
	 * Set the ordering for the table.
	 *
	 * @param {integer} order Column index to sort upon.
	 * @param {string} direction Direction of the sort to be applied (`asc` or `desc`)
	 * @returns {DataTables.Api} this
	 *//**
	 * Set the ordering for the table.
	 *
	 * @param {array} order 1D array of sorting information to be applied.
	 * @param {array} [...] Optional additional sorting conditions
	 * @returns {DataTables.Api} this
	 *//**
	 * Set the ordering for the table.
	 *
	 * @param {array} order 2D array of sorting information to be applied.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'order()', function ( order, dir ) {
		var ctx = this.context;
	
		if ( order === undefined ) {
			// get
			return ctx.length !== 0 ?
				ctx[0].aaSorting :
				undefined;
		}
	
		// set
		if ( typeof order === 'number' ) {
			// Simple column / direction passed in
			order = [ [ order, dir ] ];
		}
		else if ( order.length && ! Array.isArray( order[0] ) ) {
			// Arguments passed in (list of 1D arrays)
			order = Array.prototype.slice.call( arguments );
		}
		// otherwise a 2D array was passed in
	
		return this.iterator( 'table', function ( settings ) {
			settings.aaSorting = order.slice();
		} );
	} );
	
	
	/**
	 * Attach a sort listener to an element for a given column
	 *
	 * @param {node|jQuery|string} node Identifier for the element(s) to attach the
	 *   listener to. This can take the form of a single DOM node, a jQuery
	 *   collection of nodes or a jQuery selector which will identify the node(s).
	 * @param {integer} column the column that a click on this node will sort on
	 * @param {function} [callback] callback function when sort is run
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'order.listener()', function ( node, column, callback ) {
		return this.iterator( 'table', function ( settings ) {
			_fnSortAttachListener( settings, node, column, callback );
		} );
	} );
	
	
	_api_register( 'order.fixed()', function ( set ) {
		if ( ! set ) {
			var ctx = this.context;
			var fixed = ctx.length ?
				ctx[0].aaSortingFixed :
				undefined;
	
			return Array.isArray( fixed ) ?
				{ pre: fixed } :
				fixed;
		}
	
		return this.iterator( 'table', function ( settings ) {
			settings.aaSortingFixed = $.extend( true, {}, set );
		} );
	} );
	
	
	// Order by the selected column(s)
	_api_register( [
		'columns().order()',
		'column().order()'
	], function ( dir ) {
		var that = this;
	
		return this.iterator( 'table', function ( settings, i ) {
			var sort = [];
	
			$.each( that[i], function (j, col) {
				sort.push( [ col, dir ] );
			} );
	
			settings.aaSorting = sort;
		} );
	} );
	
	
	
	_api_register( 'search()', function ( input, regex, smart, caseInsen ) {
		var ctx = this.context;
	
		if ( input === undefined ) {
			// get
			return ctx.length !== 0 ?
				ctx[0].oPreviousSearch.sSearch :
				undefined;
		}
	
		// set
		return this.iterator( 'table', function ( settings ) {
			if ( ! settings.oFeatures.bFilter ) {
				return;
			}
	
			_fnFilterComplete( settings, $.extend( {}, settings.oPreviousSearch, {
				"sSearch": input+"",
				"bRegex":  regex === null ? false : regex,
				"bSmart":  smart === null ? true  : smart,
				"bCaseInsensitive": caseInsen === null ? true : caseInsen
			} ), 1 );
		} );
	} );
	
	
	_api_registerPlural(
		'columns().search()',
		'column().search()',
		function ( input, regex, smart, caseInsen ) {
			return this.iterator( 'column', function ( settings, column ) {
				var preSearch = settings.aoPreSearchCols;
	
				if ( input === undefined ) {
					// get
					return preSearch[ column ].sSearch;
				}
	
				// set
				if ( ! settings.oFeatures.bFilter ) {
					return;
				}
	
				$.extend( preSearch[ column ], {
					"sSearch": input+"",
					"bRegex":  regex === null ? false : regex,
					"bSmart":  smart === null ? true  : smart,
					"bCaseInsensitive": caseInsen === null ? true : caseInsen
				} );
	
				_fnFilterComplete( settings, settings.oPreviousSearch, 1 );
			} );
		}
	);
	
	/*
	 * State API methods
	 */
	
	_api_register( 'state()', function () {
		return this.context.length ?
			this.context[0].oSavedState :
			null;
	} );
	
	
	_api_register( 'state.clear()', function () {
		return this.iterator( 'table', function ( settings ) {
			// Save an empty object
			settings.fnStateSaveCallback.call( settings.oInstance, settings, {} );
		} );
	} );
	
	
	_api_register( 'state.loaded()', function () {
		return this.context.length ?
			this.context[0].oLoadedState :
			null;
	} );
	
	
	_api_register( 'state.save()', function () {
		return this.iterator( 'table', function ( settings ) {
			_fnSaveState( settings );
		} );
	} );
	
	
	
	/**
	 * Provide a common method for plug-ins to check the version of DataTables being
	 * used, in order to ensure compatibility.
	 *
	 *  @param {string} version Version string to check for, in the format "X.Y.Z".
	 *    Note that the formats "X" and "X.Y" are also acceptable.
	 *  @returns {boolean} true if this version of DataTables is greater or equal to
	 *    the required version, or false if this version of DataTales is not
	 *    suitable
	 *  @static
	 *  @dtopt API-Static
	 *
	 *  @example
	 *    alert( $.fn.dataTable.versionCheck( '1.9.0' ) );
	 */
	DataTable.versionCheck = DataTable.fnVersionCheck = function( version )
	{
		var aThis = DataTable.version.split('.');
		var aThat = version.split('.');
		var iThis, iThat;
	
		for ( var i=0, iLen=aThat.length ; i<iLen ; i++ ) {
			iThis = parseInt( aThis[i], 10 ) || 0;
			iThat = parseInt( aThat[i], 10 ) || 0;
	
			// Parts are the same, keep comparing
			if (iThis === iThat) {
				continue;
			}
	
			// Parts are different, return immediately
			return iThis > iThat;
		}
	
		return true;
	};
	
	
	/**
	 * Check if a `<table>` node is a DataTable table already or not.
	 *
	 *  @param {node|jquery|string} table Table node, jQuery object or jQuery
	 *      selector for the table to test. Note that if more than more than one
	 *      table is passed on, only the first will be checked
	 *  @returns {boolean} true the table given is a DataTable, or false otherwise
	 *  @static
	 *  @dtopt API-Static
	 *
	 *  @example
	 *    if ( ! $.fn.DataTable.isDataTable( '#example' ) ) {
	 *      $('#example').dataTable();
	 *    }
	 */
	DataTable.isDataTable = DataTable.fnIsDataTable = function ( table )
	{
		var t = $(table).get(0);
		var is = false;
	
		if ( table instanceof DataTable.Api ) {
			return true;
		}
	
		$.each( DataTable.settings, function (i, o) {
			var head = o.nScrollHead ? $('table', o.nScrollHead)[0] : null;
			var foot = o.nScrollFoot ? $('table', o.nScrollFoot)[0] : null;
	
			if ( o.nTable === t || head === t || foot === t ) {
				is = true;
			}
		} );
	
		return is;
	};
	
	
	/**
	 * Get all DataTable tables that have been initialised - optionally you can
	 * select to get only currently visible tables.
	 *
	 *  @param {boolean} [visible=false] Flag to indicate if you want all (default)
	 *    or visible tables only.
	 *  @returns {array} Array of `table` nodes (not DataTable instances) which are
	 *    DataTables
	 *  @static
	 *  @dtopt API-Static
	 *
	 *  @example
	 *    $.each( $.fn.dataTable.tables(true), function () {
	 *      $(table).DataTable().columns.adjust();
	 *    } );
	 */
	DataTable.tables = DataTable.fnTables = function ( visible )
	{
		var api = false;
	
		if ( $.isPlainObject( visible ) ) {
			api = visible.api;
			visible = visible.visible;
		}
	
		var a = $.map( DataTable.settings, function (o) {
			if ( !visible || (visible && $(o.nTable).is(':visible')) ) {
				return o.nTable;
			}
		} );
	
		return api ?
			new _Api( a ) :
			a;
	};
	
	
	/**
	 * Convert from camel case parameters to Hungarian notation. This is made public
	 * for the extensions to provide the same ability as DataTables core to accept
	 * either the 1.9 style Hungarian notation, or the 1.10+ style camelCase
	 * parameters.
	 *
	 *  @param {object} src The model object which holds all parameters that can be
	 *    mapped.
	 *  @param {object} user The object to convert from camel case to Hungarian.
	 *  @param {boolean} force When set to `true`, properties which already have a
	 *    Hungarian value in the `user` object will be overwritten. Otherwise they
	 *    won't be.
	 */
	DataTable.camelToHungarian = _fnCamelToHungarian;
	
	
	
	/**
	 *
	 */
	_api_register( '$()', function ( selector, opts ) {
		var
			rows   = this.rows( opts ).nodes(), // Get all rows
			jqRows = $(rows);
	
		return $( [].concat(
			jqRows.filter( selector ).toArray(),
			jqRows.find( selector ).toArray()
		) );
	} );
	
	
	// jQuery functions to operate on the tables
	$.each( [ 'on', 'one', 'off' ], function (i, key) {
		_api_register( key+'()', function ( /* event, handler */ ) {
			var args = Array.prototype.slice.call(arguments);
	
			// Add the `dt` namespace automatically if it isn't already present
			args[0] = $.map( args[0].split( /\s/ ), function ( e ) {
				return ! e.match(/\.dt\b/) ?
					e+'.dt' :
					e;
				} ).join( ' ' );
	
			var inst = $( this.tables().nodes() );
			inst[key].apply( inst, args );
			return this;
		} );
	} );
	
	
	_api_register( 'clear()', function () {
		return this.iterator( 'table', function ( settings ) {
			_fnClearTable( settings );
		} );
	} );
	
	
	_api_register( 'settings()', function () {
		return new _Api( this.context, this.context );
	} );
	
	
	_api_register( 'init()', function () {
		var ctx = this.context;
		return ctx.length ? ctx[0].oInit : null;
	} );
	
	
	_api_register( 'data()', function () {
		return this.iterator( 'table', function ( settings ) {
			return _pluck( settings.aoData, '_aData' );
		} ).flatten();
	} );
	
	
	_api_register( 'destroy()', function ( remove ) {
		remove = remove || false;
	
		return this.iterator( 'table', function ( settings ) {
			var orig      = settings.nTableWrapper.parentNode;
			var classes   = settings.oClasses;
			var table     = settings.nTable;
			var tbody     = settings.nTBody;
			var thead     = settings.nTHead;
			var tfoot     = settings.nTFoot;
			var jqTable   = $(table);
			var jqTbody   = $(tbody);
			var jqWrapper = $(settings.nTableWrapper);
			var rows      = $.map( settings.aoData, function (r) { return r.nTr; } );
			var i, ien;
	
			// Flag to note that the table is currently being destroyed - no action
			// should be taken
			settings.bDestroying = true;
	
			// Fire off the destroy callbacks for plug-ins etc
			_fnCallbackFire( settings, "aoDestroyCallback", "destroy", [settings] );
	
			// If not being removed from the document, make all columns visible
			if ( ! remove ) {
				new _Api( settings ).columns().visible( true );
			}
	
			// Blitz all `DT` namespaced events (these are internal events, the
			// lowercase, `dt` events are user subscribed and they are responsible
			// for removing them
			jqWrapper.off('.DT').find(':not(tbody *)').off('.DT');
			$(window).off('.DT-'+settings.sInstance);
	
			// When scrolling we had to break the table up - restore it
			if ( table != thead.parentNode ) {
				jqTable.children('thead').detach();
				jqTable.append( thead );
			}
	
			if ( tfoot && table != tfoot.parentNode ) {
				jqTable.children('tfoot').detach();
				jqTable.append( tfoot );
			}
	
			settings.aaSorting = [];
			settings.aaSortingFixed = [];
			_fnSortingClasses( settings );
	
			$( rows ).removeClass( settings.asStripeClasses.join(' ') );
	
			$('th, td', thead).removeClass( classes.sSortable+' '+
				classes.sSortableAsc+' '+classes.sSortableDesc+' '+classes.sSortableNone
			);
	
			// Add the TR elements back into the table in their original order
			jqTbody.children().detach();
			jqTbody.append( rows );
	
			// Remove the DataTables generated nodes, events and classes
			var removedMethod = remove ? 'remove' : 'detach';
			jqTable[ removedMethod ]();
			jqWrapper[ removedMethod ]();
	
			// If we need to reattach the table to the document
			if ( ! remove && orig ) {
				// insertBefore acts like appendChild if !arg[1]
				orig.insertBefore( table, settings.nTableReinsertBefore );
	
				// Restore the width of the original table - was read from the style property,
				// so we can restore directly to that
				jqTable
					.css( 'width', settings.sDestroyWidth )
					.removeClass( classes.sTable );
	
				// If the were originally stripe classes - then we add them back here.
				// Note this is not fool proof (for example if not all rows had stripe
				// classes - but it's a good effort without getting carried away
				ien = settings.asDestroyStripes.length;
	
				if ( ien ) {
					jqTbody.children().each( function (i) {
						$(this).addClass( settings.asDestroyStripes[i % ien] );
					} );
				}
			}
	
			/* Remove the settings object from the settings array */
			var idx = $.inArray( settings, DataTable.settings );
			if ( idx !== -1 ) {
				DataTable.settings.splice( idx, 1 );
			}
		} );
	} );
	
	
	// Add the `every()` method for rows, columns and cells in a compact form
	$.each( [ 'column', 'row', 'cell' ], function ( i, type ) {
		_api_register( type+'s().every()', function ( fn ) {
			var opts = this.selector.opts;
			var api = this;
	
			return this.iterator( type, function ( settings, arg1, arg2, arg3, arg4 ) {
				// Rows and columns:
				//  arg1 - index
				//  arg2 - table counter
				//  arg3 - loop counter
				//  arg4 - undefined
				// Cells:
				//  arg1 - row index
				//  arg2 - column index
				//  arg3 - table counter
				//  arg4 - loop counter
				fn.call(
					api[ type ](
						arg1,
						type==='cell' ? arg2 : opts,
						type==='cell' ? opts : undefined
					),
					arg1, arg2, arg3, arg4
				);
			} );
		} );
	} );
	
	
	// i18n method for extensions to be able to use the language object from the
	// DataTable
	_api_register( 'i18n()', function ( token, def, plural ) {
		var ctx = this.context[0];
		var resolved = _fnGetObjectDataFn( token )( ctx.oLanguage );
	
		if ( resolved === undefined ) {
			resolved = def;
		}
	
		if ( plural !== undefined && $.isPlainObject( resolved ) ) {
			resolved = resolved[ plural ] !== undefined ?
				resolved[ plural ] :
				resolved._;
		}
	
		return resolved.replace( '%d', plural ); // nb: plural might be undefined,
	} );
	/**
	 * Version string for plug-ins to check compatibility. Allowed format is
	 * `a.b.c-d` where: a:int, b:int, c:int, d:string(dev|beta|alpha). `d` is used
	 * only for non-release builds. See http://semver.org/ for more information.
	 *  @member
	 *  @type string
	 *  @default Version number
	 */
	DataTable.version = "1.10.23";

	/**
	 * Private data store, containing all of the settings objects that are
	 * created for the tables on a given page.
	 *
	 * Note that the `DataTable.settings` object is aliased to
	 * `jQuery.fn.dataTableExt` through which it may be accessed and
	 * manipulated, or `jQuery.fn.dataTable.settings`.
	 *  @member
	 *  @type array
	 *  @default []
	 *  @private
	 */
	DataTable.settings = [];

	/**
	 * Object models container, for the various models that DataTables has
	 * available to it. These models define the objects that are used to hold
	 * the active state and configuration of the table.
	 *  @namespace
	 */
	DataTable.models = {};
	
	
	
	/**
	 * Template object for the way in which DataTables holds information about
	 * search information for the global filter and individual column filters.
	 *  @namespace
	 */
	DataTable.models.oSearch = {
		/**
		 * Flag to indicate if the filtering should be case insensitive or not
		 *  @type boolean
		 *  @default true
		 */
		"bCaseInsensitive": true,
	
		/**
		 * Applied search term
		 *  @type string
		 *  @default <i>Empty string</i>
		 */
		"sSearch": "",
	
		/**
		 * Flag to indicate if the search term should be interpreted as a
		 * regular expression (true) or not (false) and therefore and special
		 * regex characters escaped.
		 *  @type boolean
		 *  @default false
		 */
		"bRegex": false,
	
		/**
		 * Flag to indicate if DataTables is to use its smart filtering or not.
		 *  @type boolean
		 *  @default true
		 */
		"bSmart": true
	};
	
	
	
	
	/**
	 * Template object for the way in which DataTables holds information about
	 * each individual row. This is the object format used for the settings
	 * aoData array.
	 *  @namespace
	 */
	DataTable.models.oRow = {
		/**
		 * TR element for the row
		 *  @type node
		 *  @default null
		 */
		"nTr": null,
	
		/**
		 * Array of TD elements for each row. This is null until the row has been
		 * created.
		 *  @type array nodes
		 *  @default []
		 */
		"anCells": null,
	
		/**
		 * Data object from the original data source for the row. This is either
		 * an array if using the traditional form of DataTables, or an object if
		 * using mData options. The exact type will depend on the passed in
		 * data from the data source, or will be an array if using DOM a data
		 * source.
		 *  @type array|object
		 *  @default []
		 */
		"_aData": [],
	
		/**
		 * Sorting data cache - this array is ostensibly the same length as the
		 * number of columns (although each index is generated only as it is
		 * needed), and holds the data that is used for sorting each column in the
		 * row. We do this cache generation at the start of the sort in order that
		 * the formatting of the sort data need be done only once for each cell
		 * per sort. This array should not be read from or written to by anything
		 * other than the master sorting methods.
		 *  @type array
		 *  @default null
		 *  @private
		 */
		"_aSortData": null,
	
		/**
		 * Per cell filtering data cache. As per the sort data cache, used to
		 * increase the performance of the filtering in DataTables
		 *  @type array
		 *  @default null
		 *  @private
		 */
		"_aFilterData": null,
	
		/**
		 * Filtering data cache. This is the same as the cell filtering cache, but
		 * in this case a string rather than an array. This is easily computed with
		 * a join on `_aFilterData`, but is provided as a cache so the join isn't
		 * needed on every search (memory traded for performance)
		 *  @type array
		 *  @default null
		 *  @private
		 */
		"_sFilterRow": null,
	
		/**
		 * Cache of the class name that DataTables has applied to the row, so we
		 * can quickly look at this variable rather than needing to do a DOM check
		 * on className for the nTr property.
		 *  @type string
		 *  @default <i>Empty string</i>
		 *  @private
		 */
		"_sRowStripe": "",
	
		/**
		 * Denote if the original data source was from the DOM, or the data source
		 * object. This is used for invalidating data, so DataTables can
		 * automatically read data from the original source, unless uninstructed
		 * otherwise.
		 *  @type string
		 *  @default null
		 *  @private
		 */
		"src": null,
	
		/**
		 * Index in the aoData array. This saves an indexOf lookup when we have the
		 * object, but want to know the index
		 *  @type integer
		 *  @default -1
		 *  @private
		 */
		"idx": -1
	};
	
	
	/**
	 * Template object for the column information object in DataTables. This object
	 * is held in the settings aoColumns array and contains all the information that
	 * DataTables needs about each individual column.
	 *
	 * Note that this object is related to {@link DataTable.defaults.column}
	 * but this one is the internal data store for DataTables's cache of columns.
	 * It should NOT be manipulated outside of DataTables. Any configuration should
	 * be done through the initialisation options.
	 *  @namespace
	 */
	DataTable.models.oColumn = {
		/**
		 * Column index. This could be worked out on-the-fly with $.inArray, but it
		 * is faster to just hold it as a variable
		 *  @type integer
		 *  @default null
		 */
		"idx": null,
	
		/**
		 * A list of the columns that sorting should occur on when this column
		 * is sorted. That this property is an array allows multi-column sorting
		 * to be defined for a column (for example first name / last name columns
		 * would benefit from this). The values are integers pointing to the
		 * columns to be sorted on (typically it will be a single integer pointing
		 * at itself, but that doesn't need to be the case).
		 *  @type array
		 */
		"aDataSort": null,
	
		/**
		 * Define the sorting directions that are applied to the column, in sequence
		 * as the column is repeatedly sorted upon - i.e. the first value is used
		 * as the sorting direction when the column if first sorted (clicked on).
		 * Sort it again (click again) and it will move on to the next index.
		 * Repeat until loop.
		 *  @type array
		 */
		"asSorting": null,
	
		/**
		 * Flag to indicate if the column is searchable, and thus should be included
		 * in the filtering or not.
		 *  @type boolean
		 */
		"bSearchable": null,
	
		/**
		 * Flag to indicate if the column is sortable or not.
		 *  @type boolean
		 */
		"bSortable": null,
	
		/**
		 * Flag to indicate if the column is currently visible in the table or not
		 *  @type boolean
		 */
		"bVisible": null,
	
		/**
		 * Store for manual type assignment using the `column.type` option. This
		 * is held in store so we can manipulate the column's `sType` property.
		 *  @type string
		 *  @default null
		 *  @private
		 */
		"_sManualType": null,
	
		/**
		 * Flag to indicate if HTML5 data attributes should be used as the data
		 * source for filtering or sorting. True is either are.
		 *  @type boolean
		 *  @default false
		 *  @private
		 */
		"_bAttrSrc": false,
	
		/**
		 * Developer definable function that is called whenever a cell is created (Ajax source,
		 * etc) or processed for input (DOM source). This can be used as a compliment to mRender
		 * allowing you to modify the DOM element (add background colour for example) when the
		 * element is available.
		 *  @type function
		 *  @param {element} nTd The TD node that has been created
		 *  @param {*} sData The Data for the cell
		 *  @param {array|object} oData The data for the whole row
		 *  @param {int} iRow The row index for the aoData data store
		 *  @default null
		 */
		"fnCreatedCell": null,
	
		/**
		 * Function to get data from a cell in a column. You should <b>never</b>
		 * access data directly through _aData internally in DataTables - always use
		 * the method attached to this property. It allows mData to function as
		 * required. This function is automatically assigned by the column
		 * initialisation method
		 *  @type function
		 *  @param {array|object} oData The data array/object for the array
		 *    (i.e. aoData[]._aData)
		 *  @param {string} sSpecific The specific data type you want to get -
		 *    'display', 'type' 'filter' 'sort'
		 *  @returns {*} The data for the cell from the given row's data
		 *  @default null
		 */
		"fnGetData": null,
	
		/**
		 * Function to set data for a cell in the column. You should <b>never</b>
		 * set the data directly to _aData internally in DataTables - always use
		 * this method. It allows mData to function as required. This function
		 * is automatically assigned by the column initialisation method
		 *  @type function
		 *  @param {array|object} oData The data array/object for the array
		 *    (i.e. aoData[]._aData)
		 *  @param {*} sValue Value to set
		 *  @default null
		 */
		"fnSetData": null,
	
		/**
		 * Property to read the value for the cells in the column from the data
		 * source array / object. If null, then the default content is used, if a
		 * function is given then the return from the function is used.
		 *  @type function|int|string|null
		 *  @default null
		 */
		"mData": null,
	
		/**
		 * Partner property to mData which is used (only when defined) to get
		 * the data - i.e. it is basically the same as mData, but without the
		 * 'set' option, and also the data fed to it is the result from mData.
		 * This is the rendering method to match the data method of mData.
		 *  @type function|int|string|null
		 *  @default null
		 */
		"mRender": null,
	
		/**
		 * Unique header TH/TD element for this column - this is what the sorting
		 * listener is attached to (if sorting is enabled.)
		 *  @type node
		 *  @default null
		 */
		"nTh": null,
	
		/**
		 * Unique footer TH/TD element for this column (if there is one). Not used
		 * in DataTables as such, but can be used for plug-ins to reference the
		 * footer for each column.
		 *  @type node
		 *  @default null
		 */
		"nTf": null,
	
		/**
		 * The class to apply to all TD elements in the table's TBODY for the column
		 *  @type string
		 *  @default null
		 */
		"sClass": null,
	
		/**
		 * When DataTables calculates the column widths to assign to each column,
		 * it finds the longest string in each column and then constructs a
		 * temporary table and reads the widths from that. The problem with this
		 * is that "mmm" is much wider then "iiii", but the latter is a longer
		 * string - thus the calculation can go wrong (doing it properly and putting
		 * it into an DOM object and measuring that is horribly(!) slow). Thus as
		 * a "work around" we provide this option. It will append its value to the
		 * text that is found to be the longest string for the column - i.e. padding.
		 *  @type string
		 */
		"sContentPadding": null,
	
		/**
		 * Allows a default value to be given for a column's data, and will be used
		 * whenever a null data source is encountered (this can be because mData
		 * is set to null, or because the data source itself is null).
		 *  @type string
		 *  @default null
		 */
		"sDefaultContent": null,
	
		/**
		 * Name for the column, allowing reference to the column by name as well as
		 * by index (needs a lookup to work by name).
		 *  @type string
		 */
		"sName": null,
	
		/**
		 * Custom sorting data type - defines which of the available plug-ins in
		 * afnSortData the custom sorting will use - if any is defined.
		 *  @type string
		 *  @default std
		 */
		"sSortDataType": 'std',
	
		/**
		 * Class to be applied to the header element when sorting on this column
		 *  @type string
		 *  @default null
		 */
		"sSortingClass": null,
	
		/**
		 * Class to be applied to the header element when sorting on this column -
		 * when jQuery UI theming is used.
		 *  @type string
		 *  @default null
		 */
		"sSortingClassJUI": null,
	
		/**
		 * Title of the column - what is seen in the TH element (nTh).
		 *  @type string
		 */
		"sTitle": null,
	
		/**
		 * Column sorting and filtering type
		 *  @type string
		 *  @default null
		 */
		"sType": null,
	
		/**
		 * Width of the column
		 *  @type string
		 *  @default null
		 */
		"sWidth": null,
	
		/**
		 * Width of the column when it was first "encountered"
		 *  @type string
		 *  @default null
		 */
		"sWidthOrig": null
	};
	
	
	/*
	 * Developer note: The properties of the object below are given in Hungarian
	 * notation, that was used as the interface for DataTables prior to v1.10, however
	 * from v1.10 onwards the primary interface is camel case. In order to avoid
	 * breaking backwards compatibility utterly with this change, the Hungarian
	 * version is still, internally the primary interface, but is is not documented
	 * - hence the @name tags in each doc comment. This allows a Javascript function
	 * to create a map from Hungarian notation to camel case (going the other direction
	 * would require each property to be listed, which would add around 3K to the size
	 * of DataTables, while this method is about a 0.5K hit).
	 *
	 * Ultimately this does pave the way for Hungarian notation to be dropped
	 * completely, but that is a massive amount of work and will break current
	 * installs (therefore is on-hold until v2).
	 */
	
	/**
	 * Initialisation options that can be given to DataTables at initialisation
	 * time.
	 *  @namespace
	 */
	DataTable.defaults = {
		/**
		 * An array of data to use for the table, passed in at initialisation which
		 * will be used in preference to any data which is already in the DOM. This is
		 * particularly useful for constructing tables purely in Javascript, for
		 * example with a custom Ajax call.
		 *  @type array
		 *  @default null
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.data
		 *
		 *  @example
		 *    // Using a 2D array data source
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "data": [
		 *          ['Trident', 'Internet Explorer 4.0', 'Win 95+', 4, 'X'],
		 *          ['Trident', 'Internet Explorer 5.0', 'Win 95+', 5, 'C'],
		 *        ],
		 *        "columns": [
		 *          { "title": "Engine" },
		 *          { "title": "Browser" },
		 *          { "title": "Platform" },
		 *          { "title": "Version" },
		 *          { "title": "Grade" }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using an array of objects as a data source (`data`)
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "data": [
		 *          {
		 *            "engine":   "Trident",
		 *            "browser":  "Internet Explorer 4.0",
		 *            "platform": "Win 95+",
		 *            "version":  4,
		 *            "grade":    "X"
		 *          },
		 *          {
		 *            "engine":   "Trident",
		 *            "browser":  "Internet Explorer 5.0",
		 *            "platform": "Win 95+",
		 *            "version":  5,
		 *            "grade":    "C"
		 *          }
		 *        ],
		 *        "columns": [
		 *          { "title": "Engine",   "data": "engine" },
		 *          { "title": "Browser",  "data": "browser" },
		 *          { "title": "Platform", "data": "platform" },
		 *          { "title": "Version",  "data": "version" },
		 *          { "title": "Grade",    "data": "grade" }
		 *        ]
		 *      } );
		 *    } );
		 */
		"aaData": null,
	
	
		/**
		 * If ordering is enabled, then DataTables will perform a first pass sort on
		 * initialisation. You can define which column(s) the sort is performed
		 * upon, and the sorting direction, with this variable. The `sorting` array
		 * should contain an array for each column to be sorted initially containing
		 * the column's index and a direction string ('asc' or 'desc').
		 *  @type array
		 *  @default [[0,'asc']]
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.order
		 *
		 *  @example
		 *    // Sort by 3rd column first, and then 4th column
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "order": [[2,'asc'], [3,'desc']]
		 *      } );
		 *    } );
		 *
		 *    // No initial sorting
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "order": []
		 *      } );
		 *    } );
		 */
		"aaSorting": [[0,'asc']],
	
	
		/**
		 * This parameter is basically identical to the `sorting` parameter, but
		 * cannot be overridden by user interaction with the table. What this means
		 * is that you could have a column (visible or hidden) which the sorting
		 * will always be forced on first - any sorting after that (from the user)
		 * will then be performed as required. This can be useful for grouping rows
		 * together.
		 *  @type array
		 *  @default null
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.orderFixed
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "orderFixed": [[0,'asc']]
		 *      } );
		 *    } )
		 */
		"aaSortingFixed": [],
	
	
		/**
		 * DataTables can be instructed to load data to display in the table from a
		 * Ajax source. This option defines how that Ajax call is made and where to.
		 *
		 * The `ajax` property has three different modes of operation, depending on
		 * how it is defined. These are:
		 *
		 * * `string` - Set the URL from where the data should be loaded from.
		 * * `object` - Define properties for `jQuery.ajax`.
		 * * `function` - Custom data get function
		 *
		 * `string`
		 * --------
		 *
		 * As a string, the `ajax` property simply defines the URL from which
		 * DataTables will load data.
		 *
		 * `object`
		 * --------
		 *
		 * As an object, the parameters in the object are passed to
		 * [jQuery.ajax](http://api.jquery.com/jQuery.ajax/) allowing fine control
		 * of the Ajax request. DataTables has a number of default parameters which
		 * you can override using this option. Please refer to the jQuery
		 * documentation for a full description of the options available, although
		 * the following parameters provide additional options in DataTables or
		 * require special consideration:
		 *
		 * * `data` - As with jQuery, `data` can be provided as an object, but it
		 *   can also be used as a function to manipulate the data DataTables sends
		 *   to the server. The function takes a single parameter, an object of
		 *   parameters with the values that DataTables has readied for sending. An
		 *   object may be returned which will be merged into the DataTables
		 *   defaults, or you can add the items to the object that was passed in and
		 *   not return anything from the function. This supersedes `fnServerParams`
		 *   from DataTables 1.9-.
		 *
		 * * `dataSrc` - By default DataTables will look for the property `data` (or
		 *   `aaData` for compatibility with DataTables 1.9-) when obtaining data
		 *   from an Ajax source or for server-side processing - this parameter
		 *   allows that property to be changed. You can use Javascript dotted
		 *   object notation to get a data source for multiple levels of nesting, or
		 *   it my be used as a function. As a function it takes a single parameter,
		 *   the JSON returned from the server, which can be manipulated as
		 *   required, with the returned value being that used by DataTables as the
		 *   data source for the table. This supersedes `sAjaxDataProp` from
		 *   DataTables 1.9-.
		 *
		 * * `success` - Should not be overridden it is used internally in
		 *   DataTables. To manipulate / transform the data returned by the server
		 *   use `ajax.dataSrc`, or use `ajax` as a function (see below).
		 *
		 * `function`
		 * ----------
		 *
		 * As a function, making the Ajax call is left up to yourself allowing
		 * complete control of the Ajax request. Indeed, if desired, a method other
		 * than Ajax could be used to obtain the required data, such as Web storage
		 * or an AIR database.
		 *
		 * The function is given four parameters and no return is required. The
		 * parameters are:
		 *
		 * 1. _object_ - Data to send to the server
		 * 2. _function_ - Callback function that must be executed when the required
		 *    data has been obtained. That data should be passed into the callback
		 *    as the only parameter
		 * 3. _object_ - DataTables settings object for the table
		 *
		 * Note that this supersedes `fnServerData` from DataTables 1.9-.
		 *
		 *  @type string|object|function
		 *  @default null
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.ajax
		 *  @since 1.10.0
		 *
		 * @example
		 *   // Get JSON data from a file via Ajax.
		 *   // Note DataTables expects data in the form `{ data: [ ...data... ] }` by default).
		 *   $('#example').dataTable( {
		 *     "ajax": "data.json"
		 *   } );
		 *
		 * @example
		 *   // Get JSON data from a file via Ajax, using `dataSrc` to change
		 *   // `data` to `tableData` (i.e. `{ tableData: [ ...data... ] }`)
		 *   $('#example').dataTable( {
		 *     "ajax": {
		 *       "url": "data.json",
		 *       "dataSrc": "tableData"
		 *     }
		 *   } );
		 *
		 * @example
		 *   // Get JSON data from a file via Ajax, using `dataSrc` to read data
		 *   // from a plain array rather than an array in an object
		 *   $('#example').dataTable( {
		 *     "ajax": {
		 *       "url": "data.json",
		 *       "dataSrc": ""
		 *     }
		 *   } );
		 *
		 * @example
		 *   // Manipulate the data returned from the server - add a link to data
		 *   // (note this can, should, be done using `render` for the column - this
		 *   // is just a simple example of how the data can be manipulated).
		 *   $('#example').dataTable( {
		 *     "ajax": {
		 *       "url": "data.json",
		 *       "dataSrc": function ( json ) {
		 *         for ( var i=0, ien=json.length ; i<ien ; i++ ) {
		 *           json[i][0] = '<a href="/message/'+json[i][0]+'>View message</a>';
		 *         }
		 *         return json;
		 *       }
		 *     }
		 *   } );
		 *
		 * @example
		 *   // Add data to the request
		 *   $('#example').dataTable( {
		 *     "ajax": {
		 *       "url": "data.json",
		 *       "data": function ( d ) {
		 *         return {
		 *           "extra_search": $('#extra').val()
		 *         };
		 *       }
		 *     }
		 *   } );
		 *
		 * @example
		 *   // Send request as POST
		 *   $('#example').dataTable( {
		 *     "ajax": {
		 *       "url": "data.json",
		 *       "type": "POST"
		 *     }
		 *   } );
		 *
		 * @example
		 *   // Get the data from localStorage (could interface with a form for
		 *   // adding, editing and removing rows).
		 *   $('#example').dataTable( {
		 *     "ajax": function (data, callback, settings) {
		 *       callback(
		 *         JSON.parse( localStorage.getItem('dataTablesData') )
		 *       );
		 *     }
		 *   } );
		 */
		"ajax": null,
	
	
		/**
		 * This parameter allows you to readily specify the entries in the length drop
		 * down menu that DataTables shows when pagination is enabled. It can be
		 * either a 1D array of options which will be used for both the displayed
		 * option and the value, or a 2D array which will use the array in the first
		 * position as the value, and the array in the second position as the
		 * displayed options (useful for language strings such as 'All').
		 *
		 * Note that the `pageLength` property will be automatically set to the
		 * first value given in this array, unless `pageLength` is also provided.
		 *  @type array
		 *  @default [ 10, 25, 50, 100 ]
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.lengthMenu
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "lengthMenu": [[10, 25, 50, -1], [10, 25, 50, "All"]]
		 *      } );
		 *    } );
		 */
		"aLengthMenu": [ 10, 25, 50, 100 ],
	
	
		/**
		 * The `columns` option in the initialisation parameter allows you to define
		 * details about the way individual columns behave. For a full list of
		 * column options that can be set, please see
		 * {@link DataTable.defaults.column}. Note that if you use `columns` to
		 * define your columns, you must have an entry in the array for every single
		 * column that you have in your table (these can be null if you don't which
		 * to specify any options).
		 *  @member
		 *
		 *  @name DataTable.defaults.column
		 */
		"aoColumns": null,
	
		/**
		 * Very similar to `columns`, `columnDefs` allows you to target a specific
		 * column, multiple columns, or all columns, using the `targets` property of
		 * each object in the array. This allows great flexibility when creating
		 * tables, as the `columnDefs` arrays can be of any length, targeting the
		 * columns you specifically want. `columnDefs` may use any of the column
		 * options available: {@link DataTable.defaults.column}, but it _must_
		 * have `targets` defined in each object in the array. Values in the `targets`
		 * array may be:
		 *   <ul>
		 *     <li>a string - class name will be matched on the TH for the column</li>
		 *     <li>0 or a positive integer - column index counting from the left</li>
		 *     <li>a negative integer - column index counting from the right</li>
		 *     <li>the string "_all" - all columns (i.e. assign a default)</li>
		 *   </ul>
		 *  @member
		 *
		 *  @name DataTable.defaults.columnDefs
		 */
		"aoColumnDefs": null,
	
	
		/**
		 * Basically the same as `search`, this parameter defines the individual column
		 * filtering state at initialisation time. The array must be of the same size
		 * as the number of columns, and each element be an object with the parameters
		 * `search` and `escapeRegex` (the latter is optional). 'null' is also
		 * accepted and the default will be used.
		 *  @type array
		 *  @default []
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.searchCols
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "searchCols": [
		 *          null,
		 *          { "search": "My filter" },
		 *          null,
		 *          { "search": "^[0-9]", "escapeRegex": false }
		 *        ]
		 *      } );
		 *    } )
		 */
		"aoSearchCols": [],
	
	
		/**
		 * An array of CSS classes that should be applied to displayed rows. This
		 * array may be of any length, and DataTables will apply each class
		 * sequentially, looping when required.
		 *  @type array
		 *  @default null <i>Will take the values determined by the `oClasses.stripe*`
		 *    options</i>
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.stripeClasses
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stripeClasses": [ 'strip1', 'strip2', 'strip3' ]
		 *      } );
		 *    } )
		 */
		"asStripeClasses": null,
	
	
		/**
		 * Enable or disable automatic column width calculation. This can be disabled
		 * as an optimisation (it takes some time to calculate the widths) if the
		 * tables widths are passed in using `columns`.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.autoWidth
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "autoWidth": false
		 *      } );
		 *    } );
		 */
		"bAutoWidth": true,
	
	
		/**
		 * Deferred rendering can provide DataTables with a huge speed boost when you
		 * are using an Ajax or JS data source for the table. This option, when set to
		 * true, will cause DataTables to defer the creation of the table elements for
		 * each row until they are needed for a draw - saving a significant amount of
		 * time.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.deferRender
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "ajax": "sources/arrays.txt",
		 *        "deferRender": true
		 *      } );
		 *    } );
		 */
		"bDeferRender": false,
	
	
		/**
		 * Replace a DataTable which matches the given selector and replace it with
		 * one which has the properties of the new initialisation object passed. If no
		 * table matches the selector, then the new DataTable will be constructed as
		 * per normal.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.destroy
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "srollY": "200px",
		 *        "paginate": false
		 *      } );
		 *
		 *      // Some time later....
		 *      $('#example').dataTable( {
		 *        "filter": false,
		 *        "destroy": true
		 *      } );
		 *    } );
		 */
		"bDestroy": false,
	
	
		/**
		 * Enable or disable filtering of data. Filtering in DataTables is "smart" in
		 * that it allows the end user to input multiple words (space separated) and
		 * will match a row containing those words, even if not in the order that was
		 * specified (this allow matching across multiple columns). Note that if you
		 * wish to use filtering in DataTables this must remain 'true' - to remove the
		 * default filtering input box and retain filtering abilities, please use
		 * {@link DataTable.defaults.dom}.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.searching
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "searching": false
		 *      } );
		 *    } );
		 */
		"bFilter": true,
	
	
		/**
		 * Enable or disable the table information display. This shows information
		 * about the data that is currently visible on the page, including information
		 * about filtered data if that action is being performed.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.info
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "info": false
		 *      } );
		 *    } );
		 */
		"bInfo": true,
	
	
		/**
		 * Allows the end user to select the size of a formatted page from a select
		 * menu (sizes are 10, 25, 50 and 100). Requires pagination (`paginate`).
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.lengthChange
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "lengthChange": false
		 *      } );
		 *    } );
		 */
		"bLengthChange": true,
	
	
		/**
		 * Enable or disable pagination.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.paging
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "paging": false
		 *      } );
		 *    } );
		 */
		"bPaginate": true,
	
	
		/**
		 * Enable or disable the display of a 'processing' indicator when the table is
		 * being processed (e.g. a sort). This is particularly useful for tables with
		 * large amounts of data where it can take a noticeable amount of time to sort
		 * the entries.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.processing
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "processing": true
		 *      } );
		 *    } );
		 */
		"bProcessing": false,
	
	
		/**
		 * Retrieve the DataTables object for the given selector. Note that if the
		 * table has already been initialised, this parameter will cause DataTables
		 * to simply return the object that has already been set up - it will not take
		 * account of any changes you might have made to the initialisation object
		 * passed to DataTables (setting this parameter to true is an acknowledgement
		 * that you understand this). `destroy` can be used to reinitialise a table if
		 * you need.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.retrieve
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      initTable();
		 *      tableActions();
		 *    } );
		 *
		 *    function initTable ()
		 *    {
		 *      return $('#example').dataTable( {
		 *        "scrollY": "200px",
		 *        "paginate": false,
		 *        "retrieve": true
		 *      } );
		 *    }
		 *
		 *    function tableActions ()
		 *    {
		 *      var table = initTable();
		 *      // perform API operations with oTable
		 *    }
		 */
		"bRetrieve": false,
	
	
		/**
		 * When vertical (y) scrolling is enabled, DataTables will force the height of
		 * the table's viewport to the given height at all times (useful for layout).
		 * However, this can look odd when filtering data down to a small data set,
		 * and the footer is left "floating" further down. This parameter (when
		 * enabled) will cause DataTables to collapse the table's viewport down when
		 * the result set will fit within the given Y height.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.scrollCollapse
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "scrollY": "200",
		 *        "scrollCollapse": true
		 *      } );
		 *    } );
		 */
		"bScrollCollapse": false,
	
	
		/**
		 * Configure DataTables to use server-side processing. Note that the
		 * `ajax` parameter must also be given in order to give DataTables a
		 * source to obtain the required data for each draw.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Features
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.serverSide
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "serverSide": true,
		 *        "ajax": "xhr.php"
		 *      } );
		 *    } );
		 */
		"bServerSide": false,
	
	
		/**
		 * Enable or disable sorting of columns. Sorting of individual columns can be
		 * disabled by the `sortable` option for each column.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.ordering
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "ordering": false
		 *      } );
		 *    } );
		 */
		"bSort": true,
	
	
		/**
		 * Enable or display DataTables' ability to sort multiple columns at the
		 * same time (activated by shift-click by the user).
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.orderMulti
		 *
		 *  @example
		 *    // Disable multiple column sorting ability
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "orderMulti": false
		 *      } );
		 *    } );
		 */
		"bSortMulti": true,
	
	
		/**
		 * Allows control over whether DataTables should use the top (true) unique
		 * cell that is found for a single column, or the bottom (false - default).
		 * This is useful when using complex headers.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.orderCellsTop
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "orderCellsTop": true
		 *      } );
		 *    } );
		 */
		"bSortCellsTop": false,
	
	
		/**
		 * Enable or disable the addition of the classes `sorting\_1`, `sorting\_2` and
		 * `sorting\_3` to the columns which are currently being sorted on. This is
		 * presented as a feature switch as it can increase processing time (while
		 * classes are removed and added) so for large data sets you might want to
		 * turn this off.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.orderClasses
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "orderClasses": false
		 *      } );
		 *    } );
		 */
		"bSortClasses": true,
	
	
		/**
		 * Enable or disable state saving. When enabled HTML5 `localStorage` will be
		 * used to save table display information such as pagination information,
		 * display length, filtering and sorting. As such when the end user reloads
		 * the page the display display will match what thy had previously set up.
		 *
		 * Due to the use of `localStorage` the default state saving is not supported
		 * in IE6 or 7. If state saving is required in those browsers, use
		 * `stateSaveCallback` to provide a storage solution such as cookies.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.stateSave
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "stateSave": true
		 *      } );
		 *    } );
		 */
		"bStateSave": false,
	
	
		/**
		 * This function is called when a TR element is created (and all TD child
		 * elements have been inserted), or registered if using a DOM source, allowing
		 * manipulation of the TR element (adding classes etc).
		 *  @type function
		 *  @param {node} row "TR" element for the current row
		 *  @param {array} data Raw data array for this row
		 *  @param {int} dataIndex The index of this row in the internal aoData array
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.createdRow
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "createdRow": function( row, data, dataIndex ) {
		 *          // Bold the grade for all 'A' grade browsers
		 *          if ( data[4] == "A" )
		 *          {
		 *            $('td:eq(4)', row).html( '<b>A</b>' );
		 *          }
		 *        }
		 *      } );
		 *    } );
		 */
		"fnCreatedRow": null,
	
	
		/**
		 * This function is called on every 'draw' event, and allows you to
		 * dynamically modify any aspect you want about the created DOM.
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.drawCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "drawCallback": function( settings ) {
		 *          alert( 'DataTables has redrawn the table' );
		 *        }
		 *      } );
		 *    } );
		 */
		"fnDrawCallback": null,
	
	
		/**
		 * Identical to fnHeaderCallback() but for the table footer this function
		 * allows you to modify the table footer on every 'draw' event.
		 *  @type function
		 *  @param {node} foot "TR" element for the footer
		 *  @param {array} data Full table data (as derived from the original HTML)
		 *  @param {int} start Index for the current display starting point in the
		 *    display array
		 *  @param {int} end Index for the current display ending point in the
		 *    display array
		 *  @param {array int} display Index array to translate the visual position
		 *    to the full data array
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.footerCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "footerCallback": function( tfoot, data, start, end, display ) {
		 *          tfoot.getElementsByTagName('th')[0].innerHTML = "Starting index is "+start;
		 *        }
		 *      } );
		 *    } )
		 */
		"fnFooterCallback": null,
	
	
		/**
		 * When rendering large numbers in the information element for the table
		 * (i.e. "Showing 1 to 10 of 57 entries") DataTables will render large numbers
		 * to have a comma separator for the 'thousands' units (e.g. 1 million is
		 * rendered as "1,000,000") to help readability for the end user. This
		 * function will override the default method DataTables uses.
		 *  @type function
		 *  @member
		 *  @param {int} toFormat number to be formatted
		 *  @returns {string} formatted string for DataTables to show the number
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.formatNumber
		 *
		 *  @example
		 *    // Format a number using a single quote for the separator (note that
		 *    // this can also be done with the language.thousands option)
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "formatNumber": function ( toFormat ) {
		 *          return toFormat.toString().replace(
		 *            /\B(?=(\d{3})+(?!\d))/g, "'"
		 *          );
		 *        };
		 *      } );
		 *    } );
		 */
		"fnFormatNumber": function ( toFormat ) {
			return toFormat.toString().replace(
				/\B(?=(\d{3})+(?!\d))/g,
				this.oLanguage.sThousands
			);
		},
	
	
		/**
		 * This function is called on every 'draw' event, and allows you to
		 * dynamically modify the header row. This can be used to calculate and
		 * display useful information about the table.
		 *  @type function
		 *  @param {node} head "TR" element for the header
		 *  @param {array} data Full table data (as derived from the original HTML)
		 *  @param {int} start Index for the current display starting point in the
		 *    display array
		 *  @param {int} end Index for the current display ending point in the
		 *    display array
		 *  @param {array int} display Index array to translate the visual position
		 *    to the full data array
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.headerCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "fheaderCallback": function( head, data, start, end, display ) {
		 *          head.getElementsByTagName('th')[0].innerHTML = "Displaying "+(end-start)+" records";
		 *        }
		 *      } );
		 *    } )
		 */
		"fnHeaderCallback": null,
	
	
		/**
		 * The information element can be used to convey information about the current
		 * state of the table. Although the internationalisation options presented by
		 * DataTables are quite capable of dealing with most customisations, there may
		 * be times where you wish to customise the string further. This callback
		 * allows you to do exactly that.
		 *  @type function
		 *  @param {object} oSettings DataTables settings object
		 *  @param {int} start Starting position in data for the draw
		 *  @param {int} end End position in data for the draw
		 *  @param {int} max Total number of rows in the table (regardless of
		 *    filtering)
		 *  @param {int} total Total number of rows in the data set, after filtering
		 *  @param {string} pre The string that DataTables has formatted using it's
		 *    own rules
		 *  @returns {string} The string to be displayed in the information element.
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.infoCallback
		 *
		 *  @example
		 *    $('#example').dataTable( {
		 *      "infoCallback": function( settings, start, end, max, total, pre ) {
		 *        return start +" to "+ end;
		 *      }
		 *    } );
		 */
		"fnInfoCallback": null,
	
	
		/**
		 * Called when the table has been initialised. Normally DataTables will
		 * initialise sequentially and there will be no need for this function,
		 * however, this does not hold true when using external language information
		 * since that is obtained using an async XHR call.
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *  @param {object} json The JSON object request from the server - only
		 *    present if client-side Ajax sourced data is used
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.initComplete
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "initComplete": function(settings, json) {
		 *          alert( 'DataTables has finished its initialisation.' );
		 *        }
		 *      } );
		 *    } )
		 */
		"fnInitComplete": null,
	
	
		/**
		 * Called at the very start of each table draw and can be used to cancel the
		 * draw by returning false, any other return (including undefined) results in
		 * the full draw occurring).
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *  @returns {boolean} False will cancel the draw, anything else (including no
		 *    return) will allow it to complete.
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.preDrawCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "preDrawCallback": function( settings ) {
		 *          if ( $('#test').val() == 1 ) {
		 *            return false;
		 *          }
		 *        }
		 *      } );
		 *    } );
		 */
		"fnPreDrawCallback": null,
	
	
		/**
		 * This function allows you to 'post process' each row after it have been
		 * generated for each table draw, but before it is rendered on screen. This
		 * function might be used for setting the row class name etc.
		 *  @type function
		 *  @param {node} row "TR" element for the current row
		 *  @param {array} data Raw data array for this row
		 *  @param {int} displayIndex The display index for the current table draw
		 *  @param {int} displayIndexFull The index of the data in the full list of
		 *    rows (after filtering)
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.rowCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "rowCallback": function( row, data, displayIndex, displayIndexFull ) {
		 *          // Bold the grade for all 'A' grade browsers
		 *          if ( data[4] == "A" ) {
		 *            $('td:eq(4)', row).html( '<b>A</b>' );
		 *          }
		 *        }
		 *      } );
		 *    } );
		 */
		"fnRowCallback": null,
	
	
		/**
		 * __Deprecated__ The functionality provided by this parameter has now been
		 * superseded by that provided through `ajax`, which should be used instead.
		 *
		 * This parameter allows you to override the default function which obtains
		 * the data from the server so something more suitable for your application.
		 * For example you could use POST data, or pull information from a Gears or
		 * AIR database.
		 *  @type function
		 *  @member
		 *  @param {string} source HTTP source to obtain the data from (`ajax`)
		 *  @param {array} data A key/value pair object containing the data to send
		 *    to the server
		 *  @param {function} callback to be called on completion of the data get
		 *    process that will draw the data on the page.
		 *  @param {object} settings DataTables settings object
		 *
		 *  @dtopt Callbacks
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.serverData
		 *
		 *  @deprecated 1.10. Please use `ajax` for this functionality now.
		 */
		"fnServerData": null,
	
	
		/**
		 * __Deprecated__ The functionality provided by this parameter has now been
		 * superseded by that provided through `ajax`, which should be used instead.
		 *
		 *  It is often useful to send extra data to the server when making an Ajax
		 * request - for example custom filtering information, and this callback
		 * function makes it trivial to send extra information to the server. The
		 * passed in parameter is the data set that has been constructed by
		 * DataTables, and you can add to this or modify it as you require.
		 *  @type function
		 *  @param {array} data Data array (array of objects which are name/value
		 *    pairs) that has been constructed by DataTables and will be sent to the
		 *    server. In the case of Ajax sourced data with server-side processing
		 *    this will be an empty array, for server-side processing there will be a
		 *    significant number of parameters!
		 *  @returns {undefined} Ensure that you modify the data array passed in,
		 *    as this is passed by reference.
		 *
		 *  @dtopt Callbacks
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.serverParams
		 *
		 *  @deprecated 1.10. Please use `ajax` for this functionality now.
		 */
		"fnServerParams": null,
	
	
		/**
		 * Load the table state. With this function you can define from where, and how, the
		 * state of a table is loaded. By default DataTables will load from `localStorage`
		 * but you might wish to use a server-side database or cookies.
		 *  @type function
		 *  @member
		 *  @param {object} settings DataTables settings object
		 *  @param {object} callback Callback that can be executed when done. It
		 *    should be passed the loaded state object.
		 *  @return {object} The DataTables state object to be loaded
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.stateLoadCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateLoadCallback": function (settings, callback) {
		 *          $.ajax( {
		 *            "url": "/state_load",
		 *            "dataType": "json",
		 *            "success": function (json) {
		 *              callback( json );
		 *            }
		 *          } );
		 *        }
		 *      } );
		 *    } );
		 */
		"fnStateLoadCallback": function ( settings ) {
			try {
				return JSON.parse(
					(settings.iStateDuration === -1 ? sessionStorage : localStorage).getItem(
						'DataTables_'+settings.sInstance+'_'+location.pathname
					)
				);
			} catch (e) {
				return {};
			}
		},
	
	
		/**
		 * Callback which allows modification of the saved state prior to loading that state.
		 * This callback is called when the table is loading state from the stored data, but
		 * prior to the settings object being modified by the saved state. Note that for
		 * plug-in authors, you should use the `stateLoadParams` event to load parameters for
		 * a plug-in.
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *  @param {object} data The state object that is to be loaded
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.stateLoadParams
		 *
		 *  @example
		 *    // Remove a saved filter, so filtering is never loaded
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateLoadParams": function (settings, data) {
		 *          data.oSearch.sSearch = "";
		 *        }
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Disallow state loading by returning false
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateLoadParams": function (settings, data) {
		 *          return false;
		 *        }
		 *      } );
		 *    } );
		 */
		"fnStateLoadParams": null,
	
	
		/**
		 * Callback that is called when the state has been loaded from the state saving method
		 * and the DataTables settings object has been modified as a result of the loaded state.
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *  @param {object} data The state object that was loaded
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.stateLoaded
		 *
		 *  @example
		 *    // Show an alert with the filtering value that was saved
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateLoaded": function (settings, data) {
		 *          alert( 'Saved filter was: '+data.oSearch.sSearch );
		 *        }
		 *      } );
		 *    } );
		 */
		"fnStateLoaded": null,
	
	
		/**
		 * Save the table state. This function allows you to define where and how the state
		 * information for the table is stored By default DataTables will use `localStorage`
		 * but you might wish to use a server-side database or cookies.
		 *  @type function
		 *  @member
		 *  @param {object} settings DataTables settings object
		 *  @param {object} data The state object to be saved
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.stateSaveCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateSaveCallback": function (settings, data) {
		 *          // Send an Ajax request to the server with the state object
		 *          $.ajax( {
		 *            "url": "/state_save",
		 *            "data": data,
		 *            "dataType": "json",
		 *            "method": "POST"
		 *            "success": function () {}
		 *          } );
		 *        }
		 *      } );
		 *    } );
		 */
		"fnStateSaveCallback": function ( settings, data ) {
			try {
				(settings.iStateDuration === -1 ? sessionStorage : localStorage).setItem(
					'DataTables_'+settings.sInstance+'_'+location.pathname,
					JSON.stringify( data )
				);
			} catch (e) {}
		},
	
	
		/**
		 * Callback which allows modification of the state to be saved. Called when the table
		 * has changed state a new state save is required. This method allows modification of
		 * the state saving object prior to actually doing the save, including addition or
		 * other state properties or modification. Note that for plug-in authors, you should
		 * use the `stateSaveParams` event to save parameters for a plug-in.
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *  @param {object} data The state object to be saved
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.stateSaveParams
		 *
		 *  @example
		 *    // Remove a saved filter, so filtering is never saved
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateSaveParams": function (settings, data) {
		 *          data.oSearch.sSearch = "";
		 *        }
		 *      } );
		 *    } );
		 */
		"fnStateSaveParams": null,
	
	
		/**
		 * Duration for which the saved state information is considered valid. After this period
		 * has elapsed the state will be returned to the default.
		 * Value is given in seconds.
		 *  @type int
		 *  @default 7200 <i>(2 hours)</i>
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.stateDuration
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateDuration": 60*60*24; // 1 day
		 *      } );
		 *    } )
		 */
		"iStateDuration": 7200,
	
	
		/**
		 * When enabled DataTables will not make a request to the server for the first
		 * page draw - rather it will use the data already on the page (no sorting etc
		 * will be applied to it), thus saving on an XHR at load time. `deferLoading`
		 * is used to indicate that deferred loading is required, but it is also used
		 * to tell DataTables how many records there are in the full table (allowing
		 * the information element and pagination to be displayed correctly). In the case
		 * where a filtering is applied to the table on initial load, this can be
		 * indicated by giving the parameter as an array, where the first element is
		 * the number of records available after filtering and the second element is the
		 * number of records without filtering (allowing the table information element
		 * to be shown correctly).
		 *  @type int | array
		 *  @default null
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.deferLoading
		 *
		 *  @example
		 *    // 57 records available in the table, no filtering applied
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "serverSide": true,
		 *        "ajax": "scripts/server_processing.php",
		 *        "deferLoading": 57
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // 57 records after filtering, 100 without filtering (an initial filter applied)
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "serverSide": true,
		 *        "ajax": "scripts/server_processing.php",
		 *        "deferLoading": [ 57, 100 ],
		 *        "search": {
		 *          "search": "my_filter"
		 *        }
		 *      } );
		 *    } );
		 */
		"iDeferLoading": null,
	
	
		/**
		 * Number of rows to display on a single page when using pagination. If
		 * feature enabled (`lengthChange`) then the end user will be able to override
		 * this to a custom setting using a pop-up menu.
		 *  @type int
		 *  @default 10
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.pageLength
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "pageLength": 50
		 *      } );
		 *    } )
		 */
		"iDisplayLength": 10,
	
	
		/**
		 * Define the starting point for data display when using DataTables with
		 * pagination. Note that this parameter is the number of records, rather than
		 * the page number, so if you have 10 records per page and want to start on
		 * the third page, it should be "20".
		 *  @type int
		 *  @default 0
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.displayStart
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "displayStart": 20
		 *      } );
		 *    } )
		 */
		"iDisplayStart": 0,
	
	
		/**
		 * By default DataTables allows keyboard navigation of the table (sorting, paging,
		 * and filtering) by adding a `tabindex` attribute to the required elements. This
		 * allows you to tab through the controls and press the enter key to activate them.
		 * The tabindex is default 0, meaning that the tab follows the flow of the document.
		 * You can overrule this using this parameter if you wish. Use a value of -1 to
		 * disable built-in keyboard navigation.
		 *  @type int
		 *  @default 0
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.tabIndex
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "tabIndex": 1
		 *      } );
		 *    } );
		 */
		"iTabIndex": 0,
	
	
		/**
		 * Classes that DataTables assigns to the various components and features
		 * that it adds to the HTML table. This allows classes to be configured
		 * during initialisation in addition to through the static
		 * {@link DataTable.ext.oStdClasses} object).
		 *  @namespace
		 *  @name DataTable.defaults.classes
		 */
		"oClasses": {},
	
	
		/**
		 * All strings that DataTables uses in the user interface that it creates
		 * are defined in this object, allowing you to modified them individually or
		 * completely replace them all as required.
		 *  @namespace
		 *  @name DataTable.defaults.language
		 */
		"oLanguage": {
			/**
			 * Strings that are used for WAI-ARIA labels and controls only (these are not
			 * actually visible on the page, but will be read by screenreaders, and thus
			 * must be internationalised as well).
			 *  @namespace
			 *  @name DataTable.defaults.language.aria
			 */
			"oAria": {
				/**
				 * ARIA label that is added to the table headers when the column may be
				 * sorted ascending by activing the column (click or return when focused).
				 * Note that the column header is prefixed to this string.
				 *  @type string
				 *  @default : activate to sort column ascending
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.aria.sortAscending
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "aria": {
				 *            "sortAscending": " - click/return to sort ascending"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sSortAscending": ": activate to sort column ascending",
	
				/**
				 * ARIA label that is added to the table headers when the column may be
				 * sorted descending by activing the column (click or return when focused).
				 * Note that the column header is prefixed to this string.
				 *  @type string
				 *  @default : activate to sort column ascending
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.aria.sortDescending
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "aria": {
				 *            "sortDescending": " - click/return to sort descending"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sSortDescending": ": activate to sort column descending"
			},
	
			/**
			 * Pagination string used by DataTables for the built-in pagination
			 * control types.
			 *  @namespace
			 *  @name DataTable.defaults.language.paginate
			 */
			"oPaginate": {
				/**
				 * Text to use when using the 'full_numbers' type of pagination for the
				 * button to take the user to the first page.
				 *  @type string
				 *  @default First
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.paginate.first
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "paginate": {
				 *            "first": "First page"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sFirst": "First",
	
	
				/**
				 * Text to use when using the 'full_numbers' type of pagination for the
				 * button to take the user to the last page.
				 *  @type string
				 *  @default Last
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.paginate.last
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "paginate": {
				 *            "last": "Last page"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sLast": "Last",
	
	
				/**
				 * Text to use for the 'next' pagination button (to take the user to the
				 * next page).
				 *  @type string
				 *  @default Next
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.paginate.next
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "paginate": {
				 *            "next": "Next page"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sNext": "Next",
	
	
				/**
				 * Text to use for the 'previous' pagination button (to take the user to
				 * the previous page).
				 *  @type string
				 *  @default Previous
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.paginate.previous
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "paginate": {
				 *            "previous": "Previous page"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sPrevious": "Previous"
			},
	
			/**
			 * This string is shown in preference to `zeroRecords` when the table is
			 * empty of data (regardless of filtering). Note that this is an optional
			 * parameter - if it is not given, the value of `zeroRecords` will be used
			 * instead (either the default or given value).
			 *  @type string
			 *  @default No data available in table
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.emptyTable
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "emptyTable": "No data available in table"
			 *        }
			 *      } );
			 *    } );
			 */
			"sEmptyTable": "No data available in table",
	
	
			/**
			 * This string gives information to the end user about the information
			 * that is current on display on the page. The following tokens can be
			 * used in the string and will be dynamically replaced as the table
			 * display updates. This tokens can be placed anywhere in the string, or
			 * removed as needed by the language requires:
			 *
			 * * `\_START\_` - Display index of the first record on the current page
			 * * `\_END\_` - Display index of the last record on the current page
			 * * `\_TOTAL\_` - Number of records in the table after filtering
			 * * `\_MAX\_` - Number of records in the table without filtering
			 * * `\_PAGE\_` - Current page number
			 * * `\_PAGES\_` - Total number of pages of data in the table
			 *
			 *  @type string
			 *  @default Showing _START_ to _END_ of _TOTAL_ entries
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.info
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "info": "Showing page _PAGE_ of _PAGES_"
			 *        }
			 *      } );
			 *    } );
			 */
			"sInfo": "Showing _START_ to _END_ of _TOTAL_ entries",
	
	
			/**
			 * Display information string for when the table is empty. Typically the
			 * format of this string should match `info`.
			 *  @type string
			 *  @default Showing 0 to 0 of 0 entries
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.infoEmpty
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "infoEmpty": "No entries to show"
			 *        }
			 *      } );
			 *    } );
			 */
			"sInfoEmpty": "Showing 0 to 0 of 0 entries",
	
	
			/**
			 * When a user filters the information in a table, this string is appended
			 * to the information (`info`) to give an idea of how strong the filtering
			 * is. The variable _MAX_ is dynamically updated.
			 *  @type string
			 *  @default (filtered from _MAX_ total entries)
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.infoFiltered
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "infoFiltered": " - filtering from _MAX_ records"
			 *        }
			 *      } );
			 *    } );
			 */
			"sInfoFiltered": "(filtered from _MAX_ total entries)",
	
	
			/**
			 * If can be useful to append extra information to the info string at times,
			 * and this variable does exactly that. This information will be appended to
			 * the `info` (`infoEmpty` and `infoFiltered` in whatever combination they are
			 * being used) at all times.
			 *  @type string
			 *  @default <i>Empty string</i>
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.infoPostFix
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "infoPostFix": "All records shown are derived from real information."
			 *        }
			 *      } );
			 *    } );
			 */
			"sInfoPostFix": "",
	
	
			/**
			 * This decimal place operator is a little different from the other
			 * language options since DataTables doesn't output floating point
			 * numbers, so it won't ever use this for display of a number. Rather,
			 * what this parameter does is modify the sort methods of the table so
			 * that numbers which are in a format which has a character other than
			 * a period (`.`) as a decimal place will be sorted numerically.
			 *
			 * Note that numbers with different decimal places cannot be shown in
			 * the same table and still be sortable, the table must be consistent.
			 * However, multiple different tables on the page can use different
			 * decimal place characters.
			 *  @type string
			 *  @default 
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.decimal
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "decimal": ","
			 *          "thousands": "."
			 *        }
			 *      } );
			 *    } );
			 */
			"sDecimal": "",
	
	
			/**
			 * DataTables has a build in number formatter (`formatNumber`) which is
			 * used to format large numbers that are used in the table information.
			 * By default a comma is used, but this can be trivially changed to any
			 * character you wish with this parameter.
			 *  @type string
			 *  @default ,
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.thousands
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "thousands": "'"
			 *        }
			 *      } );
			 *    } );
			 */
			"sThousands": ",",
	
	
			/**
			 * Detail the action that will be taken when the drop down menu for the
			 * pagination length option is changed. The '_MENU_' variable is replaced
			 * with a default select list of 10, 25, 50 and 100, and can be replaced
			 * with a custom select box if required.
			 *  @type string
			 *  @default Show _MENU_ entries
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.lengthMenu
			 *
			 *  @example
			 *    // Language change only
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "lengthMenu": "Display _MENU_ records"
			 *        }
			 *      } );
			 *    } );
			 *
			 *  @example
			 *    // Language and options change
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "lengthMenu": 'Display <select>'+
			 *            '<option value="10">10</option>'+
			 *            '<option value="20">20</option>'+
			 *            '<option value="30">30</option>'+
			 *            '<option value="40">40</option>'+
			 *            '<option value="50">50</option>'+
			 *            '<option value="-1">All</option>'+
			 *            '</select> records'
			 *        }
			 *      } );
			 *    } );
			 */
			"sLengthMenu": "Show _MENU_ entries",
	
	
			/**
			 * When using Ajax sourced data and during the first draw when DataTables is
			 * gathering the data, this message is shown in an empty row in the table to
			 * indicate to the end user the the data is being loaded. Note that this
			 * parameter is not used when loading data by server-side processing, just
			 * Ajax sourced data with client-side processing.
			 *  @type string
			 *  @default Loading...
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.loadingRecords
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "loadingRecords": "Please wait - loading..."
			 *        }
			 *      } );
			 *    } );
			 */
			"sLoadingRecords": "Loading...",
	
	
			/**
			 * Text which is displayed when the table is processing a user action
			 * (usually a sort command or similar).
			 *  @type string
			 *  @default Processing...
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.processing
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "processing": "DataTables is currently busy"
			 *        }
			 *      } );
			 *    } );
			 */
			"sProcessing": "Processing...",
	
	
			/**
			 * Details the actions that will be taken when the user types into the
			 * filtering input text box. The variable "_INPUT_", if used in the string,
			 * is replaced with the HTML text box for the filtering input allowing
			 * control over where it appears in the string. If "_INPUT_" is not given
			 * then the input box is appended to the string automatically.
			 *  @type string
			 *  @default Search:
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.search
			 *
			 *  @example
			 *    // Input text box will be appended at the end automatically
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "search": "Filter records:"
			 *        }
			 *      } );
			 *    } );
			 *
			 *  @example
			 *    // Specify where the filter should appear
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "search": "Apply filter _INPUT_ to table"
			 *        }
			 *      } );
			 *    } );
			 */
			"sSearch": "Search:",
	
	
			/**
			 * Assign a `placeholder` attribute to the search `input` element
			 *  @type string
			 *  @default 
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.searchPlaceholder
			 */
			"sSearchPlaceholder": "",
	
	
			/**
			 * All of the language information can be stored in a file on the
			 * server-side, which DataTables will look up if this parameter is passed.
			 * It must store the URL of the language file, which is in a JSON format,
			 * and the object has the same properties as the oLanguage object in the
			 * initialiser object (i.e. the above parameters). Please refer to one of
			 * the example language files to see how this works in action.
			 *  @type string
			 *  @default <i>Empty string - i.e. disabled</i>
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.url
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "url": "http://www.sprymedia.co.uk/dataTables/lang.txt"
			 *        }
			 *      } );
			 *    } );
			 */
			"sUrl": "",
	
	
			/**
			 * Text shown inside the table records when the is no information to be
			 * displayed after filtering. `emptyTable` is shown when there is simply no
			 * information in the table at all (regardless of filtering).
			 *  @type string
			 *  @default No matching records found
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.zeroRecords
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "zeroRecords": "No records to display"
			 *        }
			 *      } );
			 *    } );
			 */
			"sZeroRecords": "No matching records found"
		},
	
	
		/**
		 * This parameter allows you to have define the global filtering state at
		 * initialisation time. As an object the `search` parameter must be
		 * defined, but all other parameters are optional. When `regex` is true,
		 * the search string will be treated as a regular expression, when false
		 * (default) it will be treated as a straight string. When `smart`
		 * DataTables will use it's smart filtering methods (to word match at
		 * any point in the data), when false this will not be done.
		 *  @namespace
		 *  @extends DataTable.models.oSearch
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.search
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "search": {"search": "Initial search"}
		 *      } );
		 *    } )
		 */
		"oSearch": $.extend( {}, DataTable.models.oSearch ),
	
	
		/**
		 * __Deprecated__ The functionality provided by this parameter has now been
		 * superseded by that provided through `ajax`, which should be used instead.
		 *
		 * By default DataTables will look for the property `data` (or `aaData` for
		 * compatibility with DataTables 1.9-) when obtaining data from an Ajax
		 * source or for server-side processing - this parameter allows that
		 * property to be changed. You can use Javascript dotted object notation to
		 * get a data source for multiple levels of nesting.
		 *  @type string
		 *  @default data
		 *
		 *  @dtopt Options
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.ajaxDataProp
		 *
		 *  @deprecated 1.10. Please use `ajax` for this functionality now.
		 */
		"sAjaxDataProp": "data",
	
	
		/**
		 * __Deprecated__ The functionality provided by this parameter has now been
		 * superseded by that provided through `ajax`, which should be used instead.
		 *
		 * You can instruct DataTables to load data from an external
		 * source using this parameter (use aData if you want to pass data in you
		 * already have). Simply provide a url a JSON object can be obtained from.
		 *  @type string
		 *  @default null
		 *
		 *  @dtopt Options
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.ajaxSource
		 *
		 *  @deprecated 1.10. Please use `ajax` for this functionality now.
		 */
		"sAjaxSource": null,
	
	
		/**
		 * This initialisation variable allows you to specify exactly where in the
		 * DOM you want DataTables to inject the various controls it adds to the page
		 * (for example you might want the pagination controls at the top of the
		 * table). DIV elements (with or without a custom class) can also be added to
		 * aid styling. The follow syntax is used:
		 *   <ul>
		 *     <li>The following options are allowed:
		 *       <ul>
		 *         <li>'l' - Length changing</li>
		 *         <li>'f' - Filtering input</li>
		 *         <li>'t' - The table!</li>
		 *         <li>'i' - Information</li>
		 *         <li>'p' - Pagination</li>
		 *         <li>'r' - pRocessing</li>
		 *       </ul>
		 *     </li>
		 *     <li>The following constants are allowed:
		 *       <ul>
		 *         <li>'H' - jQueryUI theme "header" classes ('fg-toolbar ui-widget-header ui-corner-tl ui-corner-tr ui-helper-clearfix')</li>
		 *         <li>'F' - jQueryUI theme "footer" classes ('fg-toolbar ui-widget-header ui-corner-bl ui-corner-br ui-helper-clearfix')</li>
		 *       </ul>
		 *     </li>
		 *     <li>The following syntax is expected:
		 *       <ul>
		 *         <li>'&lt;' and '&gt;' - div elements</li>
		 *         <li>'&lt;"class" and '&gt;' - div with a class</li>
		 *         <li>'&lt;"#id" and '&gt;' - div with an ID</li>
		 *       </ul>
		 *     </li>
		 *     <li>Examples:
		 *       <ul>
		 *         <li>'&lt;"wrapper"flipt&gt;'</li>
		 *         <li>'&lt;lf&lt;t&gt;ip&gt;'</li>
		 *       </ul>
		 *     </li>
		 *   </ul>
		 *  @type string
		 *  @default lfrtip <i>(when `jQueryUI` is false)</i> <b>or</b>
		 *    <"H"lfr>t<"F"ip> <i>(when `jQueryUI` is true)</i>
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.dom
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "dom": '&lt;"top"i&gt;rt&lt;"bottom"flp&gt;&lt;"clear"&gt;'
		 *      } );
		 *    } );
		 */
		"sDom": "lfrtip",
	
	
		/**
		 * Search delay option. This will throttle full table searches that use the
		 * DataTables provided search input element (it does not effect calls to
		 * `dt-api search()`, providing a delay before the search is made.
		 *  @type integer
		 *  @default 0
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.searchDelay
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "searchDelay": 200
		 *      } );
		 *    } )
		 */
		"searchDelay": null,
	
	
		/**
		 * DataTables features six different built-in options for the buttons to
		 * display for pagination control:
		 *
		 * * `numbers` - Page number buttons only
		 * * `simple` - 'Previous' and 'Next' buttons only
		 * * 'simple_numbers` - 'Previous' and 'Next' buttons, plus page numbers
		 * * `full` - 'First', 'Previous', 'Next' and 'Last' buttons
		 * * `full_numbers` - 'First', 'Previous', 'Next' and 'Last' buttons, plus page numbers
		 * * `first_last_numbers` - 'First' and 'Last' buttons, plus page numbers
		 *  
		 * Further methods can be added using {@link DataTable.ext.oPagination}.
		 *  @type string
		 *  @default simple_numbers
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.pagingType
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "pagingType": "full_numbers"
		 *      } );
		 *    } )
		 */
		"sPaginationType": "simple_numbers",
	
	
		/**
		 * Enable horizontal scrolling. When a table is too wide to fit into a
		 * certain layout, or you have a large number of columns in the table, you
		 * can enable x-scrolling to show the table in a viewport, which can be
		 * scrolled. This property can be `true` which will allow the table to
		 * scroll horizontally when needed, or any CSS unit, or a number (in which
		 * case it will be treated as a pixel measurement). Setting as simply `true`
		 * is recommended.
		 *  @type boolean|string
		 *  @default <i>blank string - i.e. disabled</i>
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.scrollX
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "scrollX": true,
		 *        "scrollCollapse": true
		 *      } );
		 *    } );
		 */
		"sScrollX": "",
	
	
		/**
		 * This property can be used to force a DataTable to use more width than it
		 * might otherwise do when x-scrolling is enabled. For example if you have a
		 * table which requires to be well spaced, this parameter is useful for
		 * "over-sizing" the table, and thus forcing scrolling. This property can by
		 * any CSS unit, or a number (in which case it will be treated as a pixel
		 * measurement).
		 *  @type string
		 *  @default <i>blank string - i.e. disabled</i>
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.scrollXInner
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "scrollX": "100%",
		 *        "scrollXInner": "110%"
		 *      } );
		 *    } );
		 */
		"sScrollXInner": "",
	
	
		/**
		 * Enable vertical scrolling. Vertical scrolling will constrain the DataTable
		 * to the given height, and enable scrolling for any data which overflows the
		 * current viewport. This can be used as an alternative to paging to display
		 * a lot of data in a small area (although paging and scrolling can both be
		 * enabled at the same time). This property can be any CSS unit, or a number
		 * (in which case it will be treated as a pixel measurement).
		 *  @type string
		 *  @default <i>blank string - i.e. disabled</i>
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.scrollY
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "scrollY": "200px",
		 *        "paginate": false
		 *      } );
		 *    } );
		 */
		"sScrollY": "",
	
	
		/**
		 * __Deprecated__ The functionality provided by this parameter has now been
		 * superseded by that provided through `ajax`, which should be used instead.
		 *
		 * Set the HTTP method that is used to make the Ajax call for server-side
		 * processing or Ajax sourced data.
		 *  @type string
		 *  @default GET
		 *
		 *  @dtopt Options
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.serverMethod
		 *
		 *  @deprecated 1.10. Please use `ajax` for this functionality now.
		 */
		"sServerMethod": "GET",
	
	
		/**
		 * DataTables makes use of renderers when displaying HTML elements for
		 * a table. These renderers can be added or modified by plug-ins to
		 * generate suitable mark-up for a site. For example the Bootstrap
		 * integration plug-in for DataTables uses a paging button renderer to
		 * display pagination buttons in the mark-up required by Bootstrap.
		 *
		 * For further information about the renderers available see
		 * DataTable.ext.renderer
		 *  @type string|object
		 *  @default null
		 *
		 *  @name DataTable.defaults.renderer
		 *
		 */
		"renderer": null,
	
	
		/**
		 * Set the data property name that DataTables should use to get a row's id
		 * to set as the `id` property in the node.
		 *  @type string
		 *  @default DT_RowId
		 *
		 *  @name DataTable.defaults.rowId
		 */
		"rowId": "DT_RowId"
	};
	
	_fnHungarianMap( DataTable.defaults );
	
	
	
	/*
	 * Developer note - See note in model.defaults.js about the use of Hungarian
	 * notation and camel case.
	 */
	
	/**
	 * Column options that can be given to DataTables at initialisation time.
	 *  @namespace
	 */
	DataTable.defaults.column = {
		/**
		 * Define which column(s) an order will occur on for this column. This
		 * allows a column's ordering to take multiple columns into account when
		 * doing a sort or use the data from a different column. For example first
		 * name / last name columns make sense to do a multi-column sort over the
		 * two columns.
		 *  @type array|int
		 *  @default null <i>Takes the value of the column index automatically</i>
		 *
		 *  @name DataTable.defaults.column.orderData
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "orderData": [ 0, 1 ], "targets": [ 0 ] },
		 *          { "orderData": [ 1, 0 ], "targets": [ 1 ] },
		 *          { "orderData": 2, "targets": [ 2 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "orderData": [ 0, 1 ] },
		 *          { "orderData": [ 1, 0 ] },
		 *          { "orderData": 2 },
		 *          null,
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"aDataSort": null,
		"iDataSort": -1,
	
	
		/**
		 * You can control the default ordering direction, and even alter the
		 * behaviour of the sort handler (i.e. only allow ascending ordering etc)
		 * using this parameter.
		 *  @type array
		 *  @default [ 'asc', 'desc' ]
		 *
		 *  @name DataTable.defaults.column.orderSequence
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "orderSequence": [ "asc" ], "targets": [ 1 ] },
		 *          { "orderSequence": [ "desc", "asc", "asc" ], "targets": [ 2 ] },
		 *          { "orderSequence": [ "desc" ], "targets": [ 3 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          null,
		 *          { "orderSequence": [ "asc" ] },
		 *          { "orderSequence": [ "desc", "asc", "asc" ] },
		 *          { "orderSequence": [ "desc" ] },
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"asSorting": [ 'asc', 'desc' ],
	
	
		/**
		 * Enable or disable filtering on the data in this column.
		 *  @type boolean
		 *  @default true
		 *
		 *  @name DataTable.defaults.column.searchable
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "searchable": false, "targets": [ 0 ] }
		 *        ] } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "searchable": false },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ] } );
		 *    } );
		 */
		"bSearchable": true,
	
	
		/**
		 * Enable or disable ordering on this column.
		 *  @type boolean
		 *  @default true
		 *
		 *  @name DataTable.defaults.column.orderable
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "orderable": false, "targets": [ 0 ] }
		 *        ] } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "orderable": false },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ] } );
		 *    } );
		 */
		"bSortable": true,
	
	
		/**
		 * Enable or disable the display of this column.
		 *  @type boolean
		 *  @default true
		 *
		 *  @name DataTable.defaults.column.visible
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "visible": false, "targets": [ 0 ] }
		 *        ] } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "visible": false },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ] } );
		 *    } );
		 */
		"bVisible": true,
	
	
		/**
		 * Developer definable function that is called whenever a cell is created (Ajax source,
		 * etc) or processed for input (DOM source). This can be used as a compliment to mRender
		 * allowing you to modify the DOM element (add background colour for example) when the
		 * element is available.
		 *  @type function
		 *  @param {element} td The TD node that has been created
		 *  @param {*} cellData The Data for the cell
		 *  @param {array|object} rowData The data for the whole row
		 *  @param {int} row The row index for the aoData data store
		 *  @param {int} col The column index for aoColumns
		 *
		 *  @name DataTable.defaults.column.createdCell
		 *  @dtopt Columns
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [3],
		 *          "createdCell": function (td, cellData, rowData, row, col) {
		 *            if ( cellData == "1.7" ) {
		 *              $(td).css('color', 'blue')
		 *            }
		 *          }
		 *        } ]
		 *      });
		 *    } );
		 */
		"fnCreatedCell": null,
	
	
		/**
		 * This parameter has been replaced by `data` in DataTables to ensure naming
		 * consistency. `dataProp` can still be used, as there is backwards
		 * compatibility in DataTables for this option, but it is strongly
		 * recommended that you use `data` in preference to `dataProp`.
		 *  @name DataTable.defaults.column.dataProp
		 */
	
	
		/**
		 * This property can be used to read data from any data source property,
		 * including deeply nested objects / properties. `data` can be given in a
		 * number of different ways which effect its behaviour:
		 *
		 * * `integer` - treated as an array index for the data source. This is the
		 *   default that DataTables uses (incrementally increased for each column).
		 * * `string` - read an object property from the data source. There are
		 *   three 'special' options that can be used in the string to alter how
		 *   DataTables reads the data from the source object:
		 *    * `.` - Dotted Javascript notation. Just as you use a `.` in
		 *      Javascript to read from nested objects, so to can the options
		 *      specified in `data`. For example: `browser.version` or
		 *      `browser.name`. If your object parameter name contains a period, use
		 *      `\\` to escape it - i.e. `first\\.name`.
		 *    * `[]` - Array notation. DataTables can automatically combine data
		 *      from and array source, joining the data with the characters provided
		 *      between the two brackets. For example: `name[, ]` would provide a
		 *      comma-space separated list from the source array. If no characters
		 *      are provided between the brackets, the original array source is
		 *      returned.
		 *    * `()` - Function notation. Adding `()` to the end of a parameter will
		 *      execute a function of the name given. For example: `browser()` for a
		 *      simple function on the data source, `browser.version()` for a
		 *      function in a nested property or even `browser().version` to get an
		 *      object property if the function called returns an object. Note that
		 *      function notation is recommended for use in `render` rather than
		 *      `data` as it is much simpler to use as a renderer.
		 * * `null` - use the original data source for the row rather than plucking
		 *   data directly from it. This action has effects on two other
		 *   initialisation options:
		 *    * `defaultContent` - When null is given as the `data` option and
		 *      `defaultContent` is specified for the column, the value defined by
		 *      `defaultContent` will be used for the cell.
		 *    * `render` - When null is used for the `data` option and the `render`
		 *      option is specified for the column, the whole data source for the
		 *      row is used for the renderer.
		 * * `function` - the function given will be executed whenever DataTables
		 *   needs to set or get the data for a cell in the column. The function
		 *   takes three parameters:
		 *    * Parameters:
		 *      * `{array|object}` The data source for the row
		 *      * `{string}` The type call data requested - this will be 'set' when
		 *        setting data or 'filter', 'display', 'type', 'sort' or undefined
		 *        when gathering data. Note that when `undefined` is given for the
		 *        type DataTables expects to get the raw data for the object back<
		 *      * `{*}` Data to set when the second parameter is 'set'.
		 *    * Return:
		 *      * The return value from the function is not required when 'set' is
		 *        the type of call, but otherwise the return is what will be used
		 *        for the data requested.
		 *
		 * Note that `data` is a getter and setter option. If you just require
		 * formatting of data for output, you will likely want to use `render` which
		 * is simply a getter and thus simpler to use.
		 *
		 * Note that prior to DataTables 1.9.2 `data` was called `mDataProp`. The
		 * name change reflects the flexibility of this property and is consistent
		 * with the naming of mRender. If 'mDataProp' is given, then it will still
		 * be used by DataTables, as it automatically maps the old name to the new
		 * if required.
		 *
		 *  @type string|int|function|null
		 *  @default null <i>Use automatically calculated column index</i>
		 *
		 *  @name DataTable.defaults.column.data
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Read table data from objects
		 *    // JSON structure for each row:
		 *    //   {
		 *    //      "engine": {value},
		 *    //      "browser": {value},
		 *    //      "platform": {value},
		 *    //      "version": {value},
		 *    //      "grade": {value}
		 *    //   }
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "ajaxSource": "sources/objects.txt",
		 *        "columns": [
		 *          { "data": "engine" },
		 *          { "data": "browser" },
		 *          { "data": "platform" },
		 *          { "data": "version" },
		 *          { "data": "grade" }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Read information from deeply nested objects
		 *    // JSON structure for each row:
		 *    //   {
		 *    //      "engine": {value},
		 *    //      "browser": {value},
		 *    //      "platform": {
		 *    //         "inner": {value}
		 *    //      },
		 *    //      "details": [
		 *    //         {value}, {value}
		 *    //      ]
		 *    //   }
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "ajaxSource": "sources/deep.txt",
		 *        "columns": [
		 *          { "data": "engine" },
		 *          { "data": "browser" },
		 *          { "data": "platform.inner" },
		 *          { "data": "details.0" },
		 *          { "data": "details.1" }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `data` as a function to provide different information for
		 *    // sorting, filtering and display. In this case, currency (price)
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": function ( source, type, val ) {
		 *            if (type === 'set') {
		 *              source.price = val;
		 *              // Store the computed dislay and filter values for efficiency
		 *              source.price_display = val=="" ? "" : "$"+numberFormat(val);
		 *              source.price_filter  = val=="" ? "" : "$"+numberFormat(val)+" "+val;
		 *              return;
		 *            }
		 *            else if (type === 'display') {
		 *              return source.price_display;
		 *            }
		 *            else if (type === 'filter') {
		 *              return source.price_filter;
		 *            }
		 *            // 'sort', 'type' and undefined all just use the integer
		 *            return source.price;
		 *          }
		 *        } ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using default content
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": null,
		 *          "defaultContent": "Click to edit"
		 *        } ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using array notation - outputting a list from an array
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": "name[, ]"
		 *        } ]
		 *      } );
		 *    } );
		 *
		 */
		"mData": null,
	
	
		/**
		 * This property is the rendering partner to `data` and it is suggested that
		 * when you want to manipulate data for display (including filtering,
		 * sorting etc) without altering the underlying data for the table, use this
		 * property. `render` can be considered to be the the read only companion to
		 * `data` which is read / write (then as such more complex). Like `data`
		 * this option can be given in a number of different ways to effect its
		 * behaviour:
		 *
		 * * `integer` - treated as an array index for the data source. This is the
		 *   default that DataTables uses (incrementally increased for each column).
		 * * `string` - read an object property from the data source. There are
		 *   three 'special' options that can be used in the string to alter how
		 *   DataTables reads the data from the source object:
		 *    * `.` - Dotted Javascript notation. Just as you use a `.` in
		 *      Javascript to read from nested objects, so to can the options
		 *      specified in `data`. For example: `browser.version` or
		 *      `browser.name`. If your object parameter name contains a period, use
		 *      `\\` to escape it - i.e. `first\\.name`.
		 *    * `[]` - Array notation. DataTables can automatically combine data
		 *      from and array source, joining the data with the characters provided
		 *      between the two brackets. For example: `name[, ]` would provide a
		 *      comma-space separated list from the source array. If no characters
		 *      are provided between the brackets, the original array source is
		 *      returned.
		 *    * `()` - Function notation. Adding `()` to the end of a parameter will
		 *      execute a function of the name given. For example: `browser()` for a
		 *      simple function on the data source, `browser.version()` for a
		 *      function in a nested property or even `browser().version` to get an
		 *      object property if the function called returns an object.
		 * * `object` - use different data for the different data types requested by
		 *   DataTables ('filter', 'display', 'type' or 'sort'). The property names
		 *   of the object is the data type the property refers to and the value can
		 *   defined using an integer, string or function using the same rules as
		 *   `render` normally does. Note that an `_` option _must_ be specified.
		 *   This is the default value to use if you haven't specified a value for
		 *   the data type requested by DataTables.
		 * * `function` - the function given will be executed whenever DataTables
		 *   needs to set or get the data for a cell in the column. The function
		 *   takes three parameters:
		 *    * Parameters:
		 *      * {array|object} The data source for the row (based on `data`)
		 *      * {string} The type call data requested - this will be 'filter',
		 *        'display', 'type' or 'sort'.
		 *      * {array|object} The full data source for the row (not based on
		 *        `data`)
		 *    * Return:
		 *      * The return value from the function is what will be used for the
		 *        data requested.
		 *
		 *  @type string|int|function|object|null
		 *  @default null Use the data source value.
		 *
		 *  @name DataTable.defaults.column.render
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Create a comma separated list from an array of objects
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "ajaxSource": "sources/deep.txt",
		 *        "columns": [
		 *          { "data": "engine" },
		 *          { "data": "browser" },
		 *          {
		 *            "data": "platform",
		 *            "render": "[, ].name"
		 *          }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Execute a function to obtain data
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": null, // Use the full data source object for the renderer's source
		 *          "render": "browserName()"
		 *        } ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // As an object, extracting different data for the different types
		 *    // This would be used with a data source such as:
		 *    //   { "phone": 5552368, "phone_filter": "5552368 555-2368", "phone_display": "555-2368" }
		 *    // Here the `phone` integer is used for sorting and type detection, while `phone_filter`
		 *    // (which has both forms) is used for filtering for if a user inputs either format, while
		 *    // the formatted phone number is the one that is shown in the table.
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": null, // Use the full data source object for the renderer's source
		 *          "render": {
		 *            "_": "phone",
		 *            "filter": "phone_filter",
		 *            "display": "phone_display"
		 *          }
		 *        } ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Use as a function to create a link from the data source
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": "download_link",
		 *          "render": function ( data, type, full ) {
		 *            return '<a href="'+data+'">Download</a>';
		 *          }
		 *        } ]
		 *      } );
		 *    } );
		 */
		"mRender": null,
	
	
		/**
		 * Change the cell type created for the column - either TD cells or TH cells. This
		 * can be useful as TH cells have semantic meaning in the table body, allowing them
		 * to act as a header for a row (you may wish to add scope='row' to the TH elements).
		 *  @type string
		 *  @default td
		 *
		 *  @name DataTable.defaults.column.cellType
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Make the first column use TH cells
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "cellType": "th"
		 *        } ]
		 *      } );
		 *    } );
		 */
		"sCellType": "td",
	
	
		/**
		 * Class to give to each cell in this column.
		 *  @type string
		 *  @default <i>Empty string</i>
		 *
		 *  @name DataTable.defaults.column.class
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "class": "my_class", "targets": [ 0 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "class": "my_class" },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"sClass": "",
	
		/**
		 * When DataTables calculates the column widths to assign to each column,
		 * it finds the longest string in each column and then constructs a
		 * temporary table and reads the widths from that. The problem with this
		 * is that "mmm" is much wider then "iiii", but the latter is a longer
		 * string - thus the calculation can go wrong (doing it properly and putting
		 * it into an DOM object and measuring that is horribly(!) slow). Thus as
		 * a "work around" we provide this option. It will append its value to the
		 * text that is found to be the longest string for the column - i.e. padding.
		 * Generally you shouldn't need this!
		 *  @type string
		 *  @default <i>Empty string<i>
		 *
		 *  @name DataTable.defaults.column.contentPadding
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          null,
		 *          null,
		 *          null,
		 *          {
		 *            "contentPadding": "mmm"
		 *          }
		 *        ]
		 *      } );
		 *    } );
		 */
		"sContentPadding": "",
	
	
		/**
		 * Allows a default value to be given for a column's data, and will be used
		 * whenever a null data source is encountered (this can be because `data`
		 * is set to null, or because the data source itself is null).
		 *  @type string
		 *  @default null
		 *
		 *  @name DataTable.defaults.column.defaultContent
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          {
		 *            "data": null,
		 *            "defaultContent": "Edit",
		 *            "targets": [ -1 ]
		 *          }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          null,
		 *          null,
		 *          null,
		 *          {
		 *            "data": null,
		 *            "defaultContent": "Edit"
		 *          }
		 *        ]
		 *      } );
		 *    } );
		 */
		"sDefaultContent": null,
	
	
		/**
		 * This parameter is only used in DataTables' server-side processing. It can
		 * be exceptionally useful to know what columns are being displayed on the
		 * client side, and to map these to database fields. When defined, the names
		 * also allow DataTables to reorder information from the server if it comes
		 * back in an unexpected order (i.e. if you switch your columns around on the
		 * client-side, your server-side code does not also need updating).
		 *  @type string
		 *  @default <i>Empty string</i>
		 *
		 *  @name DataTable.defaults.column.name
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "name": "engine", "targets": [ 0 ] },
		 *          { "name": "browser", "targets": [ 1 ] },
		 *          { "name": "platform", "targets": [ 2 ] },
		 *          { "name": "version", "targets": [ 3 ] },
		 *          { "name": "grade", "targets": [ 4 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "name": "engine" },
		 *          { "name": "browser" },
		 *          { "name": "platform" },
		 *          { "name": "version" },
		 *          { "name": "grade" }
		 *        ]
		 *      } );
		 *    } );
		 */
		"sName": "",
	
	
		/**
		 * Defines a data source type for the ordering which can be used to read
		 * real-time information from the table (updating the internally cached
		 * version) prior to ordering. This allows ordering to occur on user
		 * editable elements such as form inputs.
		 *  @type string
		 *  @default std
		 *
		 *  @name DataTable.defaults.column.orderDataType
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "orderDataType": "dom-text", "targets": [ 2, 3 ] },
		 *          { "type": "numeric", "targets": [ 3 ] },
		 *          { "orderDataType": "dom-select", "targets": [ 4 ] },
		 *          { "orderDataType": "dom-checkbox", "targets": [ 5 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          null,
		 *          null,
		 *          { "orderDataType": "dom-text" },
		 *          { "orderDataType": "dom-text", "type": "numeric" },
		 *          { "orderDataType": "dom-select" },
		 *          { "orderDataType": "dom-checkbox" }
		 *        ]
		 *      } );
		 *    } );
		 */
		"sSortDataType": "std",
	
	
		/**
		 * The title of this column.
		 *  @type string
		 *  @default null <i>Derived from the 'TH' value for this column in the
		 *    original HTML table.</i>
		 *
		 *  @name DataTable.defaults.column.title
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "title": "My column title", "targets": [ 0 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "title": "My column title" },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"sTitle": null,
	
	
		/**
		 * The type allows you to specify how the data for this column will be
		 * ordered. Four types (string, numeric, date and html (which will strip
		 * HTML tags before ordering)) are currently available. Note that only date
		 * formats understood by Javascript's Date() object will be accepted as type
		 * date. For example: "Mar 26, 2008 5:03 PM". May take the values: 'string',
		 * 'numeric', 'date' or 'html' (by default). Further types can be adding
		 * through plug-ins.
		 *  @type string
		 *  @default null <i>Auto-detected from raw data</i>
		 *
		 *  @name DataTable.defaults.column.type
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "type": "html", "targets": [ 0 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "type": "html" },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"sType": null,
	
	
		/**
		 * Defining the width of the column, this parameter may take any CSS value
		 * (3em, 20px etc). DataTables applies 'smart' widths to columns which have not
		 * been given a specific width through this interface ensuring that the table
		 * remains readable.
		 *  @type string
		 *  @default null <i>Automatic</i>
		 *
		 *  @name DataTable.defaults.column.width
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "width": "20%", "targets": [ 0 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "width": "20%" },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"sWidth": null
	};
	
	_fnHungarianMap( DataTable.defaults.column );
	
	
	
	/**
	 * DataTables settings object - this holds all the information needed for a
	 * given table, including configuration, data and current application of the
	 * table options. DataTables does not have a single instance for each DataTable
	 * with the settings attached to that instance, but rather instances of the
	 * DataTable "class" are created on-the-fly as needed (typically by a
	 * $().dataTable() call) and the settings object is then applied to that
	 * instance.
	 *
	 * Note that this object is related to {@link DataTable.defaults} but this
	 * one is the internal data store for DataTables's cache of columns. It should
	 * NOT be manipulated outside of DataTables. Any configuration should be done
	 * through the initialisation options.
	 *  @namespace
	 *  @todo Really should attach the settings object to individual instances so we
	 *    don't need to create new instances on each $().dataTable() call (if the
	 *    table already exists). It would also save passing oSettings around and
	 *    into every single function. However, this is a very significant
	 *    architecture change for DataTables and will almost certainly break
	 *    backwards compatibility with older installations. This is something that
	 *    will be done in 2.0.
	 */
	DataTable.models.oSettings = {
		/**
		 * Primary features of DataTables and their enablement state.
		 *  @namespace
		 */
		"oFeatures": {
	
			/**
			 * Flag to say if DataTables should automatically try to calculate the
			 * optimum table and columns widths (true) or not (false).
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bAutoWidth": null,
	
			/**
			 * Delay the creation of TR and TD elements until they are actually
			 * needed by a driven page draw. This can give a significant speed
			 * increase for Ajax source and Javascript source data, but makes no
			 * difference at all fro DOM and server-side processing tables.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bDeferRender": null,
	
			/**
			 * Enable filtering on the table or not. Note that if this is disabled
			 * then there is no filtering at all on the table, including fnFilter.
			 * To just remove the filtering input use sDom and remove the 'f' option.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bFilter": null,
	
			/**
			 * Table information element (the 'Showing x of y records' div) enable
			 * flag.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bInfo": null,
	
			/**
			 * Present a user control allowing the end user to change the page size
			 * when pagination is enabled.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bLengthChange": null,
	
			/**
			 * Pagination enabled or not. Note that if this is disabled then length
			 * changing must also be disabled.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bPaginate": null,
	
			/**
			 * Processing indicator enable flag whenever DataTables is enacting a
			 * user request - typically an Ajax request for server-side processing.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bProcessing": null,
	
			/**
			 * Server-side processing enabled flag - when enabled DataTables will
			 * get all data from the server for every draw - there is no filtering,
			 * sorting or paging done on the client-side.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bServerSide": null,
	
			/**
			 * Sorting enablement flag.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bSort": null,
	
			/**
			 * Multi-column sorting
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bSortMulti": null,
	
			/**
			 * Apply a class to the columns which are being sorted to provide a
			 * visual highlight or not. This can slow things down when enabled since
			 * there is a lot of DOM interaction.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bSortClasses": null,
	
			/**
			 * State saving enablement flag.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bStateSave": null
		},
	
	
		/**
		 * Scrolling settings for a table.
		 *  @namespace
		 */
		"oScroll": {
			/**
			 * When the table is shorter in height than sScrollY, collapse the
			 * table container down to the height of the table (when true).
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bCollapse": null,
	
			/**
			 * Width of the scrollbar for the web-browser's platform. Calculated
			 * during table initialisation.
			 *  @type int
			 *  @default 0
			 */
			"iBarWidth": 0,
	
			/**
			 * Viewport width for horizontal scrolling. Horizontal scrolling is
			 * disabled if an empty string.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type string
			 */
			"sX": null,
	
			/**
			 * Width to expand the table to when using x-scrolling. Typically you
			 * should not need to use this.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type string
			 *  @deprecated
			 */
			"sXInner": null,
	
			/**
			 * Viewport height for vertical scrolling. Vertical scrolling is disabled
			 * if an empty string.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type string
			 */
			"sY": null
		},
	
		/**
		 * Language information for the table.
		 *  @namespace
		 *  @extends DataTable.defaults.oLanguage
		 */
		"oLanguage": {
			/**
			 * Information callback function. See
			 * {@link DataTable.defaults.fnInfoCallback}
			 *  @type function
			 *  @default null
			 */
			"fnInfoCallback": null
		},
	
		/**
		 * Browser support parameters
		 *  @namespace
		 */
		"oBrowser": {
			/**
			 * Indicate if the browser incorrectly calculates width:100% inside a
			 * scrolling element (IE6/7)
			 *  @type boolean
			 *  @default false
			 */
			"bScrollOversize": false,
	
			/**
			 * Determine if the vertical scrollbar is on the right or left of the
			 * scrolling container - needed for rtl language layout, although not
			 * all browsers move the scrollbar (Safari).
			 *  @type boolean
			 *  @default false
			 */
			"bScrollbarLeft": false,
	
			/**
			 * Flag for if `getBoundingClientRect` is fully supported or not
			 *  @type boolean
			 *  @default false
			 */
			"bBounding": false,
	
			/**
			 * Browser scrollbar width
			 *  @type integer
			 *  @default 0
			 */
			"barWidth": 0
		},
	
	
		"ajax": null,
	
	
		/**
		 * Array referencing the nodes which are used for the features. The
		 * parameters of this object match what is allowed by sDom - i.e.
		 *   <ul>
		 *     <li>'l' - Length changing</li>
		 *     <li>'f' - Filtering input</li>
		 *     <li>'t' - The table!</li>
		 *     <li>'i' - Information</li>
		 *     <li>'p' - Pagination</li>
		 *     <li>'r' - pRocessing</li>
		 *   </ul>
		 *  @type array
		 *  @default []
		 */
		"aanFeatures": [],
	
		/**
		 * Store data information - see {@link DataTable.models.oRow} for detailed
		 * information.
		 *  @type array
		 *  @default []
		 */
		"aoData": [],
	
		/**
		 * Array of indexes which are in the current display (after filtering etc)
		 *  @type array
		 *  @default []
		 */
		"aiDisplay": [],
	
		/**
		 * Array of indexes for display - no filtering
		 *  @type array
		 *  @default []
		 */
		"aiDisplayMaster": [],
	
		/**
		 * Map of row ids to data indexes
		 *  @type object
		 *  @default {}
		 */
		"aIds": {},
	
		/**
		 * Store information about each column that is in use
		 *  @type array
		 *  @default []
		 */
		"aoColumns": [],
	
		/**
		 * Store information about the table's header
		 *  @type array
		 *  @default []
		 */
		"aoHeader": [],
	
		/**
		 * Store information about the table's footer
		 *  @type array
		 *  @default []
		 */
		"aoFooter": [],
	
		/**
		 * Store the applied global search information in case we want to force a
		 * research or compare the old search to a new one.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @namespace
		 *  @extends DataTable.models.oSearch
		 */
		"oPreviousSearch": {},
	
		/**
		 * Store the applied search for each column - see
		 * {@link DataTable.models.oSearch} for the format that is used for the
		 * filtering information for each column.
		 *  @type array
		 *  @default []
		 */
		"aoPreSearchCols": [],
	
		/**
		 * Sorting that is applied to the table. Note that the inner arrays are
		 * used in the following manner:
		 * <ul>
		 *   <li>Index 0 - column number</li>
		 *   <li>Index 1 - current sorting direction</li>
		 * </ul>
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type array
		 *  @todo These inner arrays should really be objects
		 */
		"aaSorting": null,
	
		/**
		 * Sorting that is always applied to the table (i.e. prefixed in front of
		 * aaSorting).
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type array
		 *  @default []
		 */
		"aaSortingFixed": [],
	
		/**
		 * Classes to use for the striping of a table.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type array
		 *  @default []
		 */
		"asStripeClasses": null,
	
		/**
		 * If restoring a table - we should restore its striping classes as well
		 *  @type array
		 *  @default []
		 */
		"asDestroyStripes": [],
	
		/**
		 * If restoring a table - we should restore its width
		 *  @type int
		 *  @default 0
		 */
		"sDestroyWidth": 0,
	
		/**
		 * Callback functions array for every time a row is inserted (i.e. on a draw).
		 *  @type array
		 *  @default []
		 */
		"aoRowCallback": [],
	
		/**
		 * Callback functions for the header on each draw.
		 *  @type array
		 *  @default []
		 */
		"aoHeaderCallback": [],
	
		/**
		 * Callback function for the footer on each draw.
		 *  @type array
		 *  @default []
		 */
		"aoFooterCallback": [],
	
		/**
		 * Array of callback functions for draw callback functions
		 *  @type array
		 *  @default []
		 */
		"aoDrawCallback": [],
	
		/**
		 * Array of callback functions for row created function
		 *  @type array
		 *  @default []
		 */
		"aoRowCreatedCallback": [],
	
		/**
		 * Callback functions for just before the table is redrawn. A return of
		 * false will be used to cancel the draw.
		 *  @type array
		 *  @default []
		 */
		"aoPreDrawCallback": [],
	
		/**
		 * Callback functions for when the table has been initialised.
		 *  @type array
		 *  @default []
		 */
		"aoInitComplete": [],
	
	
		/**
		 * Callbacks for modifying the settings to be stored for state saving, prior to
		 * saving state.
		 *  @type array
		 *  @default []
		 */
		"aoStateSaveParams": [],
	
		/**
		 * Callbacks for modifying the settings that have been stored for state saving
		 * prior to using the stored values to restore the state.
		 *  @type array
		 *  @default []
		 */
		"aoStateLoadParams": [],
	
		/**
		 * Callbacks for operating on the settings object once the saved state has been
		 * loaded
		 *  @type array
		 *  @default []
		 */
		"aoStateLoaded": [],
	
		/**
		 * Cache the table ID for quick access
		 *  @type string
		 *  @default <i>Empty string</i>
		 */
		"sTableId": "",
	
		/**
		 * The TABLE node for the main table
		 *  @type node
		 *  @default null
		 */
		"nTable": null,
	
		/**
		 * Permanent ref to the thead element
		 *  @type node
		 *  @default null
		 */
		"nTHead": null,
	
		/**
		 * Permanent ref to the tfoot element - if it exists
		 *  @type node
		 *  @default null
		 */
		"nTFoot": null,
	
		/**
		 * Permanent ref to the tbody element
		 *  @type node
		 *  @default null
		 */
		"nTBody": null,
	
		/**
		 * Cache the wrapper node (contains all DataTables controlled elements)
		 *  @type node
		 *  @default null
		 */
		"nTableWrapper": null,
	
		/**
		 * Indicate if when using server-side processing the loading of data
		 * should be deferred until the second draw.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type boolean
		 *  @default false
		 */
		"bDeferLoading": false,
	
		/**
		 * Indicate if all required information has been read in
		 *  @type boolean
		 *  @default false
		 */
		"bInitialised": false,
	
		/**
		 * Information about open rows. Each object in the array has the parameters
		 * 'nTr' and 'nParent'
		 *  @type array
		 *  @default []
		 */
		"aoOpenRows": [],
	
		/**
		 * Dictate the positioning of DataTables' control elements - see
		 * {@link DataTable.model.oInit.sDom}.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type string
		 *  @default null
		 */
		"sDom": null,
	
		/**
		 * Search delay (in mS)
		 *  @type integer
		 *  @default null
		 */
		"searchDelay": null,
	
		/**
		 * Which type of pagination should be used.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type string
		 *  @default two_button
		 */
		"sPaginationType": "two_button",
	
		/**
		 * The state duration (for `stateSave`) in seconds.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type int
		 *  @default 0
		 */
		"iStateDuration": 0,
	
		/**
		 * Array of callback functions for state saving. Each array element is an
		 * object with the following parameters:
		 *   <ul>
		 *     <li>function:fn - function to call. Takes two parameters, oSettings
		 *       and the JSON string to save that has been thus far created. Returns
		 *       a JSON string to be inserted into a json object
		 *       (i.e. '"param": [ 0, 1, 2]')</li>
		 *     <li>string:sName - name of callback</li>
		 *   </ul>
		 *  @type array
		 *  @default []
		 */
		"aoStateSave": [],
	
		/**
		 * Array of callback functions for state loading. Each array element is an
		 * object with the following parameters:
		 *   <ul>
		 *     <li>function:fn - function to call. Takes two parameters, oSettings
		 *       and the object stored. May return false to cancel state loading</li>
		 *     <li>string:sName - name of callback</li>
		 *   </ul>
		 *  @type array
		 *  @default []
		 */
		"aoStateLoad": [],
	
		/**
		 * State that was saved. Useful for back reference
		 *  @type object
		 *  @default null
		 */
		"oSavedState": null,
	
		/**
		 * State that was loaded. Useful for back reference
		 *  @type object
		 *  @default null
		 */
		"oLoadedState": null,
	
		/**
		 * Source url for AJAX data for the table.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type string
		 *  @default null
		 */
		"sAjaxSource": null,
	
		/**
		 * Property from a given object from which to read the table data from. This
		 * can be an empty string (when not server-side processing), in which case
		 * it is  assumed an an array is given directly.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type string
		 */
		"sAjaxDataProp": null,
	
		/**
		 * Note if draw should be blocked while getting data
		 *  @type boolean
		 *  @default true
		 */
		"bAjaxDataGet": true,
	
		/**
		 * The last jQuery XHR object that was used for server-side data gathering.
		 * This can be used for working with the XHR information in one of the
		 * callbacks
		 *  @type object
		 *  @default null
		 */
		"jqXHR": null,
	
		/**
		 * JSON returned from the server in the last Ajax request
		 *  @type object
		 *  @default undefined
		 */
		"json": undefined,
	
		/**
		 * Data submitted as part of the last Ajax request
		 *  @type object
		 *  @default undefined
		 */
		"oAjaxData": undefined,
	
		/**
		 * Function to get the server-side data.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type function
		 */
		"fnServerData": null,
	
		/**
		 * Functions which are called prior to sending an Ajax request so extra
		 * parameters can easily be sent to the server
		 *  @type array
		 *  @default []
		 */
		"aoServerParams": [],
	
		/**
		 * Send the XHR HTTP method - GET or POST (could be PUT or DELETE if
		 * required).
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type string
		 */
		"sServerMethod": null,
	
		/**
		 * Format numbers for display.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type function
		 */
		"fnFormatNumber": null,
	
		/**
		 * List of options that can be used for the user selectable length menu.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type array
		 *  @default []
		 */
		"aLengthMenu": null,
	
		/**
		 * Counter for the draws that the table does. Also used as a tracker for
		 * server-side processing
		 *  @type int
		 *  @default 0
		 */
		"iDraw": 0,
	
		/**
		 * Indicate if a redraw is being done - useful for Ajax
		 *  @type boolean
		 *  @default false
		 */
		"bDrawing": false,
	
		/**
		 * Draw index (iDraw) of the last error when parsing the returned data
		 *  @type int
		 *  @default -1
		 */
		"iDrawError": -1,
	
		/**
		 * Paging display length
		 *  @type int
		 *  @default 10
		 */
		"_iDisplayLength": 10,
	
		/**
		 * Paging start point - aiDisplay index
		 *  @type int
		 *  @default 0
		 */
		"_iDisplayStart": 0,
	
		/**
		 * Server-side processing - number of records in the result set
		 * (i.e. before filtering), Use fnRecordsTotal rather than
		 * this property to get the value of the number of records, regardless of
		 * the server-side processing setting.
		 *  @type int
		 *  @default 0
		 *  @private
		 */
		"_iRecordsTotal": 0,
	
		/**
		 * Server-side processing - number of records in the current display set
		 * (i.e. after filtering). Use fnRecordsDisplay rather than
		 * this property to get the value of the number of records, regardless of
		 * the server-side processing setting.
		 *  @type boolean
		 *  @default 0
		 *  @private
		 */
		"_iRecordsDisplay": 0,
	
		/**
		 * The classes to use for the table
		 *  @type object
		 *  @default {}
		 */
		"oClasses": {},
	
		/**
		 * Flag attached to the settings object so you can check in the draw
		 * callback if filtering has been done in the draw. Deprecated in favour of
		 * events.
		 *  @type boolean
		 *  @default false
		 *  @deprecated
		 */
		"bFiltered": false,
	
		/**
		 * Flag attached to the settings object so you can check in the draw
		 * callback if sorting has been done in the draw. Deprecated in favour of
		 * events.
		 *  @type boolean
		 *  @default false
		 *  @deprecated
		 */
		"bSorted": false,
	
		/**
		 * Indicate that if multiple rows are in the header and there is more than
		 * one unique cell per column, if the top one (true) or bottom one (false)
		 * should be used for sorting / title by DataTables.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type boolean
		 */
		"bSortCellsTop": null,
	
		/**
		 * Initialisation object that is used for the table
		 *  @type object
		 *  @default null
		 */
		"oInit": null,
	
		/**
		 * Destroy callback functions - for plug-ins to attach themselves to the
		 * destroy so they can clean up markup and events.
		 *  @type array
		 *  @default []
		 */
		"aoDestroyCallback": [],
	
	
		/**
		 * Get the number of records in the current record set, before filtering
		 *  @type function
		 */
		"fnRecordsTotal": function ()
		{
			return _fnDataSource( this ) == 'ssp' ?
				this._iRecordsTotal * 1 :
				this.aiDisplayMaster.length;
		},
	
		/**
		 * Get the number of records in the current record set, after filtering
		 *  @type function
		 */
		"fnRecordsDisplay": function ()
		{
			return _fnDataSource( this ) == 'ssp' ?
				this._iRecordsDisplay * 1 :
				this.aiDisplay.length;
		},
	
		/**
		 * Get the display end point - aiDisplay index
		 *  @type function
		 */
		"fnDisplayEnd": function ()
		{
			var
				len      = this._iDisplayLength,
				start    = this._iDisplayStart,
				calc     = start + len,
				records  = this.aiDisplay.length,
				features = this.oFeatures,
				paginate = features.bPaginate;
	
			if ( features.bServerSide ) {
				return paginate === false || len === -1 ?
					start + records :
					Math.min( start+len, this._iRecordsDisplay );
			}
			else {
				return ! paginate || calc>records || len===-1 ?
					records :
					calc;
			}
		},
	
		/**
		 * The DataTables object for this table
		 *  @type object
		 *  @default null
		 */
		"oInstance": null,
	
		/**
		 * Unique identifier for each instance of the DataTables object. If there
		 * is an ID on the table node, then it takes that value, otherwise an
		 * incrementing internal counter is used.
		 *  @type string
		 *  @default null
		 */
		"sInstance": null,
	
		/**
		 * tabindex attribute value that is added to DataTables control elements, allowing
		 * keyboard navigation of the table and its controls.
		 */
		"iTabIndex": 0,
	
		/**
		 * DIV container for the footer scrolling table if scrolling
		 */
		"nScrollHead": null,
	
		/**
		 * DIV container for the footer scrolling table if scrolling
		 */
		"nScrollFoot": null,
	
		/**
		 * Last applied sort
		 *  @type array
		 *  @default []
		 */
		"aLastSort": [],
	
		/**
		 * Stored plug-in instances
		 *  @type object
		 *  @default {}
		 */
		"oPlugins": {},
	
		/**
		 * Function used to get a row's id from the row's data
		 *  @type function
		 *  @default null
		 */
		"rowIdFn": null,
	
		/**
		 * Data location where to store a row's id
		 *  @type string
		 *  @default null
		 */
		"rowId": null
	};

	/**
	 * Extension object for DataTables that is used to provide all extension
	 * options.
	 *
	 * Note that the `DataTable.ext` object is available through
	 * `jQuery.fn.dataTable.ext` where it may be accessed and manipulated. It is
	 * also aliased to `jQuery.fn.dataTableExt` for historic reasons.
	 *  @namespace
	 *  @extends DataTable.models.ext
	 */
	
	
	/**
	 * DataTables extensions
	 * 
	 * This namespace acts as a collection area for plug-ins that can be used to
	 * extend DataTables capabilities. Indeed many of the build in methods
	 * use this method to provide their own capabilities (sorting methods for
	 * example).
	 *
	 * Note that this namespace is aliased to `jQuery.fn.dataTableExt` for legacy
	 * reasons
	 *
	 *  @namespace
	 */
	DataTable.ext = _ext = {
		/**
		 * Buttons. For use with the Buttons extension for DataTables. This is
		 * defined here so other extensions can define buttons regardless of load
		 * order. It is _not_ used by DataTables core.
		 *
		 *  @type object
		 *  @default {}
		 */
		buttons: {},
	
	
		/**
		 * Element class names
		 *
		 *  @type object
		 *  @default {}
		 */
		classes: {},
	
	
		/**
		 * DataTables build type (expanded by the download builder)
		 *
		 *  @type string
		 */
		builder: "-source-",
	
	
		/**
		 * Error reporting.
		 * 
		 * How should DataTables report an error. Can take the value 'alert',
		 * 'throw', 'none' or a function.
		 *
		 *  @type string|function
		 *  @default alert
		 */
		errMode: "alert",
	
	
		/**
		 * Feature plug-ins.
		 * 
		 * This is an array of objects which describe the feature plug-ins that are
		 * available to DataTables. These feature plug-ins are then available for
		 * use through the `dom` initialisation option.
		 * 
		 * Each feature plug-in is described by an object which must have the
		 * following properties:
		 * 
		 * * `fnInit` - function that is used to initialise the plug-in,
		 * * `cFeature` - a character so the feature can be enabled by the `dom`
		 *   instillation option. This is case sensitive.
		 *
		 * The `fnInit` function has the following input parameters:
		 *
		 * 1. `{object}` DataTables settings object: see
		 *    {@link DataTable.models.oSettings}
		 *
		 * And the following return is expected:
		 * 
		 * * {node|null} The element which contains your feature. Note that the
		 *   return may also be void if your plug-in does not require to inject any
		 *   DOM elements into DataTables control (`dom`) - for example this might
		 *   be useful when developing a plug-in which allows table control via
		 *   keyboard entry
		 *
		 *  @type array
		 *
		 *  @example
		 *    $.fn.dataTable.ext.features.push( {
		 *      "fnInit": function( oSettings ) {
		 *        return new TableTools( { "oDTSettings": oSettings } );
		 *      },
		 *      "cFeature": "T"
		 *    } );
		 */
		feature: [],
	
	
		/**
		 * Row searching.
		 * 
		 * This method of searching is complimentary to the default type based
		 * searching, and a lot more comprehensive as it allows you complete control
		 * over the searching logic. Each element in this array is a function
		 * (parameters described below) that is called for every row in the table,
		 * and your logic decides if it should be included in the searching data set
		 * or not.
		 *
		 * Searching functions have the following input parameters:
		 *
		 * 1. `{object}` DataTables settings object: see
		 *    {@link DataTable.models.oSettings}
		 * 2. `{array|object}` Data for the row to be processed (same as the
		 *    original format that was passed in as the data source, or an array
		 *    from a DOM data source
		 * 3. `{int}` Row index ({@link DataTable.models.oSettings.aoData}), which
		 *    can be useful to retrieve the `TR` element if you need DOM interaction.
		 *
		 * And the following return is expected:
		 *
		 * * {boolean} Include the row in the searched result set (true) or not
		 *   (false)
		 *
		 * Note that as with the main search ability in DataTables, technically this
		 * is "filtering", since it is subtractive. However, for consistency in
		 * naming we call it searching here.
		 *
		 *  @type array
		 *  @default []
		 *
		 *  @example
		 *    // The following example shows custom search being applied to the
		 *    // fourth column (i.e. the data[3] index) based on two input values
		 *    // from the end-user, matching the data in a certain range.
		 *    $.fn.dataTable.ext.search.push(
		 *      function( settings, data, dataIndex ) {
		 *        var min = document.getElementById('min').value * 1;
		 *        var max = document.getElementById('max').value * 1;
		 *        var version = data[3] == "-" ? 0 : data[3]*1;
		 *
		 *        if ( min == "" && max == "" ) {
		 *          return true;
		 *        }
		 *        else if ( min == "" && version < max ) {
		 *          return true;
		 *        }
		 *        else if ( min < version && "" == max ) {
		 *          return true;
		 *        }
		 *        else if ( min < version && version < max ) {
		 *          return true;
		 *        }
		 *        return false;
		 *      }
		 *    );
		 */
		search: [],
	
	
		/**
		 * Selector extensions
		 *
		 * The `selector` option can be used to extend the options available for the
		 * selector modifier options (`selector-modifier` object data type) that
		 * each of the three built in selector types offer (row, column and cell +
		 * their plural counterparts). For example the Select extension uses this
		 * mechanism to provide an option to select only rows, columns and cells
		 * that have been marked as selected by the end user (`{selected: true}`),
		 * which can be used in conjunction with the existing built in selector
		 * options.
		 *
		 * Each property is an array to which functions can be pushed. The functions
		 * take three attributes:
		 *
		 * * Settings object for the host table
		 * * Options object (`selector-modifier` object type)
		 * * Array of selected item indexes
		 *
		 * The return is an array of the resulting item indexes after the custom
		 * selector has been applied.
		 *
		 *  @type object
		 */
		selector: {
			cell: [],
			column: [],
			row: []
		},
	
	
		/**
		 * Internal functions, exposed for used in plug-ins.
		 * 
		 * Please note that you should not need to use the internal methods for
		 * anything other than a plug-in (and even then, try to avoid if possible).
		 * The internal function may change between releases.
		 *
		 *  @type object
		 *  @default {}
		 */
		internal: {},
	
	
		/**
		 * Legacy configuration options. Enable and disable legacy options that
		 * are available in DataTables.
		 *
		 *  @type object
		 */
		legacy: {
			/**
			 * Enable / disable DataTables 1.9 compatible server-side processing
			 * requests
			 *
			 *  @type boolean
			 *  @default null
			 */
			ajax: null
		},
	
	
		/**
		 * Pagination plug-in methods.
		 * 
		 * Each entry in this object is a function and defines which buttons should
		 * be shown by the pagination rendering method that is used for the table:
		 * {@link DataTable.ext.renderer.pageButton}. The renderer addresses how the
		 * buttons are displayed in the document, while the functions here tell it
		 * what buttons to display. This is done by returning an array of button
		 * descriptions (what each button will do).
		 *
		 * Pagination types (the four built in options and any additional plug-in
		 * options defined here) can be used through the `paginationType`
		 * initialisation parameter.
		 *
		 * The functions defined take two parameters:
		 *
		 * 1. `{int} page` The current page index
		 * 2. `{int} pages` The number of pages in the table
		 *
		 * Each function is expected to return an array where each element of the
		 * array can be one of:
		 *
		 * * `first` - Jump to first page when activated
		 * * `last` - Jump to last page when activated
		 * * `previous` - Show previous page when activated
		 * * `next` - Show next page when activated
		 * * `{int}` - Show page of the index given
		 * * `{array}` - A nested array containing the above elements to add a
		 *   containing 'DIV' element (might be useful for styling).
		 *
		 * Note that DataTables v1.9- used this object slightly differently whereby
		 * an object with two functions would be defined for each plug-in. That
		 * ability is still supported by DataTables 1.10+ to provide backwards
		 * compatibility, but this option of use is now decremented and no longer
		 * documented in DataTables 1.10+.
		 *
		 *  @type object
		 *  @default {}
		 *
		 *  @example
		 *    // Show previous, next and current page buttons only
		 *    $.fn.dataTableExt.oPagination.current = function ( page, pages ) {
		 *      return [ 'previous', page, 'next' ];
		 *    };
		 */
		pager: {},
	
	
		renderer: {
			pageButton: {},
			header: {}
		},
	
	
		/**
		 * Ordering plug-ins - custom data source
		 * 
		 * The extension options for ordering of data available here is complimentary
		 * to the default type based ordering that DataTables typically uses. It
		 * allows much greater control over the the data that is being used to
		 * order a column, but is necessarily therefore more complex.
		 * 
		 * This type of ordering is useful if you want to do ordering based on data
		 * live from the DOM (for example the contents of an 'input' element) rather
		 * than just the static string that DataTables knows of.
		 * 
		 * The way these plug-ins work is that you create an array of the values you
		 * wish to be ordering for the column in question and then return that
		 * array. The data in the array much be in the index order of the rows in
		 * the table (not the currently ordering order!). Which order data gathering
		 * function is run here depends on the `dt-init columns.orderDataType`
		 * parameter that is used for the column (if any).
		 *
		 * The functions defined take two parameters:
		 *
		 * 1. `{object}` DataTables settings object: see
		 *    {@link DataTable.models.oSettings}
		 * 2. `{int}` Target column index
		 *
		 * Each function is expected to return an array:
		 *
		 * * `{array}` Data for the column to be ordering upon
		 *
		 *  @type array
		 *
		 *  @example
		 *    // Ordering using `input` node values
		 *    $.fn.dataTable.ext.order['dom-text'] = function  ( settings, col )
		 *    {
		 *      return this.api().column( col, {order:'index'} ).nodes().map( function ( td, i ) {
		 *        return $('input', td).val();
		 *      } );
		 *    }
		 */
		order: {},
	
	
		/**
		 * Type based plug-ins.
		 *
		 * Each column in DataTables has a type assigned to it, either by automatic
		 * detection or by direct assignment using the `type` option for the column.
		 * The type of a column will effect how it is ordering and search (plug-ins
		 * can also make use of the column type if required).
		 *
		 * @namespace
		 */
		type: {
			/**
			 * Type detection functions.
			 *
			 * The functions defined in this object are used to automatically detect
			 * a column's type, making initialisation of DataTables super easy, even
			 * when complex data is in the table.
			 *
			 * The functions defined take two parameters:
			 *
		     *  1. `{*}` Data from the column cell to be analysed
		     *  2. `{settings}` DataTables settings object. This can be used to
		     *     perform context specific type detection - for example detection
		     *     based on language settings such as using a comma for a decimal
		     *     place. Generally speaking the options from the settings will not
		     *     be required
			 *
			 * Each function is expected to return:
			 *
			 * * `{string|null}` Data type detected, or null if unknown (and thus
			 *   pass it on to the other type detection functions.
			 *
			 *  @type array
			 *
			 *  @example
			 *    // Currency type detection plug-in:
			 *    $.fn.dataTable.ext.type.detect.push(
			 *      function ( data, settings ) {
			 *        // Check the numeric part
			 *        if ( ! data.substring(1).match(/[0-9]/) ) {
			 *          return null;
			 *        }
			 *
			 *        // Check prefixed by currency
			 *        if ( data.charAt(0) == '$' || data.charAt(0) == '&pound;' ) {
			 *          return 'currency';
			 *        }
			 *        return null;
			 *      }
			 *    );
			 */
			detect: [],
	
	
			/**
			 * Type based search formatting.
			 *
			 * The type based searching functions can be used to pre-format the
			 * data to be search on. For example, it can be used to strip HTML
			 * tags or to de-format telephone numbers for numeric only searching.
			 *
			 * Note that is a search is not defined for a column of a given type,
			 * no search formatting will be performed.
			 * 
			 * Pre-processing of searching data plug-ins - When you assign the sType
			 * for a column (or have it automatically detected for you by DataTables
			 * or a type detection plug-in), you will typically be using this for
			 * custom sorting, but it can also be used to provide custom searching
			 * by allowing you to pre-processing the data and returning the data in
			 * the format that should be searched upon. This is done by adding
			 * functions this object with a parameter name which matches the sType
			 * for that target column. This is the corollary of <i>afnSortData</i>
			 * for searching data.
			 *
			 * The functions defined take a single parameter:
			 *
		     *  1. `{*}` Data from the column cell to be prepared for searching
			 *
			 * Each function is expected to return:
			 *
			 * * `{string|null}` Formatted string that will be used for the searching.
			 *
			 *  @type object
			 *  @default {}
			 *
			 *  @example
			 *    $.fn.dataTable.ext.type.search['title-numeric'] = function ( d ) {
			 *      return d.replace(/\n/g," ").replace( /<.*?>/g, "" );
			 *    }
			 */
			search: {},
	
	
			/**
			 * Type based ordering.
			 *
			 * The column type tells DataTables what ordering to apply to the table
			 * when a column is sorted upon. The order for each type that is defined,
			 * is defined by the functions available in this object.
			 *
			 * Each ordering option can be described by three properties added to
			 * this object:
			 *
			 * * `{type}-pre` - Pre-formatting function
			 * * `{type}-asc` - Ascending order function
			 * * `{type}-desc` - Descending order function
			 *
			 * All three can be used together, only `{type}-pre` or only
			 * `{type}-asc` and `{type}-desc` together. It is generally recommended
			 * that only `{type}-pre` is used, as this provides the optimal
			 * implementation in terms of speed, although the others are provided
			 * for compatibility with existing Javascript sort functions.
			 *
			 * `{type}-pre`: Functions defined take a single parameter:
			 *
		     *  1. `{*}` Data from the column cell to be prepared for ordering
			 *
			 * And return:
			 *
			 * * `{*}` Data to be sorted upon
			 *
			 * `{type}-asc` and `{type}-desc`: Functions are typical Javascript sort
			 * functions, taking two parameters:
			 *
		     *  1. `{*}` Data to compare to the second parameter
		     *  2. `{*}` Data to compare to the first parameter
			 *
			 * And returning:
			 *
			 * * `{*}` Ordering match: <0 if first parameter should be sorted lower
			 *   than the second parameter, ===0 if the two parameters are equal and
			 *   >0 if the first parameter should be sorted height than the second
			 *   parameter.
			 * 
			 *  @type object
			 *  @default {}
			 *
			 *  @example
			 *    // Numeric ordering of formatted numbers with a pre-formatter
			 *    $.extend( $.fn.dataTable.ext.type.order, {
			 *      "string-pre": function(x) {
			 *        a = (a === "-" || a === "") ? 0 : a.replace( /[^\d\-\.]/g, "" );
			 *        return parseFloat( a );
			 *      }
			 *    } );
			 *
			 *  @example
			 *    // Case-sensitive string ordering, with no pre-formatting method
			 *    $.extend( $.fn.dataTable.ext.order, {
			 *      "string-case-asc": function(x,y) {
			 *        return ((x < y) ? -1 : ((x > y) ? 1 : 0));
			 *      },
			 *      "string-case-desc": function(x,y) {
			 *        return ((x < y) ? 1 : ((x > y) ? -1 : 0));
			 *      }
			 *    } );
			 */
			order: {}
		},
	
		/**
		 * Unique DataTables instance counter
		 *
		 * @type int
		 * @private
		 */
		_unique: 0,
	
	
		//
		// Depreciated
		// The following properties are retained for backwards compatiblity only.
		// The should not be used in new projects and will be removed in a future
		// version
		//
	
		/**
		 * Version check function.
		 *  @type function
		 *  @depreciated Since 1.10
		 */
		fnVersionCheck: DataTable.fnVersionCheck,
	
	
		/**
		 * Index for what 'this' index API functions should use
		 *  @type int
		 *  @deprecated Since v1.10
		 */
		iApiIndex: 0,
	
	
		/**
		 * jQuery UI class container
		 *  @type object
		 *  @deprecated Since v1.10
		 */
		oJUIClasses: {},
	
	
		/**
		 * Software version
		 *  @type string
		 *  @deprecated Since v1.10
		 */
		sVersion: DataTable.version
	};
	
	
	//
	// Backwards compatibility. Alias to pre 1.10 Hungarian notation counter parts
	//
	$.extend( _ext, {
		afnFiltering: _ext.search,
		aTypes:       _ext.type.detect,
		ofnSearch:    _ext.type.search,
		oSort:        _ext.type.order,
		afnSortData:  _ext.order,
		aoFeatures:   _ext.feature,
		oApi:         _ext.internal,
		oStdClasses:  _ext.classes,
		oPagination:  _ext.pager
	} );
	
	
	$.extend( DataTable.ext.classes, {
		"sTable": "dataTable",
		"sNoFooter": "no-footer",
	
		/* Paging buttons */
		"sPageButton": "paginate_button",
		"sPageButtonActive": "current",
		"sPageButtonDisabled": "disabled",
	
		/* Striping classes */
		"sStripeOdd": "odd",
		"sStripeEven": "even",
	
		/* Empty row */
		"sRowEmpty": "dataTables_empty",
	
		/* Features */
		"sWrapper": "dataTables_wrapper",
		"sFilter": "dataTables_filter",
		"sInfo": "dataTables_info",
		"sPaging": "dataTables_paginate paging_", /* Note that the type is postfixed */
		"sLength": "dataTables_length",
		"sProcessing": "dataTables_processing",
	
		/* Sorting */
		"sSortAsc": "sorting_asc",
		"sSortDesc": "sorting_desc",
		"sSortable": "sorting", /* Sortable in both directions */
		"sSortableAsc": "sorting_asc_disabled",
		"sSortableDesc": "sorting_desc_disabled",
		"sSortableNone": "sorting_disabled",
		"sSortColumn": "sorting_", /* Note that an int is postfixed for the sorting order */
	
		/* Filtering */
		"sFilterInput": "",
	
		/* Page length */
		"sLengthSelect": "",
	
		/* Scrolling */
		"sScrollWrapper": "dataTables_scroll",
		"sScrollHead": "dataTables_scrollHead",
		"sScrollHeadInner": "dataTables_scrollHeadInner",
		"sScrollBody": "dataTables_scrollBody",
		"sScrollFoot": "dataTables_scrollFoot",
		"sScrollFootInner": "dataTables_scrollFootInner",
	
		/* Misc */
		"sHeaderTH": "",
		"sFooterTH": "",
	
		// Deprecated
		"sSortJUIAsc": "",
		"sSortJUIDesc": "",
		"sSortJUI": "",
		"sSortJUIAscAllowed": "",
		"sSortJUIDescAllowed": "",
		"sSortJUIWrapper": "",
		"sSortIcon": "",
		"sJUIHeader": "",
		"sJUIFooter": ""
	} );
	
	
	var extPagination = DataTable.ext.pager;
	
	function _numbers ( page, pages ) {
		var
			numbers = [],
			buttons = extPagination.numbers_length,
			half = Math.floor( buttons / 2 ),
			i = 1;
	
		if ( pages <= buttons ) {
			numbers = _range( 0, pages );
		}
		else if ( page <= half ) {
			numbers = _range( 0, buttons-2 );
			numbers.push( 'ellipsis' );
			numbers.push( pages-1 );
		}
		else if ( page >= pages - 1 - half ) {
			numbers = _range( pages-(buttons-2), pages );
			numbers.splice( 0, 0, 'ellipsis' ); // no unshift in ie6
			numbers.splice( 0, 0, 0 );
		}
		else {
			numbers = _range( page-half+2, page+half-1 );
			numbers.push( 'ellipsis' );
			numbers.push( pages-1 );
			numbers.splice( 0, 0, 'ellipsis' );
			numbers.splice( 0, 0, 0 );
		}
	
		numbers.DT_el = 'span';
		return numbers;
	}
	
	
	$.extend( extPagination, {
		simple: function ( page, pages ) {
			return [ 'previous', 'next' ];
		},
	
		full: function ( page, pages ) {
			return [  'first', 'previous', 'next', 'last' ];
		},
	
		numbers: function ( page, pages ) {
			return [ _numbers(page, pages) ];
		},
	
		simple_numbers: function ( page, pages ) {
			return [ 'previous', _numbers(page, pages), 'next' ];
		},
	
		full_numbers: function ( page, pages ) {
			return [ 'first', 'previous', _numbers(page, pages), 'next', 'last' ];
		},
		
		first_last_numbers: function (page, pages) {
	 		return ['first', _numbers(page, pages), 'last'];
	 	},
	
		// For testing and plug-ins to use
		_numbers: _numbers,
	
		// Number of number buttons (including ellipsis) to show. _Must be odd!_
		numbers_length: 7
	} );
	
	
	$.extend( true, DataTable.ext.renderer, {
		pageButton: {
			_: function ( settings, host, idx, buttons, page, pages ) {
				var classes = settings.oClasses;
				var lang = settings.oLanguage.oPaginate;
				var aria = settings.oLanguage.oAria.paginate || {};
				var btnDisplay, btnClass, counter=0;
	
				var attach = function( container, buttons ) {
					var i, ien, node, button, tabIndex;
					var disabledClass = classes.sPageButtonDisabled;
					var clickHandler = function ( e ) {
						_fnPageChange( settings, e.data.action, true );
					};
	
					for ( i=0, ien=buttons.length ; i<ien ; i++ ) {
						button = buttons[i];
	
						if ( Array.isArray( button ) ) {
							var inner = $( '<'+(button.DT_el || 'div')+'/>' )
								.appendTo( container );
							attach( inner, button );
						}
						else {
							btnDisplay = null;
							btnClass = button;
							tabIndex = settings.iTabIndex;
	
							switch ( button ) {
								case 'ellipsis':
									container.append('<span class="ellipsis">&#x2026;</span>');
									break;
	
								case 'first':
									btnDisplay = lang.sFirst;
	
									if ( page === 0 ) {
										tabIndex = -1;
										btnClass += ' ' + disabledClass;
									}
									break;
	
								case 'previous':
									btnDisplay = lang.sPrevious;
	
									if ( page === 0 ) {
										tabIndex = -1;
										btnClass += ' ' + disabledClass;
									}
									break;
	
								case 'next':
									btnDisplay = lang.sNext;
	
									if ( pages === 0 || page === pages-1 ) {
										tabIndex = -1;
										btnClass += ' ' + disabledClass;
									}
									break;
	
								case 'last':
									btnDisplay = lang.sLast;
	
									if ( pages === 0 || page === pages-1 ) {
										tabIndex = -1;
										btnClass += ' ' + disabledClass;
									}
									break;
	
								default:
									btnDisplay = settings.fnFormatNumber( button + 1 );
									btnClass = page === button ?
										classes.sPageButtonActive : '';
									break;
							}
	
							if ( btnDisplay !== null ) {
								node = $('<a>', {
										'class': classes.sPageButton+' '+btnClass,
										'aria-controls': settings.sTableId,
										'aria-label': aria[ button ],
										'data-dt-idx': counter,
										'tabindex': tabIndex,
										'id': idx === 0 && typeof button === 'string' ?
											settings.sTableId +'_'+ button :
											null
									} )
									.html( btnDisplay )
									.appendTo( container );
	
								_fnBindAction(
									node, {action: button}, clickHandler
								);
	
								counter++;
							}
						}
					}
				};
	
				// IE9 throws an 'unknown error' if document.activeElement is used
				// inside an iframe or frame. Try / catch the error. Not good for
				// accessibility, but neither are frames.
				var activeEl;
	
				try {
					// Because this approach is destroying and recreating the paging
					// elements, focus is lost on the select button which is bad for
					// accessibility. So we want to restore focus once the draw has
					// completed
					activeEl = $(host).find(document.activeElement).data('dt-idx');
				}
				catch (e) {}
	
				attach( $(host).empty(), buttons );
	
				if ( activeEl !== undefined ) {
					$(host).find( '[data-dt-idx='+activeEl+']' ).trigger('focus');
				}
			}
		}
	} );
	
	
	
	// Built in type detection. See model.ext.aTypes for information about
	// what is required from this methods.
	$.extend( DataTable.ext.type.detect, [
		// Plain numbers - first since V8 detects some plain numbers as dates
		// e.g. Date.parse('55') (but not all, e.g. Date.parse('22')...).
		function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _isNumber( d, decimal ) ? 'num'+decimal : null;
		},
	
		// Dates (only those recognised by the browser's Date.parse)
		function ( d, settings )
		{
			// V8 tries _very_ hard to make a string passed into `Date.parse()`
			// valid, so we need to use a regex to restrict date formats. Use a
			// plug-in for anything other than ISO8601 style strings
			if ( d && !(d instanceof Date) && ! _re_date.test(d) ) {
				return null;
			}
			var parsed = Date.parse(d);
			return (parsed !== null && !isNaN(parsed)) || _empty(d) ? 'date' : null;
		},
	
		// Formatted numbers
		function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _isNumber( d, decimal, true ) ? 'num-fmt'+decimal : null;
		},
	
		// HTML numeric
		function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _htmlNumeric( d, decimal ) ? 'html-num'+decimal : null;
		},
	
		// HTML numeric, formatted
		function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _htmlNumeric( d, decimal, true ) ? 'html-num-fmt'+decimal : null;
		},
	
		// HTML (this is strict checking - there must be html)
		function ( d, settings )
		{
			return _empty( d ) || (typeof d === 'string' && d.indexOf('<') !== -1) ?
				'html' : null;
		}
	] );
	
	
	
	// Filter formatting functions. See model.ext.ofnSearch for information about
	// what is required from these methods.
	// 
	// Note that additional search methods are added for the html numbers and
	// html formatted numbers by `_addNumericSort()` when we know what the decimal
	// place is
	
	
	$.extend( DataTable.ext.type.search, {
		html: function ( data ) {
			return _empty(data) ?
				data :
				typeof data === 'string' ?
					data
						.replace( _re_new_lines, " " )
						.replace( _re_html, "" ) :
					'';
		},
	
		string: function ( data ) {
			return _empty(data) ?
				data :
				typeof data === 'string' ?
					data.replace( _re_new_lines, " " ) :
					data;
		}
	} );
	
	
	
	var __numericReplace = function ( d, decimalPlace, re1, re2 ) {
		if ( d !== 0 && (!d || d === '-') ) {
			return -Infinity;
		}
	
		// If a decimal place other than `.` is used, it needs to be given to the
		// function so we can detect it and replace with a `.` which is the only
		// decimal place Javascript recognises - it is not locale aware.
		if ( decimalPlace ) {
			d = _numToDecimal( d, decimalPlace );
		}
	
		if ( d.replace ) {
			if ( re1 ) {
				d = d.replace( re1, '' );
			}
	
			if ( re2 ) {
				d = d.replace( re2, '' );
			}
		}
	
		return d * 1;
	};
	
	
	// Add the numeric 'deformatting' functions for sorting and search. This is done
	// in a function to provide an easy ability for the language options to add
	// additional methods if a non-period decimal place is used.
	function _addNumericSort ( decimalPlace ) {
		$.each(
			{
				// Plain numbers
				"num": function ( d ) {
					return __numericReplace( d, decimalPlace );
				},
	
				// Formatted numbers
				"num-fmt": function ( d ) {
					return __numericReplace( d, decimalPlace, _re_formatted_numeric );
				},
	
				// HTML numeric
				"html-num": function ( d ) {
					return __numericReplace( d, decimalPlace, _re_html );
				},
	
				// HTML numeric, formatted
				"html-num-fmt": function ( d ) {
					return __numericReplace( d, decimalPlace, _re_html, _re_formatted_numeric );
				}
			},
			function ( key, fn ) {
				// Add the ordering method
				_ext.type.order[ key+decimalPlace+'-pre' ] = fn;
	
				// For HTML types add a search formatter that will strip the HTML
				if ( key.match(/^html\-/) ) {
					_ext.type.search[ key+decimalPlace ] = _ext.type.search.html;
				}
			}
		);
	}
	
	
	// Default sort methods
	$.extend( _ext.type.order, {
		// Dates
		"date-pre": function ( d ) {
			var ts = Date.parse( d );
			return isNaN(ts) ? -Infinity : ts;
		},
	
		// html
		"html-pre": function ( a ) {
			return _empty(a) ?
				'' :
				a.replace ?
					a.replace( /<.*?>/g, "" ).toLowerCase() :
					a+'';
		},
	
		// string
		"string-pre": function ( a ) {
			// This is a little complex, but faster than always calling toString,
			// http://jsperf.com/tostring-v-check
			return _empty(a) ?
				'' :
				typeof a === 'string' ?
					a.toLowerCase() :
					! a.toString ?
						'' :
						a.toString();
		},
	
		// string-asc and -desc are retained only for compatibility with the old
		// sort methods
		"string-asc": function ( x, y ) {
			return ((x < y) ? -1 : ((x > y) ? 1 : 0));
		},
	
		"string-desc": function ( x, y ) {
			return ((x < y) ? 1 : ((x > y) ? -1 : 0));
		}
	} );
	
	
	// Numeric sorting types - order doesn't matter here
	_addNumericSort( '' );
	
	
	$.extend( true, DataTable.ext.renderer, {
		header: {
			_: function ( settings, cell, column, classes ) {
				// No additional mark-up required
				// Attach a sort listener to update on sort - note that using the
				// `DT` namespace will allow the event to be removed automatically
				// on destroy, while the `dt` namespaced event is the one we are
				// listening for
				$(settings.nTable).on( 'order.dt.DT', function ( e, ctx, sorting, columns ) {
					if ( settings !== ctx ) { // need to check this this is the host
						return;               // table, not a nested one
					}
	
					var colIdx = column.idx;
	
					cell
						.removeClass(
							column.sSortingClass +' '+
							classes.sSortAsc +' '+
							classes.sSortDesc
						)
						.addClass( columns[ colIdx ] == 'asc' ?
							classes.sSortAsc : columns[ colIdx ] == 'desc' ?
								classes.sSortDesc :
								column.sSortingClass
						);
				} );
			},
	
			jqueryui: function ( settings, cell, column, classes ) {
				$('<div/>')
					.addClass( classes.sSortJUIWrapper )
					.append( cell.contents() )
					.append( $('<span/>')
						.addClass( classes.sSortIcon+' '+column.sSortingClassJUI )
					)
					.appendTo( cell );
	
				// Attach a sort listener to update on sort
				$(settings.nTable).on( 'order.dt.DT', function ( e, ctx, sorting, columns ) {
					if ( settings !== ctx ) {
						return;
					}
	
					var colIdx = column.idx;
	
					cell
						.removeClass( classes.sSortAsc +" "+classes.sSortDesc )
						.addClass( columns[ colIdx ] == 'asc' ?
							classes.sSortAsc : columns[ colIdx ] == 'desc' ?
								classes.sSortDesc :
								column.sSortingClass
						);
	
					cell
						.find( 'span.'+classes.sSortIcon )
						.removeClass(
							classes.sSortJUIAsc +" "+
							classes.sSortJUIDesc +" "+
							classes.sSortJUI +" "+
							classes.sSortJUIAscAllowed +" "+
							classes.sSortJUIDescAllowed
						)
						.addClass( columns[ colIdx ] == 'asc' ?
							classes.sSortJUIAsc : columns[ colIdx ] == 'desc' ?
								classes.sSortJUIDesc :
								column.sSortingClassJUI
						);
				} );
			}
		}
	} );
	
	/*
	 * Public helper functions. These aren't used internally by DataTables, or
	 * called by any of the options passed into DataTables, but they can be used
	 * externally by developers working with DataTables. They are helper functions
	 * to make working with DataTables a little bit easier.
	 */
	
	var __htmlEscapeEntities = function ( d ) {
		return typeof d === 'string' ?
			d
				.replace(/&/g, '&amp;')
				.replace(/</g, '&lt;')
				.replace(/>/g, '&gt;')
				.replace(/"/g, '&quot;') :
			d;
	};
	
	/**
	 * Helpers for `columns.render`.
	 *
	 * The options defined here can be used with the `columns.render` initialisation
	 * option to provide a display renderer. The following functions are defined:
	 *
	 * * `number` - Will format numeric data (defined by `columns.data`) for
	 *   display, retaining the original unformatted data for sorting and filtering.
	 *   It takes 5 parameters:
	 *   * `string` - Thousands grouping separator
	 *   * `string` - Decimal point indicator
	 *   * `integer` - Number of decimal points to show
	 *   * `string` (optional) - Prefix.
	 *   * `string` (optional) - Postfix (/suffix).
	 * * `text` - Escape HTML to help prevent XSS attacks. It has no optional
	 *   parameters.
	 *
	 * @example
	 *   // Column definition using the number renderer
	 *   {
	 *     data: "salary",
	 *     render: $.fn.dataTable.render.number( '\'', '.', 0, '$' )
	 *   }
	 *
	 * @namespace
	 */
	DataTable.render = {
		number: function ( thousands, decimal, precision, prefix, postfix ) {
			return {
				display: function ( d ) {
					if ( typeof d !== 'number' && typeof d !== 'string' ) {
						return d;
					}
	
					var negative = d < 0 ? '-' : '';
					var flo = parseFloat( d );
	
					// If NaN then there isn't much formatting that we can do - just
					// return immediately, escaping any HTML (this was supposed to
					// be a number after all)
					if ( isNaN( flo ) ) {
						return __htmlEscapeEntities( d );
					}
	
					flo = flo.toFixed( precision );
					d = Math.abs( flo );
	
					var intPart = parseInt( d, 10 );
					var floatPart = precision ?
						decimal+(d - intPart).toFixed( precision ).substring( 2 ):
						'';
	
					return negative + (prefix||'') +
						intPart.toString().replace(
							/\B(?=(\d{3})+(?!\d))/g, thousands
						) +
						floatPart +
						(postfix||'');
				}
			};
		},
	
		text: function () {
			return {
				display: __htmlEscapeEntities,
				filter: __htmlEscapeEntities
			};
		}
	};
	
	
	/*
	 * This is really a good bit rubbish this method of exposing the internal methods
	 * publicly... - To be fixed in 2.0 using methods on the prototype
	 */
	
	
	/**
	 * Create a wrapper function for exporting an internal functions to an external API.
	 *  @param {string} fn API function name
	 *  @returns {function} wrapped function
	 *  @memberof DataTable#internal
	 */
	function _fnExternApiFunc (fn)
	{
		return function() {
			var args = [_fnSettingsFromNode( this[DataTable.ext.iApiIndex] )].concat(
				Array.prototype.slice.call(arguments)
			);
			return DataTable.ext.internal[fn].apply( this, args );
		};
	}
	
	
	/**
	 * Reference to internal functions for use by plug-in developers. Note that
	 * these methods are references to internal functions and are considered to be
	 * private. If you use these methods, be aware that they are liable to change
	 * between versions.
	 *  @namespace
	 */
	$.extend( DataTable.ext.internal, {
		_fnExternApiFunc: _fnExternApiFunc,
		_fnBuildAjax: _fnBuildAjax,
		_fnAjaxUpdate: _fnAjaxUpdate,
		_fnAjaxParameters: _fnAjaxParameters,
		_fnAjaxUpdateDraw: _fnAjaxUpdateDraw,
		_fnAjaxDataSrc: _fnAjaxDataSrc,
		_fnAddColumn: _fnAddColumn,
		_fnColumnOptions: _fnColumnOptions,
		_fnAdjustColumnSizing: _fnAdjustColumnSizing,
		_fnVisibleToColumnIndex: _fnVisibleToColumnIndex,
		_fnColumnIndexToVisible: _fnColumnIndexToVisible,
		_fnVisbleColumns: _fnVisbleColumns,
		_fnGetColumns: _fnGetColumns,
		_fnColumnTypes: _fnColumnTypes,
		_fnApplyColumnDefs: _fnApplyColumnDefs,
		_fnHungarianMap: _fnHungarianMap,
		_fnCamelToHungarian: _fnCamelToHungarian,
		_fnLanguageCompat: _fnLanguageCompat,
		_fnBrowserDetect: _fnBrowserDetect,
		_fnAddData: _fnAddData,
		_fnAddTr: _fnAddTr,
		_fnNodeToDataIndex: _fnNodeToDataIndex,
		_fnNodeToColumnIndex: _fnNodeToColumnIndex,
		_fnGetCellData: _fnGetCellData,
		_fnSetCellData: _fnSetCellData,
		_fnSplitObjNotation: _fnSplitObjNotation,
		_fnGetObjectDataFn: _fnGetObjectDataFn,
		_fnSetObjectDataFn: _fnSetObjectDataFn,
		_fnGetDataMaster: _fnGetDataMaster,
		_fnClearTable: _fnClearTable,
		_fnDeleteIndex: _fnDeleteIndex,
		_fnInvalidate: _fnInvalidate,
		_fnGetRowElements: _fnGetRowElements,
		_fnCreateTr: _fnCreateTr,
		_fnBuildHead: _fnBuildHead,
		_fnDrawHead: _fnDrawHead,
		_fnDraw: _fnDraw,
		_fnReDraw: _fnReDraw,
		_fnAddOptionsHtml: _fnAddOptionsHtml,
		_fnDetectHeader: _fnDetectHeader,
		_fnGetUniqueThs: _fnGetUniqueThs,
		_fnFeatureHtmlFilter: _fnFeatureHtmlFilter,
		_fnFilterComplete: _fnFilterComplete,
		_fnFilterCustom: _fnFilterCustom,
		_fnFilterColumn: _fnFilterColumn,
		_fnFilter: _fnFilter,
		_fnFilterCreateSearch: _fnFilterCreateSearch,
		_fnEscapeRegex: _fnEscapeRegex,
		_fnFilterData: _fnFilterData,
		_fnFeatureHtmlInfo: _fnFeatureHtmlInfo,
		_fnUpdateInfo: _fnUpdateInfo,
		_fnInfoMacros: _fnInfoMacros,
		_fnInitialise: _fnInitialise,
		_fnInitComplete: _fnInitComplete,
		_fnLengthChange: _fnLengthChange,
		_fnFeatureHtmlLength: _fnFeatureHtmlLength,
		_fnFeatureHtmlPaginate: _fnFeatureHtmlPaginate,
		_fnPageChange: _fnPageChange,
		_fnFeatureHtmlProcessing: _fnFeatureHtmlProcessing,
		_fnProcessingDisplay: _fnProcessingDisplay,
		_fnFeatureHtmlTable: _fnFeatureHtmlTable,
		_fnScrollDraw: _fnScrollDraw,
		_fnApplyToChildren: _fnApplyToChildren,
		_fnCalculateColumnWidths: _fnCalculateColumnWidths,
		_fnThrottle: _fnThrottle,
		_fnConvertToWidth: _fnConvertToWidth,
		_fnGetWidestNode: _fnGetWidestNode,
		_fnGetMaxLenString: _fnGetMaxLenString,
		_fnStringToCss: _fnStringToCss,
		_fnSortFlatten: _fnSortFlatten,
		_fnSort: _fnSort,
		_fnSortAria: _fnSortAria,
		_fnSortListener: _fnSortListener,
		_fnSortAttachListener: _fnSortAttachListener,
		_fnSortingClasses: _fnSortingClasses,
		_fnSortData: _fnSortData,
		_fnSaveState: _fnSaveState,
		_fnLoadState: _fnLoadState,
		_fnSettingsFromNode: _fnSettingsFromNode,
		_fnLog: _fnLog,
		_fnMap: _fnMap,
		_fnBindAction: _fnBindAction,
		_fnCallbackReg: _fnCallbackReg,
		_fnCallbackFire: _fnCallbackFire,
		_fnLengthOverflow: _fnLengthOverflow,
		_fnRenderer: _fnRenderer,
		_fnDataSource: _fnDataSource,
		_fnRowAttributes: _fnRowAttributes,
		_fnExtend: _fnExtend,
		_fnCalculateEnd: function () {} // Used by a lot of plug-ins, but redundant
		                                // in 1.10, so this dead-end function is
		                                // added to prevent errors
	} );
	

	// jQuery access
	$.fn.dataTable = DataTable;

	// Provide access to the host jQuery object (circular reference)
	DataTable.$ = $;

	// Legacy aliases
	$.fn.dataTableSettings = DataTable.settings;
	$.fn.dataTableExt = DataTable.ext;

	// With a capital `D` we return a DataTables API instance rather than a
	// jQuery object
	$.fn.DataTable = function ( opts ) {
		return $(this).dataTable( opts ).api();
	};

	// All properties that are available to $.fn.dataTable should also be
	// available on $.fn.DataTable
	$.each( DataTable, function ( prop, val ) {
		$.fn.DataTable[ prop ] = val;
	} );


	// Information about events fired by DataTables - for documentation.
	/**
	 * Draw event, fired whenever the table is redrawn on the page, at the same
	 * point as fnDrawCallback. This may be useful for binding events or
	 * performing calculations when the table is altered at all.
	 *  @name DataTable#draw.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 */

	/**
	 * Search event, fired when the searching applied to the table (using the
	 * built-in global search, or column filters) is altered.
	 *  @name DataTable#search.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 */

	/**
	 * Page change event, fired when the paging of the table is altered.
	 *  @name DataTable#page.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 */

	/**
	 * Order event, fired when the ordering applied to the table is altered.
	 *  @name DataTable#order.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 */

	/**
	 * DataTables initialisation complete event, fired when the table is fully
	 * drawn, including Ajax data loaded, if Ajax data is required.
	 *  @name DataTable#init.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} oSettings DataTables settings object
	 *  @param {object} json The JSON object request from the server - only
	 *    present if client-side Ajax sourced data is used</li></ol>
	 */

	/**
	 * State save event, fired when the table has changed state a new state save
	 * is required. This event allows modification of the state saving object
	 * prior to actually doing the save, including addition or other state
	 * properties (for plug-ins) or modification of a DataTables core property.
	 *  @name DataTable#stateSaveParams.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} oSettings DataTables settings object
	 *  @param {object} json The state information to be saved
	 */

	/**
	 * State load event, fired when the table is loading state from the stored
	 * data, but prior to the settings object being modified by the saved state
	 * - allowing modification of the saved state is required or loading of
	 * state for a plug-in.
	 *  @name DataTable#stateLoadParams.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} oSettings DataTables settings object
	 *  @param {object} json The saved state information
	 */

	/**
	 * State loaded event, fired when state has been loaded from stored data and
	 * the settings object has been modified by the loaded data.
	 *  @name DataTable#stateLoaded.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} oSettings DataTables settings object
	 *  @param {object} json The saved state information
	 */

	/**
	 * Processing event, fired when DataTables is doing some kind of processing
	 * (be it, order, search or anything else). It can be used to indicate to
	 * the end user that there is something happening, or that something has
	 * finished.
	 *  @name DataTable#processing.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} oSettings DataTables settings object
	 *  @param {boolean} bShow Flag for if DataTables is doing processing or not
	 */

	/**
	 * Ajax (XHR) event, fired whenever an Ajax request is completed from a
	 * request to made to the server for new data. This event is called before
	 * DataTables processed the returned data, so it can also be used to pre-
	 * process the data returned from the server, if needed.
	 *
	 * Note that this trigger is called in `fnServerData`, if you override
	 * `fnServerData` and which to use this event, you need to trigger it in you
	 * success function.
	 *  @name DataTable#xhr.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 *  @param {object} json JSON returned from the server
	 *
	 *  @example
	 *     // Use a custom property returned from the server in another DOM element
	 *     $('#table').dataTable().on('xhr.dt', function (e, settings, json) {
	 *       $('#status').html( json.status );
	 *     } );
	 *
	 *  @example
	 *     // Pre-process the data returned from the server
	 *     $('#table').dataTable().on('xhr.dt', function (e, settings, json) {
	 *       for ( var i=0, ien=json.aaData.length ; i<ien ; i++ ) {
	 *         json.aaData[i].sum = json.aaData[i].one + json.aaData[i].two;
	 *       }
	 *       // Note no return - manipulate the data directly in the JSON object.
	 *     } );
	 */

	/**
	 * Destroy event, fired when the DataTable is destroyed by calling fnDestroy
	 * or passing the bDestroy:true parameter in the initialisation object. This
	 * can be used to remove bound events, added DOM nodes, etc.
	 *  @name DataTable#destroy.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 */

	/**
	 * Page length change event, fired when number of records to show on each
	 * page (the length) is changed.
	 *  @name DataTable#length.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 *  @param {integer} len New length
	 */

	/**
	 * Column sizing has changed.
	 *  @name DataTable#column-sizing.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 */

	/**
	 * Column visibility has changed.
	 *  @name DataTable#column-visibility.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 *  @param {int} column Column index
	 *  @param {bool} vis `false` if column now hidden, or `true` if visible
	 */

	return $.fn.dataTable;
}));

/*! DataTables Bootstrap 4 integration
 * ©2011-2017 SpryMedia Ltd - datatables.net/license
 */

/**
 * DataTables integration for Bootstrap 4. This requires Bootstrap 4 and
 * DataTables 1.10 or newer.
 *
 * This file sets the defaults and adds options to DataTables to style its
 * controls using Bootstrap. See http://datatables.net/manual/styling/bootstrap
 * for further information.
 */
(function( factory ){
	if ( typeof define === 'function' && define.amd ) {
		// AMD
		define( ['jquery', 'datatables.net'], function ( $ ) {
			return factory( $, window, document );
		} );
	}
	else if ( typeof exports === 'object' ) {
		// CommonJS
		module.exports = function (root, $) {
			if ( ! root ) {
				root = window;
			}

			if ( ! $ || ! $.fn.dataTable ) {
				// Require DataTables, which attaches to jQuery, including
				// jQuery if needed and have a $ property so we can access the
				// jQuery object that is used
				$ = require('datatables.net')(root, $).$;
			}

			return factory( $, root, root.document );
		};
	}
	else {
		// Browser
		factory( jQuery, window, document );
	}
}(function( $, window, document, undefined ) {
'use strict';
var DataTable = $.fn.dataTable;


/* Set the defaults for DataTables initialisation */
$.extend( true, DataTable.defaults, {
	dom:
		"<'row'<'col-sm-12 col-md-6'l><'col-sm-12 col-md-6'f>>" +
		"<'row'<'col-sm-12'tr>>" +
		"<'row'<'col-sm-12 col-md-5'i><'col-sm-12 col-md-7'p>>",
	renderer: 'bootstrap'
} );


/* Default class modification */
$.extend( DataTable.ext.classes, {
	sWrapper:      "dataTables_wrapper dt-bootstrap4",
	sFilterInput:  "form-control form-control-sm",
	sLengthSelect: "custom-select custom-select-sm form-control form-control-sm",
	sProcessing:   "dataTables_processing card",
	sPageButton:   "paginate_button page-item"
} );


/* Bootstrap paging button renderer */
DataTable.ext.renderer.pageButton.bootstrap = function ( settings, host, idx, buttons, page, pages ) {
	var api     = new DataTable.Api( settings );
	var classes = settings.oClasses;
	var lang    = settings.oLanguage.oPaginate;
	var aria = settings.oLanguage.oAria.paginate || {};
	var btnDisplay, btnClass, counter=0;

	var attach = function( container, buttons ) {
		var i, ien, node, button;
		var clickHandler = function ( e ) {
			e.preventDefault();
			if ( !$(e.currentTarget).hasClass('disabled') && api.page() != e.data.action ) {
				api.page( e.data.action ).draw( 'page' );
			}
		};

		for ( i=0, ien=buttons.length ; i<ien ; i++ ) {
			button = buttons[i];

			if ( Array.isArray( button ) ) {
				attach( container, button );
			}
			else {
				btnDisplay = '';
				btnClass = '';

				switch ( button ) {
					case 'ellipsis':
						btnDisplay = '&#x2026;';
						btnClass = 'disabled';
						break;

					case 'first':
						btnDisplay = lang.sFirst;
						btnClass = button + (page > 0 ?
							'' : ' disabled');
						break;

					case 'previous':
						btnDisplay = lang.sPrevious;
						btnClass = button + (page > 0 ?
							'' : ' disabled');
						break;

					case 'next':
						btnDisplay = lang.sNext;
						btnClass = button + (page < pages-1 ?
							'' : ' disabled');
						break;

					case 'last':
						btnDisplay = lang.sLast;
						btnClass = button + (page < pages-1 ?
							'' : ' disabled');
						break;

					default:
						btnDisplay = button + 1;
						btnClass = page === button ?
							'active' : '';
						break;
				}

				if ( btnDisplay ) {
					node = $('<li>', {
							'class': classes.sPageButton+' '+btnClass,
							'id': idx === 0 && typeof button === 'string' ?
								settings.sTableId +'_'+ button :
								null
						} )
						.append( $('<a>', {
								'href': '#',
								'aria-controls': settings.sTableId,
								'aria-label': aria[ button ],
								'data-dt-idx': counter,
								'tabindex': settings.iTabIndex,
								'class': 'page-link'
							} )
							.html( btnDisplay )
						)
						.appendTo( container );

					settings.oApi._fnBindAction(
						node, {action: button}, clickHandler
					);

					counter++;
				}
			}
		}
	};

	// IE9 throws an 'unknown error' if document.activeElement is used
	// inside an iframe or frame. 
	var activeEl;

	try {
		// Because this approach is destroying and recreating the paging
		// elements, focus is lost on the select button which is bad for
		// accessibility. So we want to restore focus once the draw has
		// completed
		activeEl = $(host).find(document.activeElement).data('dt-idx');
	}
	catch (e) {}

	attach(
		$(host).empty().html('<ul class="pagination"/>').children('ul'),
		buttons
	);

	if ( activeEl !== undefined ) {
		$(host).find( '[data-dt-idx='+activeEl+']' ).trigger('focus');
	}
};


return DataTable;
}));

"use strict";
var defaults = {
	"language": {
		"paginate": {
			"first": '<i class="ki ki-double-arrow-back"></i>',
			"last": '<i class="ki ki-double-arrow-next"></i>',
			"next": '<i class="ki ki-arrow-next"></i>',
			"previous": '<i class="ki ki-arrow-back"></i>'
		}
	}
};

if (KTUtil.isRTL()) {
	defaults = {
		"language": {
			"paginate": {
				"first": '<i class="ki ki-double-arrow-next"></i>',
				"last": '<i class="ki ki-double-arrow-back"></i>',
				"next": '<i class="ki ki-arrow-back"></i>',
				"previous": '<i class="ki ki-arrow-next"></i>'
			}
		}
	}
}

$.extend(true, $.fn.dataTable.defaults, defaults);

// fix dropdown overflow inside datatable
KTApp.initAbsoluteDropdown('.dataTables_wrapper');

/*!
 AutoFill 2.3.5
 ©2008-2020 SpryMedia Ltd - datatables.net/license
*/
(function(e){"function"===typeof define&&define.amd?define(["jquery","datatables.net"],function(g){return e(g,window,document)}):"object"===typeof exports?module.exports=function(g,i){g||(g=window);if(!i||!i.fn.dataTable)i=require("datatables.net")(g,i).$;return e(i,g,g.document)}:e(jQuery,window,document)})(function(e,g,i,r){var l=e.fn.dataTable,t=0,k=function(b,c){if(!l.versionCheck||!l.versionCheck("1.10.8"))throw"Warning: AutoFill requires DataTables 1.10.8 or greater";this.c=e.extend(!0,{},l.defaults.autoFill,
k.defaults,c);this.s={dt:new l.Api(b),namespace:".autoFill"+t++,scroll:{},scrollInterval:null,handle:{height:0,width:0},enabled:!1};this.dom={handle:e('<div class="dt-autofill-handle"/>'),select:{top:e('<div class="dt-autofill-select top"/>'),right:e('<div class="dt-autofill-select right"/>'),bottom:e('<div class="dt-autofill-select bottom"/>'),left:e('<div class="dt-autofill-select left"/>')},background:e('<div class="dt-autofill-background"/>'),list:e('<div class="dt-autofill-list">'+this.s.dt.i18n("autoFill.info",
"")+"<ul/></div>"),dtScroll:null,offsetParent:null};this._constructor()};e.extend(k.prototype,{enabled:function(){return this.s.enabled},enable:function(b){var c=this;if(!1===b)return this.disable();this.s.enabled=!0;this._focusListener();this.dom.handle.on("mousedown",function(a){c._mousedown(a);return!1});return this},disable:function(){this.s.enabled=!1;this._focusListenerRemove();return this},_constructor:function(){var b=this,c=this.s.dt,a=e("div.dataTables_scrollBody",this.s.dt.table().container());
c.settings()[0].autoFill=this;a.length&&(this.dom.dtScroll=a,"static"===a.css("position")&&a.css("position","relative"));!1!==this.c.enable&&this.enable();c.on("destroy.autoFill",function(){b._focusListenerRemove()})},_attach:function(b){var c=this.s.dt,a=c.cell(b).index(),d=this.dom.handle,f=this.s.handle;if(!a||-1===c.columns(this.c.columns).indexes().indexOf(a.column))this._detach();else{this.dom.offsetParent||(this.dom.offsetParent=e(c.table().node()).offsetParent());if(!f.height||!f.width)d.appendTo("body"),
f.height=d.outerHeight(),f.width=d.outerWidth();c=this._getPosition(b,this.dom.offsetParent);this.dom.attachedTo=b;d.css({top:c.top+b.offsetHeight-f.height,left:c.left+b.offsetWidth-f.width}).appendTo(this.dom.offsetParent)}},_actionSelector:function(b){var c=this,a=this.s.dt,d=k.actions,f=[];e.each(d,function(c,d){d.available(a,b)&&f.push(c)});if(1===f.length&&!1===this.c.alwaysAsk){var j=d[f[0]].execute(a,b);this._update(j,b)}else if(1<f.length){var h=this.dom.list.children("ul").empty();f.push("cancel");
e.each(f,function(f,j){h.append(e("<li/>").append('<div class="dt-autofill-question">'+d[j].option(a,b)+"<div>").append(e('<div class="dt-autofill-button">').append(e('<button class="'+k.classes.btn+'">'+a.i18n("autoFill.button","&gt;")+"</button>").on("click",function(){var f=d[j].execute(a,b,e(this).closest("li"));c._update(f,b);c.dom.background.remove();c.dom.list.remove()}))))});this.dom.background.appendTo("body");this.dom.list.appendTo("body");this.dom.list.css("margin-top",-1*(this.dom.list.outerHeight()/
2))}},_detach:function(){this.dom.attachedTo=null;this.dom.handle.detach()},_drawSelection:function(b){var c=this.s.dt,a=this.s.start,d=e(this.dom.start),f={row:this.c.vertical?c.rows({page:"current"}).nodes().indexOf(b.parentNode):a.row,column:this.c.horizontal?e(b).index():a.column},b=c.column.index("toData",f.column),j=c.row(":eq("+f.row+")",{page:"current"}),j=e(c.cell(j.index(),b).node());if(c.cell(j).any()&&-1!==c.columns(this.c.columns).indexes().indexOf(b)){this.s.end=f;var h,c=a.row<f.row?
d:j;h=a.row<f.row?j:d;b=a.column<f.column?d:j;d=a.column<f.column?j:d;c=this._getPosition(c.get(0)).top;b=this._getPosition(b.get(0)).left;a=this._getPosition(h.get(0)).top+h.outerHeight()-c;d=this._getPosition(d.get(0)).left+d.outerWidth()-b;f=this.dom.select;f.top.css({top:c,left:b,width:d});f.left.css({top:c,left:b,height:a});f.bottom.css({top:c+a,left:b,width:d});f.right.css({top:c,left:b+d,height:a})}},_editor:function(b){var c=this.s.dt,a=this.c.editor;if(a){for(var d={},f=[],e=a.fields(),h=
0,i=b.length;h<i;h++)for(var p=0,k=b[h].length;p<k;p++){var n=b[h][p],g=c.settings()[0].aoColumns[n.index.column],o=g.editField;if(o===r)for(var g=g.mData,q=0,l=e.length;q<l;q++){var s=a.field(e[q]);if(s.dataSrc()===g){o=s.name();break}}if(!o)throw"Could not automatically determine field data. Please see https://datatables.net/tn/11";d[o]||(d[o]={});g=c.row(n.index.row).id();d[o][g]=n.set;f.push(n.index)}a.bubble(f,!1).multiSet(d).submit()}},_emitEvent:function(b,c){this.s.dt.iterator("table",function(a){e(a.nTable).triggerHandler(b+
".dt",c)})},_focusListener:function(){var b=this,c=this.s.dt,a=this.s.namespace,d=null!==this.c.focus?this.c.focus:c.init().keys||c.settings()[0].keytable?"focus":"hover";if("focus"===d)c.on("key-focus.autoFill",function(a,c,d){b._attach(d.node())}).on("key-blur.autoFill",function(){b._detach()});else if("click"===d)e(c.table().body()).on("click"+a,"td, th",function(){b._attach(this)}),e(i.body).on("click"+a,function(a){e(a.target).parents().filter(c.table().body()).length||b._detach()});else e(c.table().body()).on("mouseenter"+
a,"td, th",function(){b._attach(this)}).on("mouseleave"+a,function(a){e(a.relatedTarget).hasClass("dt-autofill-handle")||b._detach()})},_focusListenerRemove:function(){var b=this.s.dt;b.off(".autoFill");e(b.table().body()).off(this.s.namespace);e(i.body).off(this.s.namespace)},_getPosition:function(b,c){var a=b,d,f=0,j=0;c||(c=e(e(this.s.dt.table().node())[0].offsetParent));do{var h=a.offsetTop,i=a.offsetLeft;d=e(a.offsetParent);f+=h+1*parseInt(d.css("border-top-width"));j+=i+1*parseInt(d.css("border-left-width"));
if("body"===a.nodeName.toLowerCase())break;a=d.get(0)}while(d.get(0)!==c.get(0));return{top:f,left:j}},_mousedown:function(b){var c=this,a=this.s.dt;this.dom.start=this.dom.attachedTo;this.s.start={row:a.rows({page:"current"}).nodes().indexOf(e(this.dom.start).parent()[0]),column:e(this.dom.start).index()};e(i.body).on("mousemove.autoFill",function(a){c._mousemove(a)}).on("mouseup.autoFill",function(a){c._mouseup(a)});var d=this.dom.select,a=e(a.table().node()).offsetParent();d.top.appendTo(a);d.left.appendTo(a);
d.right.appendTo(a);d.bottom.appendTo(a);this._drawSelection(this.dom.start,b);this.dom.handle.css("display","none");b=this.dom.dtScroll;this.s.scroll={windowHeight:e(g).height(),windowWidth:e(g).width(),dtTop:b?b.offset().top:null,dtLeft:b?b.offset().left:null,dtHeight:b?b.outerHeight():null,dtWidth:b?b.outerWidth():null}},_mousemove:function(b){var c=b.target.nodeName.toLowerCase();"td"!==c&&"th"!==c||(this._drawSelection(b.target,b),this._shiftScroll(b))},_mouseup:function(b){e(i.body).off(".autoFill");
var c=this,a=this.s.dt,d=this.dom.select;d.top.remove();d.left.remove();d.right.remove();d.bottom.remove();this.dom.handle.css("display","block");var d=this.s.start,f=this.s.end;if(!(d.row===f.row&&d.column===f.column)){var j=a.cell(":eq("+d.row+")",d.column+":visible",{page:"current"});if(e("div.DTE",j.node()).length){var h=a.editor();h.on("submitSuccess.dtaf close.dtaf",function(){h.off(".dtaf");setTimeout(function(){c._mouseup(b)},100)}).on("submitComplete.dtaf preSubmitCancelled.dtaf close.dtaf",
function(){h.off(".dtaf")});h.submit()}else{for(var g=this._range(d.row,f.row),d=this._range(d.column,f.column),f=[],k=a.settings()[0],l=k.aoColumns,n=a.columns(this.c.columns).indexes(),m=0;m<g.length;m++)f.push(e.map(d,function(b){var c=a.row(":eq("+g[m]+")",{page:"current"}),b=a.cell(c.index(),b+":visible"),c=b.data(),d=b.index(),f=l[d.column].editField;f!==r&&(c=k.oApi._fnGetObjectDataFn(f)(a.row(d.row).data()));if(-1!==n.indexOf(d.column))return{cell:b,data:c,label:b.data(),index:d}}));this._actionSelector(f);
clearInterval(this.s.scrollInterval);this.s.scrollInterval=null}}},_range:function(b,c){var a=[],d;if(b<=c)for(d=b;d<=c;d++)a.push(d);else for(d=b;d>=c;d--)a.push(d);return a},_shiftScroll:function(b){var c=this,a=this.s.scroll,d=!1,f=b.pageY-i.body.scrollTop,e=b.pageX-i.body.scrollLeft,h,g,k,l;65>f?h=-5:f>a.windowHeight-65&&(h=5);65>e?g=-5:e>a.windowWidth-65&&(g=5);null!==a.dtTop&&b.pageY<a.dtTop+65?k=-5:null!==a.dtTop&&b.pageY>a.dtTop+a.dtHeight-65&&(k=5);null!==a.dtLeft&&b.pageX<a.dtLeft+65?l=
-5:null!==a.dtLeft&&b.pageX>a.dtLeft+a.dtWidth-65&&(l=5);h||g||k||l?(a.windowVert=h,a.windowHoriz=g,a.dtVert=k,a.dtHoriz=l,d=!0):this.s.scrollInterval&&(clearInterval(this.s.scrollInterval),this.s.scrollInterval=null);!this.s.scrollInterval&&d&&(this.s.scrollInterval=setInterval(function(){if(a.windowVert)i.body.scrollTop=i.body.scrollTop+a.windowVert;if(a.windowHoriz)i.body.scrollLeft=i.body.scrollLeft+a.windowHoriz;if(a.dtVert||a.dtHoriz){var b=c.dom.dtScroll[0];if(a.dtVert)b.scrollTop=b.scrollTop+
a.dtVert;if(a.dtHoriz)b.scrollLeft=b.scrollLeft+a.dtHoriz}},20))},_update:function(b,c){if(!1!==b){var a=this.s.dt,d,f=a.columns(this.c.columns).indexes();this._emitEvent("preAutoFill",[a,c]);this._editor(c);if(null!==this.c.update?this.c.update:!this.c.editor){for(var e=0,h=c.length;e<h;e++)for(var g=0,i=c[e].length;g<i;g++)d=c[e][g],-1!==f.indexOf(d.index.column)&&d.cell.data(d.set);a.draw(!1)}this._emitEvent("autoFill",[a,c])}}});k.actions={increment:{available:function(b,c){var a=c[0][0].label;
return!isNaN(a-parseFloat(a))},option:function(b){return b.i18n("autoFill.increment",'Increment / decrement each cell by: <input type="number" value="1">')},execute:function(b,c,a){for(var b=1*c[0][0].data,a=1*e("input",a).val(),d=0,f=c.length;d<f;d++)for(var j=0,g=c[d].length;j<g;j++)c[d][j].set=b,b+=a}},fill:{available:function(){return!0},option:function(b,c){return b.i18n("autoFill.fill","Fill all cells with <i>"+c[0][0].label+"</i>")},execute:function(b,c){for(var a=c[0][0].data,d=0,f=c.length;d<
f;d++)for(var e=0,g=c[d].length;e<g;e++)c[d][e].set=a}},fillHorizontal:{available:function(b,c){return 1<c.length&&1<c[0].length},option:function(b){return b.i18n("autoFill.fillHorizontal","Fill cells horizontally")},execute:function(b,c){for(var a=0,d=c.length;a<d;a++)for(var f=0,e=c[a].length;f<e;f++)c[a][f].set=c[a][0].data}},fillVertical:{available:function(b,c){return 1<c.length},option:function(b){return b.i18n("autoFill.fillVertical","Fill cells vertically")},execute:function(b,c){for(var a=
0,d=c.length;a<d;a++)for(var e=0,g=c[a].length;e<g;e++)c[a][e].set=c[0][e].data}},cancel:{available:function(){return!1},option:function(b){return b.i18n("autoFill.cancel","Cancel")},execute:function(){return!1}}};k.version="2.3.5";k.defaults={alwaysAsk:!1,focus:null,columns:"",enable:!0,update:null,editor:null,vertical:!0,horizontal:!0};k.classes={btn:"btn"};var m=e.fn.dataTable.Api;m.register("autoFill()",function(){return this});m.register("autoFill().enabled()",function(){var b=this.context[0];
return b.autoFill?b.autoFill.enabled():!1});m.register("autoFill().enable()",function(b){return this.iterator("table",function(c){c.autoFill&&c.autoFill.enable(b)})});m.register("autoFill().disable()",function(){return this.iterator("table",function(b){b.autoFill&&b.autoFill.disable()})});e(i).on("preInit.dt.autofill",function(b,c){if("dt"===b.namespace){var a=c.oInit.autoFill,d=l.defaults.autoFill;if(a||d)d=e.extend({},a,d),!1!==a&&new k(c,d)}});l.AutoFill=k;return l.AutoFill=k});

/*!
 Bootstrap integration for DataTables' AutoFill
 ©2015 SpryMedia Ltd - datatables.net/license
*/
(function(a){"function"===typeof define&&define.amd?define(["jquery","datatables.net-bs4","datatables.net-autofill"],function(b){return a(b,window,document)}):"object"===typeof exports?module.exports=function(b,c){b||(b=window);if(!c||!c.fn.dataTable)c=require("datatables.net-bs4")(b,c).$;c.fn.dataTable.AutoFill||require("datatables.net-autofill")(b,c);return a(c,b,b.document)}:a(jQuery,window,document)})(function(a){a=a.fn.dataTable;a.AutoFill.classes.btn="btn btn-primary";return a});

/*!

JSZip v3.5.0 - A JavaScript class for generating and reading zip files
<http://stuartk.com/jszip>

(c) 2009-2016 Stuart Knightley <stuart [at] stuartk.com>
Dual licenced under the MIT license or GPLv3. See https://raw.github.com/Stuk/jszip/master/LICENSE.markdown.

JSZip uses the library pako released under the MIT license :
https://github.com/nodeca/pako/blob/master/LICENSE
*/

!function(t){if("object"==typeof exports&&"undefined"!=typeof module)module.exports=t();else if("function"==typeof define&&define.amd)define([],t);else{("undefined"!=typeof window?window:"undefined"!=typeof global?global:"undefined"!=typeof self?self:this).JSZip=t()}}(function(){return function s(a,o,h){function u(r,t){if(!o[r]){if(!a[r]){var e="function"==typeof require&&require;if(!t&&e)return e(r,!0);if(l)return l(r,!0);var i=new Error("Cannot find module '"+r+"'");throw i.code="MODULE_NOT_FOUND",i}var n=o[r]={exports:{}};a[r][0].call(n.exports,function(t){var e=a[r][1][t];return u(e||t)},n,n.exports,s,a,o,h)}return o[r].exports}for(var l="function"==typeof require&&require,t=0;t<h.length;t++)u(h[t]);return u}({1:[function(t,e,r){"use strict";var c=t("./utils"),d=t("./support"),p="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";r.encode=function(t){for(var e,r,i,n,s,a,o,h=[],u=0,l=t.length,f=l,d="string"!==c.getTypeOf(t);u<t.length;)f=l-u,i=d?(e=t[u++],r=u<l?t[u++]:0,u<l?t[u++]:0):(e=t.charCodeAt(u++),r=u<l?t.charCodeAt(u++):0,u<l?t.charCodeAt(u++):0),n=e>>2,s=(3&e)<<4|r>>4,a=1<f?(15&r)<<2|i>>6:64,o=2<f?63&i:64,h.push(p.charAt(n)+p.charAt(s)+p.charAt(a)+p.charAt(o));return h.join("")},r.decode=function(t){var e,r,i,n,s,a,o=0,h=0,u="data:";if(t.substr(0,u.length)===u)throw new Error("Invalid base64 input, it looks like a data url.");var l,f=3*(t=t.replace(/[^A-Za-z0-9\+\/\=]/g,"")).length/4;if(t.charAt(t.length-1)===p.charAt(64)&&f--,t.charAt(t.length-2)===p.charAt(64)&&f--,f%1!=0)throw new Error("Invalid base64 input, bad content length.");for(l=d.uint8array?new Uint8Array(0|f):new Array(0|f);o<t.length;)e=p.indexOf(t.charAt(o++))<<2|(n=p.indexOf(t.charAt(o++)))>>4,r=(15&n)<<4|(s=p.indexOf(t.charAt(o++)))>>2,i=(3&s)<<6|(a=p.indexOf(t.charAt(o++))),l[h++]=e,64!==s&&(l[h++]=r),64!==a&&(l[h++]=i);return l}},{"./support":30,"./utils":32}],2:[function(t,e,r){"use strict";var i=t("./external"),n=t("./stream/DataWorker"),s=t("./stream/DataLengthProbe"),a=t("./stream/Crc32Probe");s=t("./stream/DataLengthProbe");function o(t,e,r,i,n){this.compressedSize=t,this.uncompressedSize=e,this.crc32=r,this.compression=i,this.compressedContent=n}o.prototype={getContentWorker:function(){var t=new n(i.Promise.resolve(this.compressedContent)).pipe(this.compression.uncompressWorker()).pipe(new s("data_length")),e=this;return t.on("end",function(){if(this.streamInfo.data_length!==e.uncompressedSize)throw new Error("Bug : uncompressed data size mismatch")}),t},getCompressedWorker:function(){return new n(i.Promise.resolve(this.compressedContent)).withStreamInfo("compressedSize",this.compressedSize).withStreamInfo("uncompressedSize",this.uncompressedSize).withStreamInfo("crc32",this.crc32).withStreamInfo("compression",this.compression)}},o.createWorkerFrom=function(t,e,r){return t.pipe(new a).pipe(new s("uncompressedSize")).pipe(e.compressWorker(r)).pipe(new s("compressedSize")).withStreamInfo("compression",e)},e.exports=o},{"./external":6,"./stream/Crc32Probe":25,"./stream/DataLengthProbe":26,"./stream/DataWorker":27}],3:[function(t,e,r){"use strict";var i=t("./stream/GenericWorker");r.STORE={magic:"\0\0",compressWorker:function(t){return new i("STORE compression")},uncompressWorker:function(){return new i("STORE decompression")}},r.DEFLATE=t("./flate")},{"./flate":7,"./stream/GenericWorker":28}],4:[function(t,e,r){"use strict";var i=t("./utils");var o=function(){for(var t,e=[],r=0;r<256;r++){t=r;for(var i=0;i<8;i++)t=1&t?3988292384^t>>>1:t>>>1;e[r]=t}return e}();e.exports=function(t,e){return void 0!==t&&t.length?"string"!==i.getTypeOf(t)?function(t,e,r,i){var n=o,s=i+r;t^=-1;for(var a=i;a<s;a++)t=t>>>8^n[255&(t^e[a])];return-1^t}(0|e,t,t.length,0):function(t,e,r,i){var n=o,s=i+r;t^=-1;for(var a=i;a<s;a++)t=t>>>8^n[255&(t^e.charCodeAt(a))];return-1^t}(0|e,t,t.length,0):0}},{"./utils":32}],5:[function(t,e,r){"use strict";r.base64=!1,r.binary=!1,r.dir=!1,r.createFolders=!0,r.date=null,r.compression=null,r.compressionOptions=null,r.comment=null,r.unixPermissions=null,r.dosPermissions=null},{}],6:[function(t,e,r){"use strict";var i=null;i="undefined"!=typeof Promise?Promise:t("lie"),e.exports={Promise:i}},{lie:37}],7:[function(t,e,r){"use strict";var i="undefined"!=typeof Uint8Array&&"undefined"!=typeof Uint16Array&&"undefined"!=typeof Uint32Array,n=t("pako"),s=t("./utils"),a=t("./stream/GenericWorker"),o=i?"uint8array":"array";function h(t,e){a.call(this,"FlateWorker/"+t),this._pako=null,this._pakoAction=t,this._pakoOptions=e,this.meta={}}r.magic="\b\0",s.inherits(h,a),h.prototype.processChunk=function(t){this.meta=t.meta,null===this._pako&&this._createPako(),this._pako.push(s.transformTo(o,t.data),!1)},h.prototype.flush=function(){a.prototype.flush.call(this),null===this._pako&&this._createPako(),this._pako.push([],!0)},h.prototype.cleanUp=function(){a.prototype.cleanUp.call(this),this._pako=null},h.prototype._createPako=function(){this._pako=new n[this._pakoAction]({raw:!0,level:this._pakoOptions.level||-1});var e=this;this._pako.onData=function(t){e.push({data:t,meta:e.meta})}},r.compressWorker=function(t){return new h("Deflate",t)},r.uncompressWorker=function(){return new h("Inflate",{})}},{"./stream/GenericWorker":28,"./utils":32,pako:38}],8:[function(t,e,r){"use strict";function A(t,e){var r,i="";for(r=0;r<e;r++)i+=String.fromCharCode(255&t),t>>>=8;return i}function i(t,e,r,i,n,s){var a,o,h=t.file,u=t.compression,l=s!==O.utf8encode,f=I.transformTo("string",s(h.name)),d=I.transformTo("string",O.utf8encode(h.name)),c=h.comment,p=I.transformTo("string",s(c)),m=I.transformTo("string",O.utf8encode(c)),_=d.length!==h.name.length,g=m.length!==c.length,b="",v="",y="",w=h.dir,k=h.date,x={crc32:0,compressedSize:0,uncompressedSize:0};e&&!r||(x.crc32=t.crc32,x.compressedSize=t.compressedSize,x.uncompressedSize=t.uncompressedSize);var S=0;e&&(S|=8),l||!_&&!g||(S|=2048);var z=0,C=0;w&&(z|=16),"UNIX"===n?(C=798,z|=function(t,e){var r=t;return t||(r=e?16893:33204),(65535&r)<<16}(h.unixPermissions,w)):(C=20,z|=function(t){return 63&(t||0)}(h.dosPermissions)),a=k.getUTCHours(),a<<=6,a|=k.getUTCMinutes(),a<<=5,a|=k.getUTCSeconds()/2,o=k.getUTCFullYear()-1980,o<<=4,o|=k.getUTCMonth()+1,o<<=5,o|=k.getUTCDate(),_&&(v=A(1,1)+A(B(f),4)+d,b+="up"+A(v.length,2)+v),g&&(y=A(1,1)+A(B(p),4)+m,b+="uc"+A(y.length,2)+y);var E="";return E+="\n\0",E+=A(S,2),E+=u.magic,E+=A(a,2),E+=A(o,2),E+=A(x.crc32,4),E+=A(x.compressedSize,4),E+=A(x.uncompressedSize,4),E+=A(f.length,2),E+=A(b.length,2),{fileRecord:R.LOCAL_FILE_HEADER+E+f+b,dirRecord:R.CENTRAL_FILE_HEADER+A(C,2)+E+A(p.length,2)+"\0\0\0\0"+A(z,4)+A(i,4)+f+b+p}}var I=t("../utils"),n=t("../stream/GenericWorker"),O=t("../utf8"),B=t("../crc32"),R=t("../signature");function s(t,e,r,i){n.call(this,"ZipFileWorker"),this.bytesWritten=0,this.zipComment=e,this.zipPlatform=r,this.encodeFileName=i,this.streamFiles=t,this.accumulate=!1,this.contentBuffer=[],this.dirRecords=[],this.currentSourceOffset=0,this.entriesCount=0,this.currentFile=null,this._sources=[]}I.inherits(s,n),s.prototype.push=function(t){var e=t.meta.percent||0,r=this.entriesCount,i=this._sources.length;this.accumulate?this.contentBuffer.push(t):(this.bytesWritten+=t.data.length,n.prototype.push.call(this,{data:t.data,meta:{currentFile:this.currentFile,percent:r?(e+100*(r-i-1))/r:100}}))},s.prototype.openedSource=function(t){this.currentSourceOffset=this.bytesWritten,this.currentFile=t.file.name;var e=this.streamFiles&&!t.file.dir;if(e){var r=i(t,e,!1,this.currentSourceOffset,this.zipPlatform,this.encodeFileName);this.push({data:r.fileRecord,meta:{percent:0}})}else this.accumulate=!0},s.prototype.closedSource=function(t){this.accumulate=!1;var e=this.streamFiles&&!t.file.dir,r=i(t,e,!0,this.currentSourceOffset,this.zipPlatform,this.encodeFileName);if(this.dirRecords.push(r.dirRecord),e)this.push({data:function(t){return R.DATA_DESCRIPTOR+A(t.crc32,4)+A(t.compressedSize,4)+A(t.uncompressedSize,4)}(t),meta:{percent:100}});else for(this.push({data:r.fileRecord,meta:{percent:0}});this.contentBuffer.length;)this.push(this.contentBuffer.shift());this.currentFile=null},s.prototype.flush=function(){for(var t=this.bytesWritten,e=0;e<this.dirRecords.length;e++)this.push({data:this.dirRecords[e],meta:{percent:100}});var r=this.bytesWritten-t,i=function(t,e,r,i,n){var s=I.transformTo("string",n(i));return R.CENTRAL_DIRECTORY_END+"\0\0\0\0"+A(t,2)+A(t,2)+A(e,4)+A(r,4)+A(s.length,2)+s}(this.dirRecords.length,r,t,this.zipComment,this.encodeFileName);this.push({data:i,meta:{percent:100}})},s.prototype.prepareNextSource=function(){this.previous=this._sources.shift(),this.openedSource(this.previous.streamInfo),this.isPaused?this.previous.pause():this.previous.resume()},s.prototype.registerPrevious=function(t){this._sources.push(t);var e=this;return t.on("data",function(t){e.processChunk(t)}),t.on("end",function(){e.closedSource(e.previous.streamInfo),e._sources.length?e.prepareNextSource():e.end()}),t.on("error",function(t){e.error(t)}),this},s.prototype.resume=function(){return!!n.prototype.resume.call(this)&&(!this.previous&&this._sources.length?(this.prepareNextSource(),!0):this.previous||this._sources.length||this.generatedError?void 0:(this.end(),!0))},s.prototype.error=function(t){var e=this._sources;if(!n.prototype.error.call(this,t))return!1;for(var r=0;r<e.length;r++)try{e[r].error(t)}catch(t){}return!0},s.prototype.lock=function(){n.prototype.lock.call(this);for(var t=this._sources,e=0;e<t.length;e++)t[e].lock()},e.exports=s},{"../crc32":4,"../signature":23,"../stream/GenericWorker":28,"../utf8":31,"../utils":32}],9:[function(t,e,r){"use strict";var u=t("../compressions"),i=t("./ZipFileWorker");r.generateWorker=function(t,a,e){var o=new i(a.streamFiles,e,a.platform,a.encodeFileName),h=0;try{t.forEach(function(t,e){h++;var r=function(t,e){var r=t||e,i=u[r];if(!i)throw new Error(r+" is not a valid compression method !");return i}(e.options.compression,a.compression),i=e.options.compressionOptions||a.compressionOptions||{},n=e.dir,s=e.date;e._compressWorker(r,i).withStreamInfo("file",{name:t,dir:n,date:s,comment:e.comment||"",unixPermissions:e.unixPermissions,dosPermissions:e.dosPermissions}).pipe(o)}),o.entriesCount=h}catch(t){o.error(t)}return o}},{"../compressions":3,"./ZipFileWorker":8}],10:[function(t,e,r){"use strict";function i(){if(!(this instanceof i))return new i;if(arguments.length)throw new Error("The constructor with parameters has been removed in JSZip 3.0, please check the upgrade guide.");this.files={},this.comment=null,this.root="",this.clone=function(){var t=new i;for(var e in this)"function"!=typeof this[e]&&(t[e]=this[e]);return t}}(i.prototype=t("./object")).loadAsync=t("./load"),i.support=t("./support"),i.defaults=t("./defaults"),i.version="3.5.0",i.loadAsync=function(t,e){return(new i).loadAsync(t,e)},i.external=t("./external"),e.exports=i},{"./defaults":5,"./external":6,"./load":11,"./object":15,"./support":30}],11:[function(t,e,r){"use strict";var i=t("./utils"),n=t("./external"),o=t("./utf8"),h=(i=t("./utils"),t("./zipEntries")),s=t("./stream/Crc32Probe"),u=t("./nodejsUtils");function l(i){return new n.Promise(function(t,e){var r=i.decompressed.getContentWorker().pipe(new s);r.on("error",function(t){e(t)}).on("end",function(){r.streamInfo.crc32!==i.decompressed.crc32?e(new Error("Corrupted zip : CRC32 mismatch")):t()}).resume()})}e.exports=function(t,s){var a=this;return s=i.extend(s||{},{base64:!1,checkCRC32:!1,optimizedBinaryString:!1,createFolders:!1,decodeFileName:o.utf8decode}),u.isNode&&u.isStream(t)?n.Promise.reject(new Error("JSZip can't accept a stream when loading a zip file.")):i.prepareContent("the loaded zip file",t,!0,s.optimizedBinaryString,s.base64).then(function(t){var e=new h(s);return e.load(t),e}).then(function(t){var e=[n.Promise.resolve(t)],r=t.files;if(s.checkCRC32)for(var i=0;i<r.length;i++)e.push(l(r[i]));return n.Promise.all(e)}).then(function(t){for(var e=t.shift(),r=e.files,i=0;i<r.length;i++){var n=r[i];a.file(n.fileNameStr,n.decompressed,{binary:!0,optimizedBinaryString:!0,date:n.date,dir:n.dir,comment:n.fileCommentStr.length?n.fileCommentStr:null,unixPermissions:n.unixPermissions,dosPermissions:n.dosPermissions,createFolders:s.createFolders})}return e.zipComment.length&&(a.comment=e.zipComment),a})}},{"./external":6,"./nodejsUtils":14,"./stream/Crc32Probe":25,"./utf8":31,"./utils":32,"./zipEntries":33}],12:[function(t,e,r){"use strict";var i=t("../utils"),n=t("../stream/GenericWorker");function s(t,e){n.call(this,"Nodejs stream input adapter for "+t),this._upstreamEnded=!1,this._bindStream(e)}i.inherits(s,n),s.prototype._bindStream=function(t){var e=this;(this._stream=t).pause(),t.on("data",function(t){e.push({data:t,meta:{percent:0}})}).on("error",function(t){e.isPaused?this.generatedError=t:e.error(t)}).on("end",function(){e.isPaused?e._upstreamEnded=!0:e.end()})},s.prototype.pause=function(){return!!n.prototype.pause.call(this)&&(this._stream.pause(),!0)},s.prototype.resume=function(){return!!n.prototype.resume.call(this)&&(this._upstreamEnded?this.end():this._stream.resume(),!0)},e.exports=s},{"../stream/GenericWorker":28,"../utils":32}],13:[function(t,e,r){"use strict";var n=t("readable-stream").Readable;function i(t,e,r){n.call(this,e),this._helper=t;var i=this;t.on("data",function(t,e){i.push(t)||i._helper.pause(),r&&r(e)}).on("error",function(t){i.emit("error",t)}).on("end",function(){i.push(null)})}t("../utils").inherits(i,n),i.prototype._read=function(){this._helper.resume()},e.exports=i},{"../utils":32,"readable-stream":16}],14:[function(t,e,r){"use strict";e.exports={isNode:"undefined"!=typeof Buffer,newBufferFrom:function(t,e){if(Buffer.from&&Buffer.from!==Uint8Array.from)return Buffer.from(t,e);if("number"==typeof t)throw new Error('The "data" argument must not be a number');return new Buffer(t,e)},allocBuffer:function(t){if(Buffer.alloc)return Buffer.alloc(t);var e=new Buffer(t);return e.fill(0),e},isBuffer:function(t){return Buffer.isBuffer(t)},isStream:function(t){return t&&"function"==typeof t.on&&"function"==typeof t.pause&&"function"==typeof t.resume}}},{}],15:[function(t,e,r){"use strict";function s(t,e,r){var i,n=u.getTypeOf(e),s=u.extend(r||{},f);s.date=s.date||new Date,null!==s.compression&&(s.compression=s.compression.toUpperCase()),"string"==typeof s.unixPermissions&&(s.unixPermissions=parseInt(s.unixPermissions,8)),s.unixPermissions&&16384&s.unixPermissions&&(s.dir=!0),s.dosPermissions&&16&s.dosPermissions&&(s.dir=!0),s.dir&&(t=g(t)),s.createFolders&&(i=_(t))&&b.call(this,i,!0);var a="string"===n&&!1===s.binary&&!1===s.base64;r&&void 0!==r.binary||(s.binary=!a),(e instanceof d&&0===e.uncompressedSize||s.dir||!e||0===e.length)&&(s.base64=!1,s.binary=!0,e="",s.compression="STORE",n="string");var o=null;o=e instanceof d||e instanceof l?e:p.isNode&&p.isStream(e)?new m(t,e):u.prepareContent(t,e,s.binary,s.optimizedBinaryString,s.base64);var h=new c(t,o,s);this.files[t]=h}var n=t("./utf8"),u=t("./utils"),l=t("./stream/GenericWorker"),a=t("./stream/StreamHelper"),f=t("./defaults"),d=t("./compressedObject"),c=t("./zipObject"),o=t("./generate"),p=t("./nodejsUtils"),m=t("./nodejs/NodejsStreamInputAdapter"),_=function(t){"/"===t.slice(-1)&&(t=t.substring(0,t.length-1));var e=t.lastIndexOf("/");return 0<e?t.substring(0,e):""},g=function(t){return"/"!==t.slice(-1)&&(t+="/"),t},b=function(t,e){return e=void 0!==e?e:f.createFolders,t=g(t),this.files[t]||s.call(this,t,null,{dir:!0,createFolders:e}),this.files[t]};function h(t){return"[object RegExp]"===Object.prototype.toString.call(t)}var i={load:function(){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},forEach:function(t){var e,r,i;for(e in this.files)this.files.hasOwnProperty(e)&&(i=this.files[e],(r=e.slice(this.root.length,e.length))&&e.slice(0,this.root.length)===this.root&&t(r,i))},filter:function(r){var i=[];return this.forEach(function(t,e){r(t,e)&&i.push(e)}),i},file:function(t,e,r){if(1!==arguments.length)return t=this.root+t,s.call(this,t,e,r),this;if(h(t)){var i=t;return this.filter(function(t,e){return!e.dir&&i.test(t)})}var n=this.files[this.root+t];return n&&!n.dir?n:null},folder:function(r){if(!r)return this;if(h(r))return this.filter(function(t,e){return e.dir&&r.test(t)});var t=this.root+r,e=b.call(this,t),i=this.clone();return i.root=e.name,i},remove:function(r){r=this.root+r;var t=this.files[r];if(t||("/"!==r.slice(-1)&&(r+="/"),t=this.files[r]),t&&!t.dir)delete this.files[r];else for(var e=this.filter(function(t,e){return e.name.slice(0,r.length)===r}),i=0;i<e.length;i++)delete this.files[e[i].name];return this},generate:function(t){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},generateInternalStream:function(t){var e,r={};try{if((r=u.extend(t||{},{streamFiles:!1,compression:"STORE",compressionOptions:null,type:"",platform:"DOS",comment:null,mimeType:"application/zip",encodeFileName:n.utf8encode})).type=r.type.toLowerCase(),r.compression=r.compression.toUpperCase(),"binarystring"===r.type&&(r.type="string"),!r.type)throw new Error("No output type specified.");u.checkSupport(r.type),"darwin"!==r.platform&&"freebsd"!==r.platform&&"linux"!==r.platform&&"sunos"!==r.platform||(r.platform="UNIX"),"win32"===r.platform&&(r.platform="DOS");var i=r.comment||this.comment||"";e=o.generateWorker(this,r,i)}catch(t){(e=new l("error")).error(t)}return new a(e,r.type||"string",r.mimeType)},generateAsync:function(t,e){return this.generateInternalStream(t).accumulate(e)},generateNodeStream:function(t,e){return(t=t||{}).type||(t.type="nodebuffer"),this.generateInternalStream(t).toNodejsStream(e)}};e.exports=i},{"./compressedObject":2,"./defaults":5,"./generate":9,"./nodejs/NodejsStreamInputAdapter":12,"./nodejsUtils":14,"./stream/GenericWorker":28,"./stream/StreamHelper":29,"./utf8":31,"./utils":32,"./zipObject":35}],16:[function(t,e,r){e.exports=t("stream")},{stream:void 0}],17:[function(t,e,r){"use strict";var i=t("./DataReader");function n(t){i.call(this,t);for(var e=0;e<this.data.length;e++)t[e]=255&t[e]}t("../utils").inherits(n,i),n.prototype.byteAt=function(t){return this.data[this.zero+t]},n.prototype.lastIndexOfSignature=function(t){for(var e=t.charCodeAt(0),r=t.charCodeAt(1),i=t.charCodeAt(2),n=t.charCodeAt(3),s=this.length-4;0<=s;--s)if(this.data[s]===e&&this.data[s+1]===r&&this.data[s+2]===i&&this.data[s+3]===n)return s-this.zero;return-1},n.prototype.readAndCheckSignature=function(t){var e=t.charCodeAt(0),r=t.charCodeAt(1),i=t.charCodeAt(2),n=t.charCodeAt(3),s=this.readData(4);return e===s[0]&&r===s[1]&&i===s[2]&&n===s[3]},n.prototype.readData=function(t){if(this.checkOffset(t),0===t)return[];var e=this.data.slice(this.zero+this.index,this.zero+this.index+t);return this.index+=t,e},e.exports=n},{"../utils":32,"./DataReader":18}],18:[function(t,e,r){"use strict";var i=t("../utils");function n(t){this.data=t,this.length=t.length,this.index=0,this.zero=0}n.prototype={checkOffset:function(t){this.checkIndex(this.index+t)},checkIndex:function(t){if(this.length<this.zero+t||t<0)throw new Error("End of data reached (data length = "+this.length+", asked index = "+t+"). Corrupted zip ?")},setIndex:function(t){this.checkIndex(t),this.index=t},skip:function(t){this.setIndex(this.index+t)},byteAt:function(t){},readInt:function(t){var e,r=0;for(this.checkOffset(t),e=this.index+t-1;e>=this.index;e--)r=(r<<8)+this.byteAt(e);return this.index+=t,r},readString:function(t){return i.transformTo("string",this.readData(t))},readData:function(t){},lastIndexOfSignature:function(t){},readAndCheckSignature:function(t){},readDate:function(){var t=this.readInt(4);return new Date(Date.UTC(1980+(t>>25&127),(t>>21&15)-1,t>>16&31,t>>11&31,t>>5&63,(31&t)<<1))}},e.exports=n},{"../utils":32}],19:[function(t,e,r){"use strict";var i=t("./Uint8ArrayReader");function n(t){i.call(this,t)}t("../utils").inherits(n,i),n.prototype.readData=function(t){this.checkOffset(t);var e=this.data.slice(this.zero+this.index,this.zero+this.index+t);return this.index+=t,e},e.exports=n},{"../utils":32,"./Uint8ArrayReader":21}],20:[function(t,e,r){"use strict";var i=t("./DataReader");function n(t){i.call(this,t)}t("../utils").inherits(n,i),n.prototype.byteAt=function(t){return this.data.charCodeAt(this.zero+t)},n.prototype.lastIndexOfSignature=function(t){return this.data.lastIndexOf(t)-this.zero},n.prototype.readAndCheckSignature=function(t){return t===this.readData(4)},n.prototype.readData=function(t){this.checkOffset(t);var e=this.data.slice(this.zero+this.index,this.zero+this.index+t);return this.index+=t,e},e.exports=n},{"../utils":32,"./DataReader":18}],21:[function(t,e,r){"use strict";var i=t("./ArrayReader");function n(t){i.call(this,t)}t("../utils").inherits(n,i),n.prototype.readData=function(t){if(this.checkOffset(t),0===t)return new Uint8Array(0);var e=this.data.subarray(this.zero+this.index,this.zero+this.index+t);return this.index+=t,e},e.exports=n},{"../utils":32,"./ArrayReader":17}],22:[function(t,e,r){"use strict";var i=t("../utils"),n=t("../support"),s=t("./ArrayReader"),a=t("./StringReader"),o=t("./NodeBufferReader"),h=t("./Uint8ArrayReader");e.exports=function(t){var e=i.getTypeOf(t);return i.checkSupport(e),"string"!==e||n.uint8array?"nodebuffer"===e?new o(t):n.uint8array?new h(i.transformTo("uint8array",t)):new s(i.transformTo("array",t)):new a(t)}},{"../support":30,"../utils":32,"./ArrayReader":17,"./NodeBufferReader":19,"./StringReader":20,"./Uint8ArrayReader":21}],23:[function(t,e,r){"use strict";r.LOCAL_FILE_HEADER="PK",r.CENTRAL_FILE_HEADER="PK",r.CENTRAL_DIRECTORY_END="PK",r.ZIP64_CENTRAL_DIRECTORY_LOCATOR="PK",r.ZIP64_CENTRAL_DIRECTORY_END="PK",r.DATA_DESCRIPTOR="PK\b"},{}],24:[function(t,e,r){"use strict";var i=t("./GenericWorker"),n=t("../utils");function s(t){i.call(this,"ConvertWorker to "+t),this.destType=t}n.inherits(s,i),s.prototype.processChunk=function(t){this.push({data:n.transformTo(this.destType,t.data),meta:t.meta})},e.exports=s},{"../utils":32,"./GenericWorker":28}],25:[function(t,e,r){"use strict";var i=t("./GenericWorker"),n=t("../crc32");function s(){i.call(this,"Crc32Probe"),this.withStreamInfo("crc32",0)}t("../utils").inherits(s,i),s.prototype.processChunk=function(t){this.streamInfo.crc32=n(t.data,this.streamInfo.crc32||0),this.push(t)},e.exports=s},{"../crc32":4,"../utils":32,"./GenericWorker":28}],26:[function(t,e,r){"use strict";var i=t("../utils"),n=t("./GenericWorker");function s(t){n.call(this,"DataLengthProbe for "+t),this.propName=t,this.withStreamInfo(t,0)}i.inherits(s,n),s.prototype.processChunk=function(t){if(t){var e=this.streamInfo[this.propName]||0;this.streamInfo[this.propName]=e+t.data.length}n.prototype.processChunk.call(this,t)},e.exports=s},{"../utils":32,"./GenericWorker":28}],27:[function(t,e,r){"use strict";var i=t("../utils"),n=t("./GenericWorker");function s(t){n.call(this,"DataWorker");var e=this;this.dataIsReady=!1,this.index=0,this.max=0,this.data=null,this.type="",this._tickScheduled=!1,t.then(function(t){e.dataIsReady=!0,e.data=t,e.max=t&&t.length||0,e.type=i.getTypeOf(t),e.isPaused||e._tickAndRepeat()},function(t){e.error(t)})}i.inherits(s,n),s.prototype.cleanUp=function(){n.prototype.cleanUp.call(this),this.data=null},s.prototype.resume=function(){return!!n.prototype.resume.call(this)&&(!this._tickScheduled&&this.dataIsReady&&(this._tickScheduled=!0,i.delay(this._tickAndRepeat,[],this)),!0)},s.prototype._tickAndRepeat=function(){this._tickScheduled=!1,this.isPaused||this.isFinished||(this._tick(),this.isFinished||(i.delay(this._tickAndRepeat,[],this),this._tickScheduled=!0))},s.prototype._tick=function(){if(this.isPaused||this.isFinished)return!1;var t=null,e=Math.min(this.max,this.index+16384);if(this.index>=this.max)return this.end();switch(this.type){case"string":t=this.data.substring(this.index,e);break;case"uint8array":t=this.data.subarray(this.index,e);break;case"array":case"nodebuffer":t=this.data.slice(this.index,e)}return this.index=e,this.push({data:t,meta:{percent:this.max?this.index/this.max*100:0}})},e.exports=s},{"../utils":32,"./GenericWorker":28}],28:[function(t,e,r){"use strict";function i(t){this.name=t||"default",this.streamInfo={},this.generatedError=null,this.extraStreamInfo={},this.isPaused=!0,this.isFinished=!1,this.isLocked=!1,this._listeners={data:[],end:[],error:[]},this.previous=null}i.prototype={push:function(t){this.emit("data",t)},end:function(){if(this.isFinished)return!1;this.flush();try{this.emit("end"),this.cleanUp(),this.isFinished=!0}catch(t){this.emit("error",t)}return!0},error:function(t){return!this.isFinished&&(this.isPaused?this.generatedError=t:(this.isFinished=!0,this.emit("error",t),this.previous&&this.previous.error(t),this.cleanUp()),!0)},on:function(t,e){return this._listeners[t].push(e),this},cleanUp:function(){this.streamInfo=this.generatedError=this.extraStreamInfo=null,this._listeners=[]},emit:function(t,e){if(this._listeners[t])for(var r=0;r<this._listeners[t].length;r++)this._listeners[t][r].call(this,e)},pipe:function(t){return t.registerPrevious(this)},registerPrevious:function(t){if(this.isLocked)throw new Error("The stream '"+this+"' has already been used.");this.streamInfo=t.streamInfo,this.mergeStreamInfo(),this.previous=t;var e=this;return t.on("data",function(t){e.processChunk(t)}),t.on("end",function(){e.end()}),t.on("error",function(t){e.error(t)}),this},pause:function(){return!this.isPaused&&!this.isFinished&&(this.isPaused=!0,this.previous&&this.previous.pause(),!0)},resume:function(){if(!this.isPaused||this.isFinished)return!1;var t=this.isPaused=!1;return this.generatedError&&(this.error(this.generatedError),t=!0),this.previous&&this.previous.resume(),!t},flush:function(){},processChunk:function(t){this.push(t)},withStreamInfo:function(t,e){return this.extraStreamInfo[t]=e,this.mergeStreamInfo(),this},mergeStreamInfo:function(){for(var t in this.extraStreamInfo)this.extraStreamInfo.hasOwnProperty(t)&&(this.streamInfo[t]=this.extraStreamInfo[t])},lock:function(){if(this.isLocked)throw new Error("The stream '"+this+"' has already been used.");this.isLocked=!0,this.previous&&this.previous.lock()},toString:function(){var t="Worker "+this.name;return this.previous?this.previous+" -> "+t:t}},e.exports=i},{}],29:[function(t,e,r){"use strict";var h=t("../utils"),n=t("./ConvertWorker"),s=t("./GenericWorker"),u=t("../base64"),i=t("../support"),a=t("../external"),o=null;if(i.nodestream)try{o=t("../nodejs/NodejsStreamOutputAdapter")}catch(t){}function l(t,o){return new a.Promise(function(e,r){var i=[],n=t._internalType,s=t._outputType,a=t._mimeType;t.on("data",function(t,e){i.push(t),o&&o(e)}).on("error",function(t){i=[],r(t)}).on("end",function(){try{var t=function(t,e,r){switch(t){case"blob":return h.newBlob(h.transformTo("arraybuffer",e),r);case"base64":return u.encode(e);default:return h.transformTo(t,e)}}(s,function(t,e){var r,i=0,n=null,s=0;for(r=0;r<e.length;r++)s+=e[r].length;switch(t){case"string":return e.join("");case"array":return Array.prototype.concat.apply([],e);case"uint8array":for(n=new Uint8Array(s),r=0;r<e.length;r++)n.set(e[r],i),i+=e[r].length;return n;case"nodebuffer":return Buffer.concat(e);default:throw new Error("concat : unsupported type '"+t+"'")}}(n,i),a);e(t)}catch(t){r(t)}i=[]}).resume()})}function f(t,e,r){var i=e;switch(e){case"blob":case"arraybuffer":i="uint8array";break;case"base64":i="string"}try{this._internalType=i,this._outputType=e,this._mimeType=r,h.checkSupport(i),this._worker=t.pipe(new n(i)),t.lock()}catch(t){this._worker=new s("error"),this._worker.error(t)}}f.prototype={accumulate:function(t){return l(this,t)},on:function(t,e){var r=this;return"data"===t?this._worker.on(t,function(t){e.call(r,t.data,t.meta)}):this._worker.on(t,function(){h.delay(e,arguments,r)}),this},resume:function(){return h.delay(this._worker.resume,[],this._worker),this},pause:function(){return this._worker.pause(),this},toNodejsStream:function(t){if(h.checkSupport("nodestream"),"nodebuffer"!==this._outputType)throw new Error(this._outputType+" is not supported by this method");return new o(this,{objectMode:"nodebuffer"!==this._outputType},t)}},e.exports=f},{"../base64":1,"../external":6,"../nodejs/NodejsStreamOutputAdapter":13,"../support":30,"../utils":32,"./ConvertWorker":24,"./GenericWorker":28}],30:[function(t,e,r){"use strict";if(r.base64=!0,r.array=!0,r.string=!0,r.arraybuffer="undefined"!=typeof ArrayBuffer&&"undefined"!=typeof Uint8Array,r.nodebuffer="undefined"!=typeof Buffer,r.uint8array="undefined"!=typeof Uint8Array,"undefined"==typeof ArrayBuffer)r.blob=!1;else{var i=new ArrayBuffer(0);try{r.blob=0===new Blob([i],{type:"application/zip"}).size}catch(t){try{var n=new(self.BlobBuilder||self.WebKitBlobBuilder||self.MozBlobBuilder||self.MSBlobBuilder);n.append(i),r.blob=0===n.getBlob("application/zip").size}catch(t){r.blob=!1}}}try{r.nodestream=!!t("readable-stream").Readable}catch(t){r.nodestream=!1}},{"readable-stream":16}],31:[function(t,e,s){"use strict";for(var o=t("./utils"),h=t("./support"),r=t("./nodejsUtils"),i=t("./stream/GenericWorker"),u=new Array(256),n=0;n<256;n++)u[n]=252<=n?6:248<=n?5:240<=n?4:224<=n?3:192<=n?2:1;u[254]=u[254]=1;function a(){i.call(this,"utf-8 decode"),this.leftOver=null}function l(){i.call(this,"utf-8 encode")}s.utf8encode=function(t){return h.nodebuffer?r.newBufferFrom(t,"utf-8"):function(t){var e,r,i,n,s,a=t.length,o=0;for(n=0;n<a;n++)55296==(64512&(r=t.charCodeAt(n)))&&n+1<a&&56320==(64512&(i=t.charCodeAt(n+1)))&&(r=65536+(r-55296<<10)+(i-56320),n++),o+=r<128?1:r<2048?2:r<65536?3:4;for(e=h.uint8array?new Uint8Array(o):new Array(o),n=s=0;s<o;n++)55296==(64512&(r=t.charCodeAt(n)))&&n+1<a&&56320==(64512&(i=t.charCodeAt(n+1)))&&(r=65536+(r-55296<<10)+(i-56320),n++),r<128?e[s++]=r:(r<2048?e[s++]=192|r>>>6:(r<65536?e[s++]=224|r>>>12:(e[s++]=240|r>>>18,e[s++]=128|r>>>12&63),e[s++]=128|r>>>6&63),e[s++]=128|63&r);return e}(t)},s.utf8decode=function(t){return h.nodebuffer?o.transformTo("nodebuffer",t).toString("utf-8"):function(t){var e,r,i,n,s=t.length,a=new Array(2*s);for(e=r=0;e<s;)if((i=t[e++])<128)a[r++]=i;else if(4<(n=u[i]))a[r++]=65533,e+=n-1;else{for(i&=2===n?31:3===n?15:7;1<n&&e<s;)i=i<<6|63&t[e++],n--;1<n?a[r++]=65533:i<65536?a[r++]=i:(i-=65536,a[r++]=55296|i>>10&1023,a[r++]=56320|1023&i)}return a.length!==r&&(a.subarray?a=a.subarray(0,r):a.length=r),o.applyFromCharCode(a)}(t=o.transformTo(h.uint8array?"uint8array":"array",t))},o.inherits(a,i),a.prototype.processChunk=function(t){var e=o.transformTo(h.uint8array?"uint8array":"array",t.data);if(this.leftOver&&this.leftOver.length){if(h.uint8array){var r=e;(e=new Uint8Array(r.length+this.leftOver.length)).set(this.leftOver,0),e.set(r,this.leftOver.length)}else e=this.leftOver.concat(e);this.leftOver=null}var i=function(t,e){var r;for((e=e||t.length)>t.length&&(e=t.length),r=e-1;0<=r&&128==(192&t[r]);)r--;return r<0?e:0===r?e:r+u[t[r]]>e?r:e}(e),n=e;i!==e.length&&(h.uint8array?(n=e.subarray(0,i),this.leftOver=e.subarray(i,e.length)):(n=e.slice(0,i),this.leftOver=e.slice(i,e.length))),this.push({data:s.utf8decode(n),meta:t.meta})},a.prototype.flush=function(){this.leftOver&&this.leftOver.length&&(this.push({data:s.utf8decode(this.leftOver),meta:{}}),this.leftOver=null)},s.Utf8DecodeWorker=a,o.inherits(l,i),l.prototype.processChunk=function(t){this.push({data:s.utf8encode(t.data),meta:t.meta})},s.Utf8EncodeWorker=l},{"./nodejsUtils":14,"./stream/GenericWorker":28,"./support":30,"./utils":32}],32:[function(t,e,a){"use strict";var o=t("./support"),h=t("./base64"),r=t("./nodejsUtils"),i=t("set-immediate-shim"),u=t("./external");function n(t){return t}function l(t,e){for(var r=0;r<t.length;++r)e[r]=255&t.charCodeAt(r);return e}a.newBlob=function(e,r){a.checkSupport("blob");try{return new Blob([e],{type:r})}catch(t){try{var i=new(self.BlobBuilder||self.WebKitBlobBuilder||self.MozBlobBuilder||self.MSBlobBuilder);return i.append(e),i.getBlob(r)}catch(t){throw new Error("Bug : can't construct the Blob.")}}};var s={stringifyByChunk:function(t,e,r){var i=[],n=0,s=t.length;if(s<=r)return String.fromCharCode.apply(null,t);for(;n<s;)"array"===e||"nodebuffer"===e?i.push(String.fromCharCode.apply(null,t.slice(n,Math.min(n+r,s)))):i.push(String.fromCharCode.apply(null,t.subarray(n,Math.min(n+r,s)))),n+=r;return i.join("")},stringifyByChar:function(t){for(var e="",r=0;r<t.length;r++)e+=String.fromCharCode(t[r]);return e},applyCanBeUsed:{uint8array:function(){try{return o.uint8array&&1===String.fromCharCode.apply(null,new Uint8Array(1)).length}catch(t){return!1}}(),nodebuffer:function(){try{return o.nodebuffer&&1===String.fromCharCode.apply(null,r.allocBuffer(1)).length}catch(t){return!1}}()}};function f(t){var e=65536,r=a.getTypeOf(t),i=!0;if("uint8array"===r?i=s.applyCanBeUsed.uint8array:"nodebuffer"===r&&(i=s.applyCanBeUsed.nodebuffer),i)for(;1<e;)try{return s.stringifyByChunk(t,r,e)}catch(t){e=Math.floor(e/2)}return s.stringifyByChar(t)}function d(t,e){for(var r=0;r<t.length;r++)e[r]=t[r];return e}a.applyFromCharCode=f;var c={};c.string={string:n,array:function(t){return l(t,new Array(t.length))},arraybuffer:function(t){return c.string.uint8array(t).buffer},uint8array:function(t){return l(t,new Uint8Array(t.length))},nodebuffer:function(t){return l(t,r.allocBuffer(t.length))}},c.array={string:f,array:n,arraybuffer:function(t){return new Uint8Array(t).buffer},uint8array:function(t){return new Uint8Array(t)},nodebuffer:function(t){return r.newBufferFrom(t)}},c.arraybuffer={string:function(t){return f(new Uint8Array(t))},array:function(t){return d(new Uint8Array(t),new Array(t.byteLength))},arraybuffer:n,uint8array:function(t){return new Uint8Array(t)},nodebuffer:function(t){return r.newBufferFrom(new Uint8Array(t))}},c.uint8array={string:f,array:function(t){return d(t,new Array(t.length))},arraybuffer:function(t){return t.buffer},uint8array:n,nodebuffer:function(t){return r.newBufferFrom(t)}},c.nodebuffer={string:f,array:function(t){return d(t,new Array(t.length))},arraybuffer:function(t){return c.nodebuffer.uint8array(t).buffer},uint8array:function(t){return d(t,new Uint8Array(t.length))},nodebuffer:n},a.transformTo=function(t,e){if(e=e||"",!t)return e;a.checkSupport(t);var r=a.getTypeOf(e);return c[r][t](e)},a.getTypeOf=function(t){return"string"==typeof t?"string":"[object Array]"===Object.prototype.toString.call(t)?"array":o.nodebuffer&&r.isBuffer(t)?"nodebuffer":o.uint8array&&t instanceof Uint8Array?"uint8array":o.arraybuffer&&t instanceof ArrayBuffer?"arraybuffer":void 0},a.checkSupport=function(t){if(!o[t.toLowerCase()])throw new Error(t+" is not supported by this platform")},a.MAX_VALUE_16BITS=65535,a.MAX_VALUE_32BITS=-1,a.pretty=function(t){var e,r,i="";for(r=0;r<(t||"").length;r++)i+="\\x"+((e=t.charCodeAt(r))<16?"0":"")+e.toString(16).toUpperCase();return i},a.delay=function(t,e,r){i(function(){t.apply(r||null,e||[])})},a.inherits=function(t,e){function r(){}r.prototype=e.prototype,t.prototype=new r},a.extend=function(){var t,e,r={};for(t=0;t<arguments.length;t++)for(e in arguments[t])arguments[t].hasOwnProperty(e)&&void 0===r[e]&&(r[e]=arguments[t][e]);return r},a.prepareContent=function(r,t,i,n,s){return u.Promise.resolve(t).then(function(i){return o.blob&&(i instanceof Blob||-1!==["[object File]","[object Blob]"].indexOf(Object.prototype.toString.call(i)))&&"undefined"!=typeof FileReader?new u.Promise(function(e,r){var t=new FileReader;t.onload=function(t){e(t.target.result)},t.onerror=function(t){r(t.target.error)},t.readAsArrayBuffer(i)}):i}).then(function(t){var e=a.getTypeOf(t);return e?("arraybuffer"===e?t=a.transformTo("uint8array",t):"string"===e&&(s?t=h.decode(t):i&&!0!==n&&(t=function(t){return l(t,o.uint8array?new Uint8Array(t.length):new Array(t.length))}(t))),t):u.Promise.reject(new Error("Can't read the data of '"+r+"'. Is it in a supported JavaScript type (String, Blob, ArrayBuffer, etc) ?"))})}},{"./base64":1,"./external":6,"./nodejsUtils":14,"./support":30,"set-immediate-shim":54}],33:[function(t,e,r){"use strict";var i=t("./reader/readerFor"),n=t("./utils"),s=t("./signature"),a=t("./zipEntry"),o=(t("./utf8"),t("./support"));function h(t){this.files=[],this.loadOptions=t}h.prototype={checkSignature:function(t){if(!this.reader.readAndCheckSignature(t)){this.reader.index-=4;var e=this.reader.readString(4);throw new Error("Corrupted zip or bug: unexpected signature ("+n.pretty(e)+", expected "+n.pretty(t)+")")}},isSignature:function(t,e){var r=this.reader.index;this.reader.setIndex(t);var i=this.reader.readString(4)===e;return this.reader.setIndex(r),i},readBlockEndOfCentral:function(){this.diskNumber=this.reader.readInt(2),this.diskWithCentralDirStart=this.reader.readInt(2),this.centralDirRecordsOnThisDisk=this.reader.readInt(2),this.centralDirRecords=this.reader.readInt(2),this.centralDirSize=this.reader.readInt(4),this.centralDirOffset=this.reader.readInt(4),this.zipCommentLength=this.reader.readInt(2);var t=this.reader.readData(this.zipCommentLength),e=o.uint8array?"uint8array":"array",r=n.transformTo(e,t);this.zipComment=this.loadOptions.decodeFileName(r)},readBlockZip64EndOfCentral:function(){this.zip64EndOfCentralSize=this.reader.readInt(8),this.reader.skip(4),this.diskNumber=this.reader.readInt(4),this.diskWithCentralDirStart=this.reader.readInt(4),this.centralDirRecordsOnThisDisk=this.reader.readInt(8),this.centralDirRecords=this.reader.readInt(8),this.centralDirSize=this.reader.readInt(8),this.centralDirOffset=this.reader.readInt(8),this.zip64ExtensibleData={};for(var t,e,r,i=this.zip64EndOfCentralSize-44;0<i;)t=this.reader.readInt(2),e=this.reader.readInt(4),r=this.reader.readData(e),this.zip64ExtensibleData[t]={id:t,length:e,value:r}},readBlockZip64EndOfCentralLocator:function(){if(this.diskWithZip64CentralDirStart=this.reader.readInt(4),this.relativeOffsetEndOfZip64CentralDir=this.reader.readInt(8),this.disksCount=this.reader.readInt(4),1<this.disksCount)throw new Error("Multi-volumes zip are not supported")},readLocalFiles:function(){var t,e;for(t=0;t<this.files.length;t++)e=this.files[t],this.reader.setIndex(e.localHeaderOffset),this.checkSignature(s.LOCAL_FILE_HEADER),e.readLocalPart(this.reader),e.handleUTF8(),e.processAttributes()},readCentralDir:function(){var t;for(this.reader.setIndex(this.centralDirOffset);this.reader.readAndCheckSignature(s.CENTRAL_FILE_HEADER);)(t=new a({zip64:this.zip64},this.loadOptions)).readCentralPart(this.reader),this.files.push(t);if(this.centralDirRecords!==this.files.length&&0!==this.centralDirRecords&&0===this.files.length)throw new Error("Corrupted zip or bug: expected "+this.centralDirRecords+" records in central dir, got "+this.files.length)},readEndOfCentral:function(){var t=this.reader.lastIndexOfSignature(s.CENTRAL_DIRECTORY_END);if(t<0)throw!this.isSignature(0,s.LOCAL_FILE_HEADER)?new Error("Can't find end of central directory : is this a zip file ? If it is, see https://stuk.github.io/jszip/documentation/howto/read_zip.html"):new Error("Corrupted zip: can't find end of central directory");this.reader.setIndex(t);var e=t;if(this.checkSignature(s.CENTRAL_DIRECTORY_END),this.readBlockEndOfCentral(),this.diskNumber===n.MAX_VALUE_16BITS||this.diskWithCentralDirStart===n.MAX_VALUE_16BITS||this.centralDirRecordsOnThisDisk===n.MAX_VALUE_16BITS||this.centralDirRecords===n.MAX_VALUE_16BITS||this.centralDirSize===n.MAX_VALUE_32BITS||this.centralDirOffset===n.MAX_VALUE_32BITS){if(this.zip64=!0,(t=this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR))<0)throw new Error("Corrupted zip: can't find the ZIP64 end of central directory locator");if(this.reader.setIndex(t),this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR),this.readBlockZip64EndOfCentralLocator(),!this.isSignature(this.relativeOffsetEndOfZip64CentralDir,s.ZIP64_CENTRAL_DIRECTORY_END)&&(this.relativeOffsetEndOfZip64CentralDir=this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_END),this.relativeOffsetEndOfZip64CentralDir<0))throw new Error("Corrupted zip: can't find the ZIP64 end of central directory");this.reader.setIndex(this.relativeOffsetEndOfZip64CentralDir),this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_END),this.readBlockZip64EndOfCentral()}var r=this.centralDirOffset+this.centralDirSize;this.zip64&&(r+=20,r+=12+this.zip64EndOfCentralSize);var i=e-r;if(0<i)this.isSignature(e,s.CENTRAL_FILE_HEADER)||(this.reader.zero=i);else if(i<0)throw new Error("Corrupted zip: missing "+Math.abs(i)+" bytes.")},prepareReader:function(t){this.reader=i(t)},load:function(t){this.prepareReader(t),this.readEndOfCentral(),this.readCentralDir(),this.readLocalFiles()}},e.exports=h},{"./reader/readerFor":22,"./signature":23,"./support":30,"./utf8":31,"./utils":32,"./zipEntry":34}],34:[function(t,e,r){"use strict";var i=t("./reader/readerFor"),s=t("./utils"),n=t("./compressedObject"),a=t("./crc32"),o=t("./utf8"),h=t("./compressions"),u=t("./support");function l(t,e){this.options=t,this.loadOptions=e}l.prototype={isEncrypted:function(){return 1==(1&this.bitFlag)},useUTF8:function(){return 2048==(2048&this.bitFlag)},readLocalPart:function(t){var e,r;if(t.skip(22),this.fileNameLength=t.readInt(2),r=t.readInt(2),this.fileName=t.readData(this.fileNameLength),t.skip(r),-1===this.compressedSize||-1===this.uncompressedSize)throw new Error("Bug or corrupted zip : didn't get enough information from the central directory (compressedSize === -1 || uncompressedSize === -1)");if(null===(e=function(t){for(var e in h)if(h.hasOwnProperty(e)&&h[e].magic===t)return h[e];return null}(this.compressionMethod)))throw new Error("Corrupted zip : compression "+s.pretty(this.compressionMethod)+" unknown (inner file : "+s.transformTo("string",this.fileName)+")");this.decompressed=new n(this.compressedSize,this.uncompressedSize,this.crc32,e,t.readData(this.compressedSize))},readCentralPart:function(t){this.versionMadeBy=t.readInt(2),t.skip(2),this.bitFlag=t.readInt(2),this.compressionMethod=t.readString(2),this.date=t.readDate(),this.crc32=t.readInt(4),this.compressedSize=t.readInt(4),this.uncompressedSize=t.readInt(4);var e=t.readInt(2);if(this.extraFieldsLength=t.readInt(2),this.fileCommentLength=t.readInt(2),this.diskNumberStart=t.readInt(2),this.internalFileAttributes=t.readInt(2),this.externalFileAttributes=t.readInt(4),this.localHeaderOffset=t.readInt(4),this.isEncrypted())throw new Error("Encrypted zip are not supported");t.skip(e),this.readExtraFields(t),this.parseZIP64ExtraField(t),this.fileComment=t.readData(this.fileCommentLength)},processAttributes:function(){this.unixPermissions=null,this.dosPermissions=null;var t=this.versionMadeBy>>8;this.dir=!!(16&this.externalFileAttributes),0==t&&(this.dosPermissions=63&this.externalFileAttributes),3==t&&(this.unixPermissions=this.externalFileAttributes>>16&65535),this.dir||"/"!==this.fileNameStr.slice(-1)||(this.dir=!0)},parseZIP64ExtraField:function(t){if(this.extraFields[1]){var e=i(this.extraFields[1].value);this.uncompressedSize===s.MAX_VALUE_32BITS&&(this.uncompressedSize=e.readInt(8)),this.compressedSize===s.MAX_VALUE_32BITS&&(this.compressedSize=e.readInt(8)),this.localHeaderOffset===s.MAX_VALUE_32BITS&&(this.localHeaderOffset=e.readInt(8)),this.diskNumberStart===s.MAX_VALUE_32BITS&&(this.diskNumberStart=e.readInt(4))}},readExtraFields:function(t){var e,r,i,n=t.index+this.extraFieldsLength;for(this.extraFields||(this.extraFields={});t.index+4<n;)e=t.readInt(2),r=t.readInt(2),i=t.readData(r),this.extraFields[e]={id:e,length:r,value:i};t.setIndex(n)},handleUTF8:function(){var t=u.uint8array?"uint8array":"array";if(this.useUTF8())this.fileNameStr=o.utf8decode(this.fileName),this.fileCommentStr=o.utf8decode(this.fileComment);else{var e=this.findExtraFieldUnicodePath();if(null!==e)this.fileNameStr=e;else{var r=s.transformTo(t,this.fileName);this.fileNameStr=this.loadOptions.decodeFileName(r)}var i=this.findExtraFieldUnicodeComment();if(null!==i)this.fileCommentStr=i;else{var n=s.transformTo(t,this.fileComment);this.fileCommentStr=this.loadOptions.decodeFileName(n)}}},findExtraFieldUnicodePath:function(){var t=this.extraFields[28789];if(t){var e=i(t.value);return 1!==e.readInt(1)?null:a(this.fileName)!==e.readInt(4)?null:o.utf8decode(e.readData(t.length-5))}return null},findExtraFieldUnicodeComment:function(){var t=this.extraFields[25461];if(t){var e=i(t.value);return 1!==e.readInt(1)?null:a(this.fileComment)!==e.readInt(4)?null:o.utf8decode(e.readData(t.length-5))}return null}},e.exports=l},{"./compressedObject":2,"./compressions":3,"./crc32":4,"./reader/readerFor":22,"./support":30,"./utf8":31,"./utils":32}],35:[function(t,e,r){"use strict";function i(t,e,r){this.name=t,this.dir=r.dir,this.date=r.date,this.comment=r.comment,this.unixPermissions=r.unixPermissions,this.dosPermissions=r.dosPermissions,this._data=e,this._dataBinary=r.binary,this.options={compression:r.compression,compressionOptions:r.compressionOptions}}var s=t("./stream/StreamHelper"),n=t("./stream/DataWorker"),a=t("./utf8"),o=t("./compressedObject"),h=t("./stream/GenericWorker");i.prototype={internalStream:function(t){var e=null,r="string";try{if(!t)throw new Error("No output type specified.");var i="string"===(r=t.toLowerCase())||"text"===r;"binarystring"!==r&&"text"!==r||(r="string"),e=this._decompressWorker();var n=!this._dataBinary;n&&!i&&(e=e.pipe(new a.Utf8EncodeWorker)),!n&&i&&(e=e.pipe(new a.Utf8DecodeWorker))}catch(t){(e=new h("error")).error(t)}return new s(e,r,"")},async:function(t,e){return this.internalStream(t).accumulate(e)},nodeStream:function(t,e){return this.internalStream(t||"nodebuffer").toNodejsStream(e)},_compressWorker:function(t,e){if(this._data instanceof o&&this._data.compression.magic===t.magic)return this._data.getCompressedWorker();var r=this._decompressWorker();return this._dataBinary||(r=r.pipe(new a.Utf8EncodeWorker)),o.createWorkerFrom(r,t,e)},_decompressWorker:function(){return this._data instanceof o?this._data.getContentWorker():this._data instanceof h?this._data:new n(this._data)}};for(var u=["asText","asBinary","asNodeBuffer","asUint8Array","asArrayBuffer"],l=function(){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},f=0;f<u.length;f++)i.prototype[u[f]]=l;e.exports=i},{"./compressedObject":2,"./stream/DataWorker":27,"./stream/GenericWorker":28,"./stream/StreamHelper":29,"./utf8":31}],36:[function(t,l,e){(function(e){"use strict";var r,i,t=e.MutationObserver||e.WebKitMutationObserver;if(t){var n=0,s=new t(u),a=e.document.createTextNode("");s.observe(a,{characterData:!0}),r=function(){a.data=n=++n%2}}else if(e.setImmediate||void 0===e.MessageChannel)r="document"in e&&"onreadystatechange"in e.document.createElement("script")?function(){var t=e.document.createElement("script");t.onreadystatechange=function(){u(),t.onreadystatechange=null,t.parentNode.removeChild(t),t=null},e.document.documentElement.appendChild(t)}:function(){setTimeout(u,0)};else{var o=new e.MessageChannel;o.port1.onmessage=u,r=function(){o.port2.postMessage(0)}}var h=[];function u(){var t,e;i=!0;for(var r=h.length;r;){for(e=h,h=[],t=-1;++t<r;)e[t]();r=h.length}i=!1}l.exports=function(t){1!==h.push(t)||i||r()}}).call(this,"undefined"!=typeof global?global:"undefined"!=typeof self?self:"undefined"!=typeof window?window:{})},{}],37:[function(t,e,r){"use strict";var n=t("immediate");function u(){}var l={},s=["REJECTED"],a=["FULFILLED"],i=["PENDING"];function o(t){if("function"!=typeof t)throw new TypeError("resolver must be a function");this.state=i,this.queue=[],this.outcome=void 0,t!==u&&c(this,t)}function h(t,e,r){this.promise=t,"function"==typeof e&&(this.onFulfilled=e,this.callFulfilled=this.otherCallFulfilled),"function"==typeof r&&(this.onRejected=r,this.callRejected=this.otherCallRejected)}function f(e,r,i){n(function(){var t;try{t=r(i)}catch(t){return l.reject(e,t)}t===e?l.reject(e,new TypeError("Cannot resolve promise with itself")):l.resolve(e,t)})}function d(t){var e=t&&t.then;if(t&&("object"==typeof t||"function"==typeof t)&&"function"==typeof e)return function(){e.apply(t,arguments)}}function c(e,t){var r=!1;function i(t){r||(r=!0,l.reject(e,t))}function n(t){r||(r=!0,l.resolve(e,t))}var s=p(function(){t(n,i)});"error"===s.status&&i(s.value)}function p(t,e){var r={};try{r.value=t(e),r.status="success"}catch(t){r.status="error",r.value=t}return r}(e.exports=o).prototype.finally=function(e){if("function"!=typeof e)return this;var r=this.constructor;return this.then(function(t){return r.resolve(e()).then(function(){return t})},function(t){return r.resolve(e()).then(function(){throw t})})},o.prototype.catch=function(t){return this.then(null,t)},o.prototype.then=function(t,e){if("function"!=typeof t&&this.state===a||"function"!=typeof e&&this.state===s)return this;var r=new this.constructor(u);this.state!==i?f(r,this.state===a?t:e,this.outcome):this.queue.push(new h(r,t,e));return r},h.prototype.callFulfilled=function(t){l.resolve(this.promise,t)},h.prototype.otherCallFulfilled=function(t){f(this.promise,this.onFulfilled,t)},h.prototype.callRejected=function(t){l.reject(this.promise,t)},h.prototype.otherCallRejected=function(t){f(this.promise,this.onRejected,t)},l.resolve=function(t,e){var r=p(d,e);if("error"===r.status)return l.reject(t,r.value);var i=r.value;if(i)c(t,i);else{t.state=a,t.outcome=e;for(var n=-1,s=t.queue.length;++n<s;)t.queue[n].callFulfilled(e)}return t},l.reject=function(t,e){t.state=s,t.outcome=e;for(var r=-1,i=t.queue.length;++r<i;)t.queue[r].callRejected(e);return t},o.resolve=function(t){if(t instanceof this)return t;return l.resolve(new this(u),t)},o.reject=function(t){var e=new this(u);return l.reject(e,t)},o.all=function(t){var r=this;if("[object Array]"!==Object.prototype.toString.call(t))return this.reject(new TypeError("must be an array"));var i=t.length,n=!1;if(!i)return this.resolve([]);var s=new Array(i),a=0,e=-1,o=new this(u);for(;++e<i;)h(t[e],e);return o;function h(t,e){r.resolve(t).then(function(t){s[e]=t,++a!==i||n||(n=!0,l.resolve(o,s))},function(t){n||(n=!0,l.reject(o,t))})}},o.race=function(t){var e=this;if("[object Array]"!==Object.prototype.toString.call(t))return this.reject(new TypeError("must be an array"));var r=t.length,i=!1;if(!r)return this.resolve([]);var n=-1,s=new this(u);for(;++n<r;)a=t[n],e.resolve(a).then(function(t){i||(i=!0,l.resolve(s,t))},function(t){i||(i=!0,l.reject(s,t))});var a;return s}},{immediate:36}],38:[function(t,e,r){"use strict";var i={};(0,t("./lib/utils/common").assign)(i,t("./lib/deflate"),t("./lib/inflate"),t("./lib/zlib/constants")),e.exports=i},{"./lib/deflate":39,"./lib/inflate":40,"./lib/utils/common":41,"./lib/zlib/constants":44}],39:[function(t,e,r){"use strict";var a=t("./zlib/deflate"),o=t("./utils/common"),h=t("./utils/strings"),n=t("./zlib/messages"),s=t("./zlib/zstream"),u=Object.prototype.toString,l=0,f=-1,d=0,c=8;function p(t){if(!(this instanceof p))return new p(t);this.options=o.assign({level:f,method:c,chunkSize:16384,windowBits:15,memLevel:8,strategy:d,to:""},t||{});var e=this.options;e.raw&&0<e.windowBits?e.windowBits=-e.windowBits:e.gzip&&0<e.windowBits&&e.windowBits<16&&(e.windowBits+=16),this.err=0,this.msg="",this.ended=!1,this.chunks=[],this.strm=new s,this.strm.avail_out=0;var r=a.deflateInit2(this.strm,e.level,e.method,e.windowBits,e.memLevel,e.strategy);if(r!==l)throw new Error(n[r]);if(e.header&&a.deflateSetHeader(this.strm,e.header),e.dictionary){var i;if(i="string"==typeof e.dictionary?h.string2buf(e.dictionary):"[object ArrayBuffer]"===u.call(e.dictionary)?new Uint8Array(e.dictionary):e.dictionary,(r=a.deflateSetDictionary(this.strm,i))!==l)throw new Error(n[r]);this._dict_set=!0}}function i(t,e){var r=new p(e);if(r.push(t,!0),r.err)throw r.msg||n[r.err];return r.result}p.prototype.push=function(t,e){var r,i,n=this.strm,s=this.options.chunkSize;if(this.ended)return!1;i=e===~~e?e:!0===e?4:0,"string"==typeof t?n.input=h.string2buf(t):"[object ArrayBuffer]"===u.call(t)?n.input=new Uint8Array(t):n.input=t,n.next_in=0,n.avail_in=n.input.length;do{if(0===n.avail_out&&(n.output=new o.Buf8(s),n.next_out=0,n.avail_out=s),1!==(r=a.deflate(n,i))&&r!==l)return this.onEnd(r),!(this.ended=!0);0!==n.avail_out&&(0!==n.avail_in||4!==i&&2!==i)||("string"===this.options.to?this.onData(h.buf2binstring(o.shrinkBuf(n.output,n.next_out))):this.onData(o.shrinkBuf(n.output,n.next_out)))}while((0<n.avail_in||0===n.avail_out)&&1!==r);return 4===i?(r=a.deflateEnd(this.strm),this.onEnd(r),this.ended=!0,r===l):2!==i||(this.onEnd(l),!(n.avail_out=0))},p.prototype.onData=function(t){this.chunks.push(t)},p.prototype.onEnd=function(t){t===l&&("string"===this.options.to?this.result=this.chunks.join(""):this.result=o.flattenChunks(this.chunks)),this.chunks=[],this.err=t,this.msg=this.strm.msg},r.Deflate=p,r.deflate=i,r.deflateRaw=function(t,e){return(e=e||{}).raw=!0,i(t,e)},r.gzip=function(t,e){return(e=e||{}).gzip=!0,i(t,e)}},{"./utils/common":41,"./utils/strings":42,"./zlib/deflate":46,"./zlib/messages":51,"./zlib/zstream":53}],40:[function(t,e,r){"use strict";var d=t("./zlib/inflate"),c=t("./utils/common"),p=t("./utils/strings"),m=t("./zlib/constants"),i=t("./zlib/messages"),n=t("./zlib/zstream"),s=t("./zlib/gzheader"),_=Object.prototype.toString;function a(t){if(!(this instanceof a))return new a(t);this.options=c.assign({chunkSize:16384,windowBits:0,to:""},t||{});var e=this.options;e.raw&&0<=e.windowBits&&e.windowBits<16&&(e.windowBits=-e.windowBits,0===e.windowBits&&(e.windowBits=-15)),!(0<=e.windowBits&&e.windowBits<16)||t&&t.windowBits||(e.windowBits+=32),15<e.windowBits&&e.windowBits<48&&0==(15&e.windowBits)&&(e.windowBits|=15),this.err=0,this.msg="",this.ended=!1,this.chunks=[],this.strm=new n,this.strm.avail_out=0;var r=d.inflateInit2(this.strm,e.windowBits);if(r!==m.Z_OK)throw new Error(i[r]);this.header=new s,d.inflateGetHeader(this.strm,this.header)}function o(t,e){var r=new a(e);if(r.push(t,!0),r.err)throw r.msg||i[r.err];return r.result}a.prototype.push=function(t,e){var r,i,n,s,a,o,h=this.strm,u=this.options.chunkSize,l=this.options.dictionary,f=!1;if(this.ended)return!1;i=e===~~e?e:!0===e?m.Z_FINISH:m.Z_NO_FLUSH,"string"==typeof t?h.input=p.binstring2buf(t):"[object ArrayBuffer]"===_.call(t)?h.input=new Uint8Array(t):h.input=t,h.next_in=0,h.avail_in=h.input.length;do{if(0===h.avail_out&&(h.output=new c.Buf8(u),h.next_out=0,h.avail_out=u),(r=d.inflate(h,m.Z_NO_FLUSH))===m.Z_NEED_DICT&&l&&(o="string"==typeof l?p.string2buf(l):"[object ArrayBuffer]"===_.call(l)?new Uint8Array(l):l,r=d.inflateSetDictionary(this.strm,o)),r===m.Z_BUF_ERROR&&!0===f&&(r=m.Z_OK,f=!1),r!==m.Z_STREAM_END&&r!==m.Z_OK)return this.onEnd(r),!(this.ended=!0);h.next_out&&(0!==h.avail_out&&r!==m.Z_STREAM_END&&(0!==h.avail_in||i!==m.Z_FINISH&&i!==m.Z_SYNC_FLUSH)||("string"===this.options.to?(n=p.utf8border(h.output,h.next_out),s=h.next_out-n,a=p.buf2string(h.output,n),h.next_out=s,h.avail_out=u-s,s&&c.arraySet(h.output,h.output,n,s,0),this.onData(a)):this.onData(c.shrinkBuf(h.output,h.next_out)))),0===h.avail_in&&0===h.avail_out&&(f=!0)}while((0<h.avail_in||0===h.avail_out)&&r!==m.Z_STREAM_END);return r===m.Z_STREAM_END&&(i=m.Z_FINISH),i===m.Z_FINISH?(r=d.inflateEnd(this.strm),this.onEnd(r),this.ended=!0,r===m.Z_OK):i!==m.Z_SYNC_FLUSH||(this.onEnd(m.Z_OK),!(h.avail_out=0))},a.prototype.onData=function(t){this.chunks.push(t)},a.prototype.onEnd=function(t){t===m.Z_OK&&("string"===this.options.to?this.result=this.chunks.join(""):this.result=c.flattenChunks(this.chunks)),this.chunks=[],this.err=t,this.msg=this.strm.msg},r.Inflate=a,r.inflate=o,r.inflateRaw=function(t,e){return(e=e||{}).raw=!0,o(t,e)},r.ungzip=o},{"./utils/common":41,"./utils/strings":42,"./zlib/constants":44,"./zlib/gzheader":47,"./zlib/inflate":49,"./zlib/messages":51,"./zlib/zstream":53}],41:[function(t,e,r){"use strict";var i="undefined"!=typeof Uint8Array&&"undefined"!=typeof Uint16Array&&"undefined"!=typeof Int32Array;r.assign=function(t){for(var e=Array.prototype.slice.call(arguments,1);e.length;){var r=e.shift();if(r){if("object"!=typeof r)throw new TypeError(r+"must be non-object");for(var i in r)r.hasOwnProperty(i)&&(t[i]=r[i])}}return t},r.shrinkBuf=function(t,e){return t.length===e?t:t.subarray?t.subarray(0,e):(t.length=e,t)};var n={arraySet:function(t,e,r,i,n){if(e.subarray&&t.subarray)t.set(e.subarray(r,r+i),n);else for(var s=0;s<i;s++)t[n+s]=e[r+s]},flattenChunks:function(t){var e,r,i,n,s,a;for(e=i=0,r=t.length;e<r;e++)i+=t[e].length;for(a=new Uint8Array(i),e=n=0,r=t.length;e<r;e++)s=t[e],a.set(s,n),n+=s.length;return a}},s={arraySet:function(t,e,r,i,n){for(var s=0;s<i;s++)t[n+s]=e[r+s]},flattenChunks:function(t){return[].concat.apply([],t)}};r.setTyped=function(t){t?(r.Buf8=Uint8Array,r.Buf16=Uint16Array,r.Buf32=Int32Array,r.assign(r,n)):(r.Buf8=Array,r.Buf16=Array,r.Buf32=Array,r.assign(r,s))},r.setTyped(i)},{}],42:[function(t,e,r){"use strict";var h=t("./common"),n=!0,s=!0;try{String.fromCharCode.apply(null,[0])}catch(t){n=!1}try{String.fromCharCode.apply(null,new Uint8Array(1))}catch(t){s=!1}for(var u=new h.Buf8(256),i=0;i<256;i++)u[i]=252<=i?6:248<=i?5:240<=i?4:224<=i?3:192<=i?2:1;function l(t,e){if(e<65537&&(t.subarray&&s||!t.subarray&&n))return String.fromCharCode.apply(null,h.shrinkBuf(t,e));for(var r="",i=0;i<e;i++)r+=String.fromCharCode(t[i]);return r}u[254]=u[254]=1,r.string2buf=function(t){var e,r,i,n,s,a=t.length,o=0;for(n=0;n<a;n++)55296==(64512&(r=t.charCodeAt(n)))&&n+1<a&&56320==(64512&(i=t.charCodeAt(n+1)))&&(r=65536+(r-55296<<10)+(i-56320),n++),o+=r<128?1:r<2048?2:r<65536?3:4;for(e=new h.Buf8(o),n=s=0;s<o;n++)55296==(64512&(r=t.charCodeAt(n)))&&n+1<a&&56320==(64512&(i=t.charCodeAt(n+1)))&&(r=65536+(r-55296<<10)+(i-56320),n++),r<128?e[s++]=r:(r<2048?e[s++]=192|r>>>6:(r<65536?e[s++]=224|r>>>12:(e[s++]=240|r>>>18,e[s++]=128|r>>>12&63),e[s++]=128|r>>>6&63),e[s++]=128|63&r);return e},r.buf2binstring=function(t){return l(t,t.length)},r.binstring2buf=function(t){for(var e=new h.Buf8(t.length),r=0,i=e.length;r<i;r++)e[r]=t.charCodeAt(r);return e},r.buf2string=function(t,e){var r,i,n,s,a=e||t.length,o=new Array(2*a);for(r=i=0;r<a;)if((n=t[r++])<128)o[i++]=n;else if(4<(s=u[n]))o[i++]=65533,r+=s-1;else{for(n&=2===s?31:3===s?15:7;1<s&&r<a;)n=n<<6|63&t[r++],s--;1<s?o[i++]=65533:n<65536?o[i++]=n:(n-=65536,o[i++]=55296|n>>10&1023,o[i++]=56320|1023&n)}return l(o,i)},r.utf8border=function(t,e){var r;for((e=e||t.length)>t.length&&(e=t.length),r=e-1;0<=r&&128==(192&t[r]);)r--;return r<0?e:0===r?e:r+u[t[r]]>e?r:e}},{"./common":41}],43:[function(t,e,r){"use strict";e.exports=function(t,e,r,i){for(var n=65535&t|0,s=t>>>16&65535|0,a=0;0!==r;){for(r-=a=2e3<r?2e3:r;s=s+(n=n+e[i++]|0)|0,--a;);n%=65521,s%=65521}return n|s<<16|0}},{}],44:[function(t,e,r){"use strict";e.exports={Z_NO_FLUSH:0,Z_PARTIAL_FLUSH:1,Z_SYNC_FLUSH:2,Z_FULL_FLUSH:3,Z_FINISH:4,Z_BLOCK:5,Z_TREES:6,Z_OK:0,Z_STREAM_END:1,Z_NEED_DICT:2,Z_ERRNO:-1,Z_STREAM_ERROR:-2,Z_DATA_ERROR:-3,Z_BUF_ERROR:-5,Z_NO_COMPRESSION:0,Z_BEST_SPEED:1,Z_BEST_COMPRESSION:9,Z_DEFAULT_COMPRESSION:-1,Z_FILTERED:1,Z_HUFFMAN_ONLY:2,Z_RLE:3,Z_FIXED:4,Z_DEFAULT_STRATEGY:0,Z_BINARY:0,Z_TEXT:1,Z_UNKNOWN:2,Z_DEFLATED:8}},{}],45:[function(t,e,r){"use strict";var o=function(){for(var t,e=[],r=0;r<256;r++){t=r;for(var i=0;i<8;i++)t=1&t?3988292384^t>>>1:t>>>1;e[r]=t}return e}();e.exports=function(t,e,r,i){var n=o,s=i+r;t^=-1;for(var a=i;a<s;a++)t=t>>>8^n[255&(t^e[a])];return-1^t}},{}],46:[function(t,e,r){"use strict";var h,d=t("../utils/common"),u=t("./trees"),c=t("./adler32"),p=t("./crc32"),i=t("./messages"),l=0,f=4,m=0,_=-2,g=-1,b=4,n=2,v=8,y=9,s=286,a=30,o=19,w=2*s+1,k=15,x=3,S=258,z=S+x+1,C=42,E=113,A=1,I=2,O=3,B=4;function R(t,e){return t.msg=i[e],e}function T(t){return(t<<1)-(4<t?9:0)}function D(t){for(var e=t.length;0<=--e;)t[e]=0}function F(t){var e=t.state,r=e.pending;r>t.avail_out&&(r=t.avail_out),0!==r&&(d.arraySet(t.output,e.pending_buf,e.pending_out,r,t.next_out),t.next_out+=r,e.pending_out+=r,t.total_out+=r,t.avail_out-=r,e.pending-=r,0===e.pending&&(e.pending_out=0))}function N(t,e){u._tr_flush_block(t,0<=t.block_start?t.block_start:-1,t.strstart-t.block_start,e),t.block_start=t.strstart,F(t.strm)}function U(t,e){t.pending_buf[t.pending++]=e}function P(t,e){t.pending_buf[t.pending++]=e>>>8&255,t.pending_buf[t.pending++]=255&e}function L(t,e){var r,i,n=t.max_chain_length,s=t.strstart,a=t.prev_length,o=t.nice_match,h=t.strstart>t.w_size-z?t.strstart-(t.w_size-z):0,u=t.window,l=t.w_mask,f=t.prev,d=t.strstart+S,c=u[s+a-1],p=u[s+a];t.prev_length>=t.good_match&&(n>>=2),o>t.lookahead&&(o=t.lookahead);do{if(u[(r=e)+a]===p&&u[r+a-1]===c&&u[r]===u[s]&&u[++r]===u[s+1]){s+=2,r++;do{}while(u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&u[++s]===u[++r]&&s<d);if(i=S-(d-s),s=d-S,a<i){if(t.match_start=e,o<=(a=i))break;c=u[s+a-1],p=u[s+a]}}}while((e=f[e&l])>h&&0!=--n);return a<=t.lookahead?a:t.lookahead}function j(t){var e,r,i,n,s,a,o,h,u,l,f=t.w_size;do{if(n=t.window_size-t.lookahead-t.strstart,t.strstart>=f+(f-z)){for(d.arraySet(t.window,t.window,f,f,0),t.match_start-=f,t.strstart-=f,t.block_start-=f,e=r=t.hash_size;i=t.head[--e],t.head[e]=f<=i?i-f:0,--r;);for(e=r=f;i=t.prev[--e],t.prev[e]=f<=i?i-f:0,--r;);n+=f}if(0===t.strm.avail_in)break;if(a=t.strm,o=t.window,h=t.strstart+t.lookahead,u=n,l=void 0,l=a.avail_in,u<l&&(l=u),r=0===l?0:(a.avail_in-=l,d.arraySet(o,a.input,a.next_in,l,h),1===a.state.wrap?a.adler=c(a.adler,o,l,h):2===a.state.wrap&&(a.adler=p(a.adler,o,l,h)),a.next_in+=l,a.total_in+=l,l),t.lookahead+=r,t.lookahead+t.insert>=x)for(s=t.strstart-t.insert,t.ins_h=t.window[s],t.ins_h=(t.ins_h<<t.hash_shift^t.window[s+1])&t.hash_mask;t.insert&&(t.ins_h=(t.ins_h<<t.hash_shift^t.window[s+x-1])&t.hash_mask,t.prev[s&t.w_mask]=t.head[t.ins_h],t.head[t.ins_h]=s,s++,t.insert--,!(t.lookahead+t.insert<x)););}while(t.lookahead<z&&0!==t.strm.avail_in)}function Z(t,e){for(var r,i;;){if(t.lookahead<z){if(j(t),t.lookahead<z&&e===l)return A;if(0===t.lookahead)break}if(r=0,t.lookahead>=x&&(t.ins_h=(t.ins_h<<t.hash_shift^t.window[t.strstart+x-1])&t.hash_mask,r=t.prev[t.strstart&t.w_mask]=t.head[t.ins_h],t.head[t.ins_h]=t.strstart),0!==r&&t.strstart-r<=t.w_size-z&&(t.match_length=L(t,r)),t.match_length>=x)if(i=u._tr_tally(t,t.strstart-t.match_start,t.match_length-x),t.lookahead-=t.match_length,t.match_length<=t.max_lazy_match&&t.lookahead>=x){for(t.match_length--;t.strstart++,t.ins_h=(t.ins_h<<t.hash_shift^t.window[t.strstart+x-1])&t.hash_mask,r=t.prev[t.strstart&t.w_mask]=t.head[t.ins_h],t.head[t.ins_h]=t.strstart,0!=--t.match_length;);t.strstart++}else t.strstart+=t.match_length,t.match_length=0,t.ins_h=t.window[t.strstart],t.ins_h=(t.ins_h<<t.hash_shift^t.window[t.strstart+1])&t.hash_mask;else i=u._tr_tally(t,0,t.window[t.strstart]),t.lookahead--,t.strstart++;if(i&&(N(t,!1),0===t.strm.avail_out))return A}return t.insert=t.strstart<x-1?t.strstart:x-1,e===f?(N(t,!0),0===t.strm.avail_out?O:B):t.last_lit&&(N(t,!1),0===t.strm.avail_out)?A:I}function W(t,e){for(var r,i,n;;){if(t.lookahead<z){if(j(t),t.lookahead<z&&e===l)return A;if(0===t.lookahead)break}if(r=0,t.lookahead>=x&&(t.ins_h=(t.ins_h<<t.hash_shift^t.window[t.strstart+x-1])&t.hash_mask,r=t.prev[t.strstart&t.w_mask]=t.head[t.ins_h],t.head[t.ins_h]=t.strstart),t.prev_length=t.match_length,t.prev_match=t.match_start,t.match_length=x-1,0!==r&&t.prev_length<t.max_lazy_match&&t.strstart-r<=t.w_size-z&&(t.match_length=L(t,r),t.match_length<=5&&(1===t.strategy||t.match_length===x&&4096<t.strstart-t.match_start)&&(t.match_length=x-1)),t.prev_length>=x&&t.match_length<=t.prev_length){for(n=t.strstart+t.lookahead-x,i=u._tr_tally(t,t.strstart-1-t.prev_match,t.prev_length-x),t.lookahead-=t.prev_length-1,t.prev_length-=2;++t.strstart<=n&&(t.ins_h=(t.ins_h<<t.hash_shift^t.window[t.strstart+x-1])&t.hash_mask,r=t.prev[t.strstart&t.w_mask]=t.head[t.ins_h],t.head[t.ins_h]=t.strstart),0!=--t.prev_length;);if(t.match_available=0,t.match_length=x-1,t.strstart++,i&&(N(t,!1),0===t.strm.avail_out))return A}else if(t.match_available){if((i=u._tr_tally(t,0,t.window[t.strstart-1]))&&N(t,!1),t.strstart++,t.lookahead--,0===t.strm.avail_out)return A}else t.match_available=1,t.strstart++,t.lookahead--}return t.match_available&&(i=u._tr_tally(t,0,t.window[t.strstart-1]),t.match_available=0),t.insert=t.strstart<x-1?t.strstart:x-1,e===f?(N(t,!0),0===t.strm.avail_out?O:B):t.last_lit&&(N(t,!1),0===t.strm.avail_out)?A:I}function M(t,e,r,i,n){this.good_length=t,this.max_lazy=e,this.nice_length=r,this.max_chain=i,this.func=n}function H(){this.strm=null,this.status=0,this.pending_buf=null,this.pending_buf_size=0,this.pending_out=0,this.pending=0,this.wrap=0,this.gzhead=null,this.gzindex=0,this.method=v,this.last_flush=-1,this.w_size=0,this.w_bits=0,this.w_mask=0,this.window=null,this.window_size=0,this.prev=null,this.head=null,this.ins_h=0,this.hash_size=0,this.hash_bits=0,this.hash_mask=0,this.hash_shift=0,this.block_start=0,this.match_length=0,this.prev_match=0,this.match_available=0,this.strstart=0,this.match_start=0,this.lookahead=0,this.prev_length=0,this.max_chain_length=0,this.max_lazy_match=0,this.level=0,this.strategy=0,this.good_match=0,this.nice_match=0,this.dyn_ltree=new d.Buf16(2*w),this.dyn_dtree=new d.Buf16(2*(2*a+1)),this.bl_tree=new d.Buf16(2*(2*o+1)),D(this.dyn_ltree),D(this.dyn_dtree),D(this.bl_tree),this.l_desc=null,this.d_desc=null,this.bl_desc=null,this.bl_count=new d.Buf16(k+1),this.heap=new d.Buf16(2*s+1),D(this.heap),this.heap_len=0,this.heap_max=0,this.depth=new d.Buf16(2*s+1),D(this.depth),this.l_buf=0,this.lit_bufsize=0,this.last_lit=0,this.d_buf=0,this.opt_len=0,this.static_len=0,this.matches=0,this.insert=0,this.bi_buf=0,this.bi_valid=0}function G(t){var e;return t&&t.state?(t.total_in=t.total_out=0,t.data_type=n,(e=t.state).pending=0,e.pending_out=0,e.wrap<0&&(e.wrap=-e.wrap),e.status=e.wrap?C:E,t.adler=2===e.wrap?0:1,e.last_flush=l,u._tr_init(e),m):R(t,_)}function K(t){var e=G(t);return e===m&&function(t){t.window_size=2*t.w_size,D(t.head),t.max_lazy_match=h[t.level].max_lazy,t.good_match=h[t.level].good_length,t.nice_match=h[t.level].nice_length,t.max_chain_length=h[t.level].max_chain,t.strstart=0,t.block_start=0,t.lookahead=0,t.insert=0,t.match_length=t.prev_length=x-1,t.match_available=0,t.ins_h=0}(t.state),e}function Y(t,e,r,i,n,s){if(!t)return _;var a=1;if(e===g&&(e=6),i<0?(a=0,i=-i):15<i&&(a=2,i-=16),n<1||y<n||r!==v||i<8||15<i||e<0||9<e||s<0||b<s)return R(t,_);8===i&&(i=9);var o=new H;return(t.state=o).strm=t,o.wrap=a,o.gzhead=null,o.w_bits=i,o.w_size=1<<o.w_bits,o.w_mask=o.w_size-1,o.hash_bits=n+7,o.hash_size=1<<o.hash_bits,o.hash_mask=o.hash_size-1,o.hash_shift=~~((o.hash_bits+x-1)/x),o.window=new d.Buf8(2*o.w_size),o.head=new d.Buf16(o.hash_size),o.prev=new d.Buf16(o.w_size),o.lit_bufsize=1<<n+6,o.pending_buf_size=4*o.lit_bufsize,o.pending_buf=new d.Buf8(o.pending_buf_size),o.d_buf=1*o.lit_bufsize,o.l_buf=3*o.lit_bufsize,o.level=e,o.strategy=s,o.method=r,K(t)}h=[new M(0,0,0,0,function(t,e){var r=65535;for(r>t.pending_buf_size-5&&(r=t.pending_buf_size-5);;){if(t.lookahead<=1){if(j(t),0===t.lookahead&&e===l)return A;if(0===t.lookahead)break}t.strstart+=t.lookahead,t.lookahead=0;var i=t.block_start+r;if((0===t.strstart||t.strstart>=i)&&(t.lookahead=t.strstart-i,t.strstart=i,N(t,!1),0===t.strm.avail_out))return A;if(t.strstart-t.block_start>=t.w_size-z&&(N(t,!1),0===t.strm.avail_out))return A}return t.insert=0,e===f?(N(t,!0),0===t.strm.avail_out?O:B):(t.strstart>t.block_start&&(N(t,!1),t.strm.avail_out),A)}),new M(4,4,8,4,Z),new M(4,5,16,8,Z),new M(4,6,32,32,Z),new M(4,4,16,16,W),new M(8,16,32,32,W),new M(8,16,128,128,W),new M(8,32,128,256,W),new M(32,128,258,1024,W),new M(32,258,258,4096,W)],r.deflateInit=function(t,e){return Y(t,e,v,15,8,0)},r.deflateInit2=Y,r.deflateReset=K,r.deflateResetKeep=G,r.deflateSetHeader=function(t,e){return t&&t.state?2!==t.state.wrap?_:(t.state.gzhead=e,m):_},r.deflate=function(t,e){var r,i,n,s;if(!t||!t.state||5<e||e<0)return t?R(t,_):_;if(i=t.state,!t.output||!t.input&&0!==t.avail_in||666===i.status&&e!==f)return R(t,0===t.avail_out?-5:_);if(i.strm=t,r=i.last_flush,i.last_flush=e,i.status===C)if(2===i.wrap)t.adler=0,U(i,31),U(i,139),U(i,8),i.gzhead?(U(i,(i.gzhead.text?1:0)+(i.gzhead.hcrc?2:0)+(i.gzhead.extra?4:0)+(i.gzhead.name?8:0)+(i.gzhead.comment?16:0)),U(i,255&i.gzhead.time),U(i,i.gzhead.time>>8&255),U(i,i.gzhead.time>>16&255),U(i,i.gzhead.time>>24&255),U(i,9===i.level?2:2<=i.strategy||i.level<2?4:0),U(i,255&i.gzhead.os),i.gzhead.extra&&i.gzhead.extra.length&&(U(i,255&i.gzhead.extra.length),U(i,i.gzhead.extra.length>>8&255)),i.gzhead.hcrc&&(t.adler=p(t.adler,i.pending_buf,i.pending,0)),i.gzindex=0,i.status=69):(U(i,0),U(i,0),U(i,0),U(i,0),U(i,0),U(i,9===i.level?2:2<=i.strategy||i.level<2?4:0),U(i,3),i.status=E);else{var a=v+(i.w_bits-8<<4)<<8;a|=(2<=i.strategy||i.level<2?0:i.level<6?1:6===i.level?2:3)<<6,0!==i.strstart&&(a|=32),a+=31-a%31,i.status=E,P(i,a),0!==i.strstart&&(P(i,t.adler>>>16),P(i,65535&t.adler)),t.adler=1}if(69===i.status)if(i.gzhead.extra){for(n=i.pending;i.gzindex<(65535&i.gzhead.extra.length)&&(i.pending!==i.pending_buf_size||(i.gzhead.hcrc&&i.pending>n&&(t.adler=p(t.adler,i.pending_buf,i.pending-n,n)),F(t),n=i.pending,i.pending!==i.pending_buf_size));)U(i,255&i.gzhead.extra[i.gzindex]),i.gzindex++;i.gzhead.hcrc&&i.pending>n&&(t.adler=p(t.adler,i.pending_buf,i.pending-n,n)),i.gzindex===i.gzhead.extra.length&&(i.gzindex=0,i.status=73)}else i.status=73;if(73===i.status)if(i.gzhead.name){n=i.pending;do{if(i.pending===i.pending_buf_size&&(i.gzhead.hcrc&&i.pending>n&&(t.adler=p(t.adler,i.pending_buf,i.pending-n,n)),F(t),n=i.pending,i.pending===i.pending_buf_size)){s=1;break}s=i.gzindex<i.gzhead.name.length?255&i.gzhead.name.charCodeAt(i.gzindex++):0,U(i,s)}while(0!==s);i.gzhead.hcrc&&i.pending>n&&(t.adler=p(t.adler,i.pending_buf,i.pending-n,n)),0===s&&(i.gzindex=0,i.status=91)}else i.status=91;if(91===i.status)if(i.gzhead.comment){n=i.pending;do{if(i.pending===i.pending_buf_size&&(i.gzhead.hcrc&&i.pending>n&&(t.adler=p(t.adler,i.pending_buf,i.pending-n,n)),F(t),n=i.pending,i.pending===i.pending_buf_size)){s=1;break}s=i.gzindex<i.gzhead.comment.length?255&i.gzhead.comment.charCodeAt(i.gzindex++):0,U(i,s)}while(0!==s);i.gzhead.hcrc&&i.pending>n&&(t.adler=p(t.adler,i.pending_buf,i.pending-n,n)),0===s&&(i.status=103)}else i.status=103;if(103===i.status&&(i.gzhead.hcrc?(i.pending+2>i.pending_buf_size&&F(t),i.pending+2<=i.pending_buf_size&&(U(i,255&t.adler),U(i,t.adler>>8&255),t.adler=0,i.status=E)):i.status=E),0!==i.pending){if(F(t),0===t.avail_out)return i.last_flush=-1,m}else if(0===t.avail_in&&T(e)<=T(r)&&e!==f)return R(t,-5);if(666===i.status&&0!==t.avail_in)return R(t,-5);if(0!==t.avail_in||0!==i.lookahead||e!==l&&666!==i.status){var o=2===i.strategy?function(t,e){for(var r;;){if(0===t.lookahead&&(j(t),0===t.lookahead)){if(e===l)return A;break}if(t.match_length=0,r=u._tr_tally(t,0,t.window[t.strstart]),t.lookahead--,t.strstart++,r&&(N(t,!1),0===t.strm.avail_out))return A}return t.insert=0,e===f?(N(t,!0),0===t.strm.avail_out?O:B):t.last_lit&&(N(t,!1),0===t.strm.avail_out)?A:I}(i,e):3===i.strategy?function(t,e){for(var r,i,n,s,a=t.window;;){if(t.lookahead<=S){if(j(t),t.lookahead<=S&&e===l)return A;if(0===t.lookahead)break}if(t.match_length=0,t.lookahead>=x&&0<t.strstart&&(i=a[n=t.strstart-1])===a[++n]&&i===a[++n]&&i===a[++n]){s=t.strstart+S;do{}while(i===a[++n]&&i===a[++n]&&i===a[++n]&&i===a[++n]&&i===a[++n]&&i===a[++n]&&i===a[++n]&&i===a[++n]&&n<s);t.match_length=S-(s-n),t.match_length>t.lookahead&&(t.match_length=t.lookahead)}if(t.match_length>=x?(r=u._tr_tally(t,1,t.match_length-x),t.lookahead-=t.match_length,t.strstart+=t.match_length,t.match_length=0):(r=u._tr_tally(t,0,t.window[t.strstart]),t.lookahead--,t.strstart++),r&&(N(t,!1),0===t.strm.avail_out))return A}return t.insert=0,e===f?(N(t,!0),0===t.strm.avail_out?O:B):t.last_lit&&(N(t,!1),0===t.strm.avail_out)?A:I}(i,e):h[i.level].func(i,e);if(o!==O&&o!==B||(i.status=666),o===A||o===O)return 0===t.avail_out&&(i.last_flush=-1),m;if(o===I&&(1===e?u._tr_align(i):5!==e&&(u._tr_stored_block(i,0,0,!1),3===e&&(D(i.head),0===i.lookahead&&(i.strstart=0,i.block_start=0,i.insert=0))),F(t),0===t.avail_out))return i.last_flush=-1,m}return e!==f?m:i.wrap<=0?1:(2===i.wrap?(U(i,255&t.adler),U(i,t.adler>>8&255),U(i,t.adler>>16&255),U(i,t.adler>>24&255),U(i,255&t.total_in),U(i,t.total_in>>8&255),U(i,t.total_in>>16&255),U(i,t.total_in>>24&255)):(P(i,t.adler>>>16),P(i,65535&t.adler)),F(t),0<i.wrap&&(i.wrap=-i.wrap),0!==i.pending?m:1)},r.deflateEnd=function(t){var e;return t&&t.state?(e=t.state.status)!==C&&69!==e&&73!==e&&91!==e&&103!==e&&e!==E&&666!==e?R(t,_):(t.state=null,e===E?R(t,-3):m):_},r.deflateSetDictionary=function(t,e){var r,i,n,s,a,o,h,u,l=e.length;if(!t||!t.state)return _;if(2===(s=(r=t.state).wrap)||1===s&&r.status!==C||r.lookahead)return _;for(1===s&&(t.adler=c(t.adler,e,l,0)),r.wrap=0,l>=r.w_size&&(0===s&&(D(r.head),r.strstart=0,r.block_start=0,r.insert=0),u=new d.Buf8(r.w_size),d.arraySet(u,e,l-r.w_size,r.w_size,0),e=u,l=r.w_size),a=t.avail_in,o=t.next_in,h=t.input,t.avail_in=l,t.next_in=0,t.input=e,j(r);r.lookahead>=x;){for(i=r.strstart,n=r.lookahead-(x-1);r.ins_h=(r.ins_h<<r.hash_shift^r.window[i+x-1])&r.hash_mask,r.prev[i&r.w_mask]=r.head[r.ins_h],r.head[r.ins_h]=i,i++,--n;);r.strstart=i,r.lookahead=x-1,j(r)}return r.strstart+=r.lookahead,r.block_start=r.strstart,r.insert=r.lookahead,r.lookahead=0,r.match_length=r.prev_length=x-1,r.match_available=0,t.next_in=o,t.input=h,t.avail_in=a,r.wrap=s,m},r.deflateInfo="pako deflate (from Nodeca project)"},{"../utils/common":41,"./adler32":43,"./crc32":45,"./messages":51,"./trees":52}],47:[function(t,e,r){"use strict";e.exports=function(){this.text=0,this.time=0,this.xflags=0,this.os=0,this.extra=null,this.extra_len=0,this.name="",this.comment="",this.hcrc=0,this.done=!1}},{}],48:[function(t,e,r){"use strict";e.exports=function(t,e){var r,i,n,s,a,o,h,u,l,f,d,c,p,m,_,g,b,v,y,w,k,x,S,z,C;r=t.state,i=t.next_in,z=t.input,n=i+(t.avail_in-5),s=t.next_out,C=t.output,a=s-(e-t.avail_out),o=s+(t.avail_out-257),h=r.dmax,u=r.wsize,l=r.whave,f=r.wnext,d=r.window,c=r.hold,p=r.bits,m=r.lencode,_=r.distcode,g=(1<<r.lenbits)-1,b=(1<<r.distbits)-1;t:do{p<15&&(c+=z[i++]<<p,p+=8,c+=z[i++]<<p,p+=8),v=m[c&g];e:for(;;){if(c>>>=y=v>>>24,p-=y,0===(y=v>>>16&255))C[s++]=65535&v;else{if(!(16&y)){if(0==(64&y)){v=m[(65535&v)+(c&(1<<y)-1)];continue e}if(32&y){r.mode=12;break t}t.msg="invalid literal/length code",r.mode=30;break t}w=65535&v,(y&=15)&&(p<y&&(c+=z[i++]<<p,p+=8),w+=c&(1<<y)-1,c>>>=y,p-=y),p<15&&(c+=z[i++]<<p,p+=8,c+=z[i++]<<p,p+=8),v=_[c&b];r:for(;;){if(c>>>=y=v>>>24,p-=y,!(16&(y=v>>>16&255))){if(0==(64&y)){v=_[(65535&v)+(c&(1<<y)-1)];continue r}t.msg="invalid distance code",r.mode=30;break t}if(k=65535&v,p<(y&=15)&&(c+=z[i++]<<p,(p+=8)<y&&(c+=z[i++]<<p,p+=8)),h<(k+=c&(1<<y)-1)){t.msg="invalid distance too far back",r.mode=30;break t}if(c>>>=y,p-=y,(y=s-a)<k){if(l<(y=k-y)&&r.sane){t.msg="invalid distance too far back",r.mode=30;break t}if(S=d,(x=0)===f){if(x+=u-y,y<w){for(w-=y;C[s++]=d[x++],--y;);x=s-k,S=C}}else if(f<y){if(x+=u+f-y,(y-=f)<w){for(w-=y;C[s++]=d[x++],--y;);if(x=0,f<w){for(w-=y=f;C[s++]=d[x++],--y;);x=s-k,S=C}}}else if(x+=f-y,y<w){for(w-=y;C[s++]=d[x++],--y;);x=s-k,S=C}for(;2<w;)C[s++]=S[x++],C[s++]=S[x++],C[s++]=S[x++],w-=3;w&&(C[s++]=S[x++],1<w&&(C[s++]=S[x++]))}else{for(x=s-k;C[s++]=C[x++],C[s++]=C[x++],C[s++]=C[x++],2<(w-=3););w&&(C[s++]=C[x++],1<w&&(C[s++]=C[x++]))}break}}break}}while(i<n&&s<o);i-=w=p>>3,c&=(1<<(p-=w<<3))-1,t.next_in=i,t.next_out=s,t.avail_in=i<n?n-i+5:5-(i-n),t.avail_out=s<o?o-s+257:257-(s-o),r.hold=c,r.bits=p}},{}],49:[function(t,e,r){"use strict";var I=t("../utils/common"),O=t("./adler32"),B=t("./crc32"),R=t("./inffast"),T=t("./inftrees"),D=1,F=2,N=0,U=-2,P=1,i=852,n=592;function L(t){return(t>>>24&255)+(t>>>8&65280)+((65280&t)<<8)+((255&t)<<24)}function s(){this.mode=0,this.last=!1,this.wrap=0,this.havedict=!1,this.flags=0,this.dmax=0,this.check=0,this.total=0,this.head=null,this.wbits=0,this.wsize=0,this.whave=0,this.wnext=0,this.window=null,this.hold=0,this.bits=0,this.length=0,this.offset=0,this.extra=0,this.lencode=null,this.distcode=null,this.lenbits=0,this.distbits=0,this.ncode=0,this.nlen=0,this.ndist=0,this.have=0,this.next=null,this.lens=new I.Buf16(320),this.work=new I.Buf16(288),this.lendyn=null,this.distdyn=null,this.sane=0,this.back=0,this.was=0}function a(t){var e;return t&&t.state?(e=t.state,t.total_in=t.total_out=e.total=0,t.msg="",e.wrap&&(t.adler=1&e.wrap),e.mode=P,e.last=0,e.havedict=0,e.dmax=32768,e.head=null,e.hold=0,e.bits=0,e.lencode=e.lendyn=new I.Buf32(i),e.distcode=e.distdyn=new I.Buf32(n),e.sane=1,e.back=-1,N):U}function o(t){var e;return t&&t.state?((e=t.state).wsize=0,e.whave=0,e.wnext=0,a(t)):U}function h(t,e){var r,i;return t&&t.state?(i=t.state,e<0?(r=0,e=-e):(r=1+(e>>4),e<48&&(e&=15)),e&&(e<8||15<e)?U:(null!==i.window&&i.wbits!==e&&(i.window=null),i.wrap=r,i.wbits=e,o(t))):U}function u(t,e){var r,i;return t?(i=new s,(t.state=i).window=null,(r=h(t,e))!==N&&(t.state=null),r):U}var l,f,d=!0;function j(t){if(d){var e;for(l=new I.Buf32(512),f=new I.Buf32(32),e=0;e<144;)t.lens[e++]=8;for(;e<256;)t.lens[e++]=9;for(;e<280;)t.lens[e++]=7;for(;e<288;)t.lens[e++]=8;for(T(D,t.lens,0,288,l,0,t.work,{bits:9}),e=0;e<32;)t.lens[e++]=5;T(F,t.lens,0,32,f,0,t.work,{bits:5}),d=!1}t.lencode=l,t.lenbits=9,t.distcode=f,t.distbits=5}function Z(t,e,r,i){var n,s=t.state;return null===s.window&&(s.wsize=1<<s.wbits,s.wnext=0,s.whave=0,s.window=new I.Buf8(s.wsize)),i>=s.wsize?(I.arraySet(s.window,e,r-s.wsize,s.wsize,0),s.wnext=0,s.whave=s.wsize):(i<(n=s.wsize-s.wnext)&&(n=i),I.arraySet(s.window,e,r-i,n,s.wnext),(i-=n)?(I.arraySet(s.window,e,r-i,i,0),s.wnext=i,s.whave=s.wsize):(s.wnext+=n,s.wnext===s.wsize&&(s.wnext=0),s.whave<s.wsize&&(s.whave+=n))),0}r.inflateReset=o,r.inflateReset2=h,r.inflateResetKeep=a,r.inflateInit=function(t){return u(t,15)},r.inflateInit2=u,r.inflate=function(t,e){var r,i,n,s,a,o,h,u,l,f,d,c,p,m,_,g,b,v,y,w,k,x,S,z,C=0,E=new I.Buf8(4),A=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15];if(!t||!t.state||!t.output||!t.input&&0!==t.avail_in)return U;12===(r=t.state).mode&&(r.mode=13),a=t.next_out,n=t.output,h=t.avail_out,s=t.next_in,i=t.input,o=t.avail_in,u=r.hold,l=r.bits,f=o,d=h,x=N;t:for(;;)switch(r.mode){case P:if(0===r.wrap){r.mode=13;break}for(;l<16;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(2&r.wrap&&35615===u){E[r.check=0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0),l=u=0,r.mode=2;break}if(r.flags=0,r.head&&(r.head.done=!1),!(1&r.wrap)||(((255&u)<<8)+(u>>8))%31){t.msg="incorrect header check",r.mode=30;break}if(8!=(15&u)){t.msg="unknown compression method",r.mode=30;break}if(l-=4,k=8+(15&(u>>>=4)),0===r.wbits)r.wbits=k;else if(k>r.wbits){t.msg="invalid window size",r.mode=30;break}r.dmax=1<<k,t.adler=r.check=1,r.mode=512&u?10:12,l=u=0;break;case 2:for(;l<16;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(r.flags=u,8!=(255&r.flags)){t.msg="unknown compression method",r.mode=30;break}if(57344&r.flags){t.msg="unknown header flags set",r.mode=30;break}r.head&&(r.head.text=u>>8&1),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0)),l=u=0,r.mode=3;case 3:for(;l<32;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}r.head&&(r.head.time=u),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,E[2]=u>>>16&255,E[3]=u>>>24&255,r.check=B(r.check,E,4,0)),l=u=0,r.mode=4;case 4:for(;l<16;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}r.head&&(r.head.xflags=255&u,r.head.os=u>>8),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0)),l=u=0,r.mode=5;case 5:if(1024&r.flags){for(;l<16;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}r.length=u,r.head&&(r.head.extra_len=u),512&r.flags&&(E[0]=255&u,E[1]=u>>>8&255,r.check=B(r.check,E,2,0)),l=u=0}else r.head&&(r.head.extra=null);r.mode=6;case 6:if(1024&r.flags&&(o<(c=r.length)&&(c=o),c&&(r.head&&(k=r.head.extra_len-r.length,r.head.extra||(r.head.extra=new Array(r.head.extra_len)),I.arraySet(r.head.extra,i,s,c,k)),512&r.flags&&(r.check=B(r.check,i,c,s)),o-=c,s+=c,r.length-=c),r.length))break t;r.length=0,r.mode=7;case 7:if(2048&r.flags){if(0===o)break t;for(c=0;k=i[s+c++],r.head&&k&&r.length<65536&&(r.head.name+=String.fromCharCode(k)),k&&c<o;);if(512&r.flags&&(r.check=B(r.check,i,c,s)),o-=c,s+=c,k)break t}else r.head&&(r.head.name=null);r.length=0,r.mode=8;case 8:if(4096&r.flags){if(0===o)break t;for(c=0;k=i[s+c++],r.head&&k&&r.length<65536&&(r.head.comment+=String.fromCharCode(k)),k&&c<o;);if(512&r.flags&&(r.check=B(r.check,i,c,s)),o-=c,s+=c,k)break t}else r.head&&(r.head.comment=null);r.mode=9;case 9:if(512&r.flags){for(;l<16;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(u!==(65535&r.check)){t.msg="header crc mismatch",r.mode=30;break}l=u=0}r.head&&(r.head.hcrc=r.flags>>9&1,r.head.done=!0),t.adler=r.check=0,r.mode=12;break;case 10:for(;l<32;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}t.adler=r.check=L(u),l=u=0,r.mode=11;case 11:if(0===r.havedict)return t.next_out=a,t.avail_out=h,t.next_in=s,t.avail_in=o,r.hold=u,r.bits=l,2;t.adler=r.check=1,r.mode=12;case 12:if(5===e||6===e)break t;case 13:if(r.last){u>>>=7&l,l-=7&l,r.mode=27;break}for(;l<3;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}switch(r.last=1&u,l-=1,3&(u>>>=1)){case 0:r.mode=14;break;case 1:if(j(r),r.mode=20,6!==e)break;u>>>=2,l-=2;break t;case 2:r.mode=17;break;case 3:t.msg="invalid block type",r.mode=30}u>>>=2,l-=2;break;case 14:for(u>>>=7&l,l-=7&l;l<32;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if((65535&u)!=(u>>>16^65535)){t.msg="invalid stored block lengths",r.mode=30;break}if(r.length=65535&u,l=u=0,r.mode=15,6===e)break t;case 15:r.mode=16;case 16:if(c=r.length){if(o<c&&(c=o),h<c&&(c=h),0===c)break t;I.arraySet(n,i,s,c,a),o-=c,s+=c,h-=c,a+=c,r.length-=c;break}r.mode=12;break;case 17:for(;l<14;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(r.nlen=257+(31&u),u>>>=5,l-=5,r.ndist=1+(31&u),u>>>=5,l-=5,r.ncode=4+(15&u),u>>>=4,l-=4,286<r.nlen||30<r.ndist){t.msg="too many length or distance symbols",r.mode=30;break}r.have=0,r.mode=18;case 18:for(;r.have<r.ncode;){for(;l<3;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}r.lens[A[r.have++]]=7&u,u>>>=3,l-=3}for(;r.have<19;)r.lens[A[r.have++]]=0;if(r.lencode=r.lendyn,r.lenbits=7,S={bits:r.lenbits},x=T(0,r.lens,0,19,r.lencode,0,r.work,S),r.lenbits=S.bits,x){t.msg="invalid code lengths set",r.mode=30;break}r.have=0,r.mode=19;case 19:for(;r.have<r.nlen+r.ndist;){for(;g=(C=r.lencode[u&(1<<r.lenbits)-1])>>>16&255,b=65535&C,!((_=C>>>24)<=l);){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(b<16)u>>>=_,l-=_,r.lens[r.have++]=b;else{if(16===b){for(z=_+2;l<z;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(u>>>=_,l-=_,0===r.have){t.msg="invalid bit length repeat",r.mode=30;break}k=r.lens[r.have-1],c=3+(3&u),u>>>=2,l-=2}else if(17===b){for(z=_+3;l<z;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}l-=_,k=0,c=3+(7&(u>>>=_)),u>>>=3,l-=3}else{for(z=_+7;l<z;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}l-=_,k=0,c=11+(127&(u>>>=_)),u>>>=7,l-=7}if(r.have+c>r.nlen+r.ndist){t.msg="invalid bit length repeat",r.mode=30;break}for(;c--;)r.lens[r.have++]=k}}if(30===r.mode)break;if(0===r.lens[256]){t.msg="invalid code -- missing end-of-block",r.mode=30;break}if(r.lenbits=9,S={bits:r.lenbits},x=T(D,r.lens,0,r.nlen,r.lencode,0,r.work,S),r.lenbits=S.bits,x){t.msg="invalid literal/lengths set",r.mode=30;break}if(r.distbits=6,r.distcode=r.distdyn,S={bits:r.distbits},x=T(F,r.lens,r.nlen,r.ndist,r.distcode,0,r.work,S),r.distbits=S.bits,x){t.msg="invalid distances set",r.mode=30;break}if(r.mode=20,6===e)break t;case 20:r.mode=21;case 21:if(6<=o&&258<=h){t.next_out=a,t.avail_out=h,t.next_in=s,t.avail_in=o,r.hold=u,r.bits=l,R(t,d),a=t.next_out,n=t.output,h=t.avail_out,s=t.next_in,i=t.input,o=t.avail_in,u=r.hold,l=r.bits,12===r.mode&&(r.back=-1);break}for(r.back=0;g=(C=r.lencode[u&(1<<r.lenbits)-1])>>>16&255,b=65535&C,!((_=C>>>24)<=l);){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(g&&0==(240&g)){for(v=_,y=g,w=b;g=(C=r.lencode[w+((u&(1<<v+y)-1)>>v)])>>>16&255,b=65535&C,!(v+(_=C>>>24)<=l);){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}u>>>=v,l-=v,r.back+=v}if(u>>>=_,l-=_,r.back+=_,r.length=b,0===g){r.mode=26;break}if(32&g){r.back=-1,r.mode=12;break}if(64&g){t.msg="invalid literal/length code",r.mode=30;break}r.extra=15&g,r.mode=22;case 22:if(r.extra){for(z=r.extra;l<z;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}r.length+=u&(1<<r.extra)-1,u>>>=r.extra,l-=r.extra,r.back+=r.extra}r.was=r.length,r.mode=23;case 23:for(;g=(C=r.distcode[u&(1<<r.distbits)-1])>>>16&255,b=65535&C,!((_=C>>>24)<=l);){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(0==(240&g)){for(v=_,y=g,w=b;g=(C=r.distcode[w+((u&(1<<v+y)-1)>>v)])>>>16&255,b=65535&C,!(v+(_=C>>>24)<=l);){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}u>>>=v,l-=v,r.back+=v}if(u>>>=_,l-=_,r.back+=_,64&g){t.msg="invalid distance code",r.mode=30;break}r.offset=b,r.extra=15&g,r.mode=24;case 24:if(r.extra){for(z=r.extra;l<z;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}r.offset+=u&(1<<r.extra)-1,u>>>=r.extra,l-=r.extra,r.back+=r.extra}if(r.offset>r.dmax){t.msg="invalid distance too far back",r.mode=30;break}r.mode=25;case 25:if(0===h)break t;if(c=d-h,r.offset>c){if((c=r.offset-c)>r.whave&&r.sane){t.msg="invalid distance too far back",r.mode=30;break}p=c>r.wnext?(c-=r.wnext,r.wsize-c):r.wnext-c,c>r.length&&(c=r.length),m=r.window}else m=n,p=a-r.offset,c=r.length;for(h<c&&(c=h),h-=c,r.length-=c;n[a++]=m[p++],--c;);0===r.length&&(r.mode=21);break;case 26:if(0===h)break t;n[a++]=r.length,h--,r.mode=21;break;case 27:if(r.wrap){for(;l<32;){if(0===o)break t;o--,u|=i[s++]<<l,l+=8}if(d-=h,t.total_out+=d,r.total+=d,d&&(t.adler=r.check=r.flags?B(r.check,n,d,a-d):O(r.check,n,d,a-d)),d=h,(r.flags?u:L(u))!==r.check){t.msg="incorrect data check",r.mode=30;break}l=u=0}r.mode=28;case 28:if(r.wrap&&r.flags){for(;l<32;){if(0===o)break t;o--,u+=i[s++]<<l,l+=8}if(u!==(4294967295&r.total)){t.msg="incorrect length check",r.mode=30;break}l=u=0}r.mode=29;case 29:x=1;break t;case 30:x=-3;break t;case 31:return-4;case 32:default:return U}return t.next_out=a,t.avail_out=h,t.next_in=s,t.avail_in=o,r.hold=u,r.bits=l,(r.wsize||d!==t.avail_out&&r.mode<30&&(r.mode<27||4!==e))&&Z(t,t.output,t.next_out,d-t.avail_out)?(r.mode=31,-4):(f-=t.avail_in,d-=t.avail_out,t.total_in+=f,t.total_out+=d,r.total+=d,r.wrap&&d&&(t.adler=r.check=r.flags?B(r.check,n,d,t.next_out-d):O(r.check,n,d,t.next_out-d)),t.data_type=r.bits+(r.last?64:0)+(12===r.mode?128:0)+(20===r.mode||15===r.mode?256:0),(0==f&&0===d||4===e)&&x===N&&(x=-5),x)},r.inflateEnd=function(t){if(!t||!t.state)return U;var e=t.state;return e.window&&(e.window=null),t.state=null,N},r.inflateGetHeader=function(t,e){var r;return t&&t.state?0==(2&(r=t.state).wrap)?U:((r.head=e).done=!1,N):U},r.inflateSetDictionary=function(t,e){var r,i=e.length;return t&&t.state?0!==(r=t.state).wrap&&11!==r.mode?U:11===r.mode&&O(1,e,i,0)!==r.check?-3:Z(t,e,i,i)?(r.mode=31,-4):(r.havedict=1,N):U},r.inflateInfo="pako inflate (from Nodeca project)"},{"../utils/common":41,"./adler32":43,"./crc32":45,"./inffast":48,"./inftrees":50}],50:[function(t,e,r){"use strict";var D=t("../utils/common"),F=[3,4,5,6,7,8,9,10,11,13,15,17,19,23,27,31,35,43,51,59,67,83,99,115,131,163,195,227,258,0,0],N=[16,16,16,16,16,16,16,16,17,17,17,17,18,18,18,18,19,19,19,19,20,20,20,20,21,21,21,21,16,72,78],U=[1,2,3,4,5,7,9,13,17,25,33,49,65,97,129,193,257,385,513,769,1025,1537,2049,3073,4097,6145,8193,12289,16385,24577,0,0],P=[16,16,16,16,17,17,18,18,19,19,20,20,21,21,22,22,23,23,24,24,25,25,26,26,27,27,28,28,29,29,64,64];e.exports=function(t,e,r,i,n,s,a,o){var h,u,l,f,d,c,p,m,_,g=o.bits,b=0,v=0,y=0,w=0,k=0,x=0,S=0,z=0,C=0,E=0,A=null,I=0,O=new D.Buf16(16),B=new D.Buf16(16),R=null,T=0;for(b=0;b<=15;b++)O[b]=0;for(v=0;v<i;v++)O[e[r+v]]++;for(k=g,w=15;1<=w&&0===O[w];w--);if(w<k&&(k=w),0===w)return n[s++]=20971520,n[s++]=20971520,o.bits=1,0;for(y=1;y<w&&0===O[y];y++);for(k<y&&(k=y),b=z=1;b<=15;b++)if(z<<=1,(z-=O[b])<0)return-1;if(0<z&&(0===t||1!==w))return-1;for(B[1]=0,b=1;b<15;b++)B[b+1]=B[b]+O[b];for(v=0;v<i;v++)0!==e[r+v]&&(a[B[e[r+v]]++]=v);if(c=0===t?(A=R=a,19):1===t?(A=F,I-=257,R=N,T-=257,256):(A=U,R=P,-1),b=y,d=s,S=v=E=0,l=-1,f=(C=1<<(x=k))-1,1===t&&852<C||2===t&&592<C)return 1;for(;;){for(p=b-S,_=a[v]<c?(m=0,a[v]):a[v]>c?(m=R[T+a[v]],A[I+a[v]]):(m=96,0),h=1<<b-S,y=u=1<<x;n[d+(E>>S)+(u-=h)]=p<<24|m<<16|_|0,0!==u;);for(h=1<<b-1;E&h;)h>>=1;if(0!==h?(E&=h-1,E+=h):E=0,v++,0==--O[b]){if(b===w)break;b=e[r+a[v]]}if(k<b&&(E&f)!==l){for(0===S&&(S=k),d+=y,z=1<<(x=b-S);x+S<w&&!((z-=O[x+S])<=0);)x++,z<<=1;if(C+=1<<x,1===t&&852<C||2===t&&592<C)return 1;n[l=E&f]=k<<24|x<<16|d-s|0}}return 0!==E&&(n[d+E]=b-S<<24|64<<16|0),o.bits=k,0}},{"../utils/common":41}],51:[function(t,e,r){"use strict";e.exports={2:"need dictionary",1:"stream end",0:"","-1":"file error","-2":"stream error","-3":"data error","-4":"insufficient memory","-5":"buffer error","-6":"incompatible version"}},{}],52:[function(t,e,r){"use strict";var n=t("../utils/common"),o=0,h=1;function i(t){for(var e=t.length;0<=--e;)t[e]=0}var s=0,a=29,u=256,l=u+1+a,f=30,d=19,_=2*l+1,g=15,c=16,p=7,m=256,b=16,v=17,y=18,w=[0,0,0,0,0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5,0],k=[0,0,0,0,1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10,11,11,12,12,13,13],x=[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,7],S=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15],z=new Array(2*(l+2));i(z);var C=new Array(2*f);i(C);var E=new Array(512);i(E);var A=new Array(256);i(A);var I=new Array(a);i(I);var O,B,R,T=new Array(f);function D(t,e,r,i,n){this.static_tree=t,this.extra_bits=e,this.extra_base=r,this.elems=i,this.max_length=n,this.has_stree=t&&t.length}function F(t,e){this.dyn_tree=t,this.max_code=0,this.stat_desc=e}function N(t){return t<256?E[t]:E[256+(t>>>7)]}function U(t,e){t.pending_buf[t.pending++]=255&e,t.pending_buf[t.pending++]=e>>>8&255}function P(t,e,r){t.bi_valid>c-r?(t.bi_buf|=e<<t.bi_valid&65535,U(t,t.bi_buf),t.bi_buf=e>>c-t.bi_valid,t.bi_valid+=r-c):(t.bi_buf|=e<<t.bi_valid&65535,t.bi_valid+=r)}function L(t,e,r){P(t,r[2*e],r[2*e+1])}function j(t,e){for(var r=0;r|=1&t,t>>>=1,r<<=1,0<--e;);return r>>>1}function Z(t,e,r){var i,n,s=new Array(g+1),a=0;for(i=1;i<=g;i++)s[i]=a=a+r[i-1]<<1;for(n=0;n<=e;n++){var o=t[2*n+1];0!==o&&(t[2*n]=j(s[o]++,o))}}function W(t){var e;for(e=0;e<l;e++)t.dyn_ltree[2*e]=0;for(e=0;e<f;e++)t.dyn_dtree[2*e]=0;for(e=0;e<d;e++)t.bl_tree[2*e]=0;t.dyn_ltree[2*m]=1,t.opt_len=t.static_len=0,t.last_lit=t.matches=0}function M(t){8<t.bi_valid?U(t,t.bi_buf):0<t.bi_valid&&(t.pending_buf[t.pending++]=t.bi_buf),t.bi_buf=0,t.bi_valid=0}function H(t,e,r,i){var n=2*e,s=2*r;return t[n]<t[s]||t[n]===t[s]&&i[e]<=i[r]}function G(t,e,r){for(var i=t.heap[r],n=r<<1;n<=t.heap_len&&(n<t.heap_len&&H(e,t.heap[n+1],t.heap[n],t.depth)&&n++,!H(e,i,t.heap[n],t.depth));)t.heap[r]=t.heap[n],r=n,n<<=1;t.heap[r]=i}function K(t,e,r){var i,n,s,a,o=0;if(0!==t.last_lit)for(;i=t.pending_buf[t.d_buf+2*o]<<8|t.pending_buf[t.d_buf+2*o+1],n=t.pending_buf[t.l_buf+o],o++,0===i?L(t,n,e):(L(t,(s=A[n])+u+1,e),0!==(a=w[s])&&P(t,n-=I[s],a),L(t,s=N(--i),r),0!==(a=k[s])&&P(t,i-=T[s],a)),o<t.last_lit;);L(t,m,e)}function Y(t,e){var r,i,n,s=e.dyn_tree,a=e.stat_desc.static_tree,o=e.stat_desc.has_stree,h=e.stat_desc.elems,u=-1;for(t.heap_len=0,t.heap_max=_,r=0;r<h;r++)0!==s[2*r]?(t.heap[++t.heap_len]=u=r,t.depth[r]=0):s[2*r+1]=0;for(;t.heap_len<2;)s[2*(n=t.heap[++t.heap_len]=u<2?++u:0)]=1,t.depth[n]=0,t.opt_len--,o&&(t.static_len-=a[2*n+1]);for(e.max_code=u,r=t.heap_len>>1;1<=r;r--)G(t,s,r);for(n=h;r=t.heap[1],t.heap[1]=t.heap[t.heap_len--],G(t,s,1),i=t.heap[1],t.heap[--t.heap_max]=r,t.heap[--t.heap_max]=i,s[2*n]=s[2*r]+s[2*i],t.depth[n]=(t.depth[r]>=t.depth[i]?t.depth[r]:t.depth[i])+1,s[2*r+1]=s[2*i+1]=n,t.heap[1]=n++,G(t,s,1),2<=t.heap_len;);t.heap[--t.heap_max]=t.heap[1],function(t,e){var r,i,n,s,a,o,h=e.dyn_tree,u=e.max_code,l=e.stat_desc.static_tree,f=e.stat_desc.has_stree,d=e.stat_desc.extra_bits,c=e.stat_desc.extra_base,p=e.stat_desc.max_length,m=0;for(s=0;s<=g;s++)t.bl_count[s]=0;for(h[2*t.heap[t.heap_max]+1]=0,r=t.heap_max+1;r<_;r++)p<(s=h[2*h[2*(i=t.heap[r])+1]+1]+1)&&(s=p,m++),h[2*i+1]=s,u<i||(t.bl_count[s]++,a=0,c<=i&&(a=d[i-c]),o=h[2*i],t.opt_len+=o*(s+a),f&&(t.static_len+=o*(l[2*i+1]+a)));if(0!==m){do{for(s=p-1;0===t.bl_count[s];)s--;t.bl_count[s]--,t.bl_count[s+1]+=2,t.bl_count[p]--,m-=2}while(0<m);for(s=p;0!==s;s--)for(i=t.bl_count[s];0!==i;)u<(n=t.heap[--r])||(h[2*n+1]!==s&&(t.opt_len+=(s-h[2*n+1])*h[2*n],h[2*n+1]=s),i--)}}(t,e),Z(s,u,t.bl_count)}function X(t,e,r){var i,n,s=-1,a=e[1],o=0,h=7,u=4;for(0===a&&(h=138,u=3),e[2*(r+1)+1]=65535,i=0;i<=r;i++)n=a,a=e[2*(i+1)+1],++o<h&&n===a||(o<u?t.bl_tree[2*n]+=o:0!==n?(n!==s&&t.bl_tree[2*n]++,t.bl_tree[2*b]++):o<=10?t.bl_tree[2*v]++:t.bl_tree[2*y]++,s=n,u=(o=0)===a?(h=138,3):n===a?(h=6,3):(h=7,4))}function V(t,e,r){var i,n,s=-1,a=e[1],o=0,h=7,u=4;for(0===a&&(h=138,u=3),i=0;i<=r;i++)if(n=a,a=e[2*(i+1)+1],!(++o<h&&n===a)){if(o<u)for(;L(t,n,t.bl_tree),0!=--o;);else 0!==n?(n!==s&&(L(t,n,t.bl_tree),o--),L(t,b,t.bl_tree),P(t,o-3,2)):o<=10?(L(t,v,t.bl_tree),P(t,o-3,3)):(L(t,y,t.bl_tree),P(t,o-11,7));s=n,u=(o=0)===a?(h=138,3):n===a?(h=6,3):(h=7,4)}}i(T);var q=!1;function J(t,e,r,i){P(t,(s<<1)+(i?1:0),3),function(t,e,r,i){M(t),i&&(U(t,r),U(t,~r)),n.arraySet(t.pending_buf,t.window,e,r,t.pending),t.pending+=r}(t,e,r,!0)}r._tr_init=function(t){q||(function(){var t,e,r,i,n,s=new Array(g+1);for(i=r=0;i<a-1;i++)for(I[i]=r,t=0;t<1<<w[i];t++)A[r++]=i;for(A[r-1]=i,i=n=0;i<16;i++)for(T[i]=n,t=0;t<1<<k[i];t++)E[n++]=i;for(n>>=7;i<f;i++)for(T[i]=n<<7,t=0;t<1<<k[i]-7;t++)E[256+n++]=i;for(e=0;e<=g;e++)s[e]=0;for(t=0;t<=143;)z[2*t+1]=8,t++,s[8]++;for(;t<=255;)z[2*t+1]=9,t++,s[9]++;for(;t<=279;)z[2*t+1]=7,t++,s[7]++;for(;t<=287;)z[2*t+1]=8,t++,s[8]++;for(Z(z,l+1,s),t=0;t<f;t++)C[2*t+1]=5,C[2*t]=j(t,5);O=new D(z,w,u+1,l,g),B=new D(C,k,0,f,g),R=new D(new Array(0),x,0,d,p)}(),q=!0),t.l_desc=new F(t.dyn_ltree,O),t.d_desc=new F(t.dyn_dtree,B),t.bl_desc=new F(t.bl_tree,R),t.bi_buf=0,t.bi_valid=0,W(t)},r._tr_stored_block=J,r._tr_flush_block=function(t,e,r,i){var n,s,a=0;0<t.level?(2===t.strm.data_type&&(t.strm.data_type=function(t){var e,r=4093624447;for(e=0;e<=31;e++,r>>>=1)if(1&r&&0!==t.dyn_ltree[2*e])return o;if(0!==t.dyn_ltree[18]||0!==t.dyn_ltree[20]||0!==t.dyn_ltree[26])return h;for(e=32;e<u;e++)if(0!==t.dyn_ltree[2*e])return h;return o}(t)),Y(t,t.l_desc),Y(t,t.d_desc),a=function(t){var e;for(X(t,t.dyn_ltree,t.l_desc.max_code),X(t,t.dyn_dtree,t.d_desc.max_code),Y(t,t.bl_desc),e=d-1;3<=e&&0===t.bl_tree[2*S[e]+1];e--);return t.opt_len+=3*(e+1)+5+5+4,e}(t),n=t.opt_len+3+7>>>3,(s=t.static_len+3+7>>>3)<=n&&(n=s)):n=s=r+5,r+4<=n&&-1!==e?J(t,e,r,i):4===t.strategy||s===n?(P(t,2+(i?1:0),3),K(t,z,C)):(P(t,4+(i?1:0),3),function(t,e,r,i){var n;for(P(t,e-257,5),P(t,r-1,5),P(t,i-4,4),n=0;n<i;n++)P(t,t.bl_tree[2*S[n]+1],3);V(t,t.dyn_ltree,e-1),V(t,t.dyn_dtree,r-1)}(t,t.l_desc.max_code+1,t.d_desc.max_code+1,a+1),K(t,t.dyn_ltree,t.dyn_dtree)),W(t),i&&M(t)},r._tr_tally=function(t,e,r){return t.pending_buf[t.d_buf+2*t.last_lit]=e>>>8&255,t.pending_buf[t.d_buf+2*t.last_lit+1]=255&e,t.pending_buf[t.l_buf+t.last_lit]=255&r,t.last_lit++,0===e?t.dyn_ltree[2*r]++:(t.matches++,e--,t.dyn_ltree[2*(A[r]+u+1)]++,t.dyn_dtree[2*N(e)]++),t.last_lit===t.lit_bufsize-1},r._tr_align=function(t){P(t,2,3),L(t,m,z),function(t){16===t.bi_valid?(U(t,t.bi_buf),t.bi_buf=0,t.bi_valid=0):8<=t.bi_valid&&(t.pending_buf[t.pending++]=255&t.bi_buf,t.bi_buf>>=8,t.bi_valid-=8)}(t)}},{"../utils/common":41}],53:[function(t,e,r){"use strict";e.exports=function(){this.input=null,this.next_in=0,this.avail_in=0,this.total_in=0,this.output=null,this.next_out=0,this.avail_out=0,this.total_out=0,this.msg="",this.state=null,this.data_type=2,this.adler=0}},{}],54:[function(t,e,r){"use strict";e.exports="function"==typeof setImmediate?setImmediate:function(){var t=[].slice.apply(arguments);t.splice(1,0,0),setTimeout.apply(null,t)}},{}]},{},[10])(10)});
/*! pdfmake v0.1.70, @license MIT, @link http://pdfmake.org */
!function webpackUniversalModuleDefinition(n,i){if("object"==typeof exports&&"object"==typeof module)module.exports=i();else if("function"==typeof define&&define.amd)define([],i);else{var o=i();for(var u in o)("object"==typeof exports?exports:n)[u]=o[u]}}("undefined"!=typeof self?self:this,(function(){return function(n){var i={};function __webpack_require__(o){if(i[o])return i[o].exports;var u=i[o]={i:o,l:!1,exports:{}};return n[o].call(u.exports,u,u.exports,__webpack_require__),u.l=!0,u.exports}return __webpack_require__.m=n,__webpack_require__.c=i,__webpack_require__.d=function(n,i,o){__webpack_require__.o(n,i)||Object.defineProperty(n,i,{enumerable:!0,get:o})},__webpack_require__.r=function(n){"undefined"!=typeof Symbol&&Symbol.toStringTag&&Object.defineProperty(n,Symbol.toStringTag,{value:"Module"}),Object.defineProperty(n,"__esModule",{value:!0})},__webpack_require__.t=function(n,i){if(1&i&&(n=__webpack_require__(n)),8&i)return n;if(4&i&&"object"==typeof n&&n&&n.__esModule)return n;var o=Object.create(null);if(__webpack_require__.r(o),Object.defineProperty(o,"default",{enumerable:!0,value:n}),2&i&&"string"!=typeof n)for(var u in n)__webpack_require__.d(o,u,function(i){return n[i]}.bind(null,u));return o},__webpack_require__.n=function(n){var i=n&&n.__esModule?function getDefault(){return n.default}:function getModuleExports(){return n};return __webpack_require__.d(i,"a",i),i},__webpack_require__.o=function(n,i){return Object.prototype.hasOwnProperty.call(n,i)},__webpack_require__.p="",__webpack_require__(__webpack_require__.s=302)}([function(n,i,o){"use strict";function isArray(n){return Array.isArray(n)}n.exports={isString:function isString(n){return"string"==typeof n||n instanceof String},isNumber:function isNumber(n){return"number"==typeof n||n instanceof Number},isBoolean:function isBoolean(n){return"boolean"==typeof n},isArray:isArray,isFunction:function isFunction(n){return"function"==typeof n},isObject:function isObject(n){return null!==n&&"object"==typeof n},isNull:function isNull(n){return null===n},isUndefined:function isUndefined(n){return void 0===n},pack:function pack(){for(var n={},i=0,o=arguments.length;i<o;i++){var u=arguments[i];if(u)for(var p in u)u.hasOwnProperty(p)&&(n[p]=u[p])}return n},fontStringify:function fontStringify(n,i){return"font"===n?"font":i},offsetVector:function offsetVector(n,i,o){switch(n.type){case"ellipse":case"rect":n.x+=i,n.y+=o;break;case"line":n.x1+=i,n.x2+=i,n.y1+=o,n.y2+=o;break;case"polyline":for(var u=0,p=n.points.length;u<p;u++)n.points[u].x+=i,n.points[u].y+=o}},getNodeId:function getNodeId(n){if(n.id)return n.id;if(isArray(n.text))for(var i=0,o=n.text.length;i<o;i++){var u=getNodeId(n.text[i]);if(u)return u}return null}}},function(n,i,o){var u=o(3),p=o(54).f,g=o(18),y=o(25),w=o(139),x=o(203),_=o(145);n.exports=function(n,i){var o,k,P,E,O,B=n.target,I=n.global,D=n.stat;if(o=I?u:D?u[B]||w(B,{}):(u[B]||{}).prototype)for(k in i){if(E=i[k],P=n.noTargetGet?(O=p(o,k))&&O.value:o[k],!_(I?k:B+(D?".":"#")+k,n.forced)&&void 0!==P){if(typeof E==typeof P)continue;x(E,P)}(n.sham||P&&P.sham)&&g(E,"sham",!0),y(o,k,E,n)}}},function(n,i,o){var u;n.exports=(u=u||function(n,i){var o=Object.create||function(){function F(){}return function(n){var i;return F.prototype=n,i=new F,F.prototype=null,i}}(),u={},p=u.lib={},g=p.Base={extend:function(n){var i=o(this);return n&&i.mixIn(n),i.hasOwnProperty("init")&&this.init!==i.init||(i.init=function(){i.$super.init.apply(this,arguments)}),i.init.prototype=i,i.$super=this,i},create:function(){var n=this.extend();return n.init.apply(n,arguments),n},init:function(){},mixIn:function(n){for(var i in n)n.hasOwnProperty(i)&&(this[i]=n[i]);n.hasOwnProperty("toString")&&(this.toString=n.toString)},clone:function(){return this.init.prototype.extend(this)}},y=p.WordArray=g.extend({init:function(n,i){n=this.words=n||[],this.sigBytes=null!=i?i:4*n.length},toString:function(n){return(n||x).stringify(this)},concat:function(n){var i=this.words,o=n.words,u=this.sigBytes,p=n.sigBytes;if(this.clamp(),u%4)for(var g=0;g<p;g++){var y=o[g>>>2]>>>24-g%4*8&255;i[u+g>>>2]|=y<<24-(u+g)%4*8}else for(g=0;g<p;g+=4)i[u+g>>>2]=o[g>>>2];return this.sigBytes+=p,this},clamp:function(){var i=this.words,o=this.sigBytes;i[o>>>2]&=4294967295<<32-o%4*8,i.length=n.ceil(o/4)},clone:function(){var n=g.clone.call(this);return n.words=this.words.slice(0),n},random:function(i){for(var o,u=[],r=function(i){i=i;var o=987654321,u=4294967295;return function(){var p=((o=36969*(65535&o)+(o>>16)&u)<<16)+(i=18e3*(65535&i)+(i>>16)&u)&u;return p/=4294967296,(p+=.5)*(n.random()>.5?1:-1)}},p=0;p<i;p+=4){var g=r(4294967296*(o||n.random()));o=987654071*g(),u.push(4294967296*g()|0)}return new y.init(u,i)}}),w=u.enc={},x=w.Hex={stringify:function(n){for(var i=n.words,o=n.sigBytes,u=[],p=0;p<o;p++){var g=i[p>>>2]>>>24-p%4*8&255;u.push((g>>>4).toString(16)),u.push((15&g).toString(16))}return u.join("")},parse:function(n){for(var i=n.length,o=[],u=0;u<i;u+=2)o[u>>>3]|=parseInt(n.substr(u,2),16)<<24-u%8*4;return new y.init(o,i/2)}},_=w.Latin1={stringify:function(n){for(var i=n.words,o=n.sigBytes,u=[],p=0;p<o;p++){var g=i[p>>>2]>>>24-p%4*8&255;u.push(String.fromCharCode(g))}return u.join("")},parse:function(n){for(var i=n.length,o=[],u=0;u<i;u++)o[u>>>2]|=(255&n.charCodeAt(u))<<24-u%4*8;return new y.init(o,i)}},k=w.Utf8={stringify:function(n){try{return decodeURIComponent(escape(_.stringify(n)))}catch(n){throw new Error("Malformed UTF-8 data")}},parse:function(n){return _.parse(unescape(encodeURIComponent(n)))}},P=p.BufferedBlockAlgorithm=g.extend({reset:function(){this._data=new y.init,this._nDataBytes=0},_append:function(n){"string"==typeof n&&(n=k.parse(n)),this._data.concat(n),this._nDataBytes+=n.sigBytes},_process:function(i){var o=this._data,u=o.words,p=o.sigBytes,g=this.blockSize,w=p/(4*g),x=(w=i?n.ceil(w):n.max((0|w)-this._minBufferSize,0))*g,_=n.min(4*x,p);if(x){for(var k=0;k<x;k+=g)this._doProcessBlock(u,k);var P=u.splice(0,x);o.sigBytes-=_}return new y.init(P,_)},clone:function(){var n=g.clone.call(this);return n._data=this._data.clone(),n},_minBufferSize:0}),E=(p.Hasher=P.extend({cfg:g.extend(),init:function(n){this.cfg=this.cfg.extend(n),this.reset()},reset:function(){P.reset.call(this),this._doReset()},update:function(n){return this._append(n),this._process(),this},finalize:function(n){return n&&this._append(n),this._doFinalize()},blockSize:16,_createHelper:function(n){return function(i,o){return new n.init(o).finalize(i)}},_createHmacHelper:function(n){return function(i,o){return new E.HMAC.init(n,o).finalize(i)}}}),u.algo={});return u}(Math),u)},function(n,i,o){(function(i){var check=function(n){return n&&n.Math==Math&&n};n.exports=check("object"==typeof globalThis&&globalThis)||check("object"==typeof window&&window)||check("object"==typeof self&&self)||check("object"==typeof i&&i)||function(){return this}()||Function("return this")()}).call(this,o(24))},function(n,i){n.exports=function(n){try{return!!n()}catch(n){return!0}}},function(n,i,o){"use strict";var u,p=o(221),g=o(13),y=o(3),w=o(14),x=o(15),_=o(117),k=o(18),P=o(25),E=o(16).f,O=o(60),B=o(45),I=o(6),D=o(112),R=y.Int8Array,N=R&&R.prototype,U=y.Uint8ClampedArray,W=U&&U.prototype,G=R&&O(R),j=N&&O(N),X=Object.prototype,Y=X.isPrototypeOf,K=I("toStringTag"),J=D("TYPED_ARRAY_TAG"),$=p&&!!B&&"Opera"!==_(y.opera),tt=!1,et={Int8Array:1,Uint8Array:1,Uint8ClampedArray:1,Int16Array:2,Uint16Array:2,Int32Array:4,Uint32Array:4,Float32Array:4,Float64Array:8},rt={BigInt64Array:8,BigUint64Array:8},isTypedArray=function(n){if(!w(n))return!1;var i=_(n);return x(et,i)||x(rt,i)};for(u in et)y[u]||($=!1);if((!$||"function"!=typeof G||G===Function.prototype)&&(G=function TypedArray(){throw TypeError("Incorrect invocation")},$))for(u in et)y[u]&&B(y[u],G);if((!$||!j||j===X)&&(j=G.prototype,$))for(u in et)y[u]&&B(y[u].prototype,j);if($&&O(W)!==j&&B(W,j),g&&!x(j,K))for(u in tt=!0,E(j,K,{get:function(){return w(this)?this[J]:void 0}}),et)y[u]&&k(y[u],J,u);n.exports={NATIVE_ARRAY_BUFFER_VIEWS:$,TYPED_ARRAY_TAG:tt&&J,aTypedArray:function(n){if(isTypedArray(n))return n;throw TypeError("Target is not a typed array")},aTypedArrayConstructor:function(n){if(B){if(Y.call(G,n))return n}else for(var i in et)if(x(et,u)){var o=y[i];if(o&&(n===o||Y.call(o,n)))return n}throw TypeError("Target is not a typed array constructor")},exportTypedArrayMethod:function(n,i,o){if(g){if(o)for(var u in et){var p=y[u];p&&x(p.prototype,n)&&delete p.prototype[n]}j[n]&&!o||P(j,n,o?i:$&&N[n]||i)}},exportTypedArrayStaticMethod:function(n,i,o){var u,p;if(g){if(B){if(o)for(u in et)(p=y[u])&&x(p,n)&&delete p[n];if(G[n]&&!o)return;try{return P(G,n,o?i:$&&R[n]||i)}catch(n){}}for(u in et)!(p=y[u])||p[n]&&!o||P(p,n,i)}},isView:function isView(n){if(!w(n))return!1;var i=_(n);return"DataView"===i||x(et,i)||x(rt,i)},isTypedArray:isTypedArray,TypedArray:G,TypedArrayPrototype:j}},function(n,i,o){var u=o(3),p=o(142),g=o(15),y=o(112),w=o(146),x=o(205),_=p("wks"),k=u.Symbol,P=x?k:k&&k.withoutSetter||y;n.exports=function(n){return g(_,n)||(w&&g(k,n)?_[n]=k[n]:_[n]=P("Symbol."+n)),_[n]}},function(n,i){var o=n.exports={version:"2.6.12"};"number"==typeof __e&&(__e=o)},function(n,i,o){var u=o(35),p=Math.min;n.exports=function(n){return n>0?p(u(n),9007199254740991):0}},function(n,i,o){var u=o(17),p=o(7),g=o(32),y=o(39),w=o(51),$export=function(n,i,o){var x,_,k,P=n&$export.F,E=n&$export.G,O=n&$export.S,B=n&$export.P,I=n&$export.B,D=n&$export.W,R=E?p:p[i]||(p[i]={}),N=R.prototype,U=E?u:O?u[i]:(u[i]||{}).prototype;for(x in E&&(o=i),o)(_=!P&&U&&void 0!==U[x])&&w(R,x)||(k=_?U[x]:o[x],R[x]=E&&"function"!=typeof U[x]?o[x]:I&&_?g(k,u):D&&U[x]==k?function(n){var F=function(i,o,u){if(this instanceof n){switch(arguments.length){case 0:return new n;case 1:return new n(i);case 2:return new n(i,o)}return new n(i,o,u)}return n.apply(this,arguments)};return F.prototype=n.prototype,F}(k):B&&"function"==typeof k?g(Function.call,k):k,B&&((R.virtual||(R.virtual={}))[x]=k,n&$export.R&&N&&!N[x]&&y(N,x,k)))};$export.F=1,$export.G=2,$export.S=4,$export.P=8,$export.B=16,$export.W=32,$export.U=64,$export.R=128,n.exports=$export},function(n,i,o){"use strict";(function(n){var u=o(304),p=o(305),g=o(200);function kMaxLength(){return Buffer.TYPED_ARRAY_SUPPORT?2147483647:1073741823}function createBuffer(n,i){if(kMaxLength()<i)throw new RangeError("Invalid typed array length");return Buffer.TYPED_ARRAY_SUPPORT?(n=new Uint8Array(i)).__proto__=Buffer.prototype:(null===n&&(n=new Buffer(i)),n.length=i),n}function Buffer(n,i,o){if(!(Buffer.TYPED_ARRAY_SUPPORT||this instanceof Buffer))return new Buffer(n,i,o);if("number"==typeof n){if("string"==typeof i)throw new Error("If encoding is specified then the first argument must be a string");return allocUnsafe(this,n)}return from(this,n,i,o)}function from(n,i,o,u){if("number"==typeof i)throw new TypeError('"value" argument must not be a number');return"undefined"!=typeof ArrayBuffer&&i instanceof ArrayBuffer?function fromArrayBuffer(n,i,o,u){if(i.byteLength,o<0||i.byteLength<o)throw new RangeError("'offset' is out of bounds");if(i.byteLength<o+(u||0))throw new RangeError("'length' is out of bounds");i=void 0===o&&void 0===u?new Uint8Array(i):void 0===u?new Uint8Array(i,o):new Uint8Array(i,o,u);Buffer.TYPED_ARRAY_SUPPORT?(n=i).__proto__=Buffer.prototype:n=fromArrayLike(n,i);return n}(n,i,o,u):"string"==typeof i?function fromString(n,i,o){"string"==typeof o&&""!==o||(o="utf8");if(!Buffer.isEncoding(o))throw new TypeError('"encoding" must be a valid string encoding');var u=0|byteLength(i,o),p=(n=createBuffer(n,u)).write(i,o);p!==u&&(n=n.slice(0,p));return n}(n,i,o):function fromObject(n,i){if(Buffer.isBuffer(i)){var o=0|checked(i.length);return 0===(n=createBuffer(n,o)).length||i.copy(n,0,0,o),n}if(i){if("undefined"!=typeof ArrayBuffer&&i.buffer instanceof ArrayBuffer||"length"in i)return"number"!=typeof i.length||function isnan(n){return n!=n}(i.length)?createBuffer(n,0):fromArrayLike(n,i);if("Buffer"===i.type&&g(i.data))return fromArrayLike(n,i.data)}throw new TypeError("First argument must be a string, Buffer, ArrayBuffer, Array, or array-like object.")}(n,i)}function assertSize(n){if("number"!=typeof n)throw new TypeError('"size" argument must be a number');if(n<0)throw new RangeError('"size" argument must not be negative')}function allocUnsafe(n,i){if(assertSize(i),n=createBuffer(n,i<0?0:0|checked(i)),!Buffer.TYPED_ARRAY_SUPPORT)for(var o=0;o<i;++o)n[o]=0;return n}function fromArrayLike(n,i){var o=i.length<0?0:0|checked(i.length);n=createBuffer(n,o);for(var u=0;u<o;u+=1)n[u]=255&i[u];return n}function checked(n){if(n>=kMaxLength())throw new RangeError("Attempt to allocate Buffer larger than maximum size: 0x"+kMaxLength().toString(16)+" bytes");return 0|n}function byteLength(n,i){if(Buffer.isBuffer(n))return n.length;if("undefined"!=typeof ArrayBuffer&&"function"==typeof ArrayBuffer.isView&&(ArrayBuffer.isView(n)||n instanceof ArrayBuffer))return n.byteLength;"string"!=typeof n&&(n=""+n);var o=n.length;if(0===o)return 0;for(var u=!1;;)switch(i){case"ascii":case"latin1":case"binary":return o;case"utf8":case"utf-8":case void 0:return utf8ToBytes(n).length;case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return 2*o;case"hex":return o>>>1;case"base64":return base64ToBytes(n).length;default:if(u)return utf8ToBytes(n).length;i=(""+i).toLowerCase(),u=!0}}function slowToString(n,i,o){var u=!1;if((void 0===i||i<0)&&(i=0),i>this.length)return"";if((void 0===o||o>this.length)&&(o=this.length),o<=0)return"";if((o>>>=0)<=(i>>>=0))return"";for(n||(n="utf8");;)switch(n){case"hex":return hexSlice(this,i,o);case"utf8":case"utf-8":return utf8Slice(this,i,o);case"ascii":return asciiSlice(this,i,o);case"latin1":case"binary":return latin1Slice(this,i,o);case"base64":return base64Slice(this,i,o);case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return utf16leSlice(this,i,o);default:if(u)throw new TypeError("Unknown encoding: "+n);n=(n+"").toLowerCase(),u=!0}}function swap(n,i,o){var u=n[i];n[i]=n[o],n[o]=u}function bidirectionalIndexOf(n,i,o,u,p){if(0===n.length)return-1;if("string"==typeof o?(u=o,o=0):o>2147483647?o=2147483647:o<-2147483648&&(o=-2147483648),o=+o,isNaN(o)&&(o=p?0:n.length-1),o<0&&(o=n.length+o),o>=n.length){if(p)return-1;o=n.length-1}else if(o<0){if(!p)return-1;o=0}if("string"==typeof i&&(i=Buffer.from(i,u)),Buffer.isBuffer(i))return 0===i.length?-1:arrayIndexOf(n,i,o,u,p);if("number"==typeof i)return i&=255,Buffer.TYPED_ARRAY_SUPPORT&&"function"==typeof Uint8Array.prototype.indexOf?p?Uint8Array.prototype.indexOf.call(n,i,o):Uint8Array.prototype.lastIndexOf.call(n,i,o):arrayIndexOf(n,[i],o,u,p);throw new TypeError("val must be string, number or Buffer")}function arrayIndexOf(n,i,o,u,p){var g,y=1,w=n.length,x=i.length;if(void 0!==u&&("ucs2"===(u=String(u).toLowerCase())||"ucs-2"===u||"utf16le"===u||"utf-16le"===u)){if(n.length<2||i.length<2)return-1;y=2,w/=2,x/=2,o/=2}function read(n,i){return 1===y?n[i]:n.readUInt16BE(i*y)}if(p){var _=-1;for(g=o;g<w;g++)if(read(n,g)===read(i,-1===_?0:g-_)){if(-1===_&&(_=g),g-_+1===x)return _*y}else-1!==_&&(g-=g-_),_=-1}else for(o+x>w&&(o=w-x),g=o;g>=0;g--){for(var k=!0,P=0;P<x;P++)if(read(n,g+P)!==read(i,P)){k=!1;break}if(k)return g}return-1}function hexWrite(n,i,o,u){o=Number(o)||0;var p=n.length-o;u?(u=Number(u))>p&&(u=p):u=p;var g=i.length;if(g%2!=0)throw new TypeError("Invalid hex string");u>g/2&&(u=g/2);for(var y=0;y<u;++y){var w=parseInt(i.substr(2*y,2),16);if(isNaN(w))return y;n[o+y]=w}return y}function utf8Write(n,i,o,u){return blitBuffer(utf8ToBytes(i,n.length-o),n,o,u)}function asciiWrite(n,i,o,u){return blitBuffer(function asciiToBytes(n){for(var i=[],o=0;o<n.length;++o)i.push(255&n.charCodeAt(o));return i}(i),n,o,u)}function latin1Write(n,i,o,u){return asciiWrite(n,i,o,u)}function base64Write(n,i,o,u){return blitBuffer(base64ToBytes(i),n,o,u)}function ucs2Write(n,i,o,u){return blitBuffer(function utf16leToBytes(n,i){for(var o,u,p,g=[],y=0;y<n.length&&!((i-=2)<0);++y)o=n.charCodeAt(y),u=o>>8,p=o%256,g.push(p),g.push(u);return g}(i,n.length-o),n,o,u)}function base64Slice(n,i,o){return 0===i&&o===n.length?u.fromByteArray(n):u.fromByteArray(n.slice(i,o))}function utf8Slice(n,i,o){o=Math.min(n.length,o);for(var u=[],p=i;p<o;){var g,y,w,x,_=n[p],k=null,P=_>239?4:_>223?3:_>191?2:1;if(p+P<=o)switch(P){case 1:_<128&&(k=_);break;case 2:128==(192&(g=n[p+1]))&&(x=(31&_)<<6|63&g)>127&&(k=x);break;case 3:g=n[p+1],y=n[p+2],128==(192&g)&&128==(192&y)&&(x=(15&_)<<12|(63&g)<<6|63&y)>2047&&(x<55296||x>57343)&&(k=x);break;case 4:g=n[p+1],y=n[p+2],w=n[p+3],128==(192&g)&&128==(192&y)&&128==(192&w)&&(x=(15&_)<<18|(63&g)<<12|(63&y)<<6|63&w)>65535&&x<1114112&&(k=x)}null===k?(k=65533,P=1):k>65535&&(k-=65536,u.push(k>>>10&1023|55296),k=56320|1023&k),u.push(k),p+=P}return function decodeCodePointsArray(n){var i=n.length;if(i<=4096)return String.fromCharCode.apply(String,n);var o="",u=0;for(;u<i;)o+=String.fromCharCode.apply(String,n.slice(u,u+=4096));return o}(u)}i.Buffer=Buffer,i.SlowBuffer=function SlowBuffer(n){+n!=n&&(n=0);return Buffer.alloc(+n)},i.INSPECT_MAX_BYTES=50,Buffer.TYPED_ARRAY_SUPPORT=void 0!==n.TYPED_ARRAY_SUPPORT?n.TYPED_ARRAY_SUPPORT:function typedArraySupport(){try{var n=new Uint8Array(1);return n.__proto__={__proto__:Uint8Array.prototype,foo:function(){return 42}},42===n.foo()&&"function"==typeof n.subarray&&0===n.subarray(1,1).byteLength}catch(n){return!1}}(),i.kMaxLength=kMaxLength(),Buffer.poolSize=8192,Buffer._augment=function(n){return n.__proto__=Buffer.prototype,n},Buffer.from=function(n,i,o){return from(null,n,i,o)},Buffer.TYPED_ARRAY_SUPPORT&&(Buffer.prototype.__proto__=Uint8Array.prototype,Buffer.__proto__=Uint8Array,"undefined"!=typeof Symbol&&Symbol.species&&Buffer[Symbol.species]===Buffer&&Object.defineProperty(Buffer,Symbol.species,{value:null,configurable:!0})),Buffer.alloc=function(n,i,o){return function alloc(n,i,o,u){return assertSize(i),i<=0?createBuffer(n,i):void 0!==o?"string"==typeof u?createBuffer(n,i).fill(o,u):createBuffer(n,i).fill(o):createBuffer(n,i)}(null,n,i,o)},Buffer.allocUnsafe=function(n){return allocUnsafe(null,n)},Buffer.allocUnsafeSlow=function(n){return allocUnsafe(null,n)},Buffer.isBuffer=function isBuffer(n){return!(null==n||!n._isBuffer)},Buffer.compare=function compare(n,i){if(!Buffer.isBuffer(n)||!Buffer.isBuffer(i))throw new TypeError("Arguments must be Buffers");if(n===i)return 0;for(var o=n.length,u=i.length,p=0,g=Math.min(o,u);p<g;++p)if(n[p]!==i[p]){o=n[p],u=i[p];break}return o<u?-1:u<o?1:0},Buffer.isEncoding=function isEncoding(n){switch(String(n).toLowerCase()){case"hex":case"utf8":case"utf-8":case"ascii":case"latin1":case"binary":case"base64":case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return!0;default:return!1}},Buffer.concat=function concat(n,i){if(!g(n))throw new TypeError('"list" argument must be an Array of Buffers');if(0===n.length)return Buffer.alloc(0);var o;if(void 0===i)for(i=0,o=0;o<n.length;++o)i+=n[o].length;var u=Buffer.allocUnsafe(i),p=0;for(o=0;o<n.length;++o){var y=n[o];if(!Buffer.isBuffer(y))throw new TypeError('"list" argument must be an Array of Buffers');y.copy(u,p),p+=y.length}return u},Buffer.byteLength=byteLength,Buffer.prototype._isBuffer=!0,Buffer.prototype.swap16=function swap16(){var n=this.length;if(n%2!=0)throw new RangeError("Buffer size must be a multiple of 16-bits");for(var i=0;i<n;i+=2)swap(this,i,i+1);return this},Buffer.prototype.swap32=function swap32(){var n=this.length;if(n%4!=0)throw new RangeError("Buffer size must be a multiple of 32-bits");for(var i=0;i<n;i+=4)swap(this,i,i+3),swap(this,i+1,i+2);return this},Buffer.prototype.swap64=function swap64(){var n=this.length;if(n%8!=0)throw new RangeError("Buffer size must be a multiple of 64-bits");for(var i=0;i<n;i+=8)swap(this,i,i+7),swap(this,i+1,i+6),swap(this,i+2,i+5),swap(this,i+3,i+4);return this},Buffer.prototype.toString=function toString(){var n=0|this.length;return 0===n?"":0===arguments.length?utf8Slice(this,0,n):slowToString.apply(this,arguments)},Buffer.prototype.equals=function equals(n){if(!Buffer.isBuffer(n))throw new TypeError("Argument must be a Buffer");return this===n||0===Buffer.compare(this,n)},Buffer.prototype.inspect=function inspect(){var n="",o=i.INSPECT_MAX_BYTES;return this.length>0&&(n=this.toString("hex",0,o).match(/.{2}/g).join(" "),this.length>o&&(n+=" ... ")),"<Buffer "+n+">"},Buffer.prototype.compare=function compare(n,i,o,u,p){if(!Buffer.isBuffer(n))throw new TypeError("Argument must be a Buffer");if(void 0===i&&(i=0),void 0===o&&(o=n?n.length:0),void 0===u&&(u=0),void 0===p&&(p=this.length),i<0||o>n.length||u<0||p>this.length)throw new RangeError("out of range index");if(u>=p&&i>=o)return 0;if(u>=p)return-1;if(i>=o)return 1;if(this===n)return 0;for(var g=(p>>>=0)-(u>>>=0),y=(o>>>=0)-(i>>>=0),w=Math.min(g,y),x=this.slice(u,p),_=n.slice(i,o),k=0;k<w;++k)if(x[k]!==_[k]){g=x[k],y=_[k];break}return g<y?-1:y<g?1:0},Buffer.prototype.includes=function includes(n,i,o){return-1!==this.indexOf(n,i,o)},Buffer.prototype.indexOf=function indexOf(n,i,o){return bidirectionalIndexOf(this,n,i,o,!0)},Buffer.prototype.lastIndexOf=function lastIndexOf(n,i,o){return bidirectionalIndexOf(this,n,i,o,!1)},Buffer.prototype.write=function write(n,i,o,u){if(void 0===i)u="utf8",o=this.length,i=0;else if(void 0===o&&"string"==typeof i)u=i,o=this.length,i=0;else{if(!isFinite(i))throw new Error("Buffer.write(string, encoding, offset[, length]) is no longer supported");i|=0,isFinite(o)?(o|=0,void 0===u&&(u="utf8")):(u=o,o=void 0)}var p=this.length-i;if((void 0===o||o>p)&&(o=p),n.length>0&&(o<0||i<0)||i>this.length)throw new RangeError("Attempt to write outside buffer bounds");u||(u="utf8");for(var g=!1;;)switch(u){case"hex":return hexWrite(this,n,i,o);case"utf8":case"utf-8":return utf8Write(this,n,i,o);case"ascii":return asciiWrite(this,n,i,o);case"latin1":case"binary":return latin1Write(this,n,i,o);case"base64":return base64Write(this,n,i,o);case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return ucs2Write(this,n,i,o);default:if(g)throw new TypeError("Unknown encoding: "+u);u=(""+u).toLowerCase(),g=!0}},Buffer.prototype.toJSON=function toJSON(){return{type:"Buffer",data:Array.prototype.slice.call(this._arr||this,0)}};function asciiSlice(n,i,o){var u="";o=Math.min(n.length,o);for(var p=i;p<o;++p)u+=String.fromCharCode(127&n[p]);return u}function latin1Slice(n,i,o){var u="";o=Math.min(n.length,o);for(var p=i;p<o;++p)u+=String.fromCharCode(n[p]);return u}function hexSlice(n,i,o){var u=n.length;(!i||i<0)&&(i=0),(!o||o<0||o>u)&&(o=u);for(var p="",g=i;g<o;++g)p+=toHex(n[g]);return p}function utf16leSlice(n,i,o){for(var u=n.slice(i,o),p="",g=0;g<u.length;g+=2)p+=String.fromCharCode(u[g]+256*u[g+1]);return p}function checkOffset(n,i,o){if(n%1!=0||n<0)throw new RangeError("offset is not uint");if(n+i>o)throw new RangeError("Trying to access beyond buffer length")}function checkInt(n,i,o,u,p,g){if(!Buffer.isBuffer(n))throw new TypeError('"buffer" argument must be a Buffer instance');if(i>p||i<g)throw new RangeError('"value" argument is out of bounds');if(o+u>n.length)throw new RangeError("Index out of range")}function objectWriteUInt16(n,i,o,u){i<0&&(i=65535+i+1);for(var p=0,g=Math.min(n.length-o,2);p<g;++p)n[o+p]=(i&255<<8*(u?p:1-p))>>>8*(u?p:1-p)}function objectWriteUInt32(n,i,o,u){i<0&&(i=4294967295+i+1);for(var p=0,g=Math.min(n.length-o,4);p<g;++p)n[o+p]=i>>>8*(u?p:3-p)&255}function checkIEEE754(n,i,o,u,p,g){if(o+u>n.length)throw new RangeError("Index out of range");if(o<0)throw new RangeError("Index out of range")}function writeFloat(n,i,o,u,g){return g||checkIEEE754(n,0,o,4),p.write(n,i,o,u,23,4),o+4}function writeDouble(n,i,o,u,g){return g||checkIEEE754(n,0,o,8),p.write(n,i,o,u,52,8),o+8}Buffer.prototype.slice=function slice(n,i){var o,u=this.length;if((n=~~n)<0?(n+=u)<0&&(n=0):n>u&&(n=u),(i=void 0===i?u:~~i)<0?(i+=u)<0&&(i=0):i>u&&(i=u),i<n&&(i=n),Buffer.TYPED_ARRAY_SUPPORT)(o=this.subarray(n,i)).__proto__=Buffer.prototype;else{var p=i-n;o=new Buffer(p,void 0);for(var g=0;g<p;++g)o[g]=this[g+n]}return o},Buffer.prototype.readUIntLE=function readUIntLE(n,i,o){n|=0,i|=0,o||checkOffset(n,i,this.length);for(var u=this[n],p=1,g=0;++g<i&&(p*=256);)u+=this[n+g]*p;return u},Buffer.prototype.readUIntBE=function readUIntBE(n,i,o){n|=0,i|=0,o||checkOffset(n,i,this.length);for(var u=this[n+--i],p=1;i>0&&(p*=256);)u+=this[n+--i]*p;return u},Buffer.prototype.readUInt8=function readUInt8(n,i){return i||checkOffset(n,1,this.length),this[n]},Buffer.prototype.readUInt16LE=function readUInt16LE(n,i){return i||checkOffset(n,2,this.length),this[n]|this[n+1]<<8},Buffer.prototype.readUInt16BE=function readUInt16BE(n,i){return i||checkOffset(n,2,this.length),this[n]<<8|this[n+1]},Buffer.prototype.readUInt32LE=function readUInt32LE(n,i){return i||checkOffset(n,4,this.length),(this[n]|this[n+1]<<8|this[n+2]<<16)+16777216*this[n+3]},Buffer.prototype.readUInt32BE=function readUInt32BE(n,i){return i||checkOffset(n,4,this.length),16777216*this[n]+(this[n+1]<<16|this[n+2]<<8|this[n+3])},Buffer.prototype.readIntLE=function readIntLE(n,i,o){n|=0,i|=0,o||checkOffset(n,i,this.length);for(var u=this[n],p=1,g=0;++g<i&&(p*=256);)u+=this[n+g]*p;return u>=(p*=128)&&(u-=Math.pow(2,8*i)),u},Buffer.prototype.readIntBE=function readIntBE(n,i,o){n|=0,i|=0,o||checkOffset(n,i,this.length);for(var u=i,p=1,g=this[n+--u];u>0&&(p*=256);)g+=this[n+--u]*p;return g>=(p*=128)&&(g-=Math.pow(2,8*i)),g},Buffer.prototype.readInt8=function readInt8(n,i){return i||checkOffset(n,1,this.length),128&this[n]?-1*(255-this[n]+1):this[n]},Buffer.prototype.readInt16LE=function readInt16LE(n,i){i||checkOffset(n,2,this.length);var o=this[n]|this[n+1]<<8;return 32768&o?4294901760|o:o},Buffer.prototype.readInt16BE=function readInt16BE(n,i){i||checkOffset(n,2,this.length);var o=this[n+1]|this[n]<<8;return 32768&o?4294901760|o:o},Buffer.prototype.readInt32LE=function readInt32LE(n,i){return i||checkOffset(n,4,this.length),this[n]|this[n+1]<<8|this[n+2]<<16|this[n+3]<<24},Buffer.prototype.readInt32BE=function readInt32BE(n,i){return i||checkOffset(n,4,this.length),this[n]<<24|this[n+1]<<16|this[n+2]<<8|this[n+3]},Buffer.prototype.readFloatLE=function readFloatLE(n,i){return i||checkOffset(n,4,this.length),p.read(this,n,!0,23,4)},Buffer.prototype.readFloatBE=function readFloatBE(n,i){return i||checkOffset(n,4,this.length),p.read(this,n,!1,23,4)},Buffer.prototype.readDoubleLE=function readDoubleLE(n,i){return i||checkOffset(n,8,this.length),p.read(this,n,!0,52,8)},Buffer.prototype.readDoubleBE=function readDoubleBE(n,i){return i||checkOffset(n,8,this.length),p.read(this,n,!1,52,8)},Buffer.prototype.writeUIntLE=function writeUIntLE(n,i,o,u){(n=+n,i|=0,o|=0,u)||checkInt(this,n,i,o,Math.pow(2,8*o)-1,0);var p=1,g=0;for(this[i]=255&n;++g<o&&(p*=256);)this[i+g]=n/p&255;return i+o},Buffer.prototype.writeUIntBE=function writeUIntBE(n,i,o,u){(n=+n,i|=0,o|=0,u)||checkInt(this,n,i,o,Math.pow(2,8*o)-1,0);var p=o-1,g=1;for(this[i+p]=255&n;--p>=0&&(g*=256);)this[i+p]=n/g&255;return i+o},Buffer.prototype.writeUInt8=function writeUInt8(n,i,o){return n=+n,i|=0,o||checkInt(this,n,i,1,255,0),Buffer.TYPED_ARRAY_SUPPORT||(n=Math.floor(n)),this[i]=255&n,i+1},Buffer.prototype.writeUInt16LE=function writeUInt16LE(n,i,o){return n=+n,i|=0,o||checkInt(this,n,i,2,65535,0),Buffer.TYPED_ARRAY_SUPPORT?(this[i]=255&n,this[i+1]=n>>>8):objectWriteUInt16(this,n,i,!0),i+2},Buffer.prototype.writeUInt16BE=function writeUInt16BE(n,i,o){return n=+n,i|=0,o||checkInt(this,n,i,2,65535,0),Buffer.TYPED_ARRAY_SUPPORT?(this[i]=n>>>8,this[i+1]=255&n):objectWriteUInt16(this,n,i,!1),i+2},Buffer.prototype.writeUInt32LE=function writeUInt32LE(n,i,o){return n=+n,i|=0,o||checkInt(this,n,i,4,4294967295,0),Buffer.TYPED_ARRAY_SUPPORT?(this[i+3]=n>>>24,this[i+2]=n>>>16,this[i+1]=n>>>8,this[i]=255&n):objectWriteUInt32(this,n,i,!0),i+4},Buffer.prototype.writeUInt32BE=function writeUInt32BE(n,i,o){return n=+n,i|=0,o||checkInt(this,n,i,4,4294967295,0),Buffer.TYPED_ARRAY_SUPPORT?(this[i]=n>>>24,this[i+1]=n>>>16,this[i+2]=n>>>8,this[i+3]=255&n):objectWriteUInt32(this,n,i,!1),i+4},Buffer.prototype.writeIntLE=function writeIntLE(n,i,o,u){if(n=+n,i|=0,!u){var p=Math.pow(2,8*o-1);checkInt(this,n,i,o,p-1,-p)}var g=0,y=1,w=0;for(this[i]=255&n;++g<o&&(y*=256);)n<0&&0===w&&0!==this[i+g-1]&&(w=1),this[i+g]=(n/y>>0)-w&255;return i+o},Buffer.prototype.writeIntBE=function writeIntBE(n,i,o,u){if(n=+n,i|=0,!u){var p=Math.pow(2,8*o-1);checkInt(this,n,i,o,p-1,-p)}var g=o-1,y=1,w=0;for(this[i+g]=255&n;--g>=0&&(y*=256);)n<0&&0===w&&0!==this[i+g+1]&&(w=1),this[i+g]=(n/y>>0)-w&255;return i+o},Buffer.prototype.writeInt8=function writeInt8(n,i,o){return n=+n,i|=0,o||checkInt(this,n,i,1,127,-128),Buffer.TYPED_ARRAY_SUPPORT||(n=Math.floor(n)),n<0&&(n=255+n+1),this[i]=255&n,i+1},Buffer.prototype.writeInt16LE=function writeInt16LE(n,i,o){return n=+n,i|=0,o||checkInt(this,n,i,2,32767,-32768),Buffer.TYPED_ARRAY_SUPPORT?(this[i]=255&n,this[i+1]=n>>>8):objectWriteUInt16(this,n,i,!0),i+2},Buffer.prototype.writeInt16BE=function writeInt16BE(n,i,o){return n=+n,i|=0,o||checkInt(this,n,i,2,32767,-32768),Buffer.TYPED_ARRAY_SUPPORT?(this[i]=n>>>8,this[i+1]=255&n):objectWriteUInt16(this,n,i,!1),i+2},Buffer.prototype.writeInt32LE=function writeInt32LE(n,i,o){return n=+n,i|=0,o||checkInt(this,n,i,4,2147483647,-2147483648),Buffer.TYPED_ARRAY_SUPPORT?(this[i]=255&n,this[i+1]=n>>>8,this[i+2]=n>>>16,this[i+3]=n>>>24):objectWriteUInt32(this,n,i,!0),i+4},Buffer.prototype.writeInt32BE=function writeInt32BE(n,i,o){return n=+n,i|=0,o||checkInt(this,n,i,4,2147483647,-2147483648),n<0&&(n=4294967295+n+1),Buffer.TYPED_ARRAY_SUPPORT?(this[i]=n>>>24,this[i+1]=n>>>16,this[i+2]=n>>>8,this[i+3]=255&n):objectWriteUInt32(this,n,i,!1),i+4},Buffer.prototype.writeFloatLE=function writeFloatLE(n,i,o){return writeFloat(this,n,i,!0,o)},Buffer.prototype.writeFloatBE=function writeFloatBE(n,i,o){return writeFloat(this,n,i,!1,o)},Buffer.prototype.writeDoubleLE=function writeDoubleLE(n,i,o){return writeDouble(this,n,i,!0,o)},Buffer.prototype.writeDoubleBE=function writeDoubleBE(n,i,o){return writeDouble(this,n,i,!1,o)},Buffer.prototype.copy=function copy(n,i,o,u){if(o||(o=0),u||0===u||(u=this.length),i>=n.length&&(i=n.length),i||(i=0),u>0&&u<o&&(u=o),u===o)return 0;if(0===n.length||0===this.length)return 0;if(i<0)throw new RangeError("targetStart out of bounds");if(o<0||o>=this.length)throw new RangeError("sourceStart out of bounds");if(u<0)throw new RangeError("sourceEnd out of bounds");u>this.length&&(u=this.length),n.length-i<u-o&&(u=n.length-i+o);var p,g=u-o;if(this===n&&o<i&&i<u)for(p=g-1;p>=0;--p)n[p+i]=this[p+o];else if(g<1e3||!Buffer.TYPED_ARRAY_SUPPORT)for(p=0;p<g;++p)n[p+i]=this[p+o];else Uint8Array.prototype.set.call(n,this.subarray(o,o+g),i);return g},Buffer.prototype.fill=function fill(n,i,o,u){if("string"==typeof n){if("string"==typeof i?(u=i,i=0,o=this.length):"string"==typeof o&&(u=o,o=this.length),1===n.length){var p=n.charCodeAt(0);p<256&&(n=p)}if(void 0!==u&&"string"!=typeof u)throw new TypeError("encoding must be a string");if("string"==typeof u&&!Buffer.isEncoding(u))throw new TypeError("Unknown encoding: "+u)}else"number"==typeof n&&(n&=255);if(i<0||this.length<i||this.length<o)throw new RangeError("Out of range index");if(o<=i)return this;var g;if(i>>>=0,o=void 0===o?this.length:o>>>0,n||(n=0),"number"==typeof n)for(g=i;g<o;++g)this[g]=n;else{var y=Buffer.isBuffer(n)?n:utf8ToBytes(new Buffer(n,u).toString()),w=y.length;for(g=0;g<o-i;++g)this[g+i]=y[g%w]}return this};var y=/[^+\/0-9A-Za-z-_]/g;function toHex(n){return n<16?"0"+n.toString(16):n.toString(16)}function utf8ToBytes(n,i){var o;i=i||1/0;for(var u=n.length,p=null,g=[],y=0;y<u;++y){if((o=n.charCodeAt(y))>55295&&o<57344){if(!p){if(o>56319){(i-=3)>-1&&g.push(239,191,189);continue}if(y+1===u){(i-=3)>-1&&g.push(239,191,189);continue}p=o;continue}if(o<56320){(i-=3)>-1&&g.push(239,191,189),p=o;continue}o=65536+(p-55296<<10|o-56320)}else p&&(i-=3)>-1&&g.push(239,191,189);if(p=null,o<128){if((i-=1)<0)break;g.push(o)}else if(o<2048){if((i-=2)<0)break;g.push(o>>6|192,63&o|128)}else if(o<65536){if((i-=3)<0)break;g.push(o>>12|224,o>>6&63|128,63&o|128)}else{if(!(o<1114112))throw new Error("Invalid code point");if((i-=4)<0)break;g.push(o>>18|240,o>>12&63|128,o>>6&63|128,63&o|128)}}return g}function base64ToBytes(n){return u.toByteArray(function base64clean(n){if((n=function stringtrim(n){return n.trim?n.trim():n.replace(/^\s+|\s+$/g,"")}(n).replace(y,"")).length<2)return"";for(;n.length%4!=0;)n+="=";return n}(n))}function blitBuffer(n,i,o,u){for(var p=0;p<u&&!(p+o>=i.length||p>=n.length);++p)i[p+o]=n[p];return p}}).call(this,o(24))},function(n,i,o){var u=o(14);n.exports=function(n){if(!u(n))throw TypeError(String(n)+" is not an object");return n}},function(n,i,o){var u,p,g,y,w,x,_,k,P,E,O,B,I,D,R,N,U,W,G;n.exports=(u=o(2),o(49),void(u.lib.Cipher||(p=u,g=p.lib,y=g.Base,w=g.WordArray,x=g.BufferedBlockAlgorithm,_=p.enc,_.Utf8,k=_.Base64,P=p.algo.EvpKDF,E=g.Cipher=x.extend({cfg:y.extend(),createEncryptor:function(n,i){return this.create(this._ENC_XFORM_MODE,n,i)},createDecryptor:function(n,i){return this.create(this._DEC_XFORM_MODE,n,i)},init:function(n,i,o){this.cfg=this.cfg.extend(o),this._xformMode=n,this._key=i,this.reset()},reset:function(){x.reset.call(this),this._doReset()},process:function(n){return this._append(n),this._process()},finalize:function(n){return n&&this._append(n),this._doFinalize()},keySize:4,ivSize:4,_ENC_XFORM_MODE:1,_DEC_XFORM_MODE:2,_createHelper:function(){function selectCipherStrategy(n){return"string"==typeof n?G:U}return function(n){return{encrypt:function(i,o,u){return selectCipherStrategy(o).encrypt(n,i,o,u)},decrypt:function(i,o,u){return selectCipherStrategy(o).decrypt(n,i,o,u)}}}}()}),g.StreamCipher=E.extend({_doFinalize:function(){return this._process(!0)},blockSize:1}),O=p.mode={},B=g.BlockCipherMode=y.extend({createEncryptor:function(n,i){return this.Encryptor.create(n,i)},createDecryptor:function(n,i){return this.Decryptor.create(n,i)},init:function(n,i){this._cipher=n,this._iv=i}}),I=O.CBC=function(){var n=B.extend();function xorBlock(n,i,o){var u=this._iv;if(u){var p=u;this._iv=void 0}else p=this._prevBlock;for(var g=0;g<o;g++)n[i+g]^=p[g]}return n.Encryptor=n.extend({processBlock:function(n,i){var o=this._cipher,u=o.blockSize;xorBlock.call(this,n,i,u),o.encryptBlock(n,i),this._prevBlock=n.slice(i,i+u)}}),n.Decryptor=n.extend({processBlock:function(n,i){var o=this._cipher,u=o.blockSize,p=n.slice(i,i+u);o.decryptBlock(n,i),xorBlock.call(this,n,i,u),this._prevBlock=p}}),n}(),D=(p.pad={}).Pkcs7={pad:function(n,i){for(var o=4*i,u=o-n.sigBytes%o,p=u<<24|u<<16|u<<8|u,g=[],y=0;y<u;y+=4)g.push(p);var x=w.create(g,u);n.concat(x)},unpad:function(n){var i=255&n.words[n.sigBytes-1>>>2];n.sigBytes-=i}},g.BlockCipher=E.extend({cfg:E.cfg.extend({mode:I,padding:D}),reset:function(){E.reset.call(this);var n=this.cfg,i=n.iv,o=n.mode;if(this._xformMode==this._ENC_XFORM_MODE)var u=o.createEncryptor;else u=o.createDecryptor,this._minBufferSize=1;this._mode&&this._mode.__creator==u?this._mode.init(this,i&&i.words):(this._mode=u.call(o,this,i&&i.words),this._mode.__creator=u)},_doProcessBlock:function(n,i){this._mode.processBlock(n,i)},_doFinalize:function(){var n=this.cfg.padding;if(this._xformMode==this._ENC_XFORM_MODE){n.pad(this._data,this.blockSize);var i=this._process(!0)}else i=this._process(!0),n.unpad(i);return i},blockSize:4}),R=g.CipherParams=y.extend({init:function(n){this.mixIn(n)},toString:function(n){return(n||this.formatter).stringify(this)}}),N=(p.format={}).OpenSSL={stringify:function(n){var i=n.ciphertext,o=n.salt;if(o)var u=w.create([1398893684,1701076831]).concat(o).concat(i);else u=i;return u.toString(k)},parse:function(n){var i=k.parse(n),o=i.words;if(1398893684==o[0]&&1701076831==o[1]){var u=w.create(o.slice(2,4));o.splice(0,4),i.sigBytes-=16}return R.create({ciphertext:i,salt:u})}},U=g.SerializableCipher=y.extend({cfg:y.extend({format:N}),encrypt:function(n,i,o,u){u=this.cfg.extend(u);var p=n.createEncryptor(o,u),g=p.finalize(i),y=p.cfg;return R.create({ciphertext:g,key:o,iv:y.iv,algorithm:n,mode:y.mode,padding:y.padding,blockSize:n.blockSize,formatter:u.format})},decrypt:function(n,i,o,u){return u=this.cfg.extend(u),i=this._parse(i,u.format),n.createDecryptor(o,u).finalize(i.ciphertext)},_parse:function(n,i){return"string"==typeof n?i.parse(n,this):n}}),W=(p.kdf={}).OpenSSL={execute:function(n,i,o,u){u||(u=w.random(8));var p=P.create({keySize:i+o}).compute(n,u),g=w.create(p.words.slice(i),4*o);return p.sigBytes=4*i,R.create({key:p,iv:g,salt:u})}},G=g.PasswordBasedCipher=U.extend({cfg:U.cfg.extend({kdf:W}),encrypt:function(n,i,o,u){var p=(u=this.cfg.extend(u)).kdf.execute(o,n.keySize,n.ivSize);u.iv=p.iv;var g=U.encrypt.call(this,n,i,p.key,u);return g.mixIn(p),g},decrypt:function(n,i,o,u){u=this.cfg.extend(u),i=this._parse(i,u.format);var p=u.kdf.execute(o,n.keySize,n.ivSize,i.salt);return u.iv=p.iv,U.decrypt.call(this,n,i,p.key,u)}}))))},function(n,i,o){var u=o(4);n.exports=!u((function(){return 7!=Object.defineProperty({},1,{get:function(){return 7}})[1]}))},function(n,i){n.exports=function(n){return"object"==typeof n?null!==n:"function"==typeof n}},function(n,i){var o={}.hasOwnProperty;n.exports=function(n,i){return o.call(n,i)}},function(n,i,o){var u=o(13),p=o(202),g=o(11),y=o(55),w=Object.defineProperty;i.f=u?w:function defineProperty(n,i,o){if(g(n),i=y(i,!0),g(o),p)try{return w(n,i,o)}catch(n){}if("get"in o||"set"in o)throw TypeError("Accessors not supported");return"value"in o&&(n[i]=o.value),n}},function(n,i){var o=n.exports="undefined"!=typeof window&&window.Math==Math?window:"undefined"!=typeof self&&self.Math==Math?self:Function("return this")();"number"==typeof __g&&(__g=o)},function(n,i,o){var u=o(13),p=o(16),g=o(40);n.exports=u?function(n,i,o){return p.f(n,i,g(1,o))}:function(n,i,o){return n[i]=o,n}},function(n,i,o){var u=o(33);n.exports=function(n){return Object(u(n))}},function(n,i,o){var u=o(96),p=o(92),g=o(19),y=o(8),w=o(209),x=[].push,createMethod=function(n){var i=1==n,o=2==n,_=3==n,k=4==n,P=6==n,E=7==n,O=5==n||P;return function(B,I,D,R){for(var N,U,W=g(B),G=p(W),j=u(I,D,3),X=y(G.length),Y=0,K=R||w,J=i?K(B,X):o||E?K(B,0):void 0;X>Y;Y++)if((O||Y in G)&&(U=j(N=G[Y],Y,W),n))if(i)J[Y]=U;else if(U)switch(n){case 3:return!0;case 5:return N;case 6:return Y;case 2:x.call(J,N)}else switch(n){case 4:return!1;case 7:x.call(J,N)}return P?-1:_||k?k:J}};n.exports={forEach:createMethod(0),map:createMethod(1),filter:createMethod(2),some:createMethod(3),every:createMethod(4),find:createMethod(5),findIndex:createMethod(6),filterOut:createMethod(7)}},function(n,i,o){var u=o(182)("wks"),p=o(131),g=o(17).Symbol,y="function"==typeof g;(n.exports=function(n){return u[n]||(u[n]=y&&g[n]||(y?g:p)("Symbol."+n))}).store=u},function(n,i){n.exports=function(n){return"object"==typeof n?null!==n:"function"==typeof n}},function(n,i,o){n.exports=!o(52)((function(){return 7!=Object.defineProperty({},"a",{get:function(){return 7}}).a}))},function(n,i){var o;o=function(){return this}();try{o=o||new Function("return this")()}catch(n){"object"==typeof window&&(o=window)}n.exports=o},function(n,i,o){var u=o(3),p=o(18),g=o(15),y=o(139),w=o(140),x=o(42),_=x.get,k=x.enforce,P=String(String).split("String");(n.exports=function(n,i,o,w){var x,_=!!w&&!!w.unsafe,E=!!w&&!!w.enumerable,O=!!w&&!!w.noTargetGet;"function"==typeof o&&("string"!=typeof i||g(o,"name")||p(o,"name",i),(x=k(o)).source||(x.source=P.join("string"==typeof i?i:""))),n!==u?(_?!O&&n[i]&&(E=!0):delete n[i],E?n[i]=o:p(n,i,o)):E?n[i]=o:y(i,o)})(Function.prototype,"toString",(function toString(){return"function"==typeof this&&_(this).source||w(this)}))},function(n,i,o){var u=o(29),p=o(255),g=o(176),y=Object.defineProperty;i.f=o(23)?Object.defineProperty:function defineProperty(n,i,o){if(u(n),i=g(i,!0),u(o),p)try{return y(n,i,o)}catch(n){}if("get"in o||"set"in o)throw TypeError("Accessors not supported!");return"value"in o&&(n[i]=o.value),n}},function(n,i,o){var u=o(92),p=o(33);n.exports=function(n){return u(p(n))}},function(n,i,o){"use strict";var u=o(27),p=o(151),g=o(97),y=o(42),w=o(217),x=y.set,_=y.getterFor("Array Iterator");n.exports=w(Array,"Array",(function(n,i){x(this,{type:"Array Iterator",target:u(n),index:0,kind:i})}),(function(){var n=_(this),i=n.target,o=n.kind,u=n.index++;return!i||u>=i.length?(n.target=void 0,{value:void 0,done:!0}):"keys"==o?{value:u,done:!1}:"values"==o?{value:i[u],done:!1}:{value:[u,i[u]],done:!1}}),"values"),g.Arguments=g.Array,p("keys"),p("values"),p("entries")},function(n,i,o){var u=o(22);n.exports=function(n){if(!u(n))throw TypeError(n+" is not an object!");return n}},function(n,i,o){var u=o(13),p=o(4),g=o(15),y=Object.defineProperty,w={},thrower=function(n){throw n};n.exports=function(n,i){if(g(w,n))return w[n];i||(i={});var o=[][n],x=!!g(i,"ACCESSORS")&&i.ACCESSORS,_=g(i,0)?i[0]:thrower,k=g(i,1)?i[1]:void 0;return w[n]=!!o&&!p((function(){if(x&&!u)return!0;var n={length:-1};x?y(n,1,{enumerable:!0,get:thrower}):n[1]=1,o.call(n,_,k)}))}},function(n,i,o){var u=o(154),p=o(25),g=o(332);u||p(Object.prototype,"toString",g,{unsafe:!0})},function(n,i,o){var u=o(101);n.exports=function(n,i,o){if(u(n),void 0===i)return n;switch(o){case 1:return function(o){return n.call(i,o)};case 2:return function(o,u){return n.call(i,o,u)};case 3:return function(o,u,p){return n.call(i,o,u,p)}}return function(){return n.apply(i,arguments)}}},function(n,i){n.exports=function(n){if(null==n)throw TypeError("Can't call method on "+n);return n}},function(n,i,o){var u=o(114),p=o(3),aFunction=function(n){return"function"==typeof n?n:void 0};n.exports=function(n,i){return arguments.length<2?aFunction(u[n])||aFunction(p[n]):u[n]&&u[n][i]||p[n]&&p[n][i]}},function(n,i){var o=Math.ceil,u=Math.floor;n.exports=function(n){return isNaN(n=+n)?0:(n>0?u:o)(n)}},function(n,i,o){var u=o(11),p=o(44),g=o(6)("species");n.exports=function(n,i){var o,y=u(n).constructor;return void 0===y||null==(o=u(y)[g])?i:p(o)}},function(n,i,o){"use strict";(function(i){var u,p=o(10),g=p.Buffer,y={};for(u in p)p.hasOwnProperty(u)&&"SlowBuffer"!==u&&"Buffer"!==u&&(y[u]=p[u]);var w=y.Buffer={};for(u in g)g.hasOwnProperty(u)&&"allocUnsafe"!==u&&"allocUnsafeSlow"!==u&&(w[u]=g[u]);if(y.Buffer.prototype=g.prototype,w.from&&w.from!==Uint8Array.from||(w.from=function(n,i,o){if("number"==typeof n)throw new TypeError('The "value" argument must not be of type number. Received type '+typeof n);if(n&&void 0===n.length)throw new TypeError("The first argument must be one of type string, Buffer, ArrayBuffer, Array, or Array-like Object. Received type "+typeof n);return g(n,i,o)}),w.alloc||(w.alloc=function(n,i,o){if("number"!=typeof n)throw new TypeError('The "size" argument must be of type number. Received type '+typeof n);if(n<0||n>=2*(1<<30))throw new RangeError('The value "'+n+'" is invalid for option "size"');var u=g(n);return i&&0!==i.length?"string"==typeof o?u.fill(i,o):u.fill(i):u.fill(0),u}),!y.kStringMaxLength)try{y.kStringMaxLength=i.binding("buffer").kStringMaxLength}catch(n){}y.constants||(y.constants={MAX_LENGTH:y.kMaxLength},y.kStringMaxLength&&(y.constants.MAX_STRING_LENGTH=y.kStringMaxLength)),n.exports=y}).call(this,o(47))},function(n,i,o){(function(){var n,u;n=o(89).Number,i.resolveLength=function(i,o,u){var p;if("number"==typeof i?p=i:"function"==typeof i?p=i.call(u,u):u&&"string"==typeof i?p=u[i]:o&&i instanceof n&&(p=i.decode(o)),isNaN(p))throw new Error("Not a fixed size");return p},u=function u(n){var i,o;for(i in null==n&&(n={}),this.enumerable=!0,this.configurable=!0,n)o=n[i],this[i]=o},i.PropertyDescriptor=u}).call(this)},function(n,i,o){var u=o(26),p=o(100);n.exports=o(23)?function(n,i,o){return u.f(n,i,p(1,o))}:function(n,i,o){return n[i]=o,n}},function(n,i){n.exports=function(n,i){return{enumerable:!(1&n),configurable:!(2&n),writable:!(4&n),value:i}}},function(n,i){var o={}.toString;n.exports=function(n){return o.call(n).slice(8,-1)}},function(n,i,o){var u,p,g,y=o(310),w=o(3),x=o(14),_=o(18),k=o(15),P=o(141),E=o(111),O=o(113),B=w.WeakMap;if(y){var I=P.state||(P.state=new B),D=I.get,R=I.has,N=I.set;u=function(n,i){return i.facade=n,N.call(I,n,i),i},p=function(n){return D.call(I,n)||{}},g=function(n){return R.call(I,n)}}else{var U=E("state");O[U]=!0,u=function(n,i){return i.facade=n,_(n,U,i),i},p=function(n){return k(n,U)?n[U]:{}},g=function(n){return k(n,U)}}n.exports={set:u,get:p,has:g,enforce:function(n){return g(n)?p(n):u(n,{})},getterFor:function(n){return function(i){var o;if(!x(i)||(o=p(i)).type!==n)throw TypeError("Incompatible receiver, "+n+" required");return o}}}},function(n,i,o){var u=o(35),p=Math.max,g=Math.min;n.exports=function(n,i){var o=u(n);return o<0?p(o+i,0):g(o,i)}},function(n,i){n.exports=function(n){if("function"!=typeof n)throw TypeError(String(n)+" is not a function");return n}},function(n,i,o){var u=o(11),p=o(321);n.exports=Object.setPrototypeOf||("__proto__"in{}?function(){var n,i=!1,o={};try{(n=Object.getOwnPropertyDescriptor(Object.prototype,"__proto__").set).call(o,[]),i=o instanceof Array}catch(n){}return function setPrototypeOf(o,g){return u(o),p(g),i?n.call(o,g):o.__proto__=g,o}}():void 0)},function(n,i,o){"use strict";var u=o(1),p=o(14),g=o(115),y=o(43),w=o(8),x=o(27),_=o(147),k=o(6),P=o(116),E=o(30),O=P("slice"),B=E("slice",{ACCESSORS:!0,0:0,1:2}),I=k("species"),D=[].slice,R=Math.max;u({target:"Array",proto:!0,forced:!O||!B},{slice:function slice(n,i){var o,u,k,P=x(this),E=w(P.length),O=y(n,E),B=y(void 0===i?E:i,E);if(g(P)&&("function"!=typeof(o=P.constructor)||o!==Array&&!g(o.prototype)?p(o)&&null===(o=o[I])&&(o=void 0):o=void 0,o===Array||void 0===o))return D.call(P,O,B);for(u=new(void 0===o?Array:o)(R(B-O,0)),k=0;O<B;O++,k++)O in P&&_(u,k,P[O]);return u.length=k,u}})},function(n,i){var o,u,p=n.exports={};function defaultSetTimout(){throw new Error("setTimeout has not been defined")}function defaultClearTimeout(){throw new Error("clearTimeout has not been defined")}function runTimeout(n){if(o===setTimeout)return setTimeout(n,0);if((o===defaultSetTimout||!o)&&setTimeout)return o=setTimeout,setTimeout(n,0);try{return o(n,0)}catch(i){try{return o.call(null,n,0)}catch(i){return o.call(this,n,0)}}}!function(){try{o="function"==typeof setTimeout?setTimeout:defaultSetTimout}catch(n){o=defaultSetTimout}try{u="function"==typeof clearTimeout?clearTimeout:defaultClearTimeout}catch(n){u=defaultClearTimeout}}();var g,y=[],w=!1,x=-1;function cleanUpNextTick(){w&&g&&(w=!1,g.length?y=g.concat(y):x=-1,y.length&&drainQueue())}function drainQueue(){if(!w){var n=runTimeout(cleanUpNextTick);w=!0;for(var i=y.length;i;){for(g=y,y=[];++x<i;)g&&g[x].run();x=-1,i=y.length}g=null,w=!1,function runClearTimeout(n){if(u===clearTimeout)return clearTimeout(n);if((u===defaultClearTimeout||!u)&&clearTimeout)return u=clearTimeout,clearTimeout(n);try{return u(n)}catch(i){try{return u.call(null,n)}catch(i){return u.call(this,n)}}}(n)}}function Item(n,i){this.fun=n,this.array=i}function noop(){}p.nextTick=function(n){var i=new Array(arguments.length-1);if(arguments.length>1)for(var o=1;o<arguments.length;o++)i[o-1]=arguments[o];y.push(new Item(n,i)),1!==y.length||w||runTimeout(drainQueue)},Item.prototype.run=function(){this.fun.apply(null,this.array)},p.title="browser",p.browser=!0,p.env={},p.argv=[],p.version="",p.versions={},p.on=noop,p.addListener=noop,p.once=noop,p.off=noop,p.removeListener=noop,p.removeAllListeners=noop,p.emit=noop,p.prependListener=noop,p.prependOnceListener=noop,p.listeners=function(n){return[]},p.binding=function(n){throw new Error("process.binding is not supported")},p.cwd=function(){return"/"},p.chdir=function(n){throw new Error("process.chdir is not supported")},p.umask=function(){return 0}},function(n,i,o){"use strict";var u=o(123),p=Object.keys||function(n){var i=[];for(var o in n)i.push(o);return i};n.exports=Duplex;var g=Object.create(o(98));g.inherits=o(86);var y=o(241),w=o(166);g.inherits(Duplex,y);for(var x=p(w.prototype),_=0;_<x.length;_++){var k=x[_];Duplex.prototype[k]||(Duplex.prototype[k]=w.prototype[k])}function Duplex(n){if(!(this instanceof Duplex))return new Duplex(n);y.call(this,n),w.call(this,n),n&&!1===n.readable&&(this.readable=!1),n&&!1===n.writable&&(this.writable=!1),this.allowHalfOpen=!0,n&&!1===n.allowHalfOpen&&(this.allowHalfOpen=!1),this.once("end",onend)}function onend(){this.allowHalfOpen||this._writableState.ended||u.nextTick(onEndNT,this)}function onEndNT(n){n.end()}Object.defineProperty(Duplex.prototype,"writableHighWaterMark",{enumerable:!1,get:function(){return this._writableState.highWaterMark}}),Object.defineProperty(Duplex.prototype,"destroyed",{get:function(){return void 0!==this._readableState&&void 0!==this._writableState&&(this._readableState.destroyed&&this._writableState.destroyed)},set:function(n){void 0!==this._readableState&&void 0!==this._writableState&&(this._readableState.destroyed=n,this._writableState.destroyed=n)}}),Duplex.prototype._destroy=function(n,i){this.push(null),this.end(),u.nextTick(i,n)}},function(n,i,o){var u,p,g,y,w,x,_,k;n.exports=(k=o(2),o(168),o(169),p=(u=k).lib,g=p.Base,y=p.WordArray,w=u.algo,x=w.MD5,_=w.EvpKDF=g.extend({cfg:g.extend({keySize:4,hasher:x,iterations:1}),init:function(n){this.cfg=this.cfg.extend(n)},compute:function(n,i){for(var o=this.cfg,u=o.hasher.create(),p=y.create(),g=p.words,w=o.keySize,x=o.iterations;g.length<w;){_&&u.update(_);var _=u.update(n).finalize(i);u.reset();for(var k=1;k<x;k++)_=u.finalize(_),u.reset();p.concat(_)}return p.sigBytes=4*w,p}}),u.EvpKDF=function(n,i,o){return _.create(o).compute(n,i)},k.EvpKDF)},function(n,i,o){var u=o(173),p=o(174);n.exports=function(n){return u(p(n))}},function(n,i){var o={}.hasOwnProperty;n.exports=function(n,i){return o.call(n,i)}},function(n,i){n.exports=function(n){try{return!!n()}catch(n){return!0}}},function(n,i,o){"use strict";(function(i,o){function VirtualFileSystem(){this.fileSystem={},this.dataSystem={}}function fixFilename(n){return 0===n.indexOf(o)&&(n=n.substring(o.length)),0===n.indexOf("/")&&(n=n.substring(1)),n}VirtualFileSystem.prototype.existsSync=function(n){return n=fixFilename(n),void 0!==this.fileSystem[n]||void 0!==this.dataSystem[n]},VirtualFileSystem.prototype.readFileSync=function(n,o){n=fixFilename(n);var u=this.dataSystem[n];if("string"==typeof u&&"utf8"===o)return u;if(u)return new i(u,"string"==typeof u?"base64":void 0);var p=this.fileSystem[n];if(p)return p;throw"File '"+n+"' not found in virtual file system"},VirtualFileSystem.prototype.writeFileSync=function(n,i){this.fileSystem[fixFilename(n)]=i},VirtualFileSystem.prototype.bindFS=function(n){this.dataSystem=n||{}},n.exports=new VirtualFileSystem}).call(this,o(10).Buffer,"/")},function(n,i,o){var u=o(13),p=o(110),g=o(40),y=o(27),w=o(55),x=o(15),_=o(202),k=Object.getOwnPropertyDescriptor;i.f=u?k:function getOwnPropertyDescriptor(n,i){if(n=y(n),i=w(i,!0),_)try{return k(n,i)}catch(n){}if(x(n,i))return g(!p.f.call(n,i),n[i])}},function(n,i,o){var u=o(14);n.exports=function(n,i){if(!u(n))return n;var o,p;if(i&&"function"==typeof(o=n.toString)&&!u(p=o.call(n)))return p;if("function"==typeof(o=n.valueOf)&&!u(p=o.call(n)))return p;if(!i&&"function"==typeof(o=n.toString)&&!u(p=o.call(n)))return p;throw TypeError("Can't convert object to primitive value")}},function(n,i){n.exports=!1},function(n,i,o){var u=o(204),p=o(143).concat("length","prototype");i.f=Object.getOwnPropertyNames||function getOwnPropertyNames(n){return u(n,p)}},function(n,i,o){var u,p=o(11),g=o(312),y=o(143),w=o(113),x=o(206),_=o(138),k=o(111),P=k("IE_PROTO"),EmptyConstructor=function(){},scriptTag=function(n){return"<script>"+n+"<\/script>"},NullProtoObject=function(){try{u=document.domain&&new ActiveXObject("htmlfile")}catch(n){}var n,i;NullProtoObject=u?function(n){n.write(scriptTag("")),n.close();var i=n.parentWindow.Object;return n=null,i}(u):((i=_("iframe")).style.display="none",x.appendChild(i),i.src=String("javascript:"),(n=i.contentWindow.document).open(),n.write(scriptTag("document.F=Object")),n.close(),n.F);for(var o=y.length;o--;)delete NullProtoObject.prototype[y[o]];return NullProtoObject()};w[P]=!0,n.exports=Object.create||function create(n,i){var o;return null!==n?(EmptyConstructor.prototype=p(n),o=new EmptyConstructor,EmptyConstructor.prototype=null,o[P]=n):o=NullProtoObject(),void 0===i?o:g(o,i)}},function(n,i,o){"use strict";var u=o(4);n.exports=function(n,i){var o=[][n];return!!o&&u((function(){o.call(null,i||function(){throw 1},1)}))}},function(n,i,o){var u=o(15),p=o(19),g=o(111),y=o(219),w=g("IE_PROTO"),x=Object.prototype;n.exports=y?Object.getPrototypeOf:function(n){return n=p(n),u(n,w)?n[w]:"function"==typeof n.constructor&&n instanceof n.constructor?n.constructor.prototype:n instanceof Object?x:null}},function(n,i,o){"use strict";var u=o(1),p=o(4),g=o(156),y=o(11),w=o(43),x=o(8),_=o(36),k=g.ArrayBuffer,P=g.DataView,E=k.prototype.slice;u({target:"ArrayBuffer",proto:!0,unsafe:!0,forced:p((function(){return!new k(2).slice(1,void 0).byteLength}))},{slice:function slice(n,i){if(void 0!==E&&void 0===i)return E.call(y(this),n);for(var o=y(this).byteLength,u=w(n,o),p=w(void 0===i?o:i,o),g=new(_(this,k))(x(p-u)),O=new P(this),B=new P(g),I=0;u<p;)B.setUint8(I++,O.getUint8(u++));return g}})},function(n,i,o){o(237)("Uint8",(function(n){return function Uint8Array(i,o,u){return n(this,i,o,u)}}))},function(n,i,o){"use strict";var u=o(5),p=o(344),g=u.aTypedArray;(0,u.exportTypedArrayMethod)("copyWithin",(function copyWithin(n,i){return p.call(g(this),n,i,arguments.length>2?arguments[2]:void 0)}))},function(n,i,o){"use strict";var u=o(5),p=o(20).every,g=u.aTypedArray;(0,u.exportTypedArrayMethod)("every",(function every(n){return p(g(this),n,arguments.length>1?arguments[1]:void 0)}))},function(n,i,o){"use strict";var u=o(5),p=o(150),g=u.aTypedArray;(0,u.exportTypedArrayMethod)("fill",(function fill(n){return p.apply(g(this),arguments)}))},function(n,i,o){"use strict";var u=o(5),p=o(20).filter,g=o(36),y=u.aTypedArray,w=u.aTypedArrayConstructor;(0,u.exportTypedArrayMethod)("filter",(function filter(n){for(var i=p(y(this),n,arguments.length>1?arguments[1]:void 0),o=g(this,this.constructor),u=0,x=i.length,_=new(w(o))(x);x>u;)_[u]=i[u++];return _}))},function(n,i,o){"use strict";var u=o(5),p=o(20).find,g=u.aTypedArray;(0,u.exportTypedArrayMethod)("find",(function find(n){return p(g(this),n,arguments.length>1?arguments[1]:void 0)}))},function(n,i,o){"use strict";var u=o(5),p=o(20).findIndex,g=u.aTypedArray;(0,u.exportTypedArrayMethod)("findIndex",(function findIndex(n){return p(g(this),n,arguments.length>1?arguments[1]:void 0)}))},function(n,i,o){"use strict";var u=o(5),p=o(20).forEach,g=u.aTypedArray;(0,u.exportTypedArrayMethod)("forEach",(function forEach(n){p(g(this),n,arguments.length>1?arguments[1]:void 0)}))},function(n,i,o){"use strict";var u=o(5),p=o(93).includes,g=u.aTypedArray;(0,u.exportTypedArrayMethod)("includes",(function includes(n){return p(g(this),n,arguments.length>1?arguments[1]:void 0)}))},function(n,i,o){"use strict";var u=o(5),p=o(93).indexOf,g=u.aTypedArray;(0,u.exportTypedArrayMethod)("indexOf",(function indexOf(n){return p(g(this),n,arguments.length>1?arguments[1]:void 0)}))},function(n,i,o){"use strict";var u=o(3),p=o(5),g=o(28),y=o(6)("iterator"),w=u.Uint8Array,x=g.values,_=g.keys,k=g.entries,P=p.aTypedArray,E=p.exportTypedArrayMethod,O=w&&w.prototype[y],B=!!O&&("values"==O.name||null==O.name),I=function values(){return x.call(P(this))};E("entries",(function entries(){return k.call(P(this))})),E("keys",(function keys(){return _.call(P(this))})),E("values",I,!B),E(y,I,!B)},function(n,i,o){"use strict";var u=o(5),p=u.aTypedArray,g=u.exportTypedArrayMethod,y=[].join;g("join",(function join(n){return y.apply(p(this),arguments)}))},function(n,i,o){"use strict";var u=o(5),p=o(345),g=u.aTypedArray;(0,u.exportTypedArrayMethod)("lastIndexOf",(function lastIndexOf(n){return p.apply(g(this),arguments)}))},function(n,i,o){"use strict";var u=o(5),p=o(20).map,g=o(36),y=u.aTypedArray,w=u.aTypedArrayConstructor;(0,u.exportTypedArrayMethod)("map",(function map(n){return p(y(this),n,arguments.length>1?arguments[1]:void 0,(function(n,i){return new(w(g(n,n.constructor)))(i)}))}))},function(n,i,o){"use strict";var u=o(5),p=o(239).left,g=u.aTypedArray;(0,u.exportTypedArrayMethod)("reduce",(function reduce(n){return p(g(this),n,arguments.length,arguments.length>1?arguments[1]:void 0)}))},function(n,i,o){"use strict";var u=o(5),p=o(239).right,g=u.aTypedArray;(0,u.exportTypedArrayMethod)("reduceRight",(function reduceRight(n){return p(g(this),n,arguments.length,arguments.length>1?arguments[1]:void 0)}))},function(n,i,o){"use strict";var u=o(5),p=u.aTypedArray,g=u.exportTypedArrayMethod,y=Math.floor;g("reverse",(function reverse(){for(var n,i=p(this).length,o=y(i/2),u=0;u<o;)n=this[u],this[u++]=this[--i],this[i]=n;return this}))},function(n,i,o){"use strict";var u=o(5),p=o(8),g=o(238),y=o(19),w=o(4),x=u.aTypedArray;(0,u.exportTypedArrayMethod)("set",(function set(n){x(this);var i=g(arguments.length>1?arguments[1]:void 0,1),o=this.length,u=y(n),w=p(u.length),_=0;if(w+i>o)throw RangeError("Wrong length");for(;_<w;)this[i+_]=u[_++]}),w((function(){new Int8Array(1).set({})})))},function(n,i,o){"use strict";var u=o(5),p=o(36),g=o(4),y=u.aTypedArray,w=u.aTypedArrayConstructor,x=u.exportTypedArrayMethod,_=[].slice;x("slice",(function slice(n,i){for(var o=_.call(y(this),n,i),u=p(this,this.constructor),g=0,x=o.length,k=new(w(u))(x);x>g;)k[g]=o[g++];return k}),g((function(){new Int8Array(1).slice()})))},function(n,i,o){"use strict";var u=o(5),p=o(20).some,g=u.aTypedArray;(0,u.exportTypedArrayMethod)("some",(function some(n){return p(g(this),n,arguments.length>1?arguments[1]:void 0)}))},function(n,i,o){"use strict";var u=o(5),p=u.aTypedArray,g=u.exportTypedArrayMethod,y=[].sort;g("sort",(function sort(n){return y.call(p(this),n)}))},function(n,i,o){"use strict";var u=o(5),p=o(8),g=o(43),y=o(36),w=u.aTypedArray;(0,u.exportTypedArrayMethod)("subarray",(function subarray(n,i){var o=w(this),u=o.length,x=g(n,u);return new(y(o,o.constructor))(o.buffer,o.byteOffset+x*o.BYTES_PER_ELEMENT,p((void 0===i?u:g(i,u))-x))}))},function(n,i,o){"use strict";var u=o(3),p=o(5),g=o(4),y=u.Int8Array,w=p.aTypedArray,x=p.exportTypedArrayMethod,_=[].toLocaleString,k=[].slice,P=!!y&&g((function(){_.call(new y(1))}));x("toLocaleString",(function toLocaleString(){return _.apply(P?k.call(w(this)):w(this),arguments)}),g((function(){return[1,2].toLocaleString()!=new y([1,2]).toLocaleString()}))||!g((function(){y.prototype.toLocaleString.call([1,2])})))},function(n,i,o){"use strict";var u=o(5).exportTypedArrayMethod,p=o(4),g=o(3).Uint8Array,y=g&&g.prototype||{},w=[].toString,x=[].join;p((function(){w.call({})}))&&(w=function toString(){return x.call(this)});var _=y.toString!=w;u("toString",w,_)},function(n,i){"function"==typeof Object.create?n.exports=function inherits(n,i){i&&(n.super_=i,n.prototype=Object.create(i.prototype,{constructor:{value:n,enumerable:!1,writable:!0,configurable:!0}}))}:n.exports=function inherits(n,i){if(i){n.super_=i;var TempCtor=function(){};TempCtor.prototype=i.prototype,n.prototype=new TempCtor,n.prototype.constructor=n}}},function(n,i,o){var u,p,g;n.exports=(g=o(2),p=(u=g).lib.WordArray,u.enc.Base64={stringify:function(n){var i=n.words,o=n.sigBytes,u=this._map;n.clamp();for(var p=[],g=0;g<o;g+=3)for(var y=(i[g>>>2]>>>24-g%4*8&255)<<16|(i[g+1>>>2]>>>24-(g+1)%4*8&255)<<8|i[g+2>>>2]>>>24-(g+2)%4*8&255,w=0;w<4&&g+.75*w<o;w++)p.push(u.charAt(y>>>6*(3-w)&63));var x=u.charAt(64);if(x)for(;p.length%4;)p.push(x);return p.join("")},parse:function(n){var i=n.length,o=this._map,u=this._reverseMap;if(!u){u=this._reverseMap=[];for(var g=0;g<o.length;g++)u[o.charCodeAt(g)]=g}var y=o.charAt(64);if(y){var w=n.indexOf(y);-1!==w&&(i=w)}return function parseLoop(n,i,o){for(var u=[],g=0,y=0;y<i;y++)if(y%4){var w=o[n.charCodeAt(y-1)]<<y%4*2,x=o[n.charCodeAt(y)]>>>6-y%4*2;u[g>>>2]|=(w|x)<<24-g%4*8,g++}return p.create(u,g)}(n,i,u)},_map:"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/="},g.enc.Base64)},function(n,i,o){var u;n.exports=(u=o(2),function(n){var i=u,o=i.lib,p=o.WordArray,g=o.Hasher,y=i.algo,w=[];!function(){for(var i=0;i<64;i++)w[i]=4294967296*n.abs(n.sin(i+1))|0}();var x=y.MD5=g.extend({_doReset:function(){this._hash=new p.init([1732584193,4023233417,2562383102,271733878])},_doProcessBlock:function(n,i){for(var o=0;o<16;o++){var u=i+o,p=n[u];n[u]=16711935&(p<<8|p>>>24)|4278255360&(p<<24|p>>>8)}var g=this._hash.words,y=n[i+0],x=n[i+1],_=n[i+2],k=n[i+3],P=n[i+4],E=n[i+5],O=n[i+6],B=n[i+7],I=n[i+8],D=n[i+9],R=n[i+10],N=n[i+11],U=n[i+12],W=n[i+13],G=n[i+14],j=n[i+15],X=g[0],Y=g[1],K=g[2],J=g[3];X=FF(X,Y,K,J,y,7,w[0]),J=FF(J,X,Y,K,x,12,w[1]),K=FF(K,J,X,Y,_,17,w[2]),Y=FF(Y,K,J,X,k,22,w[3]),X=FF(X,Y,K,J,P,7,w[4]),J=FF(J,X,Y,K,E,12,w[5]),K=FF(K,J,X,Y,O,17,w[6]),Y=FF(Y,K,J,X,B,22,w[7]),X=FF(X,Y,K,J,I,7,w[8]),J=FF(J,X,Y,K,D,12,w[9]),K=FF(K,J,X,Y,R,17,w[10]),Y=FF(Y,K,J,X,N,22,w[11]),X=FF(X,Y,K,J,U,7,w[12]),J=FF(J,X,Y,K,W,12,w[13]),K=FF(K,J,X,Y,G,17,w[14]),X=GG(X,Y=FF(Y,K,J,X,j,22,w[15]),K,J,x,5,w[16]),J=GG(J,X,Y,K,O,9,w[17]),K=GG(K,J,X,Y,N,14,w[18]),Y=GG(Y,K,J,X,y,20,w[19]),X=GG(X,Y,K,J,E,5,w[20]),J=GG(J,X,Y,K,R,9,w[21]),K=GG(K,J,X,Y,j,14,w[22]),Y=GG(Y,K,J,X,P,20,w[23]),X=GG(X,Y,K,J,D,5,w[24]),J=GG(J,X,Y,K,G,9,w[25]),K=GG(K,J,X,Y,k,14,w[26]),Y=GG(Y,K,J,X,I,20,w[27]),X=GG(X,Y,K,J,W,5,w[28]),J=GG(J,X,Y,K,_,9,w[29]),K=GG(K,J,X,Y,B,14,w[30]),X=HH(X,Y=GG(Y,K,J,X,U,20,w[31]),K,J,E,4,w[32]),J=HH(J,X,Y,K,I,11,w[33]),K=HH(K,J,X,Y,N,16,w[34]),Y=HH(Y,K,J,X,G,23,w[35]),X=HH(X,Y,K,J,x,4,w[36]),J=HH(J,X,Y,K,P,11,w[37]),K=HH(K,J,X,Y,B,16,w[38]),Y=HH(Y,K,J,X,R,23,w[39]),X=HH(X,Y,K,J,W,4,w[40]),J=HH(J,X,Y,K,y,11,w[41]),K=HH(K,J,X,Y,k,16,w[42]),Y=HH(Y,K,J,X,O,23,w[43]),X=HH(X,Y,K,J,D,4,w[44]),J=HH(J,X,Y,K,U,11,w[45]),K=HH(K,J,X,Y,j,16,w[46]),X=II(X,Y=HH(Y,K,J,X,_,23,w[47]),K,J,y,6,w[48]),J=II(J,X,Y,K,B,10,w[49]),K=II(K,J,X,Y,G,15,w[50]),Y=II(Y,K,J,X,E,21,w[51]),X=II(X,Y,K,J,U,6,w[52]),J=II(J,X,Y,K,k,10,w[53]),K=II(K,J,X,Y,R,15,w[54]),Y=II(Y,K,J,X,x,21,w[55]),X=II(X,Y,K,J,I,6,w[56]),J=II(J,X,Y,K,j,10,w[57]),K=II(K,J,X,Y,O,15,w[58]),Y=II(Y,K,J,X,W,21,w[59]),X=II(X,Y,K,J,P,6,w[60]),J=II(J,X,Y,K,N,10,w[61]),K=II(K,J,X,Y,_,15,w[62]),Y=II(Y,K,J,X,D,21,w[63]),g[0]=g[0]+X|0,g[1]=g[1]+Y|0,g[2]=g[2]+K|0,g[3]=g[3]+J|0},_doFinalize:function(){var i=this._data,o=i.words,u=8*this._nDataBytes,p=8*i.sigBytes;o[p>>>5]|=128<<24-p%32;var g=n.floor(u/4294967296),y=u;o[15+(p+64>>>9<<4)]=16711935&(g<<8|g>>>24)|4278255360&(g<<24|g>>>8),o[14+(p+64>>>9<<4)]=16711935&(y<<8|y>>>24)|4278255360&(y<<24|y>>>8),i.sigBytes=4*(o.length+1),this._process();for(var w=this._hash,x=w.words,_=0;_<4;_++){var k=x[_];x[_]=16711935&(k<<8|k>>>24)|4278255360&(k<<24|k>>>8)}return w},clone:function(){var n=g.clone.call(this);return n._hash=this._hash.clone(),n}});function FF(n,i,o,u,p,g,y){var w=n+(i&o|~i&u)+p+y;return(w<<g|w>>>32-g)+i}function GG(n,i,o,u,p,g,y){var w=n+(i&u|o&~u)+p+y;return(w<<g|w>>>32-g)+i}function HH(n,i,o,u,p,g,y){var w=n+(i^o^u)+p+y;return(w<<g|w>>>32-g)+i}function II(n,i,o,u,p,g,y){var w=n+(o^(i|~u))+p+y;return(w<<g|w>>>32-g)+i}i.MD5=g._createHelper(x),i.HmacMD5=g._createHmacHelper(x)}(Math),u.MD5)},function(n,i,o){(function(){var n,u,p,g={}.hasOwnProperty;n=o(170),p=function(){function NumberT(n,i){this.type=n,this.endian=null!=i?i:"BE",this.fn=this.type,"8"!==this.type[this.type.length-1]&&(this.fn+=this.endian)}return NumberT.prototype.size=function(){return n.TYPES[this.type]},NumberT.prototype.decode=function(n){return n["read"+this.fn]()},NumberT.prototype.encode=function(n,i){return n["write"+this.fn](i)},NumberT}(),i.Number=p,i.uint8=new p("UInt8"),i.uint16be=i.uint16=new p("UInt16","BE"),i.uint16le=new p("UInt16","LE"),i.uint24be=i.uint24=new p("UInt24","BE"),i.uint24le=new p("UInt24","LE"),i.uint32be=i.uint32=new p("UInt32","BE"),i.uint32le=new p("UInt32","LE"),i.int8=new p("Int8"),i.int16be=i.int16=new p("Int16","BE"),i.int16le=new p("Int16","LE"),i.int24be=i.int24=new p("Int24","BE"),i.int24le=new p("Int24","LE"),i.int32be=i.int32=new p("Int32","BE"),i.int32le=new p("Int32","LE"),i.floatbe=i.float=new p("Float","BE"),i.floatle=new p("Float","LE"),i.doublebe=i.double=new p("Double","BE"),i.doublele=new p("Double","LE"),u=function(n){function Fixed(n,i,o){null==o&&(o=n>>1),Fixed.__super__.constructor.call(this,"Int"+n,i),this._point=1<<o}return function(n,i){for(var o in i)g.call(i,o)&&(n[o]=i[o]);function ctor(){this.constructor=n}ctor.prototype=i.prototype,n.prototype=new ctor,n.__super__=i.prototype}(Fixed,n),Fixed.prototype.decode=function(n){return Fixed.__super__.decode.call(this,n)/this._point},Fixed.prototype.encode=function(n,i){return Fixed.__super__.encode.call(this,n,i*this._point|0)},Fixed}(p),i.Fixed=u,i.fixed16be=i.fixed16=new u(16,"BE"),i.fixed16le=new u(16,"LE"),i.fixed32be=i.fixed32=new u(32,"BE"),i.fixed32le=new u(32,"LE")}).call(this)},function(n,i,o){var u=o(174);n.exports=function(n){return Object(u(n))}},function(n,i,o){"use strict";var u=o(433)(!0);o(179)(String,"String",(function(n){this._t=String(n),this._i=0}),(function(){var n,i=this._t,o=this._i;return o>=i.length?{value:void 0,done:!0}:(n=u(i,o),this._i+=n.length,{value:n,done:!1})}))},function(n,i,o){var u=o(4),p=o(41),g="".split;n.exports=u((function(){return!Object("z").propertyIsEnumerable(0)}))?function(n){return"String"==p(n)?g.call(n,""):Object(n)}:Object},function(n,i,o){var u=o(27),p=o(8),g=o(43),createMethod=function(n){return function(i,o,y){var w,x=u(i),_=p(x.length),k=g(y,_);if(n&&o!=o){for(;_>k;)if((w=x[k++])!=w)return!0}else for(;_>k;k++)if((n||k in x)&&x[k]===o)return n||k||0;return!n&&-1}};n.exports={includes:createMethod(!0),indexOf:createMethod(!1)}},function(n,i,o){var u=o(204),p=o(143);n.exports=Object.keys||function keys(n){return u(n,p)}},function(n,i,o){var u=o(16).f,p=o(15),g=o(6)("toStringTag");n.exports=function(n,i,o){n&&!p(n=o?n:n.prototype,g)&&u(n,g,{configurable:!0,value:i})}},function(n,i,o){var u=o(44);n.exports=function(n,i,o){if(u(n),void 0===i)return n;switch(o){case 0:return function(){return n.call(i)};case 1:return function(o){return n.call(i,o)};case 2:return function(o,u){return n.call(i,o,u)};case 3:return function(o,u,p){return n.call(i,o,u,p)}}return function(){return n.apply(i,arguments)}}},function(n,i){n.exports={}},function(n,i,o){(function(n){function objectToString(n){return Object.prototype.toString.call(n)}i.isArray=function isArray(n){return Array.isArray?Array.isArray(n):"[object Array]"===objectToString(n)},i.isBoolean=function isBoolean(n){return"boolean"==typeof n},i.isNull=function isNull(n){return null===n},i.isNullOrUndefined=function isNullOrUndefined(n){return null==n},i.isNumber=function isNumber(n){return"number"==typeof n},i.isString=function isString(n){return"string"==typeof n},i.isSymbol=function isSymbol(n){return"symbol"==typeof n},i.isUndefined=function isUndefined(n){return void 0===n},i.isRegExp=function isRegExp(n){return"[object RegExp]"===objectToString(n)},i.isObject=function isObject(n){return"object"==typeof n&&null!==n},i.isDate=function isDate(n){return"[object Date]"===objectToString(n)},i.isError=function isError(n){return"[object Error]"===objectToString(n)||n instanceof Error},i.isFunction=function isFunction(n){return"function"==typeof n},i.isPrimitive=function isPrimitive(n){return null===n||"boolean"==typeof n||"number"==typeof n||"string"==typeof n||"symbol"==typeof n||void 0===n},i.isBuffer=n.isBuffer}).call(this,o(10).Buffer)},function(n,i){var o={}.toString;n.exports=function(n){return o.call(n).slice(8,-1)}},function(n,i){n.exports=function(n,i){return{enumerable:!(1&n),configurable:!(2&n),writable:!(4&n),value:i}}},function(n,i){n.exports=function(n){if("function"!=typeof n)throw TypeError(n+" is not a function!");return n}},function(n,i,o){o(428);for(var u=o(17),p=o(39),g=o(103),y=o(21)("toStringTag"),w="CSSRuleList,CSSStyleDeclaration,CSSValueList,ClientRectList,DOMRectList,DOMStringList,DOMTokenList,DataTransferItemList,FileList,HTMLAllCollection,HTMLCollection,HTMLFormElement,HTMLSelectElement,MediaList,MimeTypeArray,NamedNodeMap,NodeList,PaintRequestList,Plugin,PluginArray,SVGLengthList,SVGNumberList,SVGPathSegList,SVGPointList,SVGStringList,SVGTransformList,SourceBufferList,StyleSheetList,TextTrackCueList,TextTrackList,TouchList".split(","),x=0;x<w.length;x++){var _=w[x],k=u[_],P=k&&k.prototype;P&&!P[y]&&p(P,y,_),g[_]=g.Array}},function(n,i){n.exports={}},function(n,i){n.exports=!0},function(n,i,o){var u=o(259),p=o(183);n.exports=Object.keys||function keys(n){return u(n,p)}},function(n,i,o){var u=o(26).f,p=o(51),g=o(21)("toStringTag");n.exports=function(n,i,o){n&&!p(n=o?n:n.prototype,g)&&u(n,g,{configurable:!0,value:i})}},function(n,i,o){var u=o(32),p=o(266),g=o(267),y=o(29),w=o(130),x=o(184),_={},k={};(i=n.exports=function(n,i,o,P,E){var O,B,I,D,R=E?function(){return n}:x(n),N=u(o,P,i?2:1),U=0;if("function"!=typeof R)throw TypeError(n+" is not iterable!");if(g(R)){for(O=w(n.length);O>U;U++)if((D=i?N(y(B=n[U])[0],B[1]):N(n[U]))===_||D===k)return D}else for(I=R.call(n);!(B=I.next()).done;)if((D=p(I,N,B.value,i))===_||D===k)return D}).BREAK=_,i.RETURN=k},function(n,i,o){"use strict";var u=o(273),p="function"==typeof Symbol&&"symbol"==typeof Symbol("foo"),g=Object.prototype.toString,y=Array.prototype.concat,w=Object.defineProperty,x=w&&function(){var n={};try{for(var i in w(n,"x",{enumerable:!1,value:n}),n)return!1;return n.x===n}catch(n){return!1}}(),defineProperty=function(n,i,o,u){var p;(!(i in n)||"function"==typeof(p=u)&&"[object Function]"===g.call(p)&&u())&&(x?w(n,i,{configurable:!0,enumerable:!1,value:o,writable:!0}):n[i]=o)},defineProperties=function(n,i){var o=arguments.length>2?arguments[2]:{},g=u(i);p&&(g=y.call(g,Object.getOwnPropertySymbols(i)));for(var w=0;w<g.length;w+=1)defineProperty(n,g[w],i[g[w]],o[g[w]])};defineProperties.supportsDescriptors=!!x,n.exports=defineProperties},function(n,i,o){"use strict";var u=o(44),PromiseCapability=function(n){var i,o;this.promise=new n((function(n,u){if(void 0!==i||void 0!==o)throw TypeError("Bad Promise constructor");i=n,o=u})),this.resolve=u(i),this.reject=u(o)};n.exports.f=function(n){return new PromiseCapability(n)}},function(n,i,o){"use strict";var u={}.propertyIsEnumerable,p=Object.getOwnPropertyDescriptor,g=p&&!u.call({1:2},1);i.f=g?function propertyIsEnumerable(n){var i=p(this,n);return!!i&&i.enumerable}:u},function(n,i,o){var u=o(142),p=o(112),g=u("keys");n.exports=function(n){return g[n]||(g[n]=p(n))}},function(n,i){var o=0,u=Math.random();n.exports=function(n){return"Symbol("+String(void 0===n?"":n)+")_"+(++o+u).toString(36)}},function(n,i){n.exports={}},function(n,i,o){var u=o(3);n.exports=u},function(n,i,o){var u=o(41);n.exports=Array.isArray||function isArray(n){return"Array"==u(n)}},function(n,i,o){var u=o(4),p=o(6),g=o(148),y=p("species");n.exports=function(n){return g>=51||!u((function(){var i=[];return(i.constructor={})[y]=function(){return{foo:1}},1!==i[n](Boolean).foo}))}},function(n,i,o){var u=o(154),p=o(41),g=o(6)("toStringTag"),y="Arguments"==p(function(){return arguments}());n.exports=u?p:function(n){var i,o,u;return void 0===n?"Undefined":null===n?"Null":"string"==typeof(o=function(n,i){try{return n[i]}catch(n){}}(i=Object(n),g))?o:y?p(i):"Object"==(u=p(i))&&"function"==typeof i.callee?"Arguments":u}},function(n,i,o){"use strict";var u=o(1),p=o(119);u({target:"RegExp",proto:!0,forced:/./.exec!==p},{exec:p})},function(n,i,o){"use strict";var u,p,g=o(230),y=o(333),w=RegExp.prototype.exec,x=String.prototype.replace,_=w,k=(u=/a/,p=/b*/g,w.call(u,"a"),w.call(p,"a"),0!==u.lastIndex||0!==p.lastIndex),P=y.UNSUPPORTED_Y||y.BROKEN_CARET,E=void 0!==/()??/.exec("")[1];(k||E||P)&&(_=function exec(n){var i,o,u,p,y=this,_=P&&y.sticky,O=g.call(y),B=y.source,I=0,D=n;return _&&(-1===(O=O.replace("y","")).indexOf("g")&&(O+="g"),D=String(n).slice(y.lastIndex),y.lastIndex>0&&(!y.multiline||y.multiline&&"\n"!==n[y.lastIndex-1])&&(B="(?: "+B+")",D=" "+D,I++),o=new RegExp("^(?:"+B+")",O)),E&&(o=new RegExp("^"+B+"$(?!\\s)",O)),k&&(i=y.lastIndex),u=w.call(_?o:y,D),_?u?(u.input=u.input.slice(I),u[0]=u[0].slice(I),u.index=y.lastIndex,y.lastIndex+=u[0].length):y.lastIndex=0:k&&u&&(y.lastIndex=y.global?u.index+u[0].length:i),E&&u&&u.length>1&&x.call(u[0],o,(function(){for(p=1;p<arguments.length-2;p++)void 0===arguments[p]&&(u[p]=void 0)})),u}),n.exports=_},function(n,i,o){var u=o(3),p=o(240),g=o(28),y=o(18),w=o(6),x=w("iterator"),_=w("toStringTag"),k=g.values;for(var P in p){var E=u[P],O=E&&E.prototype;if(O){if(O[x]!==k)try{y(O,x,k)}catch(n){O[x]=k}if(O[_]||y(O,_,P),p[P])for(var B in g)if(O[B]!==g[B])try{y(O,B,g[B])}catch(n){O[B]=g[B]}}}},function(n,i,o){n.exports=Stream;var u=o(122).EventEmitter;function Stream(){u.call(this)}o(86)(Stream,u),Stream.Readable=o(165),Stream.Writable=o(352),Stream.Duplex=o(353),Stream.Transform=o(354),Stream.PassThrough=o(355),Stream.Stream=Stream,Stream.prototype.pipe=function(n,i){var o=this;function ondata(i){n.writable&&!1===n.write(i)&&o.pause&&o.pause()}function ondrain(){o.readable&&o.resume&&o.resume()}o.on("data",ondata),n.on("drain",ondrain),n._isStdio||i&&!1===i.end||(o.on("end",onend),o.on("close",onclose));var p=!1;function onend(){p||(p=!0,n.end())}function onclose(){p||(p=!0,"function"==typeof n.destroy&&n.destroy())}function onerror(n){if(cleanup(),0===u.listenerCount(this,"error"))throw n}function cleanup(){o.removeListener("data",ondata),n.removeListener("drain",ondrain),o.removeListener("end",onend),o.removeListener("close",onclose),o.removeListener("error",onerror),n.removeListener("error",onerror),o.removeListener("end",cleanup),o.removeListener("close",cleanup),n.removeListener("close",cleanup)}return o.on("error",onerror),n.on("error",onerror),o.on("end",cleanup),o.on("close",cleanup),n.on("close",cleanup),n.emit("pipe",o),n}},function(n,i,o){"use strict";var u,p="object"==typeof Reflect?Reflect:null,g=p&&"function"==typeof p.apply?p.apply:function ReflectApply(n,i,o){return Function.prototype.apply.call(n,i,o)};u=p&&"function"==typeof p.ownKeys?p.ownKeys:Object.getOwnPropertySymbols?function ReflectOwnKeys(n){return Object.getOwnPropertyNames(n).concat(Object.getOwnPropertySymbols(n))}:function ReflectOwnKeys(n){return Object.getOwnPropertyNames(n)};var y=Number.isNaN||function NumberIsNaN(n){return n!=n};function EventEmitter(){EventEmitter.init.call(this)}n.exports=EventEmitter,n.exports.once=function once(n,i){return new Promise((function(o,u){function eventListener(){void 0!==p&&n.removeListener("error",p),o([].slice.call(arguments))}var p;"error"!==i&&(p=function errorListener(o){n.removeListener(i,eventListener),u(o)},n.once("error",p)),n.once(i,eventListener)}))},EventEmitter.EventEmitter=EventEmitter,EventEmitter.prototype._events=void 0,EventEmitter.prototype._eventsCount=0,EventEmitter.prototype._maxListeners=void 0;var w=10;function checkListener(n){if("function"!=typeof n)throw new TypeError('The "listener" argument must be of type Function. Received type '+typeof n)}function _getMaxListeners(n){return void 0===n._maxListeners?EventEmitter.defaultMaxListeners:n._maxListeners}function _addListener(n,i,o,u){var p,g,y;if(checkListener(o),void 0===(g=n._events)?(g=n._events=Object.create(null),n._eventsCount=0):(void 0!==g.newListener&&(n.emit("newListener",i,o.listener?o.listener:o),g=n._events),y=g[i]),void 0===y)y=g[i]=o,++n._eventsCount;else if("function"==typeof y?y=g[i]=u?[o,y]:[y,o]:u?y.unshift(o):y.push(o),(p=_getMaxListeners(n))>0&&y.length>p&&!y.warned){y.warned=!0;var w=new Error("Possible EventEmitter memory leak detected. "+y.length+" "+String(i)+" listeners added. Use emitter.setMaxListeners() to increase limit");w.name="MaxListenersExceededWarning",w.emitter=n,w.type=i,w.count=y.length,function ProcessEmitWarning(n){console&&console.warn}()}return n}function onceWrapper(){if(!this.fired)return this.target.removeListener(this.type,this.wrapFn),this.fired=!0,0===arguments.length?this.listener.call(this.target):this.listener.apply(this.target,arguments)}function _onceWrap(n,i,o){var u={fired:!1,wrapFn:void 0,target:n,type:i,listener:o},p=onceWrapper.bind(u);return p.listener=o,u.wrapFn=p,p}function _listeners(n,i,o){var u=n._events;if(void 0===u)return[];var p=u[i];return void 0===p?[]:"function"==typeof p?o?[p.listener||p]:[p]:o?function unwrapListeners(n){for(var i=new Array(n.length),o=0;o<i.length;++o)i[o]=n[o].listener||n[o];return i}(p):arrayClone(p,p.length)}function listenerCount(n){var i=this._events;if(void 0!==i){var o=i[n];if("function"==typeof o)return 1;if(void 0!==o)return o.length}return 0}function arrayClone(n,i){for(var o=new Array(i),u=0;u<i;++u)o[u]=n[u];return o}Object.defineProperty(EventEmitter,"defaultMaxListeners",{enumerable:!0,get:function(){return w},set:function(n){if("number"!=typeof n||n<0||y(n))throw new RangeError('The value of "defaultMaxListeners" is out of range. It must be a non-negative number. Received '+n+".");w=n}}),EventEmitter.init=function(){void 0!==this._events&&this._events!==Object.getPrototypeOf(this)._events||(this._events=Object.create(null),this._eventsCount=0),this._maxListeners=this._maxListeners||void 0},EventEmitter.prototype.setMaxListeners=function setMaxListeners(n){if("number"!=typeof n||n<0||y(n))throw new RangeError('The value of "n" is out of range. It must be a non-negative number. Received '+n+".");return this._maxListeners=n,this},EventEmitter.prototype.getMaxListeners=function getMaxListeners(){return _getMaxListeners(this)},EventEmitter.prototype.emit=function emit(n){for(var i=[],o=1;o<arguments.length;o++)i.push(arguments[o]);var u="error"===n,p=this._events;if(void 0!==p)u=u&&void 0===p.error;else if(!u)return!1;if(u){var y;if(i.length>0&&(y=i[0]),y instanceof Error)throw y;var w=new Error("Unhandled error."+(y?" ("+y.message+")":""));throw w.context=y,w}var x=p[n];if(void 0===x)return!1;if("function"==typeof x)g(x,this,i);else{var _=x.length,k=arrayClone(x,_);for(o=0;o<_;++o)g(k[o],this,i)}return!0},EventEmitter.prototype.addListener=function addListener(n,i){return _addListener(this,n,i,!1)},EventEmitter.prototype.on=EventEmitter.prototype.addListener,EventEmitter.prototype.prependListener=function prependListener(n,i){return _addListener(this,n,i,!0)},EventEmitter.prototype.once=function once(n,i){return checkListener(i),this.on(n,_onceWrap(this,n,i)),this},EventEmitter.prototype.prependOnceListener=function prependOnceListener(n,i){return checkListener(i),this.prependListener(n,_onceWrap(this,n,i)),this},EventEmitter.prototype.removeListener=function removeListener(n,i){var o,u,p,g,y;if(checkListener(i),void 0===(u=this._events))return this;if(void 0===(o=u[n]))return this;if(o===i||o.listener===i)0==--this._eventsCount?this._events=Object.create(null):(delete u[n],u.removeListener&&this.emit("removeListener",n,o.listener||i));else if("function"!=typeof o){for(p=-1,g=o.length-1;g>=0;g--)if(o[g]===i||o[g].listener===i){y=o[g].listener,p=g;break}if(p<0)return this;0===p?o.shift():function spliceOne(n,i){for(;i+1<n.length;i++)n[i]=n[i+1];n.pop()}(o,p),1===o.length&&(u[n]=o[0]),void 0!==u.removeListener&&this.emit("removeListener",n,y||i)}return this},EventEmitter.prototype.off=EventEmitter.prototype.removeListener,EventEmitter.prototype.removeAllListeners=function removeAllListeners(n){var i,o,u;if(void 0===(o=this._events))return this;if(void 0===o.removeListener)return 0===arguments.length?(this._events=Object.create(null),this._eventsCount=0):void 0!==o[n]&&(0==--this._eventsCount?this._events=Object.create(null):delete o[n]),this;if(0===arguments.length){var p,g=Object.keys(o);for(u=0;u<g.length;++u)"removeListener"!==(p=g[u])&&this.removeAllListeners(p);return this.removeAllListeners("removeListener"),this._events=Object.create(null),this._eventsCount=0,this}if("function"==typeof(i=o[n]))this.removeListener(n,i);else if(void 0!==i)for(u=i.length-1;u>=0;u--)this.removeListener(n,i[u]);return this},EventEmitter.prototype.listeners=function listeners(n){return _listeners(this,n,!0)},EventEmitter.prototype.rawListeners=function rawListeners(n){return _listeners(this,n,!1)},EventEmitter.listenerCount=function(n,i){return"function"==typeof n.listenerCount?n.listenerCount(i):listenerCount.call(n,i)},EventEmitter.prototype.listenerCount=listenerCount,EventEmitter.prototype.eventNames=function eventNames(){return this._eventsCount>0?u(this._events):[]}},function(n,i,o){"use strict";(function(i){void 0===i||!i.version||0===i.version.indexOf("v0.")||0===i.version.indexOf("v1.")&&0!==i.version.indexOf("v1.8.")?n.exports={nextTick:function nextTick(n,o,u,p){if("function"!=typeof n)throw new TypeError('"callback" argument must be a function');var g,y,w=arguments.length;switch(w){case 0:case 1:return i.nextTick(n);case 2:return i.nextTick((function afterTickOne(){n.call(null,o)}));case 3:return i.nextTick((function afterTickTwo(){n.call(null,o,u)}));case 4:return i.nextTick((function afterTickThree(){n.call(null,o,u,p)}));default:for(g=new Array(w-1),y=0;y<g.length;)g[y++]=arguments[y];return i.nextTick((function afterTick(){n.apply(null,g)}))}}}:n.exports=i}).call(this,o(47))},function(n,i,o){var u=o(10),p=u.Buffer;function copyProps(n,i){for(var o in n)i[o]=n[o]}function SafeBuffer(n,i,o){return p(n,i,o)}p.from&&p.alloc&&p.allocUnsafe&&p.allocUnsafeSlow?n.exports=u:(copyProps(u,i),i.Buffer=SafeBuffer),copyProps(p,SafeBuffer),SafeBuffer.from=function(n,i,o){if("number"==typeof n)throw new TypeError("Argument must not be a number");return p(n,i,o)},SafeBuffer.alloc=function(n,i,o){if("number"!=typeof n)throw new TypeError("Argument must be a number");var u=p(n);return void 0!==i?"string"==typeof o?u.fill(i,o):u.fill(i):u.fill(0),u},SafeBuffer.allocUnsafe=function(n){if("number"!=typeof n)throw new TypeError("Argument must be a number");return p(n)},SafeBuffer.allocUnsafeSlow=function(n){if("number"!=typeof n)throw new TypeError("Argument must be a number");return u.SlowBuffer(n)}},function(n,i,o){"use strict";var u=o(124).Buffer,p=u.isEncoding||function(n){switch((n=""+n)&&n.toLowerCase()){case"hex":case"utf8":case"utf-8":case"ascii":case"binary":case"base64":case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":case"raw":return!0;default:return!1}};function StringDecoder(n){var i;switch(this.encoding=function normalizeEncoding(n){var i=function _normalizeEncoding(n){if(!n)return"utf8";for(var i;;)switch(n){case"utf8":case"utf-8":return"utf8";case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return"utf16le";case"latin1":case"binary":return"latin1";case"base64":case"ascii":case"hex":return n;default:if(i)return;n=(""+n).toLowerCase(),i=!0}}(n);if("string"!=typeof i&&(u.isEncoding===p||!p(n)))throw new Error("Unknown encoding: "+n);return i||n}(n),this.encoding){case"utf16le":this.text=utf16Text,this.end=utf16End,i=4;break;case"utf8":this.fillLast=utf8FillLast,i=4;break;case"base64":this.text=base64Text,this.end=base64End,i=3;break;default:return this.write=simpleWrite,void(this.end=simpleEnd)}this.lastNeed=0,this.lastTotal=0,this.lastChar=u.allocUnsafe(i)}function utf8CheckByte(n){return n<=127?0:n>>5==6?2:n>>4==14?3:n>>3==30?4:n>>6==2?-1:-2}function utf8FillLast(n){var i=this.lastTotal-this.lastNeed,o=function utf8CheckExtraBytes(n,i,o){if(128!=(192&i[0]))return n.lastNeed=0,"�";if(n.lastNeed>1&&i.length>1){if(128!=(192&i[1]))return n.lastNeed=1,"�";if(n.lastNeed>2&&i.length>2&&128!=(192&i[2]))return n.lastNeed=2,"�"}}(this,n);return void 0!==o?o:this.lastNeed<=n.length?(n.copy(this.lastChar,i,0,this.lastNeed),this.lastChar.toString(this.encoding,0,this.lastTotal)):(n.copy(this.lastChar,i,0,n.length),void(this.lastNeed-=n.length))}function utf16Text(n,i){if((n.length-i)%2==0){var o=n.toString("utf16le",i);if(o){var u=o.charCodeAt(o.length-1);if(u>=55296&&u<=56319)return this.lastNeed=2,this.lastTotal=4,this.lastChar[0]=n[n.length-2],this.lastChar[1]=n[n.length-1],o.slice(0,-1)}return o}return this.lastNeed=1,this.lastTotal=2,this.lastChar[0]=n[n.length-1],n.toString("utf16le",i,n.length-1)}function utf16End(n){var i=n&&n.length?this.write(n):"";if(this.lastNeed){var o=this.lastTotal-this.lastNeed;return i+this.lastChar.toString("utf16le",0,o)}return i}function base64Text(n,i){var o=(n.length-i)%3;return 0===o?n.toString("base64",i):(this.lastNeed=3-o,this.lastTotal=3,1===o?this.lastChar[0]=n[n.length-1]:(this.lastChar[0]=n[n.length-2],this.lastChar[1]=n[n.length-1]),n.toString("base64",i,n.length-o))}function base64End(n){var i=n&&n.length?this.write(n):"";return this.lastNeed?i+this.lastChar.toString("base64",0,3-this.lastNeed):i}function simpleWrite(n){return n.toString(this.encoding)}function simpleEnd(n){return n&&n.length?this.write(n):""}i.StringDecoder=StringDecoder,StringDecoder.prototype.write=function(n){if(0===n.length)return"";var i,o;if(this.lastNeed){if(void 0===(i=this.fillLast(n)))return"";o=this.lastNeed,this.lastNeed=0}else o=0;return o<n.length?i?i+this.text(n,o):this.text(n,o):i||""},StringDecoder.prototype.end=function utf8End(n){var i=n&&n.length?this.write(n):"";return this.lastNeed?i+"�":i},StringDecoder.prototype.text=function utf8Text(n,i){var o=function utf8CheckIncomplete(n,i,o){var u=i.length-1;if(u<o)return 0;var p=utf8CheckByte(i[u]);if(p>=0)return p>0&&(n.lastNeed=p-1),p;if(--u<o||-2===p)return 0;if((p=utf8CheckByte(i[u]))>=0)return p>0&&(n.lastNeed=p-2),p;if(--u<o||-2===p)return 0;if((p=utf8CheckByte(i[u]))>=0)return p>0&&(2===p?p=0:n.lastNeed=p-3),p;return 0}(this,n,i);if(!this.lastNeed)return n.toString("utf8",i);this.lastTotal=o;var u=n.length-(o-this.lastNeed);return n.copy(this.lastChar,0,u),n.toString("utf8",i,u)},StringDecoder.prototype.fillLast=function(n){if(this.lastNeed<=n.length)return n.copy(this.lastChar,this.lastTotal-this.lastNeed,0,this.lastNeed),this.lastChar.toString(this.encoding,0,this.lastTotal);n.copy(this.lastChar,this.lastTotal-this.lastNeed,0,n.length),this.lastNeed-=n.length}},function(n,i,o){"use strict";var u="undefined"!=typeof Uint8Array&&"undefined"!=typeof Uint16Array&&"undefined"!=typeof Int32Array;function _has(n,i){return Object.prototype.hasOwnProperty.call(n,i)}i.assign=function(n){for(var i=Array.prototype.slice.call(arguments,1);i.length;){var o=i.shift();if(o){if("object"!=typeof o)throw new TypeError(o+"must be non-object");for(var u in o)_has(o,u)&&(n[u]=o[u])}}return n},i.shrinkBuf=function(n,i){return n.length===i?n:n.subarray?n.subarray(0,i):(n.length=i,n)};var p={arraySet:function(n,i,o,u,p){if(i.subarray&&n.subarray)n.set(i.subarray(o,o+u),p);else for(var g=0;g<u;g++)n[p+g]=i[o+g]},flattenChunks:function(n){var i,o,u,p,g,y;for(u=0,i=0,o=n.length;i<o;i++)u+=n[i].length;for(y=new Uint8Array(u),p=0,i=0,o=n.length;i<o;i++)g=n[i],y.set(g,p),p+=g.length;return y}},g={arraySet:function(n,i,o,u,p){for(var g=0;g<u;g++)n[p+g]=i[o+g]},flattenChunks:function(n){return[].concat.apply([],n)}};i.setTyped=function(n){n?(i.Buf8=Uint8Array,i.Buf16=Uint16Array,i.Buf32=Int32Array,i.assign(i,p)):(i.Buf8=Array,i.Buf16=Array,i.Buf32=Array,i.assign(i,g))},i.setTyped(u)},function(n,i,o){var u,p,g,y,w,x;n.exports=(x=o(2),p=(u=x).lib,g=p.Base,y=p.WordArray,(w=u.x64={}).Word=g.extend({init:function(n,i){this.high=n,this.low=i}}),w.WordArray=g.extend({init:function(n,i){n=this.words=n||[],this.sigBytes=null!=i?i:8*n.length},toX32:function(){for(var n=this.words,i=n.length,o=[],u=0;u<i;u++){var p=n[u];o.push(p.high),o.push(p.low)}return y.create(o,this.sigBytes)},clone:function(){for(var n=g.clone.call(this),i=n.words=this.words.slice(0),o=i.length,u=0;u<o;u++)i[u]=i[u].clone();return n}}),x)},function(n,i){i.f={}.propertyIsEnumerable},function(n,i,o){var u=o(29),p=o(258),g=o(183),y=o(181)("IE_PROTO"),Empty=function(){},createDict=function(){var n,i=o(177)("iframe"),u=g.length;for(i.style.display="none",o(261).appendChild(i),i.src="javascript:",(n=i.contentWindow.document).open(),n.write("<script>document.F=Object<\/script>"),n.close(),createDict=n.F;u--;)delete createDict.prototype[g[u]];return createDict()};n.exports=Object.create||function create(n,i){var o;return null!==n?(Empty.prototype=u(n),o=new Empty,Empty.prototype=null,o[y]=n):o=createDict(),void 0===i?o:p(o,i)}},function(n,i,o){var u=o(180),p=Math.min;n.exports=function(n){return n>0?p(u(n),9007199254740991):0}},function(n,i){var o=0,u=Math.random();n.exports=function(n){return"Symbol(".concat(void 0===n?"":n,")_",(++o+u).toString(36))}},function(n,i,o){var u=o(131)("meta"),p=o(22),g=o(51),y=o(26).f,w=0,x=Object.isExtensible||function(){return!0},_=!o(52)((function(){return x(Object.preventExtensions({}))})),setMeta=function(n){y(n,u,{value:{i:"O"+ ++w,w:{}}})},k=n.exports={KEY:u,NEED:!1,fastKey:function(n,i){if(!p(n))return"symbol"==typeof n?n:("string"==typeof n?"S":"P")+n;if(!g(n,u)){if(!x(n))return"F";if(!i)return"E";setMeta(n)}return n[u].i},getWeak:function(n,i){if(!g(n,u)){if(!x(n))return!0;if(!i)return!1;setMeta(n)}return n[u].w},onFreeze:function(n){return _&&k.NEED&&x(n)&&!g(n,u)&&setMeta(n),n}}},function(n,i){},function(n,i,o){o(237)("Uint32",(function(n){return function Uint32Array(i,o,u){return n(this,i,o,u)}}))},function(n,i){function Tree(){this.table=new Uint16Array(16),this.trans=new Uint16Array(288)}function Data(n,i){this.source=n,this.sourceIndex=0,this.tag=0,this.bitcount=0,this.dest=i,this.destLen=0,this.ltree=new Tree,this.dtree=new Tree}var o=new Tree,u=new Tree,p=new Uint8Array(30),g=new Uint16Array(30),y=new Uint8Array(30),w=new Uint16Array(30),x=new Uint8Array([16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15]),_=new Tree,k=new Uint8Array(320);function tinf_build_bits_base(n,i,o,u){var p,g;for(p=0;p<o;++p)n[p]=0;for(p=0;p<30-o;++p)n[p+o]=p/o|0;for(g=u,p=0;p<30;++p)i[p]=g,g+=1<<n[p]}var P=new Uint16Array(16);function tinf_build_tree(n,i,o,u){var p,g;for(p=0;p<16;++p)n.table[p]=0;for(p=0;p<u;++p)n.table[i[o+p]]++;for(n.table[0]=0,g=0,p=0;p<16;++p)P[p]=g,g+=n.table[p];for(p=0;p<u;++p)i[o+p]&&(n.trans[P[i[o+p]]++]=p)}function tinf_getbit(n){n.bitcount--||(n.tag=n.source[n.sourceIndex++],n.bitcount=7);var i=1&n.tag;return n.tag>>>=1,i}function tinf_read_bits(n,i,o){if(!i)return o;for(;n.bitcount<24;)n.tag|=n.source[n.sourceIndex++]<<n.bitcount,n.bitcount+=8;var u=n.tag&65535>>>16-i;return n.tag>>>=i,n.bitcount-=i,u+o}function tinf_decode_symbol(n,i){for(;n.bitcount<24;)n.tag|=n.source[n.sourceIndex++]<<n.bitcount,n.bitcount+=8;var o=0,u=0,p=0,g=n.tag;do{u=2*u+(1&g),g>>>=1,++p,o+=i.table[p],u-=i.table[p]}while(u>=0);return n.tag=g,n.bitcount-=p,i.trans[o+u]}function tinf_decode_trees(n,i,o){var u,p,g,y,w,P;for(u=tinf_read_bits(n,5,257),p=tinf_read_bits(n,5,1),g=tinf_read_bits(n,4,4),y=0;y<19;++y)k[y]=0;for(y=0;y<g;++y){var E=tinf_read_bits(n,3,0);k[x[y]]=E}for(tinf_build_tree(_,k,0,19),w=0;w<u+p;){var O=tinf_decode_symbol(n,_);switch(O){case 16:var B=k[w-1];for(P=tinf_read_bits(n,2,3);P;--P)k[w++]=B;break;case 17:for(P=tinf_read_bits(n,3,3);P;--P)k[w++]=0;break;case 18:for(P=tinf_read_bits(n,7,11);P;--P)k[w++]=0;break;default:k[w++]=O}}tinf_build_tree(i,k,0,u),tinf_build_tree(o,k,u,p)}function tinf_inflate_block_data(n,i,o){for(;;){var u,x,_,k,P=tinf_decode_symbol(n,i);if(256===P)return 0;if(P<256)n.dest[n.destLen++]=P;else for(u=tinf_read_bits(n,p[P-=257],g[P]),x=tinf_decode_symbol(n,o),k=_=n.destLen-tinf_read_bits(n,y[x],w[x]);k<_+u;++k)n.dest[n.destLen++]=n.dest[k]}}function tinf_inflate_uncompressed_block(n){for(var i,o;n.bitcount>8;)n.sourceIndex--,n.bitcount-=8;if((i=256*(i=n.source[n.sourceIndex+1])+n.source[n.sourceIndex])!==(65535&~(256*n.source[n.sourceIndex+3]+n.source[n.sourceIndex+2])))return-3;for(n.sourceIndex+=4,o=i;o;--o)n.dest[n.destLen++]=n.source[n.sourceIndex++];return n.bitcount=0,0}!function tinf_build_fixed_trees(n,i){var o;for(o=0;o<7;++o)n.table[o]=0;for(n.table[7]=24,n.table[8]=152,n.table[9]=112,o=0;o<24;++o)n.trans[o]=256+o;for(o=0;o<144;++o)n.trans[24+o]=o;for(o=0;o<8;++o)n.trans[168+o]=280+o;for(o=0;o<112;++o)n.trans[176+o]=144+o;for(o=0;o<5;++o)i.table[o]=0;for(i.table[5]=32,o=0;o<32;++o)i.trans[o]=o}(o,u),tinf_build_bits_base(p,g,4,3),tinf_build_bits_base(y,w,2,1),p[28]=0,g[28]=258,n.exports=function tinf_uncompress(n,i){var p,g,y=new Data(n,i);do{switch(p=tinf_getbit(y),tinf_read_bits(y,2,0)){case 0:g=tinf_inflate_uncompressed_block(y);break;case 1:g=tinf_inflate_block_data(y,o,u);break;case 2:tinf_decode_trees(y,y.ltree,y.dtree),g=tinf_inflate_block_data(y,y.ltree,y.dtree);break;default:g=-3}if(0!==g)throw new Error("Data error")}while(!p);return y.destLen<y.dest.length?"function"==typeof y.dest.slice?y.dest.slice(0,y.destLen):y.dest.subarray(0,y.destLen):y.dest}},function(n,i,o){var u=o(11),p=o(152),g=o(8),y=o(96),w=o(153),x=o(215),Result=function(n,i){this.stopped=n,this.result=i};n.exports=function(n,i,o){var _,k,P,E,O,B,I,D=o&&o.that,R=!(!o||!o.AS_ENTRIES),N=!(!o||!o.IS_ITERATOR),U=!(!o||!o.INTERRUPTED),W=y(i,D,1+R+U),stop=function(n){return _&&x(_),new Result(!0,n)},callFn=function(n){return R?(u(n),U?W(n[0],n[1],stop):W(n[0],n[1])):U?W(n,stop):W(n)};if(N)_=n;else{if("function"!=typeof(k=w(n)))throw TypeError("Target is not iterable");if(p(k)){for(P=0,E=g(n.length);E>P;P++)if((O=callFn(n[P]))&&O instanceof Result)return O;return new Result(!1)}_=k.call(n)}for(B=_.next;!(I=B.call(_)).done;){try{O=callFn(I.value)}catch(n){throw x(_),n}if("object"==typeof O&&O&&O instanceof Result)return O}return new Result(!1)}},function(n,i){n.exports=function(n){try{return{error:!1,value:n()}}catch(n){return{error:!0,value:n}}}},function(n,i,o){var u=o(3),p=o(14),g=u.document,y=p(g)&&p(g.createElement);n.exports=function(n){return y?g.createElement(n):{}}},function(n,i,o){var u=o(3),p=o(18);n.exports=function(n,i){try{p(u,n,i)}catch(o){u[n]=i}return i}},function(n,i,o){var u=o(141),p=Function.toString;"function"!=typeof u.inspectSource&&(u.inspectSource=function(n){return p.call(n)}),n.exports=u.inspectSource},function(n,i,o){var u=o(3),p=o(139),g=u["__core-js_shared__"]||p("__core-js_shared__",{});n.exports=g},function(n,i,o){var u=o(56),p=o(141);(n.exports=function(n,i){return p[n]||(p[n]=void 0!==i?i:{})})("versions",[]).push({version:"3.8.2",mode:u?"pure":"global",copyright:"© 2021 Denis Pushkarev (zloirock.ru)"})},function(n,i){n.exports=["constructor","hasOwnProperty","isPrototypeOf","propertyIsEnumerable","toLocaleString","toString","valueOf"]},function(n,i){i.f=Object.getOwnPropertySymbols},function(n,i,o){var u=o(4),p=/#|\.prototype\./,isForced=function(n,i){var o=y[g(n)];return o==x||o!=w&&("function"==typeof i?u(i):!!i)},g=isForced.normalize=function(n){return String(n).replace(p,".").toLowerCase()},y=isForced.data={},w=isForced.NATIVE="N",x=isForced.POLYFILL="P";n.exports=isForced},function(n,i,o){var u=o(4);n.exports=!!Object.getOwnPropertySymbols&&!u((function(){return!String(Symbol())}))},function(n,i,o){"use strict";var u=o(55),p=o(16),g=o(40);n.exports=function(n,i,o){var y=u(i);y in n?p.f(n,y,g(0,o)):n[y]=o}},function(n,i,o){var u,p,g=o(3),y=o(149),w=g.process,x=w&&w.versions,_=x&&x.v8;_?p=(u=_.split("."))[0]+u[1]:y&&(!(u=y.match(/Edge\/(\d+)/))||u[1]>=74)&&(u=y.match(/Chrome\/(\d+)/))&&(p=u[1]),n.exports=p&&+p},function(n,i,o){var u=o(34);n.exports=u("navigator","userAgent")||""},function(n,i,o){"use strict";var u=o(19),p=o(43),g=o(8);n.exports=function fill(n){for(var i=u(this),o=g(i.length),y=arguments.length,w=p(y>1?arguments[1]:void 0,o),x=y>2?arguments[2]:void 0,_=void 0===x?o:p(x,o);_>w;)i[w++]=n;return i}},function(n,i,o){var u=o(6),p=o(58),g=o(16),y=u("unscopables"),w=Array.prototype;null==w[y]&&g.f(w,y,{configurable:!0,value:p(null)}),n.exports=function(n){w[y][n]=!0}},function(n,i,o){var u=o(6),p=o(97),g=u("iterator"),y=Array.prototype;n.exports=function(n){return void 0!==n&&(p.Array===n||y[g]===n)}},function(n,i,o){var u=o(117),p=o(97),g=o(6)("iterator");n.exports=function(n){if(null!=n)return n[g]||n["@@iterator"]||p[u(n)]}},function(n,i,o){var u={};u[o(6)("toStringTag")]="z",n.exports="[object z]"===String(u)},function(n,i,o){var u=o(6)("iterator"),p=!1;try{var g=0,y={next:function(){return{done:!!g++}},return:function(){p=!0}};y[u]=function(){return this},Array.from(y,(function(){throw 2}))}catch(n){}n.exports=function(n,i){if(!i&&!p)return!1;var o=!1;try{var g={};g[u]=function(){return{next:function(){return{done:o=!0}}}},n(g)}catch(n){}return o}},function(n,i,o){"use strict";var u=o(3),p=o(13),g=o(221),y=o(18),w=o(222),x=o(4),_=o(157),k=o(35),P=o(8),E=o(223),O=o(325),B=o(60),I=o(45),D=o(57).f,R=o(16).f,N=o(150),U=o(95),W=o(42),G=W.get,j=W.set,X=u.ArrayBuffer,Y=X,K=u.DataView,J=K&&K.prototype,$=Object.prototype,tt=u.RangeError,et=O.pack,rt=O.unpack,packInt8=function(n){return[255&n]},packInt16=function(n){return[255&n,n>>8&255]},packInt32=function(n){return[255&n,n>>8&255,n>>16&255,n>>24&255]},unpackInt32=function(n){return n[3]<<24|n[2]<<16|n[1]<<8|n[0]},packFloat32=function(n){return et(n,23,4)},packFloat64=function(n){return et(n,52,8)},addGetter=function(n,i){R(n.prototype,i,{get:function(){return G(this)[i]}})},get=function(n,i,o,u){var p=E(o),g=G(n);if(p+i>g.byteLength)throw tt("Wrong index");var y=G(g.buffer).bytes,w=p+g.byteOffset,x=y.slice(w,w+i);return u?x:x.reverse()},set=function(n,i,o,u,p,g){var y=E(o),w=G(n);if(y+i>w.byteLength)throw tt("Wrong index");for(var x=G(w.buffer).bytes,_=y+w.byteOffset,k=u(+p),P=0;P<i;P++)x[_+P]=k[g?P:i-P-1]};if(g){if(!x((function(){X(1)}))||!x((function(){new X(-1)}))||x((function(){return new X,new X(1.5),new X(NaN),"ArrayBuffer"!=X.name}))){for(var nt,it=(Y=function ArrayBuffer(n){return _(this,Y),new X(E(n))}).prototype=X.prototype,ot=D(X),at=0;ot.length>at;)(nt=ot[at++])in Y||y(Y,nt,X[nt]);it.constructor=Y}I&&B(J)!==$&&I(J,$);var st=new K(new Y(2)),lt=J.setInt8;st.setInt8(0,2147483648),st.setInt8(1,2147483649),!st.getInt8(0)&&st.getInt8(1)||w(J,{setInt8:function setInt8(n,i){lt.call(this,n,i<<24>>24)},setUint8:function setUint8(n,i){lt.call(this,n,i<<24>>24)}},{unsafe:!0})}else Y=function ArrayBuffer(n){_(this,Y,"ArrayBuffer");var i=E(n);j(this,{bytes:N.call(new Array(i),0),byteLength:i}),p||(this.byteLength=i)},K=function DataView(n,i,o){_(this,K,"DataView"),_(n,Y,"DataView");var u=G(n).byteLength,g=k(i);if(g<0||g>u)throw tt("Wrong offset");if(g+(o=void 0===o?u-g:P(o))>u)throw tt("Wrong length");j(this,{buffer:n,byteLength:o,byteOffset:g}),p||(this.buffer=n,this.byteLength=o,this.byteOffset=g)},p&&(addGetter(Y,"byteLength"),addGetter(K,"buffer"),addGetter(K,"byteLength"),addGetter(K,"byteOffset")),w(K.prototype,{getInt8:function getInt8(n){return get(this,1,n)[0]<<24>>24},getUint8:function getUint8(n){return get(this,1,n)[0]},getInt16:function getInt16(n){var i=get(this,2,n,arguments.length>1?arguments[1]:void 0);return(i[1]<<8|i[0])<<16>>16},getUint16:function getUint16(n){var i=get(this,2,n,arguments.length>1?arguments[1]:void 0);return i[1]<<8|i[0]},getInt32:function getInt32(n){return unpackInt32(get(this,4,n,arguments.length>1?arguments[1]:void 0))},getUint32:function getUint32(n){return unpackInt32(get(this,4,n,arguments.length>1?arguments[1]:void 0))>>>0},getFloat32:function getFloat32(n){return rt(get(this,4,n,arguments.length>1?arguments[1]:void 0),23)},getFloat64:function getFloat64(n){return rt(get(this,8,n,arguments.length>1?arguments[1]:void 0),52)},setInt8:function setInt8(n,i){set(this,1,n,packInt8,i)},setUint8:function setUint8(n,i){set(this,1,n,packInt8,i)},setInt16:function setInt16(n,i){set(this,2,n,packInt16,i,arguments.length>2?arguments[2]:void 0)},setUint16:function setUint16(n,i){set(this,2,n,packInt16,i,arguments.length>2?arguments[2]:void 0)},setInt32:function setInt32(n,i){set(this,4,n,packInt32,i,arguments.length>2?arguments[2]:void 0)},setUint32:function setUint32(n,i){set(this,4,n,packInt32,i,arguments.length>2?arguments[2]:void 0)},setFloat32:function setFloat32(n,i){set(this,4,n,packFloat32,i,arguments.length>2?arguments[2]:void 0)},setFloat64:function setFloat64(n,i){set(this,8,n,packFloat64,i,arguments.length>2?arguments[2]:void 0)}});U(Y,"ArrayBuffer"),U(K,"DataView"),n.exports={ArrayBuffer:Y,DataView:K}},function(n,i){n.exports=function(n,i,o){if(!(n instanceof i))throw TypeError("Incorrect "+(o?o+" ":"")+"invocation");return n}},function(n,i,o){"use strict";var u=o(34),p=o(16),g=o(6),y=o(13),w=g("species");n.exports=function(n){var i=u(n),o=p.f;y&&i&&!i[w]&&o(i,w,{configurable:!0,get:function(){return this}})}},function(n,i,o){var u=o(35),p=o(33),createMethod=function(n){return function(i,o){var g,y,w=String(p(i)),x=u(o),_=w.length;return x<0||x>=_?n?"":void 0:(g=w.charCodeAt(x))<55296||g>56319||x+1===_||(y=w.charCodeAt(x+1))<56320||y>57343?n?w.charAt(x):g:n?w.slice(x,x+2):y-56320+(g-55296<<10)+65536}};n.exports={codeAt:createMethod(!1),charAt:createMethod(!0)}},function(n,i,o){"use strict";var u=o(159).charAt,p=o(42),g=o(217),y=p.set,w=p.getterFor("String Iterator");g(String,"String",(function(n){y(this,{type:"String Iterator",string:String(n),index:0})}),(function next(){var n,i=w(this),o=i.string,p=i.index;return p>=o.length?{value:void 0,done:!0}:(n=u(o,p),i.index+=n.length,{value:n,done:!1})}))},function(n,i,o){"use strict";var u=o(162),p=o(11),g=o(8),y=o(33),w=o(163),x=o(164);u("match",1,(function(n,i,o){return[function match(i){var o=y(this),u=null==i?void 0:i[n];return void 0!==u?u.call(i,o):new RegExp(i)[n](String(o))},function(n){var u=o(i,n,this);if(u.done)return u.value;var y=p(n),_=String(this);if(!y.global)return x(y,_);var k=y.unicode;y.lastIndex=0;for(var P,E=[],O=0;null!==(P=x(y,_));){var B=String(P[0]);E[O]=B,""===B&&(y.lastIndex=w(_,g(y.lastIndex),k)),O++}return 0===O?null:E}]}))},function(n,i,o){"use strict";o(118);var u=o(25),p=o(4),g=o(6),y=o(119),w=o(18),x=g("species"),_=!p((function(){var n=/./;return n.exec=function(){var n=[];return n.groups={a:"7"},n},"7"!=="".replace(n,"$<a>")})),k="$0"==="a".replace(/./,"$0"),P=g("replace"),E=!!/./[P]&&""===/./[P]("a","$0"),O=!p((function(){var n=/(?:)/,i=n.exec;n.exec=function(){return i.apply(this,arguments)};var o="ab".split(n);return 2!==o.length||"a"!==o[0]||"b"!==o[1]}));n.exports=function(n,i,o,P){var B=g(n),I=!p((function(){var i={};return i[B]=function(){return 7},7!=""[n](i)})),D=I&&!p((function(){var i=!1,o=/a/;return"split"===n&&((o={}).constructor={},o.constructor[x]=function(){return o},o.flags="",o[B]=/./[B]),o.exec=function(){return i=!0,null},o[B](""),!i}));if(!I||!D||"replace"===n&&(!_||!k||E)||"split"===n&&!O){var R=/./[B],N=o(B,""[n],(function(n,i,o,u,p){return i.exec===y?I&&!p?{done:!0,value:R.call(i,o,u)}:{done:!0,value:n.call(o,i,u)}:{done:!1}}),{REPLACE_KEEPS_$0:k,REGEXP_REPLACE_SUBSTITUTES_UNDEFINED_CAPTURE:E}),U=N[0],W=N[1];u(String.prototype,n,U),u(RegExp.prototype,B,2==i?function(n,i){return W.call(n,this,i)}:function(n){return W.call(n,this)})}P&&w(RegExp.prototype[B],"sham",!0)}},function(n,i,o){"use strict";var u=o(159).charAt;n.exports=function(n,i,o){return i+(o?u(n,i).length:1)}},function(n,i,o){var u=o(41),p=o(119);n.exports=function(n,i){var o=n.exec;if("function"==typeof o){var g=o.call(n,i);if("object"!=typeof g)throw TypeError("RegExp exec method returned something other than an Object or null");return g}if("RegExp"!==u(n))throw TypeError("RegExp#exec called on incompatible receiver");return p.call(n,i)}},function(n,i,o){(i=n.exports=o(241)).Stream=i,i.Readable=i,i.Writable=o(166),i.Duplex=o(48),i.Transform=o(244),i.PassThrough=o(351)},function(n,i,o){"use strict";(function(i,u){var p=o(123);function CorkedRequest(n){var i=this;this.next=null,this.entry=null,this.finish=function(){!function onCorkedFinish(n,i,o){var u=n.entry;n.entry=null;for(;u;){var p=u.callback;i.pendingcb--,p(o),u=u.next}i.corkedRequestsFree?i.corkedRequestsFree.next=n:i.corkedRequestsFree=n}(i,n)}}n.exports=Writable;var g,y=!i.browser&&["v0.10","v0.9."].indexOf(i.version.slice(0,5))>-1?setImmediate:p.nextTick;Writable.WritableState=WritableState;var w=Object.create(o(98));w.inherits=o(86);var x={deprecate:o(350)},_=o(242),k=o(124).Buffer,P=u.Uint8Array||function(){};var E,O=o(243);function nop(){}function WritableState(n,i){g=g||o(48),n=n||{};var u=i instanceof g;this.objectMode=!!n.objectMode,u&&(this.objectMode=this.objectMode||!!n.writableObjectMode);var w=n.highWaterMark,x=n.writableHighWaterMark,_=this.objectMode?16:16384;this.highWaterMark=w||0===w?w:u&&(x||0===x)?x:_,this.highWaterMark=Math.floor(this.highWaterMark),this.finalCalled=!1,this.needDrain=!1,this.ending=!1,this.ended=!1,this.finished=!1,this.destroyed=!1;var k=!1===n.decodeStrings;this.decodeStrings=!k,this.defaultEncoding=n.defaultEncoding||"utf8",this.length=0,this.writing=!1,this.corked=0,this.sync=!0,this.bufferProcessing=!1,this.onwrite=function(n){!function onwrite(n,i){var o=n._writableState,u=o.sync,g=o.writecb;if(function onwriteStateUpdate(n){n.writing=!1,n.writecb=null,n.length-=n.writelen,n.writelen=0}(o),i)!function onwriteError(n,i,o,u,g){--i.pendingcb,o?(p.nextTick(g,u),p.nextTick(finishMaybe,n,i),n._writableState.errorEmitted=!0,n.emit("error",u)):(g(u),n._writableState.errorEmitted=!0,n.emit("error",u),finishMaybe(n,i))}(n,o,u,i,g);else{var w=needFinish(o);w||o.corked||o.bufferProcessing||!o.bufferedRequest||clearBuffer(n,o),u?y(afterWrite,n,o,w,g):afterWrite(n,o,w,g)}}(i,n)},this.writecb=null,this.writelen=0,this.bufferedRequest=null,this.lastBufferedRequest=null,this.pendingcb=0,this.prefinished=!1,this.errorEmitted=!1,this.bufferedRequestCount=0,this.corkedRequestsFree=new CorkedRequest(this)}function Writable(n){if(g=g||o(48),!(E.call(Writable,this)||this instanceof g))return new Writable(n);this._writableState=new WritableState(n,this),this.writable=!0,n&&("function"==typeof n.write&&(this._write=n.write),"function"==typeof n.writev&&(this._writev=n.writev),"function"==typeof n.destroy&&(this._destroy=n.destroy),"function"==typeof n.final&&(this._final=n.final)),_.call(this)}function doWrite(n,i,o,u,p,g,y){i.writelen=u,i.writecb=y,i.writing=!0,i.sync=!0,o?n._writev(p,i.onwrite):n._write(p,g,i.onwrite),i.sync=!1}function afterWrite(n,i,o,u){o||function onwriteDrain(n,i){0===i.length&&i.needDrain&&(i.needDrain=!1,n.emit("drain"))}(n,i),i.pendingcb--,u(),finishMaybe(n,i)}function clearBuffer(n,i){i.bufferProcessing=!0;var o=i.bufferedRequest;if(n._writev&&o&&o.next){var u=i.bufferedRequestCount,p=new Array(u),g=i.corkedRequestsFree;g.entry=o;for(var y=0,w=!0;o;)p[y]=o,o.isBuf||(w=!1),o=o.next,y+=1;p.allBuffers=w,doWrite(n,i,!0,i.length,p,"",g.finish),i.pendingcb++,i.lastBufferedRequest=null,g.next?(i.corkedRequestsFree=g.next,g.next=null):i.corkedRequestsFree=new CorkedRequest(i),i.bufferedRequestCount=0}else{for(;o;){var x=o.chunk,_=o.encoding,k=o.callback;if(doWrite(n,i,!1,i.objectMode?1:x.length,x,_,k),o=o.next,i.bufferedRequestCount--,i.writing)break}null===o&&(i.lastBufferedRequest=null)}i.bufferedRequest=o,i.bufferProcessing=!1}function needFinish(n){return n.ending&&0===n.length&&null===n.bufferedRequest&&!n.finished&&!n.writing}function callFinal(n,i){n._final((function(o){i.pendingcb--,o&&n.emit("error",o),i.prefinished=!0,n.emit("prefinish"),finishMaybe(n,i)}))}function finishMaybe(n,i){var o=needFinish(i);return o&&(!function prefinish(n,i){i.prefinished||i.finalCalled||("function"==typeof n._final?(i.pendingcb++,i.finalCalled=!0,p.nextTick(callFinal,n,i)):(i.prefinished=!0,n.emit("prefinish")))}(n,i),0===i.pendingcb&&(i.finished=!0,n.emit("finish"))),o}w.inherits(Writable,_),WritableState.prototype.getBuffer=function getBuffer(){for(var n=this.bufferedRequest,i=[];n;)i.push(n),n=n.next;return i},function(){try{Object.defineProperty(WritableState.prototype,"buffer",{get:x.deprecate((function(){return this.getBuffer()}),"_writableState.buffer is deprecated. Use _writableState.getBuffer instead.","DEP0003")})}catch(n){}}(),"function"==typeof Symbol&&Symbol.hasInstance&&"function"==typeof Function.prototype[Symbol.hasInstance]?(E=Function.prototype[Symbol.hasInstance],Object.defineProperty(Writable,Symbol.hasInstance,{value:function(n){return!!E.call(this,n)||this===Writable&&(n&&n._writableState instanceof WritableState)}})):E=function(n){return n instanceof this},Writable.prototype.pipe=function(){this.emit("error",new Error("Cannot pipe, not readable"))},Writable.prototype.write=function(n,i,o){var u=this._writableState,g=!1,y=!u.objectMode&&function _isUint8Array(n){return k.isBuffer(n)||n instanceof P}(n);return y&&!k.isBuffer(n)&&(n=function _uint8ArrayToBuffer(n){return k.from(n)}(n)),"function"==typeof i&&(o=i,i=null),y?i="buffer":i||(i=u.defaultEncoding),"function"!=typeof o&&(o=nop),u.ended?function writeAfterEnd(n,i){var o=new Error("write after end");n.emit("error",o),p.nextTick(i,o)}(this,o):(y||function validChunk(n,i,o,u){var g=!0,y=!1;return null===o?y=new TypeError("May not write null values to stream"):"string"==typeof o||void 0===o||i.objectMode||(y=new TypeError("Invalid non-string/buffer chunk")),y&&(n.emit("error",y),p.nextTick(u,y),g=!1),g}(this,u,n,o))&&(u.pendingcb++,g=function writeOrBuffer(n,i,o,u,p,g){if(!o){var y=function decodeChunk(n,i,o){n.objectMode||!1===n.decodeStrings||"string"!=typeof i||(i=k.from(i,o));return i}(i,u,p);u!==y&&(o=!0,p="buffer",u=y)}var w=i.objectMode?1:u.length;i.length+=w;var x=i.length<i.highWaterMark;x||(i.needDrain=!0);if(i.writing||i.corked){var _=i.lastBufferedRequest;i.lastBufferedRequest={chunk:u,encoding:p,isBuf:o,callback:g,next:null},_?_.next=i.lastBufferedRequest:i.bufferedRequest=i.lastBufferedRequest,i.bufferedRequestCount+=1}else doWrite(n,i,!1,w,u,p,g);return x}(this,u,y,n,i,o)),g},Writable.prototype.cork=function(){this._writableState.corked++},Writable.prototype.uncork=function(){var n=this._writableState;n.corked&&(n.corked--,n.writing||n.corked||n.finished||n.bufferProcessing||!n.bufferedRequest||clearBuffer(this,n))},Writable.prototype.setDefaultEncoding=function setDefaultEncoding(n){if("string"==typeof n&&(n=n.toLowerCase()),!(["hex","utf8","utf-8","ascii","binary","base64","ucs2","ucs-2","utf16le","utf-16le","raw"].indexOf((n+"").toLowerCase())>-1))throw new TypeError("Unknown encoding: "+n);return this._writableState.defaultEncoding=n,this},Object.defineProperty(Writable.prototype,"writableHighWaterMark",{enumerable:!1,get:function(){return this._writableState.highWaterMark}}),Writable.prototype._write=function(n,i,o){o(new Error("_write() is not implemented"))},Writable.prototype._writev=null,Writable.prototype.end=function(n,i,o){var u=this._writableState;"function"==typeof n?(o=n,n=null,i=null):"function"==typeof i&&(o=i,i=null),null!=n&&this.write(n,i),u.corked&&(u.corked=1,this.uncork()),u.ending||u.finished||function endWritable(n,i,o){i.ending=!0,finishMaybe(n,i),o&&(i.finished?p.nextTick(o):n.once("finish",o));i.ended=!0,n.writable=!1}(this,u,o)},Object.defineProperty(Writable.prototype,"destroyed",{get:function(){return void 0!==this._writableState&&this._writableState.destroyed},set:function(n){this._writableState&&(this._writableState.destroyed=n)}}),Writable.prototype.destroy=O.destroy,Writable.prototype._undestroy=O.undestroy,Writable.prototype._destroy=function(n,i){this.end(),i(n)}}).call(this,o(47),o(24))},function(n,i,o){(function(n){var u=Object.getOwnPropertyDescriptors||function getOwnPropertyDescriptors(n){for(var i=Object.keys(n),o={},u=0;u<i.length;u++)o[i[u]]=Object.getOwnPropertyDescriptor(n,i[u]);return o},p=/%[sdj%]/g;i.format=function(n){if(!isString(n)){for(var i=[],o=0;o<arguments.length;o++)i.push(inspect(arguments[o]));return i.join(" ")}o=1;for(var u=arguments,g=u.length,y=String(n).replace(p,(function(n){if("%%"===n)return"%";if(o>=g)return n;switch(n){case"%s":return String(u[o++]);case"%d":return Number(u[o++]);case"%j":try{return JSON.stringify(u[o++])}catch(n){return"[Circular]"}default:return n}})),w=u[o];o<g;w=u[++o])isNull(w)||!isObject(w)?y+=" "+w:y+=" "+inspect(w);return y},i.deprecate=function(o,u){if(void 0!==n&&!0===n.noDeprecation)return o;if(void 0===n)return function(){return i.deprecate(o,u).apply(this,arguments)};var p=!1;return function deprecated(){if(!p){if(n.throwDeprecation)throw new Error(u);n.traceDeprecation,p=!0}return o.apply(this,arguments)}};var g,y={};function inspect(n,o){var u={seen:[],stylize:stylizeNoColor};return arguments.length>=3&&(u.depth=arguments[2]),arguments.length>=4&&(u.colors=arguments[3]),isBoolean(o)?u.showHidden=o:o&&i._extend(u,o),isUndefined(u.showHidden)&&(u.showHidden=!1),isUndefined(u.depth)&&(u.depth=2),isUndefined(u.colors)&&(u.colors=!1),isUndefined(u.customInspect)&&(u.customInspect=!0),u.colors&&(u.stylize=stylizeWithColor),formatValue(u,n,u.depth)}function stylizeWithColor(n,i){var o=inspect.styles[i];return o?"["+inspect.colors[o][0]+"m"+n+"["+inspect.colors[o][1]+"m":n}function stylizeNoColor(n,i){return n}function formatValue(n,o,u){if(n.customInspect&&o&&isFunction(o.inspect)&&o.inspect!==i.inspect&&(!o.constructor||o.constructor.prototype!==o)){var p=o.inspect(u,n);return isString(p)||(p=formatValue(n,p,u)),p}var g=function formatPrimitive(n,i){if(isUndefined(i))return n.stylize("undefined","undefined");if(isString(i)){var o="'"+JSON.stringify(i).replace(/^"|"$/g,"").replace(/'/g,"\\'").replace(/\\"/g,'"')+"'";return n.stylize(o,"string")}if(isNumber(i))return n.stylize(""+i,"number");if(isBoolean(i))return n.stylize(""+i,"boolean");if(isNull(i))return n.stylize("null","null")}(n,o);if(g)return g;var y=Object.keys(o),w=function arrayToHash(n){var i={};return n.forEach((function(n,o){i[n]=!0})),i}(y);if(n.showHidden&&(y=Object.getOwnPropertyNames(o)),isError(o)&&(y.indexOf("message")>=0||y.indexOf("description")>=0))return formatError(o);if(0===y.length){if(isFunction(o)){var x=o.name?": "+o.name:"";return n.stylize("[Function"+x+"]","special")}if(isRegExp(o))return n.stylize(RegExp.prototype.toString.call(o),"regexp");if(isDate(o))return n.stylize(Date.prototype.toString.call(o),"date");if(isError(o))return formatError(o)}var _,k="",P=!1,E=["{","}"];(isArray(o)&&(P=!0,E=["[","]"]),isFunction(o))&&(k=" [Function"+(o.name?": "+o.name:"")+"]");return isRegExp(o)&&(k=" "+RegExp.prototype.toString.call(o)),isDate(o)&&(k=" "+Date.prototype.toUTCString.call(o)),isError(o)&&(k=" "+formatError(o)),0!==y.length||P&&0!=o.length?u<0?isRegExp(o)?n.stylize(RegExp.prototype.toString.call(o),"regexp"):n.stylize("[Object]","special"):(n.seen.push(o),_=P?function formatArray(n,i,o,u,p){for(var g=[],y=0,w=i.length;y<w;++y)hasOwnProperty(i,String(y))?g.push(formatProperty(n,i,o,u,String(y),!0)):g.push("");return p.forEach((function(p){p.match(/^\d+$/)||g.push(formatProperty(n,i,o,u,p,!0))})),g}(n,o,u,w,y):y.map((function(i){return formatProperty(n,o,u,w,i,P)})),n.seen.pop(),function reduceToSingleString(n,i,o){if(n.reduce((function(n,i){return i.indexOf("\n")>=0&&0,n+i.replace(/\u001b\[\d\d?m/g,"").length+1}),0)>60)return o[0]+(""===i?"":i+"\n ")+" "+n.join(",\n  ")+" "+o[1];return o[0]+i+" "+n.join(", ")+" "+o[1]}(_,k,E)):E[0]+k+E[1]}function formatError(n){return"["+Error.prototype.toString.call(n)+"]"}function formatProperty(n,i,o,u,p,g){var y,w,x;if((x=Object.getOwnPropertyDescriptor(i,p)||{value:i[p]}).get?w=x.set?n.stylize("[Getter/Setter]","special"):n.stylize("[Getter]","special"):x.set&&(w=n.stylize("[Setter]","special")),hasOwnProperty(u,p)||(y="["+p+"]"),w||(n.seen.indexOf(x.value)<0?(w=isNull(o)?formatValue(n,x.value,null):formatValue(n,x.value,o-1)).indexOf("\n")>-1&&(w=g?w.split("\n").map((function(n){return"  "+n})).join("\n").substr(2):"\n"+w.split("\n").map((function(n){return"   "+n})).join("\n")):w=n.stylize("[Circular]","special")),isUndefined(y)){if(g&&p.match(/^\d+$/))return w;(y=JSON.stringify(""+p)).match(/^"([a-zA-Z_][a-zA-Z_0-9]*)"$/)?(y=y.substr(1,y.length-2),y=n.stylize(y,"name")):(y=y.replace(/'/g,"\\'").replace(/\\"/g,'"').replace(/(^"|"$)/g,"'"),y=n.stylize(y,"string"))}return y+": "+w}function isArray(n){return Array.isArray(n)}function isBoolean(n){return"boolean"==typeof n}function isNull(n){return null===n}function isNumber(n){return"number"==typeof n}function isString(n){return"string"==typeof n}function isUndefined(n){return void 0===n}function isRegExp(n){return isObject(n)&&"[object RegExp]"===objectToString(n)}function isObject(n){return"object"==typeof n&&null!==n}function isDate(n){return isObject(n)&&"[object Date]"===objectToString(n)}function isError(n){return isObject(n)&&("[object Error]"===objectToString(n)||n instanceof Error)}function isFunction(n){return"function"==typeof n}function objectToString(n){return Object.prototype.toString.call(n)}i.debuglog=function(o){if(isUndefined(g)&&(g=n.env.NODE_DEBUG||""),o=o.toUpperCase(),!y[o])if(new RegExp("\\b"+o+"\\b","i").test(g)){n.pid;y[o]=function(){i.format.apply(i,arguments)}}else y[o]=function(){};return y[o]},i.inspect=inspect,inspect.colors={bold:[1,22],italic:[3,23],underline:[4,24],inverse:[7,27],white:[37,39],grey:[90,39],black:[30,39],blue:[34,39],cyan:[36,39],green:[32,39],magenta:[35,39],red:[31,39],yellow:[33,39]},inspect.styles={special:"cyan",number:"yellow",boolean:"yellow",undefined:"grey",null:"bold",string:"green",date:"magenta",regexp:"red"},i.isArray=isArray,i.isBoolean=isBoolean,i.isNull=isNull,i.isNullOrUndefined=function isNullOrUndefined(n){return null==n},i.isNumber=isNumber,i.isString=isString,i.isSymbol=function isSymbol(n){return"symbol"==typeof n},i.isUndefined=isUndefined,i.isRegExp=isRegExp,i.isObject=isObject,i.isDate=isDate,i.isError=isError,i.isFunction=isFunction,i.isPrimitive=function isPrimitive(n){return null===n||"boolean"==typeof n||"number"==typeof n||"string"==typeof n||"symbol"==typeof n||void 0===n},i.isBuffer=o(358);function hasOwnProperty(n,i){return Object.prototype.hasOwnProperty.call(n,i)}i.log=function(){},i.inherits=o(359),i._extend=function(n,i){if(!i||!isObject(i))return n;for(var o=Object.keys(i),u=o.length;u--;)n[o[u]]=i[o[u]];return n};var w="undefined"!=typeof Symbol?Symbol("util.promisify.custom"):void 0;function callbackifyOnRejected(n,i){if(!n){var o=new Error("Promise was rejected with a falsy value");o.reason=n,n=o}return i(n)}i.promisify=function promisify(n){if("function"!=typeof n)throw new TypeError('The "original" argument must be of type Function');if(w&&n[w]){var i;if("function"!=typeof(i=n[w]))throw new TypeError('The "util.promisify.custom" argument must be of type Function');return Object.defineProperty(i,w,{value:i,enumerable:!1,writable:!1,configurable:!0}),i}function i(){for(var i,o,u=new Promise((function(n,u){i=n,o=u})),p=[],g=0;g<arguments.length;g++)p.push(arguments[g]);p.push((function(n,u){n?o(n):i(u)}));try{n.apply(this,p)}catch(n){o(n)}return u}return Object.setPrototypeOf(i,Object.getPrototypeOf(n)),w&&Object.defineProperty(i,w,{value:i,enumerable:!1,writable:!1,configurable:!0}),Object.defineProperties(i,u(n))},i.promisify.custom=w,i.callbackify=function callbackify(i){if("function"!=typeof i)throw new TypeError('The "original" argument must be of type Function');function callbackified(){for(var o=[],u=0;u<arguments.length;u++)o.push(arguments[u]);var p=o.pop();if("function"!=typeof p)throw new TypeError("The last argument must be of type Function");var g=this,cb=function(){return p.apply(g,arguments)};i.apply(this,o).then((function(i){n.nextTick(cb,null,i)}),(function(i){n.nextTick(callbackifyOnRejected,i,cb)}))}return Object.setPrototypeOf(callbackified,Object.getPrototypeOf(i)),Object.defineProperties(callbackified,u(i)),callbackified}}).call(this,o(47))},function(n,i,o){var u,p,g,y,w,x,_,k;n.exports=(k=o(2),p=(u=k).lib,g=p.WordArray,y=p.Hasher,w=u.algo,x=[],_=w.SHA1=y.extend({_doReset:function(){this._hash=new g.init([1732584193,4023233417,2562383102,271733878,3285377520])},_doProcessBlock:function(n,i){for(var o=this._hash.words,u=o[0],p=o[1],g=o[2],y=o[3],w=o[4],_=0;_<80;_++){if(_<16)x[_]=0|n[i+_];else{var k=x[_-3]^x[_-8]^x[_-14]^x[_-16];x[_]=k<<1|k>>>31}var P=(u<<5|u>>>27)+w+x[_];P+=_<20?1518500249+(p&g|~p&y):_<40?1859775393+(p^g^y):_<60?(p&g|p&y|g&y)-1894007588:(p^g^y)-899497514,w=y,y=g,g=p<<30|p>>>2,p=u,u=P}o[0]=o[0]+u|0,o[1]=o[1]+p|0,o[2]=o[2]+g|0,o[3]=o[3]+y|0,o[4]=o[4]+w|0},_doFinalize:function(){var n=this._data,i=n.words,o=8*this._nDataBytes,u=8*n.sigBytes;return i[u>>>5]|=128<<24-u%32,i[14+(u+64>>>9<<4)]=Math.floor(o/4294967296),i[15+(u+64>>>9<<4)]=o,n.sigBytes=4*i.length,this._process(),this._hash},clone:function(){var n=y.clone.call(this);return n._hash=this._hash.clone(),n}}),u.SHA1=y._createHelper(_),u.HmacSHA1=y._createHmacHelper(_),k.SHA1)},function(n,i,o){var u,p,g,y;n.exports=(u=o(2),g=(p=u).lib.Base,y=p.enc.Utf8,void(p.algo.HMAC=g.extend({init:function(n,i){n=this._hasher=new n.init,"string"==typeof i&&(i=y.parse(i));var o=n.blockSize,u=4*o;i.sigBytes>u&&(i=n.finalize(i)),i.clamp();for(var p=this._oKey=i.clone(),g=this._iKey=i.clone(),w=p.words,x=g.words,_=0;_<o;_++)w[_]^=1549556828,x[_]^=909522486;p.sigBytes=g.sigBytes=u,this.reset()},reset:function(){var n=this._hasher;n.reset(),n.update(this._iKey)},update:function(n){return this._hasher.update(n),this},finalize:function(n){var i=this._hasher,o=i.finalize(n);return i.reset(),i.finalize(this._oKey.clone().concat(o))}})))},function(n,i,o){(function(i){(function(){var u,p;try{p=o(171)}catch(n){}u=function(){var n;function DecodeStream(n){this.buffer=n,this.pos=0,this.length=this.buffer.length}for(n in DecodeStream.TYPES={UInt8:1,UInt16:2,UInt24:3,UInt32:4,Int8:1,Int16:2,Int24:3,Int32:4,Float:4,Double:8},i.prototype)"read"===n.slice(0,4)&&function(n){var i;i=DecodeStream.TYPES[n.replace(/read|[BL]E/g,"")],DecodeStream.prototype[n]=function(){var o;return o=this.buffer[n](this.pos),this.pos+=i,o}}(n);return DecodeStream.prototype.readString=function(n,o){var u,g,y,w,x;switch(null==o&&(o="ascii"),o){case"utf16le":case"ucs2":case"utf8":case"ascii":return this.buffer.toString(o,this.pos,this.pos+=n);case"utf16be":for(y=w=0,x=(u=new i(this.readBuffer(n))).length-1;w<x;y=w+=2)g=u[y],u[y]=u[y+1],u[y+1]=g;return u.toString("utf16le");default:if(u=this.readBuffer(n),p)try{return p.decode(u,o)}catch(n){}return u}},DecodeStream.prototype.readBuffer=function(n){return this.buffer.slice(this.pos,this.pos+=n)},DecodeStream.prototype.readUInt24BE=function(){return(this.readUInt16BE()<<8)+this.readUInt8()},DecodeStream.prototype.readUInt24LE=function(){return this.readUInt16LE()+(this.readUInt8()<<16)},DecodeStream.prototype.readInt24BE=function(){return(this.readInt16BE()<<8)+this.readUInt8()},DecodeStream.prototype.readInt24LE=function(){return this.readUInt16LE()+(this.readInt8()<<16)},DecodeStream}(),n.exports=u}).call(this)}).call(this,o(10).Buffer)},function(n,i,o){"use strict";var u,p=o(37).Buffer,g=o(395),y=n.exports;y.encodings=null,y.defaultCharUnicode="�",y.defaultCharSingleByte="?",y.encode=function encode(n,i,o){n=""+(n||"");var u=y.getEncoder(i,o),g=u.write(n),w=u.end();return w&&w.length>0?p.concat([g,w]):g},y.decode=function decode(n,i,o){"string"==typeof n&&(y.skipDecodeWarning||(y.skipDecodeWarning=!0),n=p.from(""+(n||""),"binary"));var u=y.getDecoder(i,o),g=u.write(n),w=u.end();return w?g+w:g},y.encodingExists=function encodingExists(n){try{return y.getCodec(n),!0}catch(n){return!1}},y.toEncoding=y.encode,y.fromEncoding=y.decode,y._codecDataCache={},y.getCodec=function getCodec(n){y.encodings||(y.encodings=o(396));for(var i=y._canonicalizeEncoding(n),u={};;){var p=y._codecDataCache[i];if(p)return p;var g=y.encodings[i];switch(typeof g){case"string":i=g;break;case"object":for(var w in g)u[w]=g[w];u.encodingName||(u.encodingName=i),i=g.type;break;case"function":return u.encodingName||(u.encodingName=i),p=new g(u,y),y._codecDataCache[u.encodingName]=p,p;default:throw new Error("Encoding not recognized: '"+n+"' (searched as: '"+i+"')")}}},y._canonicalizeEncoding=function(n){return(""+n).toLowerCase().replace(/:\d{4}$|[^0-9a-z]/g,"")},y.getEncoder=function getEncoder(n,i){var o=y.getCodec(n),u=new o.encoder(i,o);return o.bomAware&&i&&i.addBOM&&(u=new g.PrependBOM(u,i)),u},y.getDecoder=function getDecoder(n,i){var o=y.getCodec(n),u=new o.decoder(i,o);return!o.bomAware||i&&!1===i.stripBOM||(u=new g.StripBOM(u,i)),u},y.enableStreamingAPI=function enableStreamingAPI(n){if(!y.supportsStreams){var i=o(411)(n);y.IconvLiteEncoderStream=i.IconvLiteEncoderStream,y.IconvLiteDecoderStream=i.IconvLiteDecoderStream,y.encodeStream=function encodeStream(n,i){return new y.IconvLiteEncoderStream(y.getEncoder(n,i),i)},y.decodeStream=function decodeStream(n,i){return new y.IconvLiteDecoderStream(y.getDecoder(n,i),i)},y.supportsStreams=!0}};try{u=o(412)}catch(n){}u&&u.Transform?y.enableStreamingAPI(u):y.encodeStream=y.decodeStream=function(){throw new Error("iconv-lite Streaming API is not enabled. Use iconv.enableStreamingAPI(require('stream')); to enable it.")}},function(n){n.exports=JSON.parse('[["0","\\u0000",127,"€"],["8140","丂丄丅丆丏丒丗丟丠両丣並丩丮丯丱丳丵丷丼乀乁乂乄乆乊乑乕乗乚乛乢乣乤乥乧乨乪",5,"乲乴",9,"乿",6,"亇亊"],["8180","亐亖亗亙亜亝亞亣亪亯亰亱亴亶亷亸亹亼亽亾仈仌仏仐仒仚仛仜仠仢仦仧仩仭仮仯仱仴仸仹仺仼仾伀伂",6,"伋伌伒",4,"伜伝伡伣伨伩伬伭伮伱伳伵伷伹伻伾",4,"佄佅佇",5,"佒佔佖佡佢佦佨佪佫佭佮佱佲併佷佸佹佺佽侀侁侂侅來侇侊侌侎侐侒侓侕侖侘侙侚侜侞侟価侢"],["8240","侤侫侭侰",4,"侶",8,"俀俁係俆俇俈俉俋俌俍俒",4,"俙俛俠俢俤俥俧俫俬俰俲俴俵俶俷俹俻俼俽俿",11],["8280","個倎倐們倓倕倖倗倛倝倞倠倢倣値倧倫倯",10,"倻倽倿偀偁偂偄偅偆偉偊偋偍偐",4,"偖偗偘偙偛偝",7,"偦",5,"偭",8,"偸偹偺偼偽傁傂傃傄傆傇傉傊傋傌傎",20,"傤傦傪傫傭",4,"傳",6,"傼"],["8340","傽",17,"僐",5,"僗僘僙僛",10,"僨僩僪僫僯僰僱僲僴僶",4,"僼",9,"儈"],["8380","儉儊儌",5,"儓",13,"儢",28,"兂兇兊兌兎兏児兒兓兗兘兙兛兝",4,"兣兤兦內兩兪兯兲兺兾兿冃冄円冇冊冋冎冏冐冑冓冔冘冚冝冞冟冡冣冦",4,"冭冮冴冸冹冺冾冿凁凂凃凅凈凊凍凎凐凒",5],["8440","凘凙凚凜凞凟凢凣凥",5,"凬凮凱凲凴凷凾刄刅刉刋刌刏刐刓刔刕刜刞刟刡刢刣別刦刧刪刬刯刱刲刴刵刼刾剄",5,"剋剎剏剒剓剕剗剘"],["8480","剙剚剛剝剟剠剢剣剤剦剨剫剬剭剮剰剱剳",9,"剾劀劃",4,"劉",6,"劑劒劔",6,"劜劤劥劦劧劮劯劰労",9,"勀勁勂勄勅勆勈勊勌勍勎勏勑勓勔動勗務",5,"勠勡勢勣勥",10,"勱",7,"勻勼勽匁匂匃匄匇匉匊匋匌匎"],["8540","匑匒匓匔匘匛匜匞匟匢匤匥匧匨匩匫匬匭匯",9,"匼匽區卂卄卆卋卌卍卐協単卙卛卝卥卨卪卬卭卲卶卹卻卼卽卾厀厁厃厇厈厊厎厏"],["8580","厐",4,"厖厗厙厛厜厞厠厡厤厧厪厫厬厭厯",6,"厷厸厹厺厼厽厾叀參",4,"収叏叐叒叓叕叚叜叝叞叡叢叧叴叺叾叿吀吂吅吇吋吔吘吙吚吜吢吤吥吪吰吳吶吷吺吽吿呁呂呄呅呇呉呌呍呎呏呑呚呝",4,"呣呥呧呩",7,"呴呹呺呾呿咁咃咅咇咈咉咊咍咑咓咗咘咜咞咟咠咡"],["8640","咢咥咮咰咲咵咶咷咹咺咼咾哃哅哊哋哖哘哛哠",4,"哫哬哯哰哱哴",5,"哻哾唀唂唃唄唅唈唊",4,"唒唓唕",5,"唜唝唞唟唡唥唦"],["8680","唨唩唫唭唲唴唵唶唸唹唺唻唽啀啂啅啇啈啋",4,"啑啒啓啔啗",4,"啝啞啟啠啢啣啨啩啫啯",5,"啹啺啽啿喅喆喌喍喎喐喒喓喕喖喗喚喛喞喠",6,"喨",8,"喲喴営喸喺喼喿",4,"嗆嗇嗈嗊嗋嗎嗏嗐嗕嗗",4,"嗞嗠嗢嗧嗩嗭嗮嗰嗱嗴嗶嗸",4,"嗿嘂嘃嘄嘅"],["8740","嘆嘇嘊嘋嘍嘐",7,"嘙嘚嘜嘝嘠嘡嘢嘥嘦嘨嘩嘪嘫嘮嘯嘰嘳嘵嘷嘸嘺嘼嘽嘾噀",11,"噏",4,"噕噖噚噛噝",4],["8780","噣噥噦噧噭噮噯噰噲噳噴噵噷噸噹噺噽",7,"嚇",6,"嚐嚑嚒嚔",14,"嚤",10,"嚰",6,"嚸嚹嚺嚻嚽",12,"囋",8,"囕囖囘囙囜団囥",5,"囬囮囯囲図囶囷囸囻囼圀圁圂圅圇國",6],["8840","園",9,"圝圞圠圡圢圤圥圦圧圫圱圲圴",4,"圼圽圿坁坃坄坅坆坈坉坋坒",4,"坘坙坢坣坥坧坬坮坰坱坲坴坵坸坹坺坽坾坿垀"],["8880","垁垇垈垉垊垍",4,"垔",6,"垜垝垞垟垥垨垪垬垯垰垱垳垵垶垷垹",8,"埄",6,"埌埍埐埑埓埖埗埛埜埞埡埢埣埥",7,"埮埰埱埲埳埵埶執埻埼埾埿堁堃堄堅堈堉堊堌堎堏堐堒堓堔堖堗堘堚堛堜堝堟堢堣堥",4,"堫",4,"報堲堳場堶",7],["8940","堾",5,"塅",6,"塎塏塐塒塓塕塖塗塙",4,"塟",5,"塦",4,"塭",16,"塿墂墄墆墇墈墊墋墌"],["8980","墍",4,"墔",4,"墛墜墝墠",7,"墪",17,"墽墾墿壀壂壃壄壆",10,"壒壓壔壖",13,"壥",5,"壭壯壱売壴壵壷壸壺",7,"夃夅夆夈",4,"夎夐夑夒夓夗夘夛夝夞夠夡夢夣夦夨夬夰夲夳夵夶夻"],["8a40","夽夾夿奀奃奅奆奊奌奍奐奒奓奙奛",4,"奡奣奤奦",12,"奵奷奺奻奼奾奿妀妅妉妋妌妎妏妐妑妔妕妘妚妛妜妝妟妠妡妢妦"],["8a80","妧妬妭妰妱妳",5,"妺妼妽妿",6,"姇姈姉姌姍姎姏姕姖姙姛姞",4,"姤姦姧姩姪姫姭",11,"姺姼姽姾娀娂娊娋娍娎娏娐娒娔娕娖娗娙娚娛娝娞娡娢娤娦娧娨娪",6,"娳娵娷",4,"娽娾娿婁",4,"婇婈婋",9,"婖婗婘婙婛",5],["8b40","婡婣婤婥婦婨婩婫",8,"婸婹婻婼婽婾媀",17,"媓",6,"媜",13,"媫媬"],["8b80","媭",4,"媴媶媷媹",4,"媿嫀嫃",5,"嫊嫋嫍",4,"嫓嫕嫗嫙嫚嫛嫝嫞嫟嫢嫤嫥嫧嫨嫪嫬",4,"嫲",22,"嬊",11,"嬘",25,"嬳嬵嬶嬸",7,"孁",6],["8c40","孈",7,"孒孖孞孠孡孧孨孫孭孮孯孲孴孶孷學孹孻孼孾孿宂宆宊宍宎宐宑宒宔宖実宧宨宩宬宭宮宯宱宲宷宺宻宼寀寁寃寈寉寊寋寍寎寏"],["8c80","寑寔",8,"寠寢寣實寧審",4,"寯寱",6,"寽対尀専尃尅將專尋尌對導尐尒尓尗尙尛尞尟尠尡尣尦尨尩尪尫尭尮尯尰尲尳尵尶尷屃屄屆屇屌屍屒屓屔屖屗屘屚屛屜屝屟屢層屧",6,"屰屲",6,"屻屼屽屾岀岃",4,"岉岊岋岎岏岒岓岕岝",4,"岤",4],["8d40","岪岮岯岰岲岴岶岹岺岻岼岾峀峂峃峅",5,"峌",5,"峓",5,"峚",6,"峢峣峧峩峫峬峮峯峱",9,"峼",4],["8d80","崁崄崅崈",5,"崏",4,"崕崗崘崙崚崜崝崟",4,"崥崨崪崫崬崯",4,"崵",7,"崿",7,"嵈嵉嵍",10,"嵙嵚嵜嵞",10,"嵪嵭嵮嵰嵱嵲嵳嵵",12,"嶃",21,"嶚嶛嶜嶞嶟嶠"],["8e40","嶡",21,"嶸",12,"巆",6,"巎",12,"巜巟巠巣巤巪巬巭"],["8e80","巰巵巶巸",4,"巿帀帄帇帉帊帋帍帎帒帓帗帞",7,"帨",4,"帯帰帲",4,"帹帺帾帿幀幁幃幆",5,"幍",6,"幖",4,"幜幝幟幠幣",14,"幵幷幹幾庁庂広庅庈庉庌庍庎庒庘庛庝庡庢庣庤庨",4,"庮",4,"庴庺庻庼庽庿",6],["8f40","廆廇廈廋",5,"廔廕廗廘廙廚廜",11,"廩廫",8,"廵廸廹廻廼廽弅弆弇弉弌弍弎弐弒弔弖弙弚弜弝弞弡弢弣弤"],["8f80","弨弫弬弮弰弲",6,"弻弽弾弿彁",14,"彑彔彙彚彛彜彞彟彠彣彥彧彨彫彮彯彲彴彵彶彸彺彽彾彿徃徆徍徎徏徑従徔徖徚徛徝從徟徠徢",5,"復徫徬徯",5,"徶徸徹徺徻徾",4,"忇忈忊忋忎忓忔忕忚忛応忞忟忢忣忥忦忨忩忬忯忰忲忳忴忶忷忹忺忼怇"],["9040","怈怉怋怌怐怑怓怗怘怚怞怟怢怣怤怬怭怮怰",4,"怶",4,"怽怾恀恄",6,"恌恎恏恑恓恔恖恗恘恛恜恞恟恠恡恥恦恮恱恲恴恵恷恾悀"],["9080","悁悂悅悆悇悈悊悋悎悏悐悑悓悕悗悘悙悜悞悡悢悤悥悧悩悪悮悰悳悵悶悷悹悺悽",7,"惇惈惉惌",4,"惒惓惔惖惗惙惛惞惡",4,"惪惱惲惵惷惸惻",4,"愂愃愄愅愇愊愋愌愐",4,"愖愗愘愙愛愜愝愞愡愢愥愨愩愪愬",18,"慀",6],["9140","慇慉態慍慏慐慒慓慔慖",6,"慞慟慠慡慣慤慥慦慩",6,"慱慲慳慴慶慸",18,"憌憍憏",4,"憕"],["9180","憖",6,"憞",8,"憪憫憭",9,"憸",5,"憿懀懁懃",4,"應懌",4,"懓懕",16,"懧",13,"懶",8,"戀",5,"戇戉戓戔戙戜戝戞戠戣戦戧戨戩戫戭戯戰戱戲戵戶戸",4,"扂扄扅扆扊"],["9240","扏扐払扖扗扙扚扜",6,"扤扥扨扱扲扴扵扷扸扺扻扽抁抂抃抅抆抇抈抋",5,"抔抙抜抝択抣抦抧抩抪抭抮抯抰抲抳抴抶抷抸抺抾拀拁"],["9280","拃拋拏拑拕拝拞拠拡拤拪拫拰拲拵拸拹拺拻挀挃挄挅挆挊挋挌挍挏挐挒挓挔挕挗挘挙挜挦挧挩挬挭挮挰挱挳",5,"挻挼挾挿捀捁捄捇捈捊捑捒捓捔捖",7,"捠捤捥捦捨捪捫捬捯捰捲捳捴捵捸捹捼捽捾捿掁掃掄掅掆掋掍掑掓掔掕掗掙",6,"採掤掦掫掯掱掲掵掶掹掻掽掿揀"],["9340","揁揂揃揅揇揈揊揋揌揑揓揔揕揗",6,"揟揢揤",4,"揫揬揮揯揰揱揳揵揷揹揺揻揼揾搃搄搆",4,"損搎搑搒搕",5,"搝搟搢搣搤"],["9380","搥搧搨搩搫搮",5,"搵",4,"搻搼搾摀摂摃摉摋",6,"摓摕摖摗摙",4,"摟",7,"摨摪摫摬摮",9,"摻",6,"撃撆撈",8,"撓撔撗撘撚撛撜撝撟",4,"撥撦撧撨撪撫撯撱撲撳撴撶撹撻撽撾撿擁擃擄擆",6,"擏擑擓擔擕擖擙據"],["9440","擛擜擝擟擠擡擣擥擧",24,"攁",7,"攊",7,"攓",4,"攙",8],["9480","攢攣攤攦",4,"攬攭攰攱攲攳攷攺攼攽敀",4,"敆敇敊敋敍敎敐敒敓敔敗敘敚敜敟敠敡敤敥敧敨敩敪敭敮敯敱敳敵敶數",14,"斈斉斊斍斎斏斒斔斕斖斘斚斝斞斠斢斣斦斨斪斬斮斱",7,"斺斻斾斿旀旂旇旈旉旊旍旐旑旓旔旕旘",7,"旡旣旤旪旫"],["9540","旲旳旴旵旸旹旻",4,"昁昄昅昇昈昉昋昍昐昑昒昖昗昘昚昛昜昞昡昢昣昤昦昩昪昫昬昮昰昲昳昷",4,"昽昿晀時晄",6,"晍晎晐晑晘"],["9580","晙晛晜晝晞晠晢晣晥晧晩",4,"晱晲晳晵晸晹晻晼晽晿暀暁暃暅暆暈暉暊暋暍暎暏暐暒暓暔暕暘",4,"暞",8,"暩",4,"暯",4,"暵暶暷暸暺暻暼暽暿",25,"曚曞",7,"曧曨曪",5,"曱曵曶書曺曻曽朁朂會"],["9640","朄朅朆朇朌朎朏朑朒朓朖朘朙朚朜朞朠",5,"朧朩朮朰朲朳朶朷朸朹朻朼朾朿杁杄杅杇杊杋杍杒杔杕杗",4,"杝杢杣杤杦杧杫杬杮東杴杶"],["9680","杸杹杺杻杽枀枂枃枅枆枈枊枌枍枎枏枑枒枓枔枖枙枛枟枠枡枤枦枩枬枮枱枲枴枹",7,"柂柅",9,"柕柖柗柛柟柡柣柤柦柧柨柪柫柭柮柲柵",7,"柾栁栂栃栄栆栍栐栒栔栕栘",4,"栞栟栠栢",6,"栫",6,"栴栵栶栺栻栿桇桋桍桏桒桖",5],["9740","桜桝桞桟桪桬",7,"桵桸",8,"梂梄梇",7,"梐梑梒梔梕梖梘",9,"梣梤梥梩梪梫梬梮梱梲梴梶梷梸"],["9780","梹",6,"棁棃",5,"棊棌棎棏棐棑棓棔棖棗棙棛",4,"棡棢棤",9,"棯棲棳棴棶棷棸棻棽棾棿椀椂椃椄椆",4,"椌椏椑椓",11,"椡椢椣椥",7,"椮椯椱椲椳椵椶椷椸椺椻椼椾楀楁楃",16,"楕楖楘楙楛楜楟"],["9840","楡楢楤楥楧楨楩楪楬業楯楰楲",4,"楺楻楽楾楿榁榃榅榊榋榌榎",5,"榖榗榙榚榝",9,"榩榪榬榮榯榰榲榳榵榶榸榹榺榼榽"],["9880","榾榿槀槂",7,"構槍槏槑槒槓槕",5,"槜槝槞槡",11,"槮槯槰槱槳",9,"槾樀",9,"樋",11,"標",5,"樠樢",5,"権樫樬樭樮樰樲樳樴樶",6,"樿",4,"橅橆橈",7,"橑",6,"橚"],["9940","橜",4,"橢橣橤橦",10,"橲",6,"橺橻橽橾橿檁檂檃檅",8,"檏檒",4,"檘",7,"檡",5],["9980","檧檨檪檭",114,"欥欦欨",6],["9a40","欯欰欱欳欴欵欶欸欻欼欽欿歀歁歂歄歅歈歊歋歍",11,"歚",7,"歨歩歫",13,"歺歽歾歿殀殅殈"],["9a80","殌殎殏殐殑殔殕殗殘殙殜",4,"殢",7,"殫",7,"殶殸",6,"毀毃毄毆",4,"毌毎毐毑毘毚毜",4,"毢",7,"毬毭毮毰毱毲毴毶毷毸毺毻毼毾",6,"氈",4,"氎氒気氜氝氞氠氣氥氫氬氭氱氳氶氷氹氺氻氼氾氿汃汄汅汈汋",4,"汑汒汓汖汘"],["9b40","汙汚汢汣汥汦汧汫",4,"汱汳汵汷汸決汻汼汿沀沄沇沊沋沍沎沑沒沕沖沗沘沚沜沝沞沠沢沨沬沯沰沴沵沶沷沺泀況泂泃泆泇泈泋泍泎泏泑泒泘"],["9b80","泙泚泜泝泟泤泦泧泩泬泭泲泴泹泿洀洂洃洅洆洈洉洊洍洏洐洑洓洔洕洖洘洜洝洟",5,"洦洨洩洬洭洯洰洴洶洷洸洺洿浀浂浄浉浌浐浕浖浗浘浛浝浟浡浢浤浥浧浨浫浬浭浰浱浲浳浵浶浹浺浻浽",4,"涃涄涆涇涊涋涍涏涐涒涖",4,"涜涢涥涬涭涰涱涳涴涶涷涹",5,"淁淂淃淈淉淊"],["9c40","淍淎淏淐淒淓淔淕淗淚淛淜淟淢淣淥淧淨淩淪淭淯淰淲淴淵淶淸淺淽",7,"渆渇済渉渋渏渒渓渕渘渙減渜渞渟渢渦渧渨渪測渮渰渱渳渵"],["9c80","渶渷渹渻",7,"湅",7,"湏湐湑湒湕湗湙湚湜湝湞湠",10,"湬湭湯",14,"満溁溂溄溇溈溊",4,"溑",6,"溙溚溛溝溞溠溡溣溤溦溨溩溫溬溭溮溰溳溵溸溹溼溾溿滀滃滄滅滆滈滉滊滌滍滎滐滒滖滘滙滛滜滝滣滧滪",5],["9d40","滰滱滲滳滵滶滷滸滺",7,"漃漄漅漇漈漊",4,"漐漑漒漖",9,"漡漢漣漥漦漧漨漬漮漰漲漴漵漷",6,"漿潀潁潂"],["9d80","潃潄潅潈潉潊潌潎",9,"潙潚潛潝潟潠潡潣潤潥潧",5,"潯潰潱潳潵潶潷潹潻潽",6,"澅澆澇澊澋澏",12,"澝澞澟澠澢",4,"澨",10,"澴澵澷澸澺",5,"濁濃",5,"濊",6,"濓",10,"濟濢濣濤濥"],["9e40","濦",7,"濰",32,"瀒",7,"瀜",6,"瀤",6],["9e80","瀫",9,"瀶瀷瀸瀺",17,"灍灎灐",13,"灟",11,"灮灱灲灳灴灷灹灺灻災炁炂炃炄炆炇炈炋炌炍炏炐炑炓炗炘炚炛炞",12,"炰炲炴炵炶為炾炿烄烅烆烇烉烋",12,"烚"],["9f40","烜烝烞烠烡烢烣烥烪烮烰",6,"烸烺烻烼烾",10,"焋",4,"焑焒焔焗焛",10,"焧",7,"焲焳焴"],["9f80","焵焷",13,"煆煇煈煉煋煍煏",12,"煝煟",4,"煥煩",4,"煯煰煱煴煵煶煷煹煻煼煾",5,"熅",4,"熋熌熍熎熐熑熒熓熕熖熗熚",4,"熡",6,"熩熪熫熭",5,"熴熶熷熸熺",8,"燄",9,"燏",4],["a040","燖",9,"燡燢燣燤燦燨",5,"燯",9,"燺",11,"爇",19],["a080","爛爜爞",9,"爩爫爭爮爯爲爳爴爺爼爾牀",6,"牉牊牋牎牏牐牑牓牔牕牗牘牚牜牞牠牣牤牥牨牪牫牬牭牰牱牳牴牶牷牸牻牼牽犂犃犅",4,"犌犎犐犑犓",11,"犠",11,"犮犱犲犳犵犺",6,"狅狆狇狉狊狋狌狏狑狓狔狕狖狘狚狛"],["a1a1","　、。·ˉˇ¨〃々—～‖…‘’“”〔〕〈",7,"〖〗【】±×÷∶∧∨∑∏∪∩∈∷√⊥∥∠⌒⊙∫∮≡≌≈∽∝≠≮≯≤≥∞∵∴♂♀°′″℃＄¤￠￡‰§№☆★○●◎◇◆□■△▲※→←↑↓〓"],["a2a1","ⅰ",9],["a2b1","⒈",19,"⑴",19,"①",9],["a2e5","㈠",9],["a2f1","Ⅰ",11],["a3a1","！＂＃￥％",88,"￣"],["a4a1","ぁ",82],["a5a1","ァ",85],["a6a1","Α",16,"Σ",6],["a6c1","α",16,"σ",6],["a6e0","︵︶︹︺︿﹀︽︾﹁﹂﹃﹄"],["a6ee","︻︼︷︸︱"],["a6f4","︳︴"],["a7a1","А",5,"ЁЖ",25],["a7d1","а",5,"ёж",25],["a840","ˊˋ˙–―‥‵℅℉↖↗↘↙∕∟∣≒≦≧⊿═",35,"▁",6],["a880","█",7,"▓▔▕▼▽◢◣◤◥☉⊕〒〝〞"],["a8a1","āáǎàēéěèīíǐìōóǒòūúǔùǖǘǚǜüêɑ"],["a8bd","ńň"],["a8c0","ɡ"],["a8c5","ㄅ",36],["a940","〡",8,"㊣㎎㎏㎜㎝㎞㎡㏄㏎㏑㏒㏕︰￢￤"],["a959","℡㈱"],["a95c","‐"],["a960","ー゛゜ヽヾ〆ゝゞ﹉",9,"﹔﹕﹖﹗﹙",8],["a980","﹢",4,"﹨﹩﹪﹫"],["a996","〇"],["a9a4","─",75],["aa40","狜狝狟狢",5,"狪狫狵狶狹狽狾狿猀猂猄",5,"猋猌猍猏猐猑猒猔猘猙猚猟猠猣猤猦猧猨猭猯猰猲猳猵猶猺猻猼猽獀",8],["aa80","獉獊獋獌獎獏獑獓獔獕獖獘",7,"獡",10,"獮獰獱"],["ab40","獲",11,"獿",4,"玅玆玈玊玌玍玏玐玒玓玔玕玗玘玙玚玜玝玞玠玡玣",5,"玪玬玭玱玴玵玶玸玹玼玽玾玿珁珃",4],["ab80","珋珌珎珒",6,"珚珛珜珝珟珡珢珣珤珦珨珪珫珬珮珯珰珱珳",4],["ac40","珸",10,"琄琇琈琋琌琍琎琑",8,"琜",5,"琣琤琧琩琫琭琯琱琲琷",4,"琽琾琿瑀瑂",11],["ac80","瑎",6,"瑖瑘瑝瑠",12,"瑮瑯瑱",4,"瑸瑹瑺"],["ad40","瑻瑼瑽瑿璂璄璅璆璈璉璊璌璍璏璑",10,"璝璟",7,"璪",15,"璻",12],["ad80","瓈",9,"瓓",8,"瓝瓟瓡瓥瓧",6,"瓰瓱瓲"],["ae40","瓳瓵瓸",6,"甀甁甂甃甅",7,"甎甐甒甔甕甖甗甛甝甞甠",4,"甦甧甪甮甴甶甹甼甽甿畁畂畃畄畆畇畉畊畍畐畑畒畓畕畖畗畘"],["ae80","畝",7,"畧畨畩畫",6,"畳畵當畷畺",4,"疀疁疂疄疅疇"],["af40","疈疉疊疌疍疎疐疓疕疘疛疜疞疢疦",4,"疭疶疷疺疻疿痀痁痆痋痌痎痏痐痑痓痗痙痚痜痝痟痠痡痥痩痬痭痮痯痲痳痵痶痷痸痺痻痽痾瘂瘄瘆瘇"],["af80","瘈瘉瘋瘍瘎瘏瘑瘒瘓瘔瘖瘚瘜瘝瘞瘡瘣瘧瘨瘬瘮瘯瘱瘲瘶瘷瘹瘺瘻瘽癁療癄"],["b040","癅",6,"癎",5,"癕癗",4,"癝癟癠癡癢癤",6,"癬癭癮癰",7,"癹発發癿皀皁皃皅皉皊皌皍皏皐皒皔皕皗皘皚皛"],["b080","皜",7,"皥",8,"皯皰皳皵",9,"盀盁盃啊阿埃挨哎唉哀皑癌蔼矮艾碍爱隘鞍氨安俺按暗岸胺案肮昂盎凹敖熬翱袄傲奥懊澳芭捌扒叭吧笆八疤巴拔跋靶把耙坝霸罢爸白柏百摆佰败拜稗斑班搬扳般颁板版扮拌伴瓣半办绊邦帮梆榜膀绑棒磅蚌镑傍谤苞胞包褒剥"],["b140","盄盇盉盋盌盓盕盙盚盜盝盞盠",4,"盦",7,"盰盳盵盶盷盺盻盽盿眀眂眃眅眆眊県眎",10,"眛眜眝眞眡眣眤眥眧眪眫"],["b180","眬眮眰",4,"眹眻眽眾眿睂睄睅睆睈",7,"睒",7,"睜薄雹保堡饱宝抱报暴豹鲍爆杯碑悲卑北辈背贝钡倍狈备惫焙被奔苯本笨崩绷甭泵蹦迸逼鼻比鄙笔彼碧蓖蔽毕毙毖币庇痹闭敝弊必辟壁臂避陛鞭边编贬扁便变卞辨辩辫遍标彪膘表鳖憋别瘪彬斌濒滨宾摈兵冰柄丙秉饼炳"],["b240","睝睞睟睠睤睧睩睪睭",11,"睺睻睼瞁瞂瞃瞆",5,"瞏瞐瞓",11,"瞡瞣瞤瞦瞨瞫瞭瞮瞯瞱瞲瞴瞶",4],["b280","瞼瞾矀",12,"矎",8,"矘矙矚矝",4,"矤病并玻菠播拨钵波博勃搏铂箔伯帛舶脖膊渤泊驳捕卜哺补埠不布步簿部怖擦猜裁材才财睬踩采彩菜蔡餐参蚕残惭惨灿苍舱仓沧藏操糙槽曹草厕策侧册测层蹭插叉茬茶查碴搽察岔差诧拆柴豺搀掺蝉馋谗缠铲产阐颤昌猖"],["b340","矦矨矪矯矰矱矲矴矵矷矹矺矻矼砃",5,"砊砋砎砏砐砓砕砙砛砞砠砡砢砤砨砪砫砮砯砱砲砳砵砶砽砿硁硂硃硄硆硈硉硊硋硍硏硑硓硔硘硙硚"],["b380","硛硜硞",11,"硯",7,"硸硹硺硻硽",6,"场尝常长偿肠厂敞畅唱倡超抄钞朝嘲潮巢吵炒车扯撤掣彻澈郴臣辰尘晨忱沉陈趁衬撑称城橙成呈乘程惩澄诚承逞骋秤吃痴持匙池迟弛驰耻齿侈尺赤翅斥炽充冲虫崇宠抽酬畴踌稠愁筹仇绸瞅丑臭初出橱厨躇锄雏滁除楚"],["b440","碄碅碆碈碊碋碏碐碒碔碕碖碙碝碞碠碢碤碦碨",7,"碵碶碷碸確碻碼碽碿磀磂磃磄磆磇磈磌磍磎磏磑磒磓磖磗磘磚",9],["b480","磤磥磦磧磩磪磫磭",4,"磳磵磶磸磹磻",5,"礂礃礄礆",6,"础储矗搐触处揣川穿椽传船喘串疮窗幢床闯创吹炊捶锤垂春椿醇唇淳纯蠢戳绰疵茨磁雌辞慈瓷词此刺赐次聪葱囱匆从丛凑粗醋簇促蹿篡窜摧崔催脆瘁粹淬翠村存寸磋撮搓措挫错搭达答瘩打大呆歹傣戴带殆代贷袋待逮"],["b540","礍",5,"礔",9,"礟",4,"礥",14,"礵",4,"礽礿祂祃祄祅祇祊",8,"祔祕祘祙祡祣"],["b580","祤祦祩祪祫祬祮祰",6,"祹祻",4,"禂禃禆禇禈禉禋禌禍禎禐禑禒怠耽担丹单郸掸胆旦氮但惮淡诞弹蛋当挡党荡档刀捣蹈倒岛祷导到稻悼道盗德得的蹬灯登等瞪凳邓堤低滴迪敌笛狄涤翟嫡抵底地蒂第帝弟递缔颠掂滇碘点典靛垫电佃甸店惦奠淀殿碉叼雕凋刁掉吊钓调跌爹碟蝶迭谍叠"],["b640","禓",6,"禛",11,"禨",10,"禴",4,"禼禿秂秄秅秇秈秊秌秎秏秐秓秔秖秗秙",5,"秠秡秢秥秨秪"],["b680","秬秮秱",6,"秹秺秼秾秿稁稄稅稇稈稉稊稌稏",4,"稕稖稘稙稛稜丁盯叮钉顶鼎锭定订丢东冬董懂动栋侗恫冻洞兜抖斗陡豆逗痘都督毒犊独读堵睹赌杜镀肚度渡妒端短锻段断缎堆兑队对墩吨蹲敦顿囤钝盾遁掇哆多夺垛躲朵跺舵剁惰堕蛾峨鹅俄额讹娥恶厄扼遏鄂饿恩而儿耳尔饵洱二"],["b740","稝稟稡稢稤",14,"稴稵稶稸稺稾穀",5,"穇",9,"穒",4,"穘",16],["b780","穩",6,"穱穲穳穵穻穼穽穾窂窅窇窉窊窋窌窎窏窐窓窔窙窚窛窞窡窢贰发罚筏伐乏阀法珐藩帆番翻樊矾钒繁凡烦反返范贩犯饭泛坊芳方肪房防妨仿访纺放菲非啡飞肥匪诽吠肺废沸费芬酚吩氛分纷坟焚汾粉奋份忿愤粪丰封枫蜂峰锋风疯烽逢冯缝讽奉凤佛否夫敷肤孵扶拂辐幅氟符伏俘服"],["b840","窣窤窧窩窪窫窮",4,"窴",10,"竀",10,"竌",9,"竗竘竚竛竜竝竡竢竤竧",5,"竮竰竱竲竳"],["b880","竴",4,"竻竼竾笀笁笂笅笇笉笌笍笎笐笒笓笖笗笘笚笜笝笟笡笢笣笧笩笭浮涪福袱弗甫抚辅俯釜斧脯腑府腐赴副覆赋复傅付阜父腹负富讣附妇缚咐噶嘎该改概钙盖溉干甘杆柑竿肝赶感秆敢赣冈刚钢缸肛纲岗港杠篙皋高膏羔糕搞镐稿告哥歌搁戈鸽胳疙割革葛格蛤阁隔铬个各给根跟耕更庚羹"],["b940","笯笰笲笴笵笶笷笹笻笽笿",5,"筆筈筊筍筎筓筕筗筙筜筞筟筡筣",10,"筯筰筳筴筶筸筺筼筽筿箁箂箃箄箆",6,"箎箏"],["b980","箑箒箓箖箘箙箚箛箞箟箠箣箤箥箮箯箰箲箳箵箶箷箹",7,"篂篃範埂耿梗工攻功恭龚供躬公宫弓巩汞拱贡共钩勾沟苟狗垢构购够辜菇咕箍估沽孤姑鼓古蛊骨谷股故顾固雇刮瓜剐寡挂褂乖拐怪棺关官冠观管馆罐惯灌贯光广逛瑰规圭硅归龟闺轨鬼诡癸桂柜跪贵刽辊滚棍锅郭国果裹过哈"],["ba40","篅篈築篊篋篍篎篏篐篒篔",4,"篛篜篞篟篠篢篣篤篧篨篩篫篬篭篯篰篲",4,"篸篹篺篻篽篿",7,"簈簉簊簍簎簐",5,"簗簘簙"],["ba80","簚",4,"簠",5,"簨簩簫",12,"簹",5,"籂骸孩海氦亥害骇酣憨邯韩含涵寒函喊罕翰撼捍旱憾悍焊汗汉夯杭航壕嚎豪毫郝好耗号浩呵喝荷菏核禾和何合盒貉阂河涸赫褐鹤贺嘿黑痕很狠恨哼亨横衡恒轰哄烘虹鸿洪宏弘红喉侯猴吼厚候后呼乎忽瑚壶葫胡蝴狐糊湖"],["bb40","籃",9,"籎",36,"籵",5,"籾",9],["bb80","粈粊",6,"粓粔粖粙粚粛粠粡粣粦粧粨粩粫粬粭粯粰粴",4,"粺粻弧虎唬护互沪户花哗华猾滑画划化话槐徊怀淮坏欢环桓还缓换患唤痪豢焕涣宦幻荒慌黄磺蝗簧皇凰惶煌晃幌恍谎灰挥辉徽恢蛔回毁悔慧卉惠晦贿秽会烩汇讳诲绘荤昏婚魂浑混豁活伙火获或惑霍货祸击圾基机畸稽积箕"],["bc40","粿糀糂糃糄糆糉糋糎",6,"糘糚糛糝糞糡",6,"糩",5,"糰",7,"糹糺糼",13,"紋",5],["bc80","紑",14,"紡紣紤紥紦紨紩紪紬紭紮細",6,"肌饥迹激讥鸡姬绩缉吉极棘辑籍集及急疾汲即嫉级挤几脊己蓟技冀季伎祭剂悸济寄寂计记既忌际妓继纪嘉枷夹佳家加荚颊贾甲钾假稼价架驾嫁歼监坚尖笺间煎兼肩艰奸缄茧检柬碱硷拣捡简俭剪减荐槛鉴践贱见键箭件"],["bd40","紷",54,"絯",7],["bd80","絸",32,"健舰剑饯渐溅涧建僵姜将浆江疆蒋桨奖讲匠酱降蕉椒礁焦胶交郊浇骄娇嚼搅铰矫侥脚狡角饺缴绞剿教酵轿较叫窖揭接皆秸街阶截劫节桔杰捷睫竭洁结解姐戒藉芥界借介疥诫届巾筋斤金今津襟紧锦仅谨进靳晋禁近烬浸"],["be40","継",12,"綧",6,"綯",42],["be80","線",32,"尽劲荆兢茎睛晶鲸京惊精粳经井警景颈静境敬镜径痉靖竟竞净炯窘揪究纠玖韭久灸九酒厩救旧臼舅咎就疚鞠拘狙疽居驹菊局咀矩举沮聚拒据巨具距踞锯俱句惧炬剧捐鹃娟倦眷卷绢撅攫抉掘倔爵觉决诀绝均菌钧军君峻"],["bf40","緻",62],["bf80","縺縼",4,"繂",4,"繈",21,"俊竣浚郡骏喀咖卡咯开揩楷凯慨刊堪勘坎砍看康慷糠扛抗亢炕考拷烤靠坷苛柯棵磕颗科壳咳可渴克刻客课肯啃垦恳坑吭空恐孔控抠口扣寇枯哭窟苦酷库裤夸垮挎跨胯块筷侩快宽款匡筐狂框矿眶旷况亏盔岿窥葵奎魁傀"],["c040","繞",35,"纃",23,"纜纝纞"],["c080","纮纴纻纼绖绤绬绹缊缐缞缷缹缻",6,"罃罆",9,"罒罓馈愧溃坤昆捆困括扩廓阔垃拉喇蜡腊辣啦莱来赖蓝婪栏拦篮阑兰澜谰揽览懒缆烂滥琅榔狼廊郎朗浪捞劳牢老佬姥酪烙涝勒乐雷镭蕾磊累儡垒擂肋类泪棱楞冷厘梨犁黎篱狸离漓理李里鲤礼莉荔吏栗丽厉励砾历利傈例俐"],["c140","罖罙罛罜罝罞罠罣",4,"罫罬罭罯罰罳罵罶罷罸罺罻罼罽罿羀羂",7,"羋羍羏",4,"羕",4,"羛羜羠羢羣羥羦羨",6,"羱"],["c180","羳",4,"羺羻羾翀翂翃翄翆翇翈翉翋翍翏",4,"翖翗翙",5,"翢翣痢立粒沥隶力璃哩俩联莲连镰廉怜涟帘敛脸链恋炼练粮凉梁粱良两辆量晾亮谅撩聊僚疗燎寥辽潦了撂镣廖料列裂烈劣猎琳林磷霖临邻鳞淋凛赁吝拎玲菱零龄铃伶羚凌灵陵岭领另令溜琉榴硫馏留刘瘤流柳六龙聋咙笼窿"],["c240","翤翧翨翪翫翬翭翯翲翴",6,"翽翾翿耂耇耈耉耊耎耏耑耓耚耛耝耞耟耡耣耤耫",5,"耲耴耹耺耼耾聀聁聄聅聇聈聉聎聏聐聑聓聕聖聗"],["c280","聙聛",13,"聫",5,"聲",11,"隆垄拢陇楼娄搂篓漏陋芦卢颅庐炉掳卤虏鲁麓碌露路赂鹿潞禄录陆戮驴吕铝侣旅履屡缕虑氯律率滤绿峦挛孪滦卵乱掠略抡轮伦仑沦纶论萝螺罗逻锣箩骡裸落洛骆络妈麻玛码蚂马骂嘛吗埋买麦卖迈脉瞒馒蛮满蔓曼慢漫"],["c340","聾肁肂肅肈肊肍",5,"肔肕肗肙肞肣肦肧肨肬肰肳肵肶肸肹肻胅胇",4,"胏",6,"胘胟胠胢胣胦胮胵胷胹胻胾胿脀脁脃脄脅脇脈脋"],["c380","脌脕脗脙脛脜脝脟",12,"脭脮脰脳脴脵脷脹",4,"脿谩芒茫盲氓忙莽猫茅锚毛矛铆卯茂冒帽貌贸么玫枚梅酶霉煤没眉媒镁每美昧寐妹媚门闷们萌蒙檬盟锰猛梦孟眯醚靡糜迷谜弥米秘觅泌蜜密幂棉眠绵冕免勉娩缅面苗描瞄藐秒渺庙妙蔑灭民抿皿敏悯闽明螟鸣铭名命谬摸"],["c440","腀",5,"腇腉腍腎腏腒腖腗腘腛",4,"腡腢腣腤腦腨腪腫腬腯腲腳腵腶腷腸膁膃",4,"膉膋膌膍膎膐膒",5,"膙膚膞",4,"膤膥"],["c480","膧膩膫",7,"膴",5,"膼膽膾膿臄臅臇臈臉臋臍",6,"摹蘑模膜磨摩魔抹末莫墨默沫漠寞陌谋牟某拇牡亩姆母墓暮幕募慕木目睦牧穆拿哪呐钠那娜纳氖乃奶耐奈南男难囊挠脑恼闹淖呢馁内嫩能妮霓倪泥尼拟你匿腻逆溺蔫拈年碾撵捻念娘酿鸟尿捏聂孽啮镊镍涅您柠狞凝宁"],["c540","臔",14,"臤臥臦臨臩臫臮",4,"臵",5,"臽臿舃與",4,"舎舏舑舓舕",5,"舝舠舤舥舦舧舩舮舲舺舼舽舿"],["c580","艀艁艂艃艅艆艈艊艌艍艎艐",7,"艙艛艜艝艞艠",7,"艩拧泞牛扭钮纽脓浓农弄奴努怒女暖虐疟挪懦糯诺哦欧鸥殴藕呕偶沤啪趴爬帕怕琶拍排牌徘湃派攀潘盘磐盼畔判叛乓庞旁耪胖抛咆刨炮袍跑泡呸胚培裴赔陪配佩沛喷盆砰抨烹澎彭蓬棚硼篷膨朋鹏捧碰坯砒霹批披劈琵毗"],["c640","艪艫艬艭艱艵艶艷艸艻艼芀芁芃芅芆芇芉芌芐芓芔芕芖芚芛芞芠芢芣芧芲芵芶芺芻芼芿苀苂苃苅苆苉苐苖苙苚苝苢苧苨苩苪苬苭苮苰苲苳苵苶苸"],["c680","苺苼",4,"茊茋茍茐茒茓茖茘茙茝",9,"茩茪茮茰茲茷茻茽啤脾疲皮匹痞僻屁譬篇偏片骗飘漂瓢票撇瞥拼频贫品聘乒坪苹萍平凭瓶评屏坡泼颇婆破魄迫粕剖扑铺仆莆葡菩蒲埔朴圃普浦谱曝瀑期欺栖戚妻七凄漆柒沏其棋奇歧畦崎脐齐旗祈祁骑起岂乞企启契砌器气迄弃汽泣讫掐"],["c740","茾茿荁荂荄荅荈荊",4,"荓荕",4,"荝荢荰",6,"荹荺荾",6,"莇莈莊莋莌莍莏莐莑莔莕莖莗莙莚莝莟莡",6,"莬莭莮"],["c780","莯莵莻莾莿菂菃菄菆菈菉菋菍菎菐菑菒菓菕菗菙菚菛菞菢菣菤菦菧菨菫菬菭恰洽牵扦钎铅千迁签仟谦乾黔钱钳前潜遣浅谴堑嵌欠歉枪呛腔羌墙蔷强抢橇锹敲悄桥瞧乔侨巧鞘撬翘峭俏窍切茄且怯窃钦侵亲秦琴勤芹擒禽寝沁青轻氢倾卿清擎晴氰情顷请庆琼穷秋丘邱球求囚酋泅趋区蛆曲躯屈驱渠"],["c840","菮華菳",4,"菺菻菼菾菿萀萂萅萇萈萉萊萐萒",5,"萙萚萛萞",5,"萩",7,"萲",5,"萹萺萻萾",7,"葇葈葉"],["c880","葊",6,"葒",4,"葘葝葞葟葠葢葤",4,"葪葮葯葰葲葴葷葹葻葼取娶龋趣去圈颧权醛泉全痊拳犬券劝缺炔瘸却鹊榷确雀裙群然燃冉染瓤壤攘嚷让饶扰绕惹热壬仁人忍韧任认刃妊纫扔仍日戎茸蓉荣融熔溶容绒冗揉柔肉茹蠕儒孺如辱乳汝入褥软阮蕊瑞锐闰润若弱撒洒萨腮鳃塞赛三叁"],["c940","葽",4,"蒃蒄蒅蒆蒊蒍蒏",7,"蒘蒚蒛蒝蒞蒟蒠蒢",12,"蒰蒱蒳蒵蒶蒷蒻蒼蒾蓀蓂蓃蓅蓆蓇蓈蓋蓌蓎蓏蓒蓔蓕蓗"],["c980","蓘",4,"蓞蓡蓢蓤蓧",4,"蓭蓮蓯蓱",10,"蓽蓾蔀蔁蔂伞散桑嗓丧搔骚扫嫂瑟色涩森僧莎砂杀刹沙纱傻啥煞筛晒珊苫杉山删煽衫闪陕擅赡膳善汕扇缮墒伤商赏晌上尚裳梢捎稍烧芍勺韶少哨邵绍奢赊蛇舌舍赦摄射慑涉社设砷申呻伸身深娠绅神沈审婶甚肾慎渗声生甥牲升绳"],["ca40","蔃",8,"蔍蔎蔏蔐蔒蔔蔕蔖蔘蔙蔛蔜蔝蔞蔠蔢",8,"蔭",9,"蔾",4,"蕄蕅蕆蕇蕋",10],["ca80","蕗蕘蕚蕛蕜蕝蕟",4,"蕥蕦蕧蕩",8,"蕳蕵蕶蕷蕸蕼蕽蕿薀薁省盛剩胜圣师失狮施湿诗尸虱十石拾时什食蚀实识史矢使屎驶始式示士世柿事拭誓逝势是嗜噬适仕侍释饰氏市恃室视试收手首守寿授售受瘦兽蔬枢梳殊抒输叔舒淑疏书赎孰熟薯暑曙署蜀黍鼠属术述树束戍竖墅庶数漱"],["cb40","薂薃薆薈",6,"薐",10,"薝",6,"薥薦薧薩薫薬薭薱",5,"薸薺",6,"藂",6,"藊",4,"藑藒"],["cb80","藔藖",5,"藝",6,"藥藦藧藨藪",14,"恕刷耍摔衰甩帅栓拴霜双爽谁水睡税吮瞬顺舜说硕朔烁斯撕嘶思私司丝死肆寺嗣四伺似饲巳松耸怂颂送宋讼诵搜艘擞嗽苏酥俗素速粟僳塑溯宿诉肃酸蒜算虽隋随绥髓碎岁穗遂隧祟孙损笋蓑梭唆缩琐索锁所塌他它她塔"],["cc40","藹藺藼藽藾蘀",4,"蘆",10,"蘒蘓蘔蘕蘗",15,"蘨蘪",13,"蘹蘺蘻蘽蘾蘿虀"],["cc80","虁",11,"虒虓處",4,"虛虜虝號虠虡虣",7,"獭挞蹋踏胎苔抬台泰酞太态汰坍摊贪瘫滩坛檀痰潭谭谈坦毯袒碳探叹炭汤塘搪堂棠膛唐糖倘躺淌趟烫掏涛滔绦萄桃逃淘陶讨套特藤腾疼誊梯剔踢锑提题蹄啼体替嚏惕涕剃屉天添填田甜恬舔腆挑条迢眺跳贴铁帖厅听烃"],["cd40","虭虯虰虲",6,"蚃",6,"蚎",4,"蚔蚖",5,"蚞",4,"蚥蚦蚫蚭蚮蚲蚳蚷蚸蚹蚻",4,"蛁蛂蛃蛅蛈蛌蛍蛒蛓蛕蛖蛗蛚蛜"],["cd80","蛝蛠蛡蛢蛣蛥蛦蛧蛨蛪蛫蛬蛯蛵蛶蛷蛺蛻蛼蛽蛿蜁蜄蜅蜆蜋蜌蜎蜏蜐蜑蜔蜖汀廷停亭庭挺艇通桐酮瞳同铜彤童桶捅筒统痛偷投头透凸秃突图徒途涂屠土吐兔湍团推颓腿蜕褪退吞屯臀拖托脱鸵陀驮驼椭妥拓唾挖哇蛙洼娃瓦袜歪外豌弯湾玩顽丸烷完碗挽晚皖惋宛婉万腕汪王亡枉网往旺望忘妄威"],["ce40","蜙蜛蜝蜟蜠蜤蜦蜧蜨蜪蜫蜬蜭蜯蜰蜲蜳蜵蜶蜸蜹蜺蜼蜽蝀",6,"蝊蝋蝍蝏蝐蝑蝒蝔蝕蝖蝘蝚",5,"蝡蝢蝦",7,"蝯蝱蝲蝳蝵"],["ce80","蝷蝸蝹蝺蝿螀螁螄螆螇螉螊螌螎",4,"螔螕螖螘",6,"螠",4,"巍微危韦违桅围唯惟为潍维苇萎委伟伪尾纬未蔚味畏胃喂魏位渭谓尉慰卫瘟温蚊文闻纹吻稳紊问嗡翁瓮挝蜗涡窝我斡卧握沃巫呜钨乌污诬屋无芜梧吾吴毋武五捂午舞伍侮坞戊雾晤物勿务悟误昔熙析西硒矽晰嘻吸锡牺"],["cf40","螥螦螧螩螪螮螰螱螲螴螶螷螸螹螻螼螾螿蟁",4,"蟇蟈蟉蟌",4,"蟔",6,"蟜蟝蟞蟟蟡蟢蟣蟤蟦蟧蟨蟩蟫蟬蟭蟯",9],["cf80","蟺蟻蟼蟽蟿蠀蠁蠂蠄",5,"蠋",7,"蠔蠗蠘蠙蠚蠜",4,"蠣稀息希悉膝夕惜熄烯溪汐犀檄袭席习媳喜铣洗系隙戏细瞎虾匣霞辖暇峡侠狭下厦夏吓掀锨先仙鲜纤咸贤衔舷闲涎弦嫌显险现献县腺馅羡宪陷限线相厢镶香箱襄湘乡翔祥详想响享项巷橡像向象萧硝霄削哮嚣销消宵淆晓"],["d040","蠤",13,"蠳",5,"蠺蠻蠽蠾蠿衁衂衃衆",5,"衎",5,"衕衖衘衚",6,"衦衧衪衭衯衱衳衴衵衶衸衹衺"],["d080","衻衼袀袃袆袇袉袊袌袎袏袐袑袓袔袕袗",4,"袝",4,"袣袥",5,"小孝校肖啸笑效楔些歇蝎鞋协挟携邪斜胁谐写械卸蟹懈泄泻谢屑薪芯锌欣辛新忻心信衅星腥猩惺兴刑型形邢行醒幸杏性姓兄凶胸匈汹雄熊休修羞朽嗅锈秀袖绣墟戌需虚嘘须徐许蓄酗叙旭序畜恤絮婿绪续轩喧宣悬旋玄"],["d140","袬袮袯袰袲",4,"袸袹袺袻袽袾袿裀裃裄裇裈裊裋裌裍裏裐裑裓裖裗裚",4,"裠裡裦裧裩",6,"裲裵裶裷裺裻製裿褀褁褃",5],["d180","褉褋",4,"褑褔",4,"褜",4,"褢褣褤褦褧褨褩褬褭褮褯褱褲褳褵褷选癣眩绚靴薛学穴雪血勋熏循旬询寻驯巡殉汛训讯逊迅压押鸦鸭呀丫芽牙蚜崖衙涯雅哑亚讶焉咽阉烟淹盐严研蜒岩延言颜阎炎沿奄掩眼衍演艳堰燕厌砚雁唁彦焰宴谚验殃央鸯秧杨扬佯疡羊洋阳氧仰痒养样漾邀腰妖瑶"],["d240","褸",8,"襂襃襅",24,"襠",5,"襧",19,"襼"],["d280","襽襾覀覂覄覅覇",26,"摇尧遥窑谣姚咬舀药要耀椰噎耶爷野冶也页掖业叶曳腋夜液一壹医揖铱依伊衣颐夷遗移仪胰疑沂宜姨彝椅蚁倚已乙矣以艺抑易邑屹亿役臆逸肄疫亦裔意毅忆义益溢诣议谊译异翼翌绎茵荫因殷音阴姻吟银淫寅饮尹引隐"],["d340","覢",30,"觃觍觓觔觕觗觘觙觛觝觟觠觡觢觤觧觨觩觪觬觭觮觰觱觲觴",6],["d380","觻",4,"訁",5,"計",21,"印英樱婴鹰应缨莹萤营荧蝇迎赢盈影颖硬映哟拥佣臃痈庸雍踊蛹咏泳涌永恿勇用幽优悠忧尤由邮铀犹油游酉有友右佑釉诱又幼迂淤于盂榆虞愚舆余俞逾鱼愉渝渔隅予娱雨与屿禹宇语羽玉域芋郁吁遇喻峪御愈欲狱育誉"],["d440","訞",31,"訿",8,"詉",21],["d480","詟",25,"詺",6,"浴寓裕预豫驭鸳渊冤元垣袁原援辕园员圆猿源缘远苑愿怨院曰约越跃钥岳粤月悦阅耘云郧匀陨允运蕴酝晕韵孕匝砸杂栽哉灾宰载再在咱攒暂赞赃脏葬遭糟凿藻枣早澡蚤躁噪造皂灶燥责择则泽贼怎增憎曾赠扎喳渣札轧"],["d540","誁",7,"誋",7,"誔",46],["d580","諃",32,"铡闸眨栅榨咋乍炸诈摘斋宅窄债寨瞻毡詹粘沾盏斩辗崭展蘸栈占战站湛绽樟章彰漳张掌涨杖丈帐账仗胀瘴障招昭找沼赵照罩兆肇召遮折哲蛰辙者锗蔗这浙珍斟真甄砧臻贞针侦枕疹诊震振镇阵蒸挣睁征狰争怔整拯正政"],["d640","諤",34,"謈",27],["d680","謤謥謧",30,"帧症郑证芝枝支吱蜘知肢脂汁之织职直植殖执值侄址指止趾只旨纸志挚掷至致置帜峙制智秩稚质炙痔滞治窒中盅忠钟衷终种肿重仲众舟周州洲诌粥轴肘帚咒皱宙昼骤珠株蛛朱猪诸诛逐竹烛煮拄瞩嘱主著柱助蛀贮铸筑"],["d740","譆",31,"譧",4,"譭",25],["d780","讇",24,"讬讱讻诇诐诪谉谞住注祝驻抓爪拽专砖转撰赚篆桩庄装妆撞壮状椎锥追赘坠缀谆准捉拙卓桌琢茁酌啄着灼浊兹咨资姿滋淄孜紫仔籽滓子自渍字鬃棕踪宗综总纵邹走奏揍租足卒族祖诅阻组钻纂嘴醉最罪尊遵昨左佐柞做作坐座"],["d840","谸",8,"豂豃豄豅豈豊豋豍",7,"豖豗豘豙豛",5,"豣",6,"豬",6,"豴豵豶豷豻",6,"貃貄貆貇"],["d880","貈貋貍",6,"貕貖貗貙",20,"亍丌兀丐廿卅丕亘丞鬲孬噩丨禺丿匕乇夭爻卮氐囟胤馗毓睾鼗丶亟鼐乜乩亓芈孛啬嘏仄厍厝厣厥厮靥赝匚叵匦匮匾赜卦卣刂刈刎刭刳刿剀剌剞剡剜蒯剽劂劁劐劓冂罔亻仃仉仂仨仡仫仞伛仳伢佤仵伥伧伉伫佞佧攸佚佝"],["d940","貮",62],["d980","賭",32,"佟佗伲伽佶佴侑侉侃侏佾佻侪佼侬侔俦俨俪俅俚俣俜俑俟俸倩偌俳倬倏倮倭俾倜倌倥倨偾偃偕偈偎偬偻傥傧傩傺僖儆僭僬僦僮儇儋仝氽佘佥俎龠汆籴兮巽黉馘冁夔勹匍訇匐凫夙兕亠兖亳衮袤亵脔裒禀嬴蠃羸冫冱冽冼"],["da40","贎",14,"贠赑赒赗赟赥赨赩赪赬赮赯赱赲赸",8,"趂趃趆趇趈趉趌",4,"趒趓趕",9,"趠趡"],["da80","趢趤",12,"趲趶趷趹趻趽跀跁跂跅跇跈跉跊跍跐跒跓跔凇冖冢冥讠讦讧讪讴讵讷诂诃诋诏诎诒诓诔诖诘诙诜诟诠诤诨诩诮诰诳诶诹诼诿谀谂谄谇谌谏谑谒谔谕谖谙谛谘谝谟谠谡谥谧谪谫谮谯谲谳谵谶卩卺阝阢阡阱阪阽阼陂陉陔陟陧陬陲陴隈隍隗隰邗邛邝邙邬邡邴邳邶邺"],["db40","跕跘跙跜跠跡跢跥跦跧跩跭跮跰跱跲跴跶跼跾",6,"踆踇踈踋踍踎踐踑踒踓踕",7,"踠踡踤",4,"踫踭踰踲踳踴踶踷踸踻踼踾"],["db80","踿蹃蹅蹆蹌",4,"蹓",5,"蹚",11,"蹧蹨蹪蹫蹮蹱邸邰郏郅邾郐郄郇郓郦郢郜郗郛郫郯郾鄄鄢鄞鄣鄱鄯鄹酃酆刍奂劢劬劭劾哿勐勖勰叟燮矍廴凵凼鬯厶弁畚巯坌垩垡塾墼壅壑圩圬圪圳圹圮圯坜圻坂坩垅坫垆坼坻坨坭坶坳垭垤垌垲埏垧垴垓垠埕埘埚埙埒垸埴埯埸埤埝"],["dc40","蹳蹵蹷",4,"蹽蹾躀躂躃躄躆躈",6,"躑躒躓躕",6,"躝躟",11,"躭躮躰躱躳",6,"躻",7],["dc80","軃",10,"軏",21,"堋堍埽埭堀堞堙塄堠塥塬墁墉墚墀馨鼙懿艹艽艿芏芊芨芄芎芑芗芙芫芸芾芰苈苊苣芘芷芮苋苌苁芩芴芡芪芟苄苎芤苡茉苷苤茏茇苜苴苒苘茌苻苓茑茚茆茔茕苠苕茜荑荛荜茈莒茼茴茱莛荞茯荏荇荃荟荀茗荠茭茺茳荦荥"],["dd40","軥",62],["dd80","輤",32,"荨茛荩荬荪荭荮莰荸莳莴莠莪莓莜莅荼莶莩荽莸荻莘莞莨莺莼菁萁菥菘堇萘萋菝菽菖萜萸萑萆菔菟萏萃菸菹菪菅菀萦菰菡葜葑葚葙葳蒇蒈葺蒉葸萼葆葩葶蒌蒎萱葭蓁蓍蓐蓦蒽蓓蓊蒿蒺蓠蒡蒹蒴蒗蓥蓣蔌甍蔸蓰蔹蔟蔺"],["de40","轅",32,"轪辀辌辒辝辠辡辢辤辥辦辧辪辬辭辮辯農辳辴辵辷辸辺辻込辿迀迃迆"],["de80","迉",4,"迏迒迖迗迚迠迡迣迧迬迯迱迲迴迵迶迺迻迼迾迿逇逈逌逎逓逕逘蕖蔻蓿蓼蕙蕈蕨蕤蕞蕺瞢蕃蕲蕻薤薨薇薏蕹薮薜薅薹薷薰藓藁藜藿蘧蘅蘩蘖蘼廾弈夼奁耷奕奚奘匏尢尥尬尴扌扪抟抻拊拚拗拮挢拶挹捋捃掭揶捱捺掎掴捭掬掊捩掮掼揲揸揠揿揄揞揎摒揆掾摅摁搋搛搠搌搦搡摞撄摭撖"],["df40","這逜連逤逥逧",5,"逰",4,"逷逹逺逽逿遀遃遅遆遈",4,"過達違遖遙遚遜",5,"遤遦遧適遪遫遬遯",4,"遶",6,"遾邁"],["df80","還邅邆邇邉邊邌",4,"邒邔邖邘邚邜邞邟邠邤邥邧邨邩邫邭邲邷邼邽邿郀摺撷撸撙撺擀擐擗擤擢攉攥攮弋忒甙弑卟叱叽叩叨叻吒吖吆呋呒呓呔呖呃吡呗呙吣吲咂咔呷呱呤咚咛咄呶呦咝哐咭哂咴哒咧咦哓哔呲咣哕咻咿哌哙哚哜咩咪咤哝哏哞唛哧唠哽唔哳唢唣唏唑唧唪啧喏喵啉啭啁啕唿啐唼"],["e040","郂郃郆郈郉郋郌郍郒郔郕郖郘郙郚郞郟郠郣郤郥郩郪郬郮郰郱郲郳郵郶郷郹郺郻郼郿鄀鄁鄃鄅",19,"鄚鄛鄜"],["e080","鄝鄟鄠鄡鄤",10,"鄰鄲",6,"鄺",8,"酄唷啖啵啶啷唳唰啜喋嗒喃喱喹喈喁喟啾嗖喑啻嗟喽喾喔喙嗪嗷嗉嘟嗑嗫嗬嗔嗦嗝嗄嗯嗥嗲嗳嗌嗍嗨嗵嗤辔嘞嘈嘌嘁嘤嘣嗾嘀嘧嘭噘嘹噗嘬噍噢噙噜噌噔嚆噤噱噫噻噼嚅嚓嚯囔囗囝囡囵囫囹囿圄圊圉圜帏帙帔帑帱帻帼"],["e140","酅酇酈酑酓酔酕酖酘酙酛酜酟酠酦酧酨酫酭酳酺酻酼醀",4,"醆醈醊醎醏醓",6,"醜",5,"醤",5,"醫醬醰醱醲醳醶醷醸醹醻"],["e180","醼",10,"釈釋釐釒",9,"針",8,"帷幄幔幛幞幡岌屺岍岐岖岈岘岙岑岚岜岵岢岽岬岫岱岣峁岷峄峒峤峋峥崂崃崧崦崮崤崞崆崛嵘崾崴崽嵬嵛嵯嵝嵫嵋嵊嵩嵴嶂嶙嶝豳嶷巅彳彷徂徇徉後徕徙徜徨徭徵徼衢彡犭犰犴犷犸狃狁狎狍狒狨狯狩狲狴狷猁狳猃狺"],["e240","釦",62],["e280","鈥",32,"狻猗猓猡猊猞猝猕猢猹猥猬猸猱獐獍獗獠獬獯獾舛夥飧夤夂饣饧",5,"饴饷饽馀馄馇馊馍馐馑馓馔馕庀庑庋庖庥庠庹庵庾庳赓廒廑廛廨廪膺忄忉忖忏怃忮怄忡忤忾怅怆忪忭忸怙怵怦怛怏怍怩怫怊怿怡恸恹恻恺恂"],["e340","鉆",45,"鉵",16],["e380","銆",7,"銏",24,"恪恽悖悚悭悝悃悒悌悛惬悻悱惝惘惆惚悴愠愦愕愣惴愀愎愫慊慵憬憔憧憷懔懵忝隳闩闫闱闳闵闶闼闾阃阄阆阈阊阋阌阍阏阒阕阖阗阙阚丬爿戕氵汔汜汊沣沅沐沔沌汨汩汴汶沆沩泐泔沭泷泸泱泗沲泠泖泺泫泮沱泓泯泾"],["e440","銨",5,"銯",24,"鋉",31],["e480","鋩",32,"洹洧洌浃浈洇洄洙洎洫浍洮洵洚浏浒浔洳涑浯涞涠浞涓涔浜浠浼浣渚淇淅淞渎涿淠渑淦淝淙渖涫渌涮渫湮湎湫溲湟溆湓湔渲渥湄滟溱溘滠漭滢溥溧溽溻溷滗溴滏溏滂溟潢潆潇漤漕滹漯漶潋潴漪漉漩澉澍澌潸潲潼潺濑"],["e540","錊",51,"錿",10],["e580","鍊",31,"鍫濉澧澹澶濂濡濮濞濠濯瀚瀣瀛瀹瀵灏灞宀宄宕宓宥宸甯骞搴寤寮褰寰蹇謇辶迓迕迥迮迤迩迦迳迨逅逄逋逦逑逍逖逡逵逶逭逯遄遑遒遐遨遘遢遛暹遴遽邂邈邃邋彐彗彖彘尻咫屐屙孱屣屦羼弪弩弭艴弼鬻屮妁妃妍妩妪妣"],["e640","鍬",34,"鎐",27],["e680","鎬",29,"鏋鏌鏍妗姊妫妞妤姒妲妯姗妾娅娆姝娈姣姘姹娌娉娲娴娑娣娓婀婧婊婕娼婢婵胬媪媛婷婺媾嫫媲嫒嫔媸嫠嫣嫱嫖嫦嫘嫜嬉嬗嬖嬲嬷孀尕尜孚孥孳孑孓孢驵驷驸驺驿驽骀骁骅骈骊骐骒骓骖骘骛骜骝骟骠骢骣骥骧纟纡纣纥纨纩"],["e740","鏎",7,"鏗",54],["e780","鐎",32,"纭纰纾绀绁绂绉绋绌绐绔绗绛绠绡绨绫绮绯绱绲缍绶绺绻绾缁缂缃缇缈缋缌缏缑缒缗缙缜缛缟缡",6,"缪缫缬缭缯",4,"缵幺畿巛甾邕玎玑玮玢玟珏珂珑玷玳珀珉珈珥珙顼琊珩珧珞玺珲琏琪瑛琦琥琨琰琮琬"],["e840","鐯",14,"鐿",43,"鑬鑭鑮鑯"],["e880","鑰",20,"钑钖钘铇铏铓铔铚铦铻锜锠琛琚瑁瑜瑗瑕瑙瑷瑭瑾璜璎璀璁璇璋璞璨璩璐璧瓒璺韪韫韬杌杓杞杈杩枥枇杪杳枘枧杵枨枞枭枋杷杼柰栉柘栊柩枰栌柙枵柚枳柝栀柃枸柢栎柁柽栲栳桠桡桎桢桄桤梃栝桕桦桁桧桀栾桊桉栩梵梏桴桷梓桫棂楮棼椟椠棹"],["e940","锧锳锽镃镈镋镕镚镠镮镴镵長",7,"門",42],["e980","閫",32,"椤棰椋椁楗棣椐楱椹楠楂楝榄楫榀榘楸椴槌榇榈槎榉楦楣楹榛榧榻榫榭槔榱槁槊槟榕槠榍槿樯槭樗樘橥槲橄樾檠橐橛樵檎橹樽樨橘橼檑檐檩檗檫猷獒殁殂殇殄殒殓殍殚殛殡殪轫轭轱轲轳轵轶轸轷轹轺轼轾辁辂辄辇辋"],["ea40","闌",27,"闬闿阇阓阘阛阞阠阣",6,"阫阬阭阯阰阷阸阹阺阾陁陃陊陎陏陑陒陓陖陗"],["ea80","陘陙陚陜陝陞陠陣陥陦陫陭",4,"陳陸",12,"隇隉隊辍辎辏辘辚軎戋戗戛戟戢戡戥戤戬臧瓯瓴瓿甏甑甓攴旮旯旰昊昙杲昃昕昀炅曷昝昴昱昶昵耆晟晔晁晏晖晡晗晷暄暌暧暝暾曛曜曦曩贲贳贶贻贽赀赅赆赈赉赇赍赕赙觇觊觋觌觎觏觐觑牮犟牝牦牯牾牿犄犋犍犏犒挈挲掰"],["eb40","隌階隑隒隓隕隖隚際隝",9,"隨",7,"隱隲隴隵隷隸隺隻隿雂雃雈雊雋雐雑雓雔雖",9,"雡",6,"雫"],["eb80","雬雭雮雰雱雲雴雵雸雺電雼雽雿霂霃霅霊霋霌霐霑霒霔霕霗",4,"霝霟霠搿擘耄毪毳毽毵毹氅氇氆氍氕氘氙氚氡氩氤氪氲攵敕敫牍牒牖爰虢刖肟肜肓肼朊肽肱肫肭肴肷胧胨胩胪胛胂胄胙胍胗朐胝胫胱胴胭脍脎胲胼朕脒豚脶脞脬脘脲腈腌腓腴腙腚腱腠腩腼腽腭腧塍媵膈膂膑滕膣膪臌朦臊膻"],["ec40","霡",8,"霫霬霮霯霱霳",4,"霺霻霼霽霿",18,"靔靕靗靘靚靜靝靟靣靤靦靧靨靪",7],["ec80","靲靵靷",4,"靽",7,"鞆",4,"鞌鞎鞏鞐鞓鞕鞖鞗鞙",4,"臁膦欤欷欹歃歆歙飑飒飓飕飙飚殳彀毂觳斐齑斓於旆旄旃旌旎旒旖炀炜炖炝炻烀炷炫炱烨烊焐焓焖焯焱煳煜煨煅煲煊煸煺熘熳熵熨熠燠燔燧燹爝爨灬焘煦熹戾戽扃扈扉礻祀祆祉祛祜祓祚祢祗祠祯祧祺禅禊禚禧禳忑忐"],["ed40","鞞鞟鞡鞢鞤",6,"鞬鞮鞰鞱鞳鞵",46],["ed80","韤韥韨韮",4,"韴韷",23,"怼恝恚恧恁恙恣悫愆愍慝憩憝懋懑戆肀聿沓泶淼矶矸砀砉砗砘砑斫砭砜砝砹砺砻砟砼砥砬砣砩硎硭硖硗砦硐硇硌硪碛碓碚碇碜碡碣碲碹碥磔磙磉磬磲礅磴礓礤礞礴龛黹黻黼盱眄眍盹眇眈眚眢眙眭眦眵眸睐睑睇睃睚睨"],["ee40","頏",62],["ee80","顎",32,"睢睥睿瞍睽瞀瞌瞑瞟瞠瞰瞵瞽町畀畎畋畈畛畲畹疃罘罡罟詈罨罴罱罹羁罾盍盥蠲钅钆钇钋钊钌钍钏钐钔钗钕钚钛钜钣钤钫钪钭钬钯钰钲钴钶",4,"钼钽钿铄铈",6,"铐铑铒铕铖铗铙铘铛铞铟铠铢铤铥铧铨铪"],["ef40","顯",5,"颋颎颒颕颙颣風",37,"飏飐飔飖飗飛飜飝飠",4],["ef80","飥飦飩",30,"铩铫铮铯铳铴铵铷铹铼铽铿锃锂锆锇锉锊锍锎锏锒",4,"锘锛锝锞锟锢锪锫锩锬锱锲锴锶锷锸锼锾锿镂锵镄镅镆镉镌镎镏镒镓镔镖镗镘镙镛镞镟镝镡镢镤",8,"镯镱镲镳锺矧矬雉秕秭秣秫稆嵇稃稂稞稔"],["f040","餈",4,"餎餏餑",28,"餯",26],["f080","饊",9,"饖",12,"饤饦饳饸饹饻饾馂馃馉稹稷穑黏馥穰皈皎皓皙皤瓞瓠甬鸠鸢鸨",4,"鸲鸱鸶鸸鸷鸹鸺鸾鹁鹂鹄鹆鹇鹈鹉鹋鹌鹎鹑鹕鹗鹚鹛鹜鹞鹣鹦",6,"鹱鹭鹳疒疔疖疠疝疬疣疳疴疸痄疱疰痃痂痖痍痣痨痦痤痫痧瘃痱痼痿瘐瘀瘅瘌瘗瘊瘥瘘瘕瘙"],["f140","馌馎馚",10,"馦馧馩",47],["f180","駙",32,"瘛瘼瘢瘠癀瘭瘰瘿瘵癃瘾瘳癍癞癔癜癖癫癯翊竦穸穹窀窆窈窕窦窠窬窨窭窳衤衩衲衽衿袂袢裆袷袼裉裢裎裣裥裱褚裼裨裾裰褡褙褓褛褊褴褫褶襁襦襻疋胥皲皴矜耒耔耖耜耠耢耥耦耧耩耨耱耋耵聃聆聍聒聩聱覃顸颀颃"],["f240","駺",62],["f280","騹",32,"颉颌颍颏颔颚颛颞颟颡颢颥颦虍虔虬虮虿虺虼虻蚨蚍蚋蚬蚝蚧蚣蚪蚓蚩蚶蛄蚵蛎蚰蚺蚱蚯蛉蛏蚴蛩蛱蛲蛭蛳蛐蜓蛞蛴蛟蛘蛑蜃蜇蛸蜈蜊蜍蜉蜣蜻蜞蜥蜮蜚蜾蝈蜴蜱蜩蜷蜿螂蜢蝽蝾蝻蝠蝰蝌蝮螋蝓蝣蝼蝤蝙蝥螓螯螨蟒"],["f340","驚",17,"驲骃骉骍骎骔骕骙骦骩",6,"骲骳骴骵骹骻骽骾骿髃髄髆",4,"髍髎髏髐髒體髕髖髗髙髚髛髜"],["f380","髝髞髠髢髣髤髥髧髨髩髪髬髮髰",8,"髺髼",6,"鬄鬅鬆蟆螈螅螭螗螃螫蟥螬螵螳蟋蟓螽蟑蟀蟊蟛蟪蟠蟮蠖蠓蟾蠊蠛蠡蠹蠼缶罂罄罅舐竺竽笈笃笄笕笊笫笏筇笸笪笙笮笱笠笥笤笳笾笞筘筚筅筵筌筝筠筮筻筢筲筱箐箦箧箸箬箝箨箅箪箜箢箫箴篑篁篌篝篚篥篦篪簌篾篼簏簖簋"],["f440","鬇鬉",5,"鬐鬑鬒鬔",10,"鬠鬡鬢鬤",10,"鬰鬱鬳",7,"鬽鬾鬿魀魆魊魋魌魎魐魒魓魕",5],["f480","魛",32,"簟簪簦簸籁籀臾舁舂舄臬衄舡舢舣舭舯舨舫舸舻舳舴舾艄艉艋艏艚艟艨衾袅袈裘裟襞羝羟羧羯羰羲籼敉粑粝粜粞粢粲粼粽糁糇糌糍糈糅糗糨艮暨羿翎翕翥翡翦翩翮翳糸絷綦綮繇纛麸麴赳趄趔趑趱赧赭豇豉酊酐酎酏酤"],["f540","魼",62],["f580","鮻",32,"酢酡酰酩酯酽酾酲酴酹醌醅醐醍醑醢醣醪醭醮醯醵醴醺豕鹾趸跫踅蹙蹩趵趿趼趺跄跖跗跚跞跎跏跛跆跬跷跸跣跹跻跤踉跽踔踝踟踬踮踣踯踺蹀踹踵踽踱蹉蹁蹂蹑蹒蹊蹰蹶蹼蹯蹴躅躏躔躐躜躞豸貂貊貅貘貔斛觖觞觚觜"],["f640","鯜",62],["f680","鰛",32,"觥觫觯訾謦靓雩雳雯霆霁霈霏霎霪霭霰霾龀龃龅",5,"龌黾鼋鼍隹隼隽雎雒瞿雠銎銮鋈錾鍪鏊鎏鐾鑫鱿鲂鲅鲆鲇鲈稣鲋鲎鲐鲑鲒鲔鲕鲚鲛鲞",5,"鲥",4,"鲫鲭鲮鲰",7,"鲺鲻鲼鲽鳄鳅鳆鳇鳊鳋"],["f740","鰼",62],["f780","鱻鱽鱾鲀鲃鲄鲉鲊鲌鲏鲓鲖鲗鲘鲙鲝鲪鲬鲯鲹鲾",4,"鳈鳉鳑鳒鳚鳛鳠鳡鳌",4,"鳓鳔鳕鳗鳘鳙鳜鳝鳟鳢靼鞅鞑鞒鞔鞯鞫鞣鞲鞴骱骰骷鹘骶骺骼髁髀髅髂髋髌髑魅魃魇魉魈魍魑飨餍餮饕饔髟髡髦髯髫髻髭髹鬈鬏鬓鬟鬣麽麾縻麂麇麈麋麒鏖麝麟黛黜黝黠黟黢黩黧黥黪黯鼢鼬鼯鼹鼷鼽鼾齄"],["f840","鳣",62],["f880","鴢",32],["f940","鵃",62],["f980","鶂",32],["fa40","鶣",62],["fa80","鷢",32],["fb40","鸃",27,"鸤鸧鸮鸰鸴鸻鸼鹀鹍鹐鹒鹓鹔鹖鹙鹝鹟鹠鹡鹢鹥鹮鹯鹲鹴",9,"麀"],["fb80","麁麃麄麅麆麉麊麌",5,"麔",8,"麞麠",5,"麧麨麩麪"],["fc40","麫",8,"麵麶麷麹麺麼麿",4,"黅黆黇黈黊黋黌黐黒黓黕黖黗黙黚點黡黣黤黦黨黫黬黭黮黰",8,"黺黽黿",6],["fc80","鼆",4,"鼌鼏鼑鼒鼔鼕鼖鼘鼚",5,"鼡鼣",8,"鼭鼮鼰鼱"],["fd40","鼲",4,"鼸鼺鼼鼿",4,"齅",10,"齒",38],["fd80","齹",5,"龁龂龍",11,"龜龝龞龡",4,"郎凉秊裏隣"],["fe40","兀嗀﨎﨏﨑﨓﨔礼﨟蘒﨡﨣﨤﨧﨨﨩"]]')},function(n,i,o){var u=o(99);n.exports=Object("z").propertyIsEnumerable(0)?Object:function(n){return"String"==u(n)?n.split(""):Object(n)}},function(n,i){n.exports=function(n){if(null==n)throw TypeError("Can't call method on  "+n);return n}},function(n,i,o){var u=o(128),p=o(100),g=o(50),y=o(176),w=o(51),x=o(255),_=Object.getOwnPropertyDescriptor;i.f=o(23)?_:function getOwnPropertyDescriptor(n,i){if(n=g(n),i=y(i,!0),x)try{return _(n,i)}catch(n){}if(w(n,i))return p(!u.f.call(n,i),n[i])}},function(n,i,o){var u=o(22);n.exports=function(n,i){if(!u(n))return n;var o,p;if(i&&"function"==typeof(o=n.toString)&&!u(p=o.call(n)))return p;if("function"==typeof(o=n.valueOf)&&!u(p=o.call(n)))return p;if(!i&&"function"==typeof(o=n.toString)&&!u(p=o.call(n)))return p;throw TypeError("Can't convert object to primitive value")}},function(n,i,o){var u=o(22),p=o(17).document,g=u(p)&&u(p.createElement);n.exports=function(n){return g?p.createElement(n):{}}},function(n,i,o){var u=o(9),p=o(7),g=o(52);n.exports=function(n,i){var o=(p.Object||{})[n]||Object[n],y={};y[n]=i(o),u(u.S+u.F*g((function(){o(1)})),"Object",y)}},function(n,i,o){"use strict";var u=o(104),p=o(9),g=o(257),y=o(39),w=o(103),x=o(430),_=o(106),k=o(432),P=o(21)("iterator"),E=!([].keys&&"next"in[].keys()),returnThis=function(){return this};n.exports=function(n,i,o,O,B,I,D){x(o,i,O);var R,N,U,getMethod=function(n){if(!E&&n in X)return X[n];switch(n){case"keys":return function keys(){return new o(this,n)};case"values":return function values(){return new o(this,n)}}return function entries(){return new o(this,n)}},W=i+" Iterator",G="values"==B,j=!1,X=n.prototype,Y=X[P]||X["@@iterator"]||B&&X[B],K=Y||getMethod(B),J=B?G?getMethod("entries"):K:void 0,$="Array"==i&&X.entries||Y;if($&&(U=k($.call(new n)))!==Object.prototype&&U.next&&(_(U,W,!0),u||"function"==typeof U[P]||y(U,P,returnThis)),G&&Y&&"values"!==Y.name&&(j=!0,K=function values(){return Y.call(this)}),u&&!D||!E&&!j&&X[P]||y(X,P,K),w[i]=K,w[W]=returnThis,B)if(R={values:G?K:getMethod("values"),keys:I?K:getMethod("keys"),entries:J},D)for(N in R)N in X||g(X,N,R[N]);else p(p.P+p.F*(E||j),i,R);return R}},function(n,i){var o=Math.ceil,u=Math.floor;n.exports=function(n){return isNaN(n=+n)?0:(n>0?u:o)(n)}},function(n,i,o){var u=o(182)("keys"),p=o(131);n.exports=function(n){return u[n]||(u[n]=p(n))}},function(n,i,o){var u=o(7),p=o(17),g=p["__core-js_shared__"]||(p["__core-js_shared__"]={});(n.exports=function(n,i){return g[n]||(g[n]=void 0!==i?i:{})})("versions",[]).push({version:u.version,mode:o(104)?"pure":"global",copyright:"© 2020 Denis Pushkarev (zloirock.ru)"})},function(n,i){n.exports="constructor,hasOwnProperty,isPrototypeOf,propertyIsEnumerable,toLocaleString,toString,valueOf".split(",")},function(n,i,o){var u=o(185),p=o(21)("iterator"),g=o(103);n.exports=o(7).getIteratorMethod=function(n){if(null!=n)return n[p]||n["@@iterator"]||g[u(n)]}},function(n,i,o){var u=o(99),p=o(21)("toStringTag"),g="Arguments"==u(function(){return arguments}());n.exports=function(n){var i,o,y;return void 0===n?"Undefined":null===n?"Null":"string"==typeof(o=function(n,i){try{return n[i]}catch(n){}}(i=Object(n),p))?o:g?u(i):"Object"==(y=u(i))&&"function"==typeof i.callee?"Arguments":y}},function(n,i,o){"use strict";i.__esModule=!0;var u=_interopRequireDefault(o(438)),p=_interopRequireDefault(o(440)),g="function"==typeof p.default&&"symbol"==typeof u.default?function(n){return typeof n}:function(n){return n&&"function"==typeof p.default&&n.constructor===p.default&&n!==p.default.prototype?"symbol":typeof n};function _interopRequireDefault(n){return n&&n.__esModule?n:{default:n}}i.default="function"==typeof p.default&&"symbol"===g(u.default)?function(n){return void 0===n?"undefined":g(n)}:function(n){return n&&"function"==typeof p.default&&n.constructor===p.default&&n!==p.default.prototype?"symbol":void 0===n?"undefined":g(n)}},function(n,i,o){i.f=o(21)},function(n,i,o){var u=o(17),p=o(7),g=o(104),y=o(187),w=o(26).f;n.exports=function(n){var i=p.Symbol||(p.Symbol=g?{}:u.Symbol||{});"_"==n.charAt(0)||n in i||w(i,n,{value:y.f(n)})}},function(n,i){i.f=Object.getOwnPropertySymbols},function(n,i,o){var u=o(39);n.exports=function(n,i,o){for(var p in i)o&&n[p]?n[p]=i[p]:u(n,p,i[p]);return n}},function(n,i){n.exports=function(n,i,o,u){if(!(n instanceof i)||void 0!==u&&u in n)throw TypeError(o+": incorrect invocation!");return n}},function(n,i,o){var u=o(22);n.exports=function(n,i){if(!u(n)||n._t!==i)throw TypeError("Incompatible receiver, "+i+" required!");return n}},function(n,i,o){"use strict";var u=o(481);n.exports=Function.prototype.bind||u},function(n,i,o){"use strict";var u=o(193),p=o(275),g=p("%Function.prototype.apply%"),y=p("%Function.prototype.call%"),w=p("%Reflect.apply%",!0)||u.call(y,g),x=p("%Object.getOwnPropertyDescriptor%",!0),_=p("%Object.defineProperty%",!0),k=p("%Math.max%");if(_)try{_({},"a",{value:1})}catch(n){_=null}n.exports=function callBind(n){var i=w(u,y,arguments);if(x&&_){var o=x(i,"length");o.configurable&&_(i,"length",{value:1+k(0,n.length-(arguments.length-1))})}return i};var P=function applyBind(){return w(u,g,arguments)};_?_(n.exports,"apply",{value:P}):n.exports.apply=P},function(n,i,o){"use strict";var u=o(1),p=o(93).indexOf,g=o(59),y=o(30),w=[].indexOf,x=!!w&&1/[1].indexOf(1,-0)<0,_=g("indexOf"),k=y("indexOf",{ACCESSORS:!0,1:0});u({target:"Array",proto:!0,forced:x||!_||!k},{indexOf:function indexOf(n){return x?w.apply(this,arguments)||0:p(this,n,arguments.length>1?arguments[1]:void 0)}})},function(n,i,o){"use strict";var u=o(101);function PromiseCapability(n){var i,o;this.promise=new n((function(n,u){if(void 0!==i||void 0!==o)throw TypeError("Bad Promise constructor");i=n,o=u})),this.resolve=u(i),this.reject=u(o)}n.exports.f=function(n){return new PromiseCapability(n)}},function(n,i,o){"use strict";var u=o(0).isString,p=o(0).isNumber,g=o(0).isObject,y=o(0).isArray,w=o(0).isUndefined,x=o(290),_=/^(\s)+/g,k=/(\s)+$/g;function TextTools(n){this.fontProvider=n}function splitWords(n,i){var o=[];if(n=n.replace(/\t/g,"    "),i)return o.push({text:n}),o;for(var u,p=new x(n),g=0;u=p.nextBreak();){var y=n.slice(g,u.position);u.required||y.match(/\r?\n$|\r$/)?(y=y.replace(/\r?\n$|\r$/,""),o.push({text:y,lineEnd:!0})):o.push({text:y}),g=u.position}return o}function copyStyle(n,i){for(var o in i=i||{},n=n||{})"text"!=o&&n.hasOwnProperty(o)&&(i[o]=n[o]);return i}function normalizeString(n){return null==n?"":p(n)?n.toString():u(n)?n:n.toString()}function getStyleProperty(n,i,o,u){var p;return void 0!==n[o]&&null!==n[o]?n[o]:i?(i.auto(n,(function(){p=i.getProperty(o)})),null!=p?p:u):u}function widthOfString(n,i,o,u,p){return i.widthOfString(n,o,p)+(u||0)*(n.length-1)}TextTools.prototype.buildInlines=function(n,i){var o,u=function measure(n,i,o){var u=function normalizeTextArray(n,i){function getOneWord(n,i,o){if(w(i[n]))return null;if(i[n].lineEnd)return null;var u=i[n].text;if(o){var p=splitWords(normalizeString(u),!1);if(w(p[p.length-1]))return null;u=p[p.length-1].text}return u}var o=[];y(n)||(n=[n]);n=function flatten(n){return n.reduce((function(n,i){var o=y(i.text)?flatten(i.text):i,u=[].concat(o).some(Array.isArray);return n.concat(u?flatten(o):o)}),[])}(n);for(var u=null,p=0,x=n.length;p<x;p++){var _,k=n[p],P=null,E=getStyleProperty(k||{},i,"noWrap",!1);if(g(k)?(k._textRef&&k._textRef._textNodeRef.text&&(k.text=k._textRef._textNodeRef.text),_=splitWords(normalizeString(k.text),E),P=copyStyle(k)):_=splitWords(normalizeString(k),E),u&&_.length){var O=getOneWord(0,_,E);1===splitWords(normalizeString(u+O),!1).length&&(o[o.length-1].noNewLine=!0)}for(var B=0,I=_.length;B<I;B++){var D={text:_[B].text};_[B].lineEnd&&(D.lineEnd=!0),copyStyle(P,D),o.push(D)}u=null,p+1<x&&(u=getOneWord(_.length-1,_,E))}return o}(i,o);if(u.length){var p=getStyleProperty(u[0],o,"leadingIndent",0);p&&(u[0].leadingCut=-p,u[0].leadingIndent=p)}return u.forEach((function(i){var u=getStyleProperty(i,o,"font","Roboto"),p=getStyleProperty(i,o,"fontSize",12),g=getStyleProperty(i,o,"fontFeatures",null),y=getStyleProperty(i,o,"bold",!1),w=getStyleProperty(i,o,"italics",!1),x=getStyleProperty(i,o,"color","black"),P=getStyleProperty(i,o,"decoration",null),E=getStyleProperty(i,o,"decorationColor",null),O=getStyleProperty(i,o,"decorationStyle",null),B=getStyleProperty(i,o,"background",null),I=getStyleProperty(i,o,"lineHeight",1),D=getStyleProperty(i,o,"characterSpacing",0),R=getStyleProperty(i,o,"link",null),N=getStyleProperty(i,o,"linkToPage",null),U=getStyleProperty(i,o,"linkToDestination",null),W=getStyleProperty(i,o,"noWrap",null),G=getStyleProperty(i,o,"preserveLeadingSpaces",!1),j=getStyleProperty(i,o,"preserveTrailingSpaces",!1),X=getStyleProperty(i,o,"opacity",1),Y=getStyleProperty(i,o,"sup",!1),K=getStyleProperty(i,o,"sub",!1);(Y||K)&&void 0===i.fontSize&&(p*=.58);var J,$,tt=n.provideFont(u,y,w);i.width=widthOfString(i.text,tt,p,D,g),i.height=tt.lineHeight(p)*I,i.leadingCut||(i.leadingCut=0),!G&&(J=i.text.match(_))&&(i.leadingCut+=widthOfString(J[0],tt,p,D,g)),!j&&($=i.text.match(k))?i.trailingCut=widthOfString($[0],tt,p,D,g):i.trailingCut=0,i.alignment=getStyleProperty(i,o,"alignment","left"),i.font=tt,i.fontSize=p,i.fontFeatures=g,i.characterSpacing=D,i.color=x,i.decoration=P,i.decorationColor=E,i.decorationStyle=O,i.background=B,i.link=R,i.linkToPage=N,i.linkToDestination=U,i.noWrap=W,i.opacity=X,i.sup=Y,i.sub=K})),u}(this.fontProvider,n,i),p=0,x=0;return u.forEach((function(n){p=Math.max(p,n.width-n.leadingCut-n.trailingCut),o||(o={width:0,leadingCut:n.leadingCut,trailingCut:0}),o.width+=n.width,o.trailingCut=n.trailingCut,x=Math.max(x,function getTrimmedWidth(n){return Math.max(0,n.width-n.leadingCut-n.trailingCut)}(o)),n.lineEnd&&(o=null)})),getStyleProperty({},i,"noWrap",!1)&&(p=x),{items:u,minWidth:p,maxWidth:x}},TextTools.prototype.sizeOfString=function(n,i){n=n?n.toString().replace(/\t/g,"    "):"";var o=getStyleProperty({},i,"font","Roboto"),u=getStyleProperty({},i,"fontSize",12),p=getStyleProperty({},i,"fontFeatures",null),g=getStyleProperty({},i,"bold",!1),y=getStyleProperty({},i,"italics",!1),w=getStyleProperty({},i,"lineHeight",1),x=getStyleProperty({},i,"characterSpacing",0),_=this.fontProvider.provideFont(o,g,y);return{width:widthOfString(n,_,u,x,p),height:_.lineHeight(u)*w,fontSize:u,lineHeight:w,ascender:_.ascender/1e3*u,descender:_.descender/1e3*u}},TextTools.prototype.sizeOfRotatedText=function(n,i,o){var u=i*Math.PI/-180,p=this.sizeOfString(n,o);return{width:Math.abs(p.height*Math.sin(u))+Math.abs(p.width*Math.cos(u)),height:Math.abs(p.width*Math.sin(u))+Math.abs(p.height*Math.cos(u))}},TextTools.prototype.widthOfString=function(n,i,o,u,p){return widthOfString(n,i,o,u,p)},n.exports=TextTools},function(n,i,o){"use strict";var u=o(0).isString;function isAutoColumn(n){return"auto"===n.width}function isStarColumn(n){return null===n.width||void 0===n.width||"*"===n.width||"star"===n.width}n.exports={buildColumnWidths:function buildColumnWidths(n,i){var o=[],p=0,g=0,y=[],w=0,x=0,_=[],k=i;n.forEach((function(n){isAutoColumn(n)?(o.push(n),p+=n._minWidth,g+=n._maxWidth):isStarColumn(n)?(y.push(n),w=Math.max(w,n._minWidth),x=Math.max(x,n._maxWidth)):_.push(n)})),_.forEach((function(n){u(n.width)&&/\d+%/.test(n.width)&&(n.width=parseFloat(n.width)*k/100),n.width<n._minWidth&&n.elasticWidth?n._calcWidth=n._minWidth:n._calcWidth=n.width,i-=n._calcWidth}));var P=p+w*y.length,E=g+x*y.length;if(P>=i)o.forEach((function(n){n._calcWidth=n._minWidth})),y.forEach((function(n){n._calcWidth=w}));else{if(E<i)o.forEach((function(n){n._calcWidth=n._maxWidth,i-=n._calcWidth}));else{var O=i-P,B=E-P;o.forEach((function(n){var o=n._maxWidth-n._minWidth;n._calcWidth=n._minWidth+o*O/B,i-=n._calcWidth}))}if(y.length>0){var I=i/y.length;y.forEach((function(n){n._calcWidth=I}))}}},measureMinMax:function measureMinMax(n){for(var i={min:0,max:0},o={min:0,max:0},u=0,p=0,g=n.length;p<g;p++){var y=n[p];isStarColumn(y)?(o.min=Math.max(o.min,y._minWidth),o.max=Math.max(o.max,y._maxWidth),u++):isAutoColumn(y)?(i.min+=y._minWidth,i.max+=y._maxWidth):(i.min+=void 0!==y.width&&y.width||y._minWidth,i.max+=void 0!==y.width&&y.width||y._maxWidth)}return u&&(i.min+=u*o.min,i.max+=u*o.max),i},isAutoColumn:isAutoColumn,isStarColumn:isStarColumn}},function(n,i,o){var u=o(41),p=o(3);n.exports="process"==u(p.process)},function(n,i){var o={}.toString;n.exports=Array.isArray||function(n){return"[object Array]"==o.call(n)}},function(n,i,o){"use strict";var u=o(1),p=o(3),g=o(34),y=o(56),w=o(13),x=o(146),_=o(205),k=o(4),P=o(15),E=o(115),O=o(14),B=o(11),I=o(19),D=o(27),R=o(55),N=o(40),U=o(58),W=o(94),G=o(57),j=o(313),X=o(144),Y=o(54),K=o(16),J=o(110),$=o(18),tt=o(25),et=o(142),rt=o(111),nt=o(113),it=o(112),ot=o(6),at=o(207),st=o(208),lt=o(95),ct=o(42),ut=o(20).forEach,ft=rt("hidden"),ht=ot("toPrimitive"),dt=ct.set,pt=ct.getterFor("Symbol"),gt=Object.prototype,yt=p.Symbol,vt=g("JSON","stringify"),mt=Y.f,bt=K.f,wt=j.f,xt=J.f,_t=et("symbols"),St=et("op-symbols"),Ct=et("string-to-symbol-registry"),kt=et("symbol-to-string-registry"),At=et("wks"),Pt=p.QObject,Tt=!Pt||!Pt.prototype||!Pt.prototype.findChild,Et=w&&k((function(){return 7!=U(bt({},"a",{get:function(){return bt(this,"a",{value:7}).a}})).a}))?function(n,i,o){var u=mt(gt,i);u&&delete gt[i],bt(n,i,o),u&&n!==gt&&bt(gt,i,u)}:bt,wrap=function(n,i){var o=_t[n]=U(yt.prototype);return dt(o,{type:"Symbol",tag:n,description:i}),w||(o.description=i),o},Ot=_?function(n){return"symbol"==typeof n}:function(n){return Object(n)instanceof yt},Bt=function defineProperty(n,i,o){n===gt&&Bt(St,i,o),B(n);var u=R(i,!0);return B(o),P(_t,u)?(o.enumerable?(P(n,ft)&&n[ft][u]&&(n[ft][u]=!1),o=U(o,{enumerable:N(0,!1)})):(P(n,ft)||bt(n,ft,N(1,{})),n[ft][u]=!0),Et(n,u,o)):bt(n,u,o)},It=function defineProperties(n,i){B(n);var o=D(i),u=W(o).concat(Mt(o));return ut(u,(function(i){w&&!Ft.call(o,i)||Bt(n,i,o[i])})),n},Ft=function propertyIsEnumerable(n){var i=R(n,!0),o=xt.call(this,i);return!(this===gt&&P(_t,i)&&!P(St,i))&&(!(o||!P(this,i)||!P(_t,i)||P(this,ft)&&this[ft][i])||o)},Lt=function getOwnPropertyDescriptor(n,i){var o=D(n),u=R(i,!0);if(o!==gt||!P(_t,u)||P(St,u)){var p=mt(o,u);return!p||!P(_t,u)||P(o,ft)&&o[ft][u]||(p.enumerable=!0),p}},Dt=function getOwnPropertyNames(n){var i=wt(D(n)),o=[];return ut(i,(function(n){P(_t,n)||P(nt,n)||o.push(n)})),o},Mt=function getOwnPropertySymbols(n){var i=n===gt,o=wt(i?St:D(n)),u=[];return ut(o,(function(n){!P(_t,n)||i&&!P(gt,n)||u.push(_t[n])})),u};(x||(tt((yt=function Symbol(){if(this instanceof yt)throw TypeError("Symbol is not a constructor");var n=arguments.length&&void 0!==arguments[0]?String(arguments[0]):void 0,i=it(n),setter=function(n){this===gt&&setter.call(St,n),P(this,ft)&&P(this[ft],i)&&(this[ft][i]=!1),Et(this,i,N(1,n))};return w&&Tt&&Et(gt,i,{configurable:!0,set:setter}),wrap(i,n)}).prototype,"toString",(function toString(){return pt(this).tag})),tt(yt,"withoutSetter",(function(n){return wrap(it(n),n)})),J.f=Ft,K.f=Bt,Y.f=Lt,G.f=j.f=Dt,X.f=Mt,at.f=function(n){return wrap(ot(n),n)},w&&(bt(yt.prototype,"description",{configurable:!0,get:function description(){return pt(this).description}}),y||tt(gt,"propertyIsEnumerable",Ft,{unsafe:!0}))),u({global:!0,wrap:!0,forced:!x,sham:!x},{Symbol:yt}),ut(W(At),(function(n){st(n)})),u({target:"Symbol",stat:!0,forced:!x},{for:function(n){var i=String(n);if(P(Ct,i))return Ct[i];var o=yt(i);return Ct[i]=o,kt[o]=i,o},keyFor:function keyFor(n){if(!Ot(n))throw TypeError(n+" is not a symbol");if(P(kt,n))return kt[n]},useSetter:function(){Tt=!0},useSimple:function(){Tt=!1}}),u({target:"Object",stat:!0,forced:!x,sham:!w},{create:function create(n,i){return void 0===i?U(n):It(U(n),i)},defineProperty:Bt,defineProperties:It,getOwnPropertyDescriptor:Lt}),u({target:"Object",stat:!0,forced:!x},{getOwnPropertyNames:Dt,getOwnPropertySymbols:Mt}),u({target:"Object",stat:!0,forced:k((function(){X.f(1)}))},{getOwnPropertySymbols:function getOwnPropertySymbols(n){return X.f(I(n))}}),vt)&&u({target:"JSON",stat:!0,forced:!x||k((function(){var n=yt();return"[null]"!=vt([n])||"{}"!=vt({a:n})||"{}"!=vt(Object(n))}))},{stringify:function stringify(n,i,o){for(var u,p=[n],g=1;arguments.length>g;)p.push(arguments[g++]);if(u=i,(O(i)||void 0!==n)&&!Ot(n))return E(i)||(i=function(n,i){if("function"==typeof u&&(i=u.call(this,n,i)),!Ot(i))return i}),p[1]=i,vt.apply(null,p)}});yt.prototype[ht]||$(yt.prototype,ht,yt.prototype.valueOf),lt(yt,"Symbol"),nt[ft]=!0},function(n,i,o){var u=o(13),p=o(4),g=o(138);n.exports=!u&&!p((function(){return 7!=Object.defineProperty(g("div"),"a",{get:function(){return 7}}).a}))},function(n,i,o){var u=o(15),p=o(311),g=o(54),y=o(16);n.exports=function(n,i){for(var o=p(i),w=y.f,x=g.f,_=0;_<o.length;_++){var k=o[_];u(n,k)||w(n,k,x(i,k))}}},function(n,i,o){var u=o(15),p=o(27),g=o(93).indexOf,y=o(113);n.exports=function(n,i){var o,w=p(n),x=0,_=[];for(o in w)!u(y,o)&&u(w,o)&&_.push(o);for(;i.length>x;)u(w,o=i[x++])&&(~g(_,o)||_.push(o));return _}},function(n,i,o){var u=o(146);n.exports=u&&!Symbol.sham&&"symbol"==typeof Symbol.iterator},function(n,i,o){var u=o(34);n.exports=u("document","documentElement")},function(n,i,o){var u=o(6);i.f=u},function(n,i,o){var u=o(114),p=o(15),g=o(207),y=o(16).f;n.exports=function(n){var i=u.Symbol||(u.Symbol={});p(i,n)||y(i,n,{value:g.f(n)})}},function(n,i,o){var u=o(14),p=o(115),g=o(6)("species");n.exports=function(n,i){var o;return p(n)&&("function"!=typeof(o=n.constructor)||o!==Array&&!p(o.prototype)?u(o)&&null===(o=o[g])&&(o=void 0):o=void 0),new(void 0===o?Array:o)(0===i?0:i)}},function(n,i,o){"use strict";var u=o(1),p=o(13),g=o(3),y=o(15),w=o(14),x=o(16).f,_=o(203),k=g.Symbol;if(p&&"function"==typeof k&&(!("description"in k.prototype)||void 0!==k().description)){var P={},E=function Symbol(){var n=arguments.length<1||void 0===arguments[0]?void 0:String(arguments[0]),i=this instanceof E?new k(n):void 0===n?k():k(n);return""===n&&(P[i]=!0),i};_(E,k);var O=E.prototype=k.prototype;O.constructor=E;var B=O.toString,I="Symbol(test)"==String(k("test")),D=/^Symbol\((.*)\)[^)]+$/;x(O,"description",{configurable:!0,get:function description(){var n=w(this)?this.valueOf():this,i=B.call(n);if(y(P,n))return"";var o=I?i.slice(7,-1):i.replace(D,"$1");return""===o?void 0:o}}),u({global:!0,forced:!0},{Symbol:E})}},function(n,i,o){o(208)("iterator")},function(n,i,o){"use strict";var u=o(1),p=o(4),g=o(115),y=o(14),w=o(19),x=o(8),_=o(147),k=o(209),P=o(116),E=o(6),O=o(148),B=E("isConcatSpreadable"),I=O>=51||!p((function(){var n=[];return n[B]=!1,n.concat()[0]!==n})),D=P("concat"),isConcatSpreadable=function(n){if(!y(n))return!1;var i=n[B];return void 0!==i?!!i:g(n)};u({target:"Array",proto:!0,forced:!I||!D},{concat:function concat(n){var i,o,u,p,g,y=w(this),P=k(y,0),E=0;for(i=-1,u=arguments.length;i<u;i++)if(isConcatSpreadable(g=-1===i?y:arguments[i])){if(E+(p=x(g.length))>9007199254740991)throw TypeError("Maximum allowed index exceeded");for(o=0;o<p;o++,E++)o in g&&_(P,E,g[o])}else{if(E>=9007199254740991)throw TypeError("Maximum allowed index exceeded");_(P,E++,g)}return P.length=E,P}})},function(n,i,o){var u=o(1),p=o(150),g=o(151);u({target:"Array",proto:!0},{fill:p}),g("fill")},function(n,i,o){"use strict";var u=o(20).forEach,p=o(59),g=o(30),y=p("forEach"),w=g("forEach");n.exports=y&&w?[].forEach:function forEach(n){return u(this,n,arguments.length>1?arguments[1]:void 0)}},function(n,i,o){var u=o(11);n.exports=function(n){var i=n.return;if(void 0!==i)return u(i.call(n)).value}},function(n,i,o){"use strict";var u=o(1),p=o(93).includes,g=o(151);u({target:"Array",proto:!0,forced:!o(30)("indexOf",{ACCESSORS:!0,1:0})},{includes:function includes(n){return p(this,n,arguments.length>1?arguments[1]:void 0)}}),g("includes")},function(n,i,o){"use strict";var u=o(1),p=o(320),g=o(60),y=o(45),w=o(95),x=o(18),_=o(25),k=o(6),P=o(56),E=o(97),O=o(218),B=O.IteratorPrototype,I=O.BUGGY_SAFARI_ITERATORS,D=k("iterator"),returnThis=function(){return this};n.exports=function(n,i,o,k,O,R,N){p(o,i,k);var U,W,G,getIterationMethod=function(n){if(n===O&&J)return J;if(!I&&n in Y)return Y[n];switch(n){case"keys":return function keys(){return new o(this,n)};case"values":return function values(){return new o(this,n)};case"entries":return function entries(){return new o(this,n)}}return function(){return new o(this)}},j=i+" Iterator",X=!1,Y=n.prototype,K=Y[D]||Y["@@iterator"]||O&&Y[O],J=!I&&K||getIterationMethod(O),$="Array"==i&&Y.entries||K;if($&&(U=g($.call(new n)),B!==Object.prototype&&U.next&&(P||g(U)===B||(y?y(U,B):"function"!=typeof U[D]&&x(U,D,returnThis)),w(U,j,!0,!0),P&&(E[j]=returnThis))),"values"==O&&K&&"values"!==K.name&&(X=!0,J=function values(){return K.call(this)}),P&&!N||Y[D]===J||x(Y,D,J),E[i]=J,O)if(W={values:getIterationMethod("values"),keys:R?J:getIterationMethod("keys"),entries:getIterationMethod("entries")},N)for(G in W)(I||X||!(G in Y))&&_(Y,G,W[G]);else u({target:i,proto:!0,forced:I||X},W);return W}},function(n,i,o){"use strict";var u,p,g,y=o(60),w=o(18),x=o(15),_=o(6),k=o(56),P=_("iterator"),E=!1;[].keys&&("next"in(g=[].keys())?(p=y(y(g)))!==Object.prototype&&(u=p):E=!0),null==u&&(u={}),k||x(u,P)||w(u,P,(function(){return this})),n.exports={IteratorPrototype:u,BUGGY_SAFARI_ITERATORS:E}},function(n,i,o){var u=o(4);n.exports=!u((function(){function F(){}return F.prototype.constructor=null,Object.getPrototypeOf(new F)!==F.prototype}))},function(n,i,o){"use strict";var u=o(1),p=o(92),g=o(27),y=o(59),w=[].join,x=p!=Object,_=y("join",",");u({target:"Array",proto:!0,forced:x||!_},{join:function join(n){return w.call(g(this),void 0===n?",":n)}})},function(n,i){n.exports="undefined"!=typeof ArrayBuffer&&"undefined"!=typeof DataView},function(n,i,o){var u=o(25);n.exports=function(n,i,o){for(var p in i)u(n,p,i[p],o);return n}},function(n,i,o){var u=o(35),p=o(8);n.exports=function(n){if(void 0===n)return 0;var i=u(n),o=p(i);if(i!==o)throw RangeError("Wrong length or index");return o}},function(n,i,o){var u=o(13),p=o(16).f,g=Function.prototype,y=g.toString,w=/^\s*function ([^ (]*)/;u&&!("name"in g)&&p(g,"name",{configurable:!0,get:function(){try{return y.call(this).match(w)[1]}catch(n){return""}}})},function(n,i,o){"use strict";var u=o(13),p=o(3),g=o(145),y=o(25),w=o(15),x=o(41),_=o(226),k=o(55),P=o(4),E=o(58),O=o(57).f,B=o(54).f,I=o(16).f,D=o(227).trim,R=p.Number,N=R.prototype,U="Number"==x(E(N)),toNumber=function(n){var i,o,u,p,g,y,w,x,_=k(n,!1);if("string"==typeof _&&_.length>2)if(43===(i=(_=D(_)).charCodeAt(0))||45===i){if(88===(o=_.charCodeAt(2))||120===o)return NaN}else if(48===i){switch(_.charCodeAt(1)){case 66:case 98:u=2,p=49;break;case 79:case 111:u=8,p=55;break;default:return+_}for(y=(g=_.slice(2)).length,w=0;w<y;w++)if((x=g.charCodeAt(w))<48||x>p)return NaN;return parseInt(g,u)}return+_};if(g("Number",!R(" 0o1")||!R("0b1")||R("+0x1"))){for(var W,G=function Number(n){var i=arguments.length<1?0:n,o=this;return o instanceof G&&(U?P((function(){N.valueOf.call(o)})):"Number"!=x(o))?_(new R(toNumber(i)),o,G):toNumber(i)},j=u?O(R):"MAX_VALUE,MIN_VALUE,NaN,NEGATIVE_INFINITY,POSITIVE_INFINITY,EPSILON,isFinite,isInteger,isNaN,isSafeInteger,MAX_SAFE_INTEGER,MIN_SAFE_INTEGER,parseFloat,parseInt,isInteger,fromString,range".split(","),X=0;j.length>X;X++)w(R,W=j[X])&&!w(G,W)&&I(G,W,B(R,W));G.prototype=N,N.constructor=G,y(p,"Number",G)}},function(n,i,o){var u=o(14),p=o(45);n.exports=function(n,i,o){var g,y;return p&&"function"==typeof(g=i.constructor)&&g!==o&&u(y=g.prototype)&&y!==o.prototype&&p(n,y),n}},function(n,i,o){var u=o(33),p="["+o(228)+"]",g=RegExp("^"+p+p+"*"),y=RegExp(p+p+"*$"),createMethod=function(n){return function(i){var o=String(u(i));return 1&n&&(o=o.replace(g,"")),2&n&&(o=o.replace(y,"")),o}};n.exports={start:createMethod(1),end:createMethod(2),trim:createMethod(3)}},function(n,i){n.exports="\t\n\v\f\r                　\u2028\u2029\ufeff"},function(n,i,o){var u=o(1),p=o(19),g=o(94);u({target:"Object",stat:!0,forced:o(4)((function(){g(1)}))},{keys:function keys(n){return g(p(n))}})},function(n,i,o){"use strict";var u=o(11);n.exports=function(){var n=u(this),i="";return n.global&&(i+="g"),n.ignoreCase&&(i+="i"),n.multiline&&(i+="m"),n.dotAll&&(i+="s"),n.unicode&&(i+="u"),n.sticky&&(i+="y"),i}},function(n,i,o){"use strict";var u=o(162),p=o(11),g=o(8),y=o(35),w=o(33),x=o(163),_=o(337),k=o(164),P=Math.max,E=Math.min;u("replace",2,(function(n,i,o,u){var O=u.REGEXP_REPLACE_SUBSTITUTES_UNDEFINED_CAPTURE,B=u.REPLACE_KEEPS_$0,I=O?"$":"$0";return[function replace(o,u){var p=w(this),g=null==o?void 0:o[n];return void 0!==g?g.call(o,p,u):i.call(String(p),o,u)},function(n,u){if(!O&&B||"string"==typeof u&&-1===u.indexOf(I)){var w=o(i,n,this,u);if(w.done)return w.value}var D=p(n),R=String(this),N="function"==typeof u;N||(u=String(u));var U=D.global;if(U){var W=D.unicode;D.lastIndex=0}for(var G=[];;){var j=k(D,R);if(null===j)break;if(G.push(j),!U)break;""===String(j[0])&&(D.lastIndex=x(R,g(D.lastIndex),W))}for(var X,Y="",K=0,J=0;J<G.length;J++){j=G[J];for(var $=String(j[0]),tt=P(E(y(j.index),R.length),0),et=[],rt=1;rt<j.length;rt++)et.push(void 0===(X=j[rt])?X:String(X));var nt=j.groups;if(N){var it=[$].concat(et,tt,R);void 0!==nt&&it.push(nt);var ot=String(u.apply(void 0,it))}else ot=_($,R,tt,et,nt,u);tt>=K&&(Y+=R.slice(K,tt)+ot,K=tt+$.length)}return Y+R.slice(K)}]}))},function(n,i,o){"use strict";var u=o(162),p=o(338),g=o(11),y=o(33),w=o(36),x=o(163),_=o(8),k=o(164),P=o(119),E=o(4),O=[].push,B=Math.min,I=!E((function(){return!RegExp(4294967295,"y")}));u("split",2,(function(n,i,o){var u;return u="c"=="abbc".split(/(b)*/)[1]||4!="test".split(/(?:)/,-1).length||2!="ab".split(/(?:ab)*/).length||4!=".".split(/(.?)(.?)/).length||".".split(/()()/).length>1||"".split(/.?/).length?function(n,o){var u=String(y(this)),g=void 0===o?4294967295:o>>>0;if(0===g)return[];if(void 0===n)return[u];if(!p(n))return i.call(u,n,g);for(var w,x,_,k=[],E=(n.ignoreCase?"i":"")+(n.multiline?"m":"")+(n.unicode?"u":"")+(n.sticky?"y":""),B=0,I=new RegExp(n.source,E+"g");(w=P.call(I,u))&&!((x=I.lastIndex)>B&&(k.push(u.slice(B,w.index)),w.length>1&&w.index<u.length&&O.apply(k,w.slice(1)),_=w[0].length,B=x,k.length>=g));)I.lastIndex===w.index&&I.lastIndex++;return B===u.length?!_&&I.test("")||k.push(""):k.push(u.slice(B)),k.length>g?k.slice(0,g):k}:"0".split(void 0,0).length?function(n,o){return void 0===n&&0===o?[]:i.call(this,n,o)}:i,[function split(i,o){var p=y(this),g=null==i?void 0:i[n];return void 0!==g?g.call(i,p,o):u.call(String(p),i,o)},function(n,p){var y=o(u,n,this,p,u!==i);if(y.done)return y.value;var P=g(n),E=String(this),O=w(P,RegExp),D=P.unicode,R=(P.ignoreCase?"i":"")+(P.multiline?"m":"")+(P.unicode?"u":"")+(I?"y":"g"),N=new O(I?P:"^(?:"+P.source+")",R),U=void 0===p?4294967295:p>>>0;if(0===U)return[];if(0===E.length)return null===k(N,E)?[E]:[];for(var W=0,G=0,j=[];G<E.length;){N.lastIndex=I?G:0;var X,Y=k(N,I?E:E.slice(G));if(null===Y||(X=B(_(N.lastIndex+(I?0:G)),E.length))===W)G=x(E,G,D);else{if(j.push(E.slice(W,G)),j.length===U)return j;for(var K=1;K<=Y.length-1;K++)if(j.push(Y[K]),j.length===U)return j;G=W=X}}return j.push(E.slice(W)),j}]}),!I)},function(n,i,o){"use strict";var u=o(1),p=o(227).trim;u({target:"String",proto:!0,forced:o(339)("trim")},{trim:function trim(){return p(this)}})},function(n,i,o){"use strict";var u=o(1),p=o(235);u({target:"String",proto:!0,forced:o(236)("link")},{link:function link(n){return p(this,"a","href",n)}})},function(n,i,o){var u=o(33),p=/"/g;n.exports=function(n,i,o,g){var y=String(u(n)),w="<"+i;return""!==o&&(w+=" "+o+'="'+String(g).replace(p,"&quot;")+'"'),w+">"+y+"</"+i+">"}},function(n,i,o){var u=o(4);n.exports=function(n){return u((function(){var i=""[n]('"');return i!==i.toLowerCase()||i.split('"').length>3}))}},function(n,i,o){"use strict";var u=o(1),p=o(3),g=o(13),y=o(341),w=o(5),x=o(156),_=o(157),k=o(40),P=o(18),E=o(8),O=o(223),B=o(238),I=o(55),D=o(15),R=o(117),N=o(14),U=o(58),W=o(45),G=o(57).f,j=o(343),X=o(20).forEach,Y=o(158),K=o(16),J=o(54),$=o(42),tt=o(226),et=$.get,rt=$.set,nt=K.f,it=J.f,ot=Math.round,at=p.RangeError,st=x.ArrayBuffer,lt=x.DataView,ct=w.NATIVE_ARRAY_BUFFER_VIEWS,ut=w.TYPED_ARRAY_TAG,ft=w.TypedArray,ht=w.TypedArrayPrototype,dt=w.aTypedArrayConstructor,pt=w.isTypedArray,fromList=function(n,i){for(var o=0,u=i.length,p=new(dt(n))(u);u>o;)p[o]=i[o++];return p},addGetter=function(n,i){nt(n,i,{get:function(){return et(this)[i]}})},isArrayBuffer=function(n){var i;return n instanceof st||"ArrayBuffer"==(i=R(n))||"SharedArrayBuffer"==i},isTypedArrayIndex=function(n,i){return pt(n)&&"symbol"!=typeof i&&i in n&&String(+i)==String(i)},gt=function getOwnPropertyDescriptor(n,i){return isTypedArrayIndex(n,i=I(i,!0))?k(2,n[i]):it(n,i)},yt=function defineProperty(n,i,o){return!(isTypedArrayIndex(n,i=I(i,!0))&&N(o)&&D(o,"value"))||D(o,"get")||D(o,"set")||o.configurable||D(o,"writable")&&!o.writable||D(o,"enumerable")&&!o.enumerable?nt(n,i,o):(n[i]=o.value,n)};g?(ct||(J.f=gt,K.f=yt,addGetter(ht,"buffer"),addGetter(ht,"byteOffset"),addGetter(ht,"byteLength"),addGetter(ht,"length")),u({target:"Object",stat:!0,forced:!ct},{getOwnPropertyDescriptor:gt,defineProperty:yt}),n.exports=function(n,i,o){var g=n.match(/\d+$/)[0]/8,w=n+(o?"Clamped":"")+"Array",x="get"+n,k="set"+n,I=p[w],D=I,R=D&&D.prototype,K={},addElement=function(n,i){nt(n,i,{get:function(){return function(n,i){var o=et(n);return o.view[x](i*g+o.byteOffset,!0)}(this,i)},set:function(n){return function(n,i,u){var p=et(n);o&&(u=(u=ot(u))<0?0:u>255?255:255&u),p.view[k](i*g+p.byteOffset,u,!0)}(this,i,n)},enumerable:!0})};ct?y&&(D=i((function(n,i,o,u){return _(n,D,w),tt(N(i)?isArrayBuffer(i)?void 0!==u?new I(i,B(o,g),u):void 0!==o?new I(i,B(o,g)):new I(i):pt(i)?fromList(D,i):j.call(D,i):new I(O(i)),n,D)})),W&&W(D,ft),X(G(I),(function(n){n in D||P(D,n,I[n])})),D.prototype=R):(D=i((function(n,i,o,u){_(n,D,w);var p,y,x,k=0,P=0;if(N(i)){if(!isArrayBuffer(i))return pt(i)?fromList(D,i):j.call(D,i);p=i,P=B(o,g);var I=i.byteLength;if(void 0===u){if(I%g)throw at("Wrong length");if((y=I-P)<0)throw at("Wrong length")}else if((y=E(u)*g)+P>I)throw at("Wrong length");x=y/g}else x=O(i),p=new st(y=x*g);for(rt(n,{buffer:p,byteOffset:P,byteLength:y,length:x,view:new lt(p)});k<x;)addElement(n,k++)})),W&&W(D,ft),R=D.prototype=U(ht)),R.constructor!==D&&P(R,"constructor",D),ut&&P(R,ut,w),K[w]=D,u({global:!0,forced:D!=I,sham:!ct},K),"BYTES_PER_ELEMENT"in D||P(D,"BYTES_PER_ELEMENT",g),"BYTES_PER_ELEMENT"in R||P(R,"BYTES_PER_ELEMENT",g),Y(w)}):n.exports=function(){}},function(n,i,o){var u=o(342);n.exports=function(n,i){var o=u(n);if(o%i)throw RangeError("Wrong offset");return o}},function(n,i,o){var u=o(44),p=o(19),g=o(92),y=o(8),createMethod=function(n){return function(i,o,w,x){u(o);var _=p(i),k=g(_),P=y(_.length),E=n?P-1:0,O=n?-1:1;if(w<2)for(;;){if(E in k){x=k[E],E+=O;break}if(E+=O,n?E<0:P<=E)throw TypeError("Reduce of empty array with no initial value")}for(;n?E>=0:P>E;E+=O)E in k&&(x=o(x,k[E],E,_));return x}};n.exports={left:createMethod(!1),right:createMethod(!0)}},function(n,i){n.exports={CSSRuleList:0,CSSStyleDeclaration:0,CSSValueList:0,ClientRectList:0,DOMRectList:0,DOMStringList:0,DOMTokenList:1,DataTransferItemList:0,FileList:0,HTMLAllCollection:0,HTMLCollection:0,HTMLFormElement:0,HTMLSelectElement:0,MediaList:0,MimeTypeArray:0,NamedNodeMap:0,NodeList:1,PaintRequestList:0,Plugin:0,PluginArray:0,SVGLengthList:0,SVGNumberList:0,SVGPathSegList:0,SVGPointList:0,SVGStringList:0,SVGTransformList:0,SourceBufferList:0,StyleSheetList:0,TextTrackCueList:0,TextTrackList:0,TouchList:0}},function(n,i,o){"use strict";(function(i,u){var p=o(123);n.exports=Readable;var g,y=o(200);Readable.ReadableState=ReadableState;o(122).EventEmitter;var EElistenerCount=function(n,i){return n.listeners(i).length},w=o(242),x=o(124).Buffer,_=i.Uint8Array||function(){};var k=Object.create(o(98));k.inherits=o(86);var P=o(347),E=void 0;E=P&&P.debuglog?P.debuglog("stream"):function(){};var O,B=o(348),I=o(243);k.inherits(Readable,w);var D=["error","close","destroy","pause","resume"];function ReadableState(n,i){n=n||{};var u=i instanceof(g=g||o(48));this.objectMode=!!n.objectMode,u&&(this.objectMode=this.objectMode||!!n.readableObjectMode);var p=n.highWaterMark,y=n.readableHighWaterMark,w=this.objectMode?16:16384;this.highWaterMark=p||0===p?p:u&&(y||0===y)?y:w,this.highWaterMark=Math.floor(this.highWaterMark),this.buffer=new B,this.length=0,this.pipes=null,this.pipesCount=0,this.flowing=null,this.ended=!1,this.endEmitted=!1,this.reading=!1,this.sync=!0,this.needReadable=!1,this.emittedReadable=!1,this.readableListening=!1,this.resumeScheduled=!1,this.destroyed=!1,this.defaultEncoding=n.defaultEncoding||"utf8",this.awaitDrain=0,this.readingMore=!1,this.decoder=null,this.encoding=null,n.encoding&&(O||(O=o(125).StringDecoder),this.decoder=new O(n.encoding),this.encoding=n.encoding)}function Readable(n){if(g=g||o(48),!(this instanceof Readable))return new Readable(n);this._readableState=new ReadableState(n,this),this.readable=!0,n&&("function"==typeof n.read&&(this._read=n.read),"function"==typeof n.destroy&&(this._destroy=n.destroy)),w.call(this)}function readableAddChunk(n,i,o,u,p){var g,y=n._readableState;null===i?(y.reading=!1,function onEofChunk(n,i){if(i.ended)return;if(i.decoder){var o=i.decoder.end();o&&o.length&&(i.buffer.push(o),i.length+=i.objectMode?1:o.length)}i.ended=!0,emitReadable(n)}(n,y)):(p||(g=function chunkInvalid(n,i){var o;(function _isUint8Array(n){return x.isBuffer(n)||n instanceof _})(i)||"string"==typeof i||void 0===i||n.objectMode||(o=new TypeError("Invalid non-string/buffer chunk"));return o}(y,i)),g?n.emit("error",g):y.objectMode||i&&i.length>0?("string"==typeof i||y.objectMode||Object.getPrototypeOf(i)===x.prototype||(i=function _uint8ArrayToBuffer(n){return x.from(n)}(i)),u?y.endEmitted?n.emit("error",new Error("stream.unshift() after end event")):addChunk(n,y,i,!0):y.ended?n.emit("error",new Error("stream.push() after EOF")):(y.reading=!1,y.decoder&&!o?(i=y.decoder.write(i),y.objectMode||0!==i.length?addChunk(n,y,i,!1):maybeReadMore(n,y)):addChunk(n,y,i,!1))):u||(y.reading=!1));return function needMoreData(n){return!n.ended&&(n.needReadable||n.length<n.highWaterMark||0===n.length)}(y)}function addChunk(n,i,o,u){i.flowing&&0===i.length&&!i.sync?(n.emit("data",o),n.read(0)):(i.length+=i.objectMode?1:o.length,u?i.buffer.unshift(o):i.buffer.push(o),i.needReadable&&emitReadable(n)),maybeReadMore(n,i)}Object.defineProperty(Readable.prototype,"destroyed",{get:function(){return void 0!==this._readableState&&this._readableState.destroyed},set:function(n){this._readableState&&(this._readableState.destroyed=n)}}),Readable.prototype.destroy=I.destroy,Readable.prototype._undestroy=I.undestroy,Readable.prototype._destroy=function(n,i){this.push(null),i(n)},Readable.prototype.push=function(n,i){var o,u=this._readableState;return u.objectMode?o=!0:"string"==typeof n&&((i=i||u.defaultEncoding)!==u.encoding&&(n=x.from(n,i),i=""),o=!0),readableAddChunk(this,n,i,!1,o)},Readable.prototype.unshift=function(n){return readableAddChunk(this,n,null,!0,!1)},Readable.prototype.isPaused=function(){return!1===this._readableState.flowing},Readable.prototype.setEncoding=function(n){return O||(O=o(125).StringDecoder),this._readableState.decoder=new O(n),this._readableState.encoding=n,this};function howMuchToRead(n,i){return n<=0||0===i.length&&i.ended?0:i.objectMode?1:n!=n?i.flowing&&i.length?i.buffer.head.data.length:i.length:(n>i.highWaterMark&&(i.highWaterMark=function computeNewHighWaterMark(n){return n>=8388608?n=8388608:(n--,n|=n>>>1,n|=n>>>2,n|=n>>>4,n|=n>>>8,n|=n>>>16,n++),n}(n)),n<=i.length?n:i.ended?i.length:(i.needReadable=!0,0))}function emitReadable(n){var i=n._readableState;i.needReadable=!1,i.emittedReadable||(E("emitReadable",i.flowing),i.emittedReadable=!0,i.sync?p.nextTick(emitReadable_,n):emitReadable_(n))}function emitReadable_(n){E("emit readable"),n.emit("readable"),flow(n)}function maybeReadMore(n,i){i.readingMore||(i.readingMore=!0,p.nextTick(maybeReadMore_,n,i))}function maybeReadMore_(n,i){for(var o=i.length;!i.reading&&!i.flowing&&!i.ended&&i.length<i.highWaterMark&&(E("maybeReadMore read 0"),n.read(0),o!==i.length);)o=i.length;i.readingMore=!1}function nReadingNextTick(n){E("readable nexttick read 0"),n.read(0)}function resume_(n,i){i.reading||(E("resume read 0"),n.read(0)),i.resumeScheduled=!1,i.awaitDrain=0,n.emit("resume"),flow(n),i.flowing&&!i.reading&&n.read(0)}function flow(n){var i=n._readableState;for(E("flow",i.flowing);i.flowing&&null!==n.read(););}function fromList(n,i){return 0===i.length?null:(i.objectMode?o=i.buffer.shift():!n||n>=i.length?(o=i.decoder?i.buffer.join(""):1===i.buffer.length?i.buffer.head.data:i.buffer.concat(i.length),i.buffer.clear()):o=function fromListPartial(n,i,o){var u;n<i.head.data.length?(u=i.head.data.slice(0,n),i.head.data=i.head.data.slice(n)):u=n===i.head.data.length?i.shift():o?function copyFromBufferString(n,i){var o=i.head,u=1,p=o.data;n-=p.length;for(;o=o.next;){var g=o.data,y=n>g.length?g.length:n;if(y===g.length?p+=g:p+=g.slice(0,n),0===(n-=y)){y===g.length?(++u,o.next?i.head=o.next:i.head=i.tail=null):(i.head=o,o.data=g.slice(y));break}++u}return i.length-=u,p}(n,i):function copyFromBuffer(n,i){var o=x.allocUnsafe(n),u=i.head,p=1;u.data.copy(o),n-=u.data.length;for(;u=u.next;){var g=u.data,y=n>g.length?g.length:n;if(g.copy(o,o.length-n,0,y),0===(n-=y)){y===g.length?(++p,u.next?i.head=u.next:i.head=i.tail=null):(i.head=u,u.data=g.slice(y));break}++p}return i.length-=p,o}(n,i);return u}(n,i.buffer,i.decoder),o);var o}function endReadable(n){var i=n._readableState;if(i.length>0)throw new Error('"endReadable()" called on non-empty stream');i.endEmitted||(i.ended=!0,p.nextTick(endReadableNT,i,n))}function endReadableNT(n,i){n.endEmitted||0!==n.length||(n.endEmitted=!0,i.readable=!1,i.emit("end"))}function indexOf(n,i){for(var o=0,u=n.length;o<u;o++)if(n[o]===i)return o;return-1}Readable.prototype.read=function(n){E("read",n),n=parseInt(n,10);var i=this._readableState,o=n;if(0!==n&&(i.emittedReadable=!1),0===n&&i.needReadable&&(i.length>=i.highWaterMark||i.ended))return E("read: emitReadable",i.length,i.ended),0===i.length&&i.ended?endReadable(this):emitReadable(this),null;if(0===(n=howMuchToRead(n,i))&&i.ended)return 0===i.length&&endReadable(this),null;var u,p=i.needReadable;return E("need readable",p),(0===i.length||i.length-n<i.highWaterMark)&&E("length less than watermark",p=!0),i.ended||i.reading?E("reading or ended",p=!1):p&&(E("do read"),i.reading=!0,i.sync=!0,0===i.length&&(i.needReadable=!0),this._read(i.highWaterMark),i.sync=!1,i.reading||(n=howMuchToRead(o,i))),null===(u=n>0?fromList(n,i):null)?(i.needReadable=!0,n=0):i.length-=n,0===i.length&&(i.ended||(i.needReadable=!0),o!==n&&i.ended&&endReadable(this)),null!==u&&this.emit("data",u),u},Readable.prototype._read=function(n){this.emit("error",new Error("_read() is not implemented"))},Readable.prototype.pipe=function(n,i){var o=this,g=this._readableState;switch(g.pipesCount){case 0:g.pipes=n;break;case 1:g.pipes=[g.pipes,n];break;default:g.pipes.push(n)}g.pipesCount+=1,E("pipe count=%d opts=%j",g.pipesCount,i);var w=(!i||!1!==i.end)&&n!==u.stdout&&n!==u.stderr?onend:unpipe;function onunpipe(i,u){E("onunpipe"),i===o&&u&&!1===u.hasUnpiped&&(u.hasUnpiped=!0,function cleanup(){E("cleanup"),n.removeListener("close",onclose),n.removeListener("finish",onfinish),n.removeListener("drain",x),n.removeListener("error",onerror),n.removeListener("unpipe",onunpipe),o.removeListener("end",onend),o.removeListener("end",unpipe),o.removeListener("data",ondata),_=!0,!g.awaitDrain||n._writableState&&!n._writableState.needDrain||x()}())}function onend(){E("onend"),n.end()}g.endEmitted?p.nextTick(w):o.once("end",w),n.on("unpipe",onunpipe);var x=function pipeOnDrain(n){return function(){var i=n._readableState;E("pipeOnDrain",i.awaitDrain),i.awaitDrain&&i.awaitDrain--,0===i.awaitDrain&&EElistenerCount(n,"data")&&(i.flowing=!0,flow(n))}}(o);n.on("drain",x);var _=!1;var k=!1;function ondata(i){E("ondata"),k=!1,!1!==n.write(i)||k||((1===g.pipesCount&&g.pipes===n||g.pipesCount>1&&-1!==indexOf(g.pipes,n))&&!_&&(E("false write response, pause",o._readableState.awaitDrain),o._readableState.awaitDrain++,k=!0),o.pause())}function onerror(i){E("onerror",i),unpipe(),n.removeListener("error",onerror),0===EElistenerCount(n,"error")&&n.emit("error",i)}function onclose(){n.removeListener("finish",onfinish),unpipe()}function onfinish(){E("onfinish"),n.removeListener("close",onclose),unpipe()}function unpipe(){E("unpipe"),o.unpipe(n)}return o.on("data",ondata),function prependListener(n,i,o){if("function"==typeof n.prependListener)return n.prependListener(i,o);n._events&&n._events[i]?y(n._events[i])?n._events[i].unshift(o):n._events[i]=[o,n._events[i]]:n.on(i,o)}(n,"error",onerror),n.once("close",onclose),n.once("finish",onfinish),n.emit("pipe",o),g.flowing||(E("pipe resume"),o.resume()),n},Readable.prototype.unpipe=function(n){var i=this._readableState,o={hasUnpiped:!1};if(0===i.pipesCount)return this;if(1===i.pipesCount)return n&&n!==i.pipes||(n||(n=i.pipes),i.pipes=null,i.pipesCount=0,i.flowing=!1,n&&n.emit("unpipe",this,o)),this;if(!n){var u=i.pipes,p=i.pipesCount;i.pipes=null,i.pipesCount=0,i.flowing=!1;for(var g=0;g<p;g++)u[g].emit("unpipe",this,o);return this}var y=indexOf(i.pipes,n);return-1===y||(i.pipes.splice(y,1),i.pipesCount-=1,1===i.pipesCount&&(i.pipes=i.pipes[0]),n.emit("unpipe",this,o)),this},Readable.prototype.on=function(n,i){var o=w.prototype.on.call(this,n,i);if("data"===n)!1!==this._readableState.flowing&&this.resume();else if("readable"===n){var u=this._readableState;u.endEmitted||u.readableListening||(u.readableListening=u.needReadable=!0,u.emittedReadable=!1,u.reading?u.length&&emitReadable(this):p.nextTick(nReadingNextTick,this))}return o},Readable.prototype.addListener=Readable.prototype.on,Readable.prototype.resume=function(){var n=this._readableState;return n.flowing||(E("resume"),n.flowing=!0,function resume(n,i){i.resumeScheduled||(i.resumeScheduled=!0,p.nextTick(resume_,n,i))}(this,n)),this},Readable.prototype.pause=function(){return E("call pause flowing=%j",this._readableState.flowing),!1!==this._readableState.flowing&&(E("pause"),this._readableState.flowing=!1,this.emit("pause")),this},Readable.prototype.wrap=function(n){var i=this,o=this._readableState,u=!1;for(var p in n.on("end",(function(){if(E("wrapped end"),o.decoder&&!o.ended){var n=o.decoder.end();n&&n.length&&i.push(n)}i.push(null)})),n.on("data",(function(p){(E("wrapped data"),o.decoder&&(p=o.decoder.write(p)),o.objectMode&&null==p)||(o.objectMode||p&&p.length)&&(i.push(p)||(u=!0,n.pause()))})),n)void 0===this[p]&&"function"==typeof n[p]&&(this[p]=function(i){return function(){return n[i].apply(n,arguments)}}(p));for(var g=0;g<D.length;g++)n.on(D[g],this.emit.bind(this,D[g]));return this._read=function(i){E("wrapped _read",i),u&&(u=!1,n.resume())},this},Object.defineProperty(Readable.prototype,"readableHighWaterMark",{enumerable:!1,get:function(){return this._readableState.highWaterMark}}),Readable._fromList=fromList}).call(this,o(24),o(47))},function(n,i,o){n.exports=o(122).EventEmitter},function(n,i,o){"use strict";var u=o(123);function emitErrorNT(n,i){n.emit("error",i)}n.exports={destroy:function destroy(n,i){var o=this,p=this._readableState&&this._readableState.destroyed,g=this._writableState&&this._writableState.destroyed;return p||g?(i?i(n):!n||this._writableState&&this._writableState.errorEmitted||u.nextTick(emitErrorNT,this,n),this):(this._readableState&&(this._readableState.destroyed=!0),this._writableState&&(this._writableState.destroyed=!0),this._destroy(n||null,(function(n){!i&&n?(u.nextTick(emitErrorNT,o,n),o._writableState&&(o._writableState.errorEmitted=!0)):i&&i(n)})),this)},undestroy:function undestroy(){this._readableState&&(this._readableState.destroyed=!1,this._readableState.reading=!1,this._readableState.ended=!1,this._readableState.endEmitted=!1),this._writableState&&(this._writableState.destroyed=!1,this._writableState.ended=!1,this._writableState.ending=!1,this._writableState.finished=!1,this._writableState.errorEmitted=!1)}}},function(n,i,o){"use strict";n.exports=Transform;var u=o(48),p=Object.create(o(98));function afterTransform(n,i){var o=this._transformState;o.transforming=!1;var u=o.writecb;if(!u)return this.emit("error",new Error("write callback called multiple times"));o.writechunk=null,o.writecb=null,null!=i&&this.push(i),u(n);var p=this._readableState;p.reading=!1,(p.needReadable||p.length<p.highWaterMark)&&this._read(p.highWaterMark)}function Transform(n){if(!(this instanceof Transform))return new Transform(n);u.call(this,n),this._transformState={afterTransform:afterTransform.bind(this),needTransform:!1,transforming:!1,writecb:null,writechunk:null,writeencoding:null},this._readableState.needReadable=!0,this._readableState.sync=!1,n&&("function"==typeof n.transform&&(this._transform=n.transform),"function"==typeof n.flush&&(this._flush=n.flush)),this.on("prefinish",prefinish)}function prefinish(){var n=this;"function"==typeof this._flush?this._flush((function(i,o){done(n,i,o)})):done(this,null,null)}function done(n,i,o){if(i)return n.emit("error",i);if(null!=o&&n.push(o),n._writableState.length)throw new Error("Calling transform done when ws.length != 0");if(n._transformState.transforming)throw new Error("Calling transform done when still transforming");return n.push(null)}p.inherits=o(86),p.inherits(Transform,u),Transform.prototype.push=function(n,i){return this._transformState.needTransform=!1,u.prototype.push.call(this,n,i)},Transform.prototype._transform=function(n,i,o){throw new Error("_transform() is not implemented")},Transform.prototype._write=function(n,i,o){var u=this._transformState;if(u.writecb=o,u.writechunk=n,u.writeencoding=i,!u.transforming){var p=this._readableState;(u.needTransform||p.needReadable||p.length<p.highWaterMark)&&this._read(p.highWaterMark)}},Transform.prototype._read=function(n){var i=this._transformState;null!==i.writechunk&&i.writecb&&!i.transforming?(i.transforming=!0,this._transform(i.writechunk,i.writeencoding,i.afterTransform)):i.needTransform=!0},Transform.prototype._destroy=function(n,i){var o=this;u.prototype._destroy.call(this,n,(function(n){i(n),o.emit("close")}))}},function(n,i,o){"use strict";(function(n){var u=o(10).Buffer,p=o(121).Transform,g=o(356),y=o(167),w=o(246).ok,x=o(10).kMaxLength,_="Cannot create final Buffer. It would be larger than 0x"+x.toString(16)+" bytes";g.Z_MIN_WINDOWBITS=8,g.Z_MAX_WINDOWBITS=15,g.Z_DEFAULT_WINDOWBITS=15,g.Z_MIN_CHUNK=64,g.Z_MAX_CHUNK=1/0,g.Z_DEFAULT_CHUNK=16384,g.Z_MIN_MEMLEVEL=1,g.Z_MAX_MEMLEVEL=9,g.Z_DEFAULT_MEMLEVEL=8,g.Z_MIN_LEVEL=-1,g.Z_MAX_LEVEL=9,g.Z_DEFAULT_LEVEL=g.Z_DEFAULT_COMPRESSION;for(var k=Object.keys(g),P=0;P<k.length;P++){var E=k[P];E.match(/^Z/)&&Object.defineProperty(i,E,{enumerable:!0,value:g[E],writable:!1})}for(var O={Z_OK:g.Z_OK,Z_STREAM_END:g.Z_STREAM_END,Z_NEED_DICT:g.Z_NEED_DICT,Z_ERRNO:g.Z_ERRNO,Z_STREAM_ERROR:g.Z_STREAM_ERROR,Z_DATA_ERROR:g.Z_DATA_ERROR,Z_MEM_ERROR:g.Z_MEM_ERROR,Z_BUF_ERROR:g.Z_BUF_ERROR,Z_VERSION_ERROR:g.Z_VERSION_ERROR},B=Object.keys(O),I=0;I<B.length;I++){var D=B[I];O[O[D]]=D}function zlibBuffer(n,i,o){var p=[],g=0;function flow(){for(var i;null!==(i=n.read());)p.push(i),g+=i.length;n.once("readable",flow)}function onEnd(){var i,y=null;g>=x?y=new RangeError(_):i=u.concat(p,g),p=[],n.close(),o(y,i)}n.on("error",(function onError(i){n.removeListener("end",onEnd),n.removeListener("readable",flow),o(i)})),n.on("end",onEnd),n.end(i),flow()}function zlibBufferSync(n,i){if("string"==typeof i&&(i=u.from(i)),!u.isBuffer(i))throw new TypeError("Not a string or buffer");var o=n._finishFlushFlag;return n._processChunk(i,o)}function Deflate(n){if(!(this instanceof Deflate))return new Deflate(n);Zlib.call(this,n,g.DEFLATE)}function Inflate(n){if(!(this instanceof Inflate))return new Inflate(n);Zlib.call(this,n,g.INFLATE)}function Gzip(n){if(!(this instanceof Gzip))return new Gzip(n);Zlib.call(this,n,g.GZIP)}function Gunzip(n){if(!(this instanceof Gunzip))return new Gunzip(n);Zlib.call(this,n,g.GUNZIP)}function DeflateRaw(n){if(!(this instanceof DeflateRaw))return new DeflateRaw(n);Zlib.call(this,n,g.DEFLATERAW)}function InflateRaw(n){if(!(this instanceof InflateRaw))return new InflateRaw(n);Zlib.call(this,n,g.INFLATERAW)}function Unzip(n){if(!(this instanceof Unzip))return new Unzip(n);Zlib.call(this,n,g.UNZIP)}function isValidFlushFlag(n){return n===g.Z_NO_FLUSH||n===g.Z_PARTIAL_FLUSH||n===g.Z_SYNC_FLUSH||n===g.Z_FULL_FLUSH||n===g.Z_FINISH||n===g.Z_BLOCK}function Zlib(n,o){var y=this;if(this._opts=n=n||{},this._chunkSize=n.chunkSize||i.Z_DEFAULT_CHUNK,p.call(this,n),n.flush&&!isValidFlushFlag(n.flush))throw new Error("Invalid flush flag: "+n.flush);if(n.finishFlush&&!isValidFlushFlag(n.finishFlush))throw new Error("Invalid flush flag: "+n.finishFlush);if(this._flushFlag=n.flush||g.Z_NO_FLUSH,this._finishFlushFlag=void 0!==n.finishFlush?n.finishFlush:g.Z_FINISH,n.chunkSize&&(n.chunkSize<i.Z_MIN_CHUNK||n.chunkSize>i.Z_MAX_CHUNK))throw new Error("Invalid chunk size: "+n.chunkSize);if(n.windowBits&&(n.windowBits<i.Z_MIN_WINDOWBITS||n.windowBits>i.Z_MAX_WINDOWBITS))throw new Error("Invalid windowBits: "+n.windowBits);if(n.level&&(n.level<i.Z_MIN_LEVEL||n.level>i.Z_MAX_LEVEL))throw new Error("Invalid compression level: "+n.level);if(n.memLevel&&(n.memLevel<i.Z_MIN_MEMLEVEL||n.memLevel>i.Z_MAX_MEMLEVEL))throw new Error("Invalid memLevel: "+n.memLevel);if(n.strategy&&n.strategy!=i.Z_FILTERED&&n.strategy!=i.Z_HUFFMAN_ONLY&&n.strategy!=i.Z_RLE&&n.strategy!=i.Z_FIXED&&n.strategy!=i.Z_DEFAULT_STRATEGY)throw new Error("Invalid strategy: "+n.strategy);if(n.dictionary&&!u.isBuffer(n.dictionary))throw new Error("Invalid dictionary: it should be a Buffer instance");this._handle=new g.Zlib(o);var w=this;this._hadError=!1,this._handle.onerror=function(n,o){_close(w),w._hadError=!0;var u=new Error(n);u.errno=o,u.code=i.codes[o],w.emit("error",u)};var x=i.Z_DEFAULT_COMPRESSION;"number"==typeof n.level&&(x=n.level);var _=i.Z_DEFAULT_STRATEGY;"number"==typeof n.strategy&&(_=n.strategy),this._handle.init(n.windowBits||i.Z_DEFAULT_WINDOWBITS,x,n.memLevel||i.Z_DEFAULT_MEMLEVEL,_,n.dictionary),this._buffer=u.allocUnsafe(this._chunkSize),this._offset=0,this._level=x,this._strategy=_,this.once("end",this.close),Object.defineProperty(this,"_closed",{get:function(){return!y._handle},configurable:!0,enumerable:!0})}function _close(i,o){o&&n.nextTick(o),i._handle&&(i._handle.close(),i._handle=null)}function emitCloseNT(n){n.emit("close")}Object.defineProperty(i,"codes",{enumerable:!0,value:Object.freeze(O),writable:!1}),i.Deflate=Deflate,i.Inflate=Inflate,i.Gzip=Gzip,i.Gunzip=Gunzip,i.DeflateRaw=DeflateRaw,i.InflateRaw=InflateRaw,i.Unzip=Unzip,i.createDeflate=function(n){return new Deflate(n)},i.createInflate=function(n){return new Inflate(n)},i.createDeflateRaw=function(n){return new DeflateRaw(n)},i.createInflateRaw=function(n){return new InflateRaw(n)},i.createGzip=function(n){return new Gzip(n)},i.createGunzip=function(n){return new Gunzip(n)},i.createUnzip=function(n){return new Unzip(n)},i.deflate=function(n,i,o){return"function"==typeof i&&(o=i,i={}),zlibBuffer(new Deflate(i),n,o)},i.deflateSync=function(n,i){return zlibBufferSync(new Deflate(i),n)},i.gzip=function(n,i,o){return"function"==typeof i&&(o=i,i={}),zlibBuffer(new Gzip(i),n,o)},i.gzipSync=function(n,i){return zlibBufferSync(new Gzip(i),n)},i.deflateRaw=function(n,i,o){return"function"==typeof i&&(o=i,i={}),zlibBuffer(new DeflateRaw(i),n,o)},i.deflateRawSync=function(n,i){return zlibBufferSync(new DeflateRaw(i),n)},i.unzip=function(n,i,o){return"function"==typeof i&&(o=i,i={}),zlibBuffer(new Unzip(i),n,o)},i.unzipSync=function(n,i){return zlibBufferSync(new Unzip(i),n)},i.inflate=function(n,i,o){return"function"==typeof i&&(o=i,i={}),zlibBuffer(new Inflate(i),n,o)},i.inflateSync=function(n,i){return zlibBufferSync(new Inflate(i),n)},i.gunzip=function(n,i,o){return"function"==typeof i&&(o=i,i={}),zlibBuffer(new Gunzip(i),n,o)},i.gunzipSync=function(n,i){return zlibBufferSync(new Gunzip(i),n)},i.inflateRaw=function(n,i,o){return"function"==typeof i&&(o=i,i={}),zlibBuffer(new InflateRaw(i),n,o)},i.inflateRawSync=function(n,i){return zlibBufferSync(new InflateRaw(i),n)},y.inherits(Zlib,p),Zlib.prototype.params=function(o,u,p){if(o<i.Z_MIN_LEVEL||o>i.Z_MAX_LEVEL)throw new RangeError("Invalid compression level: "+o);if(u!=i.Z_FILTERED&&u!=i.Z_HUFFMAN_ONLY&&u!=i.Z_RLE&&u!=i.Z_FIXED&&u!=i.Z_DEFAULT_STRATEGY)throw new TypeError("Invalid strategy: "+u);if(this._level!==o||this._strategy!==u){var y=this;this.flush(g.Z_SYNC_FLUSH,(function(){w(y._handle,"zlib binding closed"),y._handle.params(o,u),y._hadError||(y._level=o,y._strategy=u,p&&p())}))}else n.nextTick(p)},Zlib.prototype.reset=function(){return w(this._handle,"zlib binding closed"),this._handle.reset()},Zlib.prototype._flush=function(n){this._transform(u.alloc(0),"",n)},Zlib.prototype.flush=function(i,o){var p=this,y=this._writableState;("function"==typeof i||void 0===i&&!o)&&(o=i,i=g.Z_FULL_FLUSH),y.ended?o&&n.nextTick(o):y.ending?o&&this.once("end",o):y.needDrain?o&&this.once("drain",(function(){return p.flush(i,o)})):(this._flushFlag=i,this.write(u.alloc(0),"",o))},Zlib.prototype.close=function(i){_close(this,i),n.nextTick(emitCloseNT,this)},Zlib.prototype._transform=function(n,i,o){var p,y=this._writableState,w=(y.ending||y.ended)&&(!n||y.length===n.length);return null===n||u.isBuffer(n)?this._handle?(w?p=this._finishFlushFlag:(p=this._flushFlag,n.length>=y.length&&(this._flushFlag=this._opts.flush||g.Z_NO_FLUSH)),void this._processChunk(n,p,o)):o(new Error("zlib binding closed")):o(new Error("invalid input"))},Zlib.prototype._processChunk=function(n,i,o){var p=n&&n.length,g=this._chunkSize-this._offset,y=0,k=this,P="function"==typeof o;if(!P){var E,O=[],B=0;this.on("error",(function(n){E=n})),w(this._handle,"zlib binding closed");do{var I=this._handle.writeSync(i,n,y,p,this._buffer,this._offset,g)}while(!this._hadError&&callback(I[0],I[1]));if(this._hadError)throw E;if(B>=x)throw _close(this),new RangeError(_);var D=u.concat(O,B);return _close(this),D}w(this._handle,"zlib binding closed");var R=this._handle.write(i,n,y,p,this._buffer,this._offset,g);function callback(x,_){if(this&&(this.buffer=null,this.callback=null),!k._hadError){var E=g-_;if(w(E>=0,"have should not go down"),E>0){var I=k._buffer.slice(k._offset,k._offset+E);k._offset+=E,P?k.push(I):(O.push(I),B+=I.length)}if((0===_||k._offset>=k._chunkSize)&&(g=k._chunkSize,k._offset=0,k._buffer=u.allocUnsafe(k._chunkSize)),0===_){if(y+=p-x,p=x,!P)return!0;var D=k._handle.write(i,n,y,p,k._buffer,k._offset,k._chunkSize);return D.callback=callback,void(D.buffer=n)}if(!P)return!1;o()}}R.buffer=n,R.callback=callback},y.inherits(Deflate,Zlib),y.inherits(Inflate,Zlib),y.inherits(Gzip,Zlib),y.inherits(Gunzip,Zlib),y.inherits(DeflateRaw,Zlib),y.inherits(InflateRaw,Zlib),y.inherits(Unzip,Zlib)}).call(this,o(47))},function(n,i,o){"use strict";(function(i){var u=o(357);function compare(n,i){if(n===i)return 0;for(var o=n.length,u=i.length,p=0,g=Math.min(o,u);p<g;++p)if(n[p]!==i[p]){o=n[p],u=i[p];break}return o<u?-1:u<o?1:0}function isBuffer(n){return i.Buffer&&"function"==typeof i.Buffer.isBuffer?i.Buffer.isBuffer(n):!(null==n||!n._isBuffer)}var p=o(167),g=Object.prototype.hasOwnProperty,y=Array.prototype.slice,w="foo"===function foo(){}.name;function pToString(n){return Object.prototype.toString.call(n)}function isView(n){return!isBuffer(n)&&("function"==typeof i.ArrayBuffer&&("function"==typeof ArrayBuffer.isView?ArrayBuffer.isView(n):!!n&&(n instanceof DataView||!!(n.buffer&&n.buffer instanceof ArrayBuffer))))}var x=n.exports=ok,_=/\s*function\s+([^\(\s]*)\s*/;function getName(n){if(p.isFunction(n)){if(w)return n.name;var i=n.toString().match(_);return i&&i[1]}}function truncate(n,i){return"string"==typeof n?n.length<i?n:n.slice(0,i):n}function inspect(n){if(w||!p.isFunction(n))return p.inspect(n);var i=getName(n);return"[Function"+(i?": "+i:"")+"]"}function fail(n,i,o,u,p){throw new x.AssertionError({message:o,actual:n,expected:i,operator:u,stackStartFunction:p})}function ok(n,i){n||fail(n,!0,i,"==",x.ok)}function _deepEqual(n,i,o,u){if(n===i)return!0;if(isBuffer(n)&&isBuffer(i))return 0===compare(n,i);if(p.isDate(n)&&p.isDate(i))return n.getTime()===i.getTime();if(p.isRegExp(n)&&p.isRegExp(i))return n.source===i.source&&n.global===i.global&&n.multiline===i.multiline&&n.lastIndex===i.lastIndex&&n.ignoreCase===i.ignoreCase;if(null!==n&&"object"==typeof n||null!==i&&"object"==typeof i){if(isView(n)&&isView(i)&&pToString(n)===pToString(i)&&!(n instanceof Float32Array||n instanceof Float64Array))return 0===compare(new Uint8Array(n.buffer),new Uint8Array(i.buffer));if(isBuffer(n)!==isBuffer(i))return!1;var g=(u=u||{actual:[],expected:[]}).actual.indexOf(n);return-1!==g&&g===u.expected.indexOf(i)||(u.actual.push(n),u.expected.push(i),function objEquiv(n,i,o,u){if(null==n||null==i)return!1;if(p.isPrimitive(n)||p.isPrimitive(i))return n===i;if(o&&Object.getPrototypeOf(n)!==Object.getPrototypeOf(i))return!1;var g=isArguments(n),w=isArguments(i);if(g&&!w||!g&&w)return!1;if(g)return n=y.call(n),i=y.call(i),_deepEqual(n,i,o);var x,_,P=k(n),E=k(i);if(P.length!==E.length)return!1;for(P.sort(),E.sort(),_=P.length-1;_>=0;_--)if(P[_]!==E[_])return!1;for(_=P.length-1;_>=0;_--)if(x=P[_],!_deepEqual(n[x],i[x],o,u))return!1;return!0}(n,i,o,u))}return o?n===i:n==i}function isArguments(n){return"[object Arguments]"==Object.prototype.toString.call(n)}function expectedException(n,i){if(!n||!i)return!1;if("[object RegExp]"==Object.prototype.toString.call(i))return i.test(n);try{if(n instanceof i)return!0}catch(n){}return!Error.isPrototypeOf(i)&&!0===i.call({},n)}function _throws(n,i,o,u){var g;if("function"!=typeof i)throw new TypeError('"block" argument must be a function');"string"==typeof o&&(u=o,o=null),g=function _tryBlock(n){var i;try{n()}catch(n){i=n}return i}(i),u=(o&&o.name?" ("+o.name+").":".")+(u?" "+u:"."),n&&!g&&fail(g,o,"Missing expected exception"+u);var y="string"==typeof u,w=!n&&g&&!o;if((!n&&p.isError(g)&&y&&expectedException(g,o)||w)&&fail(g,o,"Got unwanted exception"+u),n&&g&&o&&!expectedException(g,o)||!n&&g)throw g}x.AssertionError=function AssertionError(n){this.name="AssertionError",this.actual=n.actual,this.expected=n.expected,this.operator=n.operator,n.message?(this.message=n.message,this.generatedMessage=!1):(this.message=function getMessage(n){return truncate(inspect(n.actual),128)+" "+n.operator+" "+truncate(inspect(n.expected),128)}(this),this.generatedMessage=!0);var i=n.stackStartFunction||fail;if(Error.captureStackTrace)Error.captureStackTrace(this,i);else{var o=new Error;if(o.stack){var u=o.stack,p=getName(i),g=u.indexOf("\n"+p);if(g>=0){var y=u.indexOf("\n",g+1);u=u.substring(y+1)}this.stack=u}}},p.inherits(x.AssertionError,Error),x.fail=fail,x.ok=ok,x.equal=function equal(n,i,o){n!=i&&fail(n,i,o,"==",x.equal)},x.notEqual=function notEqual(n,i,o){n==i&&fail(n,i,o,"!=",x.notEqual)},x.deepEqual=function deepEqual(n,i,o){_deepEqual(n,i,!1)||fail(n,i,o,"deepEqual",x.deepEqual)},x.deepStrictEqual=function deepStrictEqual(n,i,o){_deepEqual(n,i,!0)||fail(n,i,o,"deepStrictEqual",x.deepStrictEqual)},x.notDeepEqual=function notDeepEqual(n,i,o){_deepEqual(n,i,!1)&&fail(n,i,o,"notDeepEqual",x.notDeepEqual)},x.notDeepStrictEqual=function notDeepStrictEqual(n,i,o){_deepEqual(n,i,!0)&&fail(n,i,o,"notDeepStrictEqual",notDeepStrictEqual)},x.strictEqual=function strictEqual(n,i,o){n!==i&&fail(n,i,o,"===",x.strictEqual)},x.notStrictEqual=function notStrictEqual(n,i,o){n===i&&fail(n,i,o,"!==",x.notStrictEqual)},x.throws=function(n,i,o){_throws(!0,n,i,o)},x.doesNotThrow=function(n,i,o){_throws(!1,n,i,o)},x.ifError=function(n){if(n)throw n},x.strict=u((function strict(n,i){n||fail(n,!0,i,"==",strict)}),x,{equal:x.strictEqual,deepEqual:x.deepStrictEqual,notEqual:x.notStrictEqual,notDeepEqual:x.notDeepStrictEqual}),x.strict.strict=x.strict;var k=Object.keys||function(n){var i=[];for(var o in n)g.call(n,o)&&i.push(o);return i}}).call(this,o(24))},function(n,i,o){"use strict";n.exports=function adler32(n,i,o,u){for(var p=65535&n|0,g=n>>>16&65535|0,y=0;0!==o;){o-=y=o>2e3?2e3:o;do{g=g+(p=p+i[u++]|0)|0}while(--y);p%=65521,g%=65521}return p|g<<16|0}},function(n,i,o){"use strict";var u=function makeTable(){for(var n,i=[],o=0;o<256;o++){n=o;for(var u=0;u<8;u++)n=1&n?3988292384^n>>>1:n>>>1;i[o]=n}return i}();n.exports=function crc32(n,i,o,p){var g=u,y=p+o;n^=-1;for(var w=p;w<y;w++)n=n>>>8^g[255&(n^i[w])];return-1^n}},function(n,i,o){var u;n.exports=(u=o(2),function(n){var i=u,o=i.lib,p=o.WordArray,g=o.Hasher,y=i.algo,w=[],x=[];!function(){function isPrime(i){for(var o=n.sqrt(i),u=2;u<=o;u++)if(!(i%u))return!1;return!0}function getFractionalBits(n){return 4294967296*(n-(0|n))|0}for(var i=2,o=0;o<64;)isPrime(i)&&(o<8&&(w[o]=getFractionalBits(n.pow(i,.5))),x[o]=getFractionalBits(n.pow(i,1/3)),o++),i++}();var _=[],k=y.SHA256=g.extend({_doReset:function(){this._hash=new p.init(w.slice(0))},_doProcessBlock:function(n,i){for(var o=this._hash.words,u=o[0],p=o[1],g=o[2],y=o[3],w=o[4],k=o[5],P=o[6],E=o[7],O=0;O<64;O++){if(O<16)_[O]=0|n[i+O];else{var B=_[O-15],I=(B<<25|B>>>7)^(B<<14|B>>>18)^B>>>3,D=_[O-2],R=(D<<15|D>>>17)^(D<<13|D>>>19)^D>>>10;_[O]=I+_[O-7]+R+_[O-16]}var N=u&p^u&g^p&g,U=(u<<30|u>>>2)^(u<<19|u>>>13)^(u<<10|u>>>22),W=E+((w<<26|w>>>6)^(w<<21|w>>>11)^(w<<7|w>>>25))+(w&k^~w&P)+x[O]+_[O];E=P,P=k,k=w,w=y+W|0,y=g,g=p,p=u,u=W+(U+N)|0}o[0]=o[0]+u|0,o[1]=o[1]+p|0,o[2]=o[2]+g|0,o[3]=o[3]+y|0,o[4]=o[4]+w|0,o[5]=o[5]+k|0,o[6]=o[6]+P|0,o[7]=o[7]+E|0},_doFinalize:function(){var i=this._data,o=i.words,u=8*this._nDataBytes,p=8*i.sigBytes;return o[p>>>5]|=128<<24-p%32,o[14+(p+64>>>9<<4)]=n.floor(u/4294967296),o[15+(p+64>>>9<<4)]=u,i.sigBytes=4*o.length,this._process(),this._hash},clone:function(){var n=g.clone.call(this);return n._hash=this._hash.clone(),n}});i.SHA256=g._createHelper(k),i.HmacSHA256=g._createHmacHelper(k)}(Math),u.SHA256)},function(n,i,o){var u;n.exports=(u=o(2),o(127),function(){var n=u,i=n.lib.Hasher,o=n.x64,p=o.Word,g=o.WordArray,y=n.algo;function X64Word_create(){return p.create.apply(p,arguments)}var w=[X64Word_create(1116352408,3609767458),X64Word_create(1899447441,602891725),X64Word_create(3049323471,3964484399),X64Word_create(3921009573,2173295548),X64Word_create(961987163,4081628472),X64Word_create(1508970993,3053834265),X64Word_create(2453635748,2937671579),X64Word_create(2870763221,3664609560),X64Word_create(3624381080,2734883394),X64Word_create(310598401,1164996542),X64Word_create(607225278,1323610764),X64Word_create(1426881987,3590304994),X64Word_create(1925078388,4068182383),X64Word_create(2162078206,991336113),X64Word_create(2614888103,633803317),X64Word_create(3248222580,3479774868),X64Word_create(3835390401,2666613458),X64Word_create(4022224774,944711139),X64Word_create(264347078,2341262773),X64Word_create(604807628,2007800933),X64Word_create(770255983,1495990901),X64Word_create(1249150122,1856431235),X64Word_create(1555081692,3175218132),X64Word_create(1996064986,2198950837),X64Word_create(2554220882,3999719339),X64Word_create(2821834349,766784016),X64Word_create(2952996808,2566594879),X64Word_create(3210313671,3203337956),X64Word_create(3336571891,1034457026),X64Word_create(3584528711,2466948901),X64Word_create(113926993,3758326383),X64Word_create(338241895,168717936),X64Word_create(666307205,1188179964),X64Word_create(773529912,1546045734),X64Word_create(1294757372,1522805485),X64Word_create(1396182291,2643833823),X64Word_create(1695183700,2343527390),X64Word_create(1986661051,1014477480),X64Word_create(2177026350,1206759142),X64Word_create(2456956037,344077627),X64Word_create(2730485921,1290863460),X64Word_create(2820302411,3158454273),X64Word_create(3259730800,3505952657),X64Word_create(3345764771,106217008),X64Word_create(3516065817,3606008344),X64Word_create(3600352804,1432725776),X64Word_create(4094571909,1467031594),X64Word_create(275423344,851169720),X64Word_create(430227734,3100823752),X64Word_create(506948616,1363258195),X64Word_create(659060556,3750685593),X64Word_create(883997877,3785050280),X64Word_create(958139571,3318307427),X64Word_create(1322822218,3812723403),X64Word_create(1537002063,2003034995),X64Word_create(1747873779,3602036899),X64Word_create(1955562222,1575990012),X64Word_create(2024104815,1125592928),X64Word_create(2227730452,2716904306),X64Word_create(2361852424,442776044),X64Word_create(2428436474,593698344),X64Word_create(2756734187,3733110249),X64Word_create(3204031479,2999351573),X64Word_create(3329325298,3815920427),X64Word_create(3391569614,3928383900),X64Word_create(3515267271,566280711),X64Word_create(3940187606,3454069534),X64Word_create(4118630271,4000239992),X64Word_create(116418474,1914138554),X64Word_create(174292421,2731055270),X64Word_create(289380356,3203993006),X64Word_create(460393269,320620315),X64Word_create(685471733,587496836),X64Word_create(852142971,1086792851),X64Word_create(1017036298,365543100),X64Word_create(1126000580,2618297676),X64Word_create(1288033470,3409855158),X64Word_create(1501505948,4234509866),X64Word_create(1607167915,987167468),X64Word_create(1816402316,1246189591)],x=[];!function(){for(var n=0;n<80;n++)x[n]=X64Word_create()}();var _=y.SHA512=i.extend({_doReset:function(){this._hash=new g.init([new p.init(1779033703,4089235720),new p.init(3144134277,2227873595),new p.init(1013904242,4271175723),new p.init(2773480762,1595750129),new p.init(1359893119,2917565137),new p.init(2600822924,725511199),new p.init(528734635,4215389547),new p.init(1541459225,327033209)])},_doProcessBlock:function(n,i){for(var o=this._hash.words,u=o[0],p=o[1],g=o[2],y=o[3],_=o[4],k=o[5],P=o[6],E=o[7],O=u.high,B=u.low,I=p.high,D=p.low,R=g.high,N=g.low,U=y.high,W=y.low,G=_.high,j=_.low,X=k.high,Y=k.low,K=P.high,J=P.low,$=E.high,tt=E.low,et=O,rt=B,nt=I,it=D,ot=R,at=N,st=U,lt=W,ct=G,ut=j,ft=X,ht=Y,dt=K,pt=J,gt=$,yt=tt,vt=0;vt<80;vt++){var mt=x[vt];if(vt<16)var bt=mt.high=0|n[i+2*vt],wt=mt.low=0|n[i+2*vt+1];else{var xt=x[vt-15],_t=xt.high,St=xt.low,Ct=(_t>>>1|St<<31)^(_t>>>8|St<<24)^_t>>>7,kt=(St>>>1|_t<<31)^(St>>>8|_t<<24)^(St>>>7|_t<<25),At=x[vt-2],Pt=At.high,Tt=At.low,Et=(Pt>>>19|Tt<<13)^(Pt<<3|Tt>>>29)^Pt>>>6,Ot=(Tt>>>19|Pt<<13)^(Tt<<3|Pt>>>29)^(Tt>>>6|Pt<<26),Bt=x[vt-7],It=Bt.high,Ft=Bt.low,Lt=x[vt-16],Dt=Lt.high,Mt=Lt.low;bt=(bt=(bt=Ct+It+((wt=kt+Ft)>>>0<kt>>>0?1:0))+Et+((wt+=Ot)>>>0<Ot>>>0?1:0))+Dt+((wt+=Mt)>>>0<Mt>>>0?1:0),mt.high=bt,mt.low=wt}var Rt,Nt=ct&ft^~ct&dt,Ut=ut&ht^~ut&pt,Wt=et&nt^et&ot^nt&ot,zt=rt&it^rt&at^it&at,Gt=(et>>>28|rt<<4)^(et<<30|rt>>>2)^(et<<25|rt>>>7),jt=(rt>>>28|et<<4)^(rt<<30|et>>>2)^(rt<<25|et>>>7),Vt=(ct>>>14|ut<<18)^(ct>>>18|ut<<14)^(ct<<23|ut>>>9),Ht=(ut>>>14|ct<<18)^(ut>>>18|ct<<14)^(ut<<23|ct>>>9),qt=w[vt],Xt=qt.high,Zt=qt.low,Yt=gt+Vt+((Rt=yt+Ht)>>>0<yt>>>0?1:0),Kt=jt+zt;gt=dt,yt=pt,dt=ft,pt=ht,ft=ct,ht=ut,ct=st+(Yt=(Yt=(Yt=Yt+Nt+((Rt+=Ut)>>>0<Ut>>>0?1:0))+Xt+((Rt+=Zt)>>>0<Zt>>>0?1:0))+bt+((Rt+=wt)>>>0<wt>>>0?1:0))+((ut=lt+Rt|0)>>>0<lt>>>0?1:0)|0,st=ot,lt=at,ot=nt,at=it,nt=et,it=rt,et=Yt+(Gt+Wt+(Kt>>>0<jt>>>0?1:0))+((rt=Rt+Kt|0)>>>0<Rt>>>0?1:0)|0}B=u.low=B+rt,u.high=O+et+(B>>>0<rt>>>0?1:0),D=p.low=D+it,p.high=I+nt+(D>>>0<it>>>0?1:0),N=g.low=N+at,g.high=R+ot+(N>>>0<at>>>0?1:0),W=y.low=W+lt,y.high=U+st+(W>>>0<lt>>>0?1:0),j=_.low=j+ut,_.high=G+ct+(j>>>0<ut>>>0?1:0),Y=k.low=Y+ht,k.high=X+ft+(Y>>>0<ht>>>0?1:0),J=P.low=J+pt,P.high=K+dt+(J>>>0<pt>>>0?1:0),tt=E.low=tt+yt,E.high=$+gt+(tt>>>0<yt>>>0?1:0)},_doFinalize:function(){var n=this._data,i=n.words,o=8*this._nDataBytes,u=8*n.sigBytes;return i[u>>>5]|=128<<24-u%32,i[30+(u+128>>>10<<5)]=Math.floor(o/4294967296),i[31+(u+128>>>10<<5)]=o,n.sigBytes=4*i.length,this._process(),this._hash.toX32()},clone:function(){var n=i.clone.call(this);return n._hash=this._hash.clone(),n},blockSize:32});n.SHA512=i._createHelper(_),n.HmacSHA512=i._createHmacHelper(_)}(),u.SHA512)},function(n){n.exports=JSON.parse('[["a140","",62],["a180","",32],["a240","",62],["a280","",32],["a2ab","",5],["a2e3","€"],["a2ef",""],["a2fd",""],["a340","",62],["a380","",31,"　"],["a440","",62],["a480","",32],["a4f4","",10],["a540","",62],["a580","",32],["a5f7","",7],["a640","",62],["a680","",32],["a6b9","",7],["a6d9","",6],["a6ec",""],["a6f3",""],["a6f6","",8],["a740","",62],["a780","",32],["a7c2","",14],["a7f2","",12],["a896","",10],["a8bc","ḿ"],["a8bf","ǹ"],["a8c1",""],["a8ea","",20],["a958",""],["a95b",""],["a95d",""],["a989","〾⿰",11],["a997","",12],["a9f0","",14],["aaa1","",93],["aba1","",93],["aca1","",93],["ada1","",93],["aea1","",93],["afa1","",93],["d7fa","",4],["f8a1","",93],["f9a1","",93],["faa1","",93],["fba1","",93],["fca1","",93],["fda1","",93],["fe50","⺁⺄㑳㑇⺈⺋㖞㘚㘎⺌⺗㥮㤘㧏㧟㩳㧐㭎㱮㳠⺧⺪䁖䅟⺮䌷⺳⺶⺷䎱䎬⺻䏝䓖䙡䙌"],["fe80","䜣䜩䝼䞍⻊䥇䥺䥽䦂䦃䦅䦆䦟䦛䦷䦶䲣䲟䲠䲡䱷䲢䴓",6,"䶮",93],["8135f437",""]]')},function(n){n.exports=JSON.parse('[["0","\\u0000",127],["a140","　，、。．‧；：？！︰…‥﹐﹑﹒·﹔﹕﹖﹗｜–︱—︳╴︴﹏（）︵︶｛｝︷︸〔〕︹︺【】︻︼《》︽︾〈〉︿﹀「」﹁﹂『』﹃﹄﹙﹚"],["a1a1","﹛﹜﹝﹞‘’“”〝〞‵′＃＆＊※§〃○●△▲◎☆★◇◆□■▽▼㊣℅¯￣＿ˍ﹉﹊﹍﹎﹋﹌﹟﹠﹡＋－×÷±√＜＞＝≦≧≠∞≒≡﹢",4,"～∩∪⊥∠∟⊿㏒㏑∫∮∵∴♀♂⊕⊙↑↓←→↖↗↙↘∥∣／"],["a240","＼∕﹨＄￥〒￠￡％＠℃℉﹩﹪﹫㏕㎜㎝㎞㏎㎡㎎㎏㏄°兙兛兞兝兡兣嗧瓩糎▁",7,"▏▎▍▌▋▊▉┼┴┬┤├▔─│▕┌┐└┘╭"],["a2a1","╮╰╯═╞╪╡◢◣◥◤╱╲╳０",9,"Ⅰ",9,"〡",8,"十卄卅Ａ",25,"ａ",21],["a340","ｗｘｙｚΑ",16,"Σ",6,"α",16,"σ",6,"ㄅ",10],["a3a1","ㄐ",25,"˙ˉˊˇˋ"],["a3e1","€"],["a440","一乙丁七乃九了二人儿入八几刀刁力匕十卜又三下丈上丫丸凡久么也乞于亡兀刃勺千叉口土士夕大女子孑孓寸小尢尸山川工己已巳巾干廾弋弓才"],["a4a1","丑丐不中丰丹之尹予云井互五亢仁什仃仆仇仍今介仄元允內六兮公冗凶分切刈勻勾勿化匹午升卅卞厄友及反壬天夫太夭孔少尤尺屯巴幻廿弔引心戈戶手扎支文斗斤方日曰月木欠止歹毋比毛氏水火爪父爻片牙牛犬王丙"],["a540","世丕且丘主乍乏乎以付仔仕他仗代令仙仞充兄冉冊冬凹出凸刊加功包匆北匝仟半卉卡占卯卮去可古右召叮叩叨叼司叵叫另只史叱台句叭叻四囚外"],["a5a1","央失奴奶孕它尼巨巧左市布平幼弁弘弗必戊打扔扒扑斥旦朮本未末札正母民氐永汁汀氾犯玄玉瓜瓦甘生用甩田由甲申疋白皮皿目矛矢石示禾穴立丞丟乒乓乩亙交亦亥仿伉伙伊伕伍伐休伏仲件任仰仳份企伋光兇兆先全"],["a640","共再冰列刑划刎刖劣匈匡匠印危吉吏同吊吐吁吋各向名合吃后吆吒因回囝圳地在圭圬圯圩夙多夷夸妄奸妃好她如妁字存宇守宅安寺尖屹州帆并年"],["a6a1","式弛忙忖戎戌戍成扣扛托收早旨旬旭曲曳有朽朴朱朵次此死氖汝汗汙江池汐汕污汛汍汎灰牟牝百竹米糸缶羊羽老考而耒耳聿肉肋肌臣自至臼舌舛舟艮色艾虫血行衣西阡串亨位住佇佗佞伴佛何估佐佑伽伺伸佃佔似但佣"],["a740","作你伯低伶余佝佈佚兌克免兵冶冷別判利刪刨劫助努劬匣即卵吝吭吞吾否呎吧呆呃吳呈呂君吩告吹吻吸吮吵吶吠吼呀吱含吟听囪困囤囫坊坑址坍"],["a7a1","均坎圾坐坏圻壯夾妝妒妨妞妣妙妖妍妤妓妊妥孝孜孚孛完宋宏尬局屁尿尾岐岑岔岌巫希序庇床廷弄弟彤形彷役忘忌志忍忱快忸忪戒我抄抗抖技扶抉扭把扼找批扳抒扯折扮投抓抑抆改攻攸旱更束李杏材村杜杖杞杉杆杠"],["a840","杓杗步每求汞沙沁沈沉沅沛汪決沐汰沌汨沖沒汽沃汲汾汴沆汶沍沔沘沂灶灼災灸牢牡牠狄狂玖甬甫男甸皂盯矣私秀禿究系罕肖肓肝肘肛肚育良芒"],["a8a1","芋芍見角言谷豆豕貝赤走足身車辛辰迂迆迅迄巡邑邢邪邦那酉釆里防阮阱阪阬並乖乳事些亞享京佯依侍佳使佬供例來侃佰併侈佩佻侖佾侏侑佺兔兒兕兩具其典冽函刻券刷刺到刮制剁劾劻卒協卓卑卦卷卸卹取叔受味呵"],["a940","咖呸咕咀呻呷咄咒咆呼咐呱呶和咚呢周咋命咎固垃坷坪坩坡坦坤坼夜奉奇奈奄奔妾妻委妹妮姑姆姐姍始姓姊妯妳姒姅孟孤季宗定官宜宙宛尚屈居"],["a9a1","屆岷岡岸岩岫岱岳帘帚帖帕帛帑幸庚店府底庖延弦弧弩往征彿彼忝忠忽念忿怏怔怯怵怖怪怕怡性怩怫怛或戕房戾所承拉拌拄抿拂抹拒招披拓拔拋拈抨抽押拐拙拇拍抵拚抱拘拖拗拆抬拎放斧於旺昔易昌昆昂明昀昏昕昊"],["aa40","昇服朋杭枋枕東果杳杷枇枝林杯杰板枉松析杵枚枓杼杪杲欣武歧歿氓氛泣注泳沱泌泥河沽沾沼波沫法泓沸泄油況沮泗泅泱沿治泡泛泊沬泯泜泖泠"],["aaa1","炕炎炒炊炙爬爭爸版牧物狀狎狙狗狐玩玨玟玫玥甽疝疙疚的盂盲直知矽社祀祁秉秈空穹竺糾罔羌羋者肺肥肢肱股肫肩肴肪肯臥臾舍芳芝芙芭芽芟芹花芬芥芯芸芣芰芾芷虎虱初表軋迎返近邵邸邱邶采金長門阜陀阿阻附"],["ab40","陂隹雨青非亟亭亮信侵侯便俠俑俏保促侶俘俟俊俗侮俐俄係俚俎俞侷兗冒冑冠剎剃削前剌剋則勇勉勃勁匍南卻厚叛咬哀咨哎哉咸咦咳哇哂咽咪品"],["aba1","哄哈咯咫咱咻咩咧咿囿垂型垠垣垢城垮垓奕契奏奎奐姜姘姿姣姨娃姥姪姚姦威姻孩宣宦室客宥封屎屏屍屋峙峒巷帝帥帟幽庠度建弈弭彥很待徊律徇後徉怒思怠急怎怨恍恰恨恢恆恃恬恫恪恤扁拜挖按拼拭持拮拽指拱拷"],["ac40","拯括拾拴挑挂政故斫施既春昭映昧是星昨昱昤曷柿染柱柔某柬架枯柵柩柯柄柑枴柚查枸柏柞柳枰柙柢柝柒歪殃殆段毒毗氟泉洋洲洪流津洌洱洞洗"],["aca1","活洽派洶洛泵洹洧洸洩洮洵洎洫炫為炳炬炯炭炸炮炤爰牲牯牴狩狠狡玷珊玻玲珍珀玳甚甭畏界畎畋疫疤疥疢疣癸皆皇皈盈盆盃盅省盹相眉看盾盼眇矜砂研砌砍祆祉祈祇禹禺科秒秋穿突竿竽籽紂紅紀紉紇約紆缸美羿耄"],["ad40","耐耍耑耶胖胥胚胃胄背胡胛胎胞胤胝致舢苧范茅苣苛苦茄若茂茉苒苗英茁苜苔苑苞苓苟苯茆虐虹虻虺衍衫要觔計訂訃貞負赴赳趴軍軌述迦迢迪迥"],["ada1","迭迫迤迨郊郎郁郃酋酊重閂限陋陌降面革韋韭音頁風飛食首香乘亳倌倍倣俯倦倥俸倩倖倆值借倚倒們俺倀倔倨俱倡個候倘俳修倭倪俾倫倉兼冤冥冢凍凌准凋剖剜剔剛剝匪卿原厝叟哨唐唁唷哼哥哲唆哺唔哩哭員唉哮哪"],["ae40","哦唧唇哽唏圃圄埂埔埋埃堉夏套奘奚娑娘娜娟娛娓姬娠娣娩娥娌娉孫屘宰害家宴宮宵容宸射屑展屐峭峽峻峪峨峰島崁峴差席師庫庭座弱徒徑徐恙"],["aea1","恣恥恐恕恭恩息悄悟悚悍悔悌悅悖扇拳挈拿捎挾振捕捂捆捏捉挺捐挽挪挫挨捍捌效敉料旁旅時晉晏晃晒晌晅晁書朔朕朗校核案框桓根桂桔栩梳栗桌桑栽柴桐桀格桃株桅栓栘桁殊殉殷氣氧氨氦氤泰浪涕消涇浦浸海浙涓"],["af40","浬涉浮浚浴浩涌涊浹涅浥涔烊烘烤烙烈烏爹特狼狹狽狸狷玆班琉珮珠珪珞畔畝畜畚留疾病症疲疳疽疼疹痂疸皋皰益盍盎眩真眠眨矩砰砧砸砝破砷"],["afa1","砥砭砠砟砲祕祐祠祟祖神祝祗祚秤秣秧租秦秩秘窄窈站笆笑粉紡紗紋紊素索純紐紕級紜納紙紛缺罟羔翅翁耆耘耕耙耗耽耿胱脂胰脅胭胴脆胸胳脈能脊胼胯臭臬舀舐航舫舨般芻茫荒荔荊茸荐草茵茴荏茲茹茶茗荀茱茨荃"],["b040","虔蚊蚪蚓蚤蚩蚌蚣蚜衰衷袁袂衽衹記訐討訌訕訊託訓訖訏訑豈豺豹財貢起躬軒軔軏辱送逆迷退迺迴逃追逅迸邕郡郝郢酒配酌釘針釗釜釙閃院陣陡"],["b0a1","陛陝除陘陞隻飢馬骨高鬥鬲鬼乾偺偽停假偃偌做偉健偶偎偕偵側偷偏倏偯偭兜冕凰剪副勒務勘動匐匏匙匿區匾參曼商啪啦啄啞啡啃啊唱啖問啕唯啤唸售啜唬啣唳啁啗圈國圉域堅堊堆埠埤基堂堵執培夠奢娶婁婉婦婪婀"],["b140","娼婢婚婆婊孰寇寅寄寂宿密尉專將屠屜屝崇崆崎崛崖崢崑崩崔崙崤崧崗巢常帶帳帷康庸庶庵庾張強彗彬彩彫得徙從徘御徠徜恿患悉悠您惋悴惦悽"],["b1a1","情悻悵惜悼惘惕惆惟悸惚惇戚戛扈掠控捲掖探接捷捧掘措捱掩掉掃掛捫推掄授掙採掬排掏掀捻捩捨捺敝敖救教敗啟敏敘敕敔斜斛斬族旋旌旎晝晚晤晨晦晞曹勗望梁梯梢梓梵桿桶梱梧梗械梃棄梭梆梅梔條梨梟梡梂欲殺"],["b240","毫毬氫涎涼淳淙液淡淌淤添淺清淇淋涯淑涮淞淹涸混淵淅淒渚涵淚淫淘淪深淮淨淆淄涪淬涿淦烹焉焊烽烯爽牽犁猜猛猖猓猙率琅琊球理現琍瓠瓶"],["b2a1","瓷甜產略畦畢異疏痔痕疵痊痍皎盔盒盛眷眾眼眶眸眺硫硃硎祥票祭移窒窕笠笨笛第符笙笞笮粒粗粕絆絃統紮紹紼絀細紳組累終紲紱缽羞羚翌翎習耜聊聆脯脖脣脫脩脰脤舂舵舷舶船莎莞莘荸莢莖莽莫莒莊莓莉莠荷荻荼"],["b340","莆莧處彪蛇蛀蚶蛄蚵蛆蛋蚱蚯蛉術袞袈被袒袖袍袋覓規訪訝訣訥許設訟訛訢豉豚販責貫貨貪貧赧赦趾趺軛軟這逍通逗連速逝逐逕逞造透逢逖逛途"],["b3a1","部郭都酗野釵釦釣釧釭釩閉陪陵陳陸陰陴陶陷陬雀雪雩章竟頂頃魚鳥鹵鹿麥麻傢傍傅備傑傀傖傘傚最凱割剴創剩勞勝勛博厥啻喀喧啼喊喝喘喂喜喪喔喇喋喃喳單喟唾喲喚喻喬喱啾喉喫喙圍堯堪場堤堰報堡堝堠壹壺奠"],["b440","婷媚婿媒媛媧孳孱寒富寓寐尊尋就嵌嵐崴嵇巽幅帽幀幃幾廊廁廂廄弼彭復循徨惑惡悲悶惠愜愣惺愕惰惻惴慨惱愎惶愉愀愒戟扉掣掌描揀揩揉揆揍"],["b4a1","插揣提握揖揭揮捶援揪換摒揚揹敞敦敢散斑斐斯普晰晴晶景暑智晾晷曾替期朝棺棕棠棘棗椅棟棵森棧棹棒棲棣棋棍植椒椎棉棚楮棻款欺欽殘殖殼毯氮氯氬港游湔渡渲湧湊渠渥渣減湛湘渤湖湮渭渦湯渴湍渺測湃渝渾滋"],["b540","溉渙湎湣湄湲湩湟焙焚焦焰無然煮焜牌犄犀猶猥猴猩琺琪琳琢琥琵琶琴琯琛琦琨甥甦畫番痢痛痣痙痘痞痠登發皖皓皴盜睏短硝硬硯稍稈程稅稀窘"],["b5a1","窗窖童竣等策筆筐筒答筍筋筏筑粟粥絞結絨絕紫絮絲絡給絢絰絳善翔翕耋聒肅腕腔腋腑腎脹腆脾腌腓腴舒舜菩萃菸萍菠菅萋菁華菱菴著萊菰萌菌菽菲菊萸萎萄菜萇菔菟虛蛟蛙蛭蛔蛛蛤蛐蛞街裁裂袱覃視註詠評詞証詁"],["b640","詔詛詐詆訴診訶詖象貂貯貼貳貽賁費賀貴買貶貿貸越超趁跎距跋跚跑跌跛跆軻軸軼辜逮逵週逸進逶鄂郵鄉郾酣酥量鈔鈕鈣鈉鈞鈍鈐鈇鈑閔閏開閑"],["b6a1","間閒閎隊階隋陽隅隆隍陲隄雁雅雄集雇雯雲韌項順須飧飪飯飩飲飭馮馭黃黍黑亂傭債傲傳僅傾催傷傻傯僇剿剷剽募勦勤勢勣匯嗟嗨嗓嗦嗎嗜嗇嗑嗣嗤嗯嗚嗡嗅嗆嗥嗉園圓塞塑塘塗塚塔填塌塭塊塢塒塋奧嫁嫉嫌媾媽媼"],["b740","媳嫂媲嵩嵯幌幹廉廈弒彙徬微愚意慈感想愛惹愁愈慎慌慄慍愾愴愧愍愆愷戡戢搓搾搞搪搭搽搬搏搜搔損搶搖搗搆敬斟新暗暉暇暈暖暄暘暍會榔業"],["b7a1","楚楷楠楔極椰概楊楨楫楞楓楹榆楝楣楛歇歲毀殿毓毽溢溯滓溶滂源溝滇滅溥溘溼溺溫滑準溜滄滔溪溧溴煎煙煩煤煉照煜煬煦煌煥煞煆煨煖爺牒猷獅猿猾瑯瑚瑕瑟瑞瑁琿瑙瑛瑜當畸瘀痰瘁痲痱痺痿痴痳盞盟睛睫睦睞督"],["b840","睹睪睬睜睥睨睢矮碎碰碗碘碌碉硼碑碓硿祺祿禁萬禽稜稚稠稔稟稞窟窠筷節筠筮筧粱粳粵經絹綑綁綏絛置罩罪署義羨群聖聘肆肄腱腰腸腥腮腳腫"],["b8a1","腹腺腦舅艇蒂葷落萱葵葦葫葉葬葛萼萵葡董葩葭葆虞虜號蛹蜓蜈蜇蜀蛾蛻蜂蜃蜆蜊衙裟裔裙補裘裝裡裊裕裒覜解詫該詳試詩詰誇詼詣誠話誅詭詢詮詬詹詻訾詨豢貊貉賊資賈賄貲賃賂賅跡跟跨路跳跺跪跤跦躲較載軾輊"],["b940","辟農運遊道遂達逼違遐遇遏過遍遑逾遁鄒鄗酬酪酩釉鈷鉗鈸鈽鉀鈾鉛鉋鉤鉑鈴鉉鉍鉅鈹鈿鉚閘隘隔隕雍雋雉雊雷電雹零靖靴靶預頑頓頊頒頌飼飴"],["b9a1","飽飾馳馱馴髡鳩麂鼎鼓鼠僧僮僥僖僭僚僕像僑僱僎僩兢凳劃劂匱厭嗾嘀嘛嘗嗽嘔嘆嘉嘍嘎嗷嘖嘟嘈嘐嗶團圖塵塾境墓墊塹墅塽壽夥夢夤奪奩嫡嫦嫩嫗嫖嫘嫣孵寞寧寡寥實寨寢寤察對屢嶄嶇幛幣幕幗幔廓廖弊彆彰徹慇"],["ba40","愿態慷慢慣慟慚慘慵截撇摘摔撤摸摟摺摑摧搴摭摻敲斡旗旖暢暨暝榜榨榕槁榮槓構榛榷榻榫榴槐槍榭槌榦槃榣歉歌氳漳演滾漓滴漩漾漠漬漏漂漢"],["baa1","滿滯漆漱漸漲漣漕漫漯澈漪滬漁滲滌滷熔熙煽熊熄熒爾犒犖獄獐瑤瑣瑪瑰瑭甄疑瘧瘍瘋瘉瘓盡監瞄睽睿睡磁碟碧碳碩碣禎福禍種稱窪窩竭端管箕箋筵算箝箔箏箸箇箄粹粽精綻綰綜綽綾綠緊綴網綱綺綢綿綵綸維緒緇綬"],["bb40","罰翠翡翟聞聚肇腐膀膏膈膊腿膂臧臺與舔舞艋蓉蒿蓆蓄蒙蒞蒲蒜蓋蒸蓀蓓蒐蒼蓑蓊蜿蜜蜻蜢蜥蜴蜘蝕蜷蜩裳褂裴裹裸製裨褚裯誦誌語誣認誡誓誤"],["bba1","說誥誨誘誑誚誧豪貍貌賓賑賒赫趙趕跼輔輒輕輓辣遠遘遜遣遙遞遢遝遛鄙鄘鄞酵酸酷酴鉸銀銅銘銖鉻銓銜銨鉼銑閡閨閩閣閥閤隙障際雌雒需靼鞅韶頗領颯颱餃餅餌餉駁骯骰髦魁魂鳴鳶鳳麼鼻齊億儀僻僵價儂儈儉儅凜"],["bc40","劇劈劉劍劊勰厲嘮嘻嘹嘲嘿嘴嘩噓噎噗噴嘶嘯嘰墀墟增墳墜墮墩墦奭嬉嫻嬋嫵嬌嬈寮寬審寫層履嶝嶔幢幟幡廢廚廟廝廣廠彈影德徵慶慧慮慝慕憂"],["bca1","慼慰慫慾憧憐憫憎憬憚憤憔憮戮摩摯摹撞撲撈撐撰撥撓撕撩撒撮播撫撚撬撙撢撳敵敷數暮暫暴暱樣樟槨樁樞標槽模樓樊槳樂樅槭樑歐歎殤毅毆漿潼澄潑潦潔澆潭潛潸潮澎潺潰潤澗潘滕潯潠潟熟熬熱熨牖犛獎獗瑩璋璃"],["bd40","瑾璀畿瘠瘩瘟瘤瘦瘡瘢皚皺盤瞎瞇瞌瞑瞋磋磅確磊碾磕碼磐稿稼穀稽稷稻窯窮箭箱範箴篆篇篁箠篌糊締練緯緻緘緬緝編緣線緞緩綞緙緲緹罵罷羯"],["bda1","翩耦膛膜膝膠膚膘蔗蔽蔚蓮蔬蔭蔓蔑蔣蔡蔔蓬蔥蓿蔆螂蝴蝶蝠蝦蝸蝨蝙蝗蝌蝓衛衝褐複褒褓褕褊誼諒談諄誕請諸課諉諂調誰論諍誶誹諛豌豎豬賠賞賦賤賬賭賢賣賜質賡赭趟趣踫踐踝踢踏踩踟踡踞躺輝輛輟輩輦輪輜輞"],["be40","輥適遮遨遭遷鄰鄭鄧鄱醇醉醋醃鋅銻銷鋪銬鋤鋁銳銼鋒鋇鋰銲閭閱霄霆震霉靠鞍鞋鞏頡頫頜颳養餓餒餘駝駐駟駛駑駕駒駙骷髮髯鬧魅魄魷魯鴆鴉"],["bea1","鴃麩麾黎墨齒儒儘儔儐儕冀冪凝劑劓勳噙噫噹噩噤噸噪器噥噱噯噬噢噶壁墾壇壅奮嬝嬴學寰導彊憲憑憩憊懍憶憾懊懈戰擅擁擋撻撼據擄擇擂操撿擒擔撾整曆曉暹曄曇暸樽樸樺橙橫橘樹橄橢橡橋橇樵機橈歙歷氅濂澱澡"],["bf40","濃澤濁澧澳激澹澶澦澠澴熾燉燐燒燈燕熹燎燙燜燃燄獨璜璣璘璟璞瓢甌甍瘴瘸瘺盧盥瞠瞞瞟瞥磨磚磬磧禦積穎穆穌穋窺篙簑築篤篛篡篩篦糕糖縊"],["bfa1","縑縈縛縣縞縝縉縐罹羲翰翱翮耨膳膩膨臻興艘艙蕊蕙蕈蕨蕩蕃蕉蕭蕪蕞螃螟螞螢融衡褪褲褥褫褡親覦諦諺諫諱謀諜諧諮諾謁謂諷諭諳諶諼豫豭貓賴蹄踱踴蹂踹踵輻輯輸輳辨辦遵遴選遲遼遺鄴醒錠錶鋸錳錯錢鋼錫錄錚"],["c040","錐錦錡錕錮錙閻隧隨險雕霎霑霖霍霓霏靛靜靦鞘頰頸頻頷頭頹頤餐館餞餛餡餚駭駢駱骸骼髻髭鬨鮑鴕鴣鴦鴨鴒鴛默黔龍龜優償儡儲勵嚎嚀嚐嚅嚇"],["c0a1","嚏壕壓壑壎嬰嬪嬤孺尷屨嶼嶺嶽嶸幫彌徽應懂懇懦懋戲戴擎擊擘擠擰擦擬擱擢擭斂斃曙曖檀檔檄檢檜櫛檣橾檗檐檠歜殮毚氈濘濱濟濠濛濤濫濯澀濬濡濩濕濮濰燧營燮燦燥燭燬燴燠爵牆獰獲璩環璦璨癆療癌盪瞳瞪瞰瞬"],["c140","瞧瞭矯磷磺磴磯礁禧禪穗窿簇簍篾篷簌篠糠糜糞糢糟糙糝縮績繆縷縲繃縫總縱繅繁縴縹繈縵縿縯罄翳翼聱聲聰聯聳臆臃膺臂臀膿膽臉膾臨舉艱薪"],["c1a1","薄蕾薜薑薔薯薛薇薨薊虧蟀蟑螳蟒蟆螫螻螺蟈蟋褻褶襄褸褽覬謎謗謙講謊謠謝謄謐豁谿豳賺賽購賸賻趨蹉蹋蹈蹊轄輾轂轅輿避遽還邁邂邀鄹醣醞醜鍍鎂錨鍵鍊鍥鍋錘鍾鍬鍛鍰鍚鍔闊闋闌闈闆隱隸雖霜霞鞠韓顆颶餵騁"],["c240","駿鮮鮫鮪鮭鴻鴿麋黏點黜黝黛鼾齋叢嚕嚮壙壘嬸彝懣戳擴擲擾攆擺擻擷斷曜朦檳檬櫃檻檸櫂檮檯歟歸殯瀉瀋濾瀆濺瀑瀏燻燼燾燸獷獵璧璿甕癖癘"],["c2a1","癒瞽瞿瞻瞼礎禮穡穢穠竄竅簫簧簪簞簣簡糧織繕繞繚繡繒繙罈翹翻職聶臍臏舊藏薩藍藐藉薰薺薹薦蟯蟬蟲蟠覆覲觴謨謹謬謫豐贅蹙蹣蹦蹤蹟蹕軀轉轍邇邃邈醫醬釐鎔鎊鎖鎢鎳鎮鎬鎰鎘鎚鎗闔闖闐闕離雜雙雛雞霤鞣鞦"],["c340","鞭韹額顏題顎顓颺餾餿餽餮馥騎髁鬃鬆魏魎魍鯊鯉鯽鯈鯀鵑鵝鵠黠鼕鼬儳嚥壞壟壢寵龐廬懲懷懶懵攀攏曠曝櫥櫝櫚櫓瀛瀟瀨瀚瀝瀕瀘爆爍牘犢獸"],["c3a1","獺璽瓊瓣疇疆癟癡矇礙禱穫穩簾簿簸簽簷籀繫繭繹繩繪羅繳羶羹羸臘藩藝藪藕藤藥藷蟻蠅蠍蟹蟾襠襟襖襞譁譜識證譚譎譏譆譙贈贊蹼蹲躇蹶蹬蹺蹴轔轎辭邊邋醱醮鏡鏑鏟鏃鏈鏜鏝鏖鏢鏍鏘鏤鏗鏨關隴難霪霧靡韜韻類"],["c440","願顛颼饅饉騖騙鬍鯨鯧鯖鯛鶉鵡鵲鵪鵬麒麗麓麴勸嚨嚷嚶嚴嚼壤孀孃孽寶巉懸懺攘攔攙曦朧櫬瀾瀰瀲爐獻瓏癢癥礦礪礬礫竇競籌籃籍糯糰辮繽繼"],["c4a1","纂罌耀臚艦藻藹蘑藺蘆蘋蘇蘊蠔蠕襤覺觸議譬警譯譟譫贏贍躉躁躅躂醴釋鐘鐃鏽闡霰飄饒饑馨騫騰騷騵鰓鰍鹹麵黨鼯齟齣齡儷儸囁囀囂夔屬巍懼懾攝攜斕曩櫻欄櫺殲灌爛犧瓖瓔癩矓籐纏續羼蘗蘭蘚蠣蠢蠡蠟襪襬覽譴"],["c540","護譽贓躊躍躋轟辯醺鐮鐳鐵鐺鐸鐲鐫闢霸霹露響顧顥饗驅驃驀騾髏魔魑鰭鰥鶯鶴鷂鶸麝黯鼙齜齦齧儼儻囈囊囉孿巔巒彎懿攤權歡灑灘玀瓤疊癮癬"],["c5a1","禳籠籟聾聽臟襲襯觼讀贖贗躑躓轡酈鑄鑑鑒霽霾韃韁顫饕驕驍髒鬚鱉鰱鰾鰻鷓鷗鼴齬齪龔囌巖戀攣攫攪曬欐瓚竊籤籣籥纓纖纔臢蘸蘿蠱變邐邏鑣鑠鑤靨顯饜驚驛驗髓體髑鱔鱗鱖鷥麟黴囑壩攬灞癱癲矗罐羈蠶蠹衢讓讒"],["c640","讖艷贛釀鑪靂靈靄韆顰驟鬢魘鱟鷹鷺鹼鹽鼇齷齲廳欖灣籬籮蠻觀躡釁鑲鑰顱饞髖鬣黌灤矚讚鑷韉驢驥纜讜躪釅鑽鑾鑼鱷鱸黷豔鑿鸚爨驪鬱鸛鸞籲"],["c940","乂乜凵匚厂万丌乇亍囗兀屮彳丏冇与丮亓仂仉仈冘勼卬厹圠夃夬尐巿旡殳毌气爿丱丼仨仜仩仡仝仚刌匜卌圢圣夗夯宁宄尒尻屴屳帄庀庂忉戉扐氕"],["c9a1","氶汃氿氻犮犰玊禸肊阞伎优伬仵伔仱伀价伈伝伂伅伢伓伄仴伒冱刓刉刐劦匢匟卍厊吇囡囟圮圪圴夼妀奼妅奻奾奷奿孖尕尥屼屺屻屾巟幵庄异弚彴忕忔忏扜扞扤扡扦扢扙扠扚扥旯旮朾朹朸朻机朿朼朳氘汆汒汜汏汊汔汋"],["ca40","汌灱牞犴犵玎甪癿穵网艸艼芀艽艿虍襾邙邗邘邛邔阢阤阠阣佖伻佢佉体佤伾佧佒佟佁佘伭伳伿佡冏冹刜刞刡劭劮匉卣卲厎厏吰吷吪呔呅吙吜吥吘"],["caa1","吽呏呁吨吤呇囮囧囥坁坅坌坉坋坒夆奀妦妘妠妗妎妢妐妏妧妡宎宒尨尪岍岏岈岋岉岒岊岆岓岕巠帊帎庋庉庌庈庍弅弝彸彶忒忑忐忭忨忮忳忡忤忣忺忯忷忻怀忴戺抃抌抎抏抔抇扱扻扺扰抁抈扷扽扲扴攷旰旴旳旲旵杅杇"],["cb40","杙杕杌杈杝杍杚杋毐氙氚汸汧汫沄沋沏汱汯汩沚汭沇沕沜汦汳汥汻沎灴灺牣犿犽狃狆狁犺狅玕玗玓玔玒町甹疔疕皁礽耴肕肙肐肒肜芐芏芅芎芑芓"],["cba1","芊芃芄豸迉辿邟邡邥邞邧邠阰阨阯阭丳侘佼侅佽侀侇佶佴侉侄佷佌侗佪侚佹侁佸侐侜侔侞侒侂侕佫佮冞冼冾刵刲刳剆刱劼匊匋匼厒厔咇呿咁咑咂咈呫呺呾呥呬呴呦咍呯呡呠咘呣呧呤囷囹坯坲坭坫坱坰坶垀坵坻坳坴坢"],["cc40","坨坽夌奅妵妺姏姎妲姌姁妶妼姃姖妱妽姀姈妴姇孢孥宓宕屄屇岮岤岠岵岯岨岬岟岣岭岢岪岧岝岥岶岰岦帗帔帙弨弢弣弤彔徂彾彽忞忥怭怦怙怲怋"],["cca1","怴怊怗怳怚怞怬怢怍怐怮怓怑怌怉怜戔戽抭抴拑抾抪抶拊抮抳抯抻抩抰抸攽斨斻昉旼昄昒昈旻昃昋昍昅旽昑昐曶朊枅杬枎枒杶杻枘枆构杴枍枌杺枟枑枙枃杽极杸杹枔欥殀歾毞氝沓泬泫泮泙沶泔沭泧沷泐泂沺泃泆泭泲"],["cd40","泒泝沴沊沝沀泞泀洰泍泇沰泹泏泩泑炔炘炅炓炆炄炑炖炂炚炃牪狖狋狘狉狜狒狔狚狌狑玤玡玭玦玢玠玬玝瓝瓨甿畀甾疌疘皯盳盱盰盵矸矼矹矻矺"],["cda1","矷祂礿秅穸穻竻籵糽耵肏肮肣肸肵肭舠芠苀芫芚芘芛芵芧芮芼芞芺芴芨芡芩苂芤苃芶芢虰虯虭虮豖迒迋迓迍迖迕迗邲邴邯邳邰阹阽阼阺陃俍俅俓侲俉俋俁俔俜俙侻侳俛俇俖侺俀侹俬剄剉勀勂匽卼厗厖厙厘咺咡咭咥哏"],["ce40","哃茍咷咮哖咶哅哆咠呰咼咢咾呲哞咰垵垞垟垤垌垗垝垛垔垘垏垙垥垚垕壴复奓姡姞姮娀姱姝姺姽姼姶姤姲姷姛姩姳姵姠姾姴姭宨屌峐峘峌峗峋峛"],["cea1","峞峚峉峇峊峖峓峔峏峈峆峎峟峸巹帡帢帣帠帤庰庤庢庛庣庥弇弮彖徆怷怹恔恲恞恅恓恇恉恛恌恀恂恟怤恄恘恦恮扂扃拏挍挋拵挎挃拫拹挏挌拸拶挀挓挔拺挕拻拰敁敃斪斿昶昡昲昵昜昦昢昳昫昺昝昴昹昮朏朐柁柲柈枺"],["cf40","柜枻柸柘柀枷柅柫柤柟枵柍枳柷柶柮柣柂枹柎柧柰枲柼柆柭柌枮柦柛柺柉柊柃柪柋欨殂殄殶毖毘毠氠氡洨洴洭洟洼洿洒洊泚洳洄洙洺洚洑洀洝浂"],["cfa1","洁洘洷洃洏浀洇洠洬洈洢洉洐炷炟炾炱炰炡炴炵炩牁牉牊牬牰牳牮狊狤狨狫狟狪狦狣玅珌珂珈珅玹玶玵玴珫玿珇玾珃珆玸珋瓬瓮甮畇畈疧疪癹盄眈眃眄眅眊盷盻盺矧矨砆砑砒砅砐砏砎砉砃砓祊祌祋祅祄秕种秏秖秎窀"],["d040","穾竑笀笁籺籸籹籿粀粁紃紈紁罘羑羍羾耇耎耏耔耷胘胇胠胑胈胂胐胅胣胙胜胊胕胉胏胗胦胍臿舡芔苙苾苹茇苨茀苕茺苫苖苴苬苡苲苵茌苻苶苰苪"],["d0a1","苤苠苺苳苭虷虴虼虳衁衎衧衪衩觓訄訇赲迣迡迮迠郱邽邿郕郅邾郇郋郈釔釓陔陏陑陓陊陎倞倅倇倓倢倰倛俵俴倳倷倬俶俷倗倜倠倧倵倯倱倎党冔冓凊凄凅凈凎剡剚剒剞剟剕剢勍匎厞唦哢唗唒哧哳哤唚哿唄唈哫唑唅哱"],["d140","唊哻哷哸哠唎唃唋圁圂埌堲埕埒垺埆垽垼垸垶垿埇埐垹埁夎奊娙娖娭娮娕娏娗娊娞娳孬宧宭宬尃屖屔峬峿峮峱峷崀峹帩帨庨庮庪庬弳弰彧恝恚恧"],["d1a1","恁悢悈悀悒悁悝悃悕悛悗悇悜悎戙扆拲挐捖挬捄捅挶捃揤挹捋捊挼挩捁挴捘捔捙挭捇挳捚捑挸捗捀捈敊敆旆旃旄旂晊晟晇晑朒朓栟栚桉栲栳栻桋桏栖栱栜栵栫栭栯桎桄栴栝栒栔栦栨栮桍栺栥栠欬欯欭欱欴歭肂殈毦毤"],["d240","毨毣毢毧氥浺浣浤浶洍浡涒浘浢浭浯涑涍淯浿涆浞浧浠涗浰浼浟涂涘洯浨涋浾涀涄洖涃浻浽浵涐烜烓烑烝烋缹烢烗烒烞烠烔烍烅烆烇烚烎烡牂牸"],["d2a1","牷牶猀狺狴狾狶狳狻猁珓珙珥珖玼珧珣珩珜珒珛珔珝珚珗珘珨瓞瓟瓴瓵甡畛畟疰痁疻痄痀疿疶疺皊盉眝眛眐眓眒眣眑眕眙眚眢眧砣砬砢砵砯砨砮砫砡砩砳砪砱祔祛祏祜祓祒祑秫秬秠秮秭秪秜秞秝窆窉窅窋窌窊窇竘笐"],["d340","笄笓笅笏笈笊笎笉笒粄粑粊粌粈粍粅紞紝紑紎紘紖紓紟紒紏紌罜罡罞罠罝罛羖羒翃翂翀耖耾耹胺胲胹胵脁胻脀舁舯舥茳茭荄茙荑茥荖茿荁茦茜茢"],["d3a1","荂荎茛茪茈茼荍茖茤茠茷茯茩荇荅荌荓茞茬荋茧荈虓虒蚢蚨蚖蚍蚑蚞蚇蚗蚆蚋蚚蚅蚥蚙蚡蚧蚕蚘蚎蚝蚐蚔衃衄衭衵衶衲袀衱衿衯袃衾衴衼訒豇豗豻貤貣赶赸趵趷趶軑軓迾迵适迿迻逄迼迶郖郠郙郚郣郟郥郘郛郗郜郤酐"],["d440","酎酏釕釢釚陜陟隼飣髟鬯乿偰偪偡偞偠偓偋偝偲偈偍偁偛偊偢倕偅偟偩偫偣偤偆偀偮偳偗偑凐剫剭剬剮勖勓匭厜啵啶唼啍啐唴唪啑啢唶唵唰啒啅"],["d4a1","唌唲啥啎唹啈唭唻啀啋圊圇埻堔埢埶埜埴堀埭埽堈埸堋埳埏堇埮埣埲埥埬埡堎埼堐埧堁堌埱埩埰堍堄奜婠婘婕婧婞娸娵婭婐婟婥婬婓婤婗婃婝婒婄婛婈媎娾婍娹婌婰婩婇婑婖婂婜孲孮寁寀屙崞崋崝崚崠崌崨崍崦崥崏"],["d540","崰崒崣崟崮帾帴庱庴庹庲庳弶弸徛徖徟悊悐悆悾悰悺惓惔惏惤惙惝惈悱惛悷惊悿惃惍惀挲捥掊掂捽掽掞掭掝掗掫掎捯掇掐据掯捵掜捭掮捼掤挻掟"],["d5a1","捸掅掁掑掍捰敓旍晥晡晛晙晜晢朘桹梇梐梜桭桮梮梫楖桯梣梬梩桵桴梲梏桷梒桼桫桲梪梀桱桾梛梖梋梠梉梤桸桻梑梌梊桽欶欳欷欸殑殏殍殎殌氪淀涫涴涳湴涬淩淢涷淶淔渀淈淠淟淖涾淥淜淝淛淴淊涽淭淰涺淕淂淏淉"],["d640","淐淲淓淽淗淍淣涻烺焍烷焗烴焌烰焄烳焐烼烿焆焓焀烸烶焋焂焎牾牻牼牿猝猗猇猑猘猊猈狿猏猞玈珶珸珵琄琁珽琇琀珺珼珿琌琋珴琈畤畣痎痒痏"],["d6a1","痋痌痑痐皏皉盓眹眯眭眱眲眴眳眽眥眻眵硈硒硉硍硊硌砦硅硐祤祧祩祪祣祫祡离秺秸秶秷窏窔窐笵筇笴笥笰笢笤笳笘笪笝笱笫笭笯笲笸笚笣粔粘粖粣紵紽紸紶紺絅紬紩絁絇紾紿絊紻紨罣羕羜羝羛翊翋翍翐翑翇翏翉耟"],["d740","耞耛聇聃聈脘脥脙脛脭脟脬脞脡脕脧脝脢舑舸舳舺舴舲艴莐莣莨莍荺荳莤荴莏莁莕莙荵莔莩荽莃莌莝莛莪莋荾莥莯莈莗莰荿莦莇莮荶莚虙虖蚿蚷"],["d7a1","蛂蛁蛅蚺蚰蛈蚹蚳蚸蛌蚴蚻蚼蛃蚽蚾衒袉袕袨袢袪袚袑袡袟袘袧袙袛袗袤袬袌袓袎覂觖觙觕訰訧訬訞谹谻豜豝豽貥赽赻赹趼跂趹趿跁軘軞軝軜軗軠軡逤逋逑逜逌逡郯郪郰郴郲郳郔郫郬郩酖酘酚酓酕釬釴釱釳釸釤釹釪"],["d840","釫釷釨釮镺閆閈陼陭陫陱陯隿靪頄飥馗傛傕傔傞傋傣傃傌傎傝偨傜傒傂傇兟凔匒匑厤厧喑喨喥喭啷噅喢喓喈喏喵喁喣喒喤啽喌喦啿喕喡喎圌堩堷"],["d8a1","堙堞堧堣堨埵塈堥堜堛堳堿堶堮堹堸堭堬堻奡媯媔媟婺媢媞婸媦婼媥媬媕媮娷媄媊媗媃媋媩婻婽媌媜媏媓媝寪寍寋寔寑寊寎尌尰崷嵃嵫嵁嵋崿崵嵑嵎嵕崳崺嵒崽崱嵙嵂崹嵉崸崼崲崶嵀嵅幄幁彘徦徥徫惉悹惌惢惎惄愔"],["d940","惲愊愖愅惵愓惸惼惾惁愃愘愝愐惿愄愋扊掔掱掰揎揥揨揯揃撝揳揊揠揶揕揲揵摡揟掾揝揜揄揘揓揂揇揌揋揈揰揗揙攲敧敪敤敜敨敥斌斝斞斮旐旒"],["d9a1","晼晬晻暀晱晹晪晲朁椌棓椄棜椪棬棪棱椏棖棷棫棤棶椓椐棳棡椇棌椈楰梴椑棯棆椔棸棐棽棼棨椋椊椗棎棈棝棞棦棴棑椆棔棩椕椥棇欹欻欿欼殔殗殙殕殽毰毲毳氰淼湆湇渟湉溈渼渽湅湢渫渿湁湝湳渜渳湋湀湑渻渃渮湞"],["da40","湨湜湡渱渨湠湱湫渹渢渰湓湥渧湸湤湷湕湹湒湦渵渶湚焠焞焯烻焮焱焣焥焢焲焟焨焺焛牋牚犈犉犆犅犋猒猋猰猢猱猳猧猲猭猦猣猵猌琮琬琰琫琖"],["daa1","琚琡琭琱琤琣琝琩琠琲瓻甯畯畬痧痚痡痦痝痟痤痗皕皒盚睆睇睄睍睅睊睎睋睌矞矬硠硤硥硜硭硱硪确硰硩硨硞硢祴祳祲祰稂稊稃稌稄窙竦竤筊笻筄筈筌筎筀筘筅粢粞粨粡絘絯絣絓絖絧絪絏絭絜絫絒絔絩絑絟絎缾缿罥"],["db40","罦羢羠羡翗聑聏聐胾胔腃腊腒腏腇脽腍脺臦臮臷臸臹舄舼舽舿艵茻菏菹萣菀菨萒菧菤菼菶萐菆菈菫菣莿萁菝菥菘菿菡菋菎菖菵菉萉萏菞萑萆菂菳"],["dba1","菕菺菇菑菪萓菃菬菮菄菻菗菢萛菛菾蛘蛢蛦蛓蛣蛚蛪蛝蛫蛜蛬蛩蛗蛨蛑衈衖衕袺裗袹袸裀袾袶袼袷袽袲褁裉覕覘覗觝觚觛詎詍訹詙詀詗詘詄詅詒詈詑詊詌詏豟貁貀貺貾貰貹貵趄趀趉跘跓跍跇跖跜跏跕跙跈跗跅軯軷軺"],["dc40","軹軦軮軥軵軧軨軶軫軱軬軴軩逭逴逯鄆鄬鄄郿郼鄈郹郻鄁鄀鄇鄅鄃酡酤酟酢酠鈁鈊鈥鈃鈚鈦鈏鈌鈀鈒釿釽鈆鈄鈧鈂鈜鈤鈙鈗鈅鈖镻閍閌閐隇陾隈"],["dca1","隉隃隀雂雈雃雱雰靬靰靮頇颩飫鳦黹亃亄亶傽傿僆傮僄僊傴僈僂傰僁傺傱僋僉傶傸凗剺剸剻剼嗃嗛嗌嗐嗋嗊嗝嗀嗔嗄嗩喿嗒喍嗏嗕嗢嗖嗈嗲嗍嗙嗂圔塓塨塤塏塍塉塯塕塎塝塙塥塛堽塣塱壼嫇嫄嫋媺媸媱媵媰媿嫈媻嫆"],["dd40","媷嫀嫊媴媶嫍媹媐寖寘寙尟尳嵱嵣嵊嵥嵲嵬嵞嵨嵧嵢巰幏幎幊幍幋廅廌廆廋廇彀徯徭惷慉慊愫慅愶愲愮慆愯慏愩慀戠酨戣戥戤揅揱揫搐搒搉搠搤"],["dda1","搳摃搟搕搘搹搷搢搣搌搦搰搨摁搵搯搊搚摀搥搧搋揧搛搮搡搎敯斒旓暆暌暕暐暋暊暙暔晸朠楦楟椸楎楢楱椿楅楪椹楂楗楙楺楈楉椵楬椳椽楥棰楸椴楩楀楯楄楶楘楁楴楌椻楋椷楜楏楑椲楒椯楻椼歆歅歃歂歈歁殛嗀毻毼"],["de40","毹毷毸溛滖滈溏滀溟溓溔溠溱溹滆滒溽滁溞滉溷溰滍溦滏溲溾滃滜滘溙溒溎溍溤溡溿溳滐滊溗溮溣煇煔煒煣煠煁煝煢煲煸煪煡煂煘煃煋煰煟煐煓"],["dea1","煄煍煚牏犍犌犑犐犎猼獂猻猺獀獊獉瑄瑊瑋瑒瑑瑗瑀瑏瑐瑎瑂瑆瑍瑔瓡瓿瓾瓽甝畹畷榃痯瘏瘃痷痾痼痹痸瘐痻痶痭痵痽皙皵盝睕睟睠睒睖睚睩睧睔睙睭矠碇碚碔碏碄碕碅碆碡碃硹碙碀碖硻祼禂祽祹稑稘稙稒稗稕稢稓"],["df40","稛稐窣窢窞竫筦筤筭筴筩筲筥筳筱筰筡筸筶筣粲粴粯綈綆綀綍絿綅絺綎絻綃絼綌綔綄絽綒罭罫罧罨罬羦羥羧翛翜耡腤腠腷腜腩腛腢腲朡腞腶腧腯"],["dfa1","腄腡舝艉艄艀艂艅蓱萿葖葶葹蒏蒍葥葑葀蒆葧萰葍葽葚葙葴葳葝蔇葞萷萺萴葺葃葸萲葅萩菙葋萯葂萭葟葰萹葎葌葒葯蓅蒎萻葇萶萳葨葾葄萫葠葔葮葐蜋蜄蛷蜌蛺蛖蛵蝍蛸蜎蜉蜁蛶蜍蜅裖裋裍裎裞裛裚裌裐覅覛觟觥觤"],["e040","觡觠觢觜触詶誆詿詡訿詷誂誄詵誃誁詴詺谼豋豊豥豤豦貆貄貅賌赨赩趑趌趎趏趍趓趔趐趒跰跠跬跱跮跐跩跣跢跧跲跫跴輆軿輁輀輅輇輈輂輋遒逿"],["e0a1","遄遉逽鄐鄍鄏鄑鄖鄔鄋鄎酮酯鉈鉒鈰鈺鉦鈳鉥鉞銃鈮鉊鉆鉭鉬鉏鉠鉧鉯鈶鉡鉰鈱鉔鉣鉐鉲鉎鉓鉌鉖鈲閟閜閞閛隒隓隑隗雎雺雽雸雵靳靷靸靲頏頍頎颬飶飹馯馲馰馵骭骫魛鳪鳭鳧麀黽僦僔僗僨僳僛僪僝僤僓僬僰僯僣僠"],["e140","凘劀劁勩勫匰厬嘧嘕嘌嘒嗼嘏嘜嘁嘓嘂嗺嘝嘄嗿嗹墉塼墐墘墆墁塿塴墋塺墇墑墎塶墂墈塻墔墏壾奫嫜嫮嫥嫕嫪嫚嫭嫫嫳嫢嫠嫛嫬嫞嫝嫙嫨嫟孷寠"],["e1a1","寣屣嶂嶀嵽嶆嵺嶁嵷嶊嶉嶈嵾嵼嶍嵹嵿幘幙幓廘廑廗廎廜廕廙廒廔彄彃彯徶愬愨慁慞慱慳慒慓慲慬憀慴慔慺慛慥愻慪慡慖戩戧戫搫摍摛摝摴摶摲摳摽摵摦撦摎撂摞摜摋摓摠摐摿搿摬摫摙摥摷敳斠暡暠暟朅朄朢榱榶槉"],["e240","榠槎榖榰榬榼榑榙榎榧榍榩榾榯榿槄榽榤槔榹槊榚槏榳榓榪榡榞槙榗榐槂榵榥槆歊歍歋殞殟殠毃毄毾滎滵滱漃漥滸漷滻漮漉潎漙漚漧漘漻漒滭漊"],["e2a1","漶潳滹滮漭潀漰漼漵滫漇漎潃漅滽滶漹漜滼漺漟漍漞漈漡熇熐熉熀熅熂熏煻熆熁熗牄牓犗犕犓獃獍獑獌瑢瑳瑱瑵瑲瑧瑮甀甂甃畽疐瘖瘈瘌瘕瘑瘊瘔皸瞁睼瞅瞂睮瞀睯睾瞃碲碪碴碭碨硾碫碞碥碠碬碢碤禘禊禋禖禕禔禓"],["e340","禗禈禒禐稫穊稰稯稨稦窨窫窬竮箈箜箊箑箐箖箍箌箛箎箅箘劄箙箤箂粻粿粼粺綧綷緂綣綪緁緀緅綝緎緄緆緋緌綯綹綖綼綟綦綮綩綡緉罳翢翣翥翞"],["e3a1","耤聝聜膉膆膃膇膍膌膋舕蒗蒤蒡蒟蒺蓎蓂蒬蒮蒫蒹蒴蓁蓍蒪蒚蒱蓐蒝蒧蒻蒢蒔蓇蓌蒛蒩蒯蒨蓖蒘蒶蓏蒠蓗蓔蓒蓛蒰蒑虡蜳蜣蜨蝫蝀蜮蜞蜡蜙蜛蝃蜬蝁蜾蝆蜠蜲蜪蜭蜼蜒蜺蜱蜵蝂蜦蜧蜸蜤蜚蜰蜑裷裧裱裲裺裾裮裼裶裻"],["e440","裰裬裫覝覡覟覞觩觫觨誫誙誋誒誏誖谽豨豩賕賏賗趖踉踂跿踍跽踊踃踇踆踅跾踀踄輐輑輎輍鄣鄜鄠鄢鄟鄝鄚鄤鄡鄛酺酲酹酳銥銤鉶銛鉺銠銔銪銍"],["e4a1","銦銚銫鉹銗鉿銣鋮銎銂銕銢鉽銈銡銊銆銌銙銧鉾銇銩銝銋鈭隞隡雿靘靽靺靾鞃鞀鞂靻鞄鞁靿韎韍頖颭颮餂餀餇馝馜駃馹馻馺駂馽駇骱髣髧鬾鬿魠魡魟鳱鳲鳵麧僿儃儰僸儆儇僶僾儋儌僽儊劋劌勱勯噈噂噌嘵噁噊噉噆噘"],["e540","噚噀嘳嘽嘬嘾嘸嘪嘺圚墫墝墱墠墣墯墬墥墡壿嫿嫴嫽嫷嫶嬃嫸嬂嫹嬁嬇嬅嬏屧嶙嶗嶟嶒嶢嶓嶕嶠嶜嶡嶚嶞幩幝幠幜緳廛廞廡彉徲憋憃慹憱憰憢憉"],["e5a1","憛憓憯憭憟憒憪憡憍慦憳戭摮摰撖撠撅撗撜撏撋撊撌撣撟摨撱撘敶敺敹敻斲斳暵暰暩暲暷暪暯樀樆樗槥槸樕槱槤樠槿槬槢樛樝槾樧槲槮樔槷槧橀樈槦槻樍槼槫樉樄樘樥樏槶樦樇槴樖歑殥殣殢殦氁氀毿氂潁漦潾澇濆澒"],["e640","澍澉澌潢潏澅潚澖潶潬澂潕潲潒潐潗澔澓潝漀潡潫潽潧澐潓澋潩潿澕潣潷潪潻熲熯熛熰熠熚熩熵熝熥熞熤熡熪熜熧熳犘犚獘獒獞獟獠獝獛獡獚獙"],["e6a1","獢璇璉璊璆璁瑽璅璈瑼瑹甈甇畾瘥瘞瘙瘝瘜瘣瘚瘨瘛皜皝皞皛瞍瞏瞉瞈磍碻磏磌磑磎磔磈磃磄磉禚禡禠禜禢禛歶稹窲窴窳箷篋箾箬篎箯箹篊箵糅糈糌糋緷緛緪緧緗緡縃緺緦緶緱緰緮緟罶羬羰羭翭翫翪翬翦翨聤聧膣膟"],["e740","膞膕膢膙膗舖艏艓艒艐艎艑蔤蔻蔏蔀蔩蔎蔉蔍蔟蔊蔧蔜蓻蔫蓺蔈蔌蓴蔪蓲蔕蓷蓫蓳蓼蔒蓪蓩蔖蓾蔨蔝蔮蔂蓽蔞蓶蔱蔦蓧蓨蓰蓯蓹蔘蔠蔰蔋蔙蔯虢"],["e7a1","蝖蝣蝤蝷蟡蝳蝘蝔蝛蝒蝡蝚蝑蝞蝭蝪蝐蝎蝟蝝蝯蝬蝺蝮蝜蝥蝏蝻蝵蝢蝧蝩衚褅褌褔褋褗褘褙褆褖褑褎褉覢覤覣觭觰觬諏諆誸諓諑諔諕誻諗誾諀諅諘諃誺誽諙谾豍貏賥賟賙賨賚賝賧趠趜趡趛踠踣踥踤踮踕踛踖踑踙踦踧"],["e840","踔踒踘踓踜踗踚輬輤輘輚輠輣輖輗遳遰遯遧遫鄯鄫鄩鄪鄲鄦鄮醅醆醊醁醂醄醀鋐鋃鋄鋀鋙銶鋏鋱鋟鋘鋩鋗鋝鋌鋯鋂鋨鋊鋈鋎鋦鋍鋕鋉鋠鋞鋧鋑鋓"],["e8a1","銵鋡鋆銴镼閬閫閮閰隤隢雓霅霈霂靚鞊鞎鞈韐韏頞頝頦頩頨頠頛頧颲餈飺餑餔餖餗餕駜駍駏駓駔駎駉駖駘駋駗駌骳髬髫髳髲髱魆魃魧魴魱魦魶魵魰魨魤魬鳼鳺鳽鳿鳷鴇鴀鳹鳻鴈鴅鴄麃黓鼏鼐儜儓儗儚儑凞匴叡噰噠噮"],["e940","噳噦噣噭噲噞噷圜圛壈墽壉墿墺壂墼壆嬗嬙嬛嬡嬔嬓嬐嬖嬨嬚嬠嬞寯嶬嶱嶩嶧嶵嶰嶮嶪嶨嶲嶭嶯嶴幧幨幦幯廩廧廦廨廥彋徼憝憨憖懅憴懆懁懌憺"],["e9a1","憿憸憌擗擖擐擏擉撽撉擃擛擳擙攳敿敼斢曈暾曀曊曋曏暽暻暺曌朣樴橦橉橧樲橨樾橝橭橶橛橑樨橚樻樿橁橪橤橐橏橔橯橩橠樼橞橖橕橍橎橆歕歔歖殧殪殫毈毇氄氃氆澭濋澣濇澼濎濈潞濄澽澞濊澨瀄澥澮澺澬澪濏澿澸"],["ea40","澢濉澫濍澯澲澰燅燂熿熸燖燀燁燋燔燊燇燏熽燘熼燆燚燛犝犞獩獦獧獬獥獫獪瑿璚璠璔璒璕璡甋疀瘯瘭瘱瘽瘳瘼瘵瘲瘰皻盦瞚瞝瞡瞜瞛瞢瞣瞕瞙"],["eaa1","瞗磝磩磥磪磞磣磛磡磢磭磟磠禤穄穈穇窶窸窵窱窷篞篣篧篝篕篥篚篨篹篔篪篢篜篫篘篟糒糔糗糐糑縒縡縗縌縟縠縓縎縜縕縚縢縋縏縖縍縔縥縤罃罻罼罺羱翯耪耩聬膱膦膮膹膵膫膰膬膴膲膷膧臲艕艖艗蕖蕅蕫蕍蕓蕡蕘"],["eb40","蕀蕆蕤蕁蕢蕄蕑蕇蕣蔾蕛蕱蕎蕮蕵蕕蕧蕠薌蕦蕝蕔蕥蕬虣虥虤螛螏螗螓螒螈螁螖螘蝹螇螣螅螐螑螝螄螔螜螚螉褞褦褰褭褮褧褱褢褩褣褯褬褟觱諠"],["eba1","諢諲諴諵諝謔諤諟諰諈諞諡諨諿諯諻貑貒貐賵賮賱賰賳赬赮趥趧踳踾踸蹀蹅踶踼踽蹁踰踿躽輶輮輵輲輹輷輴遶遹遻邆郺鄳鄵鄶醓醐醑醍醏錧錞錈錟錆錏鍺錸錼錛錣錒錁鍆錭錎錍鋋錝鋺錥錓鋹鋷錴錂錤鋿錩錹錵錪錔錌"],["ec40","錋鋾錉錀鋻錖閼闍閾閹閺閶閿閵閽隩雔霋霒霐鞙鞗鞔韰韸頵頯頲餤餟餧餩馞駮駬駥駤駰駣駪駩駧骹骿骴骻髶髺髹髷鬳鮀鮅鮇魼魾魻鮂鮓鮒鮐魺鮕"],["eca1","魽鮈鴥鴗鴠鴞鴔鴩鴝鴘鴢鴐鴙鴟麈麆麇麮麭黕黖黺鼒鼽儦儥儢儤儠儩勴嚓嚌嚍嚆嚄嚃噾嚂噿嚁壖壔壏壒嬭嬥嬲嬣嬬嬧嬦嬯嬮孻寱寲嶷幬幪徾徻懃憵憼懧懠懥懤懨懞擯擩擣擫擤擨斁斀斶旚曒檍檖檁檥檉檟檛檡檞檇檓檎"],["ed40","檕檃檨檤檑橿檦檚檅檌檒歛殭氉濌澩濴濔濣濜濭濧濦濞濲濝濢濨燡燱燨燲燤燰燢獳獮獯璗璲璫璐璪璭璱璥璯甐甑甒甏疄癃癈癉癇皤盩瞵瞫瞲瞷瞶"],["eda1","瞴瞱瞨矰磳磽礂磻磼磲礅磹磾礄禫禨穜穛穖穘穔穚窾竀竁簅簏篲簀篿篻簎篴簋篳簂簉簃簁篸篽簆篰篱簐簊糨縭縼繂縳顈縸縪繉繀繇縩繌縰縻縶繄縺罅罿罾罽翴翲耬膻臄臌臊臅臇膼臩艛艚艜薃薀薏薧薕薠薋薣蕻薤薚薞"],["ee40","蕷蕼薉薡蕺蕸蕗薎薖薆薍薙薝薁薢薂薈薅蕹蕶薘薐薟虨螾螪螭蟅螰螬螹螵螼螮蟉蟃蟂蟌螷螯蟄蟊螴螶螿螸螽蟞螲褵褳褼褾襁襒褷襂覭覯覮觲觳謞"],["eea1","謘謖謑謅謋謢謏謒謕謇謍謈謆謜謓謚豏豰豲豱豯貕貔賹赯蹎蹍蹓蹐蹌蹇轃轀邅遾鄸醚醢醛醙醟醡醝醠鎡鎃鎯鍤鍖鍇鍼鍘鍜鍶鍉鍐鍑鍠鍭鎏鍌鍪鍹鍗鍕鍒鍏鍱鍷鍻鍡鍞鍣鍧鎀鍎鍙闇闀闉闃闅閷隮隰隬霠霟霘霝霙鞚鞡鞜"],["ef40","鞞鞝韕韔韱顁顄顊顉顅顃餥餫餬餪餳餲餯餭餱餰馘馣馡騂駺駴駷駹駸駶駻駽駾駼騃骾髾髽鬁髼魈鮚鮨鮞鮛鮦鮡鮥鮤鮆鮢鮠鮯鴳鵁鵧鴶鴮鴯鴱鴸鴰"],["efa1","鵅鵂鵃鴾鴷鵀鴽翵鴭麊麉麍麰黈黚黻黿鼤鼣鼢齔龠儱儭儮嚘嚜嚗嚚嚝嚙奰嬼屩屪巀幭幮懘懟懭懮懱懪懰懫懖懩擿攄擽擸攁攃擼斔旛曚曛曘櫅檹檽櫡櫆檺檶檷櫇檴檭歞毉氋瀇瀌瀍瀁瀅瀔瀎濿瀀濻瀦濼濷瀊爁燿燹爃燽獶"],["f040","璸瓀璵瓁璾璶璻瓂甔甓癜癤癙癐癓癗癚皦皽盬矂瞺磿礌礓礔礉礐礒礑禭禬穟簜簩簙簠簟簭簝簦簨簢簥簰繜繐繖繣繘繢繟繑繠繗繓羵羳翷翸聵臑臒"],["f0a1","臐艟艞薴藆藀藃藂薳薵薽藇藄薿藋藎藈藅薱薶藒蘤薸薷薾虩蟧蟦蟢蟛蟫蟪蟥蟟蟳蟤蟔蟜蟓蟭蟘蟣螤蟗蟙蠁蟴蟨蟝襓襋襏襌襆襐襑襉謪謧謣謳謰謵譇謯謼謾謱謥謷謦謶謮謤謻謽謺豂豵貙貘貗賾贄贂贀蹜蹢蹠蹗蹖蹞蹥蹧"],["f140","蹛蹚蹡蹝蹩蹔轆轇轈轋鄨鄺鄻鄾醨醥醧醯醪鎵鎌鎒鎷鎛鎝鎉鎧鎎鎪鎞鎦鎕鎈鎙鎟鎍鎱鎑鎲鎤鎨鎴鎣鎥闒闓闑隳雗雚巂雟雘雝霣霢霥鞬鞮鞨鞫鞤鞪"],["f1a1","鞢鞥韗韙韖韘韺顐顑顒颸饁餼餺騏騋騉騍騄騑騊騅騇騆髀髜鬈鬄鬅鬩鬵魊魌魋鯇鯆鯃鮿鯁鮵鮸鯓鮶鯄鮹鮽鵜鵓鵏鵊鵛鵋鵙鵖鵌鵗鵒鵔鵟鵘鵚麎麌黟鼁鼀鼖鼥鼫鼪鼩鼨齌齕儴儵劖勷厴嚫嚭嚦嚧嚪嚬壚壝壛夒嬽嬾嬿巃幰"],["f240","徿懻攇攐攍攉攌攎斄旞旝曞櫧櫠櫌櫑櫙櫋櫟櫜櫐櫫櫏櫍櫞歠殰氌瀙瀧瀠瀖瀫瀡瀢瀣瀩瀗瀤瀜瀪爌爊爇爂爅犥犦犤犣犡瓋瓅璷瓃甖癠矉矊矄矱礝礛"],["f2a1","礡礜礗礞禰穧穨簳簼簹簬簻糬糪繶繵繸繰繷繯繺繲繴繨罋罊羃羆羷翽翾聸臗臕艤艡艣藫藱藭藙藡藨藚藗藬藲藸藘藟藣藜藑藰藦藯藞藢蠀蟺蠃蟶蟷蠉蠌蠋蠆蟼蠈蟿蠊蠂襢襚襛襗襡襜襘襝襙覈覷覶觶譐譈譊譀譓譖譔譋譕"],["f340","譑譂譒譗豃豷豶貚贆贇贉趬趪趭趫蹭蹸蹳蹪蹯蹻軂轒轑轏轐轓辴酀鄿醰醭鏞鏇鏏鏂鏚鏐鏹鏬鏌鏙鎩鏦鏊鏔鏮鏣鏕鏄鏎鏀鏒鏧镽闚闛雡霩霫霬霨霦"],["f3a1","鞳鞷鞶韝韞韟顜顙顝顗颿颽颻颾饈饇饃馦馧騚騕騥騝騤騛騢騠騧騣騞騜騔髂鬋鬊鬎鬌鬷鯪鯫鯠鯞鯤鯦鯢鯰鯔鯗鯬鯜鯙鯥鯕鯡鯚鵷鶁鶊鶄鶈鵱鶀鵸鶆鶋鶌鵽鵫鵴鵵鵰鵩鶅鵳鵻鶂鵯鵹鵿鶇鵨麔麑黀黼鼭齀齁齍齖齗齘匷嚲"],["f440","嚵嚳壣孅巆巇廮廯忀忁懹攗攖攕攓旟曨曣曤櫳櫰櫪櫨櫹櫱櫮櫯瀼瀵瀯瀷瀴瀱灂瀸瀿瀺瀹灀瀻瀳灁爓爔犨獽獼璺皫皪皾盭矌矎矏矍矲礥礣礧礨礤礩"],["f4a1","禲穮穬穭竷籉籈籊籇籅糮繻繾纁纀羺翿聹臛臙舋艨艩蘢藿蘁藾蘛蘀藶蘄蘉蘅蘌藽蠙蠐蠑蠗蠓蠖襣襦覹觷譠譪譝譨譣譥譧譭趮躆躈躄轙轖轗轕轘轚邍酃酁醷醵醲醳鐋鐓鏻鐠鐏鐔鏾鐕鐐鐨鐙鐍鏵鐀鏷鐇鐎鐖鐒鏺鐉鏸鐊鏿"],["f540","鏼鐌鏶鐑鐆闞闠闟霮霯鞹鞻韽韾顠顢顣顟飁飂饐饎饙饌饋饓騲騴騱騬騪騶騩騮騸騭髇髊髆鬐鬒鬑鰋鰈鯷鰅鰒鯸鱀鰇鰎鰆鰗鰔鰉鶟鶙鶤鶝鶒鶘鶐鶛"],["f5a1","鶠鶔鶜鶪鶗鶡鶚鶢鶨鶞鶣鶿鶩鶖鶦鶧麙麛麚黥黤黧黦鼰鼮齛齠齞齝齙龑儺儹劘劗囃嚽嚾孈孇巋巏廱懽攛欂櫼欃櫸欀灃灄灊灈灉灅灆爝爚爙獾甗癪矐礭礱礯籔籓糲纊纇纈纋纆纍罍羻耰臝蘘蘪蘦蘟蘣蘜蘙蘧蘮蘡蘠蘩蘞蘥"],["f640","蠩蠝蠛蠠蠤蠜蠫衊襭襩襮襫觺譹譸譅譺譻贐贔趯躎躌轞轛轝酆酄酅醹鐿鐻鐶鐩鐽鐼鐰鐹鐪鐷鐬鑀鐱闥闤闣霵霺鞿韡顤飉飆飀饘饖騹騽驆驄驂驁騺"],["f6a1","騿髍鬕鬗鬘鬖鬺魒鰫鰝鰜鰬鰣鰨鰩鰤鰡鶷鶶鶼鷁鷇鷊鷏鶾鷅鷃鶻鶵鷎鶹鶺鶬鷈鶱鶭鷌鶳鷍鶲鹺麜黫黮黭鼛鼘鼚鼱齎齥齤龒亹囆囅囋奱孋孌巕巑廲攡攠攦攢欋欈欉氍灕灖灗灒爞爟犩獿瓘瓕瓙瓗癭皭礵禴穰穱籗籜籙籛籚"],["f740","糴糱纑罏羇臞艫蘴蘵蘳蘬蘲蘶蠬蠨蠦蠪蠥襱覿覾觻譾讄讂讆讅譿贕躕躔躚躒躐躖躗轠轢酇鑌鑐鑊鑋鑏鑇鑅鑈鑉鑆霿韣顪顩飋饔饛驎驓驔驌驏驈驊"],["f7a1","驉驒驐髐鬙鬫鬻魖魕鱆鱈鰿鱄鰹鰳鱁鰼鰷鰴鰲鰽鰶鷛鷒鷞鷚鷋鷐鷜鷑鷟鷩鷙鷘鷖鷵鷕鷝麶黰鼵鼳鼲齂齫龕龢儽劙壨壧奲孍巘蠯彏戁戃戄攩攥斖曫欑欒欏毊灛灚爢玂玁玃癰矔籧籦纕艬蘺虀蘹蘼蘱蘻蘾蠰蠲蠮蠳襶襴襳觾"],["f840","讌讎讋讈豅贙躘轤轣醼鑢鑕鑝鑗鑞韄韅頀驖驙鬞鬟鬠鱒鱘鱐鱊鱍鱋鱕鱙鱌鱎鷻鷷鷯鷣鷫鷸鷤鷶鷡鷮鷦鷲鷰鷢鷬鷴鷳鷨鷭黂黐黲黳鼆鼜鼸鼷鼶齃齏"],["f8a1","齱齰齮齯囓囍孎屭攭曭曮欓灟灡灝灠爣瓛瓥矕礸禷禶籪纗羉艭虃蠸蠷蠵衋讔讕躞躟躠躝醾醽釂鑫鑨鑩雥靆靃靇韇韥驞髕魙鱣鱧鱦鱢鱞鱠鸂鷾鸇鸃鸆鸅鸀鸁鸉鷿鷽鸄麠鼞齆齴齵齶囔攮斸欘欙欗欚灢爦犪矘矙礹籩籫糶纚"],["f940","纘纛纙臠臡虆虇虈襹襺襼襻觿讘讙躥躤躣鑮鑭鑯鑱鑳靉顲饟鱨鱮鱭鸋鸍鸐鸏鸒鸑麡黵鼉齇齸齻齺齹圞灦籯蠼趲躦釃鑴鑸鑶鑵驠鱴鱳鱱鱵鸔鸓黶鼊"],["f9a1","龤灨灥糷虪蠾蠽蠿讞貜躩軉靋顳顴飌饡馫驤驦驧鬤鸕鸗齈戇欞爧虌躨钂钀钁驩驨鬮鸙爩虋讟钃鱹麷癵驫鱺鸝灩灪麤齾齉龘碁銹裏墻恒粧嫺╔╦╗╠╬╣╚╩╝╒╤╕╞╪╡╘╧╛╓╥╖╟╫╢╙╨╜║═╭╮╰╯▓"]]')},function(n,i,o){(function(){var i,u,p;u=o(89).Number,p=o(38),i=function(){function ArrayT(n,i,o){this.type=n,this.length=i,this.lengthType=null!=o?o:"count"}return ArrayT.prototype.decode=function(n,i){var o,g,y,w,x,_;if(y=n.pos,w=[],o=i,null!=this.length&&(g=p.resolveLength(this.length,n,i)),this.length instanceof u&&(Object.defineProperties(w,{parent:{value:i},_startOffset:{value:y},_currentOffset:{value:0,writable:!0},_length:{value:g}}),o=w),null==g||"bytes"===this.lengthType)for(x=null!=g?n.pos+g:(null!=i?i._length:void 0)?i._startOffset+i._length:n.length;n.pos<x;)w.push(this.type.decode(n,o));else for(_=0;_<g;_+=1)w.push(this.type.decode(n,o));return w},ArrayT.prototype.size=function(n,i){var o,g,y,w;if(!n)return this.type.size(null,i)*p.resolveLength(this.length,null,i);for(g=0,this.length instanceof u&&(g+=this.length.size(),i={parent:i}),y=0,w=n.length;y<w;y++)o=n[y],g+=this.type.size(o,i);return g},ArrayT.prototype.encode=function(n,i,o){var p,g,y,w,x,_;for(p=o,this.length instanceof u&&((p={pointers:[],startOffset:n.pos,parent:o}).pointerOffset=n.pos+this.size(i,p),this.length.encode(n,i.length)),x=0,_=i.length;x<_;x++)y=i[x],this.type.encode(n,y,p);if(this.length instanceof u)for(g=0;g<p.pointers.length;)(w=p.pointers[g++]).type.encode(n,w.val)},ArrayT}(),n.exports=i}).call(this)},function(n,i,o){(function(){var i,u;u=o(38),i=function(){function Struct(n){this.fields=null!=n?n:{}}return Struct.prototype.decode=function(n,i,o){var u,p;return null==o&&(o=0),u=this._setup(n,i,o),this._parseFields(n,u,this.fields),null!=(p=this.process)&&p.call(u,n),u},Struct.prototype._setup=function(n,i,o){var u;return u={},Object.defineProperties(u,{parent:{value:i},_startOffset:{value:n.pos},_currentOffset:{value:0,writable:!0},_length:{value:o}}),u},Struct.prototype._parseFields=function(n,i,o){var p,g,y;for(p in o)void 0!==(y="function"==typeof(g=o[p])?g.call(i,i):g.decode(n,i))&&(y instanceof u.PropertyDescriptor?Object.defineProperty(i,p,y):i[p]=y),i._currentOffset=n.pos-i._startOffset},Struct.prototype.size=function(n,i,o){var u,p,g,y,w;for(p in null==n&&(n={}),null==o&&(o=!0),u={parent:i,val:n,pointerSize:0},g=0,w=this.fields)null!=(y=w[p]).size&&(g+=y.size(n[p],u));return o&&(g+=u.pointerSize),g},Struct.prototype.encode=function(n,i,o){var u,p,g,y,w,x,_;for(g in null!=(x=this.preEncode)&&x.call(i,n),(u={pointers:[],startOffset:n.pos,parent:o,val:i,pointerSize:0}).pointerOffset=n.pos+this.size(i,u,!1),_=this.fields)null!=(w=_[g]).encode&&w.encode(n,i[g],u);for(p=0;p<u.pointers.length;)(y=u.pointers[p++]).type.encode(n,y.val,y.parent)},Struct}(),n.exports=i}).call(this)},function(n,i,o){n.exports=!o(23)&&!o(52)((function(){return 7!=Object.defineProperty(o(177)("div"),"a",{get:function(){return 7}}).a}))},function(n,i){n.exports=function(n,i){return{value:i,done:!!n}}},function(n,i,o){n.exports=o(39)},function(n,i,o){var u=o(26),p=o(29),g=o(105);n.exports=o(23)?Object.defineProperties:function defineProperties(n,i){p(n);for(var o,y=g(i),w=y.length,x=0;w>x;)u.f(n,o=y[x++],i[o]);return n}},function(n,i,o){var u=o(51),p=o(50),g=o(431)(!1),y=o(181)("IE_PROTO");n.exports=function(n,i){var o,w=p(n),x=0,_=[];for(o in w)o!=y&&u(w,o)&&_.push(o);for(;i.length>x;)u(w,o=i[x++])&&(~g(_,o)||_.push(o));return _}},function(n,i,o){var u=o(180),p=Math.max,g=Math.min;n.exports=function(n,i){return(n=u(n))<0?p(n+i,0):g(n,i)}},function(n,i,o){var u=o(17).document;n.exports=u&&u.documentElement},function(n,i,o){var u=o(99);n.exports=Array.isArray||function isArray(n){return"Array"==u(n)}},function(n,i,o){var u=o(259),p=o(183).concat("length","prototype");i.f=Object.getOwnPropertyNames||function getOwnPropertyNames(n){return u(n,p)}},function(n,i,o){n.exports={default:o(450),__esModule:!0}},function(n,i,o){"use strict";var u=o(26).f,p=o(129),g=o(190),y=o(32),w=o(191),x=o(107),_=o(179),k=o(256),P=o(268),E=o(23),O=o(132).fastKey,B=o(192),I=E?"_s":"size",getEntry=function(n,i){var o,u=O(i);if("F"!==u)return n._i[u];for(o=n._f;o;o=o.n)if(o.k==i)return o};n.exports={getConstructor:function(n,i,o,_){var k=n((function(n,u){w(n,k,i,"_i"),n._t=i,n._i=p(null),n._f=void 0,n._l=void 0,n[I]=0,null!=u&&x(u,o,n[_],n)}));return g(k.prototype,{clear:function clear(){for(var n=B(this,i),o=n._i,u=n._f;u;u=u.n)u.r=!0,u.p&&(u.p=u.p.n=void 0),delete o[u.i];n._f=n._l=void 0,n[I]=0},delete:function(n){var o=B(this,i),u=getEntry(o,n);if(u){var p=u.n,g=u.p;delete o._i[u.i],u.r=!0,g&&(g.n=p),p&&(p.p=g),o._f==u&&(o._f=p),o._l==u&&(o._l=g),o[I]--}return!!u},forEach:function forEach(n){B(this,i);for(var o,u=y(n,arguments.length>1?arguments[1]:void 0,3);o=o?o.n:this._f;)for(u(o.v,o.k,this);o&&o.r;)o=o.p},has:function has(n){return!!getEntry(B(this,i),n)}}),E&&u(k.prototype,"size",{get:function(){return B(this,i)[I]}}),k},def:function(n,i,o){var u,p,g=getEntry(n,i);return g?g.v=o:(n._l=g={i:p=O(i,!0),k:i,v:o,p:u=n._l,n:void 0,r:!1},n._f||(n._f=g),u&&(u.n=g),n[I]++,"F"!==p&&(n._i[p]=g)),n},getEntry:getEntry,setStrong:function(n,i,o){_(n,i,(function(n,o){this._t=B(n,i),this._k=o,this._l=void 0}),(function(){for(var n=this._k,i=this._l;i&&i.r;)i=i.p;return this._t&&(this._l=i=i?i.n:this._t._f)?k(0,"keys"==n?i.k:"values"==n?i.v:[i.k,i.v]):(this._t=void 0,k(1))}),o?"entries":"values",!o,!0),P(i)}}},function(n,i,o){var u=o(29);n.exports=function(n,i,o,p){try{return p?i(u(o)[0],o[1]):i(o)}catch(i){var g=n.return;throw void 0!==g&&u(g.call(n)),i}}},function(n,i,o){var u=o(103),p=o(21)("iterator"),g=Array.prototype;n.exports=function(n){return void 0!==n&&(u.Array===n||g[p]===n)}},function(n,i,o){"use strict";var u=o(17),p=o(7),g=o(26),y=o(23),w=o(21)("species");n.exports=function(n){var i="function"==typeof p[n]?p[n]:u[n];y&&i&&!i[w]&&g.f(i,w,{configurable:!0,get:function(){return this}})}},function(n,i,o){"use strict";var u=o(17),p=o(9),g=o(132),y=o(52),w=o(39),x=o(190),_=o(107),k=o(191),P=o(22),E=o(106),O=o(26).f,B=o(457)(0),I=o(23);n.exports=function(n,i,o,D,R,N){var U=u[n],W=U,G=R?"set":"add",j=W&&W.prototype,X={};return I&&"function"==typeof W&&(N||j.forEach&&!y((function(){(new W).entries().next()})))?(W=i((function(i,o){k(i,W,n,"_c"),i._c=new U,null!=o&&_(o,R,i[G],i)})),B("add,clear,delete,forEach,get,has,set,keys,values,entries,toJSON".split(","),(function(n){var i="add"==n||"set"==n;!(n in j)||N&&"clear"==n||w(W.prototype,n,(function(o,u){if(k(this,W,n),!i&&N&&!P(o))return"get"==n&&void 0;var p=this._c[n](0===o?0:o,u);return i?this:p}))})),N||O(W.prototype,"size",{get:function(){return this._c.size}})):(W=D.getConstructor(i,n,R,G),x(W.prototype,o),g.NEED=!0),E(W,n),X[n]=W,p(p.G+p.W+p.F,X),N||D.setStrong(W,n,R),W}},function(n,i,o){var u=o(185),p=o(461);n.exports=function(n){return function toJSON(){if(u(this)!=n)throw TypeError(n+"#toJSON isn't generic");return p(this)}}},function(n,i,o){"use strict";var u=o(9);n.exports=function(n){u(u.S,n,{of:function of(){for(var n=arguments.length,i=new Array(n);n--;)i[n]=arguments[n];return new this(i)}})}},function(n,i,o){"use strict";var u=o(9),p=o(101),g=o(32),y=o(107);n.exports=function(n){u(u.S,n,{from:function from(n){var i,o,u,w,x=arguments[1];return p(this),(i=void 0!==x)&&p(x),null==n?new this:(o=[],i?(u=0,w=g(x,arguments[2],2),y(n,!1,(function(n){o.push(w(n,u++))}))):y(n,!1,o.push,o),new this(o))}})}},function(n,i,o){"use strict";var u=Array.prototype.slice,p=o(274),g=Object.keys,y=g?function keys(n){return g(n)}:o(477),w=Object.keys;y.shim=function shimObjectKeys(){Object.keys?function(){var n=Object.keys(arguments);return n&&n.length===arguments.length}(1,2)||(Object.keys=function keys(n){return p(n)?w(u.call(n)):w(n)}):Object.keys=y;return Object.keys||y},n.exports=y},function(n,i,o){"use strict";var u=Object.prototype.toString;n.exports=function isArguments(n){var i=u.call(n),o="[object Arguments]"===i;return o||(o="[object Array]"!==i&&null!==n&&"object"==typeof n&&"number"==typeof n.length&&n.length>=0&&"[object Function]"===u.call(n.callee)),o}},function(n,i,o){"use strict";var u=SyntaxError,p=Function,g=TypeError,getEvalledConstructor=function(n){try{return Function('"use strict"; return ('+n+").constructor;")()}catch(n){}},y=Object.getOwnPropertyDescriptor;if(y)try{y({},"")}catch(n){y=null}var throwTypeError=function(){throw new g},w=y?function(){try{return throwTypeError}catch(n){try{return y(arguments,"callee").get}catch(n){return throwTypeError}}}():throwTypeError,x=o(276)(),_=Object.getPrototypeOf||function(n){return n.__proto__},k=getEvalledConstructor("async function* () {}"),P=k?k.prototype:void 0,E=P?P.prototype:void 0,O="undefined"==typeof Uint8Array?void 0:_(Uint8Array),B={"%AggregateError%":"undefined"==typeof AggregateError?void 0:AggregateError,"%Array%":Array,"%ArrayBuffer%":"undefined"==typeof ArrayBuffer?void 0:ArrayBuffer,"%ArrayIteratorPrototype%":x?_([][Symbol.iterator]()):void 0,"%AsyncFromSyncIteratorPrototype%":void 0,"%AsyncFunction%":getEvalledConstructor("async function () {}"),"%AsyncGenerator%":P,"%AsyncGeneratorFunction%":k,"%AsyncIteratorPrototype%":E?_(E):void 0,"%Atomics%":"undefined"==typeof Atomics?void 0:Atomics,"%BigInt%":"undefined"==typeof BigInt?void 0:BigInt,"%Boolean%":Boolean,"%DataView%":"undefined"==typeof DataView?void 0:DataView,"%Date%":Date,"%decodeURI%":decodeURI,"%decodeURIComponent%":decodeURIComponent,"%encodeURI%":encodeURI,"%encodeURIComponent%":encodeURIComponent,"%Error%":Error,"%eval%":eval,"%EvalError%":EvalError,"%Float32Array%":"undefined"==typeof Float32Array?void 0:Float32Array,"%Float64Array%":"undefined"==typeof Float64Array?void 0:Float64Array,"%FinalizationRegistry%":"undefined"==typeof FinalizationRegistry?void 0:FinalizationRegistry,"%Function%":p,"%GeneratorFunction%":getEvalledConstructor("function* () {}"),"%Int8Array%":"undefined"==typeof Int8Array?void 0:Int8Array,"%Int16Array%":"undefined"==typeof Int16Array?void 0:Int16Array,"%Int32Array%":"undefined"==typeof Int32Array?void 0:Int32Array,"%isFinite%":isFinite,"%isNaN%":isNaN,"%IteratorPrototype%":x?_(_([][Symbol.iterator]())):void 0,"%JSON%":"object"==typeof JSON?JSON:void 0,"%Map%":"undefined"==typeof Map?void 0:Map,"%MapIteratorPrototype%":"undefined"!=typeof Map&&x?_((new Map)[Symbol.iterator]()):void 0,"%Math%":Math,"%Number%":Number,"%Object%":Object,"%parseFloat%":parseFloat,"%parseInt%":parseInt,"%Promise%":"undefined"==typeof Promise?void 0:Promise,"%Proxy%":"undefined"==typeof Proxy?void 0:Proxy,"%RangeError%":RangeError,"%ReferenceError%":ReferenceError,"%Reflect%":"undefined"==typeof Reflect?void 0:Reflect,"%RegExp%":RegExp,"%Set%":"undefined"==typeof Set?void 0:Set,"%SetIteratorPrototype%":"undefined"!=typeof Set&&x?_((new Set)[Symbol.iterator]()):void 0,"%SharedArrayBuffer%":"undefined"==typeof SharedArrayBuffer?void 0:SharedArrayBuffer,"%String%":String,"%StringIteratorPrototype%":x?_(""[Symbol.iterator]()):void 0,"%Symbol%":x?Symbol:void 0,"%SyntaxError%":u,"%ThrowTypeError%":w,"%TypedArray%":O,"%TypeError%":g,"%Uint8Array%":"undefined"==typeof Uint8Array?void 0:Uint8Array,"%Uint8ClampedArray%":"undefined"==typeof Uint8ClampedArray?void 0:Uint8ClampedArray,"%Uint16Array%":"undefined"==typeof Uint16Array?void 0:Uint16Array,"%Uint32Array%":"undefined"==typeof Uint32Array?void 0:Uint32Array,"%URIError%":URIError,"%WeakMap%":"undefined"==typeof WeakMap?void 0:WeakMap,"%WeakRef%":"undefined"==typeof WeakRef?void 0:WeakRef,"%WeakSet%":"undefined"==typeof WeakSet?void 0:WeakSet},I={"%ArrayBufferPrototype%":["ArrayBuffer","prototype"],"%ArrayPrototype%":["Array","prototype"],"%ArrayProto_entries%":["Array","prototype","entries"],"%ArrayProto_forEach%":["Array","prototype","forEach"],"%ArrayProto_keys%":["Array","prototype","keys"],"%ArrayProto_values%":["Array","prototype","values"],"%AsyncFunctionPrototype%":["AsyncFunction","prototype"],"%AsyncGenerator%":["AsyncGeneratorFunction","prototype"],"%AsyncGeneratorPrototype%":["AsyncGeneratorFunction","prototype","prototype"],"%BooleanPrototype%":["Boolean","prototype"],"%DataViewPrototype%":["DataView","prototype"],"%DatePrototype%":["Date","prototype"],"%ErrorPrototype%":["Error","prototype"],"%EvalErrorPrototype%":["EvalError","prototype"],"%Float32ArrayPrototype%":["Float32Array","prototype"],"%Float64ArrayPrototype%":["Float64Array","prototype"],"%FunctionPrototype%":["Function","prototype"],"%Generator%":["GeneratorFunction","prototype"],"%GeneratorPrototype%":["GeneratorFunction","prototype","prototype"],"%Int8ArrayPrototype%":["Int8Array","prototype"],"%Int16ArrayPrototype%":["Int16Array","prototype"],"%Int32ArrayPrototype%":["Int32Array","prototype"],"%JSONParse%":["JSON","parse"],"%JSONStringify%":["JSON","stringify"],"%MapPrototype%":["Map","prototype"],"%NumberPrototype%":["Number","prototype"],"%ObjectPrototype%":["Object","prototype"],"%ObjProto_toString%":["Object","prototype","toString"],"%ObjProto_valueOf%":["Object","prototype","valueOf"],"%PromisePrototype%":["Promise","prototype"],"%PromiseProto_then%":["Promise","prototype","then"],"%Promise_all%":["Promise","all"],"%Promise_reject%":["Promise","reject"],"%Promise_resolve%":["Promise","resolve"],"%RangeErrorPrototype%":["RangeError","prototype"],"%ReferenceErrorPrototype%":["ReferenceError","prototype"],"%RegExpPrototype%":["RegExp","prototype"],"%SetPrototype%":["Set","prototype"],"%SharedArrayBufferPrototype%":["SharedArrayBuffer","prototype"],"%StringPrototype%":["String","prototype"],"%SymbolPrototype%":["Symbol","prototype"],"%SyntaxErrorPrototype%":["SyntaxError","prototype"],"%TypedArrayPrototype%":["TypedArray","prototype"],"%TypeErrorPrototype%":["TypeError","prototype"],"%Uint8ArrayPrototype%":["Uint8Array","prototype"],"%Uint8ClampedArrayPrototype%":["Uint8ClampedArray","prototype"],"%Uint16ArrayPrototype%":["Uint16Array","prototype"],"%Uint32ArrayPrototype%":["Uint32Array","prototype"],"%URIErrorPrototype%":["URIError","prototype"],"%WeakMapPrototype%":["WeakMap","prototype"],"%WeakSetPrototype%":["WeakSet","prototype"]},D=o(193),R=o(482),N=D.call(Function.call,Array.prototype.concat),U=D.call(Function.apply,Array.prototype.splice),W=D.call(Function.call,String.prototype.replace),G=D.call(Function.call,String.prototype.slice),j=/[^%.[\]]+|\[(?:(-?\d+(?:\.\d+)?)|(["'])((?:(?!\2)[^\\]|\\.)*?)\2)\]|(?=(?:\.|\[\])(?:\.|\[\]|%$))/g,X=/\\(\\)?/g,Y=function stringToPath(n){var i=G(n,0,1),o=G(n,-1);if("%"===i&&"%"!==o)throw new u("invalid intrinsic syntax, expected closing `%`");if("%"===o&&"%"!==i)throw new u("invalid intrinsic syntax, expected opening `%`");var p=[];return W(n,j,(function(n,i,o,u){p[p.length]=o?W(u,X,"$1"):i||n})),p},K=function getBaseIntrinsic(n,i){var o,p=n;if(R(I,p)&&(p="%"+(o=I[p])[0]+"%"),R(B,p)){var y=B[p];if(void 0===y&&!i)throw new g("intrinsic "+n+" exists, but is not available. Please file an issue!");return{alias:o,name:p,value:y}}throw new u("intrinsic "+n+" does not exist!")};n.exports=function GetIntrinsic(n,i){if("string"!=typeof n||0===n.length)throw new g("intrinsic name must be a non-empty string");if(arguments.length>1&&"boolean"!=typeof i)throw new g('"allowMissing" argument must be a boolean');var o=Y(n),p=o.length>0?o[0]:"",w=K("%"+p+"%",i),x=w.name,_=w.value,k=!1,P=w.alias;P&&(p=P[0],U(o,N([0,1],P)));for(var E=1,O=!0;E<o.length;E+=1){var I=o[E],D=G(I,0,1),W=G(I,-1);if(('"'===D||"'"===D||"`"===D||'"'===W||"'"===W||"`"===W)&&D!==W)throw new u("property names with quotes must have matching quotes");if("constructor"!==I&&O||(k=!0),R(B,x="%"+(p+="."+I)+"%"))_=B[x];else if(null!=_){if(!(I in _)){if(!i)throw new g("base intrinsic for "+n+" exists, but the property is not available.");return}if(y&&E+1>=o.length){var j=y(_,I);_=(O=!!j)&&"get"in j&&!("originalValue"in j.get)?j.get:_[I]}else O=R(_,I),_=_[I];O&&!k&&(B[x]=_)}}return _}},function(n,i,o){"use strict";(function(i){var u=i.Symbol,p=o(480);n.exports=function hasNativeSymbols(){return"function"==typeof u&&("function"==typeof Symbol&&("symbol"==typeof u("foo")&&("symbol"==typeof Symbol("bar")&&p())))}}).call(this,o(24))},function(n,i,o){"use strict";var numberIsNaN=function(n){return n!=n};n.exports=function is(n,i){return 0===n&&0===i?1/n==1/i:n===i||!(!numberIsNaN(n)||!numberIsNaN(i))}},function(n,i,o){"use strict";var u=o(277);n.exports=function getPolyfill(){return"function"==typeof Object.is?Object.is:u}},function(n,i,o){"use strict";var u=Object,p=TypeError;n.exports=function flags(){if(null!=this&&this!==u(this))throw new p("RegExp.prototype.flags getter called on non-object");var n="";return this.global&&(n+="g"),this.ignoreCase&&(n+="i"),this.multiline&&(n+="m"),this.dotAll&&(n+="s"),this.unicode&&(n+="u"),this.sticky&&(n+="y"),n}},function(n,i,o){"use strict";var u=o(279),p=o(108).supportsDescriptors,g=Object.getOwnPropertyDescriptor,y=TypeError;n.exports=function getPolyfill(){if(!p)throw new y("RegExp.prototype.flags requires a true ES5 environment that supports property descriptors");if("gim"===/a/gim.flags){var n=g(RegExp.prototype,"flags");if(n&&"function"==typeof n.get&&"boolean"==typeof/a/.dotAll)return n.get}return u}},function(n,i,o){var u=o(21)("iterator"),p=!1;try{var g=[7][u]();g.return=function(){p=!0},Array.from(g,(function(){throw 2}))}catch(n){}n.exports=function(n,i){if(!i&&!p)return!1;var o=!1;try{var g=[7],y=g[u]();y.next=function(){return{done:o=!0}},g[u]=function(){return y},n(g)}catch(n){}return o}},function(n,i,o){var u=o(29),p=o(101),g=o(21)("species");n.exports=function(n,i){var o,y=u(n).constructor;return void 0===y||null==(o=u(y)[g])?i:p(o)}},function(n,i,o){var u,p,g,y=o(32),w=o(521),x=o(261),_=o(177),k=o(17),P=k.process,E=k.setImmediate,O=k.clearImmediate,B=k.MessageChannel,I=k.Dispatch,D=0,R={},run=function(){var n=+this;if(R.hasOwnProperty(n)){var i=R[n];delete R[n],i()}},listener=function(n){run.call(n.data)};E&&O||(E=function setImmediate(n){for(var i=[],o=1;arguments.length>o;)i.push(arguments[o++]);return R[++D]=function(){w("function"==typeof n?n:Function(n),i)},u(D),D},O=function clearImmediate(n){delete R[n]},"process"==o(99)(P)?u=function(n){P.nextTick(y(run,n,1))}:I&&I.now?u=function(n){I.now(y(run,n,1))}:B?(g=(p=new B).port2,p.port1.onmessage=listener,u=y(g.postMessage,g,1)):k.addEventListener&&"function"==typeof postMessage&&!k.importScripts?(u=function(n){k.postMessage(n+"","*")},k.addEventListener("message",listener,!1)):u="onreadystatechange"in _("script")?function(n){x.appendChild(_("script")).onreadystatechange=function(){x.removeChild(this),run.call(n)}}:function(n){setTimeout(y(run,n,1),0)}),n.exports={set:E,clear:O}},function(n,i){n.exports=function(n){try{return{e:!1,v:n()}}catch(n){return{e:!0,v:n}}}},function(n,i,o){var u=o(29),p=o(22),g=o(196);n.exports=function(n,i){if(u(n),p(i)&&i.constructor===n)return i;var o=g.f(n);return(0,o.resolve)(i),o.promise}},function(n,i,o){var u=o(287).BrotliInput,p=o(287).BrotliOutput,g=o(527),y=o(288),w=o(289).HuffmanCode,x=o(289).BrotliBuildHuffmanTable,_=o(531),k=o(532),P=o(533),E=new Uint8Array([1,2,3,4,0,5,17,6,16,7,8,9,10,11,12,13,14,15]),O=new Uint8Array([3,2,1,0,3,3,3,3,3,3,2,2,2,2,2,2]),B=new Int8Array([0,0,0,0,-1,1,-2,2,-3,3,-1,1,-2,2,-3,3]),I=new Uint16Array([256,402,436,468,500,534,566,598,630,662,694,726,758,790,822,854,886,920,952,984,1016,1048,1080]);function DecodeWindowBits(n){var i;return 0===n.readBits(1)?16:(i=n.readBits(3))>0?17+i:(i=n.readBits(3))>0?8+i:17}function DecodeVarLenUint8(n){if(n.readBits(1)){var i=n.readBits(3);return 0===i?1:n.readBits(i)+(1<<i)}return 0}function MetaBlockLength(){this.meta_block_length=0,this.input_end=0,this.is_uncompressed=0,this.is_metadata=!1}function DecodeMetaBlockLength(n){var i,o,u,p=new MetaBlockLength;if(p.input_end=n.readBits(1),p.input_end&&n.readBits(1))return p;if(7===(i=n.readBits(2)+4)){if(p.is_metadata=!0,0!==n.readBits(1))throw new Error("Invalid reserved bit");if(0===(o=n.readBits(2)))return p;for(u=0;u<o;u++){var g=n.readBits(8);if(u+1===o&&o>1&&0===g)throw new Error("Invalid size byte");p.meta_block_length|=g<<8*u}}else for(u=0;u<i;++u){var y=n.readBits(4);if(u+1===i&&i>4&&0===y)throw new Error("Invalid size nibble");p.meta_block_length|=y<<4*u}return++p.meta_block_length,p.input_end||p.is_metadata||(p.is_uncompressed=n.readBits(1)),p}function ReadSymbol(n,i,o){var u;return o.fillBitWindow(),(u=n[i+=o.val_>>>o.bit_pos_&255].bits-8)>0&&(o.bit_pos_+=8,i+=n[i].value,i+=o.val_>>>o.bit_pos_&(1<<u)-1),o.bit_pos_+=n[i].bits,n[i].value}function ReadHuffmanCode(n,i,o,u){var p,g,y=new Uint8Array(n);if(u.readMoreInput(),1===(g=u.readBits(2))){for(var _=n-1,k=0,P=new Int32Array(4),O=u.readBits(2)+1;_;)_>>=1,++k;for(B=0;B<O;++B)P[B]=u.readBits(k)%n,y[P[B]]=2;switch(y[P[0]]=1,O){case 1:break;case 3:if(P[0]===P[1]||P[0]===P[2]||P[1]===P[2])throw new Error("[ReadHuffmanCode] invalid symbols");break;case 2:if(P[0]===P[1])throw new Error("[ReadHuffmanCode] invalid symbols");y[P[1]]=1;break;case 4:if(P[0]===P[1]||P[0]===P[2]||P[0]===P[3]||P[1]===P[2]||P[1]===P[3]||P[2]===P[3])throw new Error("[ReadHuffmanCode] invalid symbols");u.readBits(1)?(y[P[2]]=3,y[P[3]]=3):y[P[0]]=2}}else{var B,I=new Uint8Array(18),D=32,R=0,N=[new w(2,0),new w(2,4),new w(2,3),new w(3,2),new w(2,0),new w(2,4),new w(2,3),new w(4,1),new w(2,0),new w(2,4),new w(2,3),new w(3,2),new w(2,0),new w(2,4),new w(2,3),new w(4,5)];for(B=g;B<18&&D>0;++B){var U,W=E[B],G=0;u.fillBitWindow(),G+=u.val_>>>u.bit_pos_&15,u.bit_pos_+=N[G].bits,U=N[G].value,I[W]=U,0!==U&&(D-=32>>U,++R)}if(1!==R&&0!==D)throw new Error("[ReadHuffmanCode] invalid num_codes or space");!function ReadHuffmanCodeLengths(n,i,o,u){for(var p=0,g=8,y=0,_=0,k=32768,P=[],E=0;E<32;E++)P.push(new w(0,0));for(x(P,0,5,n,18);p<i&&k>0;){var O,B=0;if(u.readMoreInput(),u.fillBitWindow(),B+=u.val_>>>u.bit_pos_&31,u.bit_pos_+=P[B].bits,(O=255&P[B].value)<16)y=0,o[p++]=O,0!==O&&(g=O,k-=32768>>O);else{var I,D,R=O-14,N=0;if(16===O&&(N=g),_!==N&&(y=0,_=N),I=y,y>0&&(y-=2,y<<=R),p+(D=(y+=u.readBits(R)+3)-I)>i)throw new Error("[ReadHuffmanCodeLengths] symbol + repeat_delta > num_symbols");for(var U=0;U<D;U++)o[p+U]=_;p+=D,0!==_&&(k-=D<<15-_)}}if(0!==k)throw new Error("[ReadHuffmanCodeLengths] space = "+k);for(;p<i;p++)o[p]=0}(I,n,y,u)}if(0===(p=x(i,o,8,y,n)))throw new Error("[ReadHuffmanCode] BuildHuffmanTable failed: ");return p}function ReadBlockLength(n,i,o){var u,p;return u=ReadSymbol(n,i,o),p=k.kBlockLengthPrefixCode[u].nbits,k.kBlockLengthPrefixCode[u].offset+o.readBits(p)}function TranslateShortCodes(n,i,o){var u;return n<16?(o+=O[n],u=i[o&=3]+B[n]):u=n-16+1,u}function MoveToFront(n,i){for(var o=n[i],u=i;u;--u)n[u]=n[u-1];n[0]=o}function HuffmanTreeGroup(n,i){this.alphabet_size=n,this.num_htrees=i,this.codes=new Array(i+i*I[n+31>>>5]),this.htrees=new Uint32Array(i)}function DecodeContextMap(n,i){var o,u,p={num_htrees:null,context_map:null},g=0;i.readMoreInput();var y=p.num_htrees=DecodeVarLenUint8(i)+1,x=p.context_map=new Uint8Array(n);if(y<=1)return p;for(i.readBits(1)&&(g=i.readBits(4)+1),o=[],u=0;u<1080;u++)o[u]=new w(0,0);for(ReadHuffmanCode(y+g,o,0,i),u=0;u<n;){var _;if(i.readMoreInput(),0===(_=ReadSymbol(o,0,i)))x[u]=0,++u;else if(_<=g)for(var k=1+(1<<_)+i.readBits(_);--k;){if(u>=n)throw new Error("[DecodeContextMap] i >= context_map_size");x[u]=0,++u}else x[u]=_-g,++u}return i.readBits(1)&&function InverseMoveToFrontTransform(n,i){var o,u=new Uint8Array(256);for(o=0;o<256;++o)u[o]=o;for(o=0;o<i;++o){var p=n[o];n[o]=u[p],p&&MoveToFront(u,p)}}(x,n),p}function DecodeBlockType(n,i,o,u,p,g,y){var w,x=2*o,_=o,k=ReadSymbol(i,1080*o,y);(w=0===k?p[x+(1&g[_])]:1===k?p[x+(g[_]-1&1)]+1:k-2)>=n&&(w-=n),u[o]=w,p[x+(1&g[_])]=w,++g[_]}function CopyUncompressedBlockToOutput(n,i,o,u,p,y){var w,x=p+1,_=o&p,k=y.pos_&g.IBUF_MASK;if(i<8||y.bit_pos_+(i<<3)<y.bit_end_pos_)for(;i-- >0;)y.readMoreInput(),u[_++]=y.readBits(8),_===x&&(n.write(u,x),_=0);else{if(y.bit_end_pos_<32)throw new Error("[CopyUncompressedBlockToOutput] br.bit_end_pos_ < 32");for(;y.bit_pos_<32;)u[_]=y.val_>>>y.bit_pos_,y.bit_pos_+=8,++_,--i;if(k+(w=y.bit_end_pos_-y.bit_pos_>>3)>g.IBUF_MASK){for(var P=g.IBUF_MASK+1-k,E=0;E<P;E++)u[_+E]=y.buf_[k+E];w-=P,_+=P,i-=P,k=0}for(E=0;E<w;E++)u[_+E]=y.buf_[k+E];if(i-=w,(_+=w)>=x){n.write(u,x),_-=x;for(E=0;E<_;E++)u[E]=u[x+E]}for(;_+i>=x;){if(w=x-_,y.input_.read(u,_,w)<w)throw new Error("[CopyUncompressedBlockToOutput] not enough bytes");n.write(u,x),i-=w,_=0}if(y.input_.read(u,_,i)<i)throw new Error("[CopyUncompressedBlockToOutput] not enough bytes");y.reset()}}function JumpToByteBoundary(n){var i=n.bit_pos_+7&-8;return 0==n.readBits(i-n.bit_pos_)}function BrotliDecompressedSize(n){var i=new u(n),o=new g(i);return DecodeWindowBits(o),DecodeMetaBlockLength(o).meta_block_length}function BrotliDecompress(n,i){var o,u,p,x,E,O,B,I,D,R,N=0,U=0,W=0,G=[16,15,11,4],j=0,X=0,Y=0,K=[new HuffmanTreeGroup(0,0),new HuffmanTreeGroup(0,0),new HuffmanTreeGroup(0,0)],J=128+g.READ_SIZE;p=(1<<(u=DecodeWindowBits(R=new g(n))))-16,E=(x=1<<u)-1,O=new Uint8Array(x+J+y.maxDictionaryWordLength),B=x,I=[],D=[];for(var $=0;$<3240;$++)I[$]=new w(0,0),D[$]=new w(0,0);for(;!U;){var tt,et,rt,nt,it,ot,at,st,lt,ct,ut,ft=0,ht=[1<<28,1<<28,1<<28],dt=[0],pt=[1,1,1],gt=[0,1,0,1,0,1],yt=[0],vt=null,mt=null,bt=0,wt=null,xt=0,_t=0,St=0;for(o=0;o<3;++o)K[o].codes=null,K[o].htrees=null;R.readMoreInput();var Ct=DecodeMetaBlockLength(R);if(N+(ft=Ct.meta_block_length)>i.buffer.length){var kt=new Uint8Array(N+ft);kt.set(i.buffer),i.buffer=kt}if(U=Ct.input_end,tt=Ct.is_uncompressed,Ct.is_metadata)for(JumpToByteBoundary(R);ft>0;--ft)R.readMoreInput(),R.readBits(8);else if(0!==ft)if(tt)R.bit_pos_=R.bit_pos_+7&-8,CopyUncompressedBlockToOutput(i,ft,N,O,E,R),N+=ft;else{for(o=0;o<3;++o)pt[o]=DecodeVarLenUint8(R)+1,pt[o]>=2&&(ReadHuffmanCode(pt[o]+2,I,1080*o,R),ReadHuffmanCode(26,D,1080*o,R),ht[o]=ReadBlockLength(D,1080*o,R),yt[o]=1);for(R.readMoreInput(),nt=(1<<(et=R.readBits(2)))-1,it=(rt=16+(R.readBits(4)<<et))+(48<<et),vt=new Uint8Array(pt[0]),o=0;o<pt[0];++o)R.readMoreInput(),vt[o]=R.readBits(2)<<1;var At=DecodeContextMap(pt[0]<<6,R);at=At.num_htrees,ot=At.context_map;var Pt=DecodeContextMap(pt[2]<<2,R);for(lt=Pt.num_htrees,st=Pt.context_map,K[0]=new HuffmanTreeGroup(256,at),K[1]=new HuffmanTreeGroup(704,pt[1]),K[2]=new HuffmanTreeGroup(it,lt),o=0;o<3;++o)K[o].decode(R);for(mt=0,wt=0,ct=vt[dt[0]],_t=_.lookupOffsets[ct],St=_.lookupOffsets[ct+1],ut=K[1].htrees[0];ft>0;){var Tt,Et,Ot,Bt,It,Ft,Lt,Dt,Mt,Rt,Nt,Ut;for(R.readMoreInput(),0===ht[1]&&(DecodeBlockType(pt[1],I,1,dt,gt,yt,R),ht[1]=ReadBlockLength(D,1080,R),ut=K[1].htrees[dt[1]]),--ht[1],(Et=(Tt=ReadSymbol(K[1].codes,ut,R))>>6)>=2?(Et-=2,Lt=-1):Lt=0,Ot=k.kInsertRangeLut[Et]+(Tt>>3&7),Bt=k.kCopyRangeLut[Et]+(7&Tt),It=k.kInsertLengthPrefixCode[Ot].offset+R.readBits(k.kInsertLengthPrefixCode[Ot].nbits),Ft=k.kCopyLengthPrefixCode[Bt].offset+R.readBits(k.kCopyLengthPrefixCode[Bt].nbits),X=O[N-1&E],Y=O[N-2&E],Mt=0;Mt<It;++Mt)R.readMoreInput(),0===ht[0]&&(DecodeBlockType(pt[0],I,0,dt,gt,yt,R),ht[0]=ReadBlockLength(D,0,R),mt=dt[0]<<6,ct=vt[dt[0]],_t=_.lookupOffsets[ct],St=_.lookupOffsets[ct+1]),bt=ot[mt+(_.lookup[_t+X]|_.lookup[St+Y])],--ht[0],Y=X,X=ReadSymbol(K[0].codes,K[0].htrees[bt],R),O[N&E]=X,(N&E)===E&&i.write(O,x),++N;if((ft-=It)<=0)break;if(Lt<0)if(R.readMoreInput(),0===ht[2]&&(DecodeBlockType(pt[2],I,2,dt,gt,yt,R),ht[2]=ReadBlockLength(D,2160,R),wt=dt[2]<<2),--ht[2],xt=st[wt+(255&(Ft>4?3:Ft-2))],(Lt=ReadSymbol(K[2].codes,K[2].htrees[xt],R))>=rt)Ut=(Lt-=rt)&nt,Lt=rt+((Wt=(2+(1&(Lt>>=et))<<(Nt=1+(Lt>>1)))-4)+R.readBits(Nt)<<et)+Ut;if((Dt=TranslateShortCodes(Lt,G,j))<0)throw new Error("[BrotliDecompress] invalid distance");if(Rt=N&E,Dt>(W=N<p&&W!==p?N:p)){if(!(Ft>=y.minDictionaryWordLength&&Ft<=y.maxDictionaryWordLength))throw new Error("Invalid backward reference. pos: "+N+" distance: "+Dt+" len: "+Ft+" bytes left: "+ft);var Wt=y.offsetsByLength[Ft],zt=Dt-W-1,Gt=y.sizeBitsByLength[Ft],jt=zt>>Gt;if(Wt+=(zt&(1<<Gt)-1)*Ft,!(jt<P.kNumTransforms))throw new Error("Invalid backward reference. pos: "+N+" distance: "+Dt+" len: "+Ft+" bytes left: "+ft);var Vt=P.transformDictionaryWord(O,Rt,Wt,Ft,jt);if(N+=Vt,ft-=Vt,(Rt+=Vt)>=B){i.write(O,x);for(var Ht=0;Ht<Rt-B;Ht++)O[Ht]=O[B+Ht]}}else{if(Lt>0&&(G[3&j]=Dt,++j),Ft>ft)throw new Error("Invalid backward reference. pos: "+N+" distance: "+Dt+" len: "+Ft+" bytes left: "+ft);for(Mt=0;Mt<Ft;++Mt)O[N&E]=O[N-Dt&E],(N&E)===E&&i.write(O,x),++N,--ft}X=O[N-1&E],Y=O[N-2&E]}N&=1073741823}}i.write(O,N&E)}HuffmanTreeGroup.prototype.decode=function(n){var i,o=0;for(i=0;i<this.num_htrees;++i)this.htrees[i]=o,o+=ReadHuffmanCode(this.alphabet_size,this.codes,o,n)},i.BrotliDecompressedSize=BrotliDecompressedSize,i.BrotliDecompressBuffer=function BrotliDecompressBuffer(n,i){var o=new u(n);null==i&&(i=BrotliDecompressedSize(n));var g=new Uint8Array(i),y=new p(g);return BrotliDecompress(o,y),y.pos<y.buffer.length&&(y.buffer=y.buffer.subarray(0,y.pos)),y.buffer},i.BrotliDecompress=BrotliDecompress,y.init()},function(n,i){function BrotliInput(n){this.buffer=n,this.pos=0}function BrotliOutput(n){this.buffer=n,this.pos=0}BrotliInput.prototype.read=function(n,i,o){this.pos+o>this.buffer.length&&(o=this.buffer.length-this.pos);for(var u=0;u<o;u++)n[i+u]=this.buffer[this.pos+u];return this.pos+=o,o},i.BrotliInput=BrotliInput,BrotliOutput.prototype.write=function(n,i){if(this.pos+i>this.buffer.length)throw new Error("Output buffer is not large enough");return this.buffer.set(n.subarray(0,i),this.pos),this.pos+=i,i},i.BrotliOutput=BrotliOutput},function(n,i,o){var u=o(528);i.init=function(){i.dictionary=u.init()},i.offsetsByLength=new Uint32Array([0,0,0,0,0,4096,9216,21504,35840,44032,53248,63488,74752,87040,93696,100864,104704,106752,108928,113536,115968,118528,119872,121280,122016]),i.sizeBitsByLength=new Uint8Array([0,0,0,0,10,10,11,11,10,10,10,10,10,9,9,8,7,7,8,7,7,6,6,5,5]),i.minDictionaryWordLength=4,i.maxDictionaryWordLength=24},function(n,i){function HuffmanCode(n,i){this.bits=n,this.value=i}i.HuffmanCode=HuffmanCode;function GetNextKey(n,i){for(var o=1<<i-1;n&o;)o>>=1;return(n&o-1)+o}function ReplicateValue(n,i,o,u,p){do{n[i+(u-=o)]=new HuffmanCode(p.bits,p.value)}while(u>0)}function NextTableBitSize(n,i,o){for(var u=1<<i-o;i<15&&!((u-=n[i])<=0);)++i,u<<=1;return i-o}i.BrotliBuildHuffmanTable=function(n,i,o,u,p){var g,y,w,x,_,k,P,E,O,B,I=i,D=new Int32Array(16),R=new Int32Array(16);for(B=new Int32Array(p),y=0;y<p;y++)D[u[y]]++;for(R[1]=0,g=1;g<15;g++)R[g+1]=R[g]+D[g];for(y=0;y<p;y++)0!==u[y]&&(B[R[u[y]]++]=y);if(O=E=1<<(P=o),1===R[15]){for(w=0;w<O;++w)n[i+w]=new HuffmanCode(0,65535&B[0]);return O}for(w=0,y=0,g=1,x=2;g<=o;++g,x<<=1)for(;D[g]>0;--D[g])ReplicateValue(n,i+w,x,E,new HuffmanCode(255&g,65535&B[y++])),w=GetNextKey(w,g);for(k=O-1,_=-1,g=o+1,x=2;g<=15;++g,x<<=1)for(;D[g]>0;--D[g])(w&k)!==_&&(i+=E,O+=E=1<<(P=NextTableBitSize(D,g,o)),n[I+(_=w&k)]=new HuffmanCode(P+o&255,i-I-_&65535)),ReplicateValue(n,i+(w>>o),x,E,new HuffmanCode(g-o&255,65535&B[y++])),w=GetNextKey(w,g);return O}},function(n,i,o){"use strict";var u,p,g,y,w,x,_,k,P,E,O,B,I,D,R,N=o(534),U=o(535),W=o(536);W.BK,_=W.CR,W.LF,W.NL,w=W.CB,g=W.BA,W.SP,D=W.WJ,I=W.SP,y=W.BK,k=W.LF,P=W.NL,u=W.AI,p=W.AL,O=W.SA,B=W.SG,R=W.XX,x=W.CJ,W.ID,E=W.NS;var G=o(537),j=G.DI_BRK,X=G.IN_BRK,Y=G.CI_BRK,K=G.CP_BRK,J=(G.PR_BRK,G.pairTable),$=new N(U.toByteArray("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")),tt=function mapClass(n){switch(n){case u:return p;case O:case B:case R:return p;case x:return E;default:return n}},et=function mapFirst(n){switch(n){case k:case P:return y;case w:return g;case I:return D;default:return n}},rt=function Break(n,i){void 0===i&&(i=!1),this.position=n,this.required=i},nt=function(){function LineBreaker(n){this.string=n,this.pos=0,this.lastPos=0,this.curClass=null,this.nextClass=null}var n=LineBreaker.prototype;return n.nextCodePoint=function nextCodePoint(){var n=this.string.charCodeAt(this.pos++),i=this.string.charCodeAt(this.pos);return 55296<=n&&n<=56319&&56320<=i&&i<=57343?(this.pos++,1024*(n-55296)+(i-56320)+65536):n},n.nextCharClass=function nextCharClass(){return tt($.get(this.nextCodePoint()))},n.nextBreak=function nextBreak(){for(null==this.curClass&&(this.curClass=et(this.nextCharClass()));this.pos<this.string.length;){this.lastPos=this.pos;var n=this.nextClass;if(this.nextClass=this.nextCharClass(),this.curClass===y||this.curClass===_&&this.nextClass!==k)return this.curClass=et(tt(this.nextClass)),new rt(this.lastPos,!0);var i=void 0;switch(this.nextClass){case I:i=this.curClass;break;case y:case k:case P:i=y;break;case _:i=_;break;case w:i=g}if(null==i){var o=!1;switch(J[this.curClass][this.nextClass]){case j:o=!0;break;case X:o=n===I;break;case Y:if(!(o=n===I))continue;break;case K:if(n!==I)continue}if(this.curClass=this.nextClass,o)return new rt(this.lastPos)}else if(this.curClass=i,this.nextClass===w)return new rt(this.lastPos)}if(this.pos>=this.string.length)return this.lastPos<this.string.length?(this.lastPos=this.string.length,new rt(this.string.length)):null},LineBreaker}();n.exports=nt},function(n,i,o){"use strict";function TraversalTracker(){this.events={}}TraversalTracker.prototype.startTracking=function(n,i){var o=this.events[n]||(this.events[n]=[]);o.indexOf(i)<0&&o.push(i)},TraversalTracker.prototype.stopTracking=function(n,i){var o=this.events[n];if(o){var u=o.indexOf(i);u>=0&&o.splice(u,1)}},TraversalTracker.prototype.emit=function(n){var i=Array.prototype.slice.call(arguments,1),o=this.events[n];o&&o.forEach((function(n){n.apply(this,i)}))},TraversalTracker.prototype.auto=function(n,i,o){this.startTracking(n,i),o(),this.stopTracking(n,i)},n.exports=TraversalTracker},function(n,i,o){"use strict";var u=o(0).isString,p=o(0).isArray,g=o(0).isUndefined,y=o(0).isNull;function StyleContextStack(n,i){this.defaultStyle=i||{},this.styleDictionary=n,this.styleOverrides=[]}StyleContextStack.prototype.clone=function(){var n=new StyleContextStack(this.styleDictionary,this.defaultStyle);return this.styleOverrides.forEach((function(i){n.styleOverrides.push(i)})),n},StyleContextStack.prototype.push=function(n){this.styleOverrides.push(n)},StyleContextStack.prototype.pop=function(n){for(n=n||1;n-- >0;)this.styleOverrides.pop()},StyleContextStack.prototype.autopush=function(n){if(u(n))return 0;var i=[];n.style&&(i=p(n.style)?n.style:[n.style]);for(var o=0,w=i.length;o<w;o++)this.push(i[o]);var x={},_=!1;return["font","fontSize","fontFeatures","bold","italics","alignment","color","columnGap","fillColor","fillOpacity","decoration","decorationStyle","decorationColor","background","lineHeight","characterSpacing","noWrap","markerColor","leadingIndent","sup","sub"].forEach((function(i){g(n[i])||y(n[i])||(x[i]=n[i],_=!0)})),_&&this.push(x),i.length+(_?1:0)},StyleContextStack.prototype.auto=function(n,i){var o=this.autopush(n),u=i();return o>0&&this.pop(o),u},StyleContextStack.prototype.getProperty=function(n){if(this.styleOverrides)for(var i=this.styleOverrides.length-1;i>=0;i--){var o=this.styleOverrides[i];if(u(o)){var p=this.styleDictionary[o];if(p&&!g(p[n])&&!y(p[n]))return p[n]}else if(!g(o[n])&&!y(o[n]))return o[n]}return this.defaultStyle&&this.defaultStyle[n]},n.exports=StyleContextStack},function(n,i,o){"use strict";var u=o(291),p=o(0).isString;function DocumentContext(n,i){this.pages=[],this.pageMargins=i,this.x=i.left,this.availableWidth=n.width-i.left-i.right,this.availableHeight=0,this.page=-1,this.snapshots=[],this.endingCell=null,this.tracker=new u,this.backgroundLength=[],this.addPage(n)}DocumentContext.prototype.beginColumnGroup=function(){this.snapshots.push({x:this.x,y:this.y,availableHeight:this.availableHeight,availableWidth:this.availableWidth,page:this.page,bottomMost:{x:this.x,y:this.y,availableHeight:this.availableHeight,availableWidth:this.availableWidth,page:this.page},endingCell:this.endingCell,lastColumnWidth:this.lastColumnWidth}),this.lastColumnWidth=0},DocumentContext.prototype.beginColumn=function(n,i,o){var u=this.snapshots[this.snapshots.length-1];this.calculateBottomMost(u),this.endingCell=o,this.page=u.page,this.x=this.x+this.lastColumnWidth+(i||0),this.y=u.y,this.availableWidth=n,this.availableHeight=u.availableHeight,this.lastColumnWidth=n},DocumentContext.prototype.calculateBottomMost=function(n){this.endingCell?(this.saveContextInEndingCell(this.endingCell),this.endingCell=null):n.bottomMost=function bottomMostContext(n,i){var o;o=n.page>i.page?n:i.page>n.page?i:n.y>i.y?n:i;return{page:o.page,x:o.x,y:o.y,availableHeight:o.availableHeight,availableWidth:o.availableWidth}}(this,n.bottomMost)},DocumentContext.prototype.markEnding=function(n){this.page=n._columnEndingContext.page,this.x=n._columnEndingContext.x,this.y=n._columnEndingContext.y,this.availableWidth=n._columnEndingContext.availableWidth,this.availableHeight=n._columnEndingContext.availableHeight,this.lastColumnWidth=n._columnEndingContext.lastColumnWidth},DocumentContext.prototype.saveContextInEndingCell=function(n){n._columnEndingContext={page:this.page,x:this.x,y:this.y,availableHeight:this.availableHeight,availableWidth:this.availableWidth,lastColumnWidth:this.lastColumnWidth}},DocumentContext.prototype.completeColumnGroup=function(n){var i=this.snapshots.pop();this.calculateBottomMost(i),this.endingCell=null,this.x=i.x;var o=i.bottomMost.y;n&&(i.page===i.bottomMost.page?i.y+n>o&&(o=i.y+n):o+=n),this.y=o,this.page=i.bottomMost.page,this.availableWidth=i.availableWidth,this.availableHeight=i.bottomMost.availableHeight,n&&(this.availableHeight-=o-i.bottomMost.y),this.lastColumnWidth=i.lastColumnWidth},DocumentContext.prototype.addMargin=function(n,i){this.x+=n,this.availableWidth-=n+(i||0)},DocumentContext.prototype.moveDown=function(n){return this.y+=n,this.availableHeight-=n,this.availableHeight>0},DocumentContext.prototype.initializePage=function(){this.y=this.pageMargins.top,this.availableHeight=this.getCurrentPage().pageSize.height-this.pageMargins.top-this.pageMargins.bottom,this.pageSnapshot().availableWidth=this.getCurrentPage().pageSize.width-this.pageMargins.left-this.pageMargins.right},DocumentContext.prototype.pageSnapshot=function(){return this.snapshots[0]?this.snapshots[0]:this},DocumentContext.prototype.moveTo=function(n,i){null!=n&&(this.x=n,this.availableWidth=this.getCurrentPage().pageSize.width-this.x-this.pageMargins.right),null!=i&&(this.y=i,this.availableHeight=this.getCurrentPage().pageSize.height-this.y-this.pageMargins.bottom)},DocumentContext.prototype.moveToRelative=function(n,i){null!=n&&(this.x=this.x+n),null!=i&&(this.y=this.y+i)},DocumentContext.prototype.beginDetachedBlock=function(){this.snapshots.push({x:this.x,y:this.y,availableHeight:this.availableHeight,availableWidth:this.availableWidth,page:this.page,endingCell:this.endingCell,lastColumnWidth:this.lastColumnWidth})},DocumentContext.prototype.endDetachedBlock=function(){var n=this.snapshots.pop();this.x=n.x,this.y=n.y,this.availableWidth=n.availableWidth,this.availableHeight=n.availableHeight,this.page=n.page,this.endingCell=n.endingCell,this.lastColumnWidth=n.lastColumnWidth};var getPageSize=function(n,i){return(i=function pageOrientation(n,i){return void 0===n?i:p(n)&&"landscape"===n.toLowerCase()?"landscape":"portrait"}(i,n.pageSize.orientation))!==n.pageSize.orientation?{orientation:i,width:n.pageSize.height,height:n.pageSize.width}:{orientation:n.pageSize.orientation,width:n.pageSize.width,height:n.pageSize.height}};DocumentContext.prototype.moveToNextPage=function(n){var i=this.page+1,o=this.page,u=this.y,p=i>=this.pages.length;if(p){var g=this.availableWidth,y=this.getCurrentPage().pageSize.orientation,w=getPageSize(this.getCurrentPage(),n);this.addPage(w),y===w.orientation&&(this.availableWidth=g)}else this.page=i,this.initializePage();return{newPageCreated:p,prevPage:o,prevY:u,y:this.y}},DocumentContext.prototype.addPage=function(n){var i={items:[],pageSize:n};return this.pages.push(i),this.backgroundLength.push(0),this.page=this.pages.length-1,this.initializePage(),this.tracker.emit("pageAdded"),i},DocumentContext.prototype.getCurrentPage=function(){return this.page<0||this.page>=this.pages.length?null:this.pages[this.page]},DocumentContext.prototype.getCurrentPosition=function(){var n=this.getCurrentPage().pageSize,i=n.height-this.pageMargins.top-this.pageMargins.bottom,o=n.width-this.pageMargins.left-this.pageMargins.right;return{pageNumber:this.page+1,pageOrientation:n.orientation,pageInnerHeight:i,pageInnerWidth:o,left:this.x,top:this.y,verticalRatio:(this.y-this.pageMargins.top)/i,horizontalRatio:(this.x-this.pageMargins.left)/o}},n.exports=DocumentContext},function(n,i,o){"use strict";function Line(n){this.maxWidth=n,this.leadingCut=0,this.trailingCut=0,this.inlineWidths=0,this.inlines=[]}Line.prototype.getAscenderHeight=function(){var n=0;return this.inlines.forEach((function(i){n=Math.max(n,i.font.ascender/1e3*i.fontSize)})),n},Line.prototype.hasEnoughSpaceForInline=function(n,i){if(i=i||[],0===this.inlines.length)return!0;if(this.newLineForced)return!1;var o=n.width,u=n.trailingCut||0;if(n.noNewLine)for(var p=0,g=i.length;p<g;p++){var y=i[p];if(o+=y.width,u+=y.trailingCut||0,!y.noNewLine)break}return this.inlineWidths+o-this.leadingCut-u<=this.maxWidth},Line.prototype.addInline=function(n){0===this.inlines.length&&(this.leadingCut=n.leadingCut||0),this.trailingCut=n.trailingCut||0,n.x=this.inlineWidths-this.leadingCut,this.inlines.push(n),this.inlineWidths+=n.width,n.lineEnd&&(this.newLineForced=!0)},Line.prototype.getWidth=function(){return this.inlineWidths-this.leadingCut-this.trailingCut},Line.prototype.getAvailableWidth=function(){return this.maxWidth-this.getWidth()},Line.prototype.getHeight=function(){var n=0;return this.inlines.forEach((function(i){n=Math.max(n,i.height||0)})),n},n.exports=Line},function(n,i,o){"use strict";var u=o(1),p=o(60),g=o(45),y=o(58),w=o(18),x=o(40),_=o(136),k=function AggregateError(n,i){var o=this;if(!(o instanceof k))return new k(n,i);g&&(o=g(new Error(void 0),p(o))),void 0!==i&&w(o,"message",String(i));var u=[];return _(n,u.push,{that:u}),w(o,"errors",u),o};k.prototype=y(Error.prototype,{constructor:x(5,k),message:x(5,""),name:x(5,"AggregateError")}),u({global:!0},{AggregateError:k})},function(n,i,o){var u=o(3);n.exports=u.Promise},function(n,i,o){var u,p,g,y=o(3),w=o(4),x=o(96),_=o(206),k=o(138),P=o(298),E=o(199),O=y.location,B=y.setImmediate,I=y.clearImmediate,D=y.process,R=y.MessageChannel,N=y.Dispatch,U=0,W={},run=function(n){if(W.hasOwnProperty(n)){var i=W[n];delete W[n],i()}},runner=function(n){return function(){run(n)}},listener=function(n){run(n.data)},post=function(n){y.postMessage(n+"",O.protocol+"//"+O.host)};B&&I||(B=function setImmediate(n){for(var i=[],o=1;arguments.length>o;)i.push(arguments[o++]);return W[++U]=function(){("function"==typeof n?n:Function(n)).apply(void 0,i)},u(U),U},I=function clearImmediate(n){delete W[n]},E?u=function(n){D.nextTick(runner(n))}:N&&N.now?u=function(n){N.now(runner(n))}:R&&!P?(g=(p=new R).port2,p.port1.onmessage=listener,u=x(g.postMessage,g,1)):y.addEventListener&&"function"==typeof postMessage&&!y.importScripts&&O&&"file:"!==O.protocol&&!w(post)?(u=post,y.addEventListener("message",listener,!1)):u="onreadystatechange"in k("script")?function(n){_.appendChild(k("script")).onreadystatechange=function(){_.removeChild(this),run(n)}}:function(n){setTimeout(runner(n),0)}),n.exports={set:B,clear:I}},function(n,i,o){var u=o(149);n.exports=/(iphone|ipod|ipad).*applewebkit/i.test(u)},function(n,i,o){var u=o(11),p=o(14),g=o(109);n.exports=function(n,i){if(u(n),p(i)&&i.constructor===n)return i;var o=g.f(n);return(0,o.resolve)(i),o.promise}},function(n,i,o){"use strict";var u=o(1),p=o(44),g=o(109),y=o(137),w=o(136);u({target:"Promise",stat:!0},{allSettled:function allSettled(n){var i=this,o=g.f(i),u=o.resolve,x=o.reject,_=y((function(){var o=p(i.resolve),g=[],y=0,x=1;w(n,(function(n){var p=y++,w=!1;g.push(void 0),x++,o.call(i,n).then((function(n){w||(w=!0,g[p]={status:"fulfilled",value:n},--x||u(g))}),(function(n){w||(w=!0,g[p]={status:"rejected",reason:n},--x||u(g))}))})),--x||u(g)}));return _.error&&x(_.value),o.promise}})},function(n,i,o){"use strict";var u=o(1),p=o(44),g=o(34),y=o(109),w=o(137),x=o(136);u({target:"Promise",stat:!0},{any:function any(n){var i=this,o=y.f(i),u=o.resolve,_=o.reject,k=w((function(){var o=p(i.resolve),y=[],w=0,k=1,P=!1;x(n,(function(n){var p=w++,x=!1;y.push(void 0),k++,o.call(i,n).then((function(n){x||P||(P=!0,u(n))}),(function(n){x||P||(x=!0,y[p]=n,--k||_(new(g("AggregateError"))(y,"No one promise resolved")))}))})),--k||_(new(g("AggregateError"))(y,"No one promise resolved"))}));return k.error&&_(k.value),o.promise}})},function(n,i,o){var u=o(303),p=o(571);void 0===p.pdfMake&&(p.pdfMake=u),n.exports=u},function(n,i,o){"use strict";(function(i,u){var p=o(0).isFunction,g=o(0).isUndefined,y=(o(0).isNull,o(306).saveAs),w={Roboto:{normal:"Roboto-Regular.ttf",bold:"Roboto-Medium.ttf",italics:"Roboto-Italic.ttf",bolditalics:"Roboto-MediumItalic.ttf"}};function Document(n,i,o,u){this.docDefinition=n,this.tableLayouts=i||null,this.fonts=o||w,this.vfs=u}Document.prototype._createDoc=function(n,i){n=n||{},this.tableLayouts&&(n.tableLayouts=this.tableLayouts);var u=new(o(307))(this.fonts);if(o(53).bindFS(this.vfs),!p(i))return u.createPdfKitDocument(this.docDefinition,n);var g=new(o(556))(o(53));for(var y in this.fonts)this.fonts.hasOwnProperty(y)&&(this.fonts[y].normal&&g.resolve(this.fonts[y].normal),this.fonts[y].bold&&g.resolve(this.fonts[y].bold),this.fonts[y].italics&&g.resolve(this.fonts[y].italics),this.fonts[y].bolditalics&&g.resolve(this.fonts[y].bolditalics));if(this.docDefinition.images)for(var w in this.docDefinition.images)this.docDefinition.images.hasOwnProperty(w)&&g.resolve(this.docDefinition.images[w]);var x=this;g.resolved().then((function(){var o=u.createPdfKitDocument(x.docDefinition,n);i(o)}),(function(n){throw n}))},Document.prototype._flushDoc=function(n,o){var u,p=[];n.on("readable",(function(){for(var i;null!==(i=n.read(9007199254740991));)p.push(i)})),n.on("end",(function(){u=i.concat(p),o(u,n._pdfMakePages)})),n.end()},Document.prototype._getPages=function(n,i){if(!i)throw"_getPages is an async method and needs a callback argument";var o=this;this._createDoc(n,(function(n){o._flushDoc(n,(function(n,o){i(o)}))}))},Document.prototype._bufferToBlob=function(n){var i;try{i=new Blob([n],{type:"application/pdf"})}catch(u){if("InvalidStateError"===u.name){var o=new Uint8Array(n);i=new Blob([o.buffer],{type:"application/pdf"})}}if(!i)throw"Could not generate blob";return i},Document.prototype._openWindow=function(){var n=window.open("","_blank");if(null===n)throw"Open PDF in new window blocked by browser";return n},Document.prototype._openPdf=function(n,i){i||(i=this._openWindow());try{this.getBlob((function(n){var o=(window.URL||window.webkitURL).createObjectURL(n);i.location.href=o}),n)}catch(n){throw i.close(),n}},Document.prototype.open=function(n,i){(n=n||{}).autoPrint=!1,i=i||null,this._openPdf(n,i)},Document.prototype.print=function(n,i){(n=n||{}).autoPrint=!0,i=i||null,this._openPdf(n,i)},Document.prototype.download=function(n,i,o){p(n)&&(g(i)||(o=i),i=n,n=null),n=n||"file.pdf",this.getBlob((function(o){y(o,n),p(i)&&i()}),o)},Document.prototype.getBase64=function(n,i){if(!n)throw"getBase64 is an async method and needs a callback argument";this.getBuffer((function(i){n(i.toString("base64"))}),i)},Document.prototype.getDataUrl=function(n,i){if(!n)throw"getDataUrl is an async method and needs a callback argument";this.getBuffer((function(i){n("data:application/pdf;base64,"+i.toString("base64"))}),i)},Document.prototype.getBlob=function(n,i){if(!n)throw"getBlob is an async method and needs a callback argument";var o=this;this.getBuffer((function(i){var u=o._bufferToBlob(i);n(u)}),i)},Document.prototype.getBuffer=function(n,i){if(!n)throw"getBuffer is an async method and needs a callback argument";var o=this;this._createDoc(i,(function(i){o._flushDoc(i,(function(i){n(i)}))}))},Document.prototype.getStream=function(n,i){if(!p(i))return this._createDoc(n);this._createDoc(n,(function(n){i(n)}))},n.exports={createPdf:function(n,i,o,p){if(!function canCreatePdf(){return!(!Object.keys||"undefined"==typeof Uint16Array)}())throw"Your browser does not provide the level of support needed";return new Document(n,i||u.pdfMake.tableLayouts,o||u.pdfMake.fonts,p||u.pdfMake.vfs)}}}).call(this,o(10).Buffer,o(24))},function(n,i,o){"use strict";i.byteLength=function byteLength(n){var i=getLens(n),o=i[0],u=i[1];return 3*(o+u)/4-u},i.toByteArray=function toByteArray(n){var i,o,u=getLens(n),y=u[0],w=u[1],x=new g(function _byteLength(n,i,o){return 3*(i+o)/4-o}(0,y,w)),_=0,k=w>0?y-4:y;for(o=0;o<k;o+=4)i=p[n.charCodeAt(o)]<<18|p[n.charCodeAt(o+1)]<<12|p[n.charCodeAt(o+2)]<<6|p[n.charCodeAt(o+3)],x[_++]=i>>16&255,x[_++]=i>>8&255,x[_++]=255&i;2===w&&(i=p[n.charCodeAt(o)]<<2|p[n.charCodeAt(o+1)]>>4,x[_++]=255&i);1===w&&(i=p[n.charCodeAt(o)]<<10|p[n.charCodeAt(o+1)]<<4|p[n.charCodeAt(o+2)]>>2,x[_++]=i>>8&255,x[_++]=255&i);return x},i.fromByteArray=function fromByteArray(n){for(var i,o=n.length,p=o%3,g=[],y=0,w=o-p;y<w;y+=16383)g.push(encodeChunk(n,y,y+16383>w?w:y+16383));1===p?(i=n[o-1],g.push(u[i>>2]+u[i<<4&63]+"==")):2===p&&(i=(n[o-2]<<8)+n[o-1],g.push(u[i>>10]+u[i>>4&63]+u[i<<2&63]+"="));return g.join("")};for(var u=[],p=[],g="undefined"!=typeof Uint8Array?Uint8Array:Array,y="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/",w=0,x=y.length;w<x;++w)u[w]=y[w],p[y.charCodeAt(w)]=w;function getLens(n){var i=n.length;if(i%4>0)throw new Error("Invalid string. Length must be a multiple of 4");var o=n.indexOf("=");return-1===o&&(o=i),[o,o===i?0:4-o%4]}function encodeChunk(n,i,o){for(var p,g,y=[],w=i;w<o;w+=3)p=(n[w]<<16&16711680)+(n[w+1]<<8&65280)+(255&n[w+2]),y.push(u[(g=p)>>18&63]+u[g>>12&63]+u[g>>6&63]+u[63&g]);return y.join("")}p["-".charCodeAt(0)]=62,p["_".charCodeAt(0)]=63},function(n,i){i.read=function(n,i,o,u,p){var g,y,w=8*p-u-1,x=(1<<w)-1,_=x>>1,k=-7,P=o?p-1:0,E=o?-1:1,O=n[i+P];for(P+=E,g=O&(1<<-k)-1,O>>=-k,k+=w;k>0;g=256*g+n[i+P],P+=E,k-=8);for(y=g&(1<<-k)-1,g>>=-k,k+=u;k>0;y=256*y+n[i+P],P+=E,k-=8);if(0===g)g=1-_;else{if(g===x)return y?NaN:1/0*(O?-1:1);y+=Math.pow(2,u),g-=_}return(O?-1:1)*y*Math.pow(2,g-u)},i.write=function(n,i,o,u,p,g){var y,w,x,_=8*g-p-1,k=(1<<_)-1,P=k>>1,E=23===p?Math.pow(2,-24)-Math.pow(2,-77):0,O=u?0:g-1,B=u?1:-1,I=i<0||0===i&&1/i<0?1:0;for(i=Math.abs(i),isNaN(i)||i===1/0?(w=isNaN(i)?1:0,y=k):(y=Math.floor(Math.log(i)/Math.LN2),i*(x=Math.pow(2,-y))<1&&(y--,x*=2),(i+=y+P>=1?E/x:E*Math.pow(2,1-P))*x>=2&&(y++,x/=2),y+P>=k?(w=0,y=k):y+P>=1?(w=(i*x-1)*Math.pow(2,p),y+=P):(w=i*Math.pow(2,P-1)*Math.pow(2,p),y=0));p>=8;n[o+O]=255&w,O+=B,w/=256,p-=8);for(y=y<<p|w,_+=p;_>0;n[o+O]=255&y,O+=B,y/=256,_-=8);n[o+O-B]|=128*I}},function(n,i,o){(function(o){var u,p,g;p=[],void 0===(g="function"==typeof(u=function(){"use strict";function c(n,i,o){var u=new XMLHttpRequest;u.open("GET",n),u.responseType="blob",u.onload=function(){p(u.response,i,o)},u.onerror=function(){},u.send()}function d(n){var i=new XMLHttpRequest;i.open("HEAD",n,!1);try{i.send()}catch(n){}return 200<=i.status&&299>=i.status}function e(n){try{n.dispatchEvent(new MouseEvent("click"))}catch(o){var i=document.createEvent("MouseEvents");i.initMouseEvent("click",!0,!0,window,0,0,0,80,20,!1,!1,!1,!1,0,null),n.dispatchEvent(i)}}var i="object"==typeof window&&window.window===window?window:"object"==typeof self&&self.self===self?self:"object"==typeof o&&o.global===o?o:void 0,u=i.navigator&&/Macintosh/.test(navigator.userAgent)&&/AppleWebKit/.test(navigator.userAgent)&&!/Safari/.test(navigator.userAgent),p=i.saveAs||("object"!=typeof window||window!==i?function(){}:"undefined"!=typeof HTMLAnchorElement&&"download"in HTMLAnchorElement.prototype&&!u?function(n,o,u){var p=i.URL||i.webkitURL,g=document.createElement("a");o=o||n.name||"download",g.download=o,g.rel="noopener","string"==typeof n?(g.href=n,g.origin===location.origin?e(g):d(g.href)?c(n,o,u):e(g,g.target="_blank")):(g.href=p.createObjectURL(n),setTimeout((function(){p.revokeObjectURL(g.href)}),4e4),setTimeout((function(){e(g)}),0))}:"msSaveOrOpenBlob"in navigator?function(n,i,o){if(i=i||n.name||"download","string"!=typeof n)navigator.msSaveOrOpenBlob(function b(n,i){return void 0===i?i={autoBom:!1}:"object"!=typeof i&&(i={autoBom:!i}),i.autoBom&&/^\s*(?:text\/\S*|application\/xml|\S*\/\S*\+xml)\s*;.*charset\s*=\s*utf-8/i.test(n.type)?new Blob(["\ufeff",n],{type:n.type}):n}(n,o),i);else if(d(n))c(n,i,o);else{var u=document.createElement("a");u.href=n,u.target="_blank",setTimeout((function(){e(u)}))}}:function(n,o,p,g){if((g=g||open("","_blank"))&&(g.document.title=g.document.body.innerText="downloading..."),"string"==typeof n)return c(n,o,p);var y="application/octet-stream"===n.type,w=/constructor/i.test(i.HTMLElement)||i.safari,x=/CriOS\/[\d]+/.test(navigator.userAgent);if((x||y&&w||u)&&"undefined"!=typeof FileReader){var _=new FileReader;_.onloadend=function(){var n=_.result;n=x?n:n.replace(/^data:[^;]*;/,"data:attachment/file;"),g?g.location.href=n:location=n,g=null},_.readAsDataURL(n)}else{var k=i.URL||i.webkitURL,P=k.createObjectURL(n);g?g.location=P:location.href=P,g=null,setTimeout((function(){k.revokeObjectURL(P)}),4e4)}});i.saveAs=p.saveAs=p,n.exports=p})?u.apply(i,p):u)||(n.exports=g)}).call(this,o(24))},function(n,i,o){"use strict";var u=o(308),p=o(539),g=o(540),y=o(547),w=o(548),x=o(549),_=o(553),k=o(197),P=o(0).isFunction,E=o(0).isString,O=o(0).isNumber,B=o(0).isBoolean,I=o(0).isArray,D=o(0).isUndefined;function PdfPrinter(n){this.fontDescriptors=n}function fixPageMargins(n){if(O(n))n={left:n,right:n,top:n,bottom:n};else if(I(n))if(2===n.length)n={left:n[0],top:n[1],right:n[0],bottom:n[1]};else{if(4!==n.length)throw"Invalid pageMargins definition";n={left:n[0],top:n[1],right:n[2],bottom:n[3]}}return n}function updatePageOrientationInOptions(n,i){var o=i.options.size[0]>i.options.size[1]?"landscape":"portrait";if(n.pageSize.orientation!==o){var u=i.options.size[0],p=i.options.size[1];i.options.size=[p,u]}}function offsetText(n,i){var o=n;return i.sup&&(o-=.75*i.fontSize),i.sub&&(o+=.35*i.fontSize),o}function renderLine(n,i,o,u){function preparePageNodeRefLine(n,i){var o,u,p=new k(null);if(D(n.positions))throw"Page reference id not found";var g=n.positions[0].pageNumber.toString();switch(i.text=g,o=p.widthOfString(i.text,i.font,i.fontSize,i.characterSpacing,i.fontFeatures),u=i.width-o,i.width=o,i.alignment){case"right":i.x+=u;break;case"center":i.x+=u/2}}n._pageNodeRef&&preparePageNodeRefLine(n._pageNodeRef,n.inlines[0]),i=i||0,o=o||0;var p=n.getHeight(),g=p-n.getAscenderHeight();_.drawBackground(n,i,o,u);for(var y=0,w=n.inlines.length;y<w;y++){var x=n.inlines[y],P=p-x.font.ascender/1e3*x.fontSize-g;x._pageNodeRef&&preparePageNodeRefLine(x._pageNodeRef,x);var E={lineBreak:!1,textWidth:x.width,characterSpacing:x.characterSpacing,wordCount:1,link:x.link};x.linkToDestination&&(E.goTo=x.linkToDestination),n.id&&0===y&&(E.destination=n.id),x.fontFeatures&&(E.features=x.fontFeatures);var B=O(x.opacity)?x.opacity:1;u.opacity(B),u.fill(x.color||"black"),u._font=x.font,u.fontSize(x.fontSize);var I=offsetText(o+P,x);if(u.text(x.text,i+x.x,I,E),x.linkToPage){u.ref({Type:"Action",S:"GoTo",D:[x.linkToPage,0,0]}).end();u.annotate(i+x.x,I,x.width,x.height,{Subtype:"Link",Dest:[x.linkToPage-1,"XYZ",null,null,null]})}}_.drawDecorations(n,i,o,u)}function renderWatermark(n,i){var o=n.watermark;i.fill(o.color),i.opacity(o.opacity),i.save(),i.rotate(o.angle,{origin:[i.page.width/2,i.page.height/2]});var u=i.page.width/2-o._size.size.width/2,p=i.page.height/2-o._size.size.height/2;i._font=o.font,i.fontSize(o.fontSize),i.text(o.text,u,p,{lineBreak:!1}),i.restore()}function renderVector(n,i){i.lineWidth(n.lineWidth||1),n.dash?i.dash(n.dash.length,{space:n.dash.space||n.dash.length,phase:n.dash.phase||0}):i.undash(),i.lineJoin(n.lineJoin||"miter"),i.lineCap(n.lineCap||"butt");var o=null;switch(n.type){case"ellipse":i.ellipse(n.x,n.y,n.r1,n.r2),n.linearGradient&&(o=i.linearGradient(n.x-n.r1,n.y,n.x+n.r1,n.y));break;case"rect":n.r?i.roundedRect(n.x,n.y,n.w,n.h,n.r):i.rect(n.x,n.y,n.w,n.h),n.linearGradient&&(o=i.linearGradient(n.x,n.y,n.x+n.w,n.y));break;case"line":i.moveTo(n.x1,n.y1),i.lineTo(n.x2,n.y2);break;case"polyline":if(0===n.points.length)break;i.moveTo(n.points[0].x,n.points[0].y);for(var u=1,p=n.points.length;u<p;u++)i.lineTo(n.points[u].x,n.points[u].y);if(n.points.length>1){var g=n.points[0],y=n.points[n.points.length-1];(n.closePath||g.x===y.x&&g.y===y.y)&&i.closePath()}break;case"path":i.path(n.d)}if(n.linearGradient&&o){var w=1/(n.linearGradient.length-1);for(u=0;u<n.linearGradient.length;u++)o.stop(u*w,n.linearGradient[u]);n.color=o}var x=O(n.fillOpacity)?n.fillOpacity:1,_=O(n.strokeOpacity)?n.strokeOpacity:1;n.color&&n.lineColor?(i.fillColor(n.color,x),i.strokeColor(n.lineColor,_),i.fillAndStroke()):n.color?(i.fillColor(n.color,x),i.fill()):(i.strokeColor(n.lineColor||"black",_),i.stroke())}function renderImage(n,i,o,u){var p=O(n.opacity)?n.opacity:1;if(u.opacity(p),n.cover){var g=n.cover.align||"center",y=n.cover.valign||"center",w=n.cover.width?n.cover.width:n.width,x=n.cover.height?n.cover.height:n.height;u.save(),u.rect(n.x,n.y,w,x).clip(),u.image(n.image,n.x,n.y,{cover:[w,x],align:g,valign:y}),u.restore()}else u.image(n.image,n.x,n.y,{width:n._width,height:n._height});n.link&&u.link(n.x,n.y,n._width,n._height,n.link),n.linkToPage&&(u.ref({Type:"Action",S:"GoTo",D:[n.linkToPage,0,0]}).end(),u.annotate(n.x,n.y,n._width,n._height,{Subtype:"Link",Dest:[n.linkToPage-1,"XYZ",null,null,null]})),n.linkToDestination&&u.goTo(n.x,n.y,n._width,n._height,n.linkToDestination)}function renderSVG(n,i,u,p,g){var y=Object.assign({width:n._width,height:n._height,assumePt:!0},n.options);y.fontCallback=function(i,o,u){var p=i.split(",").map((function(n){return n.trim().replace(/('|")/g,"")})),y=function(n,i,o){for(var u=0;u<i.length;u++){var p=i[u].toLowerCase();for(var g in n)if(g.toLowerCase()===p)return g}return o}(g.fonts,p,n.font||"Roboto"),w=g.getFontFile(y,o,u);if(null===w){var x=g.getFontType(o,u);throw new Error("Font '"+y+"' in style '"+x+"' is not defined in the font section of the document definition.")}return w},function(){try{return o(554)}catch(n){throw new Error("Please install svg-to-pdfkit to enable svg nodes")}}()(p,n.svg,n.x,n.y,y)}function beginClip(n,i){i.save(),i.addContent(n.x+" "+n.y+" "+n.width+" "+n.height+" re"),i.clip()}function endClip(n){n.restore()}PdfPrinter.prototype.createPdfKitDocument=function(n,i){i=i||{},n.version=n.version||"1.3",n.compress=!B(n.compress)||n.compress,n.images=n.images||{},n.pageMargins=void 0!==n.pageMargins&&null!==n.pageMargins?n.pageMargins:40;var o=function fixPageSize(n,i){n&&"auto"===n.height&&(n.height=1/0);var o=function pageSize2widthAndHeight(n){if(E(n)){var i=y[n.toUpperCase()];if(!i)throw"Page size "+n+" not recognized";return{width:i[0],height:i[1]}}return n}(n||"A4");(function isNeedSwapPageSizes(n){return!!E(n)&&("portrait"===(n=n.toLowerCase())&&o.width>o.height||"landscape"===n&&o.width<o.height)})(i)&&(o={width:o.height,height:o.width});return o.orientation=o.width>o.height?"landscape":"portrait",o}(n.pageSize,n.pageOrientation),_={size:[o.width,o.height],pdfVersion:n.version,compress:n.compress,userPassword:n.userPassword,ownerPassword:n.ownerPassword,permissions:n.permissions,fontLayoutCache:!B(i.fontLayoutCache)||i.fontLayoutCache,bufferPages:i.bufferPages||!1,autoFirstPage:!1,font:null};this.pdfKitDoc=u.createPdfDocument(_),function setMetadata(n,i){function standardizePropertyKey(n){var i=n.charAt(0).toUpperCase()+n.slice(1);return-1!==["Title","Author","Subject","Keywords","Creator","Producer","CreationDate","ModDate","Trapped"].indexOf(i)?i:n.replace(/\s+/g,"")}if(i.info.Producer="pdfmake",i.info.Creator="pdfmake",n.info)for(var o in n.info){var u=n.info[o];u&&(o=standardizePropertyKey(o),i.info[o]=u)}}(n,this.pdfKitDoc),this.fontProvider=new p(this.fontDescriptors,this.pdfKitDoc);var k=new g(o,fixPageMargins(n.pageMargins),new w(this.pdfKitDoc,n.images),new x);!function registerDefaultTableLayouts(n){n.registerTableLayouts({noBorders:{hLineWidth:function(n){return 0},vLineWidth:function(n){return 0},paddingLeft:function(n){return n?4:0},paddingRight:function(n,i){return n<i.table.widths.length-1?4:0}},headerLineOnly:{hLineWidth:function(n,i){return 0===n||n===i.table.body.length?0:n===i.table.headerRows?2:0},vLineWidth:function(n){return 0},paddingLeft:function(n){return 0===n?0:8},paddingRight:function(n,i){return n===i.table.widths.length-1?0:8}},lightHorizontalLines:{hLineWidth:function(n,i){return 0===n||n===i.table.body.length?0:n===i.table.headerRows?2:1},vLineWidth:function(n){return 0},hLineColor:function(n){return 1===n?"black":"#aaa"},paddingLeft:function(n){return 0===n?0:8},paddingRight:function(n,i){return n===i.table.widths.length-1?0:8}}})}(k),i.tableLayouts&&k.registerTableLayouts(i.tableLayouts);var I=k.layoutDocument(n.content,this.fontProvider,n.styles||{},n.defaultStyle||{fontSize:12,font:"Roboto"},n.background,n.header,n.footer,n.images,n.watermark,n.pageBreakBefore),D=n.maxPagesNumber||-1;if(O(D)&&D>-1&&(I=I.slice(0,D)),o.height===1/0){var R=function calculatePageHeight(n,i){var o=fixPageMargins(i||40),u=o.top;return n.forEach((function(n){n.items.forEach((function(n){var i=function getBottomPosition(n){var i=n.item.y||0,o=function getItemHeight(n){return P(n.item.getHeight)?n.item.getHeight():n.item._height?n.item._height:"vector"===n.type?n.item.y1>n.item.y2?n.item.y1:n.item.y2:0}(n);return i+o}(n);i>u&&(u=i)}))})),u+=o.bottom}(I,n.pageMargins);this.pdfKitDoc.options.size=[o.width,R]}if(function renderPages(n,i,o,u){o._pdfMakePages=n,o.addPage();var p=0;u&&n.forEach((function(n){p+=n.items.length}));var g=0;u=u||function(){};for(var y=0;y<n.length;y++){y>0&&(updatePageOrientationInOptions(n[y],o),o.addPage(o.options));for(var w=n[y],x=0,_=w.items.length;x<_;x++){var k=w.items[x];switch(k.type){case"vector":renderVector(k.item,o);break;case"line":renderLine(k.item,k.item.x,k.item.y,o);break;case"image":renderImage(k.item,k.item.x,k.item.y,o);break;case"svg":renderSVG(k.item,k.item.x,k.item.y,o,i);break;case"beginClip":beginClip(k.item,o);break;case"endClip":endClip(o)}g++,u(g/p)}w.watermark&&renderWatermark(w,o)}}(I,this.fontProvider,this.pdfKitDoc,i.progressCallback),i.autoPrint){var N=this.pdfKitDoc.ref({Type:"Action",S:"Named",N:"Print"});this.pdfKitDoc._root.data.OpenAction=N,N.end()}return this.pdfKitDoc},n.exports=PdfPrinter},function(n,i,o){"use strict";var u=function _interopDefault(n){return n&&"object"==typeof n&&"default"in n?n.default:n}(o(309));n.exports={getEngineInstance:function getEngineInstance(){return u},createPdfDocument:function createPdfDocument(n){return new u(n=n||{})}}},function(n,i,o){"use strict";(function(n,u){o(201),o(210),o(211),o(212),o(314),o(213),o(315),o(316),o(317),o(216),o(28),o(220),o(322),o(46),o(323),o(324),o(61),o(224),o(225),o(326),o(328),o(330),o(229),o(331),o(31),o(118),o(334),o(335),o(336),o(160),o(161),o(231),o(232),o(233),o(234),o(340),o(62),o(63),o(64),o(65),o(66),o(67),o(68),o(69),o(70),o(71),o(72),o(73),o(74),o(75),o(76),o(77),o(78),o(79),o(80),o(81),o(82),o(83),o(84),o(85),o(346),o(120),Object.defineProperty(i,"__esModule",{value:!0}),i.default=void 0;var p=_interopRequireDefault(o(121)),g=_interopRequireDefault(o(245)),y=_interopRequireDefault(o(368)),w=_interopRequireDefault(o(392)),x=o(122),_=_interopRequireDefault(o(290)),k=_interopRequireDefault(o(538));function _interopRequireDefault(n){return n&&n.__esModule?n:{default:n}}var P=o(53);function _classCallCheck(n,i){if(!(n instanceof i))throw new TypeError("Cannot call a class as a function")}function _defineProperties(n,i){for(var o=0;o<i.length;o++){var u=i[o];u.enumerable=u.enumerable||!1,u.configurable=!0,"value"in u&&(u.writable=!0),Object.defineProperty(n,u.key,u)}}function _createClass(n,i,o){return i&&_defineProperties(n.prototype,i),o&&_defineProperties(n,o),n}function _inherits(n,i){if("function"!=typeof i&&null!==i)throw new TypeError("Super expression must either be null or a function");n.prototype=Object.create(i&&i.prototype,{constructor:{value:n,writable:!0,configurable:!0}}),i&&_setPrototypeOf(n,i)}function _getPrototypeOf(n){return(_getPrototypeOf=Object.setPrototypeOf?Object.getPrototypeOf:function _getPrototypeOf(n){return n.__proto__||Object.getPrototypeOf(n)})(n)}function _setPrototypeOf(n,i){return(_setPrototypeOf=Object.setPrototypeOf||function _setPrototypeOf(n,i){return n.__proto__=i,n})(n,i)}function _assertThisInitialized(n){if(void 0===n)throw new ReferenceError("this hasn't been initialised - super() hasn't been called");return n}function _possibleConstructorReturn(n,i){return!i||"object"!=typeof i&&"function"!=typeof i?_assertThisInitialized(n):i}function _slicedToArray(n,i){return function _arrayWithHoles(n){if(Array.isArray(n))return n}(n)||function _iterableToArrayLimit(n,i){if(!(Symbol.iterator in Object(n))&&"[object Arguments]"!==Object.prototype.toString.call(n))return;var o=[],u=!0,p=!1,g=void 0;try{for(var y,w=n[Symbol.iterator]();!(u=(y=w.next()).done)&&(o.push(y.value),!i||o.length!==i);u=!0);}catch(n){p=!0,g=n}finally{try{u||null==w.return||w.return()}finally{if(p)throw g}}return o}(n,i)||function _nonIterableRest(){throw new TypeError("Invalid attempt to destructure non-iterable instance")}()}function _toConsumableArray(n){return function _arrayWithoutHoles(n){if(Array.isArray(n)){for(var i=0,o=new Array(n.length);i<n.length;i++)o[i]=n[i];return o}}(n)||function _iterableToArray(n){if(Symbol.iterator in Object(n)||"[object Arguments]"===Object.prototype.toString.call(n))return Array.from(n)}(n)||function _nonIterableSpread(){throw new TypeError("Invalid attempt to spread non-iterable instance")}()}var E=function(){function PDFAbstractReference(){_classCallCheck(this,PDFAbstractReference)}return _createClass(PDFAbstractReference,[{key:"toString",value:function toString(){throw new Error("Must be implemented by subclasses")}}]),PDFAbstractReference}(),O=function(){function PDFNameTree(){_classCallCheck(this,PDFNameTree),this._items={}}return _createClass(PDFNameTree,[{key:"add",value:function add(n,i){return this._items[n]=i}},{key:"get",value:function get(n){return this._items[n]}},{key:"toString",value:function toString(){var n=Object.keys(this._items).sort((function(n,i){return n.localeCompare(i)})),i=["<<"];if(n.length>1){var o=n[0],u=n[n.length-1];i.push("  /Limits ".concat(N.convert([new String(o),new String(u)])))}i.push("  /Names [");var p=!0,g=!1,y=void 0;try{for(var w,x=n[Symbol.iterator]();!(p=(w=x.next()).done);p=!0){var _=w.value;i.push("    ".concat(N.convert(new String(_))," ").concat(N.convert(this._items[_])))}}catch(n){g=!0,y=n}finally{try{p||null==x.return||x.return()}finally{if(g)throw y}}return i.push("]"),i.push(">>"),i.join("\n")}}]),PDFNameTree}(),B=function pad(n,i){return(Array(i+1).join("0")+n).slice(-i)},I=/[\n\r\t\b\f\(\)\\]/g,D={"\n":"\\n","\r":"\\r","\t":"\\t","\b":"\\b","\f":"\\f","\\":"\\\\","(":"\\(",")":"\\)"},R=function swapBytes(n){var i=n.length;if(1&i)throw new Error("Buffer length must be even");for(var o=0,u=i-1;o<u;o+=2){var p=n[o];n[o]=n[o+1],n[o+1]=p}return n},N=function(){function PDFObject(){_classCallCheck(this,PDFObject)}return _createClass(PDFObject,null,[{key:"convert",value:function convert(i){var o=arguments.length>1&&void 0!==arguments[1]?arguments[1]:null;if("string"==typeof i)return"/".concat(i);if(i instanceof String){for(var u,p=i,g=!1,y=0,w=p.length;y<w;y++)if(p.charCodeAt(y)>127){g=!0;break}return u=g?R(n.from("\ufeff".concat(p),"utf16le")):n.from(p.valueOf(),"ascii"),p=(p=o?o(u).toString("binary"):u.toString("binary")).replace(I,(function(n){return D[n]})),"(".concat(p,")")}if(n.isBuffer(i))return"<".concat(i.toString("hex"),">");if(i instanceof E||i instanceof O)return i.toString();if(i instanceof Date){var x="D:".concat(B(i.getUTCFullYear(),4))+B(i.getUTCMonth()+1,2)+B(i.getUTCDate(),2)+B(i.getUTCHours(),2)+B(i.getUTCMinutes(),2)+B(i.getUTCSeconds(),2)+"Z";return o&&(x=(x=o(new n(x,"ascii")).toString("binary")).replace(I,(function(n){return D[n]}))),"(".concat(x,")")}if(Array.isArray(i)){var _=i.map((function(n){return PDFObject.convert(n,o)})).join(" ");return"[".concat(_,"]")}if("[object Object]"==={}.toString.call(i)){var k=["<<"];for(var P in i){var N=i[P];k.push("/".concat(P," ").concat(PDFObject.convert(N,o)))}return k.push(">>"),k.join("\n")}return"number"==typeof i?PDFObject.number(i):"".concat(i)}},{key:"number",value:function number(n){if(n>-1e21&&n<1e21)return Math.round(1e6*n)/1e6;throw new Error("unsupported number: ".concat(n))}}]),PDFObject}(),U=function(i){function PDFReference(n,i){var o,u=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{};return _classCallCheck(this,PDFReference),(o=_possibleConstructorReturn(this,_getPrototypeOf(PDFReference).call(this))).document=n,o.id=i,o.data=u,o.gen=0,o.compress=o.document.compress&&!o.data.Filter,o.uncompressedLength=0,o.buffer=[],o}return _inherits(PDFReference,i),_createClass(PDFReference,[{key:"write",value:function write(i){if(n.isBuffer(i)||(i=new n(i+"\n","binary")),this.uncompressedLength+=i.length,null==this.data.Length&&(this.data.Length=0),this.buffer.push(i),this.data.Length+=i.length,this.compress)return this.data.Filter="FlateDecode"}},{key:"end",value:function end(n){return n&&this.write(n),this.finalize()}},{key:"finalize",value:function finalize(){this.offset=this.document._offset;var i=this.document._security?this.document._security.getEncryptFn(this.id,this.gen):null;this.buffer.length&&(this.buffer=n.concat(this.buffer),this.compress&&(this.buffer=g.default.deflateSync(this.buffer)),i&&(this.buffer=i(this.buffer)),this.data.Length=this.buffer.length),this.document._write("".concat(this.id," ").concat(this.gen," obj")),this.document._write(N.convert(this.data,i)),this.buffer.length&&(this.document._write("stream"),this.document._write(this.buffer),this.buffer=[],this.document._write("\nendstream")),this.document._write("endobj"),this.document._refEnd(this)}},{key:"toString",value:function toString(){return"".concat(this.id," ").concat(this.gen," R")}}]),PDFReference}(E),W={top:72,left:72,bottom:72,right:72},G={"4A0":[4767.87,6740.79],"2A0":[3370.39,4767.87],A0:[2383.94,3370.39],A1:[1683.78,2383.94],A2:[1190.55,1683.78],A3:[841.89,1190.55],A4:[595.28,841.89],A5:[419.53,595.28],A6:[297.64,419.53],A7:[209.76,297.64],A8:[147.4,209.76],A9:[104.88,147.4],A10:[73.7,104.88],B0:[2834.65,4008.19],B1:[2004.09,2834.65],B2:[1417.32,2004.09],B3:[1000.63,1417.32],B4:[708.66,1000.63],B5:[498.9,708.66],B6:[354.33,498.9],B7:[249.45,354.33],B8:[175.75,249.45],B9:[124.72,175.75],B10:[87.87,124.72],C0:[2599.37,3676.54],C1:[1836.85,2599.37],C2:[1298.27,1836.85],C3:[918.43,1298.27],C4:[649.13,918.43],C5:[459.21,649.13],C6:[323.15,459.21],C7:[229.61,323.15],C8:[161.57,229.61],C9:[113.39,161.57],C10:[79.37,113.39],RA0:[2437.8,3458.27],RA1:[1729.13,2437.8],RA2:[1218.9,1729.13],RA3:[864.57,1218.9],RA4:[609.45,864.57],SRA0:[2551.18,3628.35],SRA1:[1814.17,2551.18],SRA2:[1275.59,1814.17],SRA3:[907.09,1275.59],SRA4:[637.8,907.09],EXECUTIVE:[521.86,756],FOLIO:[612,936],LEGAL:[612,1008],LETTER:[612,792],TABLOID:[792,1224]},j=function(){function PDFPage(n){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};_classCallCheck(this,PDFPage),this.document=n,this.size=i.size||"letter",this.layout=i.layout||"portrait","number"==typeof i.margin?this.margins={top:i.margin,left:i.margin,bottom:i.margin,right:i.margin}:this.margins=i.margins||W;var o=Array.isArray(this.size)?this.size:G[this.size.toUpperCase()];this.width=o["portrait"===this.layout?0:1],this.height=o["portrait"===this.layout?1:0],this.content=this.document.ref(),this.resources=this.document.ref({ProcSet:["PDF","Text","ImageB","ImageC","ImageI"]}),this.dictionary=this.document.ref({Type:"Page",Parent:this.document._root.data.Pages,MediaBox:[0,0,this.width,this.height],Contents:this.content,Resources:this.resources})}return _createClass(PDFPage,[{key:"maxY",value:function maxY(){return this.height-this.margins.bottom}},{key:"write",value:function write(n){return this.content.write(n)}},{key:"end",value:function end(){return this.dictionary.end(),this.resources.end(),this.content.end()}},{key:"fonts",get:function get(){var n=this.resources.data;return null!=n.Font?n.Font:n.Font={}}},{key:"xobjects",get:function get(){var n=this.resources.data;return null!=n.XObject?n.XObject:n.XObject={}}},{key:"ext_gstates",get:function get(){var n=this.resources.data;return null!=n.ExtGState?n.ExtGState:n.ExtGState={}}},{key:"patterns",get:function get(){var n=this.resources.data;return null!=n.Pattern?n.Pattern:n.Pattern={}}},{key:"annotations",get:function get(){var n=this.dictionary.data;return null!=n.Annots?n.Annots:n.Annots=[]}}]),PDFPage}();function inRange(n,i){if(n<i[0])return!1;for(var o=0,u=i.length/2;o<=u;){var p=Math.floor((o+u)/2),g=2*p;if(n>=i[g]&&n<=i[g+1])return!0;n>i[g+1]?o=p+1:u=p-1}return!1}var X=[545,545,564,591,686,687,751,767,848,863,880,883,886,889,891,893,895,899,907,907,909,909,930,930,975,975,1015,1023,1159,1159,1231,1231,1270,1271,1274,1279,1296,1328,1367,1368,1376,1376,1416,1416,1419,1424,1442,1442,1466,1466,1477,1487,1515,1519,1525,1547,1549,1562,1564,1566,1568,1568,1595,1599,1622,1631,1774,1775,1791,1791,1806,1806,1837,1839,1867,1919,1970,2304,2308,2308,2362,2363,2382,2383,2389,2391,2417,2432,2436,2436,2445,2446,2449,2450,2473,2473,2481,2481,2483,2485,2490,2491,2493,2493,2501,2502,2505,2506,2510,2518,2520,2523,2526,2526,2532,2533,2555,2561,2563,2564,2571,2574,2577,2578,2601,2601,2609,2609,2612,2612,2615,2615,2618,2619,2621,2621,2627,2630,2633,2634,2638,2648,2653,2653,2655,2661,2677,2688,2692,2692,2700,2700,2702,2702,2706,2706,2729,2729,2737,2737,2740,2740,2746,2747,2758,2758,2762,2762,2766,2767,2769,2783,2785,2789,2800,2816,2820,2820,2829,2830,2833,2834,2857,2857,2865,2865,2868,2869,2874,2875,2884,2886,2889,2890,2894,2901,2904,2907,2910,2910,2914,2917,2929,2945,2948,2948,2955,2957,2961,2961,2966,2968,2971,2971,2973,2973,2976,2978,2981,2983,2987,2989,2998,2998,3002,3005,3011,3013,3017,3017,3022,3030,3032,3046,3059,3072,3076,3076,3085,3085,3089,3089,3113,3113,3124,3124,3130,3133,3141,3141,3145,3145,3150,3156,3159,3167,3170,3173,3184,3201,3204,3204,3213,3213,3217,3217,3241,3241,3252,3252,3258,3261,3269,3269,3273,3273,3278,3284,3287,3293,3295,3295,3298,3301,3312,3329,3332,3332,3341,3341,3345,3345,3369,3369,3386,3389,3396,3397,3401,3401,3406,3414,3416,3423,3426,3429,3440,3457,3460,3460,3479,3481,3506,3506,3516,3516,3518,3519,3527,3529,3531,3534,3541,3541,3543,3543,3552,3569,3573,3584,3643,3646,3676,3712,3715,3715,3717,3718,3721,3721,3723,3724,3726,3731,3736,3736,3744,3744,3748,3748,3750,3750,3752,3753,3756,3756,3770,3770,3774,3775,3781,3781,3783,3783,3790,3791,3802,3803,3806,3839,3912,3912,3947,3952,3980,3983,3992,3992,4029,4029,4045,4046,4048,4095,4130,4130,4136,4136,4139,4139,4147,4149,4154,4159,4186,4255,4294,4303,4345,4346,4348,4351,4442,4446,4515,4519,4602,4607,4615,4615,4679,4679,4681,4681,4686,4687,4695,4695,4697,4697,4702,4703,4743,4743,4745,4745,4750,4751,4783,4783,4785,4785,4790,4791,4799,4799,4801,4801,4806,4807,4815,4815,4823,4823,4847,4847,4879,4879,4881,4881,4886,4887,4895,4895,4935,4935,4955,4960,4989,5023,5109,5120,5751,5759,5789,5791,5873,5887,5901,5901,5909,5919,5943,5951,5972,5983,5997,5997,6001,6001,6004,6015,6109,6111,6122,6143,6159,6159,6170,6175,6264,6271,6314,7679,7836,7839,7930,7935,7958,7959,7966,7967,8006,8007,8014,8015,8024,8024,8026,8026,8028,8028,8030,8030,8062,8063,8117,8117,8133,8133,8148,8149,8156,8156,8176,8177,8181,8181,8191,8191,8275,8278,8280,8286,8292,8297,8306,8307,8335,8351,8370,8399,8427,8447,8507,8508,8524,8530,8580,8591,9167,9215,9255,9279,9291,9311,9471,9471,9748,9749,9752,9752,9854,9855,9866,9984,9989,9989,9994,9995,10024,10024,10060,10060,10062,10062,10067,10069,10071,10071,10079,10080,10133,10135,10160,10160,10175,10191,10220,10223,11008,11903,11930,11930,12020,12031,12246,12271,12284,12287,12352,12352,12439,12440,12544,12548,12589,12592,12687,12687,12728,12783,12829,12831,12868,12880,12924,12926,13004,13007,13055,13055,13175,13178,13278,13279,13311,13311,19894,19967,40870,40959,42125,42127,42183,44031,55204,55295,64046,64047,64107,64255,64263,64274,64280,64284,64311,64311,64317,64317,64319,64319,64322,64322,64325,64325,64434,64466,64832,64847,64912,64913,64968,64975,65021,65023,65040,65055,65060,65071,65095,65096,65107,65107,65127,65127,65132,65135,65141,65141,65277,65278,65280,65280,65471,65473,65480,65481,65488,65489,65496,65497,65501,65503,65511,65511,65519,65528,65536,66303,66335,66335,66340,66351,66379,66559,66598,66599,66638,118783,119030,119039,119079,119081,119262,119807,119893,119893,119965,119965,119968,119969,119971,119972,119975,119976,119981,119981,119994,119994,119996,119996,120001,120001,120004,120004,120070,120070,120075,120076,120085,120085,120093,120093,120122,120122,120127,120127,120133,120133,120135,120137,120145,120145,120484,120487,120778,120781,120832,131069,173783,194559,195102,196605,196608,262141,262144,327677,327680,393213,393216,458749,458752,524285,524288,589821,589824,655357,655360,720893,720896,786429,786432,851965,851968,917501,917504,917504,917506,917535,917632,983037],Y=function isUnassignedCodePoint(n){return inRange(n,X)},K=[173,173,847,847,6150,6150,6155,6155,6156,6156,6157,6157,8203,8203,8204,8204,8205,8205,8288,8288,65024,65024,65025,65025,65026,65026,65027,65027,65028,65028,65029,65029,65030,65030,65031,65031,65032,65032,65033,65033,65034,65034,65035,65035,65036,65036,65037,65037,65038,65038,65039,65039,65279,65279],J=[160,160,5760,5760,8192,8192,8193,8193,8194,8194,8195,8195,8196,8196,8197,8197,8198,8198,8199,8199,8200,8200,8201,8201,8202,8202,8203,8203,8239,8239,8287,8287,12288,12288],$=[128,159,1757,1757,1807,1807,6158,6158,8204,8204,8205,8205,8232,8232,8233,8233,8288,8288,8289,8289,8290,8290,8291,8291,8298,8303,65279,65279,65529,65532,119155,119162],tt=[64976,65007,65534,65535,131070,131071,196606,196607,262142,262143,327678,327679,393214,393215,458750,458751,524286,524287,589822,589823,655358,655359,720894,720895,786430,786431,851966,851967,917502,917503,983038,983039,1114110,1114111],et=[0,31,127,127,832,832,833,833,8206,8206,8207,8207,8234,8234,8235,8235,8236,8236,8237,8237,8238,8238,8298,8298,8299,8299,8300,8300,8301,8301,8302,8302,8303,8303,12272,12283,55296,57343,57344,63743,65529,65529,65530,65530,65531,65531,65532,65532,65533,65533,917505,917505,917536,917631,983040,1048573,1048576,1114109],rt=function isProhibitedCharacter(n){return inRange(n,J)||inRange(n,et)||inRange(n,$)||inRange(n,tt)},nt=[1470,1470,1472,1472,1475,1475,1488,1514,1520,1524,1563,1563,1567,1567,1569,1594,1600,1610,1645,1647,1649,1749,1757,1757,1765,1766,1786,1790,1792,1805,1808,1808,1810,1836,1920,1957,1969,1969,8207,8207,64285,64285,64287,64296,64298,64310,64312,64316,64318,64318,64320,64321,64323,64324,64326,64433,64467,64829,64848,64911,64914,64967,65008,65020,65136,65140,65142,65276],it=function isBidirectionalRAL(n){return inRange(n,nt)},ot=[65,90,97,122,170,170,181,181,186,186,192,214,216,246,248,544,546,563,592,685,688,696,699,705,720,721,736,740,750,750,890,890,902,902,904,906,908,908,910,929,931,974,976,1013,1024,1154,1162,1230,1232,1269,1272,1273,1280,1295,1329,1366,1369,1375,1377,1415,1417,1417,2307,2307,2309,2361,2365,2368,2377,2380,2384,2384,2392,2401,2404,2416,2434,2435,2437,2444,2447,2448,2451,2472,2474,2480,2482,2482,2486,2489,2494,2496,2503,2504,2507,2508,2519,2519,2524,2525,2527,2529,2534,2545,2548,2554,2565,2570,2575,2576,2579,2600,2602,2608,2610,2611,2613,2614,2616,2617,2622,2624,2649,2652,2654,2654,2662,2671,2674,2676,2691,2691,2693,2699,2701,2701,2703,2705,2707,2728,2730,2736,2738,2739,2741,2745,2749,2752,2761,2761,2763,2764,2768,2768,2784,2784,2790,2799,2818,2819,2821,2828,2831,2832,2835,2856,2858,2864,2866,2867,2870,2873,2877,2878,2880,2880,2887,2888,2891,2892,2903,2903,2908,2909,2911,2913,2918,2928,2947,2947,2949,2954,2958,2960,2962,2965,2969,2970,2972,2972,2974,2975,2979,2980,2984,2986,2990,2997,2999,3001,3006,3007,3009,3010,3014,3016,3018,3020,3031,3031,3047,3058,3073,3075,3077,3084,3086,3088,3090,3112,3114,3123,3125,3129,3137,3140,3168,3169,3174,3183,3202,3203,3205,3212,3214,3216,3218,3240,3242,3251,3253,3257,3262,3262,3264,3268,3271,3272,3274,3275,3285,3286,3294,3294,3296,3297,3302,3311,3330,3331,3333,3340,3342,3344,3346,3368,3370,3385,3390,3392,3398,3400,3402,3404,3415,3415,3424,3425,3430,3439,3458,3459,3461,3478,3482,3505,3507,3515,3517,3517,3520,3526,3535,3537,3544,3551,3570,3572,3585,3632,3634,3635,3648,3654,3663,3675,3713,3714,3716,3716,3719,3720,3722,3722,3725,3725,3732,3735,3737,3743,3745,3747,3749,3749,3751,3751,3754,3755,3757,3760,3762,3763,3773,3773,3776,3780,3782,3782,3792,3801,3804,3805,3840,3863,3866,3892,3894,3894,3896,3896,3902,3911,3913,3946,3967,3967,3973,3973,3976,3979,4030,4037,4039,4044,4047,4047,4096,4129,4131,4135,4137,4138,4140,4140,4145,4145,4152,4152,4160,4183,4256,4293,4304,4344,4347,4347,4352,4441,4447,4514,4520,4601,4608,4614,4616,4678,4680,4680,4682,4685,4688,4694,4696,4696,4698,4701,4704,4742,4744,4744,4746,4749,4752,4782,4784,4784,4786,4789,4792,4798,4800,4800,4802,4805,4808,4814,4816,4822,4824,4846,4848,4878,4880,4880,4882,4885,4888,4894,4896,4934,4936,4954,4961,4988,5024,5108,5121,5750,5761,5786,5792,5872,5888,5900,5902,5905,5920,5937,5941,5942,5952,5969,5984,5996,5998,6e3,6016,6070,6078,6085,6087,6088,6100,6106,6108,6108,6112,6121,6160,6169,6176,6263,6272,6312,7680,7835,7840,7929,7936,7957,7960,7965,7968,8005,8008,8013,8016,8023,8025,8025,8027,8027,8029,8029,8031,8061,8064,8116,8118,8124,8126,8126,8130,8132,8134,8140,8144,8147,8150,8155,8160,8172,8178,8180,8182,8188,8206,8206,8305,8305,8319,8319,8450,8450,8455,8455,8458,8467,8469,8469,8473,8477,8484,8484,8486,8486,8488,8488,8490,8493,8495,8497,8499,8505,8509,8511,8517,8521,8544,8579,9014,9082,9109,9109,9372,9449,12293,12295,12321,12329,12337,12341,12344,12348,12353,12438,12445,12447,12449,12538,12540,12543,12549,12588,12593,12686,12688,12727,12784,12828,12832,12867,12896,12923,12927,12976,12992,13003,13008,13054,13056,13174,13179,13277,13280,13310,13312,19893,19968,40869,40960,42124,44032,55203,55296,64045,64048,64106,64256,64262,64275,64279,65313,65338,65345,65370,65382,65470,65474,65479,65482,65487,65490,65495,65498,65500,66304,66334,66336,66339,66352,66378,66560,66597,66600,66637,118784,119029,119040,119078,119082,119142,119146,119154,119171,119172,119180,119209,119214,119261,119808,119892,119894,119964,119966,119967,119970,119970,119973,119974,119977,119980,119982,119993,119995,119995,119997,12e4,120002,120003,120005,120069,120071,120074,120077,120084,120086,120092,120094,120121,120123,120126,120128,120132,120134,120134,120138,120144,120146,120483,120488,120777,131072,173782,194560,195101,983040,1048573,1048576,1114109],at=function isBidirectionalL(n){return inRange(n,ot)},st=function isNonASCIISpaceCharacter(n){return inRange(n,J)},lt=function isCommonlyMappedToNothing(n){return inRange(n,K)},ct=function getCodePoint(n){return n.codePointAt(0)},ut=function first(n){return n[0]},ft=function last(n){return n[n.length-1]};function toCodePoints(n){for(var i=[],o=n.length,u=0;u<o;u+=1){var p=n.charCodeAt(u);if(p>=55296&&p<=56319&&o>u+1){var g=n.charCodeAt(u+1);if(g>=56320&&g<=57343){i.push(1024*(p-55296)+g-56320+65536),u+=1;continue}}i.push(p)}return i}function saslprep(n){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};if("string"!=typeof n)throw new TypeError("Expected string.");if(0===n.length)return"";var o=toCodePoints(n).map((function(n){return st(n)?32:n})).filter((function(n){return!lt(n)})),u=String.fromCodePoint.apply(null,o).normalize("NFKC"),p=toCodePoints(u),g=p.some(rt);if(g)throw new Error("Prohibited character, see https://tools.ietf.org/html/rfc4013#section-2.3");if(!0!==i.allowUnassigned){var y=p.some(Y);if(y)throw new Error("Unassigned code point, see https://tools.ietf.org/html/rfc4013#section-2.5")}var w=p.some(it),x=p.some(at);if(w&&x)throw new Error("String must not contain RandALCat and LCat at the same time, see https://tools.ietf.org/html/rfc3454#section-6");var _=it(ct(ut(u))),k=it(ct(ft(u)));if(w&&(!_||!k))throw new Error("Bidirectional RandALCat character must be the first and the last character of the string, see https://tools.ietf.org/html/rfc3454#section-6");return u}var ht=function(){function PDFSecurity(n){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};if(_classCallCheck(this,PDFSecurity),!i.ownerPassword&&!i.userPassword)throw new Error("None of owner password and user password is defined.");this.document=n,this._setupEncryption(i)}return _createClass(PDFSecurity,null,[{key:"generateFileID",value:function generateFileID(){var n=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{},i="".concat(n.CreationDate.getTime(),"\n");for(var o in n)n.hasOwnProperty(o)&&(i+="".concat(o,": ").concat(n[o],"\n"));return wordArrayToBuffer(y.default.MD5(i))}},{key:"generateRandomWordArray",value:function generateRandomWordArray(n){return y.default.lib.WordArray.random(n)}},{key:"create",value:function create(n){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};return i.ownerPassword||i.userPassword?new PDFSecurity(n,i):null}}]),_createClass(PDFSecurity,[{key:"_setupEncryption",value:function _setupEncryption(n){switch(n.pdfVersion){case"1.4":case"1.5":this.version=2;break;case"1.6":case"1.7":this.version=4;break;case"1.7ext3":this.version=5;break;default:this.version=1}var i={Filter:"Standard"};switch(this.version){case 1:case 2:case 4:this._setupEncryptionV1V2V4(this.version,i,n);break;case 5:this._setupEncryptionV5(i,n)}this.dictionary=this.document.ref(i)}},{key:"_setupEncryptionV1V2V4",value:function _setupEncryptionV1V2V4(n,i,o){var u,p;switch(n){case 1:u=2,this.keyBits=40,p=function getPermissionsR2(){var n=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{},i=-64;n.printing&&(i|=4);n.modifying&&(i|=8);n.copying&&(i|=16);n.annotating&&(i|=32);return i}(o.permissions);break;case 2:u=3,this.keyBits=128,p=getPermissionsR3(o.permissions);break;case 4:u=4,this.keyBits=128,p=getPermissionsR3(o.permissions)}var g,w=processPasswordR2R3R4(o.userPassword),x=o.ownerPassword?processPasswordR2R3R4(o.ownerPassword):w,_=function getOwnerPasswordR2R3R4(n,i,o,u){for(var p=u,g=n>=3?51:1,w=0;w<g;w++)p=y.default.MD5(p);var x=p.clone();x.sigBytes=i/8;var _=o;g=n>=3?20:1;for(var k=0;k<g;k++){for(var P=Math.ceil(x.sigBytes/4),E=0;E<P;E++)x.words[E]=p.words[E]^(k|k<<8|k<<16|k<<24);_=y.default.RC4.encrypt(_,x).ciphertext}return _}(u,this.keyBits,w,x);this.encryptionKey=function getEncryptionKeyR2R3R4(n,i,o,u,p,g){for(var w=u.clone().concat(p).concat(y.default.lib.WordArray.create([lsbFirstWord(g)],4)).concat(y.default.lib.WordArray.create(o)),x=n>=3?51:1,_=0;_<x;_++)(w=y.default.MD5(w)).sigBytes=i/8;return w}(u,this.keyBits,this.document._id,w,_,p),g=2===u?function getUserPasswordR2(n){return y.default.RC4.encrypt(processPasswordR2R3R4(),n).ciphertext}(this.encryptionKey):function getUserPasswordR3R4(n,i){for(var o=i.clone(),u=y.default.MD5(processPasswordR2R3R4().concat(y.default.lib.WordArray.create(n))),p=0;p<20;p++){for(var g=Math.ceil(o.sigBytes/4),w=0;w<g;w++)o.words[w]=i.words[w]^(p|p<<8|p<<16|p<<24);u=y.default.RC4.encrypt(u,o).ciphertext}return u.concat(y.default.lib.WordArray.create(null,16))}(this.document._id,this.encryptionKey),i.V=n,n>=2&&(i.Length=this.keyBits),4===n&&(i.CF={StdCF:{AuthEvent:"DocOpen",CFM:"AESV2",Length:this.keyBits/8}},i.StmF="StdCF",i.StrF="StdCF"),i.R=u,i.O=wordArrayToBuffer(_),i.U=wordArrayToBuffer(g),i.P=p}},{key:"_setupEncryptionV5",value:function _setupEncryptionV5(n,i){this.keyBits=256;var o=getPermissionsR3(i),u=processPasswordR5(i.userPassword),p=i.ownerPassword?processPasswordR5(i.ownerPassword):u;this.encryptionKey=function getEncryptionKeyR5(n){return n(32)}(PDFSecurity.generateRandomWordArray);var g=function getUserPasswordR5(n,i){var o=i(8),u=i(8);return y.default.SHA256(n.clone().concat(o)).concat(o).concat(u)}(u,PDFSecurity.generateRandomWordArray),w=function getUserEncryptionKeyR5(n,i,o){var u=y.default.SHA256(n.clone().concat(i)),p={mode:y.default.mode.CBC,padding:y.default.pad.NoPadding,iv:y.default.lib.WordArray.create(null,16)};return y.default.AES.encrypt(o,u,p).ciphertext}(u,y.default.lib.WordArray.create(g.words.slice(10,12),8),this.encryptionKey),x=function getOwnerPasswordR5(n,i,o){var u=o(8),p=o(8);return y.default.SHA256(n.clone().concat(u).concat(i)).concat(u).concat(p)}(p,g,PDFSecurity.generateRandomWordArray),_=function getOwnerEncryptionKeyR5(n,i,o,u){var p=y.default.SHA256(n.clone().concat(i).concat(o)),g={mode:y.default.mode.CBC,padding:y.default.pad.NoPadding,iv:y.default.lib.WordArray.create(null,16)};return y.default.AES.encrypt(u,p,g).ciphertext}(p,y.default.lib.WordArray.create(x.words.slice(10,12),8),g,this.encryptionKey),k=function getEncryptedPermissionsR5(n,i,o){var u=y.default.lib.WordArray.create([lsbFirstWord(n),4294967295,1415668834],12).concat(o(4)),p={mode:y.default.mode.ECB,padding:y.default.pad.NoPadding};return y.default.AES.encrypt(u,i,p).ciphertext}(o,this.encryptionKey,PDFSecurity.generateRandomWordArray);n.V=5,n.Length=this.keyBits,n.CF={StdCF:{AuthEvent:"DocOpen",CFM:"AESV3",Length:this.keyBits/8}},n.StmF="StdCF",n.StrF="StdCF",n.R=5,n.O=wordArrayToBuffer(x),n.OE=wordArrayToBuffer(_),n.U=wordArrayToBuffer(g),n.UE=wordArrayToBuffer(w),n.P=o,n.Perms=wordArrayToBuffer(k)}},{key:"getEncryptFn",value:function getEncryptFn(n,i){var o,u;if(this.version<5&&(o=this.encryptionKey.clone().concat(y.default.lib.WordArray.create([(255&n)<<24|(65280&n)<<8|n>>8&65280|255&i,(65280&i)<<16],5))),1===this.version||2===this.version){var p=y.default.MD5(o);return p.sigBytes=Math.min(16,this.keyBits/8+5),function(n){return wordArrayToBuffer(y.default.RC4.encrypt(y.default.lib.WordArray.create(n),p).ciphertext)}}u=4===this.version?y.default.MD5(o.concat(y.default.lib.WordArray.create([1933667412],4))):this.encryptionKey;var g=PDFSecurity.generateRandomWordArray(16),w={mode:y.default.mode.CBC,padding:y.default.pad.Pkcs7,iv:g};return function(n){return wordArrayToBuffer(g.clone().concat(y.default.AES.encrypt(y.default.lib.WordArray.create(n),u,w).ciphertext))}}},{key:"end",value:function end(){this.dictionary.end()}}]),PDFSecurity}();function getPermissionsR3(){var n=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{},i=-3904;return"lowResolution"===n.printing&&(i|=4),"highResolution"===n.printing&&(i|=2052),n.modifying&&(i|=8),n.copying&&(i|=16),n.annotating&&(i|=32),n.fillingForms&&(i|=256),n.contentAccessibility&&(i|=512),n.documentAssembly&&(i|=1024),i}function processPasswordR2R3R4(){for(var i=arguments.length>0&&void 0!==arguments[0]?arguments[0]:"",o=new n(32),u=i.length,p=0;p<u&&p<32;){var g=i.charCodeAt(p);if(g>255)throw new Error("Password contains one or more invalid characters.");o[p]=g,p++}for(;p<32;)o[p]=bt[p-u],p++;return y.default.lib.WordArray.create(o)}function processPasswordR5(){var i=arguments.length>0&&void 0!==arguments[0]?arguments[0]:"";i=unescape(encodeURIComponent(saslprep(i)));for(var o=Math.min(127,i.length),u=new n(o),p=0;p<o;p++)u[p]=i.charCodeAt(p);return y.default.lib.WordArray.create(u)}function lsbFirstWord(n){return(255&n)<<24|(65280&n)<<8|n>>8&65280|n>>24&255}function wordArrayToBuffer(i){for(var o=[],u=0;u<i.sigBytes;u++)o.push(i.words[Math.floor(u/4)]>>8*(3-u%4)&255);return n.from(o)}var dt,pt,gt,yt,vt,mt,bt=[40,191,78,94,78,117,138,65,100,0,78,86,255,250,1,8,46,46,0,182,208,104,62,128,47,12,169,254,100,83,105,122],wt=N.number,xt=function(){function PDFGradient(n){_classCallCheck(this,PDFGradient),this.doc=n,this.stops=[],this.embedded=!1,this.transform=[1,0,0,1,0,0]}return _createClass(PDFGradient,[{key:"stop",value:function stop(n,i,o){if(null==o&&(o=1),i=this.doc._normalizeColor(i),0===this.stops.length)if(3===i.length)this._colorSpace="DeviceRGB";else if(4===i.length)this._colorSpace="DeviceCMYK";else{if(1!==i.length)throw new Error("Unknown color space");this._colorSpace="DeviceGray"}else if("DeviceRGB"===this._colorSpace&&3!==i.length||"DeviceCMYK"===this._colorSpace&&4!==i.length||"DeviceGray"===this._colorSpace&&1!==i.length)throw new Error("All gradient stops must use the same color space");return o=Math.max(0,Math.min(1,o)),this.stops.push([n,i,o]),this}},{key:"setTransform",value:function setTransform(n,i,o,u,p,g){return this.transform=[n,i,o,u,p,g],this}},{key:"embed",value:function embed(n){var i,o=this.stops.length;if(0!==o){this.embedded=!0,this.matrix=n;var u=this.stops[o-1];u[0]<1&&this.stops.push([1,u[1],u[2]]);for(var p=[],g=[],y=[],w=0;w<o-1;w++)g.push(0,1),w+2!==o&&p.push(this.stops[w+1][0]),i=this.doc.ref({FunctionType:2,Domain:[0,1],C0:this.stops[w+0][1],C1:this.stops[w+1][1],N:1}),y.push(i),i.end();1===o?i=y[0]:(i=this.doc.ref({FunctionType:3,Domain:[0,1],Functions:y,Bounds:p,Encode:g})).end(),this.id="Sh".concat(++this.doc._gradCount);var x=this.shader(i);x.end();var _=this.doc.ref({Type:"Pattern",PatternType:2,Shading:x,Matrix:this.matrix.map(wt)});if(_.end(),this.stops.some((function(n){return n[2]<1}))){var k=this.opacityGradient();k._colorSpace="DeviceGray";var P=!0,E=!1,O=void 0;try{for(var B,I=this.stops[Symbol.iterator]();!(P=(B=I.next()).done);P=!0){var D=B.value;k.stop(D[0],[D[2]])}}catch(n){E=!0,O=n}finally{try{P||null==I.return||I.return()}finally{if(E)throw O}}k=k.embed(this.matrix);var R=[0,0,this.doc.page.width,this.doc.page.height],N=this.doc.ref({Type:"XObject",Subtype:"Form",FormType:1,BBox:R,Group:{Type:"Group",S:"Transparency",CS:"DeviceGray"},Resources:{ProcSet:["PDF","Text","ImageB","ImageC","ImageI"],Pattern:{Sh1:k}}});N.write("/Pattern cs /Sh1 scn"),N.end("".concat(R.join(" ")," re f"));var U=this.doc.ref({Type:"ExtGState",SMask:{Type:"Mask",S:"Luminosity",G:N}});U.end();var W=this.doc.ref({Type:"Pattern",PatternType:1,PaintType:1,TilingType:2,BBox:R,XStep:R[2],YStep:R[3],Resources:{ProcSet:["PDF","Text","ImageB","ImageC","ImageI"],Pattern:{Sh1:_},ExtGState:{Gs1:U}}});W.write("/Gs1 gs /Pattern cs /Sh1 scn"),W.end("".concat(R.join(" ")," re f")),this.doc.page.patterns[this.id]=W}else this.doc.page.patterns[this.id]=_;return _}}},{key:"apply",value:function apply(n){var i=_slicedToArray(this.doc._ctm,6),o=i[0],u=i[1],p=i[2],g=i[3],y=i[4],w=i[5],x=_slicedToArray(this.transform,6),_=x[0],k=x[1],P=x[2],E=x[3],O=x[4],B=x[5],I=[o*_+p*k,u*_+g*k,o*P+p*E,u*P+g*E,o*O+p*B+y,u*O+g*B+w];return this.embedded&&I.join(" ")===this.matrix.join(" ")||this.embed(I),this.doc.addContent("/".concat(this.id," ").concat(n))}}]),PDFGradient}(),_t={PDFGradient:xt,PDFLinearGradient:function(n){function PDFLinearGradient(n,i,o,u,p){var g;return _classCallCheck(this,PDFLinearGradient),(g=_possibleConstructorReturn(this,_getPrototypeOf(PDFLinearGradient).call(this,n))).x1=i,g.y1=o,g.x2=u,g.y2=p,g}return _inherits(PDFLinearGradient,n),_createClass(PDFLinearGradient,[{key:"shader",value:function shader(n){return this.doc.ref({ShadingType:2,ColorSpace:this._colorSpace,Coords:[this.x1,this.y1,this.x2,this.y2],Function:n,Extend:[!0,!0]})}},{key:"opacityGradient",value:function opacityGradient(){return new PDFLinearGradient(this.doc,this.x1,this.y1,this.x2,this.y2)}}]),PDFLinearGradient}(xt),PDFRadialGradient:function(n){function PDFRadialGradient(n,i,o,u,p,g,y){var w;return _classCallCheck(this,PDFRadialGradient),(w=_possibleConstructorReturn(this,_getPrototypeOf(PDFRadialGradient).call(this,n))).doc=n,w.x1=i,w.y1=o,w.r1=u,w.x2=p,w.y2=g,w.r2=y,w}return _inherits(PDFRadialGradient,n),_createClass(PDFRadialGradient,[{key:"shader",value:function shader(n){return this.doc.ref({ShadingType:3,ColorSpace:this._colorSpace,Coords:[this.x1,this.y1,this.r1,this.x2,this.y2,this.r2],Function:n,Extend:[!0,!0]})}},{key:"opacityGradient",value:function opacityGradient(){return new PDFRadialGradient(this.doc,this.x1,this.y1,this.r1,this.x2,this.y2,this.r2)}}]),PDFRadialGradient}(xt)},St=_t.PDFGradient,Ct=_t.PDFLinearGradient,kt=_t.PDFRadialGradient,At={initColor:function initColor(){return this._opacityRegistry={},this._opacityCount=0,this._gradCount=0},_normalizeColor:function _normalizeColor(n){if(n instanceof St)return n;if("string"==typeof n)if("#"===n.charAt(0)){4===n.length&&(n=n.replace(/#([0-9A-F])([0-9A-F])([0-9A-F])/i,"#$1$1$2$2$3$3"));var i=parseInt(n.slice(1),16);n=[i>>16,i>>8&255,255&i]}else Pt[n]&&(n=Pt[n]);return Array.isArray(n)?(3===n.length?n=n.map((function(n){return n/255})):4===n.length&&(n=n.map((function(n){return n/100}))),n):null},_setColor:function _setColor(n,i){if(!(n=this._normalizeColor(n)))return!1;var o=i?"SCN":"scn";if(n instanceof St)this._setColorSpace("Pattern",i),n.apply(o);else{var u=4===n.length?"DeviceCMYK":"DeviceRGB";this._setColorSpace(u,i),n=n.join(" "),this.addContent("".concat(n," ").concat(o))}return!0},_setColorSpace:function _setColorSpace(n,i){var o=i?"CS":"cs";return this.addContent("/".concat(n," ").concat(o))},fillColor:function fillColor(n,i){return this._setColor(n,!1)&&this.fillOpacity(i),this._fillColor=[n,i],this},strokeColor:function strokeColor(n,i){return this._setColor(n,!0)&&this.strokeOpacity(i),this},opacity:function opacity(n){return this._doOpacity(n,n),this},fillOpacity:function fillOpacity(n){return this._doOpacity(n,null),this},strokeOpacity:function strokeOpacity(n){return this._doOpacity(null,n),this},_doOpacity:function _doOpacity(n,i){var o,u;if(null!=n||null!=i){null!=n&&(n=Math.max(0,Math.min(1,n))),null!=i&&(i=Math.max(0,Math.min(1,i)));var p="".concat(n,"_").concat(i);if(this._opacityRegistry[p]){var g=_slicedToArray(this._opacityRegistry[p],2);o=g[0],u=g[1]}else{o={Type:"ExtGState"},null!=n&&(o.ca=n),null!=i&&(o.CA=i),(o=this.ref(o)).end();var y=++this._opacityCount;u="Gs".concat(y),this._opacityRegistry[p]=[o,u]}return this.page.ext_gstates[u]=o,this.addContent("/".concat(u," gs"))}},linearGradient:function linearGradient(n,i,o,u){return new Ct(this,n,i,o,u)},radialGradient:function radialGradient(n,i,o,u,p,g){return new kt(this,n,i,o,u,p,g)}},Pt={aliceblue:[240,248,255],antiquewhite:[250,235,215],aqua:[0,255,255],aquamarine:[127,255,212],azure:[240,255,255],beige:[245,245,220],bisque:[255,228,196],black:[0,0,0],blanchedalmond:[255,235,205],blue:[0,0,255],blueviolet:[138,43,226],brown:[165,42,42],burlywood:[222,184,135],cadetblue:[95,158,160],chartreuse:[127,255,0],chocolate:[210,105,30],coral:[255,127,80],cornflowerblue:[100,149,237],cornsilk:[255,248,220],crimson:[220,20,60],cyan:[0,255,255],darkblue:[0,0,139],darkcyan:[0,139,139],darkgoldenrod:[184,134,11],darkgray:[169,169,169],darkgreen:[0,100,0],darkgrey:[169,169,169],darkkhaki:[189,183,107],darkmagenta:[139,0,139],darkolivegreen:[85,107,47],darkorange:[255,140,0],darkorchid:[153,50,204],darkred:[139,0,0],darksalmon:[233,150,122],darkseagreen:[143,188,143],darkslateblue:[72,61,139],darkslategray:[47,79,79],darkslategrey:[47,79,79],darkturquoise:[0,206,209],darkviolet:[148,0,211],deeppink:[255,20,147],deepskyblue:[0,191,255],dimgray:[105,105,105],dimgrey:[105,105,105],dodgerblue:[30,144,255],firebrick:[178,34,34],floralwhite:[255,250,240],forestgreen:[34,139,34],fuchsia:[255,0,255],gainsboro:[220,220,220],ghostwhite:[248,248,255],gold:[255,215,0],goldenrod:[218,165,32],gray:[128,128,128],grey:[128,128,128],green:[0,128,0],greenyellow:[173,255,47],honeydew:[240,255,240],hotpink:[255,105,180],indianred:[205,92,92],indigo:[75,0,130],ivory:[255,255,240],khaki:[240,230,140],lavender:[230,230,250],lavenderblush:[255,240,245],lawngreen:[124,252,0],lemonchiffon:[255,250,205],lightblue:[173,216,230],lightcoral:[240,128,128],lightcyan:[224,255,255],lightgoldenrodyellow:[250,250,210],lightgray:[211,211,211],lightgreen:[144,238,144],lightgrey:[211,211,211],lightpink:[255,182,193],lightsalmon:[255,160,122],lightseagreen:[32,178,170],lightskyblue:[135,206,250],lightslategray:[119,136,153],lightslategrey:[119,136,153],lightsteelblue:[176,196,222],lightyellow:[255,255,224],lime:[0,255,0],limegreen:[50,205,50],linen:[250,240,230],magenta:[255,0,255],maroon:[128,0,0],mediumaquamarine:[102,205,170],mediumblue:[0,0,205],mediumorchid:[186,85,211],mediumpurple:[147,112,219],mediumseagreen:[60,179,113],mediumslateblue:[123,104,238],mediumspringgreen:[0,250,154],mediumturquoise:[72,209,204],mediumvioletred:[199,21,133],midnightblue:[25,25,112],mintcream:[245,255,250],mistyrose:[255,228,225],moccasin:[255,228,181],navajowhite:[255,222,173],navy:[0,0,128],oldlace:[253,245,230],olive:[128,128,0],olivedrab:[107,142,35],orange:[255,165,0],orangered:[255,69,0],orchid:[218,112,214],palegoldenrod:[238,232,170],palegreen:[152,251,152],paleturquoise:[175,238,238],palevioletred:[219,112,147],papayawhip:[255,239,213],peachpuff:[255,218,185],peru:[205,133,63],pink:[255,192,203],plum:[221,160,221],powderblue:[176,224,230],purple:[128,0,128],red:[255,0,0],rosybrown:[188,143,143],royalblue:[65,105,225],saddlebrown:[139,69,19],salmon:[250,128,114],sandybrown:[244,164,96],seagreen:[46,139,87],seashell:[255,245,238],sienna:[160,82,45],silver:[192,192,192],skyblue:[135,206,235],slateblue:[106,90,205],slategray:[112,128,144],slategrey:[112,128,144],snow:[255,250,250],springgreen:[0,255,127],steelblue:[70,130,180],tan:[210,180,140],teal:[0,128,128],thistle:[216,191,216],tomato:[255,99,71],turquoise:[64,224,208],violet:[238,130,238],wheat:[245,222,179],white:[255,255,255],whitesmoke:[245,245,245],yellow:[255,255,0],yellowgreen:[154,205,50]};dt=pt=gt=yt=vt=mt=0;var Tt={A:7,a:7,C:6,c:6,H:1,h:1,L:2,l:2,M:2,m:2,Q:4,q:4,S:4,s:4,T:2,t:2,V:1,v:1,Z:0,z:0},Et={M:function M(n,i){return dt=i[0],pt=i[1],gt=yt=null,vt=dt,mt=pt,n.moveTo(dt,pt)},m:function m(n,i){return dt+=i[0],pt+=i[1],gt=yt=null,vt=dt,mt=pt,n.moveTo(dt,pt)},C:function C(n,i){return dt=i[4],pt=i[5],gt=i[2],yt=i[3],n.bezierCurveTo.apply(n,_toConsumableArray(i))},c:function c(n,i){return n.bezierCurveTo(i[0]+dt,i[1]+pt,i[2]+dt,i[3]+pt,i[4]+dt,i[5]+pt),gt=dt+i[2],yt=pt+i[3],dt+=i[4],pt+=i[5]},S:function S(n,i){return null===gt&&(gt=dt,yt=pt),n.bezierCurveTo(dt-(gt-dt),pt-(yt-pt),i[0],i[1],i[2],i[3]),gt=i[0],yt=i[1],dt=i[2],pt=i[3]},s:function s(n,i){return null===gt&&(gt=dt,yt=pt),n.bezierCurveTo(dt-(gt-dt),pt-(yt-pt),dt+i[0],pt+i[1],dt+i[2],pt+i[3]),gt=dt+i[0],yt=pt+i[1],dt+=i[2],pt+=i[3]},Q:function Q(n,i){return gt=i[0],yt=i[1],dt=i[2],pt=i[3],n.quadraticCurveTo(i[0],i[1],dt,pt)},q:function q(n,i){return n.quadraticCurveTo(i[0]+dt,i[1]+pt,i[2]+dt,i[3]+pt),gt=dt+i[0],yt=pt+i[1],dt+=i[2],pt+=i[3]},T:function T(n,i){return null===gt?(gt=dt,yt=pt):(gt=dt-(gt-dt),yt=pt-(yt-pt)),n.quadraticCurveTo(gt,yt,i[0],i[1]),gt=dt-(gt-dt),yt=pt-(yt-pt),dt=i[0],pt=i[1]},t:function t(n,i){return null===gt?(gt=dt,yt=pt):(gt=dt-(gt-dt),yt=pt-(yt-pt)),n.quadraticCurveTo(gt,yt,dt+i[0],pt+i[1]),dt+=i[0],pt+=i[1]},A:function A(n,i){return Ot(n,dt,pt,i),dt=i[5],pt=i[6]},a:function a(n,i){return i[5]+=dt,i[6]+=pt,Ot(n,dt,pt,i),dt=i[5],pt=i[6]},L:function L(n,i){return dt=i[0],pt=i[1],gt=yt=null,n.lineTo(dt,pt)},l:function l(n,i){return dt+=i[0],pt+=i[1],gt=yt=null,n.lineTo(dt,pt)},H:function H(n,i){return dt=i[0],gt=yt=null,n.lineTo(dt,pt)},h:function h(n,i){return dt+=i[0],gt=yt=null,n.lineTo(dt,pt)},V:function V(n,i){return pt=i[0],gt=yt=null,n.lineTo(dt,pt)},v:function v(n,i){return pt+=i[0],gt=yt=null,n.lineTo(dt,pt)},Z:function Z(n){return n.closePath(),dt=vt,pt=mt},z:function z(n){return n.closePath(),dt=vt,pt=mt}},Ot=function solveArc(n,i,o,u){var p=_slicedToArray(u,7),g=p[0],y=p[1],w=p[2],x=p[3],_=p[4],k=p[5],P=p[6],E=Bt(k,P,g,y,x,_,w,i,o),O=!0,B=!1,I=void 0;try{for(var D,R=E[Symbol.iterator]();!(O=(D=R.next()).done);O=!0){var N=D.value,U=It.apply(void 0,_toConsumableArray(N));n.bezierCurveTo.apply(n,_toConsumableArray(U))}}catch(n){B=!0,I=n}finally{try{O||null==R.return||R.return()}finally{if(B)throw I}}},Bt=function arcToSegments(n,i,o,u,p,g,y,w,x){var _=y*(Math.PI/180),k=Math.sin(_),P=Math.cos(_);o=Math.abs(o),u=Math.abs(u);var E=(gt=P*(w-n)*.5+k*(x-i)*.5)*gt/(o*o)+(yt=P*(x-i)*.5-k*(w-n)*.5)*yt/(u*u);E>1&&(o*=E=Math.sqrt(E),u*=E);var O=P/o,B=k/o,I=-k/u,D=P/u,R=O*w+B*x,N=I*w+D*x,U=O*n+B*i,W=I*n+D*i,G=1/((U-R)*(U-R)+(W-N)*(W-N))-.25;G<0&&(G=0);var j=Math.sqrt(G);g===p&&(j=-j);var X=.5*(R+U)-j*(W-N),Y=.5*(N+W)+j*(U-R),K=Math.atan2(N-Y,R-X),J=Math.atan2(W-Y,U-X)-K;J<0&&1===g?J+=2*Math.PI:J>0&&0===g&&(J-=2*Math.PI);for(var $=Math.ceil(Math.abs(J/(.5*Math.PI+.001))),tt=[],et=0;et<$;et++){var rt=K+et*J/$,nt=K+(et+1)*J/$;tt[et]=[X,Y,rt,nt,o,u,k,P]}return tt},It=function segmentToBezier(n,i,o,u,p,g,y,w){var x=w*p,_=-y*g,k=y*p,P=w*g,E=.5*(u-o),O=8/3*Math.sin(.5*E)*Math.sin(.5*E)/Math.sin(E),B=n+Math.cos(o)-O*Math.sin(o),I=i+Math.sin(o)+O*Math.cos(o),D=n+Math.cos(u),R=i+Math.sin(u),N=D+O*Math.sin(u),U=R-O*Math.cos(u);return[x*B+_*I,k*B+P*I,x*N+_*U,k*N+P*U,x*D+_*R,k*D+P*R]},Ft=function(){function SVGPath(){_classCallCheck(this,SVGPath)}return _createClass(SVGPath,null,[{key:"apply",value:function apply(n,i){!function apply(n,i){dt=pt=gt=yt=vt=mt=0;for(var o=0;o<n.length;o++){var u=n[o];"function"==typeof Et[u.cmd]&&Et[u.cmd](i,u.args)}}(function parse(n){var i,o=[],u=[],p="",g=!1,y=0,w=!0,x=!1,_=void 0;try{for(var k,P=n[Symbol.iterator]();!(w=(k=P.next()).done);w=!0){var E=k.value;if(null!=Tt[E])y=Tt[E],i&&(p.length>0&&(u[u.length]=+p),o[o.length]={cmd:i,args:u},u=[],p="",g=!1),i=E;else if([" ",","].includes(E)||"-"===E&&p.length>0&&"e"!==p[p.length-1]||"."===E&&g){if(0===p.length)continue;u.length===y?(o[o.length]={cmd:i,args:u},u=[+p],"M"===i&&(i="L"),"m"===i&&(i="l")):u[u.length]=+p,g="."===E,p=["-","."].includes(E)?E:""}else p+=E,"."===E&&(g=!0)}}catch(n){x=!0,_=n}finally{try{w||null==P.return||P.return()}finally{if(x)throw _}}return p.length>0&&(u.length===y?(o[o.length]={cmd:i,args:u},u=[+p],"M"===i&&(i="L"),"m"===i&&(i="l")):u[u.length]=+p),o[o.length]={cmd:i,args:u},o}(i),n)}}]),SVGPath}(),Lt=N.number,Dt=(Math.sqrt(2)-1)/3*4,Mt={initVector:function initVector(){return this._ctm=[1,0,0,1,0,0],this._ctmStack=[]},save:function save(){return this._ctmStack.push(this._ctm.slice()),this.addContent("q")},restore:function restore(){return this._ctm=this._ctmStack.pop()||[1,0,0,1,0,0],this.addContent("Q")},closePath:function closePath(){return this.addContent("h")},lineWidth:function lineWidth(n){return this.addContent("".concat(Lt(n)," w"))},_CAP_STYLES:{BUTT:0,ROUND:1,SQUARE:2},lineCap:function lineCap(n){return"string"==typeof n&&(n=this._CAP_STYLES[n.toUpperCase()]),this.addContent("".concat(n," J"))},_JOIN_STYLES:{MITER:0,ROUND:1,BEVEL:2},lineJoin:function lineJoin(n){return"string"==typeof n&&(n=this._JOIN_STYLES[n.toUpperCase()]),this.addContent("".concat(n," j"))},miterLimit:function miterLimit(n){return this.addContent("".concat(Lt(n)," M"))},dash:function dash(n){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},o=n;Array.isArray(n)||(n=[n,i.space||n]);var u=n.every((function(n){return Number.isFinite(n)&&n>0}));if(!u)throw new Error("dash(".concat(JSON.stringify(o),", ").concat(JSON.stringify(i),") invalid, lengths must be numeric and greater than zero"));return n=n.map(Lt).join(" "),this.addContent("[".concat(n,"] ").concat(Lt(i.phase||0)," d"))},undash:function undash(){return this.addContent("[] 0 d")},moveTo:function moveTo(n,i){return this.addContent("".concat(Lt(n)," ").concat(Lt(i)," m"))},lineTo:function lineTo(n,i){return this.addContent("".concat(Lt(n)," ").concat(Lt(i)," l"))},bezierCurveTo:function bezierCurveTo(n,i,o,u,p,g){return this.addContent("".concat(Lt(n)," ").concat(Lt(i)," ").concat(Lt(o)," ").concat(Lt(u)," ").concat(Lt(p)," ").concat(Lt(g)," c"))},quadraticCurveTo:function quadraticCurveTo(n,i,o,u){return this.addContent("".concat(Lt(n)," ").concat(Lt(i)," ").concat(Lt(o)," ").concat(Lt(u)," v"))},rect:function rect(n,i,o,u){return this.addContent("".concat(Lt(n)," ").concat(Lt(i)," ").concat(Lt(o)," ").concat(Lt(u)," re"))},roundedRect:function roundedRect(n,i,o,u,p){null==p&&(p=0);var g=(p=Math.min(p,.5*o,.5*u))*(1-Dt);return this.moveTo(n+p,i),this.lineTo(n+o-p,i),this.bezierCurveTo(n+o-g,i,n+o,i+g,n+o,i+p),this.lineTo(n+o,i+u-p),this.bezierCurveTo(n+o,i+u-g,n+o-g,i+u,n+o-p,i+u),this.lineTo(n+p,i+u),this.bezierCurveTo(n+g,i+u,n,i+u-g,n,i+u-p),this.lineTo(n,i+p),this.bezierCurveTo(n,i+g,n+g,i,n+p,i),this.closePath()},ellipse:function ellipse(n,i,o,u){null==u&&(u=o);var p=o*Dt,g=u*Dt,y=(n-=o)+2*o,w=(i-=u)+2*u,x=n+o,_=i+u;return this.moveTo(n,_),this.bezierCurveTo(n,_-g,x-p,i,x,i),this.bezierCurveTo(x+p,i,y,_-g,y,_),this.bezierCurveTo(y,_+g,x+p,w,x,w),this.bezierCurveTo(x-p,w,n,_+g,n,_),this.closePath()},circle:function circle(n,i,o){return this.ellipse(n,i,o)},arc:function arc(n,i,o,u,p,g){null==g&&(g=!1);var y=2*Math.PI,w=.5*Math.PI,x=p-u;if(Math.abs(x)>y)x=y;else if(0!==x&&g!==x<0){x=(g?-1:1)*y+x}var _=Math.ceil(Math.abs(x)/w),k=x/_,P=k/w*Dt*o,E=u,O=-Math.sin(E)*P,B=Math.cos(E)*P,I=n+Math.cos(E)*o,D=i+Math.sin(E)*o;this.moveTo(I,D);for(var R=0;R<_;R++){var N=I+O,U=D+B;E+=k,I=n+Math.cos(E)*o,D=i+Math.sin(E)*o;var W=I-(O=-Math.sin(E)*P),G=D-(B=Math.cos(E)*P);this.bezierCurveTo(N,U,W,G,I,D)}return this},polygon:function polygon(){for(var n=arguments.length,i=new Array(n),o=0;o<n;o++)i[o]=arguments[o];this.moveTo.apply(this,_toConsumableArray(i.shift()||[]));for(var u=0,p=i;u<p.length;u++){var g=p[u];this.lineTo.apply(this,_toConsumableArray(g||[]))}return this.closePath()},path:function path(n){return Ft.apply(this,n),this},_windingRule:function _windingRule(n){return/even-?odd/.test(n)?"*":""},fill:function fill(n,i){return/(even-?odd)|(non-?zero)/.test(n)&&(i=n,n=null),n&&this.fillColor(n),this.addContent("f".concat(this._windingRule(i)))},stroke:function stroke(n){return n&&this.strokeColor(n),this.addContent("S")},fillAndStroke:function fillAndStroke(n,i,o){null==i&&(i=n);var u=/(even-?odd)|(non-?zero)/;return u.test(n)&&(o=n,n=null),u.test(i)&&(o=i,i=n),n&&(this.fillColor(n),this.strokeColor(i)),this.addContent("B".concat(this._windingRule(o)))},clip:function clip(n){return this.addContent("W".concat(this._windingRule(n)," n"))},transform:function transform(n,i,o,u,p,g){var y=this._ctm,w=_slicedToArray(y,6),x=w[0],_=w[1],k=w[2],P=w[3],E=w[4],O=w[5];y[0]=x*n+k*i,y[1]=_*n+P*i,y[2]=x*o+k*u,y[3]=_*o+P*u,y[4]=x*p+k*g+E,y[5]=_*p+P*g+O;var B=[n,i,o,u,p,g].map((function(n){return Lt(n)})).join(" ");return this.addContent("".concat(B," cm"))},translate:function translate(n,i){return this.transform(1,0,0,1,n,i)},rotate:function rotate(n){var i,o=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},u=n*Math.PI/180,p=Math.cos(u),g=Math.sin(u),y=i=0;if(null!=o.origin){var w=_slicedToArray(o.origin,2),x=(y=w[0])*p-(i=w[1])*g,_=y*g+i*p;y-=x,i-=_}return this.transform(p,g,-g,p,y,i)},scale:function scale(n,i){var o,u=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{};null==i&&(i=n),"object"==typeof i&&(u=i,i=n);var p=o=0;if(null!=u.origin){var g=_slicedToArray(u.origin,2);p=g[0],o=g[1],p-=n*p,o-=i*o}return this.transform(n,0,0,i,p,o)}},Rt={402:131,8211:150,8212:151,8216:145,8217:146,8218:130,8220:147,8221:148,8222:132,8224:134,8225:135,8226:149,8230:133,8364:128,8240:137,8249:139,8250:155,710:136,8482:153,338:140,339:156,732:152,352:138,353:154,376:159,381:142,382:158},Nt=".notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n  \nspace         exclam         quotedbl       numbersign\ndollar        percent        ampersand      quotesingle\nparenleft     parenright     asterisk       plus\ncomma         hyphen         period         slash\nzero          one            two            three\nfour          five           six            seven\neight         nine           colon          semicolon\nless          equal          greater        question\n  \nat            A              B              C\nD             E              F              G\nH             I              J              K\nL             M              N              O\nP             Q              R              S\nT             U              V              W\nX             Y              Z              bracketleft\nbackslash     bracketright   asciicircum    underscore\n  \ngrave         a              b              c\nd             e              f              g\nh             i              j              k\nl             m              n              o\np             q              r              s\nt             u              v              w\nx             y              z              braceleft\nbar           braceright     asciitilde     .notdef\n  \nEuro          .notdef        quotesinglbase florin\nquotedblbase  ellipsis       dagger         daggerdbl\ncircumflex    perthousand    Scaron         guilsinglleft\nOE            .notdef        Zcaron         .notdef\n.notdef       quoteleft      quoteright     quotedblleft\nquotedblright bullet         endash         emdash\ntilde         trademark      scaron         guilsinglright\noe            .notdef        zcaron         ydieresis\n  \nspace         exclamdown     cent           sterling\ncurrency      yen            brokenbar      section\ndieresis      copyright      ordfeminine    guillemotleft\nlogicalnot    hyphen         registered     macron\ndegree        plusminus      twosuperior    threesuperior\nacute         mu             paragraph      periodcentered\ncedilla       onesuperior    ordmasculine   guillemotright\nonequarter    onehalf        threequarters  questiondown\n  \nAgrave        Aacute         Acircumflex    Atilde\nAdieresis     Aring          AE             Ccedilla\nEgrave        Eacute         Ecircumflex    Edieresis\nIgrave        Iacute         Icircumflex    Idieresis\nEth           Ntilde         Ograve         Oacute\nOcircumflex   Otilde         Odieresis      multiply\nOslash        Ugrave         Uacute         Ucircumflex\nUdieresis     Yacute         Thorn          germandbls\n  \nagrave        aacute         acircumflex    atilde\nadieresis     aring          ae             ccedilla\negrave        eacute         ecircumflex    edieresis\nigrave        iacute         icircumflex    idieresis\neth           ntilde         ograve         oacute\nocircumflex   otilde         odieresis      divide\noslash        ugrave         uacute         ucircumflex\nudieresis     yacute         thorn          ydieresis".split(/\s+/),Ut=function(){function AFMFont(n){_classCallCheck(this,AFMFont),this.contents=n,this.attributes={},this.glyphWidths={},this.boundingBoxes={},this.kernPairs={},this.parse(),this.charWidths=new Array(256);for(var i=0;i<=255;i++)this.charWidths[i]=this.glyphWidths[Nt[i]];this.bbox=this.attributes.FontBBox.split(/\s+/).map((function(n){return+n})),this.ascender=+(this.attributes.Ascender||0),this.descender=+(this.attributes.Descender||0),this.xHeight=+(this.attributes.XHeight||0),this.capHeight=+(this.attributes.CapHeight||0),this.lineGap=this.bbox[3]-this.bbox[1]-(this.ascender-this.descender)}return _createClass(AFMFont,null,[{key:"open",value:function open(n){return new AFMFont(P.readFileSync(n,"utf8"))}}]),_createClass(AFMFont,[{key:"parse",value:function parse(){var n="",i=!0,o=!1,u=void 0;try{for(var p,g=this.contents.split("\n")[Symbol.iterator]();!(i=(p=g.next()).done);i=!0){var y,w,x=p.value;if(y=x.match(/^Start(\w+)/))n=y[1];else if(y=x.match(/^End(\w+)/))n="";else switch(n){case"FontMetrics":var _=(y=x.match(/(^\w+)\s+(.*)/))[1],k=y[2];(w=this.attributes[_])?(Array.isArray(w)||(w=this.attributes[_]=[w]),w.push(k)):this.attributes[_]=k;break;case"CharMetrics":if(!/^CH?\s/.test(x))continue;var P=x.match(/\bN\s+(\.?\w+)\s*;/)[1];this.glyphWidths[P]=+x.match(/\bWX\s+(\d+)\s*;/)[1];break;case"KernPairs":(y=x.match(/^KPX\s+(\.?\w+)\s+(\.?\w+)\s+(-?\d+)/))&&(this.kernPairs[y[1]+"\0"+y[2]]=parseInt(y[3]))}}}catch(n){o=!0,u=n}finally{try{i||null==g.return||g.return()}finally{if(o)throw u}}}},{key:"encodeText",value:function encodeText(n){for(var i=[],o=0,u=n.length;o<u;o++){var p=n.charCodeAt(o);p=Rt[p]||p,i.push(p.toString(16))}return i}},{key:"glyphsForString",value:function glyphsForString(n){for(var i=[],o=0,u=n.length;o<u;o++){var p=n.charCodeAt(o);i.push(this.characterToGlyph(p))}return i}},{key:"characterToGlyph",value:function characterToGlyph(n){return Nt[Rt[n]||n]||".notdef"}},{key:"widthOfGlyph",value:function widthOfGlyph(n){return this.glyphWidths[n]||0}},{key:"getKernPair",value:function getKernPair(n,i){return this.kernPairs[n+"\0"+i]||0}},{key:"advancesForGlyphs",value:function advancesForGlyphs(n){for(var i=[],o=0;o<n.length;o++){var u=n[o],p=n[o+1];i.push(this.widthOfGlyph(u)+this.getKernPair(u,p))}return i}}]),AFMFont}(),Wt=function(){function PDFFont(){_classCallCheck(this,PDFFont)}return _createClass(PDFFont,[{key:"encode",value:function encode(){throw new Error("Must be implemented by subclasses")}},{key:"widthOfString",value:function widthOfString(){throw new Error("Must be implemented by subclasses")}},{key:"ref",value:function ref(){return null!=this.dictionary?this.dictionary:this.dictionary=this.document.ref()}},{key:"finalize",value:function finalize(){if(!this.embedded&&null!=this.dictionary)return this.embed(),this.embedded=!0}},{key:"embed",value:function embed(){throw new Error("Must be implemented by subclasses")}},{key:"lineHeight",value:function lineHeight(n,i){null==i&&(i=!1);var o=i?this.lineGap:0;return(this.ascender+o-this.descender)/1e3*n}}]),PDFFont}(),zt={Courier:function Courier(){return P.readFileSync(u+"/data/Courier.afm","utf8")},"Courier-Bold":function CourierBold(){return P.readFileSync(u+"/data/Courier-Bold.afm","utf8")},"Courier-Oblique":function CourierOblique(){return P.readFileSync(u+"/data/Courier-Oblique.afm","utf8")},"Courier-BoldOblique":function CourierBoldOblique(){return P.readFileSync(u+"/data/Courier-BoldOblique.afm","utf8")},Helvetica:function Helvetica(){return P.readFileSync(u+"/data/Helvetica.afm","utf8")},"Helvetica-Bold":function HelveticaBold(){return P.readFileSync(u+"/data/Helvetica-Bold.afm","utf8")},"Helvetica-Oblique":function HelveticaOblique(){return P.readFileSync(u+"/data/Helvetica-Oblique.afm","utf8")},"Helvetica-BoldOblique":function HelveticaBoldOblique(){return P.readFileSync(u+"/data/Helvetica-BoldOblique.afm","utf8")},"Times-Roman":function TimesRoman(){return P.readFileSync(u+"/data/Times-Roman.afm","utf8")},"Times-Bold":function TimesBold(){return P.readFileSync(u+"/data/Times-Bold.afm","utf8")},"Times-Italic":function TimesItalic(){return P.readFileSync(u+"/data/Times-Italic.afm","utf8")},"Times-BoldItalic":function TimesBoldItalic(){return P.readFileSync(u+"/data/Times-BoldItalic.afm","utf8")},Symbol:function Symbol(){return P.readFileSync(u+"/data/Symbol.afm","utf8")},ZapfDingbats:function ZapfDingbats(){return P.readFileSync(u+"/data/ZapfDingbats.afm","utf8")}},Gt=function(n){function StandardFont(n,i,o){var u;_classCallCheck(this,StandardFont),(u=_possibleConstructorReturn(this,_getPrototypeOf(StandardFont).call(this))).document=n,u.name=i,u.id=o,u.font=new Ut(zt[u.name]());var p=u.font;return u.ascender=p.ascender,u.descender=p.descender,u.bbox=p.bbox,u.lineGap=p.lineGap,u.xHeight=p.xHeight,u.capHeight=p.capHeight,u}return _inherits(StandardFont,n),_createClass(StandardFont,[{key:"embed",value:function embed(){return this.dictionary.data={Type:"Font",BaseFont:this.name,Subtype:"Type1",Encoding:"WinAnsiEncoding"},this.dictionary.end()}},{key:"encode",value:function encode(n){for(var i=this.font.encodeText(n),o=this.font.glyphsForString("".concat(n)),u=this.font.advancesForGlyphs(o),p=[],g=0;g<o.length;g++){var y=o[g];p.push({xAdvance:u[g],yAdvance:0,xOffset:0,yOffset:0,advanceWidth:this.font.widthOfGlyph(y)})}return[i,p]}},{key:"widthOfString",value:function widthOfString(n,i){var o=this.font.glyphsForString("".concat(n)),u=this.font.advancesForGlyphs(o),p=0,g=!0,y=!1,w=void 0;try{for(var x,_=u[Symbol.iterator]();!(g=(x=_.next()).done);g=!0){p+=x.value}}catch(n){y=!0,w=n}finally{try{g||null==_.return||_.return()}finally{if(y)throw w}}return p*(i/1e3)}}],[{key:"isStandardFont",value:function isStandardFont(n){return n in zt}}]),StandardFont}(Wt),jt=function toHex(n){return"0000".concat(n.toString(16)).slice(-4)},Vt=function(n){function EmbeddedFont(n,i,o){var u;return _classCallCheck(this,EmbeddedFont),(u=_possibleConstructorReturn(this,_getPrototypeOf(EmbeddedFont).call(this))).document=n,u.font=i,u.id=o,u.subset=u.font.createSubset(),u.unicode=[[0]],u.widths=[u.font.getGlyph(0).advanceWidth],u.name=u.font.postscriptName,u.scale=1e3/u.font.unitsPerEm,u.ascender=u.font.ascent*u.scale,u.descender=u.font.descent*u.scale,u.xHeight=u.font.xHeight*u.scale,u.capHeight=u.font.capHeight*u.scale,u.lineGap=u.font.lineGap*u.scale,u.bbox=u.font.bbox,!1!==n.options.fontLayoutCache&&(u.layoutCache=Object.create(null)),u}return _inherits(EmbeddedFont,n),_createClass(EmbeddedFont,[{key:"layoutRun",value:function layoutRun(n,i){for(var o=this.font.layout(n,i),u=0;u<o.positions.length;u++){var p=o.positions[u];for(var g in p)p[g]*=this.scale;p.advanceWidth=o.glyphs[u].advanceWidth*this.scale}return o}},{key:"layoutCached",value:function layoutCached(n){if(!this.layoutCache)return this.layoutRun(n);var i;if(i=this.layoutCache[n])return i;var o=this.layoutRun(n);return this.layoutCache[n]=o,o}},{key:"layout",value:function layout(n,i,o){if(i)return this.layoutRun(n,i);for(var u=o?null:[],p=o?null:[],g=0,y=0,w=0;w<=n.length;){var x;if(w===n.length&&y<w||(x=n.charAt(w),[" ","\t"].includes(x))){var _=this.layoutCached(n.slice(y,++w));o||(u=u.concat(_.glyphs),p=p.concat(_.positions)),g+=_.advanceWidth,y=w}else w++}return{glyphs:u,positions:p,advanceWidth:g}}},{key:"encode",value:function encode(n,i){for(var o=this.layout(n,i),u=o.glyphs,p=o.positions,g=[],y=0;y<u.length;y++){var w=u[y],x=this.subset.includeGlyph(w.id);g.push("0000".concat(x.toString(16)).slice(-4)),null==this.widths[x]&&(this.widths[x]=w.advanceWidth*this.scale),null==this.unicode[x]&&(this.unicode[x]=w.codePoints)}return[g,p]}},{key:"widthOfString",value:function widthOfString(n,i,o){return this.layout(n,o,!0).advanceWidth*(i/1e3)}},{key:"embed",value:function embed(){var n=this,i=null!=this.subset.cff,o=this.document.ref();i&&(o.data.Subtype="CIDFontType0C"),this.subset.encodeStream().on("data",(function(n){return o.write(n)})).on("end",(function(){return o.end()}));var u=((null!=this.font["OS/2"]?this.font["OS/2"].sFamilyClass:void 0)||0)>>8,p=0;this.font.post.isFixedPitch&&(p|=1),1<=u&&u<=7&&(p|=2),p|=4,10===u&&(p|=8),this.font.head.macStyle.italic&&(p|=64);var g=[1,2,3,4,5,6].map((function(i){return String.fromCharCode((n.id.charCodeAt(i)||74)+16)})).join("")+"+"+this.font.postscriptName,y=this.font.bbox,w=this.document.ref({Type:"FontDescriptor",FontName:g,Flags:p,FontBBox:[y.minX*this.scale,y.minY*this.scale,y.maxX*this.scale,y.maxY*this.scale],ItalicAngle:this.font.italicAngle,Ascent:this.ascender,Descent:this.descender,CapHeight:(this.font.capHeight||this.font.ascent)*this.scale,XHeight:(this.font.xHeight||0)*this.scale,StemV:0});i?w.data.FontFile3=o:w.data.FontFile2=o,w.end();var x={Type:"Font",Subtype:"CIDFontType0",BaseFont:g,CIDSystemInfo:{Registry:new String("Adobe"),Ordering:new String("Identity"),Supplement:0},FontDescriptor:w,W:[0,this.widths]};i||(x.Subtype="CIDFontType2",x.CIDToGIDMap="Identity");var _=this.document.ref(x);return _.end(),this.dictionary.data={Type:"Font",Subtype:"Type0",BaseFont:g,Encoding:"Identity-H",DescendantFonts:[_],ToUnicode:this.toUnicodeCmap()},this.dictionary.end()}},{key:"toUnicodeCmap",value:function toUnicodeCmap(){var n=this.document.ref(),i=[],o=!0,u=!1,p=void 0;try{for(var g,y=this.unicode[Symbol.iterator]();!(o=(g=y.next()).done);o=!0){var w=g.value,x=[],_=!0,k=!1,P=void 0;try{for(var E,O=w[Symbol.iterator]();!(_=(E=O.next()).done);_=!0){var B=E.value;B>65535&&(B-=65536,x.push(jt(B>>>10&1023|55296)),B=56320|1023&B),x.push(jt(B))}}catch(n){k=!0,P=n}finally{try{_||null==O.return||O.return()}finally{if(k)throw P}}i.push("<".concat(x.join(" "),">"))}}catch(n){u=!0,p=n}finally{try{o||null==y.return||y.return()}finally{if(u)throw p}}return n.end("/CIDInit /ProcSet findresource begin\n12 dict begin\nbegincmap\n/CIDSystemInfo <<\n  /Registry (Adobe)\n  /Ordering (UCS)\n  /Supplement 0\n>> def\n/CMapName /Adobe-Identity-UCS def\n/CMapType 2 def\n1 begincodespacerange\n<0000><ffff>\nendcodespacerange\n1 beginbfrange\n<0000> <".concat(jt(i.length-1),"> [").concat(i.join(" "),"]\nendbfrange\nendcmap\nCMapName currentdict /CMap defineresource pop\nend\nend")),n}}]),EmbeddedFont}(Wt),Ht=function(){function PDFFontFactory(){_classCallCheck(this,PDFFontFactory)}return _createClass(PDFFontFactory,null,[{key:"open",value:function open(i,o,u,p){var g;if("string"==typeof o){if(Gt.isStandardFont(o))return new Gt(i,o,p);o=P.readFileSync(o)}if(n.isBuffer(o)?g=w.default.create(o,u):o instanceof Uint8Array?g=w.default.create(new n(o),u):o instanceof ArrayBuffer&&(g=w.default.create(new n(new Uint8Array(o)),u)),null==g)throw new Error("Not a supported font format or standard PDF font.");return new Vt(i,g,p)}}]),PDFFontFactory}(),qt={initFonts:function initFonts(){var n=arguments.length>0&&void 0!==arguments[0]?arguments[0]:"Helvetica";this._fontFamilies={},this._fontCount=0,this._fontSize=12,this._font=null,this._registeredFonts={},n&&this.font(n)},font:function font(n,i,o){var u,font;if("number"==typeof i&&(o=i,i=null),"string"==typeof n&&this._registeredFonts[n]){u=n;var p=this._registeredFonts[n];n=p.src,i=p.family}else"string"!=typeof(u=i||n)&&(u=null);if(null!=o&&this.fontSize(o),font=this._fontFamilies[u])return this._font=font,this;var g="F".concat(++this._fontCount);return this._font=Ht.open(this,n,i,g),(font=this._fontFamilies[this._font.name])?(this._font=font,this):(u&&(this._fontFamilies[u]=this._font),this._font.name&&(this._fontFamilies[this._font.name]=this._font),this)},fontSize:function fontSize(n){return this._fontSize=n,this},currentLineHeight:function currentLineHeight(n){return null==n&&(n=!1),this._font.lineHeight(this._fontSize,n)},registerFont:function registerFont(n,i,o){return this._registeredFonts[n]={src:i,family:o},this}},Xt=function(n){function LineWrapper(n,i){var o;return _classCallCheck(this,LineWrapper),(o=_possibleConstructorReturn(this,_getPrototypeOf(LineWrapper).call(this))).document=n,o.indent=i.indent||0,o.characterSpacing=i.characterSpacing||0,o.wordSpacing=0===i.wordSpacing,o.columns=i.columns||1,o.columnGap=null!=i.columnGap?i.columnGap:18,o.lineWidth=(i.width-o.columnGap*(o.columns-1))/o.columns,o.spaceLeft=o.lineWidth,o.startX=o.document.x,o.startY=o.document.y,o.column=1,o.ellipsis=i.ellipsis,o.continuedX=0,o.features=i.features,null!=i.height?(o.height=i.height,o.maxY=o.startY+i.height):o.maxY=o.document.page.maxY(),o.on("firstLine",(function(n){var i=o.continuedX||o.indent;return o.document.x+=i,o.lineWidth-=i,o.once("line",(function(){if(o.document.x-=i,o.lineWidth+=i,n.continued&&!o.continuedX&&(o.continuedX=o.indent),!n.continued)return o.continuedX=0}))})),o.on("lastLine",(function(n){var i=n.align;return"justify"===i&&(n.align="left"),o.lastLine=!0,o.once("line",(function(){return o.document.y+=n.paragraphGap||0,n.align=i,o.lastLine=!1}))})),o}return _inherits(LineWrapper,n),_createClass(LineWrapper,[{key:"wordWidth",value:function wordWidth(n){return this.document.widthOfString(n,this)+this.characterSpacing+this.wordSpacing}},{key:"eachWord",value:function eachWord(n,i){for(var o,u=new _.default(n),p=null,g=Object.create(null);o=u.nextBreak();){var y,w=n.slice((null!=p?p.position:void 0)||0,o.position),x=null!=g[w]?g[w]:g[w]=this.wordWidth(w);if(x>this.lineWidth+this.continuedX)for(var k=p,P={};w.length;){var E,O;x>this.spaceLeft?(E=Math.ceil(this.spaceLeft/(x/w.length)),O=(x=this.wordWidth(w.slice(0,E)))<=this.spaceLeft&&E<w.length):E=w.length;for(var B=x>this.spaceLeft&&E>0;B||O;)B?B=(x=this.wordWidth(w.slice(0,--E)))>this.spaceLeft&&E>0:(B=(x=this.wordWidth(w.slice(0,++E)))>this.spaceLeft&&E>0,O=x<=this.spaceLeft&&E<w.length);if(0===E&&this.spaceLeft===this.lineWidth&&(E=1),P.required=o.required||E<w.length,y=i(w.slice(0,E),x,P,k),k={required:!1},w=w.slice(E),x=this.wordWidth(w),!1===y)break}else y=i(w,x,o,p);if(!1===y)break;p=o}}},{key:"wrap",value:function wrap(n,i){var o=this;null!=i.indent&&(this.indent=i.indent),null!=i.characterSpacing&&(this.characterSpacing=i.characterSpacing),null!=i.wordSpacing&&(this.wordSpacing=i.wordSpacing),null!=i.ellipsis&&(this.ellipsis=i.ellipsis);var u=this.document.y+this.document.currentLineHeight(!0);(this.document.y>this.maxY||u>this.maxY)&&this.nextSection();var p="",g=0,y=0,w=0,x=this.document.y,_=function emitLine(){return i.textWidth=g+o.wordSpacing*(y-1),i.wordCount=y,i.lineWidth=o.lineWidth,x=o.document.y,o.emit("line",p,i,o),w++};return this.emit("sectionStart",i,this),this.eachWord(n,(function(n,u,w,x){if((null==x||x.required)&&(o.emit("firstLine",i,o),o.spaceLeft=o.lineWidth),u<=o.spaceLeft&&(p+=n,g+=u,y++),w.required||u>o.spaceLeft){var k=o.document.currentLineHeight(!0);if(null!=o.height&&o.ellipsis&&o.document.y+2*k>o.maxY&&o.column>=o.columns){for(!0===o.ellipsis&&(o.ellipsis="…"),p=p.replace(/\s+$/,""),g=o.wordWidth(p+o.ellipsis);p&&g>o.lineWidth;)p=p.slice(0,-1).replace(/\s+$/,""),g=o.wordWidth(p+o.ellipsis);g<=o.lineWidth&&(p+=o.ellipsis),g=o.wordWidth(p)}if(w.required&&(u>o.spaceLeft&&(_(),p=n,g=u,y=1),o.emit("lastLine",i,o)),_(),o.document.y+k>o.maxY)if(!o.nextSection())return y=0,p="",!1;return w.required?(o.spaceLeft=o.lineWidth,p="",g=0,y=0):(o.spaceLeft=o.lineWidth-u,p=n,g=u,y=1)}return o.spaceLeft-=u})),y>0&&(this.emit("lastLine",i,this),_()),this.emit("sectionEnd",i,this),!0===i.continued?(w>1&&(this.continuedX=0),this.continuedX+=i.textWidth||0,this.document.y=x):this.document.x=this.startX}},{key:"nextSection",value:function nextSection(n){if(this.emit("sectionEnd",n,this),++this.column>this.columns){if(null!=this.height)return!1;var i;if(this.document.addPage(),this.column=1,this.startY=this.document.page.margins.top,this.maxY=this.document.page.maxY(),this.document.x=this.startX,this.document._fillColor)(i=this.document).fillColor.apply(i,_toConsumableArray(this.document._fillColor));this.emit("pageBreak",n,this)}else this.document.x+=this.lineWidth+this.columnGap,this.document.y=this.startY,this.emit("columnBreak",n,this);return this.emit("sectionStart",n,this),!0}}]),LineWrapper}(x.EventEmitter),Zt=N.number,Yt={initText:function initText(){return this._line=this._line.bind(this),this.x=0,this.y=0,this._lineGap=0},lineGap:function lineGap(n){return this._lineGap=n,this},moveDown:function moveDown(n){return null==n&&(n=1),this.y+=this.currentLineHeight(!0)*n+this._lineGap,this},moveUp:function moveUp(n){return null==n&&(n=1),this.y-=this.currentLineHeight(!0)*n+this._lineGap,this},_text:function _text(n,i,o,u,p){if(u=this._initOptions(i,o,u),n=null==n?"":"".concat(n),u.wordSpacing&&(n=n.replace(/\s{2,}/g," ")),u.width){var g=this._wrapper;g||(g=new Xt(this,u)).on("line",p),this._wrapper=u.continued?g:null,this._textOptions=u.continued?u:null,g.wrap(n,u)}else{var y=!0,w=!1,x=void 0;try{for(var _,k=n.split("\n")[Symbol.iterator]();!(y=(_=k.next()).done);y=!0){p(_.value,u)}}catch(n){w=!0,x=n}finally{try{y||null==k.return||k.return()}finally{if(w)throw x}}}return this},text:function text(n,i,o,u){return this._text(n,i,o,u,this._line)},widthOfString:function widthOfString(n){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};return this._font.widthOfString(n,this._fontSize,i.features)+(i.characterSpacing||0)*(n.length-1)},heightOfString:function heightOfString(n,i){var o=this,u=this.x,p=this.y;(i=this._initOptions(i)).height=1/0;var g=i.lineGap||this._lineGap||0;this._text(n,this.x,this.y,i,(function(){return o.y+=o.currentLineHeight(!0)+g}));var y=this.y-p;return this.x=u,this.y=p,y},list:function list(n,i,o,u,p){var g=this,y=(u=this._initOptions(i,o,u)).listType||"bullet",w=Math.round(this._font.ascender/1e3*this._fontSize),x=w/2,_=u.bulletRadius||w/3,k=u.textIndent||("bullet"===y?5*_:2*w),P=u.bulletIndent||("bullet"===y?8*_:2*w),E=1,O=[],B=[],I=[];!function flatten(n){for(var i=1,o=0;o<n.length;o++){var u=n[o];Array.isArray(u)?(E++,flatten(u),E--):(O.push(u),B.push(E),"bullet"!==y&&I.push(i++))}}(n);(p=new Xt(this,u)).on("line",this._line),E=1;var D=0;return p.on("firstLine",(function(){var n;if((n=B[D++])!==E){var i=P*(n-E);g.x+=i,p.lineWidth-=i,E=n}switch(y){case"bullet":return g.circle(g.x-k+_,g.y+x,_),g.fill();case"numbered":case"lettered":var o=function label(n){switch(y){case"numbered":return"".concat(n,".");case"lettered":var i=String.fromCharCode((n-1)%26+65),o=Math.floor((n-1)/26+1),u=Array(o+1).join(i);return"".concat(u,".")}}(I[D-1]);return g._fragment(o,g.x-k,g.y,u)}})),p.on("sectionStart",(function(){var n=k+P*(E-1);return g.x+=n,p.lineWidth-=n})),p.on("sectionEnd",(function(){var n=k+P*(E-1);return g.x-=n,p.lineWidth+=n})),p.wrap(O.join("\n"),u),this},_initOptions:function _initOptions(){var n=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{},i=arguments.length>1?arguments[1]:void 0,o=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{};"object"==typeof n&&(o=n,n=null);var u=Object.assign({},o);if(this._textOptions)for(var p in this._textOptions){var g=this._textOptions[p];"continued"!==p&&void 0===u[p]&&(u[p]=g)}return null!=n&&(this.x=n),null!=i&&(this.y=i),!1!==u.lineBreak&&(null==u.width&&(u.width=this.page.width-this.x-this.page.margins.right),u.width=Math.max(u.width,0)),u.columns||(u.columns=0),null==u.columnGap&&(u.columnGap=18),u},_line:function _line(n){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},o=arguments.length>2?arguments[2]:void 0;this._fragment(n,this.x,this.y,i);var u=i.lineGap||this._lineGap||0;return o?this.y+=this.currentLineHeight(!0)+u:this.x+=this.widthOfString(n)},_fragment:function _fragment(n,i,o,u){var p,g,y,w,x,_,k=this;if(0!==(n="".concat(n).replace(/\n/g,"")).length){var P=u.align||"left",E=u.wordSpacing||0,O=u.characterSpacing||0;if(u.width)switch(P){case"right":x=this.widthOfString(n.replace(/\s+$/,""),u),i+=u.lineWidth-x;break;case"center":i+=u.lineWidth/2-u.textWidth/2;break;case"justify":_=n.trim().split(/\s+/),x=this.widthOfString(n.replace(/\s+/g,""),u);var B=this.widthOfString(" ")+O;E=Math.max(0,(u.lineWidth-x)/Math.max(1,_.length-1)-B)}if("number"==typeof u.baseline)p=-u.baseline;else{switch(u.baseline){case"svg-middle":p=.5*this._font.xHeight;break;case"middle":case"svg-central":p=.5*(this._font.descender+this._font.ascender);break;case"bottom":case"ideographic":p=this._font.descender;break;case"alphabetic":p=0;break;case"mathematical":p=.5*this._font.ascender;break;case"hanging":p=.8*this._font.ascender;break;case"top":p=this._font.ascender;break;default:p=this._font.ascender}p=p/1e3*this._fontSize}var I,D=u.textWidth+E*(u.wordCount-1)+O*(n.length-1);if(null!=u.link&&this.link(i,o,D,this.currentLineHeight(),u.link),null!=u.goTo&&this.goTo(i,o,D,this.currentLineHeight(),u.goTo),null!=u.destination&&this.addNamedDestination(u.destination,"XYZ",i,o,null),u.underline||u.strike){this.save(),u.stroke||this.strokeColor.apply(this,_toConsumableArray(this._fillColor||[]));var R=this._fontSize<10?.5:Math.floor(this._fontSize/10);this.lineWidth(R);var N=u.underline?1:2,U=o+this.currentLineHeight()/N;u.underline&&(U-=R),this.moveTo(i,U),this.lineTo(i+D,U),this.stroke(),this.restore()}if(this.save(),u.oblique)I="number"==typeof u.oblique?-Math.tan(u.oblique*Math.PI/180):-.25,this.transform(1,0,0,1,i,o),this.transform(1,0,I,1,-I*p,0),this.transform(1,0,0,1,-i,-o);this.transform(1,0,0,-1,0,this.page.height),o=this.page.height-o-p,null==this.page.fonts[this._font.id]&&(this.page.fonts[this._font.id]=this._font.ref()),this.addContent("BT"),this.addContent("1 0 0 1 ".concat(Zt(i)," ").concat(Zt(o)," Tm")),this.addContent("/".concat(this._font.id," ").concat(Zt(this._fontSize)," Tf"));var W=u.fill&&u.stroke?2:u.stroke?1:0;if(W&&this.addContent("".concat(W," Tr")),O&&this.addContent("".concat(Zt(O)," Tc")),E){_=n.trim().split(/\s+/),E+=this.widthOfString(" ")+O,E*=1e3/this._fontSize,g=[],w=[];var G=!0,j=!1,X=void 0;try{for(var Y,K=_[Symbol.iterator]();!(G=(Y=K.next()).done);G=!0){var J=Y.value,$=_slicedToArray(this._font.encode(J,u.features),2),tt=$[0],et=$[1];g=g.concat(tt),w=w.concat(et);var rt={},nt=w[w.length-1];for(var it in nt){var ot=nt[it];rt[it]=ot}rt.xAdvance+=E,w[w.length-1]=rt}}catch(n){j=!0,X=n}finally{try{G||null==K.return||K.return()}finally{if(j)throw X}}}else{var at=_slicedToArray(this._font.encode(n,u.features),2);g=at[0],w=at[1]}var st=this._fontSize/1e3,lt=[],ct=0,ut=!1,ft=function addSegment(n){if(ct<n){var i=g.slice(ct,n).join(""),o=w[n-1].xAdvance-w[n-1].advanceWidth;lt.push("<".concat(i,"> ").concat(Zt(-o)))}return ct=n},ht=function flush(n){if(ft(n),lt.length>0)return k.addContent("[".concat(lt.join(" "),"] TJ")),lt.length=0};for(y=0;y<w.length;y++){var dt=w[y];dt.xOffset||dt.yOffset?(ht(y),this.addContent("1 0 0 1 ".concat(Zt(i+dt.xOffset*st)," ").concat(Zt(o+dt.yOffset*st)," Tm")),ht(y+1),ut=!0):(ut&&(this.addContent("1 0 0 1 ".concat(Zt(i)," ").concat(Zt(o)," Tm")),ut=!1),dt.xAdvance-dt.advanceWidth!=0&&ft(y+1)),i+=dt.xAdvance*st}return ht(y),this.addContent("ET"),this.restore()}}},Kt=[65472,65473,65474,65475,65477,65478,65479,65480,65481,65482,65483,65484,65485,65486,65487],Jt={1:"DeviceGray",3:"DeviceRGB",4:"DeviceCMYK"},Qt=function(){function JPEG(n,i){var o;if(_classCallCheck(this,JPEG),this.data=n,this.label=i,65496!==this.data.readUInt16BE(0))throw"SOI not found in JPEG";for(var u=2;u<this.data.length&&(o=this.data.readUInt16BE(u),u+=2,!Kt.includes(o));)u+=this.data.readUInt16BE(u);if(!Kt.includes(o))throw"Invalid JPEG.";u+=2,this.bits=this.data[u++],this.height=this.data.readUInt16BE(u),u+=2,this.width=this.data.readUInt16BE(u),u+=2;var p=this.data[u++];this.colorSpace=Jt[p],this.obj=null}return _createClass(JPEG,[{key:"embed",value:function embed(n){if(!this.obj)return this.obj=n.ref({Type:"XObject",Subtype:"Image",BitsPerComponent:this.bits,Width:this.width,Height:this.height,ColorSpace:this.colorSpace,Filter:"DCTDecode"}),"DeviceCMYK"===this.colorSpace&&(this.obj.data.Decode=[1,0,1,0,1,0,1,0]),this.obj.end(this.data),this.data=null}}]),JPEG}(),$t=function(){function PNGImage(n,i){_classCallCheck(this,PNGImage),this.label=i,this.image=new k.default(n),this.width=this.image.width,this.height=this.image.height,this.imgData=this.image.imgData,this.obj=null}return _createClass(PNGImage,[{key:"embed",value:function embed(i){var o=!1;if(this.document=i,!this.obj){var u=this.image.hasAlphaChannel,p=1===this.image.interlaceMethod;if(this.obj=this.document.ref({Type:"XObject",Subtype:"Image",BitsPerComponent:u?8:this.image.bits,Width:this.width,Height:this.height,Filter:"FlateDecode"}),!u){var g=this.document.ref({Predictor:p?1:15,Colors:this.image.colors,BitsPerComponent:this.image.bits,Columns:this.width});this.obj.data.DecodeParms=g,g.end()}if(0===this.image.palette.length)this.obj.data.ColorSpace=this.image.colorSpace;else{var y=this.document.ref();y.end(new n(this.image.palette)),this.obj.data.ColorSpace=["Indexed","DeviceRGB",this.image.palette.length/3-1,y]}if(null!=this.image.transparency.grayscale){var w=this.image.transparency.grayscale;this.obj.data.Mask=[w,w]}else if(this.image.transparency.rgb){var x=this.image.transparency.rgb,_=[],k=!0,P=!1,E=void 0;try{for(var O,B=x[Symbol.iterator]();!(k=(O=B.next()).done);k=!0){var I=O.value;_.push(I,I)}}catch(n){P=!0,E=n}finally{try{k||null==B.return||B.return()}finally{if(P)throw E}}this.obj.data.Mask=_}else{if(this.image.transparency.indexed)return o=!0,this.loadIndexedAlphaChannel();if(u)return o=!0,this.splitAlphaChannel()}if(p&&!o)return this.decodeData();this.finalize()}}},{key:"finalize",value:function finalize(){if(this.alphaChannel){var n=this.document.ref({Type:"XObject",Subtype:"Image",Height:this.height,Width:this.width,BitsPerComponent:8,Filter:"FlateDecode",ColorSpace:"DeviceGray",Decode:[0,1]});n.end(this.alphaChannel),this.obj.data.SMask=n}return this.obj.end(this.imgData),this.image=null,this.imgData=null}},{key:"splitAlphaChannel",value:function splitAlphaChannel(){var i=this;return this.image.decodePixels((function(o){for(var u,p,y=i.image.colors,w=i.width*i.height,x=new n(w*y),_=new n(w),k=p=u=0,P=o.length,E=16===i.image.bits?1:0;k<P;){for(var O=0;O<y;O++)x[p++]=o[k++],k+=E;_[u++]=o[k++],k+=E}return i.imgData=g.default.deflateSync(x),i.alphaChannel=g.default.deflateSync(_),i.finalize()}))}},{key:"loadIndexedAlphaChannel",value:function loadIndexedAlphaChannel(){var i=this,o=this.image.transparency.indexed;return this.image.decodePixels((function(u){for(var p=new n(i.width*i.height),y=0,w=0,x=u.length;w<x;w++)p[y++]=o[u[w]];return i.alphaChannel=g.default.deflateSync(p),i.finalize()}))}},{key:"decodeData",value:function decodeData(){var n=this;this.image.decodePixels((function(i){n.imgData=g.default.deflateSync(i),n.finalize()}))}}]),PNGImage}(),te=function(){function PDFImage(){_classCallCheck(this,PDFImage)}return _createClass(PDFImage,null,[{key:"open",value:function open(i,o){var u;if(n.isBuffer(i))u=i;else if(i instanceof ArrayBuffer)u=new n(new Uint8Array(i));else{var p;if(p=/^data:.+;base64,(.*)$/.exec(i))u=new n(p[1],"base64");else if(!(u=P.readFileSync(i)))return}if(255===u[0]&&216===u[1])return new Qt(u,o);if(137===u[0]&&"PNG"===u.toString("ascii",1,4))return new $t(u,o);throw new Error("Unknown image format.")}}]),PDFImage}(),ee={initImages:function initImages(){return this._imageRegistry={},this._imageCount=0},image:function image(n,i,o){var u,p,g,image,y,w,x,_=arguments.length>3&&void 0!==arguments[3]?arguments[3]:{};"object"==typeof i&&(_=i,i=null),i=null!=(w=null!=i?i:_.x)?w:this.x,o=null!=(x=null!=o?o:_.y)?x:this.y,"string"==typeof n&&(image=this._imageRegistry[n]),image||(image=n.width&&n.height?n:this.openImage(n)),image.obj||image.embed(this),null==this.page.xobjects[image.label]&&(this.page.xobjects[image.label]=image.obj);var k=_.width||image.width,P=_.height||image.height;if(_.width&&!_.height){var E=k/image.width;k=image.width*E,P=image.height*E}else if(_.height&&!_.width){var O=P/image.height;k=image.width*O,P=image.height*O}else if(_.scale)k=image.width*_.scale,P=image.height*_.scale;else if(_.fit){var B=_slicedToArray(_.fit,2);p=(g=B[0])/(u=B[1]),(y=image.width/image.height)>p?(k=g,P=g/y):(P=u,k=u*y)}else if(_.cover){var I=_slicedToArray(_.cover,2);p=(g=I[0])/(u=I[1]),(y=image.width/image.height)>p?(P=u,k=u*y):(k=g,P=g/y)}return(_.fit||_.cover)&&("center"===_.align?i=i+g/2-k/2:"right"===_.align&&(i=i+g-k),"center"===_.valign?o=o+u/2-P/2:"bottom"===_.valign&&(o=o+u-P)),null!=_.link&&this.link(i,o,k,P,_.link),null!=_.goTo&&this.goTo(i,o,k,P,_.goTo),null!=_.destination&&this.addNamedDestination(_.destination,"XYZ",i,o,null),this.y===o&&(this.y+=P),this.save(),this.transform(k,0,0,-P,i,o+P),this.addContent("/".concat(image.label," Do")),this.restore(),this},openImage:function openImage(n){var i;return"string"==typeof n&&(i=this._imageRegistry[n]),i||(i=te.open(n,"I".concat(++this._imageCount)),"string"==typeof n&&(this._imageRegistry[n]=i)),i}},re={annotate:function annotate(n,i,o,u,p){for(var g in p.Type="Annot",p.Rect=this._convertRect(n,i,o,u),p.Border=[0,0,0],"Link"===p.Subtype&&void 0===p.F&&(p.F=4),"Link"!==p.Subtype&&null==p.C&&(p.C=this._normalizeColor(p.color||[0,0,0])),delete p.color,"string"==typeof p.Dest&&(p.Dest=new String(p.Dest)),p){var y=p[g];p[g[0].toUpperCase()+g.slice(1)]=y}var w=this.ref(p);return this.page.annotations.push(w),w.end(),this},note:function note(n,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return g.Subtype="Text",g.Contents=new String(p),g.Name="Comment",null==g.color&&(g.color=[243,223,92]),this.annotate(n,i,o,u,g)},goTo:function goTo(n,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return g.Subtype="Link",g.A=this.ref({S:"GoTo",D:new String(p)}),g.A.end(),this.annotate(n,i,o,u,g)},link:function link(n,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};if(g.Subtype="Link","number"==typeof p){var y=this._root.data.Pages.data;if(!(p>=0&&p<y.Kids.length))throw new Error("The document has no page ".concat(p));g.A=this.ref({S:"GoTo",D:[y.Kids[p],"XYZ",null,null,null]}),g.A.end()}else g.A=this.ref({S:"URI",URI:new String(p)}),g.A.end();return this.annotate(n,i,o,u,g)},_markup:function _markup(n,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{},g=this._convertRect(n,i,o,u),y=_slicedToArray(g,4),w=y[0],x=y[1],_=y[2],k=y[3];return p.QuadPoints=[w,k,_,k,w,x,_,x],p.Contents=new String,this.annotate(n,i,o,u,p)},highlight:function highlight(n,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="Highlight",null==p.color&&(p.color=[241,238,148]),this._markup(n,i,o,u,p)},underline:function underline(n,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="Underline",this._markup(n,i,o,u,p)},strike:function strike(n,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="StrikeOut",this._markup(n,i,o,u,p)},lineAnnotation:function lineAnnotation(n,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="Line",p.Contents=new String,p.L=[n,this.page.height-i,o,this.page.height-u],this.annotate(n,i,o,u,p)},rectAnnotation:function rectAnnotation(n,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="Square",p.Contents=new String,this.annotate(n,i,o,u,p)},ellipseAnnotation:function ellipseAnnotation(n,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return p.Subtype="Circle",p.Contents=new String,this.annotate(n,i,o,u,p)},textAnnotation:function textAnnotation(n,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return g.Subtype="FreeText",g.Contents=new String(p),g.DA=new String,this.annotate(n,i,o,u,g)},_convertRect:function _convertRect(n,i,o,u){var p=i;i+=u;var g=n+o,y=_slicedToArray(this._ctm,6),w=y[0],x=y[1],_=y[2],k=y[3],P=y[4],E=y[5];return[n=w*n+_*i+P,i=x*n+k*i+E,g=w*g+_*p+P,p=x*g+k*p+E]}},ne=function(){function PDFOutline(n,i,o,u){var p=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{expanded:!1};_classCallCheck(this,PDFOutline),this.document=n,this.options=p,this.outlineData={},null!==u&&(this.outlineData.Dest=[u.dictionary,"Fit"]),null!==i&&(this.outlineData.Parent=i),null!==o&&(this.outlineData.Title=new String(o)),this.dictionary=this.document.ref(this.outlineData),this.children=[]}return _createClass(PDFOutline,[{key:"addItem",value:function addItem(n){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{expanded:!1},o=new PDFOutline(this.document,this.dictionary,n,this.document.page,i);return this.children.push(o),o}},{key:"endOutline",value:function endOutline(){if(this.children.length>0){this.options.expanded&&(this.outlineData.Count=this.children.length);var n=this.children[0],i=this.children[this.children.length-1];this.outlineData.First=n.dictionary,this.outlineData.Last=i.dictionary;for(var o=0,u=this.children.length;o<u;o++){var p=this.children[o];o>0&&(p.outlineData.Prev=this.children[o-1].dictionary),o<this.children.length-1&&(p.outlineData.Next=this.children[o+1].dictionary),p.endOutline()}}return this.dictionary.end()}}]),PDFOutline}(),ie={initOutline:function initOutline(){return this.outline=new ne(this,null,null,null)},endOutline:function endOutline(){if(this.outline.endOutline(),this.outline.children.length>0)return this._root.data.Outlines=this.outline.dictionary,this._root.data.PageMode="UseOutlines"}},oe={readOnly:1,required:2,noExport:4,multiline:4096,password:8192,toggleToOffButton:16384,radioButton:32768,pushButton:65536,combo:131072,edit:262144,sort:524288,multiSelect:2097152,noSpell:4194304},ae={left:0,center:1,right:2},se={value:"V",defaultValue:"DV"},le={zip:"0",zipPlus4:"1",zip4:"1",phone:"2",ssn:"3"},ce={nDec:0,sepComma:!1,negStyle:"MinusBlack",currency:"",currencyPrepend:!0},ue={nDec:0,sepComma:!1},fe={initForm:function initForm(){if(!this._font)throw new Error("Must set a font before calling initForm method");this._acroform={fonts:{},defaultFont:this._font.name},this._acroform.fonts[this._font.id]=this._font.ref();var n={Fields:[],NeedAppearances:!0,DA:new String("/".concat(this._font.id," 0 Tf 0 g")),DR:{Font:{}}};n.DR.Font[this._font.id]=this._font.ref();var i=this.ref(n);return this._root.data.AcroForm=i,this},endAcroForm:function endAcroForm(){var n=this;if(this._root.data.AcroForm){if(!Object.keys(this._acroform.fonts).length&&!this._acroform.defaultFont)throw new Error("No fonts specified for PDF form");var i=this._root.data.AcroForm.data.DR.Font;Object.keys(this._acroform.fonts).forEach((function(o){i[o]=n._acroform.fonts[o]})),this._root.data.AcroForm.data.Fields.forEach((function(i){n._endChild(i)})),this._root.data.AcroForm.end()}return this},_endChild:function _endChild(n){var i=this;return Array.isArray(n.data.Kids)&&(n.data.Kids.forEach((function(n){i._endChild(n)})),n.end()),this},formField:function formField(n){var i=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},o=this._fieldDict(n,null,i),u=this.ref(o);return this._addToParent(u),u},formAnnotation:function formAnnotation(n,i,o,u,p,g){var y=arguments.length>6&&void 0!==arguments[6]?arguments[6]:{},w=this._fieldDict(n,i,y);w.Subtype="Widget",void 0===w.F&&(w.F=4),this.annotate(o,u,p,g,w);var x=this.page.annotations[this.page.annotations.length-1];return this._addToParent(x)},formText:function formText(n,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(n,"text",i,o,u,p,g)},formPushButton:function formPushButton(n,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(n,"pushButton",i,o,u,p,g)},formCombo:function formCombo(n,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(n,"combo",i,o,u,p,g)},formList:function formList(n,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(n,"list",i,o,u,p,g)},formRadioButton:function formRadioButton(n,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(n,"radioButton",i,o,u,p,g)},formCheckbox:function formCheckbox(n,i,o,u,p){var g=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(n,"checkbox",i,o,u,p,g)},_addToParent:function _addToParent(n){var i=n.data.Parent;return i?(i.data.Kids||(i.data.Kids=[]),i.data.Kids.push(n)):this._root.data.AcroForm.data.Fields.push(n),this},_fieldDict:function _fieldDict(n,i){var o=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{};if(!this._acroform)throw new Error("Call document.initForms() method before adding form elements to document");var u=Object.assign({},o);return null!==i&&(u=this._resolveType(i,o)),u=this._resolveFlags(u),u=this._resolveJustify(u),u=this._resolveFont(u),u=this._resolveStrings(u),u=this._resolveColors(u),(u=this._resolveFormat(u)).T=new String(n),u.parent&&(u.Parent=u.parent,delete u.parent),u},_resolveType:function _resolveType(n,i){if("text"===n)i.FT="Tx";else if("pushButton"===n)i.FT="Btn",i.pushButton=!0;else if("radioButton"===n)i.FT="Btn",i.radioButton=!0;else if("checkbox"===n)i.FT="Btn";else if("combo"===n)i.FT="Ch",i.combo=!0;else{if("list"!==n)throw new Error("Invalid form annotation type '".concat(n,"'"));i.FT="Ch"}return i},_resolveFormat:function _resolveFormat(n){var i=n.format;if(i&&i.type){var o,u,p="";if(void 0!==le[i.type])o="AFSpecial_Keystroke",u="AFSpecial_Format",p=le[i.type];else{var g=i.type.charAt(0).toUpperCase()+i.type.slice(1);if(o="AF".concat(g,"_Keystroke"),u="AF".concat(g,"_Format"),"date"===i.type)o+="Ex",p=String(i.param);else if("time"===i.type)p=String(i.param);else if("number"===i.type){var y=Object.assign({},ce,i);p=String([String(y.nDec),y.sepComma?"0":"1",'"'+y.negStyle+'"',"null",'"'+y.currency+'"',String(y.currencyPrepend)].join(","))}else if("percent"===i.type){var w=Object.assign({},ue,i);p=String([String(w.nDec),w.sepComma?"0":"1"].join(","))}}n.AA=n.AA?n.AA:{},n.AA.K={S:"JavaScript",JS:new String("".concat(o,"(").concat(p,");"))},n.AA.F={S:"JavaScript",JS:new String("".concat(u,"(").concat(p,");"))}}return delete n.format,n},_resolveColors:function _resolveColors(n){var i=this._normalizeColor(n.backgroundColor);return i&&(n.MK||(n.MK={}),n.MK.BG=i),(i=this._normalizeColor(n.borderColor))&&(n.MK||(n.MK={}),n.MK.BC=i),delete n.backgroundColor,delete n.borderColor,n},_resolveFlags:function _resolveFlags(n){var i=0;return Object.keys(n).forEach((function(o){oe[o]&&(i|=oe[o],delete n[o])})),0!==i&&(n.Ff=n.Ff?n.Ff:0,n.Ff|=i),n},_resolveJustify:function _resolveJustify(n){var i=0;return void 0!==n.align&&("number"==typeof ae[n.align]&&(i=ae[n.align]),delete n.align),0!==i&&(n.Q=i),n},_resolveFont:function _resolveFont(n){return null===this._acroform.fonts[this._font.id]&&(this._acroform.fonts[this._font.id]=this._font.ref()),this._acroform.defaultFont!==this._font.name&&(n.DR={Font:{}},n.DR.Font[this._font.id]=this._font.ref(),n.DA=new String("/".concat(this._font.id," 0 Tf 0 g"))),n},_resolveStrings:function _resolveStrings(n){var i=[];function appendChoices(n){if(Array.isArray(n))for(var o=0;o<n.length;o++)"string"==typeof n[o]?i.push(new String(n[o])):i.push(n[o])}return appendChoices(n.Opt),n.select&&(appendChoices(n.select),delete n.select),i.length&&(n.Opt=i),n.value||n.defaultValue,Object.keys(se).forEach((function(i){void 0!==n[i]&&(n[se[i]]=n[i],delete n[i])})),["V","DV"].forEach((function(i){"string"==typeof n[i]&&(n[i]=new String(n[i]))})),n.MK&&n.MK.CA&&(n.MK.CA=new String(n.MK.CA)),n.label&&(n.MK=n.MK?n.MK:{},n.MK.CA=new String(n.label),delete n.label),n}},he=function(i){function PDFDocument(){var n,i=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{};switch(_classCallCheck(this,PDFDocument),(n=_possibleConstructorReturn(this,_getPrototypeOf(PDFDocument).call(this,i))).options=i,i.pdfVersion){case"1.4":n.version=1.4;break;case"1.5":n.version=1.5;break;case"1.6":n.version=1.6;break;case"1.7":case"1.7ext3":n.version=1.7;break;default:n.version=1.3}n.compress=null==n.options.compress||n.options.compress,n._pageBuffer=[],n._pageBufferStart=0,n._offsets=[],n._waiting=0,n._ended=!1,n._offset=0;var o=n.ref({Type:"Pages",Count:0,Kids:[]}),u=n.ref({Dests:new O});if(n._root=n.ref({Type:"Catalog",Pages:o,Names:u}),n.page=null,n.initColor(),n.initVector(),n.initFonts(i.font),n.initText(),n.initImages(),n.initOutline(),n.info={Producer:"PDFKit",Creator:"PDFKit",CreationDate:new Date},n.options.info)for(var p in n.options.info){var g=n.options.info[p];n.info[p]=g}return n._id=ht.generateFileID(n.info),n._security=ht.create(_assertThisInitialized(n),i),n._write("%PDF-".concat(n.version)),n._write("%ÿÿÿÿ"),!1!==n.options.autoFirstPage&&n.addPage(),n}return _inherits(PDFDocument,i),_createClass(PDFDocument,[{key:"addPage",value:function addPage(n){null==n&&(n=this.options),this.options.bufferPages||this.flushPages(),this.page=new j(this,n),this._pageBuffer.push(this.page);var i=this._root.data.Pages.data;return i.Kids.push(this.page.dictionary),i.Count++,this.x=this.page.margins.left,this.y=this.page.margins.top,this._ctm=[1,0,0,1,0,0],this.transform(1,0,0,-1,0,this.page.height),this.emit("pageAdded"),this}},{key:"bufferedPageRange",value:function bufferedPageRange(){return{start:this._pageBufferStart,count:this._pageBuffer.length}}},{key:"switchToPage",value:function switchToPage(n){var i;if(!(i=this._pageBuffer[n-this._pageBufferStart]))throw new Error("switchToPage(".concat(n,") out of bounds, current buffer covers pages ").concat(this._pageBufferStart," to ").concat(this._pageBufferStart+this._pageBuffer.length-1));return this.page=i}},{key:"flushPages",value:function flushPages(){var n=this._pageBuffer;this._pageBuffer=[],this._pageBufferStart+=n.length;var i=!0,o=!1,u=void 0;try{for(var p,g=n[Symbol.iterator]();!(i=(p=g.next()).done);i=!0){p.value.end()}}catch(n){o=!0,u=n}finally{try{i||null==g.return||g.return()}finally{if(o)throw u}}}},{key:"addNamedDestination",value:function addNamedDestination(n){for(var i=arguments.length,o=new Array(i>1?i-1:0),u=1;u<i;u++)o[u-1]=arguments[u];0===o.length&&(o=["XYZ",null,null,null]),"XYZ"===o[0]&&null!==o[2]&&(o[2]=this.page.height-o[2]),o.unshift(this.page.dictionary),this._root.data.Names.data.Dests.add(n,o)}},{key:"addNamedJavaScript",value:function addNamedJavaScript(n,i){this._root.data.Names.data.JavaScript||(this._root.data.Names.data.JavaScript=new O);var o={JS:new String(i),S:"JavaScript"};this._root.data.Names.data.JavaScript.add(n,o)}},{key:"ref",value:function ref(n){var ref=new U(this,this._offsets.length+1,n);return this._offsets.push(null),this._waiting++,ref}},{key:"_read",value:function _read(){}},{key:"_write",value:function _write(i){return n.isBuffer(i)||(i=new n(i+"\n","binary")),this.push(i),this._offset+=i.length}},{key:"addContent",value:function addContent(n){return this.page.write(n),this}},{key:"_refEnd",value:function _refEnd(n){if(this._offsets[n.id-1]=n.offset,0==--this._waiting&&this._ended)return this._finalize(),this._ended=!1}},{key:"write",value:function write(n,i){new Error("PDFDocument#write is deprecated, and will be removed in a future version of PDFKit. Please pipe the document into a Node stream.");return this.pipe(P.createWriteStream(n)),this.end(),this.once("end",i)}},{key:"end",value:function end(){for(var n in this.flushPages(),this._info=this.ref(),this.info){var i=this.info[n];"string"==typeof i&&(i=new String(i));var o=this.ref(i);o.end(),this._info.data[n]=o}for(var u in this._info.end(),this._fontFamilies){this._fontFamilies[u].finalize()}return this.endOutline(),this._root.end(),this._root.data.Pages.end(),this._root.data.Names.end(),this.endAcroForm(),this._security&&this._security.end(),0===this._waiting?this._finalize():this._ended=!0}},{key:"_finalize",value:function _finalize(n){var i=this._offset;this._write("xref"),this._write("0 ".concat(this._offsets.length+1)),this._write("0000000000 65535 f ");var o=!0,u=!1,p=void 0;try{for(var g,y=this._offsets[Symbol.iterator]();!(o=(g=y.next()).done);o=!0){var w=g.value;w="0000000000".concat(w).slice(-10),this._write(w+" 00000 n ")}}catch(n){u=!0,p=n}finally{try{o||null==y.return||y.return()}finally{if(u)throw p}}var x={Size:this._offsets.length+1,Root:this._root,Info:this._info,ID:[this._id,this._id]};return this._security&&(x.Encrypt=this._security.dictionary),this._write("trailer"),this._write(N.convert(x)),this._write("startxref"),this._write("".concat(i)),this._write("%%EOF"),this.push(null)}},{key:"toString",value:function toString(){return"[object PDFDocument]"}}]),PDFDocument}(p.default.Readable),de=function mixin(n){Object.assign(he.prototype,n)};de(At),de(Mt),de(qt),de(Yt),de(ee),de(re),de(ie),de(fe);var pe=he;i.default=pe}).call(this,o(10).Buffer,"/")},function(n,i,o){var u=o(3),p=o(140),g=u.WeakMap;n.exports="function"==typeof g&&/native code/.test(p(g))},function(n,i,o){var u=o(34),p=o(57),g=o(144),y=o(11);n.exports=u("Reflect","ownKeys")||function ownKeys(n){var i=p.f(y(n)),o=g.f;return o?i.concat(o(n)):i}},function(n,i,o){var u=o(13),p=o(16),g=o(11),y=o(94);n.exports=u?Object.defineProperties:function defineProperties(n,i){g(n);for(var o,u=y(i),w=u.length,x=0;w>x;)p.f(n,o=u[x++],i[o]);return n}},function(n,i,o){var u=o(27),p=o(57).f,g={}.toString,y="object"==typeof window&&window&&Object.getOwnPropertyNames?Object.getOwnPropertyNames(window):[];n.exports.f=function getOwnPropertyNames(n){return y&&"[object Window]"==g.call(n)?function(n){try{return p(n)}catch(n){return y.slice()}}(n):p(u(n))}},function(n,i,o){"use strict";var u=o(1),p=o(20).every,g=o(59),y=o(30),w=g("every"),x=y("every");u({target:"Array",proto:!0,forced:!w||!x},{every:function every(n){return p(this,n,arguments.length>1?arguments[1]:void 0)}})},function(n,i,o){"use strict";var u=o(1),p=o(20).filter,g=o(116),y=o(30),w=g("filter"),x=y("filter");u({target:"Array",proto:!0,forced:!w||!x},{filter:function filter(n){return p(this,n,arguments.length>1?arguments[1]:void 0)}})},function(n,i,o){"use strict";var u=o(1),p=o(214);u({target:"Array",proto:!0,forced:[].forEach!=p},{forEach:p})},function(n,i,o){var u=o(1),p=o(318);u({target:"Array",stat:!0,forced:!o(155)((function(n){Array.from(n)}))},{from:p})},function(n,i,o){"use strict";var u=o(96),p=o(19),g=o(319),y=o(152),w=o(8),x=o(147),_=o(153);n.exports=function from(n){var i,o,k,P,E,O,B=p(n),I="function"==typeof this?this:Array,D=arguments.length,R=D>1?arguments[1]:void 0,N=void 0!==R,U=_(B),W=0;if(N&&(R=u(R,D>2?arguments[2]:void 0,2)),null==U||I==Array&&y(U))for(o=new I(i=w(B.length));i>W;W++)O=N?R(B[W],W):B[W],x(o,W,O);else for(E=(P=U.call(B)).next,o=new I;!(k=E.call(P)).done;W++)O=N?g(P,R,[k.value,W],!0):k.value,x(o,W,O);return o.length=W,o}},function(n,i,o){var u=o(11),p=o(215);n.exports=function(n,i,o,g){try{return g?i(u(o)[0],o[1]):i(o)}catch(i){throw p(n),i}}},function(n,i,o){"use strict";var u=o(218).IteratorPrototype,p=o(58),g=o(40),y=o(95),w=o(97),returnThis=function(){return this};n.exports=function(n,i,o){var x=i+" Iterator";return n.prototype=p(u,{next:g(1,o)}),y(n,x,!1,!0),w[x]=returnThis,n}},function(n,i,o){var u=o(14);n.exports=function(n){if(!u(n)&&null!==n)throw TypeError("Can't set "+String(n)+" as a prototype");return n}},function(n,i,o){"use strict";var u=o(1),p=o(20).map,g=o(116),y=o(30),w=g("map"),x=y("map");u({target:"Array",proto:!0,forced:!w||!x},{map:function map(n){return p(this,n,arguments.length>1?arguments[1]:void 0)}})},function(n,i,o){"use strict";var u=o(1),p=o(20).some,g=o(59),y=o(30),w=g("some"),x=y("some");u({target:"Array",proto:!0,forced:!w||!x},{some:function some(n){return p(this,n,arguments.length>1?arguments[1]:void 0)}})},function(n,i,o){"use strict";var u=o(1),p=o(3),g=o(156),y=o(158),w=g.ArrayBuffer;u({global:!0,forced:p.ArrayBuffer!==w},{ArrayBuffer:w}),y("ArrayBuffer")},function(n,i){var o=Math.abs,u=Math.pow,p=Math.floor,g=Math.log,y=Math.LN2;n.exports={pack:function(n,i,w){var x,_,k,P=new Array(w),E=8*w-i-1,O=(1<<E)-1,B=O>>1,I=23===i?u(2,-24)-u(2,-77):0,D=n<0||0===n&&1/n<0?1:0,R=0;for((n=o(n))!=n||n===1/0?(_=n!=n?1:0,x=O):(x=p(g(n)/y),n*(k=u(2,-x))<1&&(x--,k*=2),(n+=x+B>=1?I/k:I*u(2,1-B))*k>=2&&(x++,k/=2),x+B>=O?(_=0,x=O):x+B>=1?(_=(n*k-1)*u(2,i),x+=B):(_=n*u(2,B-1)*u(2,i),x=0));i>=8;P[R++]=255&_,_/=256,i-=8);for(x=x<<i|_,E+=i;E>0;P[R++]=255&x,x/=256,E-=8);return P[--R]|=128*D,P},unpack:function(n,i){var o,p=n.length,g=8*p-i-1,y=(1<<g)-1,w=y>>1,x=g-7,_=p-1,k=n[_--],P=127&k;for(k>>=7;x>0;P=256*P+n[_],_--,x-=8);for(o=P&(1<<-x)-1,P>>=-x,x+=i;x>0;o=256*o+n[_],_--,x-=8);if(0===P)P=1-w;else{if(P===y)return o?NaN:k?-1/0:1/0;o+=u(2,i),P-=w}return(k?-1:1)*o*u(2,P-i)}}},function(n,i,o){o(1)({target:"Number",stat:!0},{isFinite:o(327)})},function(n,i,o){var u=o(3).isFinite;n.exports=Number.isFinite||function isFinite(n){return"number"==typeof n&&u(n)}},function(n,i,o){var u=o(1),p=o(329);u({target:"Object",stat:!0,forced:Object.assign!==p},{assign:p})},function(n,i,o){"use strict";var u=o(13),p=o(4),g=o(94),y=o(144),w=o(110),x=o(19),_=o(92),k=Object.assign,P=Object.defineProperty;n.exports=!k||p((function(){if(u&&1!==k({b:1},k(P({},"a",{enumerable:!0,get:function(){P(this,"b",{value:3,enumerable:!1})}}),{b:2})).b)return!0;var n={},i={},o=Symbol();return n[o]=7,"abcdefghijklmnopqrst".split("").forEach((function(n){i[n]=n})),7!=k({},n)[o]||"abcdefghijklmnopqrst"!=g(k({},i)).join("")}))?function assign(n,i){for(var o=x(n),p=arguments.length,k=1,P=y.f,E=w.f;p>k;)for(var O,B=_(arguments[k++]),I=P?g(B).concat(P(B)):g(B),D=I.length,R=0;D>R;)O=I[R++],u&&!E.call(B,O)||(o[O]=B[O]);return o}:k},function(n,i,o){var u=o(1),p=o(4),g=o(19),y=o(60),w=o(219);u({target:"Object",stat:!0,forced:p((function(){y(1)})),sham:!w},{getPrototypeOf:function getPrototypeOf(n){return y(g(n))}})},function(n,i,o){o(1)({target:"Object",stat:!0},{setPrototypeOf:o(45)})},function(n,i,o){"use strict";var u=o(154),p=o(117);n.exports=u?{}.toString:function toString(){return"[object "+p(this)+"]"}},function(n,i,o){"use strict";var u=o(4);function RE(n,i){return RegExp(n,i)}i.UNSUPPORTED_Y=u((function(){var n=RE("a","y");return n.lastIndex=2,null!=n.exec("abcd")})),i.BROKEN_CARET=u((function(){var n=RE("^r","gy");return n.lastIndex=2,null!=n.exec("str")}))},function(n,i,o){"use strict";var u=o(25),p=o(11),g=o(4),y=o(230),w=RegExp.prototype,x=w.toString,_=g((function(){return"/a/b"!=x.call({source:"a",flags:"b"})})),k="toString"!=x.name;(_||k)&&u(RegExp.prototype,"toString",(function toString(){var n=p(this),i=String(n.source),o=n.flags;return"/"+i+"/"+String(void 0===o&&n instanceof RegExp&&!("flags"in w)?y.call(n):o)}),{unsafe:!0})},function(n,i,o){"use strict";var u=o(1),p=o(159).codeAt;u({target:"String",proto:!0},{codePointAt:function codePointAt(n){return p(this,n)}})},function(n,i,o){var u=o(1),p=o(43),g=String.fromCharCode,y=String.fromCodePoint;u({target:"String",stat:!0,forced:!!y&&1!=y.length},{fromCodePoint:function fromCodePoint(n){for(var i,o=[],u=arguments.length,y=0;u>y;){if(i=+arguments[y++],p(i,1114111)!==i)throw RangeError(i+" is not a valid code point");o.push(i<65536?g(i):g(55296+((i-=65536)>>10),i%1024+56320))}return o.join("")}})},function(n,i,o){var u=o(19),p=Math.floor,g="".replace,y=/\$([$&'`]|\d\d?|<[^>]*>)/g,w=/\$([$&'`]|\d\d?)/g;n.exports=function(n,i,o,x,_,k){var P=o+n.length,E=x.length,O=w;return void 0!==_&&(_=u(_),O=y),g.call(k,O,(function(u,g){var y;switch(g.charAt(0)){case"$":return"$";case"&":return n;case"`":return i.slice(0,o);case"'":return i.slice(P);case"<":y=_[g.slice(1,-1)];break;default:var w=+g;if(0===w)return u;if(w>E){var k=p(w/10);return 0===k?u:k<=E?void 0===x[k-1]?g.charAt(1):x[k-1]+g.charAt(1):u}y=x[w-1]}return void 0===y?"":y}))}},function(n,i,o){var u=o(14),p=o(41),g=o(6)("match");n.exports=function(n){var i;return u(n)&&(void 0!==(i=n[g])?!!i:"RegExp"==p(n))}},function(n,i,o){var u=o(4),p=o(228);n.exports=function(n){return u((function(){return!!p[n]()||"​᠎"!="​᠎"[n]()||p[n].name!==n}))}},function(n,i,o){"use strict";var u=o(1),p=o(235);u({target:"String",proto:!0,forced:o(236)("strike")},{strike:function strike(){return p(this,"strike","","")}})},function(n,i,o){var u=o(3),p=o(4),g=o(155),y=o(5).NATIVE_ARRAY_BUFFER_VIEWS,w=u.ArrayBuffer,x=u.Int8Array;n.exports=!y||!p((function(){x(1)}))||!p((function(){new x(-1)}))||!g((function(n){new x,new x(null),new x(1.5),new x(n)}),!0)||p((function(){return 1!==new x(new w(2),1,void 0).length}))},function(n,i,o){var u=o(35);n.exports=function(n){var i=u(n);if(i<0)throw RangeError("The argument can't be less than 0");return i}},function(n,i,o){var u=o(19),p=o(8),g=o(153),y=o(152),w=o(96),x=o(5).aTypedArrayConstructor;n.exports=function from(n){var i,o,_,k,P,E,O=u(n),B=arguments.length,I=B>1?arguments[1]:void 0,D=void 0!==I,R=g(O);if(null!=R&&!y(R))for(E=(P=R.call(O)).next,O=[];!(k=E.call(P)).done;)O.push(k.value);for(D&&B>2&&(I=w(I,arguments[2],2)),o=p(O.length),_=new(x(this))(o),i=0;o>i;i++)_[i]=D?I(O[i],i):O[i];return _}},function(n,i,o){"use strict";var u=o(19),p=o(43),g=o(8),y=Math.min;n.exports=[].copyWithin||function copyWithin(n,i){var o=u(this),w=g(o.length),x=p(n,w),_=p(i,w),k=arguments.length>2?arguments[2]:void 0,P=y((void 0===k?w:p(k,w))-_,w-x),E=1;for(_<x&&x<_+P&&(E=-1,_+=P-1,x+=P-1);P-- >0;)_ in o?o[x]=o[_]:delete o[x],x+=E,_+=E;return o}},function(n,i,o){"use strict";var u=o(27),p=o(35),g=o(8),y=o(59),w=o(30),x=Math.min,_=[].lastIndexOf,k=!!_&&1/[1].lastIndexOf(1,-0)<0,P=y("lastIndexOf"),E=w("indexOf",{ACCESSORS:!0,1:0}),O=k||!P||!E;n.exports=O?function lastIndexOf(n){if(k)return _.apply(this,arguments)||0;var i=u(this),o=g(i.length),y=o-1;for(arguments.length>1&&(y=x(y,p(arguments[1]))),y<0&&(y=o+y);y>=0;y--)if(y in i&&i[y]===n)return y||0;return-1}:_},function(n,i,o){var u=o(3),p=o(240),g=o(214),y=o(18);for(var w in p){var x=u[w],_=x&&x.prototype;if(_&&_.forEach!==g)try{y(_,"forEach",g)}catch(n){_.forEach=g}}},function(n,i){},function(n,i,o){"use strict";var u=o(124).Buffer,p=o(349);n.exports=function(){function BufferList(){!function _classCallCheck(n,i){if(!(n instanceof i))throw new TypeError("Cannot call a class as a function")}(this,BufferList),this.head=null,this.tail=null,this.length=0}return BufferList.prototype.push=function push(n){var i={data:n,next:null};this.length>0?this.tail.next=i:this.head=i,this.tail=i,++this.length},BufferList.prototype.unshift=function unshift(n){var i={data:n,next:this.head};0===this.length&&(this.tail=i),this.head=i,++this.length},BufferList.prototype.shift=function shift(){if(0!==this.length){var n=this.head.data;return 1===this.length?this.head=this.tail=null:this.head=this.head.next,--this.length,n}},BufferList.prototype.clear=function clear(){this.head=this.tail=null,this.length=0},BufferList.prototype.join=function join(n){if(0===this.length)return"";for(var i=this.head,o=""+i.data;i=i.next;)o+=n+i.data;return o},BufferList.prototype.concat=function concat(n){if(0===this.length)return u.alloc(0);if(1===this.length)return this.head.data;for(var i,o,p,g=u.allocUnsafe(n>>>0),y=this.head,w=0;y;)i=y.data,o=g,p=w,i.copy(o,p),w+=y.data.length,y=y.next;return g},BufferList}(),p&&p.inspect&&p.inspect.custom&&(n.exports.prototype[p.inspect.custom]=function(){var n=p.inspect({length:this.length});return this.constructor.name+" "+n})},function(n,i){},function(n,i,o){(function(i){function config(n){try{if(!i.localStorage)return!1}catch(n){return!1}var o=i.localStorage[n];return null!=o&&"true"===String(o).toLowerCase()}n.exports=function deprecate(n,i){if(config("noDeprecation"))return n;var o=!1;return function deprecated(){if(!o){if(config("throwDeprecation"))throw new Error(i);config("traceDeprecation"),o=!0}return n.apply(this,arguments)}}}).call(this,o(24))},function(n,i,o){"use strict";n.exports=PassThrough;var u=o(244),p=Object.create(o(98));function PassThrough(n){if(!(this instanceof PassThrough))return new PassThrough(n);u.call(this,n)}p.inherits=o(86),p.inherits(PassThrough,u),PassThrough.prototype._transform=function(n,i,o){o(null,n)}},function(n,i,o){n.exports=o(166)},function(n,i,o){n.exports=o(48)},function(n,i,o){n.exports=o(165).Transform},function(n,i,o){n.exports=o(165).PassThrough},function(n,i,o){"use strict";(function(n,u){var p=o(246),g=o(360),y=o(361),w=o(364),x=o(367);for(var _ in x)i[_]=x[_];i.NONE=0,i.DEFLATE=1,i.INFLATE=2,i.GZIP=3,i.GUNZIP=4,i.DEFLATERAW=5,i.INFLATERAW=6,i.UNZIP=7;function Zlib(n){if("number"!=typeof n||n<i.DEFLATE||n>i.UNZIP)throw new TypeError("Bad argument");this.dictionary=null,this.err=0,this.flush=0,this.init_done=!1,this.level=0,this.memLevel=0,this.mode=n,this.strategy=0,this.windowBits=0,this.write_in_progress=!1,this.pending_close=!1,this.gzip_id_bytes_read=0}Zlib.prototype.close=function(){this.write_in_progress?this.pending_close=!0:(this.pending_close=!1,p(this.init_done,"close before init"),p(this.mode<=i.UNZIP),this.mode===i.DEFLATE||this.mode===i.GZIP||this.mode===i.DEFLATERAW?y.deflateEnd(this.strm):this.mode!==i.INFLATE&&this.mode!==i.GUNZIP&&this.mode!==i.INFLATERAW&&this.mode!==i.UNZIP||w.inflateEnd(this.strm),this.mode=i.NONE,this.dictionary=null)},Zlib.prototype.write=function(n,i,o,u,p,g,y){return this._write(!0,n,i,o,u,p,g,y)},Zlib.prototype.writeSync=function(n,i,o,u,p,g,y){return this._write(!1,n,i,o,u,p,g,y)},Zlib.prototype._write=function(o,g,y,w,x,_,k,P){if(p.equal(arguments.length,8),p(this.init_done,"write before init"),p(this.mode!==i.NONE,"already finalized"),p.equal(!1,this.write_in_progress,"write already in progress"),p.equal(!1,this.pending_close,"close is pending"),this.write_in_progress=!0,p.equal(!1,void 0===g,"must provide flush value"),this.write_in_progress=!0,g!==i.Z_NO_FLUSH&&g!==i.Z_PARTIAL_FLUSH&&g!==i.Z_SYNC_FLUSH&&g!==i.Z_FULL_FLUSH&&g!==i.Z_FINISH&&g!==i.Z_BLOCK)throw new Error("Invalid flush value");if(null==y&&(y=n.alloc(0),x=0,w=0),this.strm.avail_in=x,this.strm.input=y,this.strm.next_in=w,this.strm.avail_out=P,this.strm.output=_,this.strm.next_out=k,this.flush=g,!o)return this._process(),this._checkError()?this._afterSync():void 0;var E=this;return u.nextTick((function(){E._process(),E._after()})),this},Zlib.prototype._afterSync=function(){var n=this.strm.avail_out,i=this.strm.avail_in;return this.write_in_progress=!1,[i,n]},Zlib.prototype._process=function(){var n=null;switch(this.mode){case i.DEFLATE:case i.GZIP:case i.DEFLATERAW:this.err=y.deflate(this.strm,this.flush);break;case i.UNZIP:switch(this.strm.avail_in>0&&(n=this.strm.next_in),this.gzip_id_bytes_read){case 0:if(null===n)break;if(31!==this.strm.input[n]){this.mode=i.INFLATE;break}if(this.gzip_id_bytes_read=1,n++,1===this.strm.avail_in)break;case 1:if(null===n)break;139===this.strm.input[n]?(this.gzip_id_bytes_read=2,this.mode=i.GUNZIP):this.mode=i.INFLATE;break;default:throw new Error("invalid number of gzip magic number bytes read")}case i.INFLATE:case i.GUNZIP:case i.INFLATERAW:for(this.err=w.inflate(this.strm,this.flush),this.err===i.Z_NEED_DICT&&this.dictionary&&(this.err=w.inflateSetDictionary(this.strm,this.dictionary),this.err===i.Z_OK?this.err=w.inflate(this.strm,this.flush):this.err===i.Z_DATA_ERROR&&(this.err=i.Z_NEED_DICT));this.strm.avail_in>0&&this.mode===i.GUNZIP&&this.err===i.Z_STREAM_END&&0!==this.strm.next_in[0];)this.reset(),this.err=w.inflate(this.strm,this.flush);break;default:throw new Error("Unknown mode "+this.mode)}},Zlib.prototype._checkError=function(){switch(this.err){case i.Z_OK:case i.Z_BUF_ERROR:if(0!==this.strm.avail_out&&this.flush===i.Z_FINISH)return this._error("unexpected end of file"),!1;break;case i.Z_STREAM_END:break;case i.Z_NEED_DICT:return null==this.dictionary?this._error("Missing dictionary"):this._error("Bad dictionary"),!1;default:return this._error("Zlib error"),!1}return!0},Zlib.prototype._after=function(){if(this._checkError()){var n=this.strm.avail_out,i=this.strm.avail_in;this.write_in_progress=!1,this.callback(i,n),this.pending_close&&this.close()}},Zlib.prototype._error=function(n){this.strm.msg&&(n=this.strm.msg),this.onerror(n,this.err),this.write_in_progress=!1,this.pending_close&&this.close()},Zlib.prototype.init=function(n,o,u,g,y){p(4===arguments.length||5===arguments.length,"init(windowBits, level, memLevel, strategy, [dictionary])"),p(n>=8&&n<=15,"invalid windowBits"),p(o>=-1&&o<=9,"invalid compression level"),p(u>=1&&u<=9,"invalid memlevel"),p(g===i.Z_FILTERED||g===i.Z_HUFFMAN_ONLY||g===i.Z_RLE||g===i.Z_FIXED||g===i.Z_DEFAULT_STRATEGY,"invalid strategy"),this._init(o,n,u,g,y),this._setDictionary()},Zlib.prototype.params=function(){throw new Error("deflateParams Not supported")},Zlib.prototype.reset=function(){this._reset(),this._setDictionary()},Zlib.prototype._init=function(n,o,u,p,x){switch(this.level=n,this.windowBits=o,this.memLevel=u,this.strategy=p,this.flush=i.Z_NO_FLUSH,this.err=i.Z_OK,this.mode!==i.GZIP&&this.mode!==i.GUNZIP||(this.windowBits+=16),this.mode===i.UNZIP&&(this.windowBits+=32),this.mode!==i.DEFLATERAW&&this.mode!==i.INFLATERAW||(this.windowBits=-1*this.windowBits),this.strm=new g,this.mode){case i.DEFLATE:case i.GZIP:case i.DEFLATERAW:this.err=y.deflateInit2(this.strm,this.level,i.Z_DEFLATED,this.windowBits,this.memLevel,this.strategy);break;case i.INFLATE:case i.GUNZIP:case i.INFLATERAW:case i.UNZIP:this.err=w.inflateInit2(this.strm,this.windowBits);break;default:throw new Error("Unknown mode "+this.mode)}this.err!==i.Z_OK&&this._error("Init error"),this.dictionary=x,this.write_in_progress=!1,this.init_done=!0},Zlib.prototype._setDictionary=function(){if(null!=this.dictionary){switch(this.err=i.Z_OK,this.mode){case i.DEFLATE:case i.DEFLATERAW:this.err=y.deflateSetDictionary(this.strm,this.dictionary)}this.err!==i.Z_OK&&this._error("Failed to set dictionary")}},Zlib.prototype._reset=function(){switch(this.err=i.Z_OK,this.mode){case i.DEFLATE:case i.DEFLATERAW:case i.GZIP:this.err=y.deflateReset(this.strm);break;case i.INFLATE:case i.INFLATERAW:case i.GUNZIP:this.err=w.inflateReset(this.strm)}this.err!==i.Z_OK&&this._error("Failed to reset stream")},i.Zlib=Zlib}).call(this,o(10).Buffer,o(47))},function(n,i,o){"use strict";var u=Object.getOwnPropertySymbols,p=Object.prototype.hasOwnProperty,g=Object.prototype.propertyIsEnumerable;function toObject(n){if(null==n)throw new TypeError("Object.assign cannot be called with null or undefined");return Object(n)}n.exports=function shouldUseNative(){try{if(!Object.assign)return!1;var n=new String("abc");if(n[5]="de","5"===Object.getOwnPropertyNames(n)[0])return!1;for(var i={},o=0;o<10;o++)i["_"+String.fromCharCode(o)]=o;if("0123456789"!==Object.getOwnPropertyNames(i).map((function(n){return i[n]})).join(""))return!1;var u={};return"abcdefghijklmnopqrst".split("").forEach((function(n){u[n]=n})),"abcdefghijklmnopqrst"===Object.keys(Object.assign({},u)).join("")}catch(n){return!1}}()?Object.assign:function(n,i){for(var o,y,w=toObject(n),x=1;x<arguments.length;x++){for(var _ in o=Object(arguments[x]))p.call(o,_)&&(w[_]=o[_]);if(u){y=u(o);for(var k=0;k<y.length;k++)g.call(o,y[k])&&(w[y[k]]=o[y[k]])}}return w}},function(n,i){n.exports=function isBuffer(n){return n&&"object"==typeof n&&"function"==typeof n.copy&&"function"==typeof n.fill&&"function"==typeof n.readUInt8}},function(n,i){"function"==typeof Object.create?n.exports=function inherits(n,i){n.super_=i,n.prototype=Object.create(i.prototype,{constructor:{value:n,enumerable:!1,writable:!0,configurable:!0}})}:n.exports=function inherits(n,i){n.super_=i;var TempCtor=function(){};TempCtor.prototype=i.prototype,n.prototype=new TempCtor,n.prototype.constructor=n}},function(n,i,o){"use strict";n.exports=function ZStream(){this.input=null,this.next_in=0,this.avail_in=0,this.total_in=0,this.output=null,this.next_out=0,this.avail_out=0,this.total_out=0,this.msg="",this.state=null,this.data_type=2,this.adler=0}},function(n,i,o){"use strict";var u,p=o(126),g=o(362),y=o(247),w=o(248),x=o(363);function err(n,i){return n.msg=x[i],i}function rank(n){return(n<<1)-(n>4?9:0)}function zero(n){for(var i=n.length;--i>=0;)n[i]=0}function flush_pending(n){var i=n.state,o=i.pending;o>n.avail_out&&(o=n.avail_out),0!==o&&(p.arraySet(n.output,i.pending_buf,i.pending_out,o,n.next_out),n.next_out+=o,i.pending_out+=o,n.total_out+=o,n.avail_out-=o,i.pending-=o,0===i.pending&&(i.pending_out=0))}function flush_block_only(n,i){g._tr_flush_block(n,n.block_start>=0?n.block_start:-1,n.strstart-n.block_start,i),n.block_start=n.strstart,flush_pending(n.strm)}function put_byte(n,i){n.pending_buf[n.pending++]=i}function putShortMSB(n,i){n.pending_buf[n.pending++]=i>>>8&255,n.pending_buf[n.pending++]=255&i}function longest_match(n,i){var o,u,p=n.max_chain_length,g=n.strstart,y=n.prev_length,w=n.nice_match,x=n.strstart>n.w_size-262?n.strstart-(n.w_size-262):0,_=n.window,k=n.w_mask,P=n.prev,E=n.strstart+258,O=_[g+y-1],B=_[g+y];n.prev_length>=n.good_match&&(p>>=2),w>n.lookahead&&(w=n.lookahead);do{if(_[(o=i)+y]===B&&_[o+y-1]===O&&_[o]===_[g]&&_[++o]===_[g+1]){g+=2,o++;do{}while(_[++g]===_[++o]&&_[++g]===_[++o]&&_[++g]===_[++o]&&_[++g]===_[++o]&&_[++g]===_[++o]&&_[++g]===_[++o]&&_[++g]===_[++o]&&_[++g]===_[++o]&&g<E);if(u=258-(E-g),g=E-258,u>y){if(n.match_start=i,y=u,u>=w)break;O=_[g+y-1],B=_[g+y]}}}while((i=P[i&k])>x&&0!=--p);return y<=n.lookahead?y:n.lookahead}function fill_window(n){var i,o,u,g,x,_,k,P,E,O,B=n.w_size;do{if(g=n.window_size-n.lookahead-n.strstart,n.strstart>=B+(B-262)){p.arraySet(n.window,n.window,B,B,0),n.match_start-=B,n.strstart-=B,n.block_start-=B,i=o=n.hash_size;do{u=n.head[--i],n.head[i]=u>=B?u-B:0}while(--o);i=o=B;do{u=n.prev[--i],n.prev[i]=u>=B?u-B:0}while(--o);g+=B}if(0===n.strm.avail_in)break;if(_=n.strm,k=n.window,P=n.strstart+n.lookahead,E=g,O=void 0,(O=_.avail_in)>E&&(O=E),o=0===O?0:(_.avail_in-=O,p.arraySet(k,_.input,_.next_in,O,P),1===_.state.wrap?_.adler=y(_.adler,k,O,P):2===_.state.wrap&&(_.adler=w(_.adler,k,O,P)),_.next_in+=O,_.total_in+=O,O),n.lookahead+=o,n.lookahead+n.insert>=3)for(x=n.strstart-n.insert,n.ins_h=n.window[x],n.ins_h=(n.ins_h<<n.hash_shift^n.window[x+1])&n.hash_mask;n.insert&&(n.ins_h=(n.ins_h<<n.hash_shift^n.window[x+3-1])&n.hash_mask,n.prev[x&n.w_mask]=n.head[n.ins_h],n.head[n.ins_h]=x,x++,n.insert--,!(n.lookahead+n.insert<3)););}while(n.lookahead<262&&0!==n.strm.avail_in)}function deflate_fast(n,i){for(var o,u;;){if(n.lookahead<262){if(fill_window(n),n.lookahead<262&&0===i)return 1;if(0===n.lookahead)break}if(o=0,n.lookahead>=3&&(n.ins_h=(n.ins_h<<n.hash_shift^n.window[n.strstart+3-1])&n.hash_mask,o=n.prev[n.strstart&n.w_mask]=n.head[n.ins_h],n.head[n.ins_h]=n.strstart),0!==o&&n.strstart-o<=n.w_size-262&&(n.match_length=longest_match(n,o)),n.match_length>=3)if(u=g._tr_tally(n,n.strstart-n.match_start,n.match_length-3),n.lookahead-=n.match_length,n.match_length<=n.max_lazy_match&&n.lookahead>=3){n.match_length--;do{n.strstart++,n.ins_h=(n.ins_h<<n.hash_shift^n.window[n.strstart+3-1])&n.hash_mask,o=n.prev[n.strstart&n.w_mask]=n.head[n.ins_h],n.head[n.ins_h]=n.strstart}while(0!=--n.match_length);n.strstart++}else n.strstart+=n.match_length,n.match_length=0,n.ins_h=n.window[n.strstart],n.ins_h=(n.ins_h<<n.hash_shift^n.window[n.strstart+1])&n.hash_mask;else u=g._tr_tally(n,0,n.window[n.strstart]),n.lookahead--,n.strstart++;if(u&&(flush_block_only(n,!1),0===n.strm.avail_out))return 1}return n.insert=n.strstart<2?n.strstart:2,4===i?(flush_block_only(n,!0),0===n.strm.avail_out?3:4):n.last_lit&&(flush_block_only(n,!1),0===n.strm.avail_out)?1:2}function deflate_slow(n,i){for(var o,u,p;;){if(n.lookahead<262){if(fill_window(n),n.lookahead<262&&0===i)return 1;if(0===n.lookahead)break}if(o=0,n.lookahead>=3&&(n.ins_h=(n.ins_h<<n.hash_shift^n.window[n.strstart+3-1])&n.hash_mask,o=n.prev[n.strstart&n.w_mask]=n.head[n.ins_h],n.head[n.ins_h]=n.strstart),n.prev_length=n.match_length,n.prev_match=n.match_start,n.match_length=2,0!==o&&n.prev_length<n.max_lazy_match&&n.strstart-o<=n.w_size-262&&(n.match_length=longest_match(n,o),n.match_length<=5&&(1===n.strategy||3===n.match_length&&n.strstart-n.match_start>4096)&&(n.match_length=2)),n.prev_length>=3&&n.match_length<=n.prev_length){p=n.strstart+n.lookahead-3,u=g._tr_tally(n,n.strstart-1-n.prev_match,n.prev_length-3),n.lookahead-=n.prev_length-1,n.prev_length-=2;do{++n.strstart<=p&&(n.ins_h=(n.ins_h<<n.hash_shift^n.window[n.strstart+3-1])&n.hash_mask,o=n.prev[n.strstart&n.w_mask]=n.head[n.ins_h],n.head[n.ins_h]=n.strstart)}while(0!=--n.prev_length);if(n.match_available=0,n.match_length=2,n.strstart++,u&&(flush_block_only(n,!1),0===n.strm.avail_out))return 1}else if(n.match_available){if((u=g._tr_tally(n,0,n.window[n.strstart-1]))&&flush_block_only(n,!1),n.strstart++,n.lookahead--,0===n.strm.avail_out)return 1}else n.match_available=1,n.strstart++,n.lookahead--}return n.match_available&&(u=g._tr_tally(n,0,n.window[n.strstart-1]),n.match_available=0),n.insert=n.strstart<2?n.strstart:2,4===i?(flush_block_only(n,!0),0===n.strm.avail_out?3:4):n.last_lit&&(flush_block_only(n,!1),0===n.strm.avail_out)?1:2}function Config(n,i,o,u,p){this.good_length=n,this.max_lazy=i,this.nice_length=o,this.max_chain=u,this.func=p}function DeflateState(){this.strm=null,this.status=0,this.pending_buf=null,this.pending_buf_size=0,this.pending_out=0,this.pending=0,this.wrap=0,this.gzhead=null,this.gzindex=0,this.method=8,this.last_flush=-1,this.w_size=0,this.w_bits=0,this.w_mask=0,this.window=null,this.window_size=0,this.prev=null,this.head=null,this.ins_h=0,this.hash_size=0,this.hash_bits=0,this.hash_mask=0,this.hash_shift=0,this.block_start=0,this.match_length=0,this.prev_match=0,this.match_available=0,this.strstart=0,this.match_start=0,this.lookahead=0,this.prev_length=0,this.max_chain_length=0,this.max_lazy_match=0,this.level=0,this.strategy=0,this.good_match=0,this.nice_match=0,this.dyn_ltree=new p.Buf16(1146),this.dyn_dtree=new p.Buf16(122),this.bl_tree=new p.Buf16(78),zero(this.dyn_ltree),zero(this.dyn_dtree),zero(this.bl_tree),this.l_desc=null,this.d_desc=null,this.bl_desc=null,this.bl_count=new p.Buf16(16),this.heap=new p.Buf16(573),zero(this.heap),this.heap_len=0,this.heap_max=0,this.depth=new p.Buf16(573),zero(this.depth),this.l_buf=0,this.lit_bufsize=0,this.last_lit=0,this.d_buf=0,this.opt_len=0,this.static_len=0,this.matches=0,this.insert=0,this.bi_buf=0,this.bi_valid=0}function deflateResetKeep(n){var i;return n&&n.state?(n.total_in=n.total_out=0,n.data_type=2,(i=n.state).pending=0,i.pending_out=0,i.wrap<0&&(i.wrap=-i.wrap),i.status=i.wrap?42:113,n.adler=2===i.wrap?0:1,i.last_flush=0,g._tr_init(i),0):err(n,-2)}function deflateReset(n){var i=deflateResetKeep(n);return 0===i&&function lm_init(n){n.window_size=2*n.w_size,zero(n.head),n.max_lazy_match=u[n.level].max_lazy,n.good_match=u[n.level].good_length,n.nice_match=u[n.level].nice_length,n.max_chain_length=u[n.level].max_chain,n.strstart=0,n.block_start=0,n.lookahead=0,n.insert=0,n.match_length=n.prev_length=2,n.match_available=0,n.ins_h=0}(n.state),i}function deflateInit2(n,i,o,u,g,y){if(!n)return-2;var w=1;if(-1===i&&(i=6),u<0?(w=0,u=-u):u>15&&(w=2,u-=16),g<1||g>9||8!==o||u<8||u>15||i<0||i>9||y<0||y>4)return err(n,-2);8===u&&(u=9);var x=new DeflateState;return n.state=x,x.strm=n,x.wrap=w,x.gzhead=null,x.w_bits=u,x.w_size=1<<x.w_bits,x.w_mask=x.w_size-1,x.hash_bits=g+7,x.hash_size=1<<x.hash_bits,x.hash_mask=x.hash_size-1,x.hash_shift=~~((x.hash_bits+3-1)/3),x.window=new p.Buf8(2*x.w_size),x.head=new p.Buf16(x.hash_size),x.prev=new p.Buf16(x.w_size),x.lit_bufsize=1<<g+6,x.pending_buf_size=4*x.lit_bufsize,x.pending_buf=new p.Buf8(x.pending_buf_size),x.d_buf=1*x.lit_bufsize,x.l_buf=3*x.lit_bufsize,x.level=i,x.strategy=y,x.method=o,deflateReset(n)}u=[new Config(0,0,0,0,(function deflate_stored(n,i){var o=65535;for(o>n.pending_buf_size-5&&(o=n.pending_buf_size-5);;){if(n.lookahead<=1){if(fill_window(n),0===n.lookahead&&0===i)return 1;if(0===n.lookahead)break}n.strstart+=n.lookahead,n.lookahead=0;var u=n.block_start+o;if((0===n.strstart||n.strstart>=u)&&(n.lookahead=n.strstart-u,n.strstart=u,flush_block_only(n,!1),0===n.strm.avail_out))return 1;if(n.strstart-n.block_start>=n.w_size-262&&(flush_block_only(n,!1),0===n.strm.avail_out))return 1}return n.insert=0,4===i?(flush_block_only(n,!0),0===n.strm.avail_out?3:4):(n.strstart>n.block_start&&(flush_block_only(n,!1),n.strm.avail_out),1)})),new Config(4,4,8,4,deflate_fast),new Config(4,5,16,8,deflate_fast),new Config(4,6,32,32,deflate_fast),new Config(4,4,16,16,deflate_slow),new Config(8,16,32,32,deflate_slow),new Config(8,16,128,128,deflate_slow),new Config(8,32,128,256,deflate_slow),new Config(32,128,258,1024,deflate_slow),new Config(32,258,258,4096,deflate_slow)],i.deflateInit=function deflateInit(n,i){return deflateInit2(n,i,8,15,8,0)},i.deflateInit2=deflateInit2,i.deflateReset=deflateReset,i.deflateResetKeep=deflateResetKeep,i.deflateSetHeader=function deflateSetHeader(n,i){return n&&n.state?2!==n.state.wrap?-2:(n.state.gzhead=i,0):-2},i.deflate=function deflate(n,i){var o,p,y,x;if(!n||!n.state||i>5||i<0)return n?err(n,-2):-2;if(p=n.state,!n.output||!n.input&&0!==n.avail_in||666===p.status&&4!==i)return err(n,0===n.avail_out?-5:-2);if(p.strm=n,o=p.last_flush,p.last_flush=i,42===p.status)if(2===p.wrap)n.adler=0,put_byte(p,31),put_byte(p,139),put_byte(p,8),p.gzhead?(put_byte(p,(p.gzhead.text?1:0)+(p.gzhead.hcrc?2:0)+(p.gzhead.extra?4:0)+(p.gzhead.name?8:0)+(p.gzhead.comment?16:0)),put_byte(p,255&p.gzhead.time),put_byte(p,p.gzhead.time>>8&255),put_byte(p,p.gzhead.time>>16&255),put_byte(p,p.gzhead.time>>24&255),put_byte(p,9===p.level?2:p.strategy>=2||p.level<2?4:0),put_byte(p,255&p.gzhead.os),p.gzhead.extra&&p.gzhead.extra.length&&(put_byte(p,255&p.gzhead.extra.length),put_byte(p,p.gzhead.extra.length>>8&255)),p.gzhead.hcrc&&(n.adler=w(n.adler,p.pending_buf,p.pending,0)),p.gzindex=0,p.status=69):(put_byte(p,0),put_byte(p,0),put_byte(p,0),put_byte(p,0),put_byte(p,0),put_byte(p,9===p.level?2:p.strategy>=2||p.level<2?4:0),put_byte(p,3),p.status=113);else{var _=8+(p.w_bits-8<<4)<<8;_|=(p.strategy>=2||p.level<2?0:p.level<6?1:6===p.level?2:3)<<6,0!==p.strstart&&(_|=32),_+=31-_%31,p.status=113,putShortMSB(p,_),0!==p.strstart&&(putShortMSB(p,n.adler>>>16),putShortMSB(p,65535&n.adler)),n.adler=1}if(69===p.status)if(p.gzhead.extra){for(y=p.pending;p.gzindex<(65535&p.gzhead.extra.length)&&(p.pending!==p.pending_buf_size||(p.gzhead.hcrc&&p.pending>y&&(n.adler=w(n.adler,p.pending_buf,p.pending-y,y)),flush_pending(n),y=p.pending,p.pending!==p.pending_buf_size));)put_byte(p,255&p.gzhead.extra[p.gzindex]),p.gzindex++;p.gzhead.hcrc&&p.pending>y&&(n.adler=w(n.adler,p.pending_buf,p.pending-y,y)),p.gzindex===p.gzhead.extra.length&&(p.gzindex=0,p.status=73)}else p.status=73;if(73===p.status)if(p.gzhead.name){y=p.pending;do{if(p.pending===p.pending_buf_size&&(p.gzhead.hcrc&&p.pending>y&&(n.adler=w(n.adler,p.pending_buf,p.pending-y,y)),flush_pending(n),y=p.pending,p.pending===p.pending_buf_size)){x=1;break}x=p.gzindex<p.gzhead.name.length?255&p.gzhead.name.charCodeAt(p.gzindex++):0,put_byte(p,x)}while(0!==x);p.gzhead.hcrc&&p.pending>y&&(n.adler=w(n.adler,p.pending_buf,p.pending-y,y)),0===x&&(p.gzindex=0,p.status=91)}else p.status=91;if(91===p.status)if(p.gzhead.comment){y=p.pending;do{if(p.pending===p.pending_buf_size&&(p.gzhead.hcrc&&p.pending>y&&(n.adler=w(n.adler,p.pending_buf,p.pending-y,y)),flush_pending(n),y=p.pending,p.pending===p.pending_buf_size)){x=1;break}x=p.gzindex<p.gzhead.comment.length?255&p.gzhead.comment.charCodeAt(p.gzindex++):0,put_byte(p,x)}while(0!==x);p.gzhead.hcrc&&p.pending>y&&(n.adler=w(n.adler,p.pending_buf,p.pending-y,y)),0===x&&(p.status=103)}else p.status=103;if(103===p.status&&(p.gzhead.hcrc?(p.pending+2>p.pending_buf_size&&flush_pending(n),p.pending+2<=p.pending_buf_size&&(put_byte(p,255&n.adler),put_byte(p,n.adler>>8&255),n.adler=0,p.status=113)):p.status=113),0!==p.pending){if(flush_pending(n),0===n.avail_out)return p.last_flush=-1,0}else if(0===n.avail_in&&rank(i)<=rank(o)&&4!==i)return err(n,-5);if(666===p.status&&0!==n.avail_in)return err(n,-5);if(0!==n.avail_in||0!==p.lookahead||0!==i&&666!==p.status){var k=2===p.strategy?function deflate_huff(n,i){for(var o;;){if(0===n.lookahead&&(fill_window(n),0===n.lookahead)){if(0===i)return 1;break}if(n.match_length=0,o=g._tr_tally(n,0,n.window[n.strstart]),n.lookahead--,n.strstart++,o&&(flush_block_only(n,!1),0===n.strm.avail_out))return 1}return n.insert=0,4===i?(flush_block_only(n,!0),0===n.strm.avail_out?3:4):n.last_lit&&(flush_block_only(n,!1),0===n.strm.avail_out)?1:2}(p,i):3===p.strategy?function deflate_rle(n,i){for(var o,u,p,y,w=n.window;;){if(n.lookahead<=258){if(fill_window(n),n.lookahead<=258&&0===i)return 1;if(0===n.lookahead)break}if(n.match_length=0,n.lookahead>=3&&n.strstart>0&&(u=w[p=n.strstart-1])===w[++p]&&u===w[++p]&&u===w[++p]){y=n.strstart+258;do{}while(u===w[++p]&&u===w[++p]&&u===w[++p]&&u===w[++p]&&u===w[++p]&&u===w[++p]&&u===w[++p]&&u===w[++p]&&p<y);n.match_length=258-(y-p),n.match_length>n.lookahead&&(n.match_length=n.lookahead)}if(n.match_length>=3?(o=g._tr_tally(n,1,n.match_length-3),n.lookahead-=n.match_length,n.strstart+=n.match_length,n.match_length=0):(o=g._tr_tally(n,0,n.window[n.strstart]),n.lookahead--,n.strstart++),o&&(flush_block_only(n,!1),0===n.strm.avail_out))return 1}return n.insert=0,4===i?(flush_block_only(n,!0),0===n.strm.avail_out?3:4):n.last_lit&&(flush_block_only(n,!1),0===n.strm.avail_out)?1:2}(p,i):u[p.level].func(p,i);if(3!==k&&4!==k||(p.status=666),1===k||3===k)return 0===n.avail_out&&(p.last_flush=-1),0;if(2===k&&(1===i?g._tr_align(p):5!==i&&(g._tr_stored_block(p,0,0,!1),3===i&&(zero(p.head),0===p.lookahead&&(p.strstart=0,p.block_start=0,p.insert=0))),flush_pending(n),0===n.avail_out))return p.last_flush=-1,0}return 4!==i?0:p.wrap<=0?1:(2===p.wrap?(put_byte(p,255&n.adler),put_byte(p,n.adler>>8&255),put_byte(p,n.adler>>16&255),put_byte(p,n.adler>>24&255),put_byte(p,255&n.total_in),put_byte(p,n.total_in>>8&255),put_byte(p,n.total_in>>16&255),put_byte(p,n.total_in>>24&255)):(putShortMSB(p,n.adler>>>16),putShortMSB(p,65535&n.adler)),flush_pending(n),p.wrap>0&&(p.wrap=-p.wrap),0!==p.pending?0:1)},i.deflateEnd=function deflateEnd(n){var i;return n&&n.state?42!==(i=n.state.status)&&69!==i&&73!==i&&91!==i&&103!==i&&113!==i&&666!==i?err(n,-2):(n.state=null,113===i?err(n,-3):0):-2},i.deflateSetDictionary=function deflateSetDictionary(n,i){var o,u,g,w,x,_,k,P,E=i.length;if(!n||!n.state)return-2;if(2===(w=(o=n.state).wrap)||1===w&&42!==o.status||o.lookahead)return-2;for(1===w&&(n.adler=y(n.adler,i,E,0)),o.wrap=0,E>=o.w_size&&(0===w&&(zero(o.head),o.strstart=0,o.block_start=0,o.insert=0),P=new p.Buf8(o.w_size),p.arraySet(P,i,E-o.w_size,o.w_size,0),i=P,E=o.w_size),x=n.avail_in,_=n.next_in,k=n.input,n.avail_in=E,n.next_in=0,n.input=i,fill_window(o);o.lookahead>=3;){u=o.strstart,g=o.lookahead-2;do{o.ins_h=(o.ins_h<<o.hash_shift^o.window[u+3-1])&o.hash_mask,o.prev[u&o.w_mask]=o.head[o.ins_h],o.head[o.ins_h]=u,u++}while(--g);o.strstart=u,o.lookahead=2,fill_window(o)}return o.strstart+=o.lookahead,o.block_start=o.strstart,o.insert=o.lookahead,o.lookahead=0,o.match_length=o.prev_length=2,o.match_available=0,n.next_in=_,n.input=k,n.avail_in=x,o.wrap=w,0},i.deflateInfo="pako deflate (from Nodeca project)"},function(n,i,o){"use strict";var u=o(126);function zero(n){for(var i=n.length;--i>=0;)n[i]=0}var p=[0,0,0,0,0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5,0],g=[0,0,0,0,1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10,11,11,12,12,13,13],y=[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,7],w=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15],x=new Array(576);zero(x);var _=new Array(60);zero(_);var k=new Array(512);zero(k);var P=new Array(256);zero(P);var E=new Array(29);zero(E);var O,B,I,D=new Array(30);function StaticTreeDesc(n,i,o,u,p){this.static_tree=n,this.extra_bits=i,this.extra_base=o,this.elems=u,this.max_length=p,this.has_stree=n&&n.length}function TreeDesc(n,i){this.dyn_tree=n,this.max_code=0,this.stat_desc=i}function d_code(n){return n<256?k[n]:k[256+(n>>>7)]}function put_short(n,i){n.pending_buf[n.pending++]=255&i,n.pending_buf[n.pending++]=i>>>8&255}function send_bits(n,i,o){n.bi_valid>16-o?(n.bi_buf|=i<<n.bi_valid&65535,put_short(n,n.bi_buf),n.bi_buf=i>>16-n.bi_valid,n.bi_valid+=o-16):(n.bi_buf|=i<<n.bi_valid&65535,n.bi_valid+=o)}function send_code(n,i,o){send_bits(n,o[2*i],o[2*i+1])}function bi_reverse(n,i){var o=0;do{o|=1&n,n>>>=1,o<<=1}while(--i>0);return o>>>1}function gen_codes(n,i,o){var u,p,g=new Array(16),y=0;for(u=1;u<=15;u++)g[u]=y=y+o[u-1]<<1;for(p=0;p<=i;p++){var w=n[2*p+1];0!==w&&(n[2*p]=bi_reverse(g[w]++,w))}}function init_block(n){var i;for(i=0;i<286;i++)n.dyn_ltree[2*i]=0;for(i=0;i<30;i++)n.dyn_dtree[2*i]=0;for(i=0;i<19;i++)n.bl_tree[2*i]=0;n.dyn_ltree[512]=1,n.opt_len=n.static_len=0,n.last_lit=n.matches=0}function bi_windup(n){n.bi_valid>8?put_short(n,n.bi_buf):n.bi_valid>0&&(n.pending_buf[n.pending++]=n.bi_buf),n.bi_buf=0,n.bi_valid=0}function smaller(n,i,o,u){var p=2*i,g=2*o;return n[p]<n[g]||n[p]===n[g]&&u[i]<=u[o]}function pqdownheap(n,i,o){for(var u=n.heap[o],p=o<<1;p<=n.heap_len&&(p<n.heap_len&&smaller(i,n.heap[p+1],n.heap[p],n.depth)&&p++,!smaller(i,u,n.heap[p],n.depth));)n.heap[o]=n.heap[p],o=p,p<<=1;n.heap[o]=u}function compress_block(n,i,o){var u,y,w,x,_=0;if(0!==n.last_lit)do{u=n.pending_buf[n.d_buf+2*_]<<8|n.pending_buf[n.d_buf+2*_+1],y=n.pending_buf[n.l_buf+_],_++,0===u?send_code(n,y,i):(send_code(n,(w=P[y])+256+1,i),0!==(x=p[w])&&send_bits(n,y-=E[w],x),send_code(n,w=d_code(--u),o),0!==(x=g[w])&&send_bits(n,u-=D[w],x))}while(_<n.last_lit);send_code(n,256,i)}function build_tree(n,i){var o,u,p,g=i.dyn_tree,y=i.stat_desc.static_tree,w=i.stat_desc.has_stree,x=i.stat_desc.elems,_=-1;for(n.heap_len=0,n.heap_max=573,o=0;o<x;o++)0!==g[2*o]?(n.heap[++n.heap_len]=_=o,n.depth[o]=0):g[2*o+1]=0;for(;n.heap_len<2;)g[2*(p=n.heap[++n.heap_len]=_<2?++_:0)]=1,n.depth[p]=0,n.opt_len--,w&&(n.static_len-=y[2*p+1]);for(i.max_code=_,o=n.heap_len>>1;o>=1;o--)pqdownheap(n,g,o);p=x;do{o=n.heap[1],n.heap[1]=n.heap[n.heap_len--],pqdownheap(n,g,1),u=n.heap[1],n.heap[--n.heap_max]=o,n.heap[--n.heap_max]=u,g[2*p]=g[2*o]+g[2*u],n.depth[p]=(n.depth[o]>=n.depth[u]?n.depth[o]:n.depth[u])+1,g[2*o+1]=g[2*u+1]=p,n.heap[1]=p++,pqdownheap(n,g,1)}while(n.heap_len>=2);n.heap[--n.heap_max]=n.heap[1],function gen_bitlen(n,i){var o,u,p,g,y,w,x=i.dyn_tree,_=i.max_code,k=i.stat_desc.static_tree,P=i.stat_desc.has_stree,E=i.stat_desc.extra_bits,O=i.stat_desc.extra_base,B=i.stat_desc.max_length,I=0;for(g=0;g<=15;g++)n.bl_count[g]=0;for(x[2*n.heap[n.heap_max]+1]=0,o=n.heap_max+1;o<573;o++)(g=x[2*x[2*(u=n.heap[o])+1]+1]+1)>B&&(g=B,I++),x[2*u+1]=g,u>_||(n.bl_count[g]++,y=0,u>=O&&(y=E[u-O]),w=x[2*u],n.opt_len+=w*(g+y),P&&(n.static_len+=w*(k[2*u+1]+y)));if(0!==I){do{for(g=B-1;0===n.bl_count[g];)g--;n.bl_count[g]--,n.bl_count[g+1]+=2,n.bl_count[B]--,I-=2}while(I>0);for(g=B;0!==g;g--)for(u=n.bl_count[g];0!==u;)(p=n.heap[--o])>_||(x[2*p+1]!==g&&(n.opt_len+=(g-x[2*p+1])*x[2*p],x[2*p+1]=g),u--)}}(n,i),gen_codes(g,_,n.bl_count)}function scan_tree(n,i,o){var u,p,g=-1,y=i[1],w=0,x=7,_=4;for(0===y&&(x=138,_=3),i[2*(o+1)+1]=65535,u=0;u<=o;u++)p=y,y=i[2*(u+1)+1],++w<x&&p===y||(w<_?n.bl_tree[2*p]+=w:0!==p?(p!==g&&n.bl_tree[2*p]++,n.bl_tree[32]++):w<=10?n.bl_tree[34]++:n.bl_tree[36]++,w=0,g=p,0===y?(x=138,_=3):p===y?(x=6,_=3):(x=7,_=4))}function send_tree(n,i,o){var u,p,g=-1,y=i[1],w=0,x=7,_=4;for(0===y&&(x=138,_=3),u=0;u<=o;u++)if(p=y,y=i[2*(u+1)+1],!(++w<x&&p===y)){if(w<_)do{send_code(n,p,n.bl_tree)}while(0!=--w);else 0!==p?(p!==g&&(send_code(n,p,n.bl_tree),w--),send_code(n,16,n.bl_tree),send_bits(n,w-3,2)):w<=10?(send_code(n,17,n.bl_tree),send_bits(n,w-3,3)):(send_code(n,18,n.bl_tree),send_bits(n,w-11,7));w=0,g=p,0===y?(x=138,_=3):p===y?(x=6,_=3):(x=7,_=4)}}zero(D);var R=!1;function _tr_stored_block(n,i,o,p){send_bits(n,0+(p?1:0),3),function copy_block(n,i,o,p){bi_windup(n),p&&(put_short(n,o),put_short(n,~o)),u.arraySet(n.pending_buf,n.window,i,o,n.pending),n.pending+=o}(n,i,o,!0)}i._tr_init=function _tr_init(n){R||(!function tr_static_init(){var n,i,o,u,w,R=new Array(16);for(o=0,u=0;u<28;u++)for(E[u]=o,n=0;n<1<<p[u];n++)P[o++]=u;for(P[o-1]=u,w=0,u=0;u<16;u++)for(D[u]=w,n=0;n<1<<g[u];n++)k[w++]=u;for(w>>=7;u<30;u++)for(D[u]=w<<7,n=0;n<1<<g[u]-7;n++)k[256+w++]=u;for(i=0;i<=15;i++)R[i]=0;for(n=0;n<=143;)x[2*n+1]=8,n++,R[8]++;for(;n<=255;)x[2*n+1]=9,n++,R[9]++;for(;n<=279;)x[2*n+1]=7,n++,R[7]++;for(;n<=287;)x[2*n+1]=8,n++,R[8]++;for(gen_codes(x,287,R),n=0;n<30;n++)_[2*n+1]=5,_[2*n]=bi_reverse(n,5);O=new StaticTreeDesc(x,p,257,286,15),B=new StaticTreeDesc(_,g,0,30,15),I=new StaticTreeDesc(new Array(0),y,0,19,7)}(),R=!0),n.l_desc=new TreeDesc(n.dyn_ltree,O),n.d_desc=new TreeDesc(n.dyn_dtree,B),n.bl_desc=new TreeDesc(n.bl_tree,I),n.bi_buf=0,n.bi_valid=0,init_block(n)},i._tr_stored_block=_tr_stored_block,i._tr_flush_block=function _tr_flush_block(n,i,o,u){var p,g,y=0;n.level>0?(2===n.strm.data_type&&(n.strm.data_type=function detect_data_type(n){var i,o=4093624447;for(i=0;i<=31;i++,o>>>=1)if(1&o&&0!==n.dyn_ltree[2*i])return 0;if(0!==n.dyn_ltree[18]||0!==n.dyn_ltree[20]||0!==n.dyn_ltree[26])return 1;for(i=32;i<256;i++)if(0!==n.dyn_ltree[2*i])return 1;return 0}(n)),build_tree(n,n.l_desc),build_tree(n,n.d_desc),y=function build_bl_tree(n){var i;for(scan_tree(n,n.dyn_ltree,n.l_desc.max_code),scan_tree(n,n.dyn_dtree,n.d_desc.max_code),build_tree(n,n.bl_desc),i=18;i>=3&&0===n.bl_tree[2*w[i]+1];i--);return n.opt_len+=3*(i+1)+5+5+4,i}(n),p=n.opt_len+3+7>>>3,(g=n.static_len+3+7>>>3)<=p&&(p=g)):p=g=o+5,o+4<=p&&-1!==i?_tr_stored_block(n,i,o,u):4===n.strategy||g===p?(send_bits(n,2+(u?1:0),3),compress_block(n,x,_)):(send_bits(n,4+(u?1:0),3),function send_all_trees(n,i,o,u){var p;for(send_bits(n,i-257,5),send_bits(n,o-1,5),send_bits(n,u-4,4),p=0;p<u;p++)send_bits(n,n.bl_tree[2*w[p]+1],3);send_tree(n,n.dyn_ltree,i-1),send_tree(n,n.dyn_dtree,o-1)}(n,n.l_desc.max_code+1,n.d_desc.max_code+1,y+1),compress_block(n,n.dyn_ltree,n.dyn_dtree)),init_block(n),u&&bi_windup(n)},i._tr_tally=function _tr_tally(n,i,o){return n.pending_buf[n.d_buf+2*n.last_lit]=i>>>8&255,n.pending_buf[n.d_buf+2*n.last_lit+1]=255&i,n.pending_buf[n.l_buf+n.last_lit]=255&o,n.last_lit++,0===i?n.dyn_ltree[2*o]++:(n.matches++,i--,n.dyn_ltree[2*(P[o]+256+1)]++,n.dyn_dtree[2*d_code(i)]++),n.last_lit===n.lit_bufsize-1},i._tr_align=function _tr_align(n){send_bits(n,2,3),send_code(n,256,x),function bi_flush(n){16===n.bi_valid?(put_short(n,n.bi_buf),n.bi_buf=0,n.bi_valid=0):n.bi_valid>=8&&(n.pending_buf[n.pending++]=255&n.bi_buf,n.bi_buf>>=8,n.bi_valid-=8)}(n)}},function(n,i,o){"use strict";n.exports={2:"need dictionary",1:"stream end",0:"","-1":"file error","-2":"stream error","-3":"data error","-4":"insufficient memory","-5":"buffer error","-6":"incompatible version"}},function(n,i,o){"use strict";var u=o(126),p=o(247),g=o(248),y=o(365),w=o(366);function zswap32(n){return(n>>>24&255)+(n>>>8&65280)+((65280&n)<<8)+((255&n)<<24)}function InflateState(){this.mode=0,this.last=!1,this.wrap=0,this.havedict=!1,this.flags=0,this.dmax=0,this.check=0,this.total=0,this.head=null,this.wbits=0,this.wsize=0,this.whave=0,this.wnext=0,this.window=null,this.hold=0,this.bits=0,this.length=0,this.offset=0,this.extra=0,this.lencode=null,this.distcode=null,this.lenbits=0,this.distbits=0,this.ncode=0,this.nlen=0,this.ndist=0,this.have=0,this.next=null,this.lens=new u.Buf16(320),this.work=new u.Buf16(288),this.lendyn=null,this.distdyn=null,this.sane=0,this.back=0,this.was=0}function inflateResetKeep(n){var i;return n&&n.state?(i=n.state,n.total_in=n.total_out=i.total=0,n.msg="",i.wrap&&(n.adler=1&i.wrap),i.mode=1,i.last=0,i.havedict=0,i.dmax=32768,i.head=null,i.hold=0,i.bits=0,i.lencode=i.lendyn=new u.Buf32(852),i.distcode=i.distdyn=new u.Buf32(592),i.sane=1,i.back=-1,0):-2}function inflateReset(n){var i;return n&&n.state?((i=n.state).wsize=0,i.whave=0,i.wnext=0,inflateResetKeep(n)):-2}function inflateReset2(n,i){var o,u;return n&&n.state?(u=n.state,i<0?(o=0,i=-i):(o=1+(i>>4),i<48&&(i&=15)),i&&(i<8||i>15)?-2:(null!==u.window&&u.wbits!==i&&(u.window=null),u.wrap=o,u.wbits=i,inflateReset(n))):-2}function inflateInit2(n,i){var o,u;return n?(u=new InflateState,n.state=u,u.window=null,0!==(o=inflateReset2(n,i))&&(n.state=null),o):-2}var x,_,k=!0;function fixedtables(n){if(k){var i;for(x=new u.Buf32(512),_=new u.Buf32(32),i=0;i<144;)n.lens[i++]=8;for(;i<256;)n.lens[i++]=9;for(;i<280;)n.lens[i++]=7;for(;i<288;)n.lens[i++]=8;for(w(1,n.lens,0,288,x,0,n.work,{bits:9}),i=0;i<32;)n.lens[i++]=5;w(2,n.lens,0,32,_,0,n.work,{bits:5}),k=!1}n.lencode=x,n.lenbits=9,n.distcode=_,n.distbits=5}function updatewindow(n,i,o,p){var g,y=n.state;return null===y.window&&(y.wsize=1<<y.wbits,y.wnext=0,y.whave=0,y.window=new u.Buf8(y.wsize)),p>=y.wsize?(u.arraySet(y.window,i,o-y.wsize,y.wsize,0),y.wnext=0,y.whave=y.wsize):((g=y.wsize-y.wnext)>p&&(g=p),u.arraySet(y.window,i,o-p,g,y.wnext),(p-=g)?(u.arraySet(y.window,i,o-p,p,0),y.wnext=p,y.whave=y.wsize):(y.wnext+=g,y.wnext===y.wsize&&(y.wnext=0),y.whave<y.wsize&&(y.whave+=g))),0}i.inflateReset=inflateReset,i.inflateReset2=inflateReset2,i.inflateResetKeep=inflateResetKeep,i.inflateInit=function inflateInit(n){return inflateInit2(n,15)},i.inflateInit2=inflateInit2,i.inflate=function inflate(n,i){var o,x,_,k,P,E,O,B,I,D,R,N,U,W,G,j,X,Y,K,J,$,tt,et,rt,nt=0,it=